// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Types/hkSignalSlots.h>
#include <Common/Base/Config/hkConfigVersion.h>
#include <Common/Base/System/Io/Socket/hkInetAddr.h>
#include <Common/Visualize/hkVersionReporter.h>

#include <VisualDebugger/VdbServices/System/Connection/hkVdbConnection.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmd.h>




#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbCmdHandler.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbSetupHandler.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbProcessHandler.h>

struct hkVdbProcessInfo;
struct hkVdbFrame;
class hkVdbCmdInput;
class hkVdbCmdOutput;
class hkVdbCmdDispatcher;
class hkVdbCache;
class hkVdbPersistentStateCache;
class hkVdbInternalHandler;
class hkVdbProgressReporter;
class hkStreamWriter;
class hkMemoryTrack;
class hkInetAddr;
struct UpdateConnectionStatesHelper;

//////////////////////////////////////////////////////////////////////////
/// This class is used to connect with sources/destinations of vdb cmd data.
//////////////////////////////////////////////////////////////////////////
class HK_EXPORT_VDB hkVdbClient : public hkVdbDefaultErrorReporter
{
public:
    HK_DECLARE_CLASS( hkVdbClient, New );

    hkVdbClient();
    ~hkVdbClient();

    //
    // Processing
    //

    
    /// The current application step state. These values offer a rough measure
    /// of new or completed steps coming from the application.
    /// In addition to being able to retrieve this state, there are also
    /// step-related signals you can listen to.
    /// If you are interacting with the system in any nuanced way based on step/frame
    /// boundaries, it is recommended to use information and signals provided by
    /// the hkVdbPlaybackHandler instead.
    enum StepState
    {
        /// We have started a new application connection and have not yet
        /// received any data.
        STEP_NOT_STARTED,

        /// We have started receiving data for a new step from the application.
        STEP_STARTED,

        /// We have completed processing an application step.
        STEP_COMPLETED,
    };

    /// Process more from the application connection.
    /// This must be called at regular intervals for the framework to move forward.
    hkResult process( hkVdbProcessInfo processingInfo = hkVdbProcessInfo() );

    /// Call this after processStep to determine what the step state is.
    StepState getStepState() { return m_state; }

    /// Not currently used, but intended for future MT work.
    hkResult waitForCompletion();
    

    //
    // Commands
    //

    
    /// For advanced use only.
    /// Get the cmd input object. All cmds are placed into this input object.
    /// Cmds are usually responded to via hkVdbCmdHandler signals and playback
    /// affected via the hkVdbPlaybackHandler.
    hkVdbCmdInput& getCmdInput() { return *m_input; }

    /// For advanced use only.
    /// Get the cmd dispatcher responsible for sending cmds out to registered handlers.
    /// Cmds are usually responded to via hkVdbCmdHandler signals and playback
    /// affected via the hkVdbPlaybackHandler.
    hkVdbCmdDispatcher& getCmdDispatcher() { return *m_dispatcher; }

    /// For advanced use only.
    /// Get the cmd output object. This can be used to send a cmd back to the server.
    hkVdbCmdOutput& getCmdOutput() { return *m_output; }
    

    //
    // Caches
    //

    
    /// For advanced use only.
    /// Get the cmd cache object. This holds extra data per-frame created by the cmd handlers
    /// for use during replay, saving, etc.
    hkVdbCache& getCache() { return *m_cache; }

    /// For advanced use only.
    /// Get the cmd persistent state cache object. This class is responsible for persisting
    /// needed data at the beginning of the circular buffer for use during replay, saving, etc.
    hkVdbPersistentStateCache& getPersistentStateCache() { return *m_persistentStateCache; }
    

    //
    // Connections
    //

    
    /// Create a connection between the vdb client and another machine with the given name/port.
    /// The hkVdbConnectionUse enum determines how this connection will be used.
    /// If disconnectCurrentConnection, the current hkVdbConnection (if any) will be disconnected.
    hkResult connectMachine(
        const char* machineName = HK_VDB_DEFAULT_MACHINE_NAME,
        hkUint16 port = HK_VDB_DEFAULT_PORT,
        hkReal timeoutMs = HK_VDB_DEFAULT_NETWORK_TIMEOUT_MS,
        hkVdbConnectionUse::Enum use = hkVdbConnectionUse::APPLICATION,
        bool disconnectCurrentConnection = true );

    /// Create a connection for the vdb client which listens on the given port.
    /// poll() must be called to realize new connections.
    /// timeoutMs can be provided for APPLICATION_SERVER_DISCOVERY use (currently ignored otherwise).
    ///<todo.jg.4vdb apply timeoutMs to listening port network connection?
    /// The hkVdbConnectionUse enum determines how this connection will be used.
    /// If disconnectCurrentConnection, the current hkVdbConnection (if any) will be disconnected.
    hkResult connectPort(
        hkUint16 port,
        hkReal timeoutMs = HK_VDB_DEFAULT_NETWORK_TIMEOUT_MS,
        hkVdbConnectionUse::Enum use = hkVdbConnectionUse::APPLICATION,
        bool disconnectCurrentConnection = true );

    /// Create a connection between the vdb client and a stream of data on disk.
    /// The hkVdbConnectionUse enum determines how this connection will be used.
    /// If disconnectCurrentConnection, the current hkVdbConnection (if any) will be disconnected.
    hkResult connectFile( const char* filepath, hkVdbConnectionUse::Enum use = hkVdbConnectionUse::APPLICATION, bool disconnectCurrentConnection = true );

    /// Create a connection between the vdb client and a stream of data in memory.
    /// The hkVdbConnectionUse enum determines how this connection will be used.
    /// If disconnectCurrentConnection, the current hkVdbConnection (if any) will be disconnected.
    hkResult connectMemory( hkMemoryTrack* memory, hkVdbConnectionUse::Enum use = hkVdbConnectionUse::APPLICATION, bool disconnectCurrentConnection = true );

    /// Create a no-op-connection which produces no errors but does not read/write any data.
    /// The hkVdbConnectionUse enum determines how this connection will be used.
    /// If disconnectCurrentConnection, the current hkVdbConnection (if any) will be disconnected.
    hkResult connectNoOp( hkVdbConnectionUse::Enum use = hkVdbConnectionUse::APPLICATION, bool disconnectCurrentConnection = true );

    /// Disconnect the current connection for the specified use.
    hkResult disconnect( hkVdbConnectionUse::Enum use = hkVdbConnectionUse::APPLICATION );

    /// Disconnect all connections.
    hkResult disconnectAll();

    /// Call poll() on all hkVdbConnections.
    /// Some connections use this to realize their connection or gather more data, etc.
    hkResult poll();

    /// For advanced use only.
    /// Set the connection for the specified use.
    /// If disconnectCurrentConnection, the current hkVdbConnection (if any) will be disconnected.
    hkResult setConnection( hkVdbConnection* connection, hkVdbConnectionUse::Enum use = hkVdbConnectionUse::APPLICATION, bool disconnectCurrentConnection = true );

    /// Get the current connection for the specified use.
    HK_INLINE hkVdbConnection* getConnection( hkVdbConnectionUse::Enum use = hkVdbConnectionUse::APPLICATION ) { return m_connections[use]; }
    

    //
    // Handlers
    //

    
    /// Get a cmd handler of a given type (Eg. hkVdbPlaybackHandler* handler = getCmdHandler<hkVdbPlaybackHandler>() ).
    /// Cmds handlers are the main way to listen for incoming cmds (via cmd handler signals) and control
    /// various aspects of the system (such as playback via the hkVdbPlaybackHandler).
    
    template<typename AnyVdbCmdHandler>
    AnyVdbCmdHandler* getCmdHandler();
    

    //
    // Progress
    //

    
    /// Get the progress reporter.
    /// Parts of the framework use this to report on progress for lengthy operations.
    /// It can be used to understand stalling work-loads as well as cancel them.
    hkVdbProgressReporter& getProgressReporter() { return *m_progressReporter; }
    

    //
    // Signals
    //

    
    HK_DECLARE_SIGNAL( StepStartedSignal, hkSignal2< const hkVdbFrame&, const hkVdbCmd& > );
    /// Occurs when a we've started processing a new step from the application connection.
    StepStartedSignal m_stepStarted;

    HK_DECLARE_SIGNAL( StepCompletedSignal, hkSignal2< const hkVdbFrame&, const hkVdbCmd& > );
    /// Occurs when a we've finished processing a step from the application connection.
    StepCompletedSignal m_stepCompleted;

    HK_DECLARE_SIGNAL( ConnectingSignal, hkSignal3< hkVdbConnectionUse::Enum, hkVdbConnection&, hkVdbSignalResult& > );
    /// Occurs when any new connection is set and told to connect. When the connection reports it has connected
    /// the m_connected signal will fire.
    ConnectingSignal m_connecting;

    HK_DECLARE_SIGNAL( ConnectedSignal, hkSignal2< hkVdbConnectionUse::Enum, hkVdbConnection& > );
    /// Occurs when a connection first reports it has connected. When the connection is first
    /// set and told to connect, the m_connecting signal is fired.
    ConnectedSignal m_connected;

    HK_DECLARE_SIGNAL( DisconnectingSignal, hkSignal3< hkVdbConnectionUse::Enum, hkVdbConnection&, hkVdbSignalResult& > );
    /// Occurs when a connection has been told to disconnect. When the connection reports it has disconnected
    /// the m_disconnected signal will fire.
    DisconnectingSignal m_disconnecting;

    HK_DECLARE_SIGNAL( DisconnectedSignal, hkSignal2< hkVdbConnectionUse::Enum, hkVdbConnection& > );
    /// Occurs when a connection reports it has disconnected. When the connection is first told to disconnect,
    /// the m_disconnecting signal is fired.
    DisconnectedSignal m_disconnected;

    HK_DECLARE_SIGNAL( WritingToSinkSignal, hkSignal5< hkVdbConnection&, const hkVdbFrame&, const hkVdbCmd&, int, const hkVdbCmd*& > );
    /// For advanced use only.
    /// This signal occurs when a cmd is about to bet written to the sink connection.
    WritingToSinkSignal m_writingToSink;
    

    //
    // Internal use
    //
    void onCmdDispatchingSignal( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbSignalResult& result );
    void onCmdDispatchedSignal( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbSignalResult& result );
    void onConnectingSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection, hkVdbSignalResult& result );
    void onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection );
    void onDisconnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection );
    void onServerInfoReceivedSignal( const hkVdbSetupHandler::ClientInfo&, const hkVdbSetupHandler::ServerInfo&, hkVdbCapabilities, hkVdbSignalResult& );
    void onProcessCmdReceivedSignal( const hkVdbProcessHandler::ProcessCmd& processCmd, hkVdbSignalResult& result );

protected:

    enum SinkInitializationState
    {
        SINK_STATE_NONE,
        SINK_WAITING_FOR_PREREQS,
        SINK_UNINITIALIZED,
        SINK_RECONNECTING, // only for pre-2016.1 connections
        SINK_INITIALIZED
    };

    enum SinkInitializationStatePreReqs
    {
        SINK_PREREQ_NONE = 0,
        SINK_PREREQ_2_FRAMES = ( 1 << 0 ),
        SINK_PREREQ_SETUP_INFO = ( 1 << 1 ),
        SINK_ALL_PREREQS = ( SINK_PREREQ_2_FRAMES | SINK_PREREQ_SETUP_INFO )
    };

    struct UpdateConnectionStatesHelper
    {
        UpdateConnectionStatesHelper( hkVdbClient& client ) :
            m_client( client ),
            m_returnedResult( false )
        {}
        ~UpdateConnectionStatesHelper();
        operator hkResult();
        hkVdbClient& m_client;
        hkBool32 m_returnedResult;
    };

    hkResult updateSinkConnection( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol );
    hkResult updateConnectionStates();
    hkResult setConnectionState( hkVdbConnectionUse::Enum use, hkVdbConnectionState::Enum state, bool disconnectConnectionOnDisconnectState );
    hkResult updateConnectionState( hkVdbConnectionUse::Enum use );
    hkResult readCmd( const hkVdbFrame*& frameOut, const hkVdbCmd*& cmdOut, StepState& prevStepStateOut );
    hkResult clearCmdHandlers();
    hkResult setCmdHandlers();

    StepState m_state;
    hkRefPtr<hkVdbCmdInput> m_input;
    hkRefPtr<hkVdbCmdDispatcher> m_dispatcher;
    hkRefPtr<hkVdbCmdOutput> m_output;
    hkRefPtr<hkVdbCache> m_cache;
    hkRefPtr<hkVdbPersistentStateCache> m_persistentStateCache;
    hkRefPtr<hkVdbConnection> m_connections[hkVdbConnectionUse::NUM_USES];
    hkVdbConnectionState::Enum m_connectionStates[hkVdbConnectionUse::NUM_USES];
    SinkInitializationState m_sinkInitializationState;
    hkFlags<SinkInitializationStatePreReqs, hkUint8> m_sinkInitializationCompletedPreReqs;
    hkArray<hkStringPtr> m_savedSelectedProcesses;
    hkRefPtr<hkVdbAnyCmdHandler> m_handlers[hkVdbCmdHandlerType::NUM_TYPES + 1];
    hkRefPtr<hkVdbProgressReporter> m_progressReporter;
};

#include <VisualDebugger/VdbServices/hkVdbClient.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
