// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <Common/Visualize/hkVisualDebugger.h>

#include <VisualDebugger/VdbServices/hkVdbServices.h>
#include <VisualDebugger/VdbServices/hkVdbClient.h>

#include <VisualDebugger/VdbServices/System/hkVdbProgressReporter.h>

#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdInput.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdOutput.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdDispatcher.h>

#include <VisualDebugger/VdbServices/System/Connection/hkVdbNetworkConnection.h>
#include <VisualDebugger/VdbServices/System/Connection/hkVdbFileConnection.h>
#include <VisualDebugger/VdbServices/System/Connection/hkVdbMemoryConnection.h>
#include <VisualDebugger/VdbServices/System/Connection/hkVdbNoOpConnection.h>
#include <VisualDebugger/VdbServices/System/Connection/hkVdbServerDiscoveryConnection.h>

#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbSetupHandler.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbPlaybackHandler.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbDisplayHandler.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbObjectHandler.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbStatsHandler.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbTextHandler.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbFileHandler.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbCustomHandler.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbInternalHandler.h>

#include <VisualDebugger/VdbServices/System/Cache/hkVdbCache.h>
#include <VisualDebugger/VdbServices/System/Cache/hkVdbPersistentStateCache.h>

#include <VisualDebugger/VdbServices/System/Utils/hkVdbBlockingWriter.h>

#define DEBUG_LOG_IDENTIFIER "vdb.Client"
#include <Common/Base/System/Log/hkLog.hxx>

#define VERIFY_IS_CONNECTING( USE, CONNECTION ) \
    HK_MULTILINE_MACRO_BEGIN \
        if ( updateConnectionState( USE ).isFailure() ) { return HK_FAILURE; } \
        HK_VDB_VERIFY_CONDITION_MSG( \
            CONNECTION && ( ( m_connectionStates[USE] == hkVdbConnectionState::CONNECTING ) || ( m_connectionStates[USE] == hkVdbConnectionState::CONNECTED ) ), \
            0xedb00005, \
            hkVdbError::NO_CONNECTION, \
            "No connected connection for use [" << USE << "]" ); \
    HK_MULTILINE_MACRO_END

//#define USE_NO_OP_CONNECTION


#include <Common/Base/System/Stopwatch/hkSystemClock.h>
hkBool32 hkVdbProcessInfo::hasMoreTime() const
{
    return ( hkSystemClock::getTickCounter() <= m_endTick );
}
void hkVdbProcessInfo::computeEndTickCount()
{
    const hkUint64 processingTicks = hkUint64( m_processingTime * hkSystemClock::getTicksPerSecond() );
    m_endTick = ( m_processingTime > 0 ) * ( hkSystemClock::getTickCounter() + processingTicks );
}

hkVdbClient::hkVdbClient() :
    hkVdbDefaultErrorReporter( &s_debugLog ),
    m_state( STEP_NOT_STARTED ),
    m_input( hkRefNew<hkVdbCmdInput>( new hkVdbCmdInput() ) ),
    m_dispatcher( hkRefNew<hkVdbCmdDispatcher>( new hkVdbCmdDispatcher() ) ),
    m_output( hkRefNew<hkVdbCmdOutput>( new hkVdbCmdOutput( *m_input ) ) ),
    m_cache( hkRefNew<hkVdbCache>( new hkVdbCache( *m_input ) ) ),
    m_persistentStateCache( hkRefNew<hkVdbPersistentStateCache>( new hkVdbPersistentStateCache( *m_input ) ) ),
    m_sinkInitializationState( SINK_STATE_NONE ),
    m_sinkInitializationCompletedPreReqs( SINK_PREREQ_NONE ),
    m_progressReporter( hkRefNew<hkVdbProgressReporter>( new hkVdbDefaultProgressReporter() ) )
{
    // Clear this array
    hkString::memSet( &m_connectionStates, 0, sizeof( m_connectionStates ) );

    // Register subsystems to connect/disconnect *before* handlers so that they won't remove data
    // handlers may rely on when they respond to connect/disconnect.
    HK_SUBSCRIBE_TO_SIGNAL( m_connecting, this, hkVdbClient );
    HK_SUBSCRIBE_TO_SIGNAL( m_connected, this, hkVdbClient );
    HK_SUBSCRIBE_TO_SIGNAL( m_disconnected, this, hkVdbClient );
    HK_SUBSCRIBE_TO_SIGNAL( m_connected, m_input.val(), hkVdbCmdInput );
    HK_SUBSCRIBE_TO_SIGNAL( m_connected, m_output.val(), hkVdbCmdOutput );
    HK_SUBSCRIBE_TO_SIGNAL( m_disconnecting, m_output.val(), hkVdbCmdOutput );
    HK_SUBSCRIBE_TO_SIGNAL( m_connected, m_persistentStateCache.val(), hkVdbPersistentStateCache );

    
    setCmdHandlers();

    // Register subsystems to other signals *after* handlers so that their state will be current
    // for handlers that reference them.
    
    HK_SUBSCRIBE_TO_SIGNAL( m_dispatcher->m_cmdDispatching, this, hkVdbClient );
    HK_SUBSCRIBE_TO_SIGNAL( m_dispatcher->m_cmdDispatched, this, hkVdbClient );
    HK_SUBSCRIBE_TO_SIGNAL( getCmdHandler<hkVdbSetupHandler>()->m_serverInfoReceived, this, hkVdbClient );
    HK_SUBSCRIBE_TO_SIGNAL( getCmdHandler<hkVdbProcessHandler>()->m_processCmdReceived, this, hkVdbClient );
    HK_SUBSCRIBE_TO_SIGNAL( m_stepCompleted, m_input.val(), hkVdbCmdInput );
    HK_SUBSCRIBE_TO_SIGNAL( getCmdHandler<hkVdbPlaybackHandler>()->m_playbackInfoReceivedInternal, m_output.val(), hkVdbCmdOutput );
}

hkVdbClient::~hkVdbClient()
{
    disconnectAll();

    // Shouldn't need to do this since these signals are owned by hkVdbClient, but it's good practice
    {
        m_connecting.unsubscribeAll( this );
        m_connected.unsubscribeAll( this );
        m_disconnected.unsubscribeAll( this );
        m_dispatcher->m_cmdDispatching.unsubscribeAll( this );
        m_dispatcher->m_cmdDispatched.unsubscribeAll( this );
        getCmdHandler<hkVdbSetupHandler>()->m_serverInfoReceived.unsubscribeAll( this );
        getCmdHandler<hkVdbProcessHandler>()->m_processCmdReceived.unsubscribeAll( this );
        m_connected.unsubscribeAll( m_input );
        m_stepCompleted.unsubscribeAll( m_input );
        m_connected.unsubscribeAll( m_output );
        m_disconnecting.unsubscribeAll( m_output );
        getCmdHandler<hkVdbPlaybackHandler>()->m_playbackInfoReceivedInternal.unsubscribeAll( m_output );
        m_connected.unsubscribeAll( m_persistentStateCache );
    }

    clearCmdHandlers();
}

hkResult hkVdbClient::process( hkVdbProcessInfo processingInfo )
{
    UpdateConnectionStatesHelper updateConnectionStatesHelper( *this );

    const hkVdbFrame* frame;
    const hkVdbCmd* cmd;
    StepState prevState;

    HK_VDB_FAIL_IF_OPERATION_FAILED( readCmd( frame, cmd, prevState ) );

    processingInfo.computeEndTickCount();
    if ( processingInfo.m_processingTime > 0 )
    {
        if ( hkDynCast<hkVdbNetworkConnection>( m_connections[hkVdbConnectionUse::APPLICATION] ) )
        {
            HK_WARN_ONCE(
                0x22441295,
                "Performance Warning: Waiting for more cmds after a completed frame while connected"
                " over a network is not recommended as the server typically has lots"
                " of work to do between steps and you will unnecessarily stall the client" );
        }
    }

    while ( cmd || processingInfo.hasMoreTime() )
    {
        if ( cmd )
        {
            
            
            
            
            HK_VDB_VERIFY_REPORTER_OPERATION(
                m_dispatcher->dispatch( *frame, *cmd, m_input->getProtocol(), &processingInfo ),
                *m_dispatcher,
                hkVdbError::CMD_DISPATCH_ERROR );

            if ( ( m_state == STEP_COMPLETED ) && !processingInfo.hasMoreTime() )
            {
                // Return (updates connection states)
                return updateConnectionStatesHelper;
            }
        }

        HK_VDB_FAIL_IF_OPERATION_FAILED( readCmd( frame, cmd, prevState ) );
    }

    // Always try and flush pending writes
    bool flushed;
    HK_VDB_VERIFY_REPORTER_OPERATION(
        m_output->flush( flushed, false ),
        *m_output,
        hkVdbError::CONNECTION_WRITE_ERROR );

    // Return (updates connection states)
    return updateConnectionStatesHelper;
}

hkResult hkVdbClient::waitForCompletion()
{
    hkVdbSignalResult result;
    for ( int i = 0; i < hkVdbCmdHandlerType::NUM_TYPES; i++ )
    {
        if ( hkVdbAnyCmdHandler* handler = m_handlers[i] )
        {
            if ( handler->waitForCompletion().isFailure() )
            {
                result.signalError( *handler );
            }
        }
    }
    HK_VDB_VERIFY_SIGNAL_RESULT( result );
    return HK_SUCCESS;
}

hkVdbClient::UpdateConnectionStatesHelper::~UpdateConnectionStatesHelper()
{
    if ( !m_returnedResult ) m_client.updateConnectionStates();
}

hkVdbClient::UpdateConnectionStatesHelper::operator hkResult()
{
    m_returnedResult = true;
    return m_client.updateConnectionStates();
}

hkResult hkVdbClient::updateConnectionStates()
{
    bool succeeded = true;
    for ( int i = 0; i < hkVdbConnectionUse::NUM_USES; i++ )
    {
        succeeded &= ( updateConnectionState( hkVdbConnectionUse::Enum( i ) ).isSuccess() );
    }
    return succeeded ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkVdbClient::setConnectionState( hkVdbConnectionUse::Enum use, hkVdbConnectionState::Enum state, bool disconnectConnectionOnDisconnectState )
{
    if ( hkVdbConnection* connection = m_connections[use] )
    {
        hkVdbConnectionState::Enum lastKnownState = m_connectionStates[use];
        if ( lastKnownState != state )
        {
            HK_ASSERT_NO_MSG( 0x22441016, state != hkVdbConnectionState::ERRORED );
            hkVdbSignalResult result;
            switch ( lastKnownState )
            {
                case hkVdbConnectionState::ERRORED:
                case hkVdbConnectionState::DISCONNECTED:
                {
                    // Try to connect
                    if ( connection->connect().isFailure() ) result.signalError( *connection );

                    // Update connection state
                    hkVdbConnectionState::Enum connectionState = connection->getState();

                    if ( ( result.isSuccess() ) &&
                        ( ( connectionState == hkVdbConnectionState::CONNECTING ) || ( connectionState == hkVdbConnectionState::CONNECTED ) ) )
                    {
                        // We always move through connecting state.
                        m_connectionStates[use] = hkVdbConnectionState::CONNECTING;
                        m_connecting.fire( use, *connection, result );

                        // See if user rejected connection and bail.
                        if ( result.isFailure() )
                        {
                            setConnectionState( use, hkVdbConnectionState::DISCONNECTED, disconnectConnectionOnDisconnectState );
                            HK_VDB_VERIFY_SIGNAL_RESULT( result );
                        }

                        // See if we need to signal connected.
                        m_connectionStates[use] = connectionState;
                        if ( m_connectionStates[use] == hkVdbConnectionState::CONNECTED )
                        {
                            if ( use == hkVdbConnectionUse::APPLICATION ) m_cache->clear();
                            m_connected.fire( use, *connection );
                        }
                    }
                    else
                    {
                        HK_ASSERT( 0x22441019, result.isFailure(), "Connection should no longer report disconnected state" );
                    }
                    break;
                }
                case hkVdbConnectionState::CONNECTING:
                case hkVdbConnectionState::CONNECTED:
                {
                    if ( disconnectConnectionOnDisconnectState )
                    {
                        // Hold ref so we can set m_connections[use] to null and still use connection in signal
                        hkRefPtr<hkVdbConnection> connectionRef = connection;

                        // Do disconnection (this cannot be canceled by the user, they can just indicate an error occurred)
                        if ( connection->disconnect().isFailure() ) result.signalError( *connection );

                        // We always move through disconnecting state.
                        m_disconnecting.fire( use, *connection, result );

                        // Clear our connection
                        m_connectionStates[use] = connection->getState();
                        m_connections[use] = HK_NULL;
                        HK_ASSERT(
                            0x22441020,
                            ( m_connectionStates[use] != hkVdbConnectionState::CONNECTING ) && ( m_connectionStates[use] != hkVdbConnectionState::CONNECTED ),
                            "Connection should no longer report connect state" );

                        // Signal we have disconnected if we had reached the connected state before
                        if ( lastKnownState == hkVdbConnectionState::CONNECTED ) m_disconnected.fire( use, *connection );
                    }
                    else
                    {
                        // We are replacing current connection, don't do any signaling or disconnecting
                        m_connectionStates[use] = hkVdbConnectionState::DISCONNECTED;
                        m_connections[use] = HK_NULL;
                    }
                    break;
                }
                default:
                {
                    HK_ASSERT_NOT_IMPLEMENTED( 0x22441018 );
                    break;
                }
            }

            HK_VDB_VERIFY_SIGNAL_RESULT( result );
        }
    }
    return HK_SUCCESS;
}

hkResult hkVdbClient::updateSinkConnection( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol )
{
    if ( hkVdbConnection* sinkConnection = m_connections[hkVdbConnectionUse::SINK] )
    {
        if ( m_connectionStates[hkVdbConnectionUse::SINK] == hkVdbConnectionState::CONNECTED )
        {
            // Wait until we are at a frame boundary,
            // then advance our sink through it's initialization stages.
            if ( cmd.getType() == hkVdbCmdType::STEP )
            {
                switch ( m_sinkInitializationState )
                {
                    case SINK_WAITING_FOR_PREREQS:
                    {
                        if ( !m_sinkInitializationCompletedPreReqs.allAreSet( SINK_ALL_PREREQS ) )
                        {
                            // Still waiting
                            break;
                        }
                        // Fall-through, we have our prereqs.
                    }
                    case SINK_UNINITIALIZED:
                    {
                        hkVdbSetupHandler* setup = getCmdHandler<hkVdbSetupHandler>();
                        HK_VDB_VERIFY_CONDITION( setup, 0xedb00129, hkVdbError::CMD_HANDLER_NOT_REGISTERED );
                        if (// We must be capable of fast "catch-up" recording to use this codepath
                            setup->isCapableOf( hkVdbCapabilities::QUICK_START_RECORDING ) )
                        {
                            hkVdbPlaybackHandler* handler = getCmdHandler<hkVdbPlaybackHandler>();
                            HK_VDB_VERIFY_CONDITION( handler, 0xedb00006, hkVdbError::CMD_HANDLER_NOT_REGISTERED );
                            HK_VDB_VERIFY_REPORTER_OPERATION(
                                handler->saveReplay( *sinkConnection, frame.getFrameNumber(), frame.getFrameNumber() ),
                                *handler,
                                hkVdbError::CMD_HANDLER_WRITE_ERROR );
                            m_sinkInitializationState = SINK_INITIALIZED;
                        }
                        else if ( hkRefPtr<hkVdbNetworkConnection> appConnection = hkDynCast<hkVdbNetworkConnection>( m_connections[hkVdbConnectionUse::APPLICATION] ) )
                        {
                            // Before 2016.1, the cmds were not written in such a way to make persisting state easy.
                            // So we don't support it. In this case, we can't do our "catch-up" recording. Instead,
                            // we must disconnect/reconnect and then start recording :(.
                            hkVdbProcessHandler* handler = getCmdHandler<hkVdbProcessHandler>();
                            HK_VDB_VERIFY_CONDITION( handler, 0xedb00127, hkVdbError::CMD_HANDLER_NOT_REGISTERED );
                            handler->getSelectedProcessNames( m_savedSelectedProcesses );
                            m_sinkInitializationState = SINK_RECONNECTING;
                            if ( setConnection( HK_NULL ).isFailure() ) { disconnect( hkVdbConnectionUse::SINK ); return HK_FAILURE; }
                            if ( setConnection( appConnection ).isFailure() ) { disconnect( hkVdbConnectionUse::SINK ); return HK_FAILURE; }
                            // Note: m_sinkInitializationState = SINK_INITIALIZED handled by onConnectedSignal as it can take some time to connect
                        }
                        else
                        {
                            // This is a non-network connection (so disconnect/reconnect will not get us into a good state).
                            HK_VDB_SIGNAL_ERROR_MSG(
                                0xedb00158,
                                hkVdbError::CMD_HANDLER_WRITE_ERROR,
                                "Recording from a frame other than the start frame isn't supported for this connection" );
                        }
                        break;
                    }
                    default:
                    {
                        HK_ASSERT( 0x22441094, m_sinkInitializationState == SINK_INITIALIZED, "Unknown sink initialization state" );
                        break;
                    }
                }
            }

            // Write to sink if we are ready.
            if ( m_sinkInitializationState == SINK_INITIALIZED )
            {
                // Signal for custom behavior
                const hkVdbCmd* cmdToWrite = &cmd;
                m_writingToSink.fire( *sinkConnection, frame, cmd, m_input->getProtocol(), cmdToWrite );

                if ( cmdToWrite )
                {
                    hkStreamWriter* sinkWriter = sinkConnection->getWriter();
                    HK_VDB_VERIFY_CONDITION_MSG(
                        sinkWriter,
                        0xedb00007,
                        hkVdbError::CONNECTION_WRITE_ERROR,
                        "Sink connection has no writer" );

                    // Ensure that our writer always writes all the bits we need
                    hkVdbBlockingWriter writer( *sinkWriter );
                    HK_ON_DEBUG( int written = ) writer.write( cmdToWrite, cmdToWrite->getCmdSize() );
                    HK_ASSERT_NO_MSG( 0x22440916, hkUint32( written ) == cmdToWrite->getCmdSize() );
                    HK_VDB_VERIFY_REPORTER( *sinkConnection );
                }
            }
        }
        else
        {
            // Still waiting for connection or we dropped our connection.
            // Check for the later.
            HK_VDB_VERIFY_REPORTER( *sinkConnection );
            HK_VDB_VERIFY_CONDITION_MSG(
                ( m_sinkInitializationState != SINK_INITIALIZED ),
                0xedb00008,
                hkVdbError::CONNECTION_ERROR,
                "Sink connection has dropped" );
        }
    }
    return HK_SUCCESS;
}

hkResult hkVdbClient::updateConnectionState( hkVdbConnectionUse::Enum use )
{
    if ( hkVdbConnection* connection = m_connections[use] )
    {
        hkVdbConnectionState::Enum lastKnownState = m_connectionStates[use];
        hkVdbConnectionState::Enum state = connection->getState();
        if ( lastKnownState != state )
        {
            hkVdbSignalResult result;
            m_connectionStates[use] = state;
            switch ( lastKnownState )
            {
                case hkVdbConnectionState::CONNECTING:
                {
                    if ( HK_VERY_LIKELY( state == hkVdbConnectionState::CONNECTED ) )
                    {
                        
                        
                        if ( use == hkVdbConnectionUse::APPLICATION ) m_cache->clear();
                        m_connected.fire( use, *connection );
                        break;
                    }

                    // Fall through to cleanup connection
                }
                case hkVdbConnectionState::CONNECTED:
                {
                    if ( HK_VERY_LIKELY( state == hkVdbConnectionState::CONNECTING ) )
                    {
                        // No-op, hopefully will reestablish connection later
                    }
                    else if ( connection->getError() )
                    {
                        result.signalError( *connection );
                    }
                    else
                    {
                        HK_VDB_SIGNAL_ERROR_MSG( 0xedb00096, hkVdbError::INVALID_CONNECTION, "The connection has been dropped" );
                        result.signalError( *this );
                    }
                    break;
                }
            }

            if ( HK_VERY_UNLIKELY( result.isFailure() ) )
            {
                // Hold ref so we can set m_connections[use] to null and still use connection in signal
                hkRefPtr<hkVdbConnection> connectionRef = connection;
                if ( ( lastKnownState == hkVdbConnectionState::CONNECTING ) || ( lastKnownState == hkVdbConnectionState::CONNECTED ) ) m_disconnecting.fire( use, *connection, result );
                if ( connection->disconnect().isFailure() ) result.signalError( *connection );
                m_connectionStates[use] = connection->getState();
                m_connections[use] = HK_NULL;
                HK_ASSERT(
                    0x22441021,
                    ( m_connectionStates[use] != hkVdbConnectionState::CONNECTING ) && ( m_connectionStates[use] != hkVdbConnectionState::CONNECTED ),
                    "Connection should no longer report connect state" );
                if ( lastKnownState == hkVdbConnectionState::CONNECTED ) m_disconnected.fire( use, *connection );
            }

            HK_VDB_VERIFY_SIGNAL_RESULT( result );
        }
    }
    return HK_SUCCESS;
}

hkResult hkVdbClient::readCmd( const hkVdbFrame*& frameOut, const hkVdbCmd*& cmdOut, StepState& prevStepStateOut )
{
    HK_VDB_FAIL_IF_OPERATION_FAILED( updateConnectionStates() );

    frameOut = HK_NULL;
    cmdOut = HK_NULL;

    hkVdbPlaybackHandler* handler = getCmdHandler<hkVdbPlaybackHandler>();
    HK_VDB_VERIFY_CONDITION( handler, 0xedb00009, hkVdbError::CMD_HANDLER_NOT_REGISTERED );

    // Get next from our connection if we have it and it's ready
    hkVdbConnection* appConnection = m_connections[hkVdbConnectionUse::APPLICATION];
    if ( appConnection && ( m_connectionStates[hkVdbConnectionUse::APPLICATION] == hkVdbConnectionState::CONNECTED ) )
    {
        HK_VDB_VERIFY_REPORTER_OPERATION( handler->getNextCmd( appConnection, frameOut, cmdOut ), *handler, hkVdbError::CMD_BUFFER_ERROR );
    }
    // Otherwise, just let the playback handler return the next from the buffer
    else
    {
        HK_VDB_VERIFY_REPORTER_OPERATION( handler->getNextCmd( HK_NULL, frameOut, cmdOut ), *handler, hkVdbError::CMD_BUFFER_ERROR );
    }

    if ( cmdOut )
    {
        prevStepStateOut = m_state;
        if ( prevStepStateOut != STEP_STARTED )
        {
            m_stepStarted.fire( *frameOut, *cmdOut );
        }
        m_state = STEP_STARTED;
    }

    return HK_SUCCESS;
}

hkResult hkVdbClient::connectMachine(
    const char* machineName,
    hkUint16 port,
    hkReal timeoutMs,
    hkVdbConnectionUse::Enum use,
    bool disconnectCurrentConnection )
{
    
    
    if ( use == hkVdbConnectionUse::APPLICATION_SERVER_DISCOVERY )
    {
        HK_VDB_SIGNAL_ERROR_MSG( 0xf04ef3e7, hkVdbError::INVALID_ARGUMENTS, "APPLICATION_SERVER_DISCOVERY is not supported by this connection type" );
        return HK_FAILURE;
    }

    if ( setConnectionState( use, hkVdbConnectionState::DISCONNECTED, disconnectCurrentConnection ).isFailure() ) return HK_FAILURE;
    m_connections[use] = hkRefNew<hkVdbConnection>( new hkVdbNetworkConnection( machineName, port, timeoutMs ) );
#ifdef USE_NO_OP_CONNECTION
    m_connections[use] = hkRefNew<hkVdbNoOpConnection>( new hkVdbNoOpConnection( m_connections[use] ) );
#endif
    if ( setConnectionState( use, hkVdbConnectionState::CONNECTING, false ).isFailure() ) return HK_FAILURE;
    return HK_SUCCESS;
}

hkResult hkVdbClient::connectPort(
    hkUint16 port,
    hkReal timeoutMs,
    hkVdbConnectionUse::Enum use,
    bool disconnectCurrentConnection )
{
    if ( setConnectionState( use, hkVdbConnectionState::DISCONNECTED, disconnectCurrentConnection ).isFailure() ) return HK_FAILURE;
    m_connections[use] = hkRefNew<hkVdbConnection>(
        ( use == hkVdbConnectionUse::APPLICATION_SERVER_DISCOVERY ) ?
        new hkVdbServerDiscoveryConnection( port, HK_VDB_ADVERTISE_SERVER_DEFAULT_GROUP_ID, timeoutMs ) :
        ( hkVdbConnection* ) new hkVdbNetworkConnection( port ) );
#ifdef USE_NO_OP_CONNECTION
    m_connections[use] = hkRefNew<hkVdbNoOpConnection>( new hkVdbNoOpConnection( m_connections[use] ) );
#endif
    if ( setConnectionState( use, hkVdbConnectionState::CONNECTING, false ).isFailure() ) return HK_FAILURE;
    return HK_SUCCESS;
}

hkResult hkVdbClient::connectFile( const char* filepath, hkVdbConnectionUse::Enum use, bool disconnectCurrentConnection )
{
    if ( ( use != hkVdbConnectionUse::APPLICATION ) && ( use != hkVdbConnectionUse::SINK ) )
    {
        HK_VDB_SIGNAL_ERROR_MSG( 0xf4ef3fe7, hkVdbError::INVALID_ARGUMENTS, "Only APPLICATION and SINK connection types are supported for files" );
        return HK_FAILURE;
    }
    if ( setConnectionState( use, hkVdbConnectionState::DISCONNECTED, disconnectCurrentConnection ).isFailure() ) return HK_FAILURE;
    m_connections[use] = hkRefNew<hkVdbConnection>( new hkVdbFileConnection( filepath, ( use == hkVdbConnectionUse::APPLICATION ) ? hkFileSystem::ACCESS_READ : hkFileSystem::ACCESS_WRITE ) );
#ifdef USE_NO_OP_CONNECTION
    m_connections[use] = hkRefNew<hkVdbNoOpConnection>( new hkVdbNoOpConnection( m_connections[use] ) );
#endif
    if ( setConnectionState( use, hkVdbConnectionState::CONNECTING, false ).isFailure() ) return HK_FAILURE;
    return HK_SUCCESS;
}

hkResult hkVdbClient::connectMemory( hkMemoryTrack* memory, hkVdbConnectionUse::Enum use, bool disconnectCurrentConnection )
{
    if ( use == hkVdbConnectionUse::APPLICATION_SERVER_DISCOVERY )
    {
        HK_VDB_SIGNAL_ERROR_MSG( 0xf04ef3e7, hkVdbError::INVALID_ARGUMENTS, "APPLICATION_SERVER_DISCOVERY is not supported by this connection type" );
        return HK_FAILURE;
    }

    if ( setConnectionState( use, hkVdbConnectionState::DISCONNECTED, disconnectCurrentConnection ).isFailure() ) return HK_FAILURE;
    m_connections[use] = hkRefNew<hkVdbConnection>( new hkVdbMemoryConnection( memory ) );
#ifdef USE_NO_OP_CONNECTION
    m_connections[use] = hkRefNew<hkVdbNoOpConnection>( new hkVdbNoOpConnection( m_connections[use] ) );
#endif
    if ( setConnectionState( use, hkVdbConnectionState::CONNECTING, false ).isFailure() ) return HK_FAILURE;
    return HK_SUCCESS;
}

hkResult hkVdbClient::connectNoOp( hkVdbConnectionUse::Enum use, bool disconnectCurrentConnection )
{
    if (use == hkVdbConnectionUse::APPLICATION_SERVER_DISCOVERY)
    {
        HK_VDB_SIGNAL_ERROR_MSG(0xf4ef3fe7, hkVdbError::INVALID_ARGUMENTS, "APPLICATION_SERVER_DISCOVERY is not supported by this connection type");
        return HK_FAILURE;
    }

    if ( setConnectionState( use, hkVdbConnectionState::DISCONNECTED, disconnectCurrentConnection ).isFailure() ) return HK_FAILURE;
    m_connections[use] = hkRefNew<hkVdbConnection>( new hkVdbNoOpConnection() );
    if ( setConnectionState( use, hkVdbConnectionState::CONNECTING, false ).isFailure() ) return HK_FAILURE;
    return HK_SUCCESS;
}

hkResult hkVdbClient::disconnect( hkVdbConnectionUse::Enum use )
{
    return setConnectionState( use, hkVdbConnectionState::DISCONNECTED, true );
}

hkResult hkVdbClient::disconnectAll()
{
    bool succeeded = true;
    for ( int i = 0; i < hkVdbConnectionUse::NUM_USES; i++ )
    {
        succeeded &= ( disconnect( hkVdbConnectionUse::Enum( i ) ).isSuccess() );
    }
    return succeeded ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkVdbClient::poll()
{
    hkVdbSignalResult signal;
    for ( int i = 0; i < hkVdbConnectionUse::NUM_USES; i++ )
    {
        if ( hkVdbConnection* connection = m_connections[i] )
        {
            if ( connection->poll().isFailure() )
            {
                signal.signalError( *connection );
            }
        }
    }
    HK_VDB_VERIFY_SIGNAL_RESULT( signal );
    return HK_SUCCESS;
}

hkResult hkVdbClient::setConnection( hkVdbConnection* connection, hkVdbConnectionUse::Enum use, bool disconnectCurrentConnection )
{
    if ( setConnectionState( use, hkVdbConnectionState::DISCONNECTED, disconnectCurrentConnection ).isFailure() ) return HK_FAILURE;
    m_connections[use] = connection;
#ifdef USE_NO_OP_CONNECTION
    m_connections[use] = hkRefNew<hkVdbNoOpConnection>( new hkVdbNoOpConnection( m_connections[use] ) );
#endif
    if ( setConnectionState( use, hkVdbConnectionState::CONNECTING, false ).isFailure() ) return HK_FAILURE;
    return HK_SUCCESS;
}

hkResult hkVdbClient::clearCmdHandlers()
{
    hkVdbSignalResult result;
    for ( int i = 0; i < HK_COUNT_OF( m_handlers ); i++ )
    {
        if ( hkVdbAnyCmdHandler* handler = m_handlers[i] )
        {
            if ( handler->unregisterSelf( *this ).isFailure() )
            {
                result.signalError( *handler );
            }

            m_connected.unsubscribeAll( handler );
        }
    }
    HK_VDB_VERIFY_SIGNAL_RESULT( result );
    return HK_SUCCESS;
}

hkResult hkVdbClient::setCmdHandlers()
{
    hkRefPtr<hkVdbSetupHandler> setupHandler = hkRefNew<hkVdbSetupHandler>( new hkVdbSetupHandler() );
    hkRefPtr<hkVdbProcessHandler> processHandler = hkRefNew<hkVdbProcessHandler>( new hkVdbProcessHandler() );
    hkRefPtr<hkVdbPlaybackHandler> playbackHandler = hkRefNew<hkVdbPlaybackHandler>( new hkVdbPlaybackHandler() );
    hkRefPtr<hkVdbDisplayHandler> displayHandler = hkRefNew<hkVdbDisplayHandler>( new hkVdbDisplayHandler() );
    hkRefPtr<hkVdbObjectHandler> objectHandler = hkRefNew<hkVdbObjectHandler>( new hkVdbObjectHandler() );
    hkRefPtr<hkVdbStatsHandler> statsHandler = hkRefNew<hkVdbStatsHandler>( new hkVdbStatsHandler() );
    hkRefPtr<hkVdbTextHandler> textHandler = hkRefNew<hkVdbTextHandler>( new hkVdbTextHandler() );
    hkRefPtr<hkVdbFileHandler> fileHandler = hkRefNew<hkVdbFileHandler>( new hkVdbFileHandler() );
    hkRefPtr<hkVdbCustomHandler> customHandler = hkRefNew<hkVdbCustomHandler>( new hkVdbCustomHandler() );
    hkRefPtr<hkVdbInternalHandler> internalHandler = hkRefNew<hkVdbInternalHandler>( new hkVdbInternalHandler() );

    hkVdbSignalResult result;

    // First, set handlers to null to force an unregister.
    if ( clearCmdHandlers().isFailure() ) result.signalError( 0xedb00012, hkVdbError::CMD_HANDLER_REGISTRATION_ERROR );

    // Second, manually set all handlers (this way they are all available to each other during registration.
    m_handlers[hkVdbSetupHandler::getStaticType()] = reinterpret_cast< hkVdbAnyCmdHandler*>( setupHandler.val() );
    m_handlers[hkVdbProcessHandler::getStaticType()] = reinterpret_cast< hkVdbAnyCmdHandler*>( processHandler.val() );
    m_handlers[hkVdbPlaybackHandler::getStaticType()] = reinterpret_cast< hkVdbAnyCmdHandler*>( playbackHandler.val() );
    m_handlers[hkVdbDisplayHandler::getStaticType()] = reinterpret_cast< hkVdbAnyCmdHandler*>( displayHandler.val() );
    m_handlers[hkVdbObjectHandler::getStaticType()] = reinterpret_cast< hkVdbAnyCmdHandler*>( objectHandler.val() );
    m_handlers[hkVdbStatsHandler::getStaticType()] = reinterpret_cast< hkVdbAnyCmdHandler*>( statsHandler.val() );
    m_handlers[hkVdbTextHandler::getStaticType()] = reinterpret_cast< hkVdbAnyCmdHandler*>( textHandler.val() );
    m_handlers[hkVdbFileHandler::getStaticType()] = reinterpret_cast< hkVdbAnyCmdHandler*>( fileHandler.val() );
    m_handlers[hkVdbCustomHandler::getStaticType()] = reinterpret_cast< hkVdbAnyCmdHandler*>( customHandler.val() );
    m_handlers[hkVdbInternalHandler::getStaticType()] = reinterpret_cast< hkVdbAnyCmdHandler*>( internalHandler.val() );

    // Lastly, manually register all handlers
    for ( int i = 0; i < HK_COUNT_OF( m_handlers ); i++ )
    {
        if ( m_handlers[i]->registerSelf( *this ).isFailure() ) result.signalError( *m_handlers[i] );
    }
    // Signals last to register to ensure these signal calls happen before any others
    
    for ( int i = 0; i < HK_COUNT_OF( m_handlers ); i++ )
    {
        m_connected.implicitSubscribe( m_handlers[i].val(), m_handlers[i]->getClassName() );
    }

    HK_VDB_VERIFY_SIGNAL_RESULT( result );
    return HK_SUCCESS;
}

void hkVdbClient::onCmdDispatchingSignal( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbSignalResult& result )
{
    m_sinkInitializationCompletedPreReqs.orWith( ( frame.getFrameNumber() >= 1 ) * SINK_PREREQ_2_FRAMES );
    HK_VDB_VERIFY_SIGNALLED_REPORTER_OPERATION(
        updateSinkConnection( frame, cmd, protocol ),
        *this,
        result );
}

void hkVdbClient::onCmdDispatchedSignal( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbSignalResult& result )
{
    if ( cmd.getType() == hkVdbCmdType::STEP )
    {
        m_state = STEP_COMPLETED;

        HK_VDB_VERIFY_SIGNALLED_REPORTER_OPERATION(
            m_output->ackFrame( frame ),
            *m_output,
            result );

        m_stepCompleted.fire( frame, cmd );
    }
    else
    {
        m_state = STEP_STARTED;
    }
}

void hkVdbClient::onConnectingSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection, hkVdbSignalResult& result )
{
    // Reset app-connection-based state
    if ( use == hkVdbConnectionUse::APPLICATION )
    {
        m_state = STEP_NOT_STARTED;
    }
}

void hkVdbClient::onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection )
{
    // Reset app-connection-based state
    if ( use == hkVdbConnectionUse::APPLICATION )
    {
        m_state = STEP_NOT_STARTED;
        hkVdbFileConnection* fileConnection = hkDynCast( &connection );
        m_input->reset( ( fileConnection ) ? 0 : HK_VDB_DEFAULT_INPUT_BUFFER_DURATION_MS );
        if ( m_sinkInitializationState == SINK_RECONNECTING )
        {
            m_sinkInitializationState = SINK_INITIALIZED;
        }
        else
        {
            m_sinkInitializationCompletedPreReqs = SINK_PREREQ_NONE;
        }
    }
    // Mark sink as needing initialization
    else if ( use == hkVdbConnectionUse::SINK )
    {
        m_sinkInitializationState = SINK_WAITING_FOR_PREREQS;
        m_savedSelectedProcesses.clear();
    }
}

void hkVdbClient::onDisconnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection )
{
    // Disconnect SINK if we've disconnected the application
    if ( ( use == hkVdbConnectionUse::APPLICATION ) && ( m_sinkInitializationState != SINK_RECONNECTING ) )
    {
        disconnect( hkVdbConnectionUse::SINK );
    }
}

void hkVdbClient::onServerInfoReceivedSignal( const hkVdbSetupHandler::ClientInfo&, const hkVdbSetupHandler::ServerInfo&, hkVdbCapabilities, hkVdbSignalResult& )
{
    m_sinkInitializationCompletedPreReqs.orWith( SINK_PREREQ_SETUP_INFO );
}

void hkVdbClient::onProcessCmdReceivedSignal( const hkVdbProcessHandler::ProcessCmd& processCmd, hkVdbSignalResult& result )
{
    if ( m_savedSelectedProcesses.getSize() )
    {
        if ( hkVdbProcessHandler* processHandler = getCmdHandler<hkVdbProcessHandler>() )
        {
            if ( const hkVdbProcessHandler::ProcessRegisteredCmd* registeredCmd = processCmd.asRegisteredCmd() )
            {
                if ( m_savedSelectedProcesses.indexOf( registeredCmd->m_name ) != -1 )
                {
                    processHandler->setProcessSelected( registeredCmd->m_tag, true );
                }
            }
            else if ( const hkVdbProcessHandler::ProcessSelectionChangedCmd* selectionChangedCmd = processCmd.asSelectionChangedCmd() )
            {
                if ( selectionChangedCmd->isSelected() )
                {
                    const char* name = processHandler->getProcessName( selectionChangedCmd->m_tag );
                    int idx = m_savedSelectedProcesses.indexOf( name );
                    if ( idx != -1 )
                    {
                        m_savedSelectedProcesses.removeAt( idx );
                    }
                }
            }
        }
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
