// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Types/hkSignalSlots.h>
#include <Common/Base/Container/BitField/hkBitField.h>
#include <Common/Base/System/Stopwatch/hkWindowedAverage.h>

class hkCriticalSection;


//////////////////////////////////////////////////////////////////////////
/// Struct describing work that is being reported through the
/// hkVdbProgressReporter system.
//////////////////////////////////////////////////////////////////////////
struct hkVdbProgressWorkEstimate
{
    HK_DECLARE_CLASS( hkVdbProgressWorkEstimate, New );
    HK_INLINE hkVdbProgressWorkEstimate();
    HK_INLINE void operator+=( const hkVdbProgressWorkEstimate& other );
    HK_INLINE hkVdbProgressWorkEstimate operator+( const hkVdbProgressWorkEstimate& other ) const;
    HK_INLINE void operator-=( const hkVdbProgressWorkEstimate& other );
    HK_INLINE hkVdbProgressWorkEstimate operator-( const hkVdbProgressWorkEstimate& other ) const;

    /// The number of completed items.
    /// When a work load completes, it's completed items are removed from the estimate.
    hkUint32 m_completedWorkItems;

    /// The number of estimated items.
    /// This is used to estimate the time remaining.
    hkUint32 m_estWorkItems;

    /// The estimated time remaining.
    /// If this is non-zero, we have active work.
    /// If the value is negative, then estimations cannot be made but we are still doing work.
    hkReal m_estTimeRemaining;

    /// Uninterruptible work items cannot be canceled.
    /// For combined work estimates, this is the number of uninterruptible sub work estimates.
    hkBool32 m_uninterruptible;
};


//////////////////////////////////////////////////////////////////////////
/// Defined an interface for reporting on work progress.
//////////////////////////////////////////////////////////////////////////
class hkVdbProgressReporter : public hkReferencedObject
{
public:

    //
    // Reporting
    //

    
    /// Returns the number of active work loads.
    HK_INLINE_VIRTUAL virtual hkUint32 getWorkLoadCount() const { hkArray<hkUint64> ids; getWorkLoads( ids ); return hkUint32( ids.getSize() ); }

    /// Returns the number of active work loads that are uninterruptible.
    HK_INLINE_VIRTUAL virtual hkUint32 getUninterruptibleWorkLoadCount() const { return getTotalEstimate().m_uninterruptible; }

    /// Returns a total estimate for all active work loads.
    HK_INLINE_VIRTUAL virtual hkVdbProgressWorkEstimate getTotalEstimate() const;

    /// Get active work load ids.
    virtual hkUint32 getWorkLoads( hkArray<hkUint64>& idsOut ) const = 0;

    /// Get an estimate of work for the provided work load id.
    virtual hkVdbProgressWorkEstimate getEstimate( hkUint64 id ) const = 0;

    /// Get the current message for the provided work load id.
    virtual const char* getWorkMessage( hkUint64 id ) const = 0;

    /// Cancel the work load associated with the provided id.
    /// (The work load should respond by ceasing at it's earliest stopping point.)
    virtual hkBool32 cancelWorkLoad( hkUint64 id ) = 0;

    /// Call cancelWorkLoad(...) for all active work load ids.
    HK_INLINE_VIRTUAL virtual hkBool32 cancelAllWorkLoads();
    

    //
    // Work Registration
    //

    /// Notify the progress reporter that you are starting work.
    /// Optionally provide initial work estimates and specify interruptibility.
    /// Returns a unique id associated with the started work load.
    virtual hkUint64 startWorkLoad( hkUint32 estWorkItems = 0, hkReal estTime = 0, hkBool32 canCancel = true ) = 0;

    /// Set a work message for the specified work load id.
    virtual void setWorkMessage( hkUint64 id, const char* msg ) = 0;

    /// Notify the progress reporter that you have done some work for the provided work load id.
    virtual void didWorkFor( hkUint64 id ) = 0;

    /// Signal to the progress reporter that a warning has occurred while executing work.
    /// The default progress reporter reports this warning to its hkLog.
    virtual void signalWarningFor( hkUint64 id, hkVdbSignalResult& warningResult ) = 0;

    /// Signal to the progress reporter that an error has occurred while executing work.
    /// The default progress reporter reports this error to its hkLog.
    /// workTillCompleted will also terminate the work and return HK_FAILURE.
    virtual void signalErrorFor( hkUint64 id, hkVdbSignalResult& errorResult ) = 0;

    /// Notify the progress reporter that you have ended work associated with the provided id.
    virtual void endWorkLoad( hkUint64 id ) = 0;

    /// Returns true if work should continue for the provided id.
    /// This should be called at regular intervals and work should cease if
    /// the work is interruptible and this returns false.
    virtual hkBool32 shouldContinueWorkFor( hkUint64 id ) const = 0;

    /// Helper for executing work until completed in a way that updates the progress reporter.
    /// This is the recommended way for reporting work.
    /// Use:
    ///  hkResult result = reporter->workTillCompleted(
    ///      // Initial work message
    ///         workMsgStr,
    ///      // Work function with optional lamda captures
    ///         [/*captures*/]( hkVdbSignalResult& warningResult, hkVdbSignalResult& errorResult, hkStringBuf& workMsgOut )
    ///             { /*returns true while working, false when done; errorResult.isFailure() will also discontinue work*/ }
    ///      // Optional args for startWorkLoad(...)
    ///         /*, estWorkItems, estTime, canCancel*/);
    ///  /*optionally error on result.isFailure()*/
    template<typename Func, typename ... Args>
    hkResult workTillCompleted(
        const char* workMessage,
        Func func,
        Args ... rest )
    {
        hkUint64 id = startWorkLoad( rest... );
        setWorkMessage( id, workMessage );
        hkBool32 working = true;
        hkVdbSignalResult warningSignal;
        hkVdbSignalResult errorSignal;
        hkStringBuf updatedWorkMsg;
        while ( working && shouldContinueWorkFor( id ) )
        {
            working = func( warningSignal, errorSignal, updatedWorkMsg );

            if ( warningSignal.isFailure() )
            {
                signalWarningFor( id, warningSignal );
            }

            if ( errorSignal.isFailure() )
            {
                signalErrorFor( id, errorSignal );
                return HK_FAILURE;
            }

            if ( updatedWorkMsg.getLength() )
            {
                setWorkMessage( id, updatedWorkMsg );
                updatedWorkMsg.clear();
            }

            didWorkFor( id );
        }

        endWorkLoad( id );
        return HK_SUCCESS;
    }

    //
    // Signals
    //

    HK_DECLARE_SIGNAL( EstUpdatedSignal, hkSignal2< const hkVdbProgressReporter&, hkUint64 > );
    /// Called when the progress reporter has updated it's estimate for a particular work load id.
    EstUpdatedSignal m_estUpdated;
};

class hkVdbDefaultProgressReporter : public hkVdbProgressReporter
{
public:
    HK_DECLARE_CLASS( hkVdbDefaultProgressReporter, New );
    hkVdbDefaultProgressReporter( int timeEstWindow = HK_VDB_PROGRESS_TIME_EST_AVERAGING_WINDOW );
    ~hkVdbDefaultProgressReporter();

    //
    // hkVdbProgressReporter interface
    //

    HK_INLINE_VIRTUAL virtual hkUint32 getWorkLoadCount() const HK_OVERRIDE;
    HK_INLINE_VIRTUAL virtual hkUint32 getUninterruptibleWorkLoadCount() const HK_OVERRIDE;
    virtual hkVdbProgressWorkEstimate getTotalEstimate() const HK_OVERRIDE;
    virtual hkUint32 getWorkLoads( hkArray<hkUint64>& idsOut ) const HK_OVERRIDE;
    virtual hkVdbProgressWorkEstimate getEstimate( hkUint64 id ) const HK_OVERRIDE;
    virtual const char* getWorkMessage( hkUint64 id ) const HK_OVERRIDE;
    virtual hkBool32 cancelWorkLoad( hkUint64 id ) HK_OVERRIDE;
    virtual hkBool32 cancelAllWorkLoads() HK_OVERRIDE;
    virtual hkUint64 startWorkLoad( hkUint32 estWorkItems = 0, hkReal estTime = 0, hkBool32 canCancel = true ) HK_OVERRIDE;
    virtual void setWorkMessage( hkUint64 id, const char* msg ) HK_OVERRIDE;
    virtual void didWorkFor( hkUint64 id ) HK_OVERRIDE;
    virtual void signalWarningFor( hkUint64 id, hkVdbSignalResult& warningResult ) HK_OVERRIDE;
    virtual void signalErrorFor( hkUint64 id, hkVdbSignalResult& errorResult ) HK_OVERRIDE;
    virtual void endWorkLoad( hkUint64 id ) HK_OVERRIDE;
    virtual hkBool32 shouldContinueWorkFor( hkUint64 id ) const HK_OVERRIDE;

protected:

    struct TickEstimate : public hkVdbProgressWorkEstimate
    {
        HK_DECLARE_CLASS( TickEstimate, New );
        TickEstimate() : m_estTicksRemaining( 0 ), m_unEstWorkDatas( 0 ), m_estWorkDatas( 0 ) {}
        hkUint64 m_estTicksRemaining;
        hkUint32 m_unEstWorkDatas;
        hkUint32 m_estWorkDatas;
    };

    struct WorkData : private hkWindowedAverage
    {
        HK_DECLARE_CLASS( WorkData, New );
        void init( int sampleWindow, hkUint32 estWorkItems, hkReal estTime, hkBool32 canCancel );
        bool estimatable() const { return ( m_estimate.m_estWorkItems > m_estimate.m_completedWorkItems ) || ( m_initialEstTicksRemaining > ( m_lastTick - m_startTick ) ); }
        void removeContribution( TickEstimate& globalEstimate ) const;
        void addContribution( TickEstimate& globalEstimate ) const;
        void pushNewTick( hkUint64 ticks );
        TickEstimate m_estimate;
        hkUint64 m_startTick;
        hkUint64 m_lastTick;
        hkUint64 m_initialEstAvgTick;
        hkUint64 m_weightedInitialEstAvgTick;
        hkUint64 m_initialEstTicksRemaining;
        hkBool32 m_pendingCancel;
        hkReal m_rollingAvgWeight;
        hkStringPtr m_msg;
    };

    const WorkData& getWorkData( hkUint64 id ) const;

    mutable hkCriticalSection* m_lock;
    int m_timeEstWindow;
    mutable TickEstimate m_estimate;
    hkArray<WorkData> m_workDatas;
    hkUint32 m_uninterruptibleWorkLoadCount;
    hkBitField m_workDataIndicesPool;
};

#include <VisualDebugger/VdbServices/System/hkVdbProgressReporter.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
