// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbServices/hkVdbServices.h>
#include <VisualDebugger/VdbServices/System/hkVdbProgressReporter.h>

#include <Common/Base/Thread/CriticalSection/hkCriticalSection.h>
#include <Common/Base/System/Stopwatch/hkSystemClock.h>

// Time weight should be between 0-1 (see didWorkFor())
HK_COMPILE_TIME_ASSERT( HK_VDB_PROGRESS_TIME_EST_UPDATE_WEIGHTING >= 0 && HK_VDB_PROGRESS_TIME_EST_UPDATE_WEIGHTING <= 1.0f );

#define DEBUG_LOG_IDENTIFIER "vdb.ProgressReporter"
#include <Common/Base/System/Log/hkLog.hxx>

namespace
{
    void reportResults( hkLog::Level::Enum level, hkVdbSignalResult& result )
    {
        if ( result.getReporter() )
        {
            result.getReporter()->reportError( &DEBUG_LOG_OBJECT );
        }
        else
        {
            hkLog_Log(
                DEBUG_LOG_OBJECT, level, "VDB error [ log={0} type={1} id={2:x} ]",
                DEBUG_LOG_OBJECT.getName(), int( result.getError() ), result.getId() ).
                    setId( result.getId() );
        }
    }
}

hkVdbDefaultProgressReporter::hkVdbDefaultProgressReporter( int timeEstWindow ) :
    m_lock( new hkCriticalSection() ),
    m_timeEstWindow( timeEstWindow ),
    m_uninterruptibleWorkLoadCount( 0 )
{
    HK_ASSERT( 0x22441113, timeEstWindow >= 0, "Only positive estimate windows allowed" );
    m_estimate.m_estTicksRemaining = 0;
}

hkVdbDefaultProgressReporter::~hkVdbDefaultProgressReporter()
{
    delete m_lock;
}

hkUint32 hkVdbDefaultProgressReporter::getWorkLoadCount() const
{
    hkCriticalSectionLock lock( m_lock );
    return ( m_estimate.m_unEstWorkDatas + m_estimate.m_estWorkDatas );
}

hkUint32 hkVdbDefaultProgressReporter::getUninterruptibleWorkLoadCount() const
{
    hkCriticalSectionLock lock( m_lock );
    return m_uninterruptibleWorkLoadCount;
}

hkVdbProgressWorkEstimate hkVdbDefaultProgressReporter::getTotalEstimate() const
{
    hkCriticalSectionLock lock( m_lock );
    m_estimate.m_estTimeRemaining = hkSystemClock::secondsFromTicks( m_estimate.m_estTicksRemaining );
    // Make this negative if it's not a valid estimate
    if ( m_estimate.m_unEstWorkDatas ) m_estimate.m_estTimeRemaining = ( m_estimate.m_estTimeRemaining != 0 ) ? -m_estimate.m_estTimeRemaining : -1;
    return m_estimate;
}

hkUint32 hkVdbDefaultProgressReporter::getWorkLoads( hkArray<hkUint64>& idsOut ) const
{
    hkCriticalSectionLock lock( m_lock );
    hkUint32 prevCount = idsOut.getSize();
    idsOut.reserve( idsOut.getSize() + m_workDatas.getSize() );
    for ( int i = 0; i < m_workDatas.getSize(); i++ )
    {
        if ( !m_workDataIndicesPool.get( i ) )
        {
            idsOut.pushBack( i );
        }
    }
    return ( idsOut.getSize() - prevCount );
}

hkVdbProgressWorkEstimate hkVdbDefaultProgressReporter::getEstimate( hkUint64 id ) const
{
    hkCriticalSectionLock lock( m_lock );
    WorkData& data = const_cast<WorkData&>( getWorkData( id ) );
    data.m_estimate.m_estTimeRemaining = hkSystemClock::secondsFromTicks( data.m_estimate.m_estTicksRemaining );
    // Make this negative if it's not a valid estimate
    if ( !data.estimatable() ) data.m_estimate.m_estTimeRemaining = ( data.m_estimate.m_estTimeRemaining != 0 ) ? -data.m_estimate.m_estTimeRemaining : -1;
    return data.m_estimate;
}

const char* hkVdbDefaultProgressReporter::getWorkMessage( hkUint64 id ) const
{
    hkCriticalSectionLock lock( m_lock );
    WorkData& data = const_cast< WorkData& >( getWorkData( id ) );
    return data.m_msg;
}

hkBool32 hkVdbDefaultProgressReporter::cancelWorkLoad( hkUint64 id )
{
    hkCriticalSectionLock lock( m_lock );
    WorkData& data = const_cast< WorkData& >( getWorkData( id ) );
    if ( !data.m_estimate.m_uninterruptible )
    {
        data.m_pendingCancel = 1; // (use 1 so we can inc. inside didWork to assert on extra work being done after cancel)
        return true;
    }
    else
    {
        return false;
    }
}

hkBool32 hkVdbDefaultProgressReporter::cancelAllWorkLoads()
{
    hkCriticalSectionLock lock( m_lock );
    return hkVdbProgressReporter::cancelAllWorkLoads();
}

hkUint64 hkVdbDefaultProgressReporter::startWorkLoad( hkUint32 estWorkItems, hkReal estTime, hkBool32 canCancel )
{
    hkCriticalSectionLock lock( m_lock );

    HK_ASSERT( 0x22441112, estTime >= 0, "Only positive estimates allowed" );

    
    

    int idx = m_workDatas.getSize();

    // Check our index pool for an available index slot
    if ( m_workDataIndicesPool.getSize() )
    {
        const int availableIdx = m_workDataIndicesPool.findFirstSet();
        if ( availableIdx != -1 )
        {
            m_workDataIndicesPool.clear( availableIdx );
            idx = availableIdx;
        }
        else
        {
            m_workDataIndicesPool.clearStorage();
        }
    }

    // Get our work data from the correct place in the array
    WorkData* data;
    if ( idx < m_workDatas.getSize() )
    {
        HK_ASSERT( 0x22440823, m_workDatas[idx].m_lastTick == 0, "Slot is occupied" );
        data = &m_workDatas[idx];
    }
    else
    {
        data = &m_workDatas.expandOne();
        m_workDataIndicesPool.setSizeAndFill( 0, m_workDatas.getSize(), 0 );
    }

    // Set data
    {
        data->init( m_timeEstWindow, estWorkItems, estTime, canCancel );
        data->addContribution( m_estimate );
        m_uninterruptibleWorkLoadCount += bool( data->m_estimate.m_uninterruptible );
    }

    return idx;
}

void hkVdbDefaultProgressReporter::setWorkMessage( hkUint64 id, const char* msg )
{
    hkCriticalSectionLock lock( m_lock );
    WorkData& data = const_cast< WorkData& >( getWorkData( id ) );
    data.m_msg = msg;
}

void hkVdbDefaultProgressReporter::didWorkFor( hkUint64 id )
{
    // Do update
    m_lock->enter();
    {
        WorkData& data = const_cast< WorkData& >( getWorkData( id ) );
        HK_ASSERT( 0x22441118, data.m_pendingCancel <= 1, "Shouldn't do extra work when there's a cancel pending" );
        // Grow our pending cancels if we have one, that way, the next time this is called we'll assert.
        HK_ON_DEBUG( data.m_pendingCancel += bool( data.m_pendingCancel ); )

        // Get change ticks immediately
        const hkUint64 current = hkSystemClock::getTickCounter();
        const hkUint64 ticks = ( current - data.m_lastTick );

        // Update our estimates
        data.removeContribution( m_estimate );
        {
            data.m_lastTick = current;
            data.m_estimate.m_completedWorkItems++;
            data.m_estimate.m_estWorkItems = hkMath::max2( data.m_estimate.m_completedWorkItems, data.m_estimate.m_estWorkItems );
            m_estimate.m_completedWorkItems++;
            data.pushNewTick( ticks );
        }
        data.addContribution( m_estimate );
    }
    m_lock->leave();

    // Signal
    m_estUpdated.fire( *this, id );
}

void hkVdbDefaultProgressReporter::signalWarningFor( hkUint64 id, hkVdbSignalResult& warningResult )
{
    hkCriticalSectionLock lock( m_lock );
    reportResults( hkLog::Level::Warning, warningResult );
}

void hkVdbDefaultProgressReporter::signalErrorFor( hkUint64 id, hkVdbSignalResult& errorResult )
{
    hkCriticalSectionLock lock( m_lock );
    endWorkLoad( id );
    reportResults( hkLog::Level::Error, errorResult );
}

void hkVdbDefaultProgressReporter::endWorkLoad( hkUint64 id )
{
    m_lock->enter();
    {
        // Correct our global estimate
        {
            WorkData& data = const_cast< WorkData& >( getWorkData( id ) );
            data.removeContribution( m_estimate );
            m_uninterruptibleWorkLoadCount -= bool( data.m_estimate.m_uninterruptible );
            m_estimate.m_estWorkItems += data.m_estimate.m_completedWorkItems;
        }

        // If this the last work item, we are done
        if ( ( m_estimate.m_unEstWorkDatas + m_estimate.m_estWorkDatas ) == 0 )
        {
            // Assert if our other data is inconsistent
            HK_ASSERT( 0x22441400, ( m_uninterruptibleWorkLoadCount == 0 ) && ( m_workDatas.getSize() == 1 ), "Work data internal inconsistency" );
            m_estimate = TickEstimate();
            m_uninterruptibleWorkLoadCount = 0;
        }
    }
    m_lock->leave();

    // Signal
    m_estUpdated.fire( *this, id );

    m_lock->enter();
    {
        // Check for shrink
        int idx = int( id );
        if ( idx == ( m_workDatas.getSize() - 1 ) )
        {
            m_workDatas.setSize( idx );
            idx--; id--;
            while ( ( idx >= 0 ) && ( getWorkData( id ).m_lastTick == 0 ) ) { idx--; id--; };
            idx++;
            m_workDatas.setSize( idx );
            m_workDataIndicesPool.resize( 0, idx );
        }
        // Otherwise, reuse
        else
        {
            WorkData& data = const_cast< WorkData& >( getWorkData( id ) );
            data.m_lastTick = 0;
            m_workDataIndicesPool.set( idx );
        }

        HK_ASSERT(
            0x22441120,
            bool( m_workDatas.getSize() ) == bool( m_estimate.m_unEstWorkDatas + m_estimate.m_estWorkDatas ) &&
            ( bool( m_workDatas.getSize() ) || !m_workDataIndicesPool.bitCount() ),
            "Desync in work data counts" );
    }
    m_lock->leave();
}

hkBool32 hkVdbDefaultProgressReporter::shouldContinueWorkFor( hkUint64 id ) const
{
    hkCriticalSectionLock lock( m_lock );
    WorkData& data = const_cast< WorkData& >( getWorkData( id ) );
    return !data.m_pendingCancel;
}

void hkVdbDefaultProgressReporter::WorkData::init( int sampleWindow, hkUint32 estWorkItems, hkReal estTime, hkBool32 canCancel )
{
    HK_WARN_ON_DEBUG_IF( estTime && !estWorkItems, 0x22441119, "estTime cannot be properly refined without estWorkItems" );

    // work items
    m_estimate.m_completedWorkItems = 0;
    m_estimate.m_estWorkItems = estWorkItems;

    // ticks
    reset( sampleWindow );
    m_estimate.m_estTicksRemaining = hkUint64( hkMath::max2( estTime, 0.0f ) * hkSystemClock::getTicksPerSecond() );
    m_startTick = hkSystemClock::getTickCounter();
    m_lastTick = m_startTick;
    m_initialEstAvgTick = m_estimate.m_estWorkItems ? ( m_estimate.m_estTicksRemaining / m_estimate.m_estWorkItems ) : 0;
    m_initialEstTicksRemaining = m_estimate.m_estTicksRemaining;

    // derived data
    m_rollingAvgWeight = ( 1 - ( bool( m_initialEstAvgTick ) * HK_VDB_PROGRESS_TIME_EST_UPDATE_WEIGHTING ) );
    hkReal initialEstWeight = ( 1 - m_rollingAvgWeight );
    m_weightedInitialEstAvgTick = hkUint64( initialEstWeight * m_initialEstAvgTick );

    // other
    m_estimate.m_uninterruptible = !canCancel;
    m_pendingCancel = false;
}

void hkVdbDefaultProgressReporter::WorkData::removeContribution( TickEstimate& globalEstimate ) const
{
    globalEstimate.m_unEstWorkDatas -= !estimatable();
    globalEstimate.m_estWorkDatas -= estimatable();
    globalEstimate.m_estWorkItems -= m_estimate.m_estWorkItems;
    globalEstimate.m_estTicksRemaining -= ( estimatable() * m_estimate.m_estTicksRemaining );
}

void hkVdbDefaultProgressReporter::WorkData::addContribution( TickEstimate& globalEstimate ) const
{
    globalEstimate.m_unEstWorkDatas += !estimatable();
    globalEstimate.m_estWorkDatas += estimatable();
    globalEstimate.m_estWorkItems += m_estimate.m_estWorkItems;
    globalEstimate.m_estTicksRemaining += ( estimatable() * m_estimate.m_estTicksRemaining );
}

void hkVdbDefaultProgressReporter::WorkData::pushNewTick( hkUint64 ticks )
{
    HK_ASSERT( 0x22441116, ticks <= HK_REAL_MAX, "Exceeded hkReal precision" );

    // Push to our window.
    hkWindowedAverage::pushNewValue( hkReal( hkMath::min2( ticks, HK_REAL_MAX ) ) );

    // Get our current rolling average.
    const hkUint64 avgTick = hkUint64( hkWindowedAverage::getWindowedMean() );

    // Compute our best guess at future avg ticks by weighting our current guess against rolling average.
    // Note: if we were never provided a time estimate, then our best guess will simply be our rolling average.
    const hkUint64 weightedAvgTick =
        hkUint64( m_rollingAvgWeight * avgTick ) +
        m_weightedInitialEstAvgTick;

#if 1
    m_estimate.m_estTicksRemaining = weightedAvgTick * ( m_estimate.m_estWorkItems - m_estimate.m_completedWorkItems );
#else
    
    // Compute our best guess at estimated time remaining by weighting our initial est against weighted avg ticks.
    // Note: if we've never ran this before and we *only* have a work items estimate, then our best guess should be based on weighted avg ticks.
    hkReal avgTickWeight = ( 1 - ( bool( estimatable() && m_estimate.m_estTicksRemaining ) * HK_VDB_PROGRESS_TIME_EST_UPDATE_WEIGHTING ) );
    hkReal timeEstWeight = ( 1 - avgTickWeight );
    m_estimate.m_estTicksRemaining =
        hkUint64( avgTickWeight * ( weightedAvgTick * ( m_estimate.m_estWorkItems - m_estimate.m_completedWorkItems ) ) ) +
        hkUint64( timeEstWeight * m_initialEstTicksRemaining );
#endif
}

const hkVdbDefaultProgressReporter::WorkData& hkVdbDefaultProgressReporter::getWorkData( hkUint64 id ) const
{
    // Check that the user didn't already call signalErrorFor or endWorkLoad.
    HK_ASSERT( 0x22441114, ( id < m_workDatas.getSize() ) && ( m_workDatas[int( id )].m_lastTick != 0 ), "Invalid id" );
    return m_workDatas[int( id )];
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
