// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#define HK_VDB_ERRORS_TABLE \
/*////////////////////////////////////////////////////////////////////////
// General errors
////////////////////////////////////////////////////////////////////////*/ \
START_VDB_ERR_CAT(NONE, "None", GENERAL_ERRORS) \
VDB_ERR(INVALID_ARGUMENTS, "Invalid Arguments") \
VDB_ERR(INVALID_STATE, "Invalid State") \
VDB_ERR(SAVE_ERROR, "Save Error") \
VDB_ERR(IO_ERROR, "IO Error") \
VDB_ERR(PROTOCOL_ERROR, "Protocol Error") \
END_VDB_ERR_CAT(UNKNOWN_ERROR, "Unknown Error", GENERAL_ERRORS) \
/*////////////////////////////////////////////////////////////////////////
// Plugin errors
////////////////////////////////////////////////////////////////////////*/ \
START_VDB_ERR_CAT(PLUGIN_ERROR, "Plugin Error", PLUGIN_ERRORS) \
VDB_ERR(PLUGIN_INIT_ERROR, "Plugin Initialization Error") \
VDB_ERR(PLUGIN_QUIT_ERROR, "Plugin Quit Error") \
VDB_ERR(PLUGIN_SETUP_INVALID, "Plugin Setup Invalid") \
VDB_ERR(PLUGIN_REGISTERED, "Plugin Registered") \
VDB_ERR(PLUGIN_DLL_INVALID, "Plugin DLL Invalid") \
VDB_ERR(PLUGIN_READ_ERROR, "Plugin Read Error") \
END_VDB_ERR_CAT(PLUGIN_WRITE_ERROR, "Plugin Write Error", PLUGIN_ERRORS) \
/*////////////////////////////////////////////////////////////////////////
// Connection errors
////////////////////////////////////////////////////////////////////////*/ \
START_VDB_ERR_CAT(CONNECTION_ERROR, "Connection Error", CONNECTION_ERRORS) \
VDB_ERR(NO_CONNECTION, "No Connection") \
VDB_ERR(INVALID_CONNECTION, "Invalid Connection") \
VDB_ERR(CONNECTION_READ_ERROR, "Connection Read Error") \
VDB_ERR(CONNECTION_WRITE_ERROR, "Connection Write Error") \
/* Network Connection */ \
VDB_ERR(NETWORK_CONNECTION_ERROR, "Network Connection Error") \
VDB_ERR(NETWORK_CONNECTION_TIMED_OUT, "Network Connection Timed Out") \
VDB_ERR(SOCKET_INVALID, "Socket Invalid") \
/* File Connection */ \
VDB_ERR(FILE_CONNECTION_ERROR, "File Connection Error") \
/* Memory Connection */ \
END_VDB_ERR_CAT(MEMORY_CONNECTION_ERROR, "Memory Connection Error", CONNECTION_ERRORS) \
/*////////////////////////////////////////////////////////////////////////
// Command errors
////////////////////////////////////////////////////////////////////////*/ \
START_VDB_ERR_CAT(CMD_BUFFER_ERROR, "Command Buffer Error", COMMAND_ERRORS) \
VDB_ERR(CMD_DISPATCH_ERROR, "Command Dispatch Error") \
VDB_ERR(CMD_HANDLER_NOT_REGISTERED, "Command Handler Not Registered") \
VDB_ERR(CMD_HANDLER_REGISTRATION_ERROR, "Command Handler Registration Error") \
VDB_ERR(CMD_HANDLER_ERROR, "Command Handler Error") \
VDB_ERR(CMD_HANDLER_READ_ERROR, "Command Handler Read Error") \
VDB_ERR(CMD_HANDLER_WRITE_ERROR, "Command Handler Write Error") \
VDB_ERR(CMD_UNKNOWN, "Unknown Command") \
VDB_ERR(CMD_ERROR, "Command Error") \
/* Setup Handler */ \
VDB_ERR(VERSION_READ_ERROR, "Version Read Error") \
VDB_ERR(UNSUPPORTED_VERSION, "Unsupported Version Error") \
/* Process Handler */ \
VDB_ERR(PROCESS_REGISTERED, "Process Registered") \
VDB_ERR(PROCESS_NOT_REGISTERED, "Process Not Registered") \
/* Playback Handler */ \
VDB_ERR(FRAME_SYNC_ERROR, "Frame Sync Error") \
/* Display Handler */ \
VDB_ERR(GEOMETRY_NOT_FOUND, "Geometry Not Found") \
/* Object Handler */ \
VDB_ERR(OBJECT_TYPE_NOT_REGISTERED, "Object Type Not Registered") \
END_VDB_ERR_CAT(OBJECT_SERIALIZATION_ERROR, "Object Serialization Error", COMMAND_ERRORS) \
/* Stats Handler */ \
/* Text Handler */ \
VDB_ERR(NUM_ERRORS, "")

//////////////////////////////////////////////////////////////////////////
/// Enum for vdb error values.
//////////////////////////////////////////////////////////////////////////
namespace hkVdbError
{
    enum Enum
    {
#define START_VDB_ERR_CAT(a, b, c) c##_START, a = c##_START,
#define END_VDB_ERR_CAT(a, b, c) a, c##_END = a,
#define VDB_ERR(a, b) a,
        HK_VDB_ERRORS_TABLE
#undef START_VDB_ERR_CAT
#undef END_VDB_ERR_CAT
#undef VDB_ERR
    };
}

/// Table for vdb error enum strings.
/// For example:
///  const char* errorEnumStr = hkVdbErrorStr[hkVdbError::GEOMETRY_NOT_FOUND];
extern const char** hkVdbErrorStr;

namespace hkLog { class Origin; }


//////////////////////////////////////////////////////////////////////////
/// Interface for an object which reports errors
//////////////////////////////////////////////////////////////////////////
















struct hkVdbErrorReporter : public hkReferencedObject
{
    /// Get the current error or hkVdbError::NONE if there is no errors.
    virtual hkVdbError::Enum getError() const = 0;

    /// Report the current error.
    /// Can optionally provide a log origin to report from.
    virtual void reportError( hkLog::Origin* reportFrom = HK_NULL ) const = 0;

    /// Clear the current error state.
    virtual void clearError() = 0;

    /// Should be called at the line of code where the error occurred
    /// for breakpoint debugging.
    virtual void errorSignalled() const {} // for debug use
};


//////////////////////////////////////////////////////////////////////////
/// Result from a signal callback which allows the listener to convey
/// errors back to the signaler.
//////////////////////////////////////////////////////////////////////////

struct hkVdbSignalResult
{
    HK_DECLARE_CLASS( hkVdbSignalResult, New );
    HK_DECLARE_POD_TYPE();

    hkVdbSignalResult() :
        m_id( 0 ),
        m_error( hkVdbError::NONE ),
        m_reporter( HK_NULL )
    {}

    /// Signal that an error has occurred and provide a reporter for that error.
    HK_ALWAYS_INLINE void signalError( hkVdbErrorReporter& reporter )
    {
        if ( ( m_error == hkVdbError::NONE ) && !m_reporter )
        {
            m_error = hkVdbError::UNKNOWN_ERROR;
            m_reporter = &reporter;
            reporter.errorSignalled();
        }
    }

    /// Signal that an error has occurred and provide an id and error value.
    HK_ALWAYS_INLINE void signalError( hkUint32 id, hkVdbError::Enum error )
    {
        if ( ( m_error == hkVdbError::NONE ) && !m_reporter )
        {
            m_error = error;
            m_reporter = HK_NULL;
        }
    }

    /// Signal success (not necessary as this is the default).
    HK_ALWAYS_INLINE void signalSuccess() {}

    /// Returns true if the signal result is success.
    HK_ALWAYS_INLINE bool isSuccess() const
    {
        return (this->operator hkVdbError::Enum()) == hkVdbError::NONE;
    }

    /// Returns true if the signal result is failure.
    HK_ALWAYS_INLINE bool isFailure() const
    {
        return (this->operator hkVdbError::Enum()) != hkVdbError::NONE;
    }

    /// Get the error id (if it's been set by signalError(...))
    HK_ALWAYS_INLINE hkUint32 getId() const
    {
        return m_id;
    }

    /// Get the error value (if it's been set by signalError(...))
    HK_ALWAYS_INLINE hkVdbError::Enum getError() const
    {
        return ( *this );
    }

    /// Get the error reporter (if it's been set by signalError(...))
    HK_ALWAYS_INLINE hkVdbErrorReporter* getReporter() const
    {
        return m_reporter;
    }

    HK_ALWAYS_INLINE bool operator==( hkVdbError::Enum error ) const
    {
        return ( ( this->operator hkVdbError::Enum() ) == error );
    }

    HK_ALWAYS_INLINE operator hkResult() const
    {
        return ((this->operator hkVdbError::Enum()) == hkVdbError::NONE) ? HK_SUCCESS : HK_FAILURE;
    }

    HK_ALWAYS_INLINE operator hkVdbError::Enum() const
    {
        return ( m_reporter && m_reporter->getError() ) ? ( m_reporter->getError() ) : m_error;
    }

private:

    hkUint32 m_id;
    hkVdbError::Enum m_error;
    hkRefPtr<hkVdbErrorReporter> m_reporter;
};

#ifdef HK_DEBUG
#define _HK_VDB_EXTRA_REPORT_ERROR_FUNC HK_REPORT2
#else
#define _HK_VDB_EXTRA_REPORT_ERROR_FUNC( id, message )
#endif

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
