// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

namespace hkLog { class Origin; }

//////////////////////////////////////////////////////////////////////////
/// A singleton which is checked by instances of hkVdbDefaultErrorReporter
/// when considering if an hkVdbError should be filtered or not.
//////////////////////////////////////////////////////////////////////////
class hkVdbDefaultErrorFilter : public hkReferencedObject
{
public:
    HK_DECLARE_CLASS( hkVdbDefaultErrorFilter, New, Singleton );
    HK_INLINE void setIsEnabled( hkVdbError::Enum error, bool enabled ) { if ( enabled ) m_disabledErrors.remove( error ); else m_disabledErrors.insert( error ); }
    HK_INLINE hkBool32 isEnabled( hkVdbError::Enum error ) const { return !m_disabledErrors.contains( error ); }
    HK_INLINE void setIsEnabled( hkUint32 id, bool enabled ) { if ( enabled ) m_disabledErrorIds.remove( id ); else m_disabledErrorIds.insert( id ); }
    HK_INLINE hkBool32 isEnabled( hkUint32 id ) const { return !m_disabledErrorIds.contains( id ); }
protected:
    hkVdbSet<hkVdbError::Enum> m_disabledErrors;
    hkVdbSet<hkUint32> m_disabledErrorIds;
};

//////////////////////////////////////////////////////////////////////////
/// A default implementation for reporting errors.
/// Manages reporting correctly from sub-reporters and channeling
/// output to the hkLog system.
//////////////////////////////////////////////////////////////////////////




class hkVdbDefaultErrorReporter : public hkVdbErrorReporter
{
public:

    HK_DECLARE_CLASS( hkVdbDefaultErrorReporter, New );

    hkVdbDefaultErrorReporter( hkLog::Origin* log ) :
        m_reporter( HK_NULL ),
        m_log( log )
    {
        clearError();
    }

    //
    // hkVdbErrorReporter interface
    //

    HK_INLINE_VIRTUAL virtual hkVdbError::Enum getError() const HK_OVERRIDE
    {
        return reporterHasError() ? m_reporter->getError() : m_error;
    }

    virtual void reportError( hkLog::Origin* reportFrom = HK_NULL ) const HK_OVERRIDE;

    HK_INLINE_VIRTUAL virtual void clearError() HK_OVERRIDE
    {
        m_errorId = 0;
        m_error = hkVdbError::NONE;
        m_errorMsg = HK_NULL;
        if ( m_reporter && ( m_reporter.val() != this ) ) m_reporter->clearError();
        m_reporter = HK_NULL;
    }

    virtual void errorSignalled() const;

protected:

    /// Returns true if the current error is filtered.
    virtual hkBool32 isFiltered() const;

    /// Returns true if we have a sub-reporter with an error.
    HK_INLINE hkBool32 reporterHasError() const
    {
        return ( m_reporter && ( m_reporter.val() != this ) && m_reporter->getError() );
    }

    /// Signal that we have an error with the given id, value, and message.
    HK_INLINE void signalError( hkUint32 id, hkVdbError::Enum error, const char* message ) const
    {
        m_errorId = id;
        m_errorMsg = message;
        m_reporter = HK_NULL;
        m_error = error;
        errorSignalled();
    }

    /// Signal that we have an error and the provided reporter should offer more information.
    /// However, if it does not, a default error value should be provided.
    HK_INLINE void signalError( hkVdbErrorReporter& reporter, hkVdbError::Enum defaultError ) const
    {
        m_errorId = 0;
        m_errorMsg = HK_NULL;
        m_reporter = &reporter;
        m_error = defaultError;
        errorSignalled();
    }

    /// If no hkLog::Origin is set, than this function will be called when the class is asked
    /// to report an error.
    virtual void reportErrorCustom( hkUint32 id, hkVdbError::Enum error, const char* msg ) const;

private:

    mutable hkUint32 m_errorId;
    mutable hkVdbError::Enum m_error;
    mutable hkStringPtr m_errorMsg;
    mutable hkRefPtr<hkVdbErrorReporter> m_reporter;
    hkLog::Origin* m_log;
};




/// If the reporter has an error, update our error state with the reporter
/// and return HK_FAILURE
#define HK_VDB_VERIFY_REPORTER( ERROR_REPORTER ) \
    HK_MULTILINE_MACRO_BEGIN \
        if ( HK_VERY_UNLIKELY( ( ERROR_REPORTER ).getError() ) ) \
        { \
            signalError( ( ERROR_REPORTER ), hkVdbError::NONE ); \
            return HK_FAILURE; \
        } \
    HK_MULTILINE_MACRO_END

/// If the condition is false, update our error state with the id and error value
/// and return HK_FAILURE
#define HK_VDB_VERIFY_CONDITION( CONDITION, ID, ERROR ) \
    HK_MULTILINE_MACRO_BEGIN \
        if ( HK_VERY_UNLIKELY( !( CONDITION ) ) ) \
        { \
            signalError( ( ID ), ( ERROR ), HK_NULL ); \
            return HK_FAILURE; \
        } \
    HK_MULTILINE_MACRO_END

/// If the condition is false, update our error state with the id, error value, and message
/// and return HK_FAILURE
#define HK_VDB_VERIFY_CONDITION_MSG( CONDITION, ID, ERROR, MSG ) \
    HK_MULTILINE_MACRO_BEGIN \
        if ( HK_VERY_UNLIKELY( !( CONDITION ) ) ) \
        { \
            HK_ERROR_BUFFER_ALIGNMENT( char _buf[512] ); \
            hkErrStream _ostr( _buf, sizeof(_buf) ); \
            _ostr << MSG; \
            signalError( ( ID ), ( ERROR ), _buf ); \
            return HK_FAILURE; \
        } \
    HK_MULTILINE_MACRO_END

/// If the condition is false, update our error state with the reporter
/// and return HK_FAILURE
#define HK_VDB_VERIFY_REPORTER_CONDITION( CONDITION, ERROR_REPORTER, DEFAULT_ERROR ) \
    HK_MULTILINE_MACRO_BEGIN \
        if ( HK_VERY_UNLIKELY( !( CONDITION ) ) ) \
        { \
            signalError( ( ERROR_REPORTER ), DEFAULT_ERROR ); \
            return HK_FAILURE; \
        } \
    HK_MULTILINE_MACRO_END

/// If the operation is HK_FAILURE, update our error state with the id and error value
/// and return HK_FAILURE
#define HK_VDB_VERIFY_OPERATION( OPERATION, ID, ERROR ) \
    HK_VDB_VERIFY_CONDITION( ( ( OPERATION ).isSuccess() ), ID, ERROR )

/// If the operation is HK_FAILURE, update our error state with the id, error value, and message
/// and return HK_FAILURE
#define HK_VDB_VERIFY_OPERATION_MSG( OPERATION, ID, ERROR, MSG ) \
    HK_VDB_VERIFY_CONDITION_MSG( ( ( OPERATION ).isSuccess() ), ID, ERROR, MSG )

/// If the operation is HK_FAILURE, update our error state with the reporter
/// and return HK_FAILURE
#define HK_VDB_VERIFY_REPORTER_OPERATION( OPERATION, ERROR_REPORTER, DEFAULT_ERROR ) \
    HK_VDB_VERIFY_REPORTER_CONDITION( ( ( OPERATION ).isSuccess() ), ERROR_REPORTER, DEFAULT_ERROR )

/// Signal we have an error with the id and error value
#define HK_VDB_SIGNAL_ERROR( ID, ERROR ) \
    HK_MULTILINE_MACRO_BEGIN \
        signalError( ( ID ), ( ERROR ), HK_NULL ); \
    HK_MULTILINE_MACRO_END

/// Signal we have an error with the id, error value, and message
#define HK_VDB_SIGNAL_ERROR_MSG( ID, ERROR, MSG ) \
    HK_MULTILINE_MACRO_BEGIN \
        HK_ERROR_BUFFER_ALIGNMENT( char _buf[512] ); \
        hkErrStream _ostr( _buf, sizeof(_buf) ); \
        _ostr << MSG; \
        signalError( ( ID ), ( ERROR ), _buf ); \
    HK_MULTILINE_MACRO_END

/// Signal we have an error with the reporter
#define HK_VDB_SIGNAL_REPORTER_ERROR( ERROR_REPORTER, DEFAULT_ERROR ) \
    HK_MULTILINE_MACRO_BEGIN \
        signalError( ( ERROR_REPORTER ), DEFAULT_ERROR ); \
    HK_MULTILINE_MACRO_END

/// If the hkVdbSignalResult is HK_FAILURE, update our error state with it's data
/// and return HK_FAILURE
#define HK_VDB_VERIFY_SIGNAL_RESULT( RESULT ) \
    HK_MULTILINE_MACRO_BEGIN \
        if ( hkVdbErrorReporter* _reporter = ( RESULT ).getReporter() ) { \
            if ( _reporter == this ) { if ( HK_VERY_UNLIKELY( ( RESULT ).isFailure() ) ) return HK_FAILURE; } \
            else { HK_VDB_VERIFY_REPORTER_OPERATION( ( RESULT ), *_reporter, ( RESULT ).getError() ); } \
        } \
        else { HK_VDB_VERIFY_OPERATION( ( RESULT ), ( RESULT ).getId(), ( RESULT ).getError() ); } \
    HK_MULTILINE_MACRO_END

/// If the condition is false, update our hkVdbSignalResult with the id and error value.

#define HK_VDB_VERIFY_SIGNALLED_CONDITION( CONDITION, ID, ERROR, RESULT ) \
    HK_MULTILINE_MACRO_BEGIN \
        if ( HK_VERY_UNLIKELY( !( CONDITION ) ) ) ( RESULT ).signalError( ( ID ), ( ERROR ) ); \
    HK_MULTILINE_MACRO_END

/// If the condition is false, update our hkVdbSignalResult with the reporter.
#define HK_VDB_VERIFY_SIGNALLED_REPORTER_CONDITION( CONDITION, ERROR_REPORTER, RESULT ) \
    HK_MULTILINE_MACRO_BEGIN \
        if ( HK_VERY_UNLIKELY( !( CONDITION ) ) ) ( RESULT ).signalError( ( ERROR_REPORTER ) ); \
    HK_MULTILINE_MACRO_END

/// If the operation is HK_FAILURE, update our hkVdbSignalResult with the id and error value.
#define HK_VDB_VERIFY_SIGNALLED_OPERATION( OPERATION, ID, ERROR, RESULT ) \
    HK_VDB_VERIFY_SIGNALLED_CONDITION( ( ( OPERATION ).isSuccess() ), ID, ERROR, RESULT )

/// If the operation is HK_FAILURE, update our hkVdbSignalResult with the reporter.
#define HK_VDB_VERIFY_SIGNALLED_REPORTER_OPERATION( OPERATION, ERROR_REPORTER, RESULT ) \
    HK_VDB_VERIFY_SIGNALLED_REPORTER_CONDITION( ( ( OPERATION ).isSuccess() ), ERROR_REPORTER, RESULT )



/// If the condition is false, return HK_FAILURE.
#define HK_VDB_FAIL_IF_CONDITION_FALSE( CONDITION ) \
    if ( HK_VERY_UNLIKELY( !( CONDITION ) ) ) { return HK_FAILURE; }

/// If the condition is false, return.
#define HK_VDB_RETURN_IF_CONDITION_FALSE( OPERATION ) \
    if ( HK_VERY_UNLIKELY( !( CONDITION ) ) ) { return; }

/// If the operation is HK_FAILURE, return HK_FAILURE.
#define HK_VDB_FAIL_IF_OPERATION_FAILED( OPERATION ) \
    HK_VDB_FAIL_IF_CONDITION_FALSE( ( ( OPERATION ).isSuccess() ) )

/// If the operation is HK_FAILURE, return.
#define HK_VDB_RETURN_IF_OPERATION_FAILED( OPERATION ) \
    HK_VDB_RETURN_IF_CONDITION_FALSE( ( ( OPERATION ).isSuccess() ) )

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
