// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbServices/hkVdbServices.h>
#include <VisualDebugger/VdbServices/System/Utils/hkVdbDebugWriter.h>

#include <Common/Base/System/Io/Writer/Array/hkArrayStreamWriter.h>
#include <Common/Base/System/Io/FileSystem/hkFileSystem.h>

#include <Common/Visualize/hkVersionReporter.h>

#include <VisualDebugger/VdbServices/System/Command/hkVdbCmd.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdInput.h>
#include <VisualDebugger/VdbServices/System/Utils/hkVdbCmdUtils.h>

hkVdbDebugWriter::hkVdbDebugWriter( hkVdbCmdInput& input, hkCmdWriteFunc func, void* handle ) :
    m_input( &input ),
    m_writeHeader( true ),
    m_currentCmdBytesWritten( 0 ),
    m_currentCmdTotalBytes( HK_INT32_MAX ),
    m_currentCmdWriter( &m_currentCmdStorage, hkArrayStreamWriter::ARRAY_BORROW ),
    m_writeFunc( func ),
    m_writeFuncHandle( handle )
{}

hkVdbDebugWriter::hkVdbDebugWriter( hkVdbCmdInput& input, const char* filename ) :
    hkVdbDebugWriter( input, HK_NULL, HK_NULL )
{
    if ( filename )
    {
        m_internalWriter = hkFileSystem::getInstance().openWriter( filename );
        m_writeFunc = []( const hkVdbCmd& cmd, int protocol, void* handle )
        {
            hkVdbDebugWriter::writeCmd( cmd, *static_cast<hkStreamWriter*>( handle ), protocol );
        };
        m_writeFuncHandle = m_internalWriter;
    }
}

int hkVdbDebugWriter::write( const void* buf, int nbytes )
{
    int bytesWritten = m_currentCmdWriter.write( buf, nbytes );
    m_currentCmdBytesWritten += bytesWritten;
    m_currentCmdTotalBytes = ( m_currentCmdBytesWritten >= 4 ) ? currentCmd()->getCmdSize() : HK_INT32_MAX;

    while ( m_currentCmdBytesWritten >= m_currentCmdTotalBytes )
    {
        hkVdbCmd* cmd = currentCmd();
        if ( m_writeFunc != HK_NULL )
        {
            // If we internally created our writer, we write the header here.
            if ( m_writeHeader )
            {
                if ( m_writeFuncHandle && ( m_writeFuncHandle == m_internalWriter ) )
                {
                    hkVdbDebugWriter::writeHeader( *cmd, *m_internalWriter, m_input->getProtocol() );
                }
                m_writeHeader = false;
            }

            // Notify the write func we've processed a cmd.
            m_writeFunc( *cmd, m_input->getProtocol(), m_writeFuncHandle );
        }

        // Reset to next command
        m_currentCmdWriter.seek( 0, hkStreamWriter::STREAM_SET );
        m_currentCmdStorage.removeAtAndCopy( 0, cmd->getCmdSize() );
        m_currentCmdBytesWritten = ( m_currentCmdBytesWritten - m_currentCmdTotalBytes );
        m_currentCmdTotalBytes = ( m_currentCmdBytesWritten >= 4 ) ? currentCmd()->getCmdSize() : HK_INT32_MAX;
        m_currentCmdWriter.seek( m_currentCmdBytesWritten, hkStreamWriter::STREAM_SET );
    }

    return bytesWritten;
}

void hkVdbDebugWriter::writeCmd( const hkVdbCmd& cmd, hkStreamWriter& writerIn, int protocol )
{
    hkOstream writer( &writerIn );

    hkUint64 id = 0;
    hkUint64 referenceId = 0;
    hkVdbLocalIStream cmdStream( cmd.getData(), cmd.getDataSize() );
    hkVdbCmdType::Enum type = hkVdbCmdType::patchType<hkVdbCmdType::Direction::SERVER_TO_CLIENT>( protocol, cmd.getServerType() );
    if ( hkVdbCmdUtils::extractUniqueIds( type, id, referenceId, *cmdStream.getStreamReader() ).isSuccess() )
    {
        const char* debugTag = HK_NULL;
        hkStringBuf debugTagBuf;
        if ( ( type == hkVdbCmdType::CLIENT_INTERNAL ) &&
            ( id == hkVdbInternalCmdType::DEBUG_TAG ) )
        {
            // Note: cannot use hkVdbInternalHandler functions because
            // we don't have meta data here.
            if ( cmdStream.readString( debugTagBuf ) )
            {
                if ( debugTagBuf[0] == '>' )
                {
                    writer << "\n*** " << (debugTagBuf.cString()+1) << " ***\n\n";
                }
                else if ( debugTagBuf[0] == '<' )
                {
                    debugTag = (debugTagBuf.cString()+1);
                }
                else
                {
                    debugTag = debugTagBuf;
                }
            }
        }

        if ( referenceId > 0 )
        {
            writer << "[c" << int( type ) << "|s" << int( cmd.getServerType() ) << "]" << " id=" << ((void*)id) << " referenceId=" << ((void*)referenceId) << " size=" << cmd.getBodySize();
        }
        else
        {
            writer << "[c" << int( type ) << "|s" << int( cmd.getServerType() ) << "]" << " id=" << ((void*)id) << " size=" << cmd.getBodySize();
        }

        if ( debugTag )
        {
            writer << " \"" << debugTag << "\"";
        }

        writer << "\n";
    }
    else
    {
        writer << "[c" << int( type ) << "|s" << int( cmd.getServerType() ) << "] size=" << cmd.getBodySize() << "\n";
    }
}

void hkVdbDebugWriter::writeHeader( const hkVdbCmd& cmd, hkStreamWriter& writerIn, int protocol )
{
    hkOstream writer( &writerIn );
    writer << "This writer assumes up-to-date cmd formats for retrieving ids in the cmd.\n";
    writer << "When connecting to an older protocol version, you may see corrupt ids if their place\n";
    writer << "in the cmd data has changed.\n";
    writer << "\n";
    writer << "Protocol: Client=" << hkVersionReporter::m_protocolVersion << " | Server=" << protocol << "\n";
    writer << "\n";
    writer << "Cmd Format (() => optional; based on cmd type):\n";
    writer << "\t[cCLIENT_CMD_TYPE|sSERVER_CMD_TYPE] (id=ID) (referenceId=REFERENCED_ID) size=SIZE (\"Debug tag\")\n";
    writer << "\t*** Debug tag written during saving of this file (didn't exist in the stream) ***\n";
    writer << "\n";
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
