// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbServices/System/Connection/hkVdbServerDiscoveryConnection.h>

#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Base/Container/LocalArray/hkLocalArray.h>

#define DEBUG_LOG_IDENTIFIER "vdb.Connection.ServerDiscovery"
#include <Common/Base/System/Log/hkLog.hxx>

namespace
{
    class hkNetLobby_SessionInfo_less
    {
    public:
        HK_INLINE hkBool32 operator() ( const hkNetLobby::SessionInfo& a, const hkNetLobby::SessionInfo& b )
        {
            return ( a.m_serverAddr.toUint64() < b.m_serverAddr.toUint64() );
        }
    };

    void addSessionInfo( hkArray<hkNetLobby::SessionInfo>& infos, const hkNetLobby::SessionInfo& infoToAdd )
    {
        int idx = hkAlgorithm::findInsertionIndex(
            infoToAdd,
            infos.begin(),
            infos.getSize(),
            hkNetLobby_SessionInfo_less() );

        if ( ( idx < infos.getSize() ) &&
            ( infoToAdd.m_serverAddr.toUint64() == infos[idx].m_serverAddr.toUint64() ) )
        {
            infos[idx] = infoToAdd;
        }
        else
        {
            infos.insertAt( idx, infoToAdd );
        }
    }
}

hkVdbServerDiscoveryConnection::hkVdbServerDiscoveryConnection(
    hkUint16 broadcastPort,
    int serverGroupId,
    hkReal serverTimeoutMs )
    : hkVdbConnection( &s_debugLog ),
    m_broadcastPort( broadcastPort ),
    m_serverGroupId( serverGroupId ),
    m_serverTimeoutMs( serverTimeoutMs ),
    m_processedResults( 0 )
{}

hkVdbConnectionState::Enum hkVdbServerDiscoveryConnection::getState() const
{
    
    if ( m_netLobby )
    {
        return hkVdbConnectionState::CONNECTED;
    }
    else
    {
        return hkVdbConnectionState::DISCONNECTED;
    }
}

hkResult hkVdbServerDiscoveryConnection::connect()
{
    disconnect();
    m_netLobby = hkRefNew<hkNetLobby>( new hkNetLobby() );
    if ( findSessions().isFailure() ) return HK_FAILURE;
    return HK_SUCCESS;
}

hkResult hkVdbServerDiscoveryConnection::disconnect()
{
    m_netLobby = HK_NULL;
    m_sortedConnectedSessions.clear();
    return HK_SUCCESS;
}

hkResult hkVdbServerDiscoveryConnection::poll()
{
    HK_VDB_VERIFY_CONDITION_MSG(
        m_netLobby,
        0xedb00172,
        hkVdbError::NETWORK_CONNECTION_ERROR,
        "Server discovery connection has been disconnected" );

    m_netLobby->step();
    addNewResults();

    if ( m_netLobby->getStatus() == hkNetLobby::STATUS_DISCOVERY_FINISHED )
    {
        if ( findSessions().isFailure() ) return HK_FAILURE;
    }

    return HK_SUCCESS;
}

hkResult hkVdbServerDiscoveryConnection::findSessions()
{
    HK_VDB_VERIFY_CONDITION_MSG(
        m_netLobby,
        0xedb00174,
        hkVdbError::NETWORK_CONNECTION_ERROR,
        "Server discovery connection has been disconnected" );

    cleanupStaleResults();

    m_processedResults = 0;
    hkResult result = m_netLobby->findSessions( m_serverGroupId, m_broadcastPort, m_serverTimeoutMs );
    if ( result.isFailure() )
    {
        disconnect();
        signalError(
            0xedb00173,
            hkVdbError::NETWORK_CONNECTION_ERROR,
            "Failed to start server discovery connection" );
        return HK_FAILURE;
    }

    addNewResults();

    return HK_SUCCESS;
}

void hkVdbServerDiscoveryConnection::addNewResults()
{
    if ( m_netLobby )
    {
        for ( ; m_processedResults < m_netLobby->getNumResults(); m_processedResults++ )
        {
            hkNetLobby::SessionInfo newSessionInfo;
            m_netLobby->getSessionInfo( newSessionInfo, m_processedResults );
            addSessionInfo( m_sortedConnectedSessions, newSessionInfo );
        }
    }
}

void hkVdbServerDiscoveryConnection::cleanupStaleResults()
{
    if ( m_netLobby )
    {
        // Collect from net lobby once to avoid more copies
        hkLocalArray<hkNetLobby::SessionInfo> foundSessions( m_netLobby->getNumResults() );
        for ( int i = 0; i < m_netLobby->getNumResults(); i++ )
        {
            hkNetLobby::SessionInfo sessionInfo;
            m_netLobby->getSessionInfo( sessionInfo, i );
            addSessionInfo( foundSessions, sessionInfo );
        }

        // Now remove any of our connections that we didn't find this time
        for ( int i = 0; i < m_sortedConnectedSessions.getSize(); i++ )
        {
            hkNetLobby::SessionInfo& sessionInfo = m_sortedConnectedSessions[i];
            int idx = hkAlgorithm::binarySearch(
                sessionInfo,
                foundSessions.begin(),
                foundSessions.getSize(),
                hkNetLobby_SessionInfo_less() );
            if ( idx == -1 )
            {
                m_sortedConnectedSessions.removeAtAndCopy( i );
            }
        }
    }
    else
    {
        m_sortedConnectedSessions.clear();
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
