// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbServices/hkVdbServices.h>
#include <VisualDebugger/VdbServices/System/Connection/hkVdbNetworkConnection.h>

#include <Common/Base/Thread/Atomic/hkAtomicPrimitives.h>
#include <Common/Base/System/Io/Socket/hkSocket.h>
#include <Common/Base/System/Io/Reader/Buffered/hkBufferedStreamReader.h>

#define DEBUG_LOG_IDENTIFIER "vdb.Connection.Network"
#include <Common/Base/System/Log/hkLog.hxx>

#ifdef HK_PLATFORM_WIN32
#define ENABLE_BACKGROUND_NETWORK_THREAD
#endif

#ifdef ENABLE_BACKGROUND_NETWORK_THREAD
#include <winsock2.h>
#include <Common/Base/System/Io/Platform/Bsd/hkBsdSocket.h>
#define SET_STATE( STATE ) \
    HK_MULTILINE_MACRO_BEGIN \
        hkAtomic::swap( data->m_state, (hkUint32)hkVdbConnectionState::##STATE ); \
        data->m_signal->fire( hkVdbNetworkEvent::##STATE ); \
    HK_MULTILINE_MACRO_END
#define SIGNAL_ERROR( ID, _ERROR, _MSG ) \
    HK_MULTILINE_MACRO_BEGIN \
        SET_STATE( ERRORED ); \
        data->m_errorId = ID; \
        data->m_error = _ERROR; \
        data->m_errorMsg = _MSG; \
    HK_MULTILINE_MACRO_END
#endif

namespace
{
#ifdef ENABLE_BACKGROUND_NETWORK_THREAD

    struct ThreadData
    {
        HK_DECLARE_CLASS( ThreadData, New );
        ThreadData() : m_error( hkVdbError::NONE ) {}

        // Inputs
        hkVdbNetworkConnection::NetworkEventReceivedSignal* m_signal;
        hkUint32* m_state;
        hkSocket* m_socket;
        hkStopwatch* m_timeoutWatch;
        hkReal m_timeoutMs;
        hkBool m_running;

        // Outputs
        hkVdbError::Enum m_error;
        hkUint32 m_errorId;
        const char* m_errorMsg;
    };

    
    
    
    void* networkThreadMain( void* dataIn )
    {
        ThreadData* data = reinterpret_cast< ThreadData* >( dataIn );
        hkUint32* state = data->m_state;
        hkSocket* socket = data->m_socket;
        hkStopwatch* timeoutWatch = data->m_timeoutWatch;
        SET_STATE( CONNECTING );

        WSAEVENT netEvent = WSACreateEvent();
        SOCKET socketToWatch = ( SOCKET ) static_cast< hkBsdSocket* >( socket )->getSocketInternal();
        if ( WSAEventSelect( socketToWatch, netEvent, FD_READ/* | FD_CONNECT */| FD_CLOSE ) != 0 )
        {
            SIGNAL_ERROR( 0xedb00117, hkVdbError::NETWORK_CONNECTION_ERROR, "Could not watch socket" );
        }

        while ( ( ( *state ) != hkVdbConnectionState::DISCONNECTED ) &&
            ( ( *state ) != hkVdbConnectionState::ERRORED ) && data->m_running )
        {
            DWORD ret = WSAWaitForMultipleEvents(
                // Number of events we are listening to
                1,
                // Our events array to place results into
                &netEvent,
                // Determines if we wait for all or some of our events, we only have one so doesn't actually matter
                false,
                // Timeout on waiting for events
                1000,
                // If I/O occurs as part of the wait, we will wait for that I/O event to complete
                true );

            // We are only watching one event
            if ( ret == WSA_WAIT_EVENT_0 )
            {
                // Signaled, find out what happened and set state
                WSANETWORKEVENTS events;
                if ( WSAEnumNetworkEvents( socketToWatch, netEvent, &events ) != 0 )
                {
                    SIGNAL_ERROR( 0xedb00118, hkVdbError::NETWORK_CONNECTION_ERROR, "Failed to enum network events" );
                }
                else
                {
                    
                    if ( events.lNetworkEvents & FD_READ )
                    {
                        
                        
                        
                        if ( ( *state ) == hkVdbConnectionState::CONNECTING )
                        {
                            SET_STATE( CONNECTED );
                        }
                        data->m_signal->fire( hkVdbNetworkEvent::READ_READY );
                    }
                    
                    
                    
                    
                    
                    
                    if ( events.lNetworkEvents & FD_CLOSE )
                    {
                        SET_STATE( DISCONNECTED );
                    }
                }
            }
            else if ( ( ret != WSA_WAIT_IO_COMPLETION ) && ( ret != WSA_WAIT_TIMEOUT ) )
            {
                // Any other response is an error
                SIGNAL_ERROR( 0xedb00119, hkVdbError::NETWORK_CONNECTION_ERROR, "Failed to watch network events" );
            }

            // Check for time-out.
            if ( timeoutWatch->isRunning() && ( ( *state ) == hkVdbConnectionState::CONNECTING ) )
            {
                if ( timeoutWatch->getElapsedSeconds() * 1000 > data->m_timeoutMs )
                {
                    SIGNAL_ERROR( 0xedb00159, hkVdbError::NETWORK_CONNECTION_TIMED_OUT, HK_NULL );
                }
            }
        }

        // We don't clear the internal socket notification state here.
        // That is we skip: WSAEventSelect(socketToWatch, netEvent, 0);
        // When the networkThreadMain thread is here and about to terminate, another networkThreadMain might
        // have started already and be using the same socketToWatch socket internal handle. We need notifications
        // to be correctly received in that thread or execution will hang in a WSAWaitForMultipleEvents() loop.
        
        
        WSACloseEvent( netEvent );

        return HK_NULL;
    }
#endif
}

hkVdbNetworkConnection::hkVdbNetworkConnection( const char* machineName, int port, hkReal timeoutMs ) :
    hkVdbConnection( &s_debugLog ),
    m_state( hkVdbConnectionState::DISCONNECTED ),
    m_networkThread( HK_NULL ),
    m_networkThreadData( HK_NULL ),
    // non-null ensures we are treated as an outgoing connection, not a listening connection
    m_machineName( machineName ? machineName : "" ),
    m_port( port ),
    m_timeoutMs( timeoutMs )
{
    clearError();
}

hkVdbNetworkConnection::hkVdbNetworkConnection( int listeningPort ) :
    hkVdbConnection( &s_debugLog ),
    m_state( hkVdbConnectionState::DISCONNECTED ),
    m_networkThread( HK_NULL ),
    m_port( listeningPort )
{
    clearError();
}

hkVdbNetworkConnection::~hkVdbNetworkConnection()
{
    disconnect();
    waitForBackgroundThreadCleanup();
}

hkVdbConnectionState::Enum hkVdbNetworkConnection::getState() const
{
    updateState();
    return hkVdbConnectionState::Enum( m_state );
}

hkResult hkVdbNetworkConnection::connect()
{
    disconnect();

    m_socket = hkRefNew<hkSocket>( hkSocket::create() );
    HK_VDB_VERIFY_OPERATION_MSG(
        m_socket->setBlocking( false ),
        0xedb00015,
        hkVdbError::NETWORK_CONNECTION_ERROR,
        "Could not make socket async" );

    if ( m_machineName )
    {
        HK_VDB_VERIFY_OPERATION_MSG(
            m_socket->connect( m_machineName, m_port ),
            0xedb00016,
            hkVdbError::NETWORK_CONNECTION_ERROR,
            "Socket failed to connect to \"" << m_machineName.cString() << ":" << m_port << "\"" );
        m_timeoutWatch.start();
    }
    else
    {
        HK_VDB_VERIFY_OPERATION_MSG(
            m_socket->bind( m_port ),
            0xedb00017,
            hkVdbError::NETWORK_CONNECTION_ERROR,
            "Socket failed to listen on port " << m_port );
    }

    m_socketReader = hkRefNew<hkBufferedStreamReader>( new hkBufferedStreamReader( &m_socket->getReader(), HK_VDB_MIN_NETWORK_READ ) );

#ifdef ENABLE_BACKGROUND_NETWORK_THREAD
    // Create a thread for network traffic.
    // ATM, we only use this to update our socket state as it's very
    // expensive to query the state whenever getState() is called.
    // In the future, we could expand the thread's responsibilities
    // to do reads, etc. if it helps with performance.
    
    
    
    {
        waitForBackgroundThreadCleanup();
        m_state = hkVdbConnectionState::CONNECTING;
        ThreadData* networkThreadData = new ThreadData();
        {
            networkThreadData->m_signal = &m_networkEventReceived;
            networkThreadData->m_socket = m_socket;
            networkThreadData->m_state = &m_state;
            networkThreadData->m_timeoutWatch = &m_timeoutWatch;
            networkThreadData->m_timeoutMs = m_timeoutMs;
            networkThreadData->m_running = true;
        }
        m_networkThread = new hkThread();
        m_networkThread->startThread( networkThreadMain, networkThreadData, "hkVdbNetworkConnection.NetworkThread" );
        m_networkThreadData = networkThreadData;
    }
#endif

    return HK_SUCCESS;
}

hkResult hkVdbNetworkConnection::disconnect()
{
    hkResult result = signalBackgroundThreadCleanup();

    
    if ( m_socket )
    {
        m_socketReader = HK_NULL;
        m_socket = HK_NULL;
    }

    m_timeoutWatch.reset();

    return result;
}

hkStreamReader* hkVdbNetworkConnection::getReader()
{
    return m_socketReader;
}

hkStreamWriter* hkVdbNetworkConnection::getWriter()
{
    return ( m_socket ) ? &m_socket->getWriter() : HK_NULL;
}

hkResult hkVdbNetworkConnection::poll()
{
    // If this isn't a listening network connection, just return.
    if ( m_machineName )
    {
        return HK_SUCCESS;
    }

    if ( m_socket )
    {
        if ( isValidSocket( *m_socket ) )
        {
            if ( hkSocket* socket = m_socket->pollForNewClient() )
            {
                m_connectedSockets.pushBack( hkRefNew<hkSocket>( socket ) );
            }
            else
            {
                
            }
            return HK_SUCCESS;
        }
        else
        {
            return HK_FAILURE;
        }
    }
    else
    {
        HK_VDB_SIGNAL_ERROR_MSG(
            0xedb00022,
            hkVdbError::NO_CONNECTION,
            "Cannot poll if network isn't connected" );
        return HK_FAILURE;
    }
}

hkResult hkVdbNetworkConnection::getConnectedAddresses( hkArray<hkInetAddr>& addrsOut )
{
    if ( m_socket )
    {
        if ( isValidSocket( *m_socket ) )
        {
            if ( m_machineName )
            {
                hkInetAddr inetAddrOut;
                HK_VDB_VERIFY_OPERATION_MSG(
                    m_socket->getInetAddr( inetAddrOut ),
                    0xedb00018,
                    hkVdbError::NETWORK_CONNECTION_ERROR,
                    "Could not get connected addresses for \"" << m_machineName.cString() << ":" << m_port << "\"" );
                addrsOut.pushBack( inetAddrOut );
            }
            else
            {
                for ( int i = 0; i < m_connectedSockets.getSize(); i++ )
                {
                    hkInetAddr inetAddrOut;
                    
                    if ( m_connectedSockets[i]->getInetAddr( inetAddrOut ).isSuccess() )
                    {
                        addrsOut.pushBack( inetAddrOut );
                    }
                }

                if ( addrsOut.getSize() != m_connectedSockets.getSize() )
                {
                    HK_VDB_SIGNAL_ERROR_MSG(
                        0xedb00019,
                        hkVdbError::NETWORK_CONNECTION_ERROR,
                        "One or more sockets connected to " << m_port << " failed to report their address" );
                    return HK_FAILURE;
                }
            }
            return HK_SUCCESS;
        }
        else
        {
            return HK_FAILURE;
        }
    }
    else
    {
        HK_VDB_SIGNAL_ERROR_MSG(
            0xedb00023,
            hkVdbError::NO_CONNECTION,
            "Cannot get addresses if connection is not connected" );
        return HK_FAILURE;
    }
}

void hkVdbNetworkConnection::updateState() const
{
    if ( HK_VERY_LIKELY( m_socket ) )
    {
#ifdef ENABLE_BACKGROUND_NETWORK_THREAD
        // Background network thread maintains up-to-date state.
        // Just check for error.
        checkBackgroundThreadForError();
#else
        if ( HK_VERY_LIKELY( isValidSocket( *m_socket ) ) )
        {
            if ( m_socket->canWrite() )
            {
                m_state = hkVdbConnectionState::CONNECTED;
            }
            else
            {
                m_state = hkVdbConnectionState::CONNECTING;
            }
        }
        else
        {
            m_state = hkVdbConnectionState::ERRORED;
        }
#endif
    }
    else if ( m_state != hkVdbConnectionState::ERRORED )
    {
        m_state = hkVdbConnectionState::DISCONNECTED;
    }

#ifndef ENABLE_BACKGROUND_NETWORK_THREAD
    // Check for time-out.
    if ( m_timeoutWatch.isRunning() && ( m_state == hkVdbConnectionState::CONNECTING ) )
    {
        if ( m_timeoutWatch.getElapsedSeconds() * 1000 > m_timeout )
        {
            m_state = hkVdbConnectionState::ERRORED;
            HK_VDB_SIGNAL_ERROR_MSG(
                0xedb00019,
                hkVdbError::NETWORK_CONNECTION_TIMED_OUT,
                "Timed-out while trying to connect to \"" << m_machineName.cString() << ":" << m_port << "\"" );
        }
    }
#endif
}

hkBool32 hkVdbNetworkConnection::isValidSocket( hkSocket& socket ) const
{
    if ( HK_VERY_UNLIKELY( !socket.isOk() ) )
    {
        if ( m_machineName )
        {
            HK_VDB_SIGNAL_ERROR_MSG(
                0xedb00020,
                hkVdbError::SOCKET_INVALID,
                "Socket for \"" << m_machineName.cString() << ":" << m_port << "\" is not valid" );
        }
        else
        {
            HK_VDB_SIGNAL_ERROR_MSG(
                0xedb00021,
                hkVdbError::SOCKET_INVALID,
                "Socket for " << m_port << " is not valid" );
        }
    }
    return true;
}

hkResult hkVdbNetworkConnection::checkBackgroundThreadForError() const
{
    if ( HK_VERY_LIKELY( ThreadData* data = reinterpret_cast< ThreadData* >( m_networkThreadData ) ) )
    {
        if ( HK_VERY_UNLIKELY( m_state == hkVdbConnectionState::ERRORED ) )
        {
            if ( data->m_running )
            {
                data->m_running = false;
                m_networkThread->joinThread(); // in case it's writing out error data

                hkStringBuf msgBuf;
                const char* errorMsg;
                if ( data->m_error == hkVdbError::NETWORK_CONNECTION_TIMED_OUT )
                {
                    msgBuf.printf( "Timed-out while trying to connect to \"%s:%i\"", m_machineName.cString(), m_port );
                    errorMsg = msgBuf;
                }
                else
                {
                    errorMsg = data->m_errorMsg;
                    HK_ASSERT( 0x22441033, errorMsg, "Error message not reported by background thread" );
                }
                signalError( data->m_errorId, data->m_error, errorMsg );
            }
            return HK_FAILURE;
        }
    }
    return HK_SUCCESS;
}

hkResult hkVdbNetworkConnection::signalBackgroundThreadCleanup()
{
    if ( ThreadData* data = reinterpret_cast< ThreadData* >( m_networkThreadData ) )
    {
        data->m_running = false;
    }
    return HK_SUCCESS;
}

hkResult hkVdbNetworkConnection::waitForBackgroundThreadCleanup()
{
    hkResult result = signalBackgroundThreadCleanup();

    if ( m_networkThread )
    {
        m_networkThread->joinThread();
        delete m_networkThread;
        m_networkThread = HK_NULL;
    }

    if ( m_networkThreadData )
    {
        delete reinterpret_cast< ThreadData* >( m_networkThreadData );
        m_networkThreadData = HK_NULL;
    }

    return result;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
