// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Visualize/hkVisualDebuggerCmdType.h>

#include <VisualDebugger/VdbServices/System/Connection/hkVdbConnection.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbPlaybackHandler.h>

class hkVdbCmdInput;
class hkVdbCmdOutput;
class hkArrayStreamWriter;

//////////////////////////////////////////////////////////////////////////
/// RVO-reliant class for writing a cmd to the output in a scoped block.
/// Intended use is:
/// if ( hkVdbScopedStreamWriter writer = cmdOutput.startCmd( type, autoFlushMode ) )
/// { /* do stuff */ }
/// else
/// { /* an error occurred and should be reported */ }
/// Note: You need to check "canFlush" if you wish to know if the cmd
/// will actually get flushed to the connection.
//////////////////////////////////////////////////////////////////////////
struct hkVdbScopedStreamWriter : public hkViewPtr<hkStreamWriter>
{
    ~hkVdbScopedStreamWriter();
private:
    hkVdbScopedStreamWriter( hkVdbCmdOutput& owner, hkStreamWriter* writer, int flushMode ) :
        hkViewPtr<hkStreamWriter>( writer ),
        m_owner( &owner ),
        m_flushMode( flushMode )
    {}
    hkViewPtr<hkVdbCmdOutput> m_owner;
    int m_flushMode;
    friend class hkVdbCmdOutput;
};

//////////////////////////////////////////////////////////////////////////
/// This class holds all cmd output data.
//////////////////////////////////////////////////////////////////////////
class hkVdbCmdOutput : public hkVdbDefaultErrorReporter
{
public:
    HK_DECLARE_CLASS( hkVdbCmdOutput, New );

    /// Creates a new output buffer.
    /// Requires the input so it can have a property cmd protocol value to write output cmds with.
    hkVdbCmdOutput( hkVdbCmdInput& input );

    /// Different flushing modes for use with completed a output cmd.
    /// Note: this can only be flushed if there's a valid app connection
    
    enum FlushMode
    {
        /// The cmd output will not be flushed to the app connection.
        NONE = 0,

        /// The cmd output will be flushed to the app connection but we won't block for it all to be written.
        NON_BLOCKING = 1,

        /// The cmd output will be flushed to the app connection and we'll wait till all of it is written.
        BLOCKING = 2
    };

    /// Start writing a new output cmd.
    hkVdbScopedStreamWriter startCmd( hkVdbCmdType::Enum type, FlushMode autoFlushMode = NONE );

    /// Returns true if the output can flush to the connection.
    HK_INLINE hkBool32 canFlush() const
    {
        return
            m_appConnection &&
            m_appConnection->getWriter();
            // A non-okay writer doesn't impact our ability to flush...the underlying connection is in an errored state.
            // && m_appConnection->getWriter()->isOk();
    }

    //
    // Internal use
    //
    void onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection );
    void onDisconnectingSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection, hkVdbSignalResult& result );
    void onPlaybackInfoReceivedSignalInternal( const hkVdbPlaybackHandler::PlaybackInfo& info, hkVdbSignalResult& result );

private:

    hkResult flush( bool& flushed, bool blocking );
    hkResult ackFrame( const hkVdbFrame& frame );

    HK_DEBUG_ONLY_MEMBER( hkBool32, m_cmdStarted );
    hkArray<char> m_outputBufferStorage;
    hkRefPtr<hkArrayStreamWriter> m_outputStream;
    int m_outputBufferBytesWritten;
    hkRefPtr<hkVdbConnection> m_appConnection;

    hkRefPtr<hkVdbCmdInput> m_input;
    int m_flushToOffset;
    hkBool32 m_isReplaying;

    hkUint32 m_acknowledgedFrames;

    friend struct hkVdbScopedStreamWriter;
    friend class hkVdbClient;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
