// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbServices/hkVdbServices.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdOutput.h>

#include <Common/Base/System/Io/Writer/Array/hkArrayStreamWriter.h>

#include <VisualDebugger/VdbServices/hkVdbClient.h>
#include <VisualDebugger/VdbServices/System/Connection/hkVdbConnection.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdInput.h>

#define DEBUG_LOG_IDENTIFIER "vdb.Cmd.Output"
#include <Common/Base/System/Log/hkLog.hxx>

hkVdbScopedStreamWriter::~hkVdbScopedStreamWriter()
{
    if ( m_flushMode )
    {
        bool flushed;
        m_owner->flush( flushed, m_flushMode == hkVdbCmdOutput::BLOCKING );
    }
    HK_ON_DEBUG( m_owner->m_cmdStarted = false; )
}

hkVdbCmdOutput::hkVdbCmdOutput( hkVdbCmdInput& input ) :
    hkVdbDefaultErrorReporter( &s_debugLog ),
    m_outputStream( hkRefNew<hkArrayStreamWriter>( new hkArrayStreamWriter( &m_outputBufferStorage, hkArrayStreamWriter::ARRAY_BORROW ) ) ),
    m_outputBufferBytesWritten( 0 ),
    m_input( &input ),
    m_flushToOffset( HK_INT32_MAX ),
    m_isReplaying( false ),
    m_acknowledgedFrames( 0 )
{
    HK_ON_DEBUG( m_cmdStarted = false );
}

hkVdbScopedStreamWriter hkVdbCmdOutput::startCmd( hkVdbCmdType::Enum typeIn, FlushMode mode )
{
    using namespace hkVdbCmdType;

    HK_ASSERT( 0x22440904, !m_cmdStarted, "You cannot start another cmd until you end the current one" );
    HK_ON_DEBUG( m_cmdStarted = true );

    if ( m_input->getProtocol() < 0 )
    {
        HK_ASSERT( 0x22440905, false, "Protocol hasn't been established" );
        return hkVdbScopedStreamWriter( *this, HK_NULL, mode );
    }

    if ( !m_isReplaying && ( typeIn != hkVdbCmdType::INVALID ) )
    {
        int type = patchType<Direction::CLIENT_TO_SERVER>( m_input->getProtocol(), typeIn );
        hkVdbOStream writer( m_outputStream );
        writer.write8u( hkUint8( type ) );
    }

    return hkVdbScopedStreamWriter( *this, m_outputStream, mode );
}

void hkVdbCmdOutput::onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection )
{
    if ( use == hkVdbConnectionUse::APPLICATION )
    {
        m_appConnection = &connection;
        m_flushToOffset = HK_INT32_MAX;
        m_acknowledgedFrames = 0;
    }
}

void hkVdbCmdOutput::onDisconnectingSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection, hkVdbSignalResult& result )
{
    if ( use == hkVdbConnectionUse::APPLICATION )
    {
        bool flushed;
        HK_VDB_VERIFY_SIGNALLED_REPORTER_OPERATION(
            flush( flushed, true ),
            *this,
            result );
        m_appConnection = HK_NULL;
    }
}

void hkVdbCmdOutput::onPlaybackInfoReceivedSignalInternal( const hkVdbPlaybackHandler::PlaybackInfo& info, hkVdbSignalResult& result )
{
    const hkBool32 wasReplaying = m_isReplaying;
    const hkBool32 isReplaying = info.m_targetFlags.anyIsSet( hkVdbPlaybackFlags::REPLAYING );

    // If we are starting replay, then mark where we should flush to the next time we flush.
    // Any data after this is occurring during replay and should be ignored.
    if ( !wasReplaying && isReplaying )
    {
        m_flushToOffset = m_outputStream->tell();
    }
    // If we are completing replay, then reset so we can start writing again.
    else if ( wasReplaying && !isReplaying )
    {
        m_outputStream->seek( 0, hkStreamWriter::STREAM_SET );
        m_flushToOffset = HK_INT32_MAX;
    }

    m_isReplaying = isReplaying;
}

hkResult hkVdbCmdOutput::flush( bool& flushed, bool blocking )
{
    // Get our writer.
    // Note: appConnections aren't required to have a writer (such as movie file replay).
    
    
    
    
    hkStreamWriter* appWriter = ( m_appConnection ) ? m_appConnection->getWriter() : HK_NULL;

    // If we don't have a writer, or we have nothing we should be flushing, simply ignore outgoing data
    if ( !appWriter || ( m_flushToOffset == 0 ) )
    {
        HK_ASSERT( 0x22440938, !m_outputBufferBytesWritten, "Contents in buffer wasn't flushed before losing writer or replaying" );
        m_outputStream->seek( 0, hkStreamWriter::STREAM_SET );
        return HK_SUCCESS;
    }

    
    const int flushToOffset = hkMath::min2( m_flushToOffset, m_outputStream->tell() );
    flushed = ( m_outputBufferBytesWritten == flushToOffset );
    if ( !flushed )
    {
        do
        {
            
            const int bytesToWrite = ( flushToOffset - m_outputBufferBytesWritten );
            const int bytesWritten = appWriter->write(
                reinterpret_cast< hkInt8* >( m_outputStream->getData() ) + m_outputBufferBytesWritten,
                bytesToWrite );
            m_outputBufferBytesWritten += bytesWritten;
            HK_VDB_VERIFY_REPORTER_CONDITION( appWriter->isOk(), *m_appConnection, hkVdbError::CONNECTION_WRITE_ERROR );
            flushed = ( m_outputBufferBytesWritten == flushToOffset );
        } while ( blocking && !flushed );

        appWriter->flush();
    }

    const hkBool32 bufferLimitReached = ( m_outputStream->getDataSize() > HK_VDB_MAX_OUTPUT_BUFFER_SIZE );
    if ( flushed || bufferLimitReached )
    {
        if ( bufferLimitReached )
        {
            int offset = flushToOffset;
            int newSize = ( offset - m_outputBufferBytesWritten );
            if ( newSize > HK_VDB_MAX_OUTPUT_BUFFER_SIZE )
            {
                HK_VDB_SIGNAL_ERROR_MSG(
                    0xedb00024,
                    hkVdbError::CONNECTION_WRITE_ERROR,
                    "The cmd output buffer has exceeded the max size of " << HK_VDB_MAX_OUTPUT_BUFFER_SIZE );
                return HK_FAILURE;
            }

            hkArray<char> newBuf( newSize );
            hkString::memCpy(
                newBuf.begin(),
                reinterpret_cast< char* >( m_outputStream->getData() ) + m_outputBufferBytesWritten,
                newBuf.getSize() );

            m_outputBufferStorage.swap( newBuf );
        }

        m_outputStream->seek( 0, hkStreamWriter::STREAM_SET );
        m_outputBufferBytesWritten = 0;
        m_flushToOffset = m_isReplaying ? 0 : HK_INT32_MAX;
    }

    return HK_SUCCESS;
}

hkResult hkVdbCmdOutput::ackFrame( const hkVdbFrame& frame )
{
    // We should always dispatch sequential commands/frames.  If this hits, we aren't doing that.
    // Because the acks are so critical, we try to recover with a while loop.
    HK_ASSERT( 0x22441080, m_isReplaying || ( frame.getFrameNumber() == m_acknowledgedFrames ), "The framework should not skip dispatching frames" );

    // Ignore steps during replay
    if ( m_isReplaying )
    {
        return HK_SUCCESS;
    }

    while ( frame.getFrameNumber() >= m_acknowledgedFrames )
    {
        if ( hkVdbScopedStreamWriter writer = startCmd(
            hkVdbCmdType::STEP_ACK,
            hkVdbCmdOutput::NON_BLOCKING ) )
        {
            m_acknowledgedFrames++;
        }
        else
        {
            return HK_FAILURE;
        }
    }
    return HK_SUCCESS;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
