// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Types/hkSignalSlots.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmd.h>

#include <VisualDebugger/VdbServices/System/Connection/hkVdbConnection.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbPlaybackHandler.h>




struct hkVdbCmd;
struct hkVdbSignalResult;
class hkStreamWriter;

//////////////////////////////////////////////////////////////////////////
/// This struct holds a frame of vdb command data.
/// These should not be created directly but instead access from the input.
//////////////////////////////////////////////////////////////////////////
struct HK_EXPORT_VDB hkVdbFrame
{
    HK_DECLARE_CLASS( hkVdbFrame, New );
    hkVdbFrame();
    hkVdbFrame( const hkVdbFrame& src ) :
        hkVdbFrame()
    {
#ifdef HK_DEBUG
        // This is only expected to insert a default impl.
        hkVdbFrame defaultFrame;
        HK_ASSERT_NO_MSG( 0x22441163, !src.m_cmdStorage[0].m_storage.getSize() );
        HK_ASSERT_NO_MSG( 0x22441165, !src.m_customStorage.getSize() );
        HK_ASSERT_NO_MSG( 0x22441166, !src.m_customStorageById.getSize() );
        HK_ASSERT_NO_MSG( 0x22441167, m_frameNumber == defaultFrame.m_frameNumber );
        HK_ASSERT_NO_MSG( 0x22441168, m_startTime == defaultFrame.m_startTime );
        HK_ASSERT_NO_MSG( 0x22441169, m_duration == defaultFrame.m_duration );
#endif
    }

    //
    // Iteration
    //

    typedef hkInt32 Iterator;
    static Iterator InvalidIterator;
    HK_INLINE Iterator getIterator( hkInt64 cmdIndex = -1 ) const;
    HK_INLINE Iterator getNext( Iterator i ) const;
    HK_INLINE const hkVdbCmd* getValue( Iterator i ) const;
    HK_INLINE hkBool32 isValid( Iterator i ) const;

    //
    // Query
    //

    /// The the frame number.
    HK_INLINE hkUint32 getFrameNumber() const { return m_frameNumber; }

    /// Get the number of cmds in this frame.
    HK_INLINE hkUint32 getNumCmds() const { hkUint32 s = 0; for ( int i = 0; i < m_cmdStorage.getSize(); i++ ) { s += m_cmdStorage[i].m_cmdOffsets.getSize(); } return s; }

    /// Get the size in bytes of the cmds in this frame.
    HK_INLINE hkUint32 getCmdBytes() const { hkUint32 s = 0; for ( int i = 0; i < m_cmdStorage.getSize(); i++ ) { s += m_cmdStorage[i].m_storage.getSize(); } return s; }

    /// Get the total size in bytes of this frame (included custom storage used by cache).
    HK_INLINE hkUint32 getTotalBytes() const { return getCmdBytes() + m_customStorage.getSize(); }

    /// Get the time this frame started (in vdb step units).
    HK_INLINE float getStartTime() const { return m_startTime; }

    /// Get the time this frame ended (in vdb step units).
    HK_INLINE float getEndTime() const { return m_startTime + hkMath::max2( m_duration, 0 ); }

    /// Get the frame's duration (in vdb step units).
    HK_INLINE float getDuration() const { return m_duration; }

    //
    // Custom Data
    // This storage is defined by calling hkVdbCmdInput::setCustomFrameBytes()
    //

    /// Get custom storage that was specified for the id as raw bytes.
    /// (see hkVdbCmdInput::setCustomFrameBytes)
    hkInt8* getCustomBytes( hkUint32 id, int& numBytesOut ) const;

    /// Get custom storage that was specified for the id as an array of Type objects.
    /// (see hkVdbCmdInput::setCustomFrameBytes)
    template<typename Type> Type* getCustomObjects( hkUint32 id, int numObjects = 1 ) const
    {
        const int expectedBytes = sizeof( Type ) * numObjects;
        int numBytes;
        if ( HK_VERY_LIKELY( hkInt8* customBytes = getCustomBytes( id, numBytes ) ) )
        {
            if ( HK_VERY_LIKELY( expectedBytes <= numBytes ) )
            {
                return reinterpret_cast<Type*>( customBytes );
            }
        }
        return HK_NULL;
    }

private:

    /// Peek at the current command we are reading.
    HK_INLINE const hkVdbCmd* peek() const { return m_currentCmd.peek(); }

    /// Read as much from the connection as is available and advance to the next command
    /// if we complete our current command.
    hkResult readAndAdvance( hkVdbConnection& connection, int protocol, const hkVdbCmd*& cmdOut );

    /// Reset the frame so it can be reused.
    /// No memory is deallocated.
    void reset();

    /// This struct is a block of cmd storage since a single frame can grow larger than hkArray can handle.
    struct CommandStorageBlock
    {
        CommandStorageBlock() : m_startingCmdIndex( 0 ) {}
        hkInt32 m_startingCmdIndex;
        hkArray<hkInt8> m_storage;
        hkArray<hkInt32> m_cmdOffsets;
    };

    /// This class represents a vdb command that isn't fully read yet.
    struct hkVdbPartialCmd
    {
        HK_DECLARE_CLASS( hkVdbPartialCmd, New );

        hkVdbPartialCmd( hkArray<CommandStorageBlock>* cmdStorage );

        /// Peek at the current, incomplete, vdb command.
        HK_INLINE const hkVdbCmd* peek() const;

        /// We are done with the current command, advance in the backing store to a new vdb command.
        /// Return the command we just advanced over (not the new one).
        HK_INLINE const hkVdbCmd* advance();

        /// Get the size of the vdb command header.
        HK_INLINE hkUint32 getHeaderSize() const;

        /// Get the size of the vdb command body (only valid after the header has been read).
        HK_INLINE hkUint32 getBodySize() const;

        /// Get the location in the current vdb command to write more data.
        HK_INLINE hkInt8* getWriteLoc();

        /// Set the num of bytes reserved for this vdb command.
        hkResult reserve( int numBytes );

        /// Set the number of bytes we've written to for this vdb command.
        HK_INLINE void setCompleted( int numBytes );

        /// Get the number of bytes we've written to for this vdb command.
        HK_INLINE hkUint32 getNumCompleted() const;

        /// Get the index of this cmd.
        HK_INLINE hkUint32 getCmdIndex() const { return hkUint32( currentStorage()->m_startingCmdIndex + currentStorage()->m_cmdOffsets.getSize() ); }

        /// Clear our completed bytes.
        HK_INLINE void clear();

    private:

        HK_INLINE CommandStorageBlock* currentStorage() const { return &(*m_cmdStorage)[m_currentStorageIndex]; }
        hkInt32 m_partialCmdStartIndex;
        int m_currentStorageIndex;
        hkArray<CommandStorageBlock>* m_cmdStorage;

        friend struct hkVdbFrame;
    };

    /// This class wraps a subset of bytes that live in an external hkArray of hkInt8s to be used for custom data.
    struct CustomStorage : public hkArray<hkInt8>
    {
        HK_DECLARE_CLASS( CustomStorage, New );
        CustomStorage( hkInt8* storage = HK_NULL, int size = 0 ) : hkArray<hkInt8>( storage, size, size ) {}
        CustomStorage( const CustomStorage& cs ) { operator=( cs ); }
        HK_INLINE CustomStorage& operator= ( const CustomStorage& cs )
        {
            CustomStorage* mutableThis = const_cast< CustomStorage* >( this );
            hkString::memCpy( mutableThis, &cs, sizeof( CustomStorage ) );
            return *mutableThis;
        }
    };

    /// Get the command storage block for the iterator
    
    
    HK_INLINE const CommandStorageBlock* getStorageFor( Iterator i ) const
    {
        for ( int storageIt = 0; storageIt < m_cmdStorage.getSize(); storageIt++ )
        {
            const CommandStorageBlock& block = m_cmdStorage[storageIt];
            int relativeI = ( i - block.m_startingCmdIndex );
            if ( relativeI >= 0 )
            {
                if ( relativeI < block.m_cmdOffsets.getSize() )
                {
                    return &block;
                }
            }
            else
            {
                return HK_NULL;
            }
        }
        return HK_NULL;
    }

    hkUint32 m_frameNumber;
    float m_startTime;
    float m_duration;

    
    
    hkArray<CommandStorageBlock> m_cmdStorage;
    hkVdbPartialCmd m_currentCmd;
    hkArray<hkInt8> m_customStorage;
    mutable hkVdbMap<hkUint32, CustomStorage> m_customStorageById;

    friend class hkVdbCmdInput;
};

//////////////////////////////////////////////////////////////////////////
/// This class holds all cmd input data.
/// It is divided into frames which are determined by vdb step cmds.
//////////////////////////////////////////////////////////////////////////
class HK_EXPORT_VDB hkVdbCmdInput : public hkVdbDefaultErrorReporter
{
public:

    HK_DECLARE_CLASS( hkVdbCmdInput, New );

    /// Create a new input buffer that will be buffer enough vdb frames to cover "duration" milliseconds.
    
    
    
    
    
    
    hkVdbCmdInput( float duration = HK_VDB_DEFAULT_INPUT_BUFFER_DURATION_MS );
    ~hkVdbCmdInput();

    /// Reset the buffer to an initial state with an optional new duration.
    /// If duration is not provided, the original duration is used.
    void reset( float duration = -1.0f );

    //
    // Advancing
    //

    /// Read command data from the connection.
    /// If the command is completed during the call, it is returned.
    /// Otherwise, HK_NULL is returned.
    hkResult readNext( hkVdbConnection& connection, const hkVdbFrame*& frameOut, const hkVdbCmd*& cmdOut );

    //
    // Iteration
    //

    typedef hkInt32 Iterator;
    static Iterator InvalidIterator;
    HK_INLINE Iterator getIterator( hkInt64 frame = -1 ) const;
    HK_INLINE Iterator getNext( Iterator i ) const;
    HK_INLINE Iterator getPrevious( Iterator i ) const;
    HK_INLINE const hkVdbFrame* getValue( Iterator i ) const;
    HK_INLINE hkBool32 isValid( Iterator i, bool includeCurrentFrame = false ) const;

    //
    // Query
    //

    /// Get the number of frames in the input.
    HK_INLINE hkUint32 getNumFrames() const;

    /// Get the duration of the entire input buffer in ms.
    HK_INLINE float getDuration() const { return m_duration; }

    /// Get the starting frame's number.
    HK_INLINE hkUint32 getStartingFrameNumber() const { return m_startingFrameNumber; }

    /// Get the number of the last frame we finished reading.
    HK_INLINE hkUint32 getLastCompletedFrameNumber() const { return hkUint32( hkMath::max2( hkInt64( m_startingFrameNumber + getNumFrames() ) - 1, 0 ) ); }

    /// Get the number of the frame we are currently reading.
    HK_INLINE hkUint32 getCurrentFrameNumber() const { hkUint32 frame = ( m_startingFrameNumber + getNumFrames() ); HK_ASSERT_NO_MSG( 0x22440932, !getNumFrames() || const_cast< hkVdbCmdInput* >( this )->current()->getFrameNumber() == frame ); return frame; }

    /// Get the number for the last frame that was processed (dispatched to the system).
    HK_INLINE hkUint32 getProcessingFrameNumber() const { return m_processedFrames; }

    /// Get the range of frame numbers in the input buffer.
    HK_INLINE hkResult getFrameRange( hkUint32& minFrameOut, hkUint32& maxFrameOut ) const;

    /// Returns true if the input buffer has filled.
    HK_INLINE hkBool32 isFull() const;

    /// Peek at the current frame we are working on.
    /// This is not available through the Iterator interface.
    HK_INLINE const hkVdbFrame* peek() const;

    /// Get the oldest frame in the buffer.
    HK_INLINE const hkVdbFrame* begin() const;

    /// Get the newest completed frame in the buffer.
    HK_INLINE const hkVdbFrame* back() const;

    /// Get a specific frame from the buffer.
    /// This can be the current frame, which is not available through the Iterator interface.
    HK_INLINE const hkVdbFrame* getFrame( int frame ) const;

    /// Get the protocol version of the commands in the buffer
    HK_INLINE int getProtocol() const { return m_protocol; }

    //
    // Custom Frame Storage
    //

    /// Enum of different initialization operations which can be performed on custom data storage
    /// when it's reserved for a new vdb frame.
    enum CustomStorageInitOp
    {
        /// Don't do anything to the reserved memory.
        UNINITIALIZED = 0,
        /// Zero out the reserved memory.
        ZERO = 1,
        /// Copy the memory from the previous frame (if available).
        COPY_FROM_PREVIOUS = 2
    };

    /// Info on reserving custom data storage
    struct CustomStorageReserveInfo
    {
        CustomStorageReserveInfo() : m_bytes( 0 ) {}
        hkUint32 m_id;
        int m_bytes;
        CustomStorageInitOp m_initOp;
    };

    /// Stop reserving custom storage for a particular id.
    HK_INLINE void unreserveCustomStorage( hkUint32 id );

    //
    // Signals
    //

    HK_DECLARE_SIGNAL( FrameCompletedSignal, hkSignal2< hkVdbCmdInput&, const hkVdbFrame& > );
    /// This signal occurs after a frame is completed (before we advance to the next frame).
    /// It's useful for setting up custom frame bytes for the next frame (which will occur
    /// between this signal and the FrameAdvancedSignal).
    FrameCompletedSignal m_frameCompleted;

    HK_DECLARE_SIGNAL( FrameAdvancedSignal, hkSignal2< hkVdbCmdInput&, const hkVdbFrame& > );
    /// This signal occurs after we've advanced to a new frame.
    FrameAdvancedSignal m_frameAdvanced;

    HK_DECLARE_SIGNAL( FrameEvictedSignal, hkSignal2< hkVdbCmdInput&, const hkVdbFrame& > );
    /// This signal occurs just before a frame is evicted (so it can be accessed).
    FrameEvictedSignal m_frameEvicted;

    HK_DECLARE_SIGNAL( PreCustomStorageAllocSignal, hkSignal2< const hkVdbFrame&, hkArray<CustomStorageReserveInfo>::Temp& > );
    /// This signal occurs just before we allocate custom storage for a new frame.
    /// It is used to reserve custom storage for new frames.
    PreCustomStorageAllocSignal m_preCustomStorageAlloc;

    HK_DECLARE_SIGNAL( PostCustomStorageAllocSignal, hkSignal1< const hkVdbFrame& > );
    /// This signal occurs just after we allocate custom storage for a new frame.
    /// Callbacks can use this to allocate the newly reserved storage, if desired.
    PostCustomStorageAllocSignal m_postCustomStorageAlloc;

    HK_DECLARE_SIGNAL( PreCustomStorageDeallocSignal, hkSignal1< const hkVdbFrame& > );
    /// This signal occurs just before we deallocate the custom storage for a frame.
    /// Callbacks should clean up and custom-storage-dependent data.
    PreCustomStorageDeallocSignal m_preCustomStorageDealloc;

    //
    // Internal use
    //

    void onFrameEvictedSignal( hkVdbCmdInput& input, const hkVdbFrame& frame );
    void onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection );
    void onStepCompletedSignal( const hkVdbFrame& frame, const hkVdbCmd& cmd );

private:

    /// Information about reserving new memory.
    struct MemoryReservationInfo
    {
        int m_bytesToReserve;
        CustomStorageInitOp m_initOp;
    };

    /// Update the remote version protocol
    hkResult updateProtocol( const hkVdbCmd& cmd );

    /// Advance the buffered to the next frame.
    /// "timeStep" should be the time the last step took to complete.
    hkVdbFrame* advance( float timeStep );

    /// Get the current frame we are working on.
    hkVdbFrame* current();

    /// This function sets the number of custom bytes that will be allocated for a particular id
    /// on any vdb frames that have not yet had their custom bytes allocated.
    
    
    void setCustomFrameReserveBytes( hkUint32 id, int numBytesToReserve, CustomStorageInitOp initOp );

    float m_duration;
    hkInt32 m_startingFrameIdx;
    hkUint32 m_startingFrameNumber;
    hkInt32 m_currentFrameIdx;
    hkUint32 m_processedFrames;
    hkArray<hkVdbFrame> m_frameStorage;
    hkVdbMap<hkUint32, MemoryReservationInfo> m_customReserveSizes;
    int m_protocol;

    // This violates the low-levelness of this class, but is needed for some important debug sanity checks
    HK_ON_DEBUG( friend class hkVdbPlaybackHandler; )
};

#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdInput.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
