// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbServices/hkVdbServices.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdInput.h>

#include <Common/Base/System/Io/Reader/hkStreamReader.h>
#include <Common/Visualize/hkVersionReporter.h>

#include <VisualDebugger/VdbServices/System/Connection/hkVdbConnection.h>

#ifndef HK_VDB_USE_HASH_MAP
#include <Common/Base/Container/PointerMap/hkMap.h>
#include <Common/Base/Container/PointerMap/hkMap.hxx>
template class HK_EXPORT_COMMON hkMapBase<hkUint32, hkVdbFrame::CustomStorage>;
template class HK_EXPORT_COMMON hkMap<hkUint32, hkVdbFrame::CustomStorage>;
template class HK_EXPORT_COMMON hkMapBase<hkUint32, hkVdbCmdInput::MemoryReservationInfo>;
template class HK_EXPORT_COMMON hkMap<hkUint32, hkVdbCmdInput::MemoryReservationInfo>;
#endif

#define DEBUG_LOG_IDENTIFIER "vdb.Cmd.Input"
#include <Common/Base/System/Log/hkLog.hxx>



// Useful debugging aids
#if 0

static int evicted = -1;
static int targetStartFrame = 250;
static int targetEndFrame = 300;

void beforeEvict( hkVdbCmdInput* cache, const hkVdbFrame* frame )
{
    evicted++;

    if ( cache->getDuration() != -1 && evicted >= targetStartFrame && evicted < targetEndFrame )
    {
        hkStringBuf filename;
        filename.printf( "Movies/vdbmovie%i.before.hkm", evicted );
        hkRefPtr<hkStreamWriter> writer = hkFileSystem::getInstance().openWriter( filename );
        filename.printf( "Movies/vdbmovie%i.before.txt", evicted );
        g_vdbDebugFilename = filename;
        cache->save( writer );
    }
}

void afterEvict( hkVdbCmdInput* cache, const hkVdbFrame* frame )
{
    if ( cache->getDuration() != -1 && evicted >= targetStartFrame && evicted < targetEndFrame )
    {
        hkStringBuf filename;
        filename.printf( "Movies/vdbmovie%i.after.hkm", evicted );
        hkRefPtr<hkStreamWriter> writer = hkFileSystem::getInstance().openWriter( filename );
        filename.printf( "Movies/vdbmovie%i.after.txt", evicted );
        g_vdbDebugFilename = filename;
        cache->save( writer );
    }
}

#endif

//////////////////////////////////////////////////////////////////////////
// hkVdbPartialCmd
//////////////////////////////////////////////////////////////////////////

hkVdbFrame::hkVdbPartialCmd::hkVdbPartialCmd( hkArray<CommandStorageBlock>* cmdStorage ) :
    m_partialCmdStartIndex( 0 ),
    m_currentStorageIndex( cmdStorage->getSize() - 1 ),
    m_cmdStorage( cmdStorage )
{}

hkResult hkVdbFrame::hkVdbPartialCmd::reserve( int numBytes )
{
    CommandStorageBlock* storageBlock = currentStorage();
    int capacity = storageBlock->m_storage.getCapacity();
    int reqBytes = ( m_partialCmdStartIndex + numBytes );
    int defaultReserveBytes = ( reqBytes <= capacity ) ? reqBytes : ( storageBlock->m_storage.RESERVE_GROWTH_FACTOR * capacity );
    hkBool32 defaultReserveOk = ( ( defaultReserveBytes & storageBlock->m_storage.CAPACITY_MASK ) == defaultReserveBytes );
    hkBool32 reqBytesReserveOk = ( ( reqBytes & storageBlock->m_storage.CAPACITY_MASK ) == reqBytes );

    if ( HK_VERY_LIKELY( defaultReserveOk || reqBytesReserveOk ) )
    {
        if ( defaultReserveOk )
        {
            // Standard resize will be okay.
            // If we fail here, it means our memory allocator can't provide the memory.
            return storageBlock->m_storage.reserve( reqBytes );
        }
        else // reqBytesReserveOk
        {
            // Just reserve the max so subsequent calls won't cause allocations.
            // If we fail here, it means our memory allocator can't provide the memory.
            return storageBlock->m_storage.reserveExactly( storageBlock->m_storage.CAPACITY_MASK );
        }
    }
    else if ( storageBlock->m_cmdOffsets.getSize() )
    {
        HK_ASSERT_NO_MSG( 0x22441341, storageBlock == &m_cmdStorage->back() );
        CommandStorageBlock* newStorageBlock = &m_cmdStorage->expandOne();
        storageBlock = currentStorage();
        {
            newStorageBlock->m_startingCmdIndex = ( storageBlock->m_startingCmdIndex + storageBlock->m_cmdOffsets.getSize() );
            newStorageBlock->m_storage.append(
                storageBlock->m_storage.begin() + m_partialCmdStartIndex,
                storageBlock->m_storage.getSize() - m_partialCmdStartIndex );
            storageBlock->m_storage.setSize( m_partialCmdStartIndex );
        }
        m_currentStorageIndex = ( m_cmdStorage->getSize() - 1 );
        m_partialCmdStartIndex = 0;
        return newStorageBlock->m_storage.reserve( numBytes );
    }
    else
    {
        HK_ASSERT( 0x22441342, false, "Single cmd is larger than hkArray can allocate" );
        return HK_FAILURE;
    }
}

//////////////////////////////////////////////////////////////////////////
// hkVdbFrame
//////////////////////////////////////////////////////////////////////////

hkVdbFrame::Iterator hkVdbFrame::InvalidIterator = -1;

hkVdbFrame::hkVdbFrame() :
    m_frameNumber( 0 ),
    m_startTime( 0 ),
    m_currentCmd( &m_cmdStorage )
{
    reset();
}

hkInt8* hkVdbFrame::getCustomBytes( hkUint32 id, int& numBytesOut ) const
{
    hkVdbMap<hkUint32, CustomStorage >::Iterator iter = m_customStorageById.findKey( id );
    if ( HK_VERY_LIKELY( m_customStorageById.isValid( iter ) ) )
    {
        CustomStorage& reservedMemory = m_customStorageById.getValue( iter );
        numBytesOut = reservedMemory.getSize();
        return reservedMemory.begin();
    }
    return HK_NULL;
}

hkResult hkVdbFrame::readAndAdvance( hkVdbConnection& connection, int protocol, const hkVdbCmd*& cmdOut )
{
    cmdOut = HK_NULL;

    hkStreamReader* reader = connection.getReader();
    if ( HK_VERY_UNLIKELY( !reader || !reader->isOk() ) )
    {
        return HK_FAILURE;
    }

    const int headerBytes = m_currentCmd.getHeaderSize();
    int completedBytes = m_currentCmd.getNumCompleted();

    // Read the header first
    if ( completedBytes < headerBytes )
    {
        const int toRead = headerBytes - completedBytes;
        HK_VDB_FAIL_IF_OPERATION_FAILED( m_currentCmd.reserve( completedBytes + toRead ) );
        completedBytes += reader->read( m_currentCmd.getWriteLoc(), toRead );
        m_currentCmd.setCompleted( completedBytes );
    }

    // If we've completed the header, we can move onto the body of the vdb command
    if ( completedBytes >= headerBytes )
    {
        const int bodyBytes = m_currentCmd.getBodySize();

        const hkVdbCmd* partialCmd = m_currentCmd.peek();
        if ( HK_VERY_UNLIKELY( !partialCmd->isOk( false ) ) )
        {
            HK_WARN_ALWAYS( 0x654ABE2F, "Received a corrupt command over the network. The command has either"
                " a bad type or is larger than the max allowed size of " << ( HK_VDB_MAX_CMD_SIZE / 1024 / 1024 ) << "MB" );

            
            connection.disconnect();
            return HK_FAILURE;
        }

        if ( HK_VERY_UNLIKELY( bodyBytes == 0 ) )
        {
            
            
            m_currentCmd.clear();
            return HK_SUCCESS;
        }

        hkInt32 totalBytes = bodyBytes + headerBytes;
        hkInt32 toReadRemaining = totalBytes - completedBytes;
        HK_VDB_FAIL_IF_OPERATION_FAILED( m_currentCmd.reserve( completedBytes + toReadRemaining ) );
        completedBytes += reader->read( m_currentCmd.getWriteLoc(), toReadRemaining );
        m_currentCmd.setCompleted( completedBytes );

        if ( completedBytes >= totalBytes )
        {
            // Write our meta data
            {
                // Write our patched type
                {
                    const hkUint8 serverType = peek()->getServerType();
                    const hkUint8 type = hkUint8( hkVdbCmdType::patchType<hkVdbCmdType::Direction::SERVER_TO_CLIENT>( protocol, serverType ) );

                    completedBytes += sizeof( type );
                    HK_VDB_FAIL_IF_OPERATION_FAILED( m_currentCmd.reserve( completedBytes ) );
                    *m_currentCmd.getWriteLoc() = type;
                    m_currentCmd.setCompleted( completedBytes );
                }

                // Write our index in the frame, this is used for cmd order comparisons during playback
                {
                    const hkUint32 commandIdx = m_currentCmd.getCmdIndex();

                    completedBytes += sizeof( commandIdx );
                    HK_VDB_FAIL_IF_OPERATION_FAILED( m_currentCmd.reserve( completedBytes ) );
                    *reinterpret_cast<hkUint32*>( m_currentCmd.getWriteLoc() ) = commandIdx;
                    m_currentCmd.setCompleted( completedBytes );
                }
            }

            // Return the completed command and advance to the next one.
            cmdOut = m_currentCmd.advance();

            
            HK_ASSERT( 0x22440875, cmdOut->isOk(), "Cmd is not ok" );
        }
    }

    return HK_VERY_LIKELY( reader->isOk() ) ? HK_SUCCESS : HK_FAILURE;
}

void hkVdbFrame::reset()
{
    m_cmdStorage.clear(); // ensures we get dtor followed by ctor for setSize( 1 )
    m_cmdStorage.setSize( 1 );
    new ( &m_currentCmd ) hkVdbPartialCmd( &m_cmdStorage );
    HK_ASSERT_NO_MSG( 0x22441361, m_currentCmd.m_currentStorageIndex >= 0 );
    HK_ASSERT_NO_MSG( 0x22440809, m_currentCmd.getNumCompleted() == 0 );
    m_customStorage.clear();
    m_customStorageById.clear();
    m_duration = -1.0f;
}

//////////////////////////////////////////////////////////////////////////
// hkVdbCmdInput
//////////////////////////////////////////////////////////////////////////

hkVdbCmdInput::Iterator hkVdbCmdInput::InvalidIterator = -1;

hkVdbCmdInput::hkVdbCmdInput( float duration ) :
hkVdbDefaultErrorReporter( &s_debugLog ),
m_duration( duration ),
m_processedFrames( 0 ),
m_protocol( -1 )
{
    reset();
    HK_SUBSCRIBE_TO_SIGNAL( m_frameEvicted, this, hkVdbCmdInput );
}

hkVdbCmdInput::~hkVdbCmdInput()
{}

void hkVdbCmdInput::reset( float duration )
{
    if ( duration >= 0.0f )
    {
        m_duration = duration;
    }

    m_startingFrameIdx = 0;
    m_startingFrameNumber = 0;
    m_currentFrameIdx = 0;
    m_processedFrames = 0;
    m_frameStorage.clear(); // ensures we get dtor followed by ctor for setSize( 1 )
#if 0
    
    // Find a way to do some dynamic prediction of needed memory?
    m_frameStorage.setSize( int( hkMath::ceil( m_duration / HK_VDB_AVG_FRAME_DURATION ) ) );
    for ( int i = 0; i < m_frameStorage.getSize(); i++ )
    {
        m_frameStorage[i].m_commandStorage.reserve( HK_VDB_AVG_FRAME_MEMORY );
        m_frameStorage[i].m_customStorage.reserve( HK_VDB_AVG_FRAME_MEMORY );
    }
    m_frameStorage.setSize( 1 );
#else
    m_frameStorage.reserve( int( hkMath::ceil( m_duration / HK_VDB_AVG_FRAME_DURATION ) ) );
    m_frameStorage.setSize( 1 );
#endif
}

hkResult hkVdbCmdInput::readNext( hkVdbConnection& connection, const hkVdbFrame*& frameOut, const hkVdbCmd*& cmdOut )
{
    hkVdbFrame* frame = current();
    if ( HK_VERY_UNLIKELY( frame->readAndAdvance( connection, m_protocol, cmdOut ).isFailure() ) )
    {
        if ( connection.getError() )
        {
            signalError( connection, hkVdbError::CONNECTION_READ_ERROR );
        }
        else
        {
            signalError( 0xedb00149, hkVdbError::CMD_BUFFER_ERROR, "Could not store cmd in input buffer" );
        }
    }

    if ( cmdOut )
    {
        HK_VDB_FAIL_IF_OPERATION_FAILED( updateProtocol( *cmdOut ) );

        if ( cmdOut->getType() == hkVdbCmdType::STEP )
        {
            hkVdbLocalIStream reader( cmdOut->getData(), cmdOut->getDataSize() );
            advance( reader.readFloat32() );
            frameOut = back();
        }
        else
        {
            frameOut = frame;
        }
    }

    return HK_SUCCESS;
}

void hkVdbCmdInput::onFrameEvictedSignal( hkVdbCmdInput& input, const hkVdbFrame& frame )
{
    m_preCustomStorageDealloc.fire( frame );
}

void hkVdbCmdInput::onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection )
{
    if ( use == hkVdbConnectionUse::APPLICATION )
    {
        m_protocol = -1;
        reset();
    }
}

void hkVdbCmdInput::onStepCompletedSignal( const hkVdbFrame& frame, const hkVdbCmd& cmd )
{
    // We should always dispatch sequential commands/frames.  If this hits, we aren't doing that.
    // Because the custom data reserves are so critical, we try to recover with a while loop.
    HK_ASSERT( 0x22441081, ( frame.getFrameNumber() <= m_processedFrames ), "The framework should not skip dispatching frames" );

    while ( frame.getFrameNumber() >= m_processedFrames )
    {
        // Get the nextFrame, this will always succeed because if we completed a frame
        // we've advanced the input to a new frame for processing.
        Iterator frameIt = getIterator( m_processedFrames );
        const hkVdbFrame* previousFrame = getValue( frameIt );
        hkVdbFrame* nextFrame = const_cast<hkVdbFrame*>( getValue( getNext( frameIt ) ) );
        HK_ASSERT( 0x22440983, nextFrame, "Should not have completed step without allocating next frame" );

        // Signal before alloc
        {
            hkArray<CustomStorageReserveInfo>::Temp reserveInfos;
            m_preCustomStorageAlloc.fire( *nextFrame, reserveInfos );

            // Update our reserve data
            for ( int i = 0; i < reserveInfos.getSize(); i++ )
            {
                CustomStorageReserveInfo& reserveInfo = reserveInfos[i];
                setCustomFrameReserveBytes( reserveInfo.m_id, reserveInfo.m_bytes, reserveInfo.m_initOp );
            }
        }

        // Setup frames custom data storage.
        if ( m_customReserveSizes.getSize() )
        {
            // Compute/set the total number of custom bytes we need for this next frame.
            int totalCustomReserveSize = 0;
            for ( hkVdbMap<hkUint32, MemoryReservationInfo>::Iterator iter = m_customReserveSizes.getIterator();
                m_customReserveSizes.isValid( iter );
                iter = m_customReserveSizes.getNext( iter ) )
            {
                const MemoryReservationInfo& info = m_customReserveSizes.getValue( iter );
                totalCustomReserveSize += HK_NEXT_MULTIPLE_OF( 16, info.m_bytesToReserve );
            }
            nextFrame->m_customStorage.setSize( totalCustomReserveSize );

            // Create custom storage blobs for each id which point to the underlying data buffer
            hkInt8* customStorageOffset = nextFrame->m_customStorage.begin();
            for ( hkVdbMap<hkUint32, MemoryReservationInfo>::Iterator iter = m_customReserveSizes.getIterator();
                m_customReserveSizes.isValid( iter );
                iter = m_customReserveSizes.getNext( iter ) )
            {
                const hkUint32 key = m_customReserveSizes.getKey( iter );
                const MemoryReservationInfo& info = m_customReserveSizes.getValue( iter );
                nextFrame->m_customStorageById.insert(
                    key,
                    hkVdbFrame::CustomStorage(
                        customStorageOffset,
                        info.m_bytesToReserve ) );

                HK_ASSERT_NO_MSG( 0x22440816, nextFrame->m_customStorageById.getWithDefault( key, hkVdbFrame::CustomStorage() ).begin() == customStorageOffset );

                switch ( info.m_initOp )
                {
                    case COPY_FROM_PREVIOUS:
                    {
                        int previousNumBytes;
                        if ( hkInt8* previousCustomData = previousFrame->getCustomBytes( key, previousNumBytes ) )
                        {
                            
                            
                            
                            
                            
                            hkString::memCpy16(
                                customStorageOffset,
                                previousCustomData,
                                HK_NEXT_MULTIPLE_OF( 16, hkMath::min2( info.m_bytesToReserve, previousNumBytes ) ) >> 4 );
                        }
                        break;
                    }

                    case ZERO:
                    {
                        hkString::memClear16(
                            customStorageOffset,
                            HK_NEXT_MULTIPLE_OF( 16, info.m_bytesToReserve ) >> 4 );
                        break;
                    }

                    // No-ops
                    default:
                        HK_ASSERT_NO_MSG( 0x22440817, false );
                    case UNINITIALIZED:
                        break;
                }

                customStorageOffset += HK_NEXT_MULTIPLE_OF( 16, info.m_bytesToReserve );
            }
        }

        m_postCustomStorageAlloc.fire( *nextFrame );

        m_processedFrames++;
    }
}

hkResult hkVdbCmdInput::updateProtocol( const hkVdbCmd& cmd )
{
    const hkVdbCmdType::Enum type = cmd.getType();

    if ( type == hkVdbCmdType::SEND_SERVER_INFO )
    {
        hkVdbLocalIStream protocolReader( cmd.getData(), cmd.getDataSize() );
        int protocolVersion = protocolReader.read32();
        if ( ( protocolVersion > 0 ) && ( protocolVersion <= hkVersionReporter::m_protocolVersion ) )
        {
            m_protocol = protocolVersion;
            return HK_SUCCESS;
        }
        else
        {
            HK_VDB_SIGNAL_ERROR_MSG( 0xedb00106, hkVdbError::PROTOCOL_ERROR, "A corrupt protocol was encountered" );
            return HK_FAILURE;
        }
    }

    // It's unsafe to proceed if we don't have a protocol version.
    // However, custom/internal cmds might be used before the protocol custom behavior, so we allow those.
    HK_VDB_VERIFY_CONDITION_MSG(
        ( type == hkVdbCmdType::INVALID ) ||
        ( type == hkVdbCmdType::CUSTOM ) ||
        ( type == hkVdbCmdType::CLIENT_INTERNAL ) ||
        ( m_protocol != -1 ),
        0xedb00131,
        hkVdbError::PROTOCOL_ERROR,
        "The first cmd received must establish a remove protocol version" );

    return HK_SUCCESS;
}

hkVdbFrame* hkVdbCmdInput::advance( float timeStep )
{
    // Gather information about current state.
    hkVdbFrame* currentFrame = current();
    currentFrame->m_duration = timeStep;
    m_frameCompleted.fire( *this, *currentFrame );

    if ( !isFull() )
    {
        // We haven't filled our buffer, we are still growing it.
        // This can happen during initial fill or if we get a shorter frame after getting longer frames.
        // So we use insertAt to accommodate both cases.
        m_frameStorage.insertAt( m_currentFrameIdx + 1, hkVdbFrame() );

        // If we are currently wrapped, we need to push our starting frame idx forward.
        m_startingFrameIdx += bool( m_startingFrameIdx > m_currentFrameIdx );

        // We may have resized, so re-get our current frame pointer
        // Note: backing store of frame's partial command may not be correct as it's a pointer to the frame's member m_commandStorage,
        // but we don't care because we are moving onto another frame and we always call reset() on the frame we will be overwriting.
        currentFrame = current();
    }
    else
    {
        // Evict frames as long as we have time in the buffer to do so.
        // More than one frame could get evicted in variable time step scenarios.
        const hkVdbFrame* oldestFrame = begin();
        do
        {
            m_frameEvicted.fire( *this, *oldestFrame );
            m_startingFrameIdx = ( m_startingFrameIdx + 1 ) % m_frameStorage.getSize();
            m_startingFrameNumber++;
            oldestFrame = begin();
        } while ( ( currentFrame->m_startTime - oldestFrame->m_startTime - oldestFrame->m_duration ) > m_duration );
    }

    // Increment the index (wrapping around if necessary) to advance to the next frame.
    m_currentFrameIdx = ( m_currentFrameIdx + 1 ) % m_frameStorage.getSize();
    HK_ASSERT_NO_MSG( 0x22440815, back() == currentFrame );
    HK_ASSERT_NO_MSG( 0x22440842, m_startingFrameIdx != m_currentFrameIdx );

    // Update the next frame we just advanced to.
    hkVdbFrame* nextFrame = current();
    nextFrame->reset();
    nextFrame->m_startTime = ( currentFrame->m_startTime + currentFrame->m_duration );
    nextFrame->m_frameNumber = ( currentFrame->m_frameNumber + 1 );

    m_frameAdvanced.fire( *this, *nextFrame );

    return nextFrame;
}

void hkVdbCmdInput::setCustomFrameReserveBytes( hkUint32 id, int numBytesToReserve, CustomStorageInitOp initOp )
{
    HK_ASSERT( 0x22440814, numBytesToReserve >= 0, "Negative numBytesToReserve" );
    if ( numBytesToReserve <= 0 )
    {
        m_customReserveSizes.remove( id );
    }
    else
    {
        MemoryReservationInfo info = { numBytesToReserve, initOp };
        m_customReserveSizes.insert( id, info );
    }
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
