// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbServices/hkVdbServices.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdDispatcher.h>

#define DEBUG_LOG_IDENTIFIER "vdb.Cmd.Dispatcher"
#include <Common/Base/System/Log/hkLog.hxx>

namespace
{
    hkResult ignoreCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
        return HK_SUCCESS;
    }
}

hkVdbCmdDispatcher::hkVdbCmdDispatcher() :
    hkVdbDefaultErrorReporter( &s_debugLog ),
    m_dispatchTable( hkVdbCmd::getMaxTypeValue() + 1 )
{
    // Register no-ops for framework or ignored commands.
    // These don't *need* to be handled to be of use, as they are interpretted elsewhere.
    registerHandler( hkVdbCmdType::INVALID, ignoreCmdFunc, HK_NULL, HK_NULL );
    registerHandler( hkVdbCmdType::STEP, ignoreCmdFunc, HK_NULL, HK_NULL );
    registerHandler( hkVdbCmdType::SEND_SERVER_INFO, ignoreCmdFunc, HK_NULL, HK_NULL );
    registerHandler( hkVdbCmdType::CUSTOM, ignoreCmdFunc, HK_NULL, HK_NULL );
    registerHandler( hkVdbCmdType::CLIENT_INTERNAL, ignoreCmdFunc, HK_NULL, HK_NULL );
    // Register no-ops for deprecated cmds that just go into the stream (for now).
    registerHandler( hkVdbCmdType::DEPRECATED_LIVE_OBJECT, ignoreCmdFunc, HK_NULL, HK_NULL );
    HK_ON_DEBUG( m_lastDispatchedCmd = HK_NULL; )
}

hkResult hkVdbCmdDispatcher::dispatch(
    const hkVdbFrame& frame,
    const hkVdbCmd& cmd,
    int protocol,
    hkVdbProcessInfo* processInfo )
{
    HK_ASSERT( 0x22440874, cmd.isOk(), "Cmd is not ok" );

    const hkVdbCmdType::Enum type = cmd.getType();
    const hkUint8 serverType = cmd.getServerType();

    if ( HK_VERY_UNLIKELY( type == hkVdbCmdType::ERROR_MESSAGE ) )
    {
        hkVdbLocalIStream istream( cmd.getData(), cmd.getDataSize() );
        hkUint32 id;
        hkStringBuf msg;
        hkVdbCmdType::readErrorCmd( *istream.getStreamReader(), id, msg );
        HK_VDB_SIGNAL_ERROR_MSG( id, hkVdbError::CMD_ERROR, msg.cString() );
        HK_ON_DEBUG( m_lastDispatchedCmd = &cmd; )
        return HK_FAILURE;
    }

    DispatchEntry& entry = m_dispatchTable[type];

    if ( HK_VERY_LIKELY( entry.m_func ) )
    {
        hkVdbSignalResult result;
        m_cmdDispatching.fire( frame, cmd, protocol, result );
        HK_VDB_VERIFY_SIGNAL_RESULT( result );

        hkVdbLocalIStream reader( cmd.getData(), cmd.getDataSize() );
        if ( HK_VERY_UNLIKELY( entry.m_func( frame, cmd, protocol, reader, processInfo, entry.m_handle ).isFailure() ) )
        {
            if ( entry.m_errorReporter && entry.m_errorReporter->getError() )
            {
                HK_VDB_SIGNAL_REPORTER_ERROR( *entry.m_errorReporter, hkVdbError::CMD_HANDLER_ERROR );
            }
            else
            {
                HK_VDB_VERIFY_CONDITION_MSG(
                    reader.isOk(),
                    0xedb00000,
                    hkVdbError::CMD_HANDLER_READ_ERROR,
                    "Stream corruption after reading cmd [c" << type << "|s" << serverType );

                HK_VDB_SIGNAL_ERROR_MSG(
                    0xedb00001,
                    hkVdbError::CMD_HANDLER_ERROR,
                    "General error after reading cmd [c" << type << "|s" << serverType );
            }
            HK_ON_DEBUG( m_lastDispatchedCmd = &cmd; )
            return HK_FAILURE;
        }

        m_cmdDispatched.fire( frame, cmd, protocol, result );
        HK_ON_DEBUG( m_lastDispatchedCmd = &cmd; )
        HK_VDB_VERIFY_SIGNAL_RESULT( result );
        return HK_SUCCESS;
    }

    HK_VDB_SIGNAL_ERROR_MSG(
        0xedb00002,
        hkVdbError::CMD_HANDLER_NOT_REGISTERED,
        "There is not handler registered for the cmd [c" << int( type ) << "|s" << int( serverType ) );

    HK_ON_DEBUG( m_lastDispatchedCmd = &cmd; )
    return HK_FAILURE;
}

hkResult hkVdbCmdDispatcher::registerHandler( hkVdbCmdType::Enum type, hkVdbCmdHandlerFunc func, hkVdbErrorReporter* errorReporter, void* userHandle )
{
    DispatchEntry& entry = m_dispatchTable[type];

    HK_VDB_VERIFY_CONDITION_MSG(
        !entry.m_func || ( entry.m_func == &ignoreCmdFunc ),
        0xedb00003,
        hkVdbError::CMD_HANDLER_REGISTRATION_ERROR,
        "Handler is already registered for cmd [" << type << "]" );

    entry.m_func = func;
    entry.m_errorReporter = errorReporter;
    entry.m_handle = userHandle;
    return HK_SUCCESS;
}

hkResult hkVdbCmdDispatcher::unregisterHandler( hkVdbCmdType::Enum type )
{
    DispatchEntry& entry = m_dispatchTable[type];

    HK_VDB_VERIFY_CONDITION_MSG(
        entry.m_func,
        0xedb00004,
        hkVdbError::CMD_HANDLER_REGISTRATION_ERROR,
        "No handler registered for cmd [" << type << "]" );

    entry.m_func = HK_NULL;
    HK_ON_DEBUG( entry.m_errorReporter = HK_NULL );
    HK_ON_DEBUG( entry.m_handle = HK_NULL );
    return HK_SUCCESS;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
