// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Memory/System/hkMemorySystem.h>
#include <Common/Base/System/Stopwatch/hkStopwatch.h>
#include <Common/Base/System/Stopwatch/hkWindowedAverage.h>
#include <Common/Base/Monitor/MonitorStreamAnalyzer/hkMonitorStreamAnalyzer.h>

#include <Common/Visualize/hkVisualDebuggerCmdType.h>

#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbCmdHandler.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbSetupHandler.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbPlaybackHandler.h>

struct hkVdbFrame;
struct hkVdbCmd;
class hkVdbSetupHandler;
class hkVdbPlaybackHandler;
namespace hkSerialize { class ReadFormat; }

//////////////////////////////////////////////////////////////////////////
/// A handler for stats-related cmds.
//////////////////////////////////////////////////////////////////////////
class HK_EXPORT_VDB hkVdbStatsHandler : public hkVdbCmdHandler<hkVdbCmdHandlerType::STATS>
{
public:
    HK_DECLARE_CLASS( hkVdbStatsHandler, New );

    /// Create a new stats handler which optionally uses an hkPersistableReadFormat internally.
    hkVdbStatsHandler( bool persistable = false );

    //
    // hkVdbCmdHandler interface.
    //

    virtual hkResult waitForCompletion() HK_OVERRIDE;
    virtual hkResult registerSelf( hkVdbClient& client ) HK_OVERRIDE;
    virtual hkResult unregisterSelf( hkVdbClient& client ) HK_OVERRIDE;
    virtual void onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection ) HK_OVERRIDE;

    //
    // Control and Query
    //

    
    /// Sets the size of the sample window for computing frame performance stats.
    void setFrameSampleWindowSize( hkUint16 windowSize ) { m_frameStats.m_stepStats.reset( windowSize ); m_frameStats.m_vdbStepStats.reset( windowSize ); }

    /// Gets the size of the sample window for computing frame performance stats.
    hkUint16 getFrameSampleWindowSize() const { return hkUint16( m_frameStats.m_stepStats.getWindowSize() ); }

    struct PerfStats;
    /// Retrieve current stats relating to detailed performance on the server.
    HK_INLINE const PerfStats& getPerfStats() const { return m_perfStats; }

    struct MemStats;
    /// Retrieve current stats relating to memory use on the server.
    HK_INLINE const MemStats& getMemStats() const { return m_memStats; }

    struct FrameStats;
    /// Retrieve current stats relating to overall timings/performance for
    /// frame throughput (server and client related).
    HK_INLINE const FrameStats& getFrameStats() const { return m_frameStats; }
    

    //
    // Signals
    //

    
    /// Server stats for detailed performance analysis of the server.
    struct PerfStats
    {
        HK_DECLARE_CLASS( PerfStats, New );
        PerfStats() : m_numThreads( 0 ) {}
        ~PerfStats() { reset(); }

        /// Filled out with server info with the format/config of the stream's information.
        hkMonitorStream::CommandStreamConfig m_streamConfig;

        /// Raw monitor streams from the server.
        hkArray< hkArray<char> > m_monitorStreams;

        /// A frame tree of stats built from m_threadRootNodes.
        hkMonitorStreamAnalyzer::Tree m_statsTree;

        /// The root nodes for each thread built from m_monitorStreams.
        hkArray<hkMonitorStreamAnalyzer::ThreadRootNodes> m_threadRootNodes;

        /// Filled out with server info to interpret a detailed frame of stats.
        hkMonitorStreamFrameInfo m_frameInfo;

    protected:
        HK_INLINE void cleanupPrevious()
        {
            for ( int ti = 0; ti < m_threadRootNodes.getSize(); ++ti )
            {
                delete m_threadRootNodes[ti].m_node;
            }
            m_threadRootNodes.clear();
            m_monitorStreams.clear();
            m_statsTree.clearTags();
        }
        HK_INLINE void resetStats()
        {
            cleanupPrevious();
            m_statsTree.~Tree();
            new ( &m_statsTree ) hkMonitorStreamAnalyzer::Tree();
        }
        HK_INLINE void reset()
        {
            resetStats();
            m_streamConfig = hkMonitorStream::CommandStreamConfig();
            m_frameInfo = hkMonitorStreamFrameInfo();
        }
        HK_INLINE void setNumThreads( int numThreads )
        {
            if ( numThreads != m_numThreads )
            {
                resetStats();
                m_numThreads = numThreads;
            }
            m_threadRootNodes.reserve( numThreads );
            m_monitorStreams.setSize( numThreads );
        }
        int m_numThreads;
        friend class hkVdbStatsHandler;
    };

    /// Stats about server memory use.
    struct MemStats : hkMemorySystem::MemoryStatistics
    {
        HK_DECLARE_CLASS( MemStats, New );

    protected:
        HK_INLINE void reset()
        {
            this->~MemStats();
            new ( this ) hkMemorySystem::MemoryStatistics();
        }
        friend class hkVdbStatsHandler;
    };

    /// Server/client frame/step stats.
    struct FrameStats
    {
        HK_DECLARE_CLASS( FrameStats, New );
        FrameStats() : m_stepStats( HK_VDB_FPS_AVERAGING_WINDOW ), m_vdbStepStats( HK_VDB_FPS_AVERAGING_WINDOW ), m_numFrames( 0 ) {}

        /// Gets the timing for step throughput (server/client)
        const hkWindowedAverage& getStepStats() const { return m_stepStats; }

        /// Gets the timing between steps as reported by the vdb server (hkVisualDebugger::step(...)).
        const hkWindowedAverage& getVdbStepStats() const { return m_vdbStepStats; }

        /// Gets the total number of frames processed.
        hkUint64 getNumFrames() const { return m_numFrames; }

        /// Gets the timing for the last step in seconds (server/client).
        hkReal getLastStepElapsedSeconds() const { return m_stepStats.getLastTime(); }

    protected:
        HK_INLINE void reset()
        {
            this->~FrameStats();
            new ( this ) FrameStats();
        }
        hkWindowedAverage m_stepStats;
        hkWindowedAverage m_vdbStepStats;
        hkUint64 m_numFrames;
        friend class hkVdbStatsHandler;
    };

    HK_DECLARE_SIGNAL( PerfStatsReceivedSignal, hkSignal2< const PerfStats&, hkVdbSignalResult& > );
    /// Fired when new detailed performance stats have been received from the server.
    PerfStatsReceivedSignal m_perfStatsReceived;

    HK_DECLARE_SIGNAL( MemStatsReceivedSignal, hkSignal2< const MemStats&, hkVdbSignalResult& > );
    /// Fired when new memory use stats have been received from the server.
    MemStatsReceivedSignal m_memStatsReceived;

    HK_DECLARE_SIGNAL( FrameStatsUpdatedSignal, hkSignal1< const FrameStats& > );
    /// Fired when our total frame/step stats have been updated (server/client).
    FrameStatsUpdatedSignal m_frameStatsUpdated;
    

    //
    // Internal use
    //

    hkResult processPerfStatsCmd( hkUint32 frameNumber, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader );
    hkResult processMemStatsCmd( hkUint32 frameNumber, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader );
    hkResult processStatsMapsCmd( hkUint32 frameNumber, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader );
    void onStepCompletedSignal( const hkVdbFrame& frame, const hkVdbCmd& cmd );
    void onPlaybackInfoReceivedSignalInternal( const hkVdbPlaybackHandler::PlaybackInfo& info, hkVdbSignalResult& result );
    void onServerInfoReceivedSignal( const hkVdbSetupHandler::ClientInfo& clientInfo, const hkVdbSetupHandler::ServerInfo& serverInfo, hkVdbCapabilities capabilities, hkVdbSignalResult& result );

protected:

    void setLoadedReader( hkSerialize::ReadFormat* readFormat );
    hkResult loadStatsMaps( const hkArray<hkInt8>& dataIn ); // Deprecated

    void updateSessionId( hkUint32 sessionId );
    hkResult parseStatsMapsInternal( int protocol, hkVdbLocalIStream& dataReader );

    // Debug checks on protocol
    HK_DEBUG_ONLY_MEMBER( int, m_protocol );

    // Avoid allocations during dispatch by having these declared and just reusing them
    PerfStats m_perfStats;
    MemStats m_memStats;

    struct LocalizedData : public hkReferencedObject
    {
        LocalizedData() :
            m_stringMap( hkRefNew<hkMonitorStreamStringMap>( new hkMonitorStreamStringMap() ) ),
            m_typeMap( hkRefNew<hkMonitorStreamTypeMap>( new hkMonitorStreamTypeMap() ) )
        {}
        ~LocalizedData();
        hkArray<hkStringPtr> m_statsStringTable;
        hkRefPtr<hkMonitorStreamStringMap> m_stringMap;
        hkRefPtr<hkMonitorStreamTypeMap> m_typeMap;
        hkVdbSet<const char*> m_localStringMap;
    };
    hkRefPtr<LocalizedData> m_localizedDatas[2];
    int m_localizedDataIdx;
    HK_INLINE LocalizedData& getLocalizedData() { return *m_localizedDatas[m_localizedDataIdx]; }
    void cycleLocalizedData();

    hkRefPtr<hkSerialize::ReadFormat> m_readFormat;
    hkInt64 m_sessionId;
    hkBool32 m_statsMapsLoaded;
    hkBool32 m_persistable;

    // Frame stats
    FrameStats m_frameStats;
    hkStopwatch m_stopwatch;

    hkRefPtr<hkVdbSetupHandler> m_setupHandler;
    hkRefPtr<hkVdbPlaybackHandler> m_playbackHandler;

    HK_ON_DEBUG( hkInt64 m_lastProcessedFrame; )
    HK_ON_DEBUG( hkInt64 m_lastProcessedCmdIdx; )

    friend class hkVdbInternalHandler;
    friend class hkVdbPersistentStateCache;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
