// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbServices/hkVdbServices.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbStatsHandler.h>

#include <Common/Visualize/hkProcessSerializeUtils.h>
#include <Common/Visualize/Serialize/hkPersistableReadFormat.h>

#include <VisualDebugger/VdbServices/hkVdbClient.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdInput.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdDispatcher.h>








#define DEBUG_LOG_IDENTIFIER "vdb.Cmd.Handler.Stats"
#include <Common/Base/System/Log/hkLog.hxx>

#define CONSUME_PROTOCOL( _STREAM ) \
    if ( ( ( cmd.getType() == hkVdbCmdType::SEND_PERF_STATS ) || ( cmd.getType() == hkVdbCmdType::SEND_STATS_MAPS ) ) && ( protocol < hkProtocolVersion::VDB_2017_1 ) ) { _STREAM.read32(); } HK_MUST_END_WITH_SEMICOLON

namespace
{
    hkResult handlePerfStatsCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
        hkVdbStatsHandler* handler = reinterpret_cast< hkVdbStatsHandler* >( userHandle );
        return handler->processPerfStatsCmd( frame.getFrameNumber(), cmd, protocol, dataReader );
    }

    hkResult handleMemStatsCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
        hkVdbStatsHandler* handler = reinterpret_cast< hkVdbStatsHandler* >( userHandle );
        return handler->processMemStatsCmd( frame.getFrameNumber(), cmd, protocol, dataReader );
    }

    hkResult handleStatsMapsCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
        hkVdbStatsHandler* handler = reinterpret_cast< hkVdbStatsHandler* >( userHandle );
        return handler->processStatsMapsCmd( frame.getFrameNumber(), cmd, protocol, dataReader );
    }

    void _makeDynamicStringsLocal( hkMonitorStreamAnalyzer::Node* node, hkVdbSet<const char*>& dynamicMap, hkMonitorStreamStringMap& staticMap )
    {
        const char* str = node->m_name;
        if ( !dynamicMap.contains( ( char* ) str ) && !staticMap.m_runtimeMap.contains( ( hkUlong ) str ) )
        {
            char* localStr = hkString::strDup( str );
            dynamicMap.insert( localStr );
            node->m_name = localStr;
        }
        for ( int c = 0; c < node->m_children.getSize(); ++c )
        {
            _makeDynamicStringsLocal( node->m_children[c], dynamicMap, staticMap );
        }
    }

    void _deallocateStrings( hkVdbSet<const char*>& map )
    {
        hkVdbSet<const char*>::Iterator iter = map.getIterator();
        while ( map.isValid( iter ) )
        {
            hkDeallocate<char>( const_cast<char*>( map.getElement( iter ) ) );
            iter = map.getNext( iter );
        }
        map.clear();
    }
};

namespace VdbBackCompat
{
    class DepObjectSerialize
    {
    public:
        HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_BASE, DepObjectSerialize );
        struct LocalFixup
        {
            HK_DECLARE_NONVIRTUAL_CLASS_ALLOCATOR( HK_MEMORY_CLASS_UTILITIES, DepObjectSerialize::LocalFixup );
            int fromOffset;
            int toOffset;
        };
    };

    void readFrameInfo( hkVdbLocalIStream& stream, hkMonitorStreamFrameInfo& frameInfo )
    {
        hkStreamReader* rawStreamReader = stream.getStreamReader();

        hkUint64 klassID = 0;

        // FrameInfo
        hkUint32 infoSize = stream.read32u();
        infoSize = infoSize;

        {
            hkUint32 dataSize = 0;
            hkStreamReader* s = rawStreamReader;
            // id
            hkUint64 id;
            s->read( &id, sizeof( id ) );

            // read class ID, as it may be a virtual class we asked for, and here we know its exact, instance class type.
            s->read( &klassID, sizeof( klassID ) );

            // local fixups
            hkUint32 numLocal;
            s->read( &numLocal, sizeof( hkUint32 ) );
            HK_ASSERT_NO_MSG( 0x33c05925, numLocal == 0 );

            // global fixups
            hkUint32 numGlobal;
            s->read( &numGlobal, sizeof( hkUint32 ) );
            HK_ASSERT_NO_MSG( 0x67c63ccc, numGlobal == 0 );

            // data
            s->read( &dataSize, sizeof( hkUint32 ) );

            s->skip( 4 ); // hkStringPtr m_heading
            frameInfo.m_indexOfTimer0 = stream.read32();
            frameInfo.m_indexOfTimer1 = stream.read32();
            frameInfo.m_absoluteTimeCounter = ( hkMonitorStreamFrameInfo::AbsoluteTimeCounter ) stream.read32();
            frameInfo.m_timerFactor0 = stream.readFloat32();
            frameInfo.m_timerFactor1 = stream.readFloat32();
            frameInfo.m_threadId = stream.read32();
            frameInfo.m_frameStreamStart = stream.read32();
            frameInfo.m_frameStreamEnd = stream.read32();
            s->skip( 12 ); // padding
        }
    }

    void readStringTable( hkVdbLocalIStream& stream, hkMonitorStreamStringMap& masterStringMap )
    {
        hkUint32 stringmapSize = stream.read32u();
        hkSeekableStreamReader* seekable = stream.getStreamReader()->isSeekTellSupported();
        if ( seekable )
        {
            int positionOfStringMap = seekable->tell();

            hkUint32 infoDataSize = 0;
            bool newStringsAdded = false;
            {
                //void* infoPtr = HK_NULL;
                hkUint64 klassID = 0;

                // manually decode the stream
                {
                    hkArray<DepObjectSerialize::LocalFixup> localFixups;

                    hkUint64 id;
                    seekable->read( &id, sizeof( id ) );

                    // read class ID, as it may be a virtual class we asked for, and here we know its exact, instance class type.
                    seekable->read( &klassID, sizeof( klassID ) );

                    // local fixups
                    hkUint32 numLocal;
                    seekable->read( &numLocal, sizeof( hkUint32 ) );
                    localFixups.setSize( numLocal );
                    {
                        for ( hkUint32 li = 0; ( li < numLocal ); ++li )
                        {
                            DepObjectSerialize::LocalFixup& lf = localFixups[li];
                            seekable->read( &lf.fromOffset, sizeof( hkInt32 ) );
                            seekable->read( &lf.toOffset, sizeof( hkInt32 ) );
                        }
                    }

                    // global fixups
                    hkUint32 numGlobal;
                    seekable->read( &numGlobal, sizeof( hkUint32 ) );
                    HK_ASSERT_NO_MSG( 0x5230d5da, numGlobal == 0 );

                    // data
                    {
                        seekable->read( &infoDataSize, sizeof( hkUint32 ) );
                        int orinalPosition = seekable->tell();

                        char* dataPtr = hkAllocateStack<char>( infoDataSize, "dataptr" );
                        seekable->read( dataPtr, infoDataSize );
                        if ( seekable->isOk() )
                        {
                            seekable->seek( orinalPosition, hkSeekableStreamReader::STREAM_SET );
                            hkUint32 pointer;
                            seekable->read( &pointer, sizeof( hkUint32 ) );
                            hkUint32 size;
                            seekable->read( &size, sizeof( hkUint32 ) );
                            hkUint32 capacityAndFlags;
                            seekable->read( &capacityAndFlags, sizeof( hkUint32 ) );

                            seekable->skip( 4 ); // padding

                            masterStringMap.m_map.reserve( masterStringMap.m_map.getSize() + size );
                            for ( hkUint32 i = 0; i < size; i++ )
                            {
                                seekable->seek( orinalPosition + 16 + i * 16, hkSeekableStreamReader::STREAM_SET );

                                hkUint64 stringId;
                                seekable->read( &stringId, sizeof( hkUint64 ) );

                                int pointerOffset = seekable->tell() - orinalPosition;
                                pointerOffset = pointerOffset;
                                hkUint32 originalPointer;
                                seekable->read( &originalPointer, sizeof( hkUint32 ) );

                                seekable->skip( 4 ); // padding

                                HK_ASSERT_NO_MSG( 0x3f0ab951, localFixups[i + 1].fromOffset == pointerOffset );
                                const char* string = dataPtr + localFixups[i + 1].toOffset;

                                //newStringMapEntry->m_id = stringId;
                                //newStringMapEntry->m_string.set(string);
                                const char* curValue = HK_NULL;
                                if ( masterStringMap.m_runtimeMap.get( stringId, &curValue ).isFailure() )
                                {
                                    hkMonitorStreamStringMap::StringMap& sm = masterStringMap.m_map.expandOne();
                                    sm.m_id = stringId;
                                    sm.m_string = string;

                                    newStringsAdded = true;
                                }
                            }
                        }
                        hkDeallocateStack<char>( dataPtr, infoDataSize );
                    }
                }
            }

            seekable->seek( positionOfStringMap + stringmapSize, hkSeekableStreamReader::STREAM_SET );

            if ( newStringsAdded )
            {
                masterStringMap.createRuntime();
            }
        }
    }
}

hkVdbStatsHandler::hkVdbStatsHandler( bool persistable ) :
    hkVdbCmdHandler<hkVdbCmdHandlerType::STATS>( &s_debugLog ),
    m_localizedDataIdx( 0 ),
    m_sessionId( 0 ),
    m_statsMapsLoaded( false ),
    m_persistable( persistable )
    HK_ON_DEBUG( , m_lastProcessedFrame( -1 ) )
    HK_ON_DEBUG( , m_lastProcessedCmdIdx( -1 ) )
{
    cycleLocalizedData();
}

hkResult hkVdbStatsHandler::waitForCompletion()
{
    
    return hkVdbCmdHandler<hkVdbCmdHandlerType::STATS>::waitForCompletion();
}

hkResult hkVdbStatsHandler::registerSelf( hkVdbClient& client )
{
    using namespace hkVdbCmdType;

    bool succeeded = true;
    hkVdbCmdDispatcher& dispatcher = client.getCmdDispatcher();

    succeeded &= ( dispatcher.registerHandler( SEND_PERF_STATS, handlePerfStatsCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( DEPRECATED_SEND_STATISTICS_DUMP, handlePerfStatsCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( SEND_MEM_STATS, handleMemStatsCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( SEND_STATS_MAPS, handleStatsMapsCmdFunc, this, this ).isSuccess() );

    m_setupHandler = client.getCmdHandler<hkVdbSetupHandler>();
    m_playbackHandler = client.getCmdHandler<hkVdbPlaybackHandler>();

    HK_SUBSCRIBE_TO_SIGNAL( client.m_stepCompleted, this, hkVdbStatsHandler );
    HK_SUBSCRIBE_TO_SIGNAL( m_setupHandler->m_serverInfoReceived, this, hkVdbStatsHandler );
    HK_SUBSCRIBE_TO_SIGNAL( m_playbackHandler->m_playbackInfoReceivedInternal, this, hkVdbStatsHandler );

    HK_VDB_VERIFY_REPORTER_CONDITION( succeeded, dispatcher, hkVdbError::CMD_HANDLER_REGISTRATION_ERROR );

    return HK_SUCCESS;
}

hkResult hkVdbStatsHandler::unregisterSelf( hkVdbClient& client )
{
    using namespace hkVdbCmdType;

    bool succeeded = true;
    hkVdbCmdDispatcher& dispatcher = client.getCmdDispatcher();

    succeeded &= ( dispatcher.unregisterHandler( SEND_PERF_STATS ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( DEPRECATED_SEND_STATISTICS_DUMP ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( SEND_MEM_STATS ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( SEND_STATS_MAPS ).isSuccess() );

    client.m_stepCompleted.unsubscribeAll( this );
    m_setupHandler->m_serverInfoReceived.unsubscribeAll( this );
    m_playbackHandler->m_playbackInfoReceivedInternal.unsubscribeAll( this );

    m_setupHandler = HK_NULL;
    m_playbackHandler = HK_NULL;

    // ensure old stuff gets cleared out
    for ( int i = 0; i < HK_COUNT_OF( m_localizedDatas ); i++ )
    {
        cycleLocalizedData();
    }

    HK_VDB_VERIFY_REPORTER_CONDITION( succeeded, dispatcher, hkVdbError::CMD_HANDLER_REGISTRATION_ERROR );

    return HK_SUCCESS;
}

void hkVdbStatsHandler::onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection )
{
    if ( use == hkVdbConnectionUse::APPLICATION )
    {
        m_perfStats.reset();
        m_memStats.reset();
        m_frameStats.m_stepStats.reset();
        m_frameStats.m_vdbStepStats.reset();
        // Notify listeners of "no data".
        {
            hkVdbSignalResult result;
            m_perfStatsReceived.fire( m_perfStats, result );
            m_memStatsReceived.fire( m_memStats, result );
            m_frameStatsUpdated.fire( m_frameStats );
        }
        cycleLocalizedData();
        m_readFormat = HK_NULL;
        m_sessionId = 0;
        m_statsMapsLoaded = false;
        m_stopwatch.reset();
        HK_ON_DEBUG( m_lastProcessedFrame = -1; )
        HK_ON_DEBUG( m_lastProcessedCmdIdx = -1; )
    }
}

hkResult hkVdbStatsHandler::processPerfStatsCmd( hkUint32 frameNumber, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader )
{
    using namespace hkVdbCmdType;
    typedef hkMonitorStreamAnalyzer::Tree Tree;
    typedef hkMonitorStreamAnalyzer::Node Node;

    // No need to process stats during caching
    if ( ( protocol >= hkProtocolVersion::SUPPORTS_CIRCULAR_BUFFER ) && m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::CACHING ) )
    {
        return HK_SUCCESS;
    }

    // Refresh/setup our stream config
    
    LocalizedData& data = getLocalizedData();
    {
        m_perfStats.m_streamConfig.m_protocolVersion = protocol;
        m_perfStats.m_streamConfig.m_stringMap = &data.m_stringMap->m_runtimeMap;
        m_perfStats.m_streamConfig.m_typeMap = &data.m_typeMap->m_runtimeMap;
    }

    CONSUME_PROTOCOL( dataReader );

    if ( protocol >= hkProtocolVersion::REFLECTION_2015_1 )
    {
        if ( protocol < hkProtocolVersion::SUPPORTS_CIRCULAR_BUFFER )
        {
            updateSessionId( dataReader.read32u() );
        }

        // Set timer-specific stream config info
        hkUint8 bitsPerTimer = dataReader.read8u();
        m_perfStats.m_streamConfig.m_timersAre64Bit = ( bitsPerTimer == 64 );
        hkUint8 timersPerEntry = dataReader.read8u();
        m_perfStats.m_streamConfig.m_twoTimers = ( timersPerEntry == 2 );
        m_perfStats.m_frameInfo.m_timerFactor0 = dataReader.readFloat32();

        // Before the circular buffer, we didn't have two different commands.
        // The circular buffer needs to be able to preserve state of frames that
        // get evicted from the buffer.  We needed to split the map data out into its own command
        // so we can store it as part of the persistent state saved with the buffer.
        if ( protocol < hkProtocolVersion::SUPPORTS_CIRCULAR_BUFFER )
        {
            parseStatsMapsInternal( protocol, dataReader );
            HK_ON_DEBUG( m_lastProcessedFrame = frameNumber; )
            HK_ON_DEBUG( m_lastProcessedCmdIdx = cmd.getIndex(); )
        }
    }
    else
    {
        VdbBackCompat::readFrameInfo( dataReader, m_perfStats.m_frameInfo );
        VdbBackCompat::readStringTable( dataReader, *data.m_stringMap );
        m_perfStats.m_streamConfig.m_timersAre64Bit = false; // nearly always 32bit in older dumps
        m_perfStats.m_streamConfig.m_twoTimers = true; // always 2 in older dataReaders, but second timer normally garbage unless used for L2 miss etc counts
        HK_ON_DEBUG( m_lastProcessedFrame = frameNumber; )
        HK_ON_DEBUG( m_lastProcessedCmdIdx = cmd.getIndex(); )
    }

    // No need to process stats during caching, we already processed stats maps
    if ( m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::CACHING ) )
    {
        return HK_SUCCESS;
    }

    // Clean-up previous frame
    m_perfStats.cleanupPrevious();

    // Read the large Monitor dataReader(s) in off the network
    int numThreads = dataReader.read32u();
    if ( numThreads < 1 )
    {
        return HK_SUCCESS;
    }
    else if ( numThreads > 64 )
    {
        // Sanity check
        HK_VDB_SIGNAL_ERROR_MSG(
            0xedb00025,
            hkVdbError::CMD_HANDLER_READ_ERROR,
            "The number of perf threads reported by server has exceeded the max of " << 64 );
        return HK_FAILURE;
    }

    // Set number of threads (this must be called every frame after cleanupPrevious)
    m_perfStats.setNumThreads( numThreads );

    // Create roots
    for ( int si = 0; si < m_perfStats.m_monitorStreams.getSize(); ++si )
    {
        hkArray<char>& thisStream = m_perfStats.m_monitorStreams[si];

        hkUint32 monSize = dataReader.read32u();
        thisStream.setSize( monSize );
        dataReader.readRaw( thisStream.begin(), monSize );

        // Set all char* (id) ptrs to be proper local string ptrs
        const char* dataReaderEnd = thisStream.end();

        // Make the tree from the info given
        if ( Tree* rootNode = hkMonitorStreamAnalyzer::makeStatisticsTreeForSingleFrame(
            m_perfStats.m_streamConfig,
            thisStream.begin(),
            dataReaderEnd,
            m_perfStats.m_frameInfo,
            "/",
            true ) )
        {
            // Make a thread root that is a directory node wrapping the root tree.
            hkMonitorStreamAnalyzer::ThreadRootNodes& threadRootNode = m_perfStats.m_threadRootNodes.expandOne();
            threadRootNode.m_node = rootNode;
        }
    }

    // Merge into single tree
    {
        hkMonitorStreamAnalyzer::scaleTimerValuesRec( &m_perfStats.m_statsTree, 0.0f );
        for ( int ti = 0; ( ti < m_perfStats.m_threadRootNodes.getSize() ); ++ti )
        {
            Node* rootNode = m_perfStats.m_threadRootNodes[ti].m_node;
            hkMonitorStreamAnalyzer::mergeTreesForCombinedThreadSummary( &m_perfStats.m_statsTree, (hkMonitorStreamAnalyzer::Tree*) rootNode, ti, 0/*src*/ );
        }
        hkMonitorStreamAnalyzer::fixupTags( &m_perfStats.m_statsTree );

        // whenever we blend trees, we may get nodes that have const char* that refer to objectnames embeded in the monitordataReader (and thus temporary / will become invalid on next reset )
        // hkMonitorStreamAnalyzer::makeStringsLocal will gather all const char*, but what we can do here is, as we have string map of all the normal static const char*, is just take the ones not in that map
        // and we then just copy the dynamic / dataReader ones
        _makeDynamicStringsLocal( &m_perfStats.m_statsTree, data.m_localStringMap, *data.m_stringMap );
    }

    HK_VDB_VERIFY_CONDITION( dataReader.isOk(), 0xedb00026, hkVdbError::CMD_HANDLER_READ_ERROR );

    hkVdbSignalResult result;
    m_perfStatsReceived.fire( m_perfStats, result );
    HK_VDB_VERIFY_SIGNAL_RESULT( result );

    return HK_SUCCESS;
}

hkResult hkVdbStatsHandler::processMemStatsCmd( hkUint32 frameNumber, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader )
{
    // Just ignore earlier stats (they were disable in 2015.1 anyway and radically different before that)
    if ( protocol < hkProtocolVersion::VDB_2017_1 )
    {
        return HK_SUCCESS;
    }

    // No need to process stats during caching
    if ( m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::CACHING ) )
    {
        return HK_SUCCESS;
    }

    hkStringBuf buf;
    hkUint16 numEntries = dataReader.read16u();
    m_memStats.m_entries.clear(); // clear first to force ctors in setSize, in case new members are added we aren't dealing with
    m_memStats.m_entries.setSize( numEntries );
    for ( int i = 0; i < numEntries; i++ )
    {
        hkMemorySystem::MemoryStatistics::Entry& entry = m_memStats.m_entries[i];
        {
            dataReader.readString( buf );
            entry.m_allocatorName = buf;
            entry.m_allocatorStats.m_allocated = hkLong( dataReader.read64() );
            entry.m_allocatorStats.m_inUse = hkLong( dataReader.read64() );
            entry.m_allocatorStats.m_peakInUse = hkLong( dataReader.read64() );
            entry.m_allocatorStats.m_available = hkLong( dataReader.read64() );
            entry.m_allocatorStats.m_totalAvailable = hkLong( dataReader.read64() );
            entry.m_allocatorStats.m_largestBlock = hkLong( dataReader.read64() );
        }
    }

    HK_VDB_VERIFY_CONDITION( dataReader.isOk(), 0xedb00086, hkVdbError::CMD_HANDLER_READ_ERROR );

    hkVdbSignalResult result;
    m_memStatsReceived.fire( m_memStats, result );
    HK_VDB_VERIFY_SIGNAL_RESULT( result );

    return HK_SUCCESS;
}

hkResult hkVdbStatsHandler::processStatsMapsCmd( hkUint32 frameNumber, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader )
{
    // Note: Null check necessary because persistent cached uses this class w/o registering
    if ( m_playbackHandler && m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::REPLAYING ) )
    {
        return HK_SUCCESS;
    }

    CONSUME_PROTOCOL( dataReader );

    updateSessionId( dataReader.read32u() );

    HK_VDB_VERIFY_OPERATION( parseStatsMapsInternal( protocol, dataReader ), 0xedb00027, hkVdbError::CMD_HANDLER_READ_ERROR );
    HK_VDB_VERIFY_CONDITION( dataReader.isOk(), 0xedb00028, hkVdbError::CMD_HANDLER_READ_ERROR );

    HK_ON_DEBUG( m_lastProcessedFrame = frameNumber; )
    HK_ON_DEBUG( m_lastProcessedCmdIdx = cmd.getIndex(); )

    return HK_SUCCESS;
}

void hkVdbStatsHandler::onStepCompletedSignal( const hkVdbFrame& frame, const hkVdbCmd& cmd )
{
    if ( hkReal elapsed = m_stopwatch.getElapsedSeconds() )
    {
        m_frameStats.m_stepStats.pushNewValue( elapsed );
    }
    m_frameStats.m_vdbStepStats.pushNewValue( frame.getDuration() / 1000.0f );
    m_frameStats.m_numFrames++;

    m_frameStatsUpdated.fire( m_frameStats );

    m_stopwatch.reset();
    m_stopwatch.start();

    HK_ON_DEBUG( m_lastProcessedCmdIdx = -1; )
}

void hkVdbStatsHandler::onPlaybackInfoReceivedSignalInternal( const hkVdbPlaybackHandler::PlaybackInfo& info, hkVdbSignalResult& result )
{
    if ( info.m_previousState != info.m_targetState )
    {
        m_stopwatch.reset();
        m_frameStats.m_stepStats.reset();
        m_frameStats.m_vdbStepStats.reset();
    }
}


void hkVdbStatsHandler::onServerInfoReceivedSignal( const hkVdbSetupHandler::ClientInfo& clientInfo, const hkVdbSetupHandler::ServerInfo& serverInfo, hkVdbCapabilities capabilities, hkVdbSignalResult& result )
{
    m_perfStats.m_streamConfig.m_needEndianSwap = ( serverInfo.m_isLittleEndian != hkBool( HK_ENDIAN_LITTLE ) );
    m_perfStats.m_streamConfig.m_pointersAre64Bit = ( serverInfo.m_sizeOfPointer == 8 );
}

void hkVdbStatsHandler::setLoadedReader( hkSerialize::ReadFormat* readFormat )
{
    m_sessionId = -1;
    m_readFormat = readFormat;
}

hkResult hkVdbStatsHandler::loadStatsMaps( const hkArray<hkInt8>& dataIn )
{
    // Save framework doesn't have sync for hkInt8 array, only char
    // COM-3780
    const hkArray<char>& byteArray = reinterpret_cast< const hkArray<char>& >( dataIn );
    hkSerialize::Load load;
    hkUlong used;
    LocalizedData& data = getLocalizedData();
    data.m_stringMap.setAndDontIncrementRefCount( load.toObject<hkMonitorStreamStringMap>( byteArray.begin(), byteArray.getSize(), &used ) );
    data.m_typeMap.setAndDontIncrementRefCount( load.toObject<hkMonitorStreamTypeMap>( byteArray.begin() + used, byteArray.getSize() - used ) );

    if ( data.m_stringMap && data.m_typeMap )
    {
        data.m_stringMap->createRuntime();
        data.m_typeMap->createRuntime();
        m_statsMapsLoaded = true;
        return HK_SUCCESS;
    }

    cycleLocalizedData();
    HK_ASSERT( 0x22440804, false, "Could not read stats maps" );
    return HK_FAILURE;
}

void hkVdbStatsHandler::updateSessionId( hkUint32 sessionId )
{
    if ( m_sessionId == -1 )
    {
        // This indicates we loaded a "ready-to-go" read format.
        HK_ASSERT( 0x22440937, m_readFormat, "No read format loaded" );
        m_sessionId = sessionId;
    }
    else if ( m_sessionId != sessionId )
    {
        // Writer will expect clean slate, will give new types etc.
        m_readFormat = HK_NULL;
        m_sessionId = sessionId;
    }
}

hkResult hkVdbStatsHandler::parseStatsMapsInternal( int protocol, hkVdbLocalIStream& dataReader )
{
    // Note: Null check necessary because persistent cached uses this class w/o registering
    hkBool32 isInReplay = ( m_playbackHandler && m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::REPLAYING ) );

    // [16100, +inf), we split up the commands, so we don't have to waste time reading map data just to get to perf data.
    // [16100, 16300), we saved off the maps themselves and reloaded when loading a movie; don't squash over them here (m_statsMapsLoaded == true).
    hkBool32 skipRead = isInReplay && ( protocol >= hkProtocolVersion::SUPPORTS_CIRCULAR_BUFFER ) || m_statsMapsLoaded;

    // (-inf, 16100), all the information was sent over in one cmd; so read the map data here so later reads are correct.
    hkBool32 skipProcess = ( isInReplay && ( protocol < hkProtocolVersion::SUPPORTS_CIRCULAR_BUFFER ) );

    if ( skipRead )
    {
        return HK_SUCCESS;
    }

    if ( m_statsMapsLoaded )
    {
        HK_ASSERT( 0x22440868, m_protocol >= hkProtocolVersion::SUPPORTS_CIRCULAR_BUFFER, "Skipping map reads will corrupt stream" );
        return HK_SUCCESS;
    }

    // Initialize a new read format if needed
    if ( !m_readFormat )
    {
        m_readFormat = hkRefNew<hkSerialize::TagfileReadFormat>(
            m_persistable ?
            new hkPersistableReadFormat() :
            new hkSerialize::TagfileReadFormat() );
    }

    LocalizedData& data = getLocalizedData();

    // String Map
    {
        hkUint32 stringMapDataSize = dataReader.read32u();
        hkArray<char>::Temp buffer( stringMapDataSize );
        dataReader.readRaw( buffer.begin(), stringMapDataSize );
        if ( !skipProcess )
        {
            hkReflect::Var stringTable;
            hkProcessSerializeUtils::bufferToVar( m_readFormat, hkArrayView<char>( buffer.begin(), buffer.getSize() ), stringTable );
            if ( stringTable )
            {
                hkMonitorStreamStringMap* sm = stringTable.dynCast<hkMonitorStreamStringMap>();
                data.m_stringMap->merge( *sm );
                sm->removeReference();
            }
            else
            {
                return HK_FAILURE;
            }
        }
    }

    // Type Map
    {
        hkUint32 typeMapDataSize = dataReader.read32u();
        hkArray<char>::Temp buffer( typeMapDataSize );
        buffer.setSize( typeMapDataSize );
        dataReader.readRaw( buffer.begin(), typeMapDataSize );
        if ( !skipProcess )
        {
            hkReflect::Var typeTable;
            hkProcessSerializeUtils::bufferToVar( m_readFormat, hkArrayView<char>( buffer.begin(), buffer.getSize() ), typeTable );
            if ( typeTable )
            {
                hkMonitorStreamTypeMap* tm = typeTable.dynCast<hkMonitorStreamTypeMap>();
                data.m_typeMap->merge( *tm );
                tm->removeReference();
            }
            else
            {
                return HK_FAILURE;
            }
        }
    }

    return HK_SUCCESS;
}

hkVdbStatsHandler::LocalizedData::~LocalizedData()
{
    _deallocateStrings( m_localStringMap );
}

void hkVdbStatsHandler::cycleLocalizedData()
{
    m_localizedDataIdx = ( m_localizedDataIdx + 1 ) % HK_COUNT_OF( m_localizedDatas );
    m_localizedDatas[m_localizedDataIdx] = hkRefNew<LocalizedData>( new LocalizedData );
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
