// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbServices/hkVdbServices.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbSetupHandler.h>

#include <Common/Visualize/hkVersionReporter.h>

#include <VisualDebugger/VdbServices/hkVdbClient.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdDispatcher.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdOutput.h>

#define DEBUG_LOG_IDENTIFIER "vdb.Cmd.Handler.Setup"
#include <Common/Base/System/Log/hkLog.hxx>

namespace
{
    hkResult handleServerInfoCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
        hkVdbSetupHandler* handler = reinterpret_cast< hkVdbSetupHandler* >( userHandle );
        return handler->processServerInfoCmd( cmd.getType(), protocol, dataReader );
    }

    hkBool32 processRestrictsCapabilities( const char* name )
    {
        return
            ( hkString::strCasecmp( name, "AI - World Replay" ) == 0 );
    }
};

hkVdbSetupHandler::hkVdbSetupHandler() :
    hkVdbCmdHandler<hkVdbCmdHandlerType::SETUP>( &s_debugLog ),
    m_capabilities( hkVdbCapabilities::UNKNOWN ),
    m_baseCapabilities( hkVdbCapabilities::UNKNOWN ),
    m_numProcessesRestrictingCapabilities( 0 )
{
    hkVersionReporter::computeInformation( m_clientInfo );
}

hkResult hkVdbSetupHandler::registerSelf( hkVdbClient& client )
{
    using namespace hkVdbCmdType;

    bool succeeded = true;
    hkVdbCmdDispatcher& dispatcher = client.getCmdDispatcher();

    succeeded &= ( dispatcher.registerHandler( SEND_SERVER_INFO, handleServerInfoCmdFunc, this, this ).isSuccess() );

    m_output = &client.getCmdOutput();
    m_processHandler = client.getCmdHandler<hkVdbProcessHandler>();

    HK_SUBSCRIBE_TO_SIGNAL( m_processHandler->m_processCmdReceived, this, hkVdbProcessHandler );

    HK_VDB_VERIFY_REPORTER_CONDITION( succeeded, dispatcher, hkVdbError::CMD_HANDLER_REGISTRATION_ERROR );

    return HK_SUCCESS;
}

hkResult hkVdbSetupHandler::unregisterSelf( hkVdbClient& client )
{
    using namespace hkVdbCmdType;

    bool succeeded = true;
    hkVdbCmdDispatcher& dispatcher = client.getCmdDispatcher();

    succeeded &= ( dispatcher.unregisterHandler( SEND_SERVER_INFO ).isSuccess() );

    m_processHandler->m_processCmdReceived.unsubscribeAll( this );

    m_output = HK_NULL;
    m_processHandler = HK_NULL;

    HK_VDB_VERIFY_REPORTER_CONDITION( succeeded, dispatcher, hkVdbError::CMD_HANDLER_REGISTRATION_ERROR );

    return HK_SUCCESS;
}

void hkVdbSetupHandler::onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection )
{
    if ( use == hkVdbConnectionUse::APPLICATION )
    {
        m_capabilities = hkVdbCapabilities::UNKNOWN;
        m_serverInfo.m_protocol = -1;
        m_numProcessesRestrictingCapabilities = 0;
    }
}

hkResult hkVdbSetupHandler::getServerInfo( ServerInfo& serverInfoOut ) const
{
    if ( m_capabilities.noneIsSet( hkVdbCapabilities::UNKNOWN ) )
    {
        serverInfoOut = m_serverInfo;
        return HK_SUCCESS;
    }

    HK_VDB_SIGNAL_ERROR_MSG(
        0xedb00133,
        hkVdbError::PROTOCOL_ERROR,
        "We do not yet have valid server information" );

    serverInfoOut.m_protocol = -1;
    return HK_FAILURE;
}

int hkVdbSetupHandler::getServerProtocol() const
{
    if ( m_capabilities.noneIsSet( hkVdbCapabilities::UNKNOWN ) )
    {
        return m_serverInfo.m_protocol;
    }

    HK_ASSERT( 0x22441096, false, "Shouldn't get protocol outside valid connection context" );
    return -1;
}

void hkVdbSetupHandler::onProcessCmdReceivedSignal( const hkVdbProcessHandler::ProcessCmd& processCmd, hkVdbSignalResult& result )
{
    
    if ( const hkVdbProcessHandler::ProcessSelectionChangedCmd* selectionChangedCmd = processCmd.asSelectionChangedCmd() )
    {
        if ( processRestrictsCapabilities( m_processHandler->getProcessName( selectionChangedCmd->m_tag ) ) )
        {
            m_numProcessesRestrictingCapabilities += ( selectionChangedCmd->isSelected() ) ? 1 : -1;
            updateCapabilities(
                // We currently restrict *all* capabilities if we have any unsupported processes.
                ( m_numProcessesRestrictingCapabilities ) ? 0 : m_baseCapabilities,
                false,
                true );
        }
    }
}

hkResult hkVdbSetupHandler::processServerInfoCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader )
{
    using namespace hkVdbCmdType;

    hkUint64 authenticationHashAck;
    HK_VDB_VERIFY_OPERATION( hkVersionReporter::readServerInformationCmdData(
        dataReader,
        m_serverInfo,
        authenticationHashAck ),
        0xedb00029,
        hkVdbError::VERSION_READ_ERROR );

    // If the server is asking for an authentication ack, send it back.
    if ( authenticationHashAck )
    {
        if ( hkVdbScopedStreamWriter writer = m_output->startCmd( INVALID, hkVdbCmdOutput::NON_BLOCKING ) )
        {
            hkVdbOStream oarchive( writer );
            oarchive.write64u( authenticationHashAck );
        }
        else
        {
            HK_VDB_SIGNAL_REPORTER_ERROR( *m_output, hkVdbError::CMD_HANDLER_WRITE_ERROR );
            return HK_FAILURE;
        }
    }

    HK_VDB_VERIFY_CONDITION_MSG(
        m_serverInfo.m_protocol == protocol,
        0xedb00132,
        hkVdbError::PROTOCOL_ERROR,
        "Corrupt protocol cmd encountered" );

    // Clear our UNKOWN flag, but don't signal yet.
    hkVdbCapabilities newCapabilities = m_capabilities;
    newCapabilities.clear( hkVdbCapabilities::UNKNOWN );
    updateCapabilities( newCapabilities, true, false );

    HK_VDB_VERIFY_CONDITION_MSG(
        ( m_serverInfo.m_protocol >= hkVersionReporter::m_protocolMinimumCompatible ) &&
        ( m_serverInfo.m_protocol <= hkVersionReporter::m_protocolVersion ),
        0xedb00030,
        hkVdbError::UNSUPPORTED_VERSION,
        "The server's protocol version of " << m_serverInfo.m_protocol << " is not in range " << hkVersionReporter::m_protocolMinimumCompatible << " - " << hkVersionReporter::m_protocolVersion );

    
    
    
    
    
    
    

    HK_VDB_VERIFY_CONDITION( dataReader.isOk(), 0xedb00031, hkVdbError::CMD_HANDLER_READ_ERROR );

    // Set our save/quick-start capabilities and signal
    
    
    newCapabilities.orWith(
        ( hkVdbCapabilities::SAVE_CMD_INPUT_BUFFER | hkVdbCapabilities::QUICK_START_RECORDING )
        * bool( m_serverInfo.m_protocol >= hkProtocolVersion::SUPPORTS_CIRCULAR_BUFFER ) );
    hkVdbSignalResult result = updateCapabilities( newCapabilities, true, true );
    HK_VDB_VERIFY_SIGNAL_RESULT( result );

    return HK_SUCCESS;
}

hkVdbSignalResult hkVdbSetupHandler::updateCapabilities( hkVdbCapabilities capabilities, hkBool32 init, hkBool32 signal )
{
    hkVdbSignalResult result;

    // We never allow rejection to take us back to "unknown".
    // We are setting the capabilities so we now know them.
    hkVdbCapabilities prevCapabilities =
        ( m_capabilities == hkVdbCapabilities::UNKNOWN ) ?
        hkVdbCapabilities::NONE :
        m_capabilities;

    // Update the capabilities
    m_capabilities = capabilities;

    // Signal if desired and needed
    if ( signal && ( ( prevCapabilities != m_capabilities ) || init ) )
    {
        m_serverInfoReceived.fire( m_clientInfo, m_serverInfo, m_capabilities, result );

        // If server info was rejected, revert our capabilities
        if ( result.isFailure() )
        {
            m_capabilities = prevCapabilities;
        }
    }

    // When initializing, update our base capabilities (so we can go back to them when
    // processes restricting capabilities go away).
    if ( init )
    {
        m_baseCapabilities = capabilities;
    }

    return result;
}


#if 0

    if ( ( m_protocol < hkVersionReporter::m_protocolMinimumCompatible ) ||
        ( m_protocol > hkVersionReporter::m_protocolVersion ) )

        // version
        if ( !g_pDeserializer->gotRemoteVersion() )
        {
            versionOk = false;

            stream << "A connection has been established but the remote server has not supplied any version information.\n";
            stream << "This probably means that your game/server was built with Havok libs without any version information and are old.\n";
            stream << "This client application has protocol version [" << hkVersionReporter::m_protocolVersion << "].\n";
            stream << "It can work with servers of version [" << hkVersionReporter::m_protocolMinimumCompatible << "] or greater.\n";
            stream << "Please use the client that shipped with your Havok SDK.\n\n";
            stream << "Disconnecting...\n\n";
        }
        else if ( status == hkVdbDeserializer::HK_VERSION_CONFLICT )
        {
            versionOk = false;

            stream << "A connection has been established but the protocol on the remote server is not compatible with this client.\n";
            stream << "This client application has protocol version [" << hkVersionReporter::m_protocolVersion << "].\n";
            stream << "It can work with servers of version [" << hkVersionReporter::m_protocolMinimumCompatible << "] or greater.\n";
            stream << "The remote server/game has protocol version [" << g_pDeserializer->getRemoteProtocolVersion() << "],\n";
            stream << "and it can work with clients of version [" << g_pDeserializer->getRemoteProtocolMinimumCompatible() << "] or greater.\n";
            stream << "Please use the client that shipped with your Havok SDK.\n\n";
            stream << "Disconnecting...\n\n";
        }
        else if ( status == hkVdbDeserializer::HK_VERSION_CLIENT_NEWER )
        {
            stream << "A connection has been established but the protocol on the remote server is older then the version in this client.\n";
            stream << "Since the server is old some of the new features of this client will not be used.\n";
            stream << "This client application has protocol version [" << hkVersionReporter::m_protocolVersion << "].\n";
            stream << "It can work with servers of version [" << hkVersionReporter::m_protocolMinimumCompatible << "] or greater.\n";
            stream << "The remote server/game has protocol version [" << g_pDeserializer->getRemoteProtocolVersion() << "],\n";
            stream << "and it can work with clients of version [" << g_pDeserializer->getRemoteProtocolMinimumCompatible() << "] or greater.\n";
            stream << "Please use the client that shipped with your Havok SDK.\n\n";
        }
        else if ( status == hkVdbDeserializer::HK_VERSION_CLIENT_OLDER )
        {
            stream << "A connection has been established but the protocol on the remote server is newer then the version in this client.\n";
            stream << "Since this client is old some of the data which the server is trying to display may not appear!\n";
            stream << "This client application has protocol version [" << hkVersionReporter::m_protocolVersion << "].\n";
            stream << "It can work with servers of version [" << hkVersionReporter::m_protocolMinimumCompatible << "] or greater.\n";
            stream << "The remote server/game has protocol version [" << g_pDeserializer->getRemoteProtocolVersion() << "],\n";
            stream << "and it can work with clients of version [" << g_pDeserializer->getRemoteProtocolMinimumCompatible() << "] or greater.\n";
            stream << "Please use the client that shipped with your Havok SDK.\n\n";
        }
#endif

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
