// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Visualize/hkVisualDebuggerCmdType.h>

#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbCmdHandler.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbPlaybackHandler.h>

class hkVdbCmdOutput;
class hkVdbPlaybackHandler;

//////////////////////////////////////////////////////////////////////////
/// A handler for process/viewer-related cmds.
//////////////////////////////////////////////////////////////////////////
class HK_EXPORT_VDB hkVdbProcessHandler : public hkVdbCmdHandler<hkVdbCmdHandlerType::PROCESS>
{
public:
    HK_DECLARE_CLASS( hkVdbProcessHandler, New );
    hkVdbProcessHandler();

    //
    // hkVdbCmdHandler interface
    //

    virtual hkResult registerSelf( hkVdbClient& client ) HK_OVERRIDE;
    virtual hkResult unregisterSelf( hkVdbClient& client ) HK_OVERRIDE;
    virtual void onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection ) HK_OVERRIDE;

    //
    // Control and Query
    //

    
    /// Returns true if the process has been selected (see setProcessSelected).
    
    hkResult isProcessSelected( int tag, hkBool32& isSelected ) const;

    /// Sets the process' selected state (selected processes are created on the server).
    hkResult setProcessSelected( int tag, hkBool32 selected );

    /// Get the tag ids of all selected processes (see setProcessSelected).
    int getSelectedProcessTags( hkArray<int>& tagsOut ) const;

    /// Get the names of all the selected processes (see setProcessSelected).
    int getSelectedProcessNames( hkArray<hkStringPtr>& processNamesOut ) const;

    /// Sets multiple process' selected state by tag ids (see setProcessSelected).
    hkResult setSelectedProcessTags( const hkArrayView<int>& tags, hkBool32 selected );

    /// Sets multiple process' selected state by name (see setProcessSelected).
    hkResult setSelectedProcessNames( const hkArrayView<hkStringPtr>& processNames, hkBool32 selected );

    /// Get a process tag from its name.
    int getProcessTag( const char* name ) const;

    /// Get a process name from its tag id.
    const char* getProcessName( int tag ) const;
    
    

    //
    // Signals
    //

    
    struct ProcessRegisteredCmd;
    struct ProcessUpdatedCmd;
    struct ProcessSelectionChangedCmd;

    /// Base class for all process-related cmd data.
    struct ProcessCmd
    {
        HK_DECLARE_CLASS( ProcessCmd, New );
        ProcessCmd() : m_cmdType( hkVdbCmdType::INVALID ) {}

        /// The cmd type.
        hkVdbCmdType::Enum m_cmdType;

        /// The process tag id.
        hkInt32 m_tag;

        /// Returns the cmd as a registered cmd if that is the type; null otherwise.
        HK_INLINE const ProcessRegisteredCmd* asRegisteredCmd() const;

        /// Returns the cmd as an updated cmd if that is the type; null otherwise.
        HK_INLINE const ProcessUpdatedCmd* asUpdatedCmd() const;

        /// Returns the cmd as a selection cmd if that is the type; null otherwise.
        HK_INLINE const ProcessSelectionChangedCmd* asSelectionChangedCmd() const;
    };

    /// Cmd data for a process-registered cmd.
    struct ProcessRegisteredCmd : public ProcessCmd
    {
        HK_DECLARE_CLASS( ProcessRegisteredCmd, New );

        /// The name of the registered process.
        hkStringPtr m_name;

        /// Extended information (not currently used).
        struct Ex
        {
            HK_DECLARE_CLASS( Ex, New );
            
        };

        /// Gets extended information if available (not currently used).
        HK_INLINE const Ex* getExtended() const;

    protected:
        Ex m_extended;
        friend class hkVdbProcessHandler;
    };

    /// Cmd data for a process-updated cmd (not currently implemented).
    struct ProcessUpdatedCmd : public ProcessCmd
    {
        HK_DECLARE_CLASS( ProcessUpdatedCmd, New );
        
        ProcessUpdatedCmd() { m_cmdType = hkVdbCmdType::UPDATE_PROCESS; }
    };

    /// Cmd data for a process-selected cmd.
    struct ProcessSelectionChangedCmd : public ProcessCmd
    {
        HK_DECLARE_CLASS( ProcessSelectionChangedCmd, New );

        /// Returns true if the process was selected, false if unselected.
        HK_INLINE hkBool32 isSelected() const { return m_cmdType == hkVdbCmdType::SELECT_PROCESS;   }
    };

    HK_DECLARE_SIGNAL( ProcessCmdReceivedSignal, hkSignal2< const ProcessCmd&, hkVdbSignalResult& > );
    /// Fires when we receive a process cmd or the client changes some process state.
    ProcessCmdReceivedSignal m_processCmdReceived;
    

    //
    // Internal use
    //

    void onPlaybackInfoReceivedSignalInternal( const hkVdbPlaybackHandler::PlaybackInfo& info, hkVdbSignalResult& result );
    hkResult processRegisteredCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader );
    hkResult processUpdatedCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader );
    hkResult processSelectedCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader );

protected:

    struct Process
    {
        HK_DECLARE_CLASS( Process, New );
        Process() : m_selected( false ) {}
        int m_tag;
        hkStringPtr m_name;
        hkBool32 m_selected;
    };

    hkResult setProcessSelectedServer( int tag, hkBool32 selected ) const;
    hkResult setProcessSelectedInternal( int tag, hkBool32 selected, hkBool32 notifyServer );

    // Avoid allocations during dispatch by having these declared and just reusing them
    ProcessRegisteredCmd m_registeredCmd;
    ProcessUpdatedCmd m_tweakedCmd;
    ProcessSelectionChangedCmd m_selectionChangedCmd;

    hkRefPtr<hkVdbCmdOutput> m_output;
    hkRefPtr<hkVdbPlaybackHandler> m_playbackHandler;

    hkVdbMap<int, Process> m_registeredProcesses;
    hkVdbSet<int> m_dirtyProcesses;
};

#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbProcessHandler.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
