// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbServices/hkVdbServices.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbProcessHandler.h>

#include <VisualDebugger/VdbServices/hkVdbClient.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdDispatcher.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdOutput.h>

#ifndef HK_VDB_USE_HASH_MAP
#include <Common/Base/Container/PointerMap/hkMap.h>
#include <Common/Base/Container/PointerMap/hkMap.hxx>
template class HK_EXPORT_COMMON hkMapBase<int, hkVdbProcessHandler::Process>;
template class HK_EXPORT_COMMON hkMap<int, hkVdbProcessHandler::Process>;
#endif

#define DEBUG_LOG_IDENTIFIER "vdb.Cmd.Handler.Process"
#include <Common/Base/System/Log/hkLog.hxx>






#define READ_STRING( _STRBUF, _STREAM ) \
    HK_MULTILINE_MACRO_BEGIN \
        if ( protocol >= hkProtocolVersion::VDB_2017_1 ) { _STREAM.readString(_STRBUF); } \
        else \
        { \
            unsigned int length = _STREAM.read16u(); \
            hkInplaceArray<char, 128> buffer; \
            buffer.setSize( length + 1 ); \
            _STREAM.readRaw( buffer.begin(), length ); \
            buffer[length] = '\0'; \
            _STRBUF = &buffer[0]; \
        } \
    HK_MULTILINE_MACRO_END

namespace
{
    hkResult handleProcessRegisteredCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
        hkVdbProcessHandler* handler = reinterpret_cast< hkVdbProcessHandler* >( userHandle );
        return handler->processRegisteredCmd( cmd.getType(), protocol, dataReader );
    }

    hkResult handleProcessUpdatedCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
        hkVdbProcessHandler* handler = reinterpret_cast< hkVdbProcessHandler* >( userHandle );
        return handler->processUpdatedCmd( cmd.getType(), protocol, dataReader );
    }

    hkResult handleProcessSelectedCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
        hkVdbProcessHandler* handler = reinterpret_cast< hkVdbProcessHandler* >( userHandle );
        return handler->processSelectedCmd( cmd.getType(), protocol, dataReader );
    }
};

hkVdbProcessHandler::hkVdbProcessHandler() :
    hkVdbCmdHandler<hkVdbCmdHandlerType::PROCESS>( &s_debugLog )
{}

hkResult hkVdbProcessHandler::registerSelf( hkVdbClient& client )
{
    using namespace hkVdbCmdType;

    bool succeeded = true;
    hkVdbCmdDispatcher& dispatcher = client.getCmdDispatcher();

    succeeded &= ( dispatcher.registerHandler( REGISTER_PROCESS, handleProcessRegisteredCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( REGISTER_PROCESS_EX, handleProcessRegisteredCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( UPDATE_PROCESS, handleProcessUpdatedCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( SELECT_PROCESS, handleProcessSelectedCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( UNSELECT_PROCESS, handleProcessSelectedCmdFunc, this, this ).isSuccess() );

    m_output = &client.getCmdOutput();
    m_playbackHandler = client.getCmdHandler<hkVdbPlaybackHandler>();

    HK_SUBSCRIBE_TO_SIGNAL( m_playbackHandler->m_playbackInfoReceivedInternal, this, hkVdbProcessHandler );

    HK_VDB_VERIFY_REPORTER_CONDITION( succeeded, dispatcher, hkVdbError::CMD_HANDLER_REGISTRATION_ERROR );

    return HK_SUCCESS;
}

hkResult hkVdbProcessHandler::unregisterSelf( hkVdbClient& client )
{
    using namespace hkVdbCmdType;

    bool succeeded = true;
    hkVdbCmdDispatcher& dispatcher = client.getCmdDispatcher();

    succeeded &= ( dispatcher.unregisterHandler( REGISTER_PROCESS ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( REGISTER_PROCESS_EX ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( UPDATE_PROCESS ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( SELECT_PROCESS ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( UNSELECT_PROCESS ).isSuccess() );

    client.getCmdHandler<hkVdbPlaybackHandler>()->m_playbackInfoReceivedInternal.unsubscribeAll( this );

    m_output = HK_NULL;
    m_playbackHandler = HK_NULL;

    HK_VDB_VERIFY_REPORTER_CONDITION( succeeded, dispatcher, hkVdbError::CMD_HANDLER_REGISTRATION_ERROR );

    return HK_SUCCESS;
}

void hkVdbProcessHandler::onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection )
{
    if ( use == hkVdbConnectionUse::APPLICATION )
    {
        m_registeredProcesses.clear();
        m_dirtyProcesses.clear();
    }
}

hkResult hkVdbProcessHandler::isProcessSelected( int tag, hkBool32& isSelected ) const
{
    isSelected = false;

    hkVdbMap<int, Process>::Iterator iter = m_registeredProcesses.findKey( tag );
    HK_VDB_VERIFY_CONDITION_MSG(
        m_registeredProcesses.isValid( iter ),
        0xedb00032,
        hkVdbError::PROCESS_NOT_REGISTERED,
        "Process with tag " << tag << " has not been registered" );

    const Process& process = m_registeredProcesses.getValue( iter );
    isSelected = process.m_selected;
    return HK_SUCCESS;
}

hkResult hkVdbProcessHandler::setProcessSelected( int tag, hkBool32 selected )
{
    return setProcessSelectedInternal( tag, selected, true );
}

int hkVdbProcessHandler::getSelectedProcessTags( hkArray<int>& tagsOut ) const
{
    const int prevCount = tagsOut.getSize();
    for ( hkVdbMap<int, Process>::Iterator iter = m_registeredProcesses.getIterator();
        m_registeredProcesses.isValid( iter );
        iter = m_registeredProcesses.getNext( iter ) )
    {
        const Process& process = m_registeredProcesses.getValue( iter );
        if ( process.m_selected )
        {
            tagsOut.pushBack( process.m_tag );
        }
    }
    return ( tagsOut.getSize() - prevCount );
}

int hkVdbProcessHandler::getSelectedProcessNames( hkArray<hkStringPtr>& processNamesOut ) const
{
    const int prevCount = processNamesOut.getSize();
    for ( hkVdbMap<int, Process>::Iterator iter = m_registeredProcesses.getIterator();
        m_registeredProcesses.isValid( iter );
        iter = m_registeredProcesses.getNext( iter ) )
    {
        const Process& process = m_registeredProcesses.getValue( iter );
        if ( process.m_selected )
        {
            processNamesOut.pushBack( process.m_name );
        }
    }
    return ( processNamesOut.getSize() - prevCount );
}

hkResult hkVdbProcessHandler::setSelectedProcessTags( const hkArrayView<int>& tags, hkBool32 selected )
{
    bool succeeded = true;
    for ( int i = 0; i < tags.getSize(); i++ )
    {
        const int tag = tags[i];
        succeeded &= ( setProcessSelected( tag, selected ).isSuccess() );
    }
    return succeeded ? HK_SUCCESS : HK_FAILURE;
}

hkResult hkVdbProcessHandler::setSelectedProcessNames( const hkArrayView<hkStringPtr>& processNames, hkBool32 selected )
{
    bool succeeded = true;
    for ( int i = 0; i < processNames.getSize(); i++ )
    {
        const char* name = processNames[i];
        const int tag = getProcessTag( name );
        if ( tag != -1 )
        {
            succeeded &= ( setProcessSelected( tag, selected ).isSuccess() );
        }
        else
        {
            succeeded = false;
            HK_VDB_SIGNAL_ERROR_MSG( 0xedb00126, hkVdbError::PROCESS_NOT_REGISTERED, "Could not find process \"" << name << "\" to select" );
        }
    }
    return succeeded ? HK_SUCCESS : HK_FAILURE;
}

int hkVdbProcessHandler::getProcessTag( const char* name ) const
{
    for ( hkVdbMap<int, Process>::Iterator iter = m_registeredProcesses.getIterator();
        m_registeredProcesses.isValid( iter );
        iter = m_registeredProcesses.getNext( iter ) )
    {
        const Process& process = m_registeredProcesses.getValue( iter );
        if ( process.m_name == name )
        {
            return process.m_tag;
        }
    }
    return -1;
}

const char* hkVdbProcessHandler::getProcessName( int tag ) const
{
    hkVdbMap<int, Process>::Iterator iter = m_registeredProcesses.findKey( tag );
    if ( m_registeredProcesses.isValid( iter ) )
    {
        const Process& process = m_registeredProcesses.getValue( iter );
        return process.m_name;
    }
    return HK_NULL;
}

void hkVdbProcessHandler::onPlaybackInfoReceivedSignalInternal( const hkVdbPlaybackHandler::PlaybackInfo& info, hkVdbSignalResult& result )
{
    if (// If we stopped replaying
        info.flagWasCleared( hkVdbPlaybackFlags::REPLAYING ) ||
        // Or we aren't replaying, but we unpaused
        info.m_targetFlags.noneIsSet( hkVdbPlaybackFlags::REPLAYING ) &&
        ( info.m_previousState == hkVdbPlaybackState::PAUSED ) &&
        ( info.m_targetState == hkVdbPlaybackState::PLAYING ) )
    {
        // Update server.
        // We rely on server side for no-ops if someone toggled a process an even number of times during replay.
        for ( hkVdbSet<int>::Iterator iter = m_dirtyProcesses.getIterator();
            m_dirtyProcesses.isValid( iter );
            iter = m_dirtyProcesses.getNext( iter ) )
        {
            const int tag = m_dirtyProcesses.getElement( iter );
            hkVdbMap<int, Process>::Iterator i = m_registeredProcesses.findKey( tag );
            HK_VDB_VERIFY_SIGNALLED_CONDITION(
                m_registeredProcesses.isValid( i ),
                0xedb00115,
                hkVdbError::CMD_HANDLER_ERROR,
                result );
            Process& process = m_registeredProcesses.getValue( i );
            HK_VDB_VERIFY_SIGNALLED_OPERATION(
                setProcessSelectedServer( process.m_tag, process.m_selected ),
                0xedb00116,
                hkVdbError::CMD_HANDLER_ERROR,
                result );
        }
        m_dirtyProcesses.clear();
    }
}

hkResult hkVdbProcessHandler::processRegisteredCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader )
{
    using namespace hkVdbCmdType;

    
    if ( m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::REPLAYING ) )
    {
        return HK_SUCCESS;
    }

    m_registeredCmd.m_cmdType = type;
    m_registeredCmd.m_tag = dataReader.read32();

    // read text
    hkStringBuf buf;
    READ_STRING( buf, dataReader );
    m_registeredCmd.m_name = buf;

    

    HK_VDB_VERIFY_CONDITION( dataReader.isOk(), 0xedb00034, hkVdbError::CMD_HANDLER_READ_ERROR );

    
    
    
    
    
    //HK_VDB_VERIFY_CONDITION_MSG(
    //  !m_registeredProcesses.contains( m_registeredCmd.m_tag ),
    //  0xedb00035,
    //  hkVdbError::PROCESS_REGISTERED,
    //  "Process with tag " << m_registeredCmd.m_tag << " has already been registered" );

    hkVdbMap<int, Process>::Iterator iter = m_registeredProcesses.findOrInsertKey(
        m_registeredCmd.m_tag,
        Process() );

    Process& process = m_registeredProcesses.getValue( iter );
    process.m_tag = m_registeredCmd.m_tag;
    process.m_name = m_registeredCmd.m_name;

    // Signal
    hkVdbSignalResult result;
    m_processCmdReceived.fire( m_registeredCmd, result );
    HK_VDB_VERIFY_SIGNAL_RESULT( result );

    return HK_SUCCESS;
}

hkResult hkVdbProcessHandler::processUpdatedCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader )
{
    using namespace hkVdbCmdType;

    
    if ( m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::REPLAYING ) )
    {
        return HK_SUCCESS;
    }

    m_tweakedCmd.m_tag = dataReader.read32();
    {
        
    }

    HK_VDB_VERIFY_CONDITION( dataReader.isOk(), 0xedb00036, hkVdbError::CMD_HANDLER_READ_ERROR );

    // Signal
    hkVdbSignalResult result;
    m_processCmdReceived.fire( m_tweakedCmd, result );
    HK_VDB_VERIFY_SIGNAL_RESULT( result );

    return HK_SUCCESS;
}

hkResult hkVdbProcessHandler::processSelectedCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader )
{
    using namespace hkVdbCmdType;

    
    if ( m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::REPLAYING ) )
    {
        return HK_SUCCESS;
    }

    const hkInt32 tag = dataReader.read32();

    HK_VDB_VERIFY_CONDITION( dataReader.isOk(), 0xedb00037, hkVdbError::CMD_HANDLER_READ_ERROR );

    // Default processes require a handshake, so we must notify the server of selection received.
    // The catch is once-off viewers, which need to get skipped in this hand-shaking.
    hkVdbMap<int, Process>::Iterator iter = m_registeredProcesses.findKey( tag );
    hkBool32 notifyServer = m_registeredProcesses.isValid( iter ) && !m_registeredProcesses.getValue( iter ).m_name.startsWith( "*" );

    return setProcessSelectedInternal( tag, ( type == hkVdbCmdType::SELECT_PROCESS ), notifyServer );
}

hkResult hkVdbProcessHandler::setProcessSelectedServer( int tag, hkBool32 selected ) const
{
    using namespace hkVdbCmdType;

    if ( selected )
    {
        if ( hkVdbScopedStreamWriter writer = m_output->startCmd( CREATE_PROCESS ) )
        {
            hkVdbOStream oarchive( writer );
            oarchive.write32( tag );
        }
        else
        {
            HK_VDB_SIGNAL_REPORTER_ERROR( *m_output, hkVdbError::CMD_HANDLER_WRITE_ERROR );
            return HK_FAILURE;
        }
    }
    else
    {
        if ( hkVdbScopedStreamWriter writer = m_output->startCmd( DELETE_PROCESS ) )
        {
            hkVdbOStream oarchive( writer );
            oarchive.write32( tag );
        }
        else
        {
            HK_VDB_SIGNAL_REPORTER_ERROR( *m_output, hkVdbError::CMD_HANDLER_WRITE_ERROR );
            return HK_FAILURE;
        }
    }

    return HK_SUCCESS;
}

hkResult hkVdbProcessHandler::setProcessSelectedInternal( int tag, hkBool32 selected, hkBool32 notifyServer )
{
    hkVdbMap<int, Process>::Iterator iter = m_registeredProcesses.findKey( tag );
    HK_VDB_VERIFY_CONDITION_MSG(
        m_registeredProcesses.isValid( iter ),
        0xedb00033,
        hkVdbError::PROCESS_NOT_REGISTERED,
        "Process with tag " << tag << " has not been registered" );

    Process& process = m_registeredProcesses.getValue( iter );
    if ( bool( process.m_selected ) != bool( selected ) )
    {
        // Set our selection state (needs to happen first for reentrancy purposes)
        process.m_selected = selected;

        // Notify server if requested
        if ( notifyServer )
        {
            // Queue for notifying server later if we are replaying and redo the current frame to ensure
            // proper visualizations.
            if ( m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::REPLAYING ) ||
                ( m_playbackHandler->getState() == hkVdbPlaybackState::PAUSED ) )
            {
                m_dirtyProcesses.insert( tag );
                m_playbackHandler->redoCurrentFrame();
            }
            // Otherwise, notify now
            else if ( setProcessSelectedServer( process.m_tag, selected ).isFailure() ) return HK_FAILURE;
        }

        // Signal selection change
        hkVdbSignalResult result;
        m_selectionChangedCmd.m_cmdType = process.m_selected ? hkVdbCmdType::SELECT_PROCESS : hkVdbCmdType::UNSELECT_PROCESS;
        m_selectionChangedCmd.m_tag = tag;
        m_processCmdReceived.fire( m_selectionChangedCmd, result );
        HK_VDB_VERIFY_SIGNAL_RESULT( result );

        
        // Viewers that start with "*" are once-off and don't remain selected
        process.m_selected = ( process.m_selected && !process.m_name.startsWith( "*" ) );

        // Signal unselected once-offs
        if ( bool( process.m_selected ) != bool( selected ) )
        {
            m_selectionChangedCmd.m_cmdType = process.m_selected ? hkVdbCmdType::SELECT_PROCESS : hkVdbCmdType::UNSELECT_PROCESS;
            m_processCmdReceived.fire( m_selectionChangedCmd, result );
            HK_VDB_VERIFY_SIGNAL_RESULT( result );
        }
    }

    return HK_SUCCESS;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
