// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/System/Stopwatch/hkStopwatch.h>
#include <Common/Visualize/hkVisualDebuggerCmdType.h>

#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbCmdHandler.h>

class hkVdbCmdInput;
struct hkVdbFrame;
struct hkVdbCmd;
class hkVdbClient;
class hkVdbProgressReporter;
class hkVdbCmdDispatcher;
class hkVdbSetupHandler;
class hkVdbDisplayHandler;
class hkVdbProcessHandler;
class hkVdbCache;
class hkVdbPersistentStateCache;
struct Playhead;
struct MaxPlayhead;
struct BufferPlayhead;
struct ConnectionPlayhead;













//////////////////////////////////////////////////////////////////////////
/// Enum describing the playback state
//////////////////////////////////////////////////////////////////////////
struct hkVdbPlaybackState
{
    enum Enum
    {
        /// The playback handler is currently playing as normal.
        PLAYING = 0,

        /// The playback handler will not produce new cmds, it is paused.
        PAUSED,
    };
};

//////////////////////////////////////////////////////////////////////////
/// Enum describing the playback direction
//////////////////////////////////////////////////////////////////////////
struct hkVdbPlaybackDirection
{
    enum Enum
    {
        /// If we are playing, we will be playing forward.
        FORWARD = 0,

        /// If we are playing, we will be playing backward (reverse).
        BACKWARD
    };
};

//////////////////////////////////////////////////////////////////////////
/// Enum describing modes for jumping between frames.
//////////////////////////////////////////////////////////////////////////
struct hkVdbPlaybackFrameMode
{
    enum Enum
    {
        /// Jump to the new frame as fast as possible.
        /// Some data will not be up-to-date until you use COMPLETE mode.
        FAST = 0,

        /// Jump to the new frame processing all available data.
        COMPLETE
    };
};

//////////////////////////////////////////////////////////////////////////
/// Flags for affecting playback.
//////////////////////////////////////////////////////////////////////////
struct hkVdbPlaybackFlags : public hkFlagsEx<hkUint16>
{
    enum Bits
    {
        //
        // User flags
        //

        /// No flags.
        NONE = 0,

        /// When set, the playback will be throttled to the vdb step value (hkVisualDebugger::step(...))
        /// When cleared, the playback will happen as fast as possible.
        THROTTLE_FRAMERATE = ( 1 << 0 ),

        /// Default flags.
        DEFAULT = ( THROTTLE_FRAMERATE ),

        /// Mask for user flag values.
        USER_FLAGS_MASK = ( 0x00ff ),

        //
        // Implementation flags (cannot be set externally)
        //

        /// Set if we are currently recording.
        RECORDING = ( 1 << 8 ),

        /// Set if we are currently replaying from the cmd input buffer.
        REPLAYING = ( 1 << 9 ),

        /// Set if we are replaying fast (see hkVdbPlaybackFrameModehkVdbPlaybackFrameMode).
        FAST_REPLAY = ( 1 << 10 ),

        /// Set when we are caching frames for later replay (as opposed to playing real-time).
        CACHING = ( 1 << 11 ),

        /// Set when are at the end of a frame boundary.
        FRAME_ENDED = ( 1 << 12 ),

        /// Set when all available cmds have been processed (dispatched).
        AVAILABLE_CMDS_PROCESSED = ( 1 << 13 ),

        /// Mask for implementation flag values.
        IMPLEMENTATION_FLAGS_MASK = ( 0xff00 ),
    };

    HK_DECLARE_FLAGS_EX_CLASS( hkVdbPlaybackFlags, hkVdbPlaybackFlags::Bits );
};

//////////////////////////////////////////////////////////////////////////
/// Enum describing different playheads.
/// Usually PROCESSED <= BUFFER <= CONNECTION.
/// If PROCESSED is moved passed BUFFER, then we enter into a CACHING mode
/// where we catch-up and don't playback in real time.
//////////////////////////////////////////////////////////////////////////
struct hkVdbPlayheadType
{
    enum Enum
    {
        /// Indicates where in the cmd input buffer we are currently playing from.
        BUFFER = 1,

        /// Set to the max frame/cmd we have processed from the cmd input buffer.
        PROCESSED = 2,

        /// Set to the end of the cmd input buffer where we are reading information
        /// in from the app connection.
        CONNECTION = 3,
    };
};

//////////////////////////////////////////////////////////////////////////
/// A struct describing a conceptual "playhead" along a timeline defined
/// by the cmd input buffer.
//////////////////////////////////////////////////////////////////////////
struct hkVdbPlayhead
{
    HK_DECLARE_CLASS( hkVdbPlayhead, New );
    hkVdbPlayhead( hkVdbCmdInput& input );

    hkInt64 compare( const hkVdbPlayhead& other ) const;
    HK_INLINE bool operator <( const hkVdbPlayhead& other ) const { return compare( other ) < 0; }
    HK_INLINE bool operator >( const hkVdbPlayhead& other ) const { return compare( other ) > 0; }
    HK_INLINE bool operator <=( const hkVdbPlayhead& other ) const { return compare( other ) <= 0; }
    HK_INLINE bool operator >=( const hkVdbPlayhead& other ) const { return compare( other ) >= 0; }
    HK_INLINE bool operator ==( const hkVdbPlayhead& other ) const { return compare( other ) == 0; }

    /// Gets the frame the playhead is in.
    const hkVdbFrame* getFrame() const;

    /// Gets the specific cmd the playhead is on.
    const hkVdbCmd* getCmd() const;

    /// Gets the frame number the playhead is in.
    hkInt64 getFrameNumber() const;

    /// Gets the cmd index the playhead is on.
    hkInt64 getCmdIndex() const;

    /// hkVdbCmdInput::Iterator which iterates over frames in the buffer.
    
    hkInt32 m_frameIt;

    /// hkVdbFrame::Iterator which iterates over cmds in the frame.
    hkInt32 m_cmdIt;

protected:
    hkRefPtr<hkVdbCmdInput> m_input;
};

///
/// Const value representing the server frame rate (whatever it may be).
const hkReal HK_VDB_SERVER_FRAME_RATE = 0.0f;

//////////////////////////////////////////////////////////////////////////
/// A handler for playback-related cmds.
/// This is the main class for controlling how playback occurs in the system.
//////////////////////////////////////////////////////////////////////////
class HK_EXPORT_VDB hkVdbPlaybackHandler : public hkVdbCmdHandler<hkVdbCmdHandlerType::PLAYBACK>
{
public:

    HK_DECLARE_CLASS( hkVdbPlaybackHandler, New );
    hkVdbPlaybackHandler();
    ~hkVdbPlaybackHandler();

    //
    // hkVdbCmdHandler interface
    //

    virtual hkResult registerSelf( hkVdbClient& client ) HK_OVERRIDE;
    virtual hkResult unregisterSelf( hkVdbClient& client ) HK_OVERRIDE;
    virtual void onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection ) HK_OVERRIDE;

    //
    // Control and Query
    //

    
    /// Get the current playback state of the handler.
    HK_INLINE hkVdbPlaybackState::Enum getState() const { return m_state.m_targetState; }

    /// Set the current playback state of the handler.
    hkResult setState( hkVdbPlaybackState::Enum state );

    /// Get the current playback direction of the handler.
    HK_INLINE hkVdbPlaybackDirection::Enum getDirection() const { return m_state.m_targetDirection; }

    /// Set the current playback direction of the handler.
    hkResult setDirection( hkVdbPlaybackDirection::Enum dir );

    /// Get flags affecting playback or indicating special playback properties.
    HK_INLINE hkVdbPlaybackFlags getFlags() const { return m_state.m_targetFlags; }

    /// Set user flags affecting playback (implementation flags may not be used).
    hkResult setFlags( hkVdbPlaybackFlags flags );

    /// Get the target frame rate used when THROTTLE_FRAMERATE flag is set.
    HK_INLINE hkReal getFrameRate() const { return m_state.m_targetFrameRate; }

    /// Set the target frame rate used when THROTTLE_FRAMERATE flag is set.
    hkResult setFrameRate( hkReal fps = HK_VDB_SERVER_FRAME_RATE );

    /// Get the last frame that we finished (FRAME_ENDED was set for this frame already).
    
    HK_INLINE hkUint32 getLastCompletedFrame() const { return m_state.m_previousFrame; }

    /// Get the current frame we are playing from.
    HK_INLINE hkUint32 getCurrentFrame() const { return m_state.m_targetFrame; }

    /// Get the next frame given our internal state (playing fwd/bwd, etc.).
    hkInt64 getNextFrame( hkInt64 fromFrame = -1 ) const;

    /// Get a safe frame range for use with setCurrentFrame.
    hkResult getFrameRange( hkUint32& minFrameOut, hkUint32& maxFrameOut ) const;

    /// Jump to any frame using the given mode for processing.
    hkResult setCurrentFrame( hkUint32 frameNumber, hkVdbPlaybackFrameMode::Enum mode );

    /// For advanced use only.
    /// Redo/reprocess the data from the current frame.
    hkResult redoCurrentFrame();

    /// Save replay from cmd input buffer. Optionally from [startFrame, endFrame] inclusively,
    /// otherwise the entire replay buffer is saved.
    hkResult saveReplay( const char* filepath, hkInt64 startFrame = -1, hkInt64 endFrame = -1) const;
    hkResult saveReplay( hkVdbConnection& connection, hkInt64 startFrame = -1, hkInt64 endFrame = -1 ) const;

    /// Start recording. This simply sets the hkVdbClient's SINK connection.
    hkResult startRecording( const char* filepath );
    hkResult startRecording( hkVdbConnection& connection );

    /// Stop recording. This simply clears the hkVdbClient's SINK connection.
    hkResult stopRecording();

    /// This will cause the playback handler to read as much as is possible from the provided connection
    /// (or the default APP connection if not provided) and dispatch those commands to the system.
    hkResult syncToConnection();
    hkResult syncToConnection( hkVdbConnection& connection );

    /// Query the playhead positions.
    hkVdbPlayhead getPlayhead( hkVdbPlayheadType::Enum type ) const;
    

    //
    // Signals
    //

    
    /// Contains information about playback which is provided to signals.
    /// Listeners can act when a value of interest changes, for instance
    /// maybe special behavior is executed when "unpausing":
    ///     if ( m_previousState != m_targetState && m_targetState == hkVdbPlaybackState::PLAYING )
    ///     { /* act */ }
    struct PlaybackInfo
    {
        HK_DECLARE_CLASS( PlaybackInfo, New );
        PlaybackInfo() :
            m_previousState( hkVdbPlaybackState::PLAYING ),
            m_targetState( hkVdbPlaybackState::PLAYING ),
            m_previousDirection( hkVdbPlaybackDirection::FORWARD ),
            m_targetDirection( hkVdbPlaybackDirection::FORWARD ),
            m_previousFlags( hkVdbPlaybackFlags::DEFAULT ),
            m_targetFlags( hkVdbPlaybackFlags::DEFAULT ),
            m_previousFrameRate( HK_VDB_SERVER_FRAME_RATE ),
            m_targetFrameRate( HK_VDB_SERVER_FRAME_RATE ),
            m_previousFrame( 0 ),
            m_targetFrame( 0 )
        {}

        hkVdbPlaybackState::Enum m_previousState;
        hkVdbPlaybackState::Enum m_targetState;
        hkVdbPlaybackDirection::Enum m_previousDirection;
        hkVdbPlaybackDirection::Enum m_targetDirection;
        hkVdbPlaybackFlags m_previousFlags;
        hkVdbPlaybackFlags m_targetFlags;
        hkReal m_previousFrameRate;
        hkReal m_targetFrameRate;
        hkUint32 m_previousFrame;
        hkUint32 m_targetFrame;
        HK_INLINE hkBool32 flagWasSet( hkVdbPlaybackFlags flag ) const
        {
            return
                m_previousFlags.noneIsSet( flag ) &&
                m_targetFlags.allAreSet( flag );
        }
        HK_INLINE hkBool32 flagWasCleared( hkVdbPlaybackFlags flag ) const
        {
            return
                m_previousFlags.allAreSet( flag ) &&
                m_targetFlags.noneIsSet( flag );
        }
        HK_INLINE hkBool32 flagChanged( hkVdbPlaybackFlags flag ) const
        {
            return
                ( m_previousFlags & flag ) != ( m_targetFlags & flag );
        }
    };

    HK_DECLARE_SIGNAL( PlaybackInfoReceivedSignal, hkSignal2< const PlaybackInfo&, hkVdbSignalResult& > );
    /// Called whenever playback state changes.
    PlaybackInfoReceivedSignal m_playbackInfoReceived;
    

    //
    // Internal use
    //

    void onDisconnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection );
    void onStepCompletedSignal( const hkVdbFrame& frame, const hkVdbCmd& cmd );
    void onFrameEvictedSignal( hkVdbCmdInput& input, const hkVdbFrame& frame );
    void onCmdDispatchingSignal( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbSignalResult& result );
    void onCmdDispatchedSignal( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbSignalResult& result );
    HK_INLINE void onPlaybackInfoReceivedSignalInternal( const PlaybackInfo& info, hkVdbSignalResult& result ) { m_playbackInfoReceived.fire( info, result ); }
    hkResult processPlaybackInfoCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader );
    hkResult processSaveReplayInfoCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader );

protected:

    HK_DECLARE_SIGNAL( PlaybackInfoReceivedSignalInternal, hkSignal2< const PlaybackInfo&, hkVdbSignalResult& > );
    PlaybackInfoReceivedSignalInternal m_playbackInfoReceivedInternal;

    hkResult getNextCmd(
        hkVdbConnection* connection,
        const hkVdbFrame*& frameOut,
        const hkVdbCmd*& cmdOut );
    hkBool32 shouldThrottlingConnection( const hkVdbFrame& frame );
    hkResult resetPlaybackStopwatch();
    hkResult updateStateAndSignal( hkVdbPlaybackState::Enum state );
    hkResult updateDirectionAndSignal( hkVdbPlaybackDirection::Enum dir );
    hkResult updateFlagsAndSignal( hkVdbPlaybackFlags flags );
    hkResult updateFrameRateAndSignal( hkReal fps );
    hkResult updateFrameAndSignal( hkUint32 frameNumber );
    hkResult updateReplayFlagAndSignal();
    hkResult updateProcessedAvailableFlagAndSignal();
    void checkPlayheads( const hkVdbFrame* frame, const hkVdbCmd* cmd );
    hkResult resolveFrameArgs(
        hkInt64 startFrameIn,
        hkInt64 endFrameIn,
        hkBool32 includeMargins,
        hkUint32& startFrameOut,
        hkUint32& endFrameOut ) const;
    hkResult setCurrentFrameInternal( hkUint32 frameNumber );

    
    

    hkViewPtr<hkVdbClient> m_client;
    hkRefPtr<hkVdbProgressReporter> m_progressReporter;
    hkRefPtr<hkVdbCmdInput> m_input;
    hkRefPtr<hkVdbCmdDispatcher> m_dispatcher;
    hkRefPtr<hkVdbSetupHandler> m_setupHandler;
    hkRefPtr<hkVdbDisplayHandler> m_displayHandler;
    hkRefPtr<hkVdbProcessHandler> m_processHandler;
    hkRefPtr<hkVdbCache> m_cache;
    hkRefPtr<hkVdbPersistentStateCache> m_persistentCache;

    PlaybackInfo m_state;

    hkStopwatch m_playbackStopwatch;
    hkReal m_playbackStartTime;
    hkUint32 m_playbackStartFrame;

    hkRefPtr<BufferPlayhead> m_bufferPlayhead;
    hkRefPtr<ConnectionPlayhead> m_connectionPlayhead;
    hkRefPtr<MaxPlayhead> m_processedPlayhead;

    friend class hkVdbClient;
    friend struct Playhead;
    friend struct MaxPlayhead;
    friend struct BufferPlayhead;
    friend struct ConnectionPlayhead;

    // For internal subscription to get around COM-4170
    friend class hkVdbDisplayHandler;
    friend class hkVdbObjectHandler;
    friend class hkVdbProcessHandler;
    friend class hkVdbStatsHandler;
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
