// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbServices/hkVdbServices.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbPlaybackHandler.h>

#include <VisualDebugger/VdbServices/hkVdbClient.h>
#include <VisualDebugger/VdbServices/System/hkVdbProgressReporter.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmd.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdInput.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdOutput.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdDispatcher.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbSetupHandler.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbDisplayHandler.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbProcessHandler.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbInternalHandler.h>
#include <VisualDebugger/VdbServices/System/Cache/hkVdbCache.h>
#include <VisualDebugger/VdbServices/System/Cache/hkVdbPersistentStateCache.h>
#include <VisualDebugger/VdbServices/System/Connection/hkVdbFileConnection.h>
#include <VisualDebugger/VdbServices/System/Utils/hkVdbBlockingWriter.h>







#define DEBUG_LOG_IDENTIFIER "vdb.Cmd.Handler.Playback"
#include <Common/Base/System/Log/hkLog.hxx>

//#define REPORT_PLAYHEADS
//#define REPORT_DIRECTION_CHANGES
//#define REPORT_STATE_CHANGES
//#define REPORT_FLAG_CHANGES
//#define REPORT_FRAME_CHANGES
//#define REPORT_FRAME_RATE_CHANGES
//#define REPORT_THROTTLING_CHANGES

struct PlayheadDiff
{
    HK_DECLARE_CLASS( PlayheadDiff, New );
    PlayheadDiff( hkInt64 numFrames = 0, hkInt64 numCmds = 0 ) :
        m_numFrames( numFrames ), m_numCmds( numCmds )
    {}
    hkInt64 m_numFrames;
    hkInt64 m_numCmds;
    PlayheadDiff operator -( const PlayheadDiff& other ) { return PlayheadDiff( m_numFrames - other.m_numFrames ); }
    PlayheadDiff operator +( const PlayheadDiff& other ) { return PlayheadDiff( m_numFrames + other.m_numFrames ); }
};

struct PlayheadProcessingFlags : public hkFlagsEx<hkUint8>
{
    enum Bits
    {
        // Skip
        NONE = 0,
        // Include the current cmd in processing
        INCLUSIVE = ( 1 << 0 ),
        // Include non-current cmds in processing
        EXCLUSIVE = ( 1 << 1 ),
        // Include step-cmds in processing, otherwise STEP is a termination condition
        STEP = ( 1 << 2 )
    };

    HK_DECLARE_FLAGS_EX_CLASS( PlayheadProcessingFlags, PlayheadProcessingFlags::Bits );
};

struct Playhead : public hkVdbPlayhead, public hkVdbDefaultErrorReporter
{
    HK_DECLARE_CLASS( Playhead, New );
    Playhead( hkVdbPlaybackHandler& handler );
    Playhead( const Playhead& other );

    virtual hkResult processFrame( PlayheadProcessingFlags flags );
    virtual hkResult advance();
    virtual Playhead peek() const;
    virtual hkBool32 isValid() const;

    PlayheadDiff operator -( const Playhead& other ) const;
    virtual Playhead& operator =( const Playhead& other ) { m_frameIt = other.m_frameIt; m_cmdIt = other.m_cmdIt; return *this; }

    hkResult set( const hkVdbFrame* frame, const hkVdbCmd* cmd );

protected:
    hkVdbPlaybackHandler& m_handler;
};

struct MaxPlayhead : public Playhead
{
    HK_DECLARE_CLASS( MaxPlayhead, New );
    MaxPlayhead( hkVdbPlaybackHandler& handler );
    MaxPlayhead( const Playhead& other );

    hkBool32 update( const hkVdbFrame* frame, const hkVdbCmd* cmd );
    hkResult tryAdvanceTo( const Playhead& other, hkBool32& reachedOtherOut );

    MaxPlayhead& operator =( const Playhead& other ) { return static_cast< MaxPlayhead& >( Playhead::operator =( other ) ); }
};

struct BufferPlayhead : public Playhead
{
    enum InterFrameLoc
    {
        START = 0,
        END
    };

    HK_DECLARE_CLASS( BufferPlayhead, New );
    BufferPlayhead( hkVdbPlaybackHandler& handler );
    BufferPlayhead( const Playhead& other );

    virtual hkResult processFrame( PlayheadProcessingFlags flags ) HK_OVERRIDE;
    virtual hkResult advance() HK_OVERRIDE;
    virtual Playhead peek() const HK_OVERRIDE;
    virtual hkBool32 isValid() const HK_OVERRIDE;

    hkResult play( const hkVdbFrame*& frameOut, const hkVdbCmd*& cmdOut );
    hkResult set( hkUint32 frameNumber, InterFrameLoc loc );

    virtual BufferPlayhead& operator =( const Playhead& other ) HK_OVERRIDE
    {
        Playhead::operator =( other );
        m_nextFrameIt = m_frameIt;
        m_nextCmdIt = m_cmdIt;
        return *this;
    }

    hkVdbCmdInput::Iterator m_nextFrameIt;
    hkVdbFrame::Iterator m_nextCmdIt;

protected:
    hkResult setNoSignal( hkUint32 frameNumber, InterFrameLoc loc );
    hkBool32 m_atBeginningPlayingBackwards;
};

struct ConnectionPlayhead : public Playhead
{
    HK_DECLARE_CLASS( ConnectionPlayhead, New );
    ConnectionPlayhead( hkVdbPlaybackHandler& handler );
    ConnectionPlayhead( const Playhead& other );

    hkResult play( hkVdbConnection& connection, const hkVdbFrame*& frameOut, const hkVdbCmd*& cmdOut );

    ConnectionPlayhead& operator =( const Playhead& other ) { return static_cast< ConnectionPlayhead& >( Playhead::operator =( other ) ); }
};

namespace
{
    hkResult handlePlaybackInfoCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
        hkVdbPlaybackHandler* handler = reinterpret_cast< hkVdbPlaybackHandler* >( userHandle );
        return handler->processPlaybackInfoCmd( cmd.getType(), protocol, dataReader );
    }

    hkResult handleSaveReplayInfoCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
        hkVdbPlaybackHandler* handler = reinterpret_cast< hkVdbPlaybackHandler* >( userHandle );
        return handler->processSaveReplayInfoCmd( cmd.getType(), protocol, dataReader );
    }
};

hkVdbPlaybackHandler::hkVdbPlaybackHandler() :
    hkVdbCmdHandler<hkVdbCmdHandlerType::PLAYBACK>( &s_debugLog ),
    m_playbackStartTime( -1 ),
    m_playbackStartFrame( 0 )
{
    // Internal signaler to get around COM-4170.
    // The first subscriber to this signal is our fwder to m_playbackInfoReceived.
    // When other subscribers from the framework listen to this signal, they will get
    // prepended to the subscribers list and happen *before* the fwder to the public
    // m_playbackInfoReceived signal.
    HK_SUBSCRIBE_TO_SIGNAL( m_playbackInfoReceivedInternal, this, hkVdbPlaybackHandler );
}

hkVdbPlaybackHandler::~hkVdbPlaybackHandler()
{}

hkResult hkVdbPlaybackHandler::registerSelf( hkVdbClient& client )
{
    using namespace hkVdbCmdType;

    bool succeeded = true;
    hkVdbCmdDispatcher& dispatcher = client.getCmdDispatcher();

    succeeded &= ( dispatcher.registerHandler( SET_CLIENT_PLAYBACK_STATE, handlePlaybackInfoCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( SET_CLIENT_PLAYBACK_FRAME, handlePlaybackInfoCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( SAVE_REPLAY, handleSaveReplayInfoCmdFunc, this, this ).isSuccess() );

    m_client = &client;
    m_progressReporter = &client.getProgressReporter();
    m_input = &client.getCmdInput();
    m_dispatcher = &client.getCmdDispatcher();
    m_setupHandler = client.getCmdHandler<hkVdbSetupHandler>();
    m_displayHandler = client.getCmdHandler<hkVdbDisplayHandler>();
    m_processHandler = client.getCmdHandler<hkVdbProcessHandler>();
    m_cache = &client.getCache();
    m_persistentCache = &client.getPersistentStateCache();

    HK_SUBSCRIBE_TO_SIGNAL( client.m_disconnected, this, hkVdbPlaybackHandler );
    HK_SUBSCRIBE_TO_SIGNAL( client.m_stepCompleted, this, hkVdbPlaybackHandler );
    HK_SUBSCRIBE_TO_SIGNAL( m_input->m_frameEvicted, this, hkVdbPlaybackHandler );
    HK_SUBSCRIBE_TO_SIGNAL( m_dispatcher->m_cmdDispatching, this, hkVdbPlaybackHandler );
    HK_SUBSCRIBE_TO_SIGNAL( m_dispatcher->m_cmdDispatched, this, hkVdbPlaybackHandler );

    m_state.m_previousFrame = m_state.m_targetFrame;
    m_state.m_targetFrame = m_input->getCurrentFrameNumber();

    // Note: must be after m_input is initialized
    m_bufferPlayhead = hkRefNew<BufferPlayhead>( new BufferPlayhead( *this ) );
    m_connectionPlayhead = hkRefNew<ConnectionPlayhead>( new ConnectionPlayhead( *this ) );
    m_processedPlayhead = hkRefNew<MaxPlayhead>( new MaxPlayhead( *this ) );

    HK_VDB_VERIFY_REPORTER_CONDITION( succeeded, dispatcher, hkVdbError::CMD_HANDLER_REGISTRATION_ERROR );

    return HK_SUCCESS;
}

hkResult hkVdbPlaybackHandler::unregisterSelf( hkVdbClient& client )
{
    using namespace hkVdbCmdType;

    bool succeeded = true;
    hkVdbCmdDispatcher& dispatcher = client.getCmdDispatcher();

    succeeded &= ( dispatcher.unregisterHandler( SET_CLIENT_PLAYBACK_STATE ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( SET_CLIENT_PLAYBACK_FRAME ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( SAVE_REPLAY ).isSuccess() );

    client.m_disconnected.unsubscribeAll( this );
    client.m_stepCompleted.unsubscribeAll( this );
    m_input->m_frameEvicted.unsubscribeAll( this );
    m_dispatcher->m_cmdDispatching.unsubscribeAll( this );
    m_dispatcher->m_cmdDispatched.unsubscribeAll( this );

    m_client = HK_NULL;
    
    
    
    
    //m_progressReporter = HK_NULL;
    //m_input = HK_NULL;
    //m_dispatcher = HK_NULL;
    //m_setupHandler = HK_NULL;
    //m_displayHandler = HK_NULL;
    //m_processHandler = HK_NULL;
    //m_cache = HK_NULL;
    //m_persistentCache = HK_NULL;

    HK_VDB_VERIFY_REPORTER_CONDITION( succeeded, dispatcher, hkVdbError::CMD_HANDLER_REGISTRATION_ERROR );

    return HK_SUCCESS;
}

void hkVdbPlaybackHandler::onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection )
{
    if ( use == hkVdbConnectionUse::APPLICATION )
    {
        
        
        
        // Reset frame
        {
            // Signaling here causes more problems than it solves.
            // Mostly because this isn't distinguishable from a skip to start.
            // It seems reasonable that listeners who are concerned about initialization
            // should also listen to connected signal anyway.
            m_state.m_targetFrame = m_state.m_previousFrame = 0;
            m_state.m_targetFlags.clear( hkVdbPlaybackFlags::FRAME_ENDED );
        }
        // Reset replaying
        {
            hkVdbPlaybackFlags flags = m_state.m_targetFlags;
            flags.clear( hkVdbPlaybackFlags::REPLAYING );
            updateFlagsAndSignal( flags );
            m_state.m_previousFlags.clear( hkVdbPlaybackFlags::REPLAYING );
        }
        // Reset playheads
        {
            m_bufferPlayhead = hkRefNew<BufferPlayhead>( new BufferPlayhead( *this ) );
            m_connectionPlayhead = hkRefNew<ConnectionPlayhead>( new ConnectionPlayhead( *this ) );
            m_processedPlayhead = hkRefNew<MaxPlayhead>( new MaxPlayhead( *this ) );
        }
        resetPlaybackStopwatch();
    }
    else if ( use == hkVdbConnectionUse::SINK )
    {
        updateFlagsAndSignal( m_state.m_targetFlags | hkVdbPlaybackFlags::RECORDING );
    }
}

hkResult hkVdbPlaybackHandler::setState( hkVdbPlaybackState::Enum state )
{
#ifdef REPORT_FLAG_CHANGES
    if ( m_state.m_targetState != state ) HK_REPORT( "User" );
#endif
    return updateStateAndSignal( state );
}

hkResult hkVdbPlaybackHandler::setDirection( hkVdbPlaybackDirection::Enum dir )
{
#ifdef REPORT_FLAG_CHANGES
    if ( m_state.m_targetDirection != dir ) HK_REPORT( "User" );
#endif
    return updateDirectionAndSignal( dir );
}

hkResult hkVdbPlaybackHandler::setFlags( hkVdbPlaybackFlags flags )
{
    HK_VDB_VERIFY_CONDITION_MSG(
        flags.noneIsSet( hkVdbPlaybackFlags::IMPLEMENTATION_FLAGS_MASK ),
        0xedb00105,
        hkVdbError::INVALID_ARGUMENTS,
        "You can not set implementation flags" );

#ifdef REPORT_FLAG_CHANGES
    if ( m_state.m_targetFlags != flags ) HK_REPORT( "User" );
#endif
    return updateFlagsAndSignal( flags );
}

hkResult hkVdbPlaybackHandler::setFrameRate( hkReal fps )
{
#ifdef REPORT_FRAME_RATE_CHANGES
    if ( m_state.m_targetFrameRate != fps ) HK_REPORT( "User" );
#endif
    return updateFrameRateAndSignal( fps );
}

hkInt64 hkVdbPlaybackHandler::getNextFrame( hkInt64 fromFrame ) const
{
    // Branchless calculation which will use the current frame if fromFrame is -1,
    // otherwise uses fromFrame.  It then either adds 1 or subtracts one based on
    // playback direction.
    return
        hkInt64(
            bool( fromFrame != -1 ) * fromFrame +
            bool( fromFrame == -1 ) * m_state.m_targetFrame +
            bool( m_state.m_targetDirection == hkVdbPlaybackDirection::FORWARD ) -
            bool( m_state.m_targetDirection == hkVdbPlaybackDirection::BACKWARD ) );
}

hkResult hkVdbPlaybackHandler::getFrameRange( hkUint32& minFrameOut, hkUint32& maxFrameOut ) const
{
    HK_VDB_VERIFY_CONDITION_MSG(
        m_input,
        0xedb00039,
        hkVdbError::CMD_HANDLER_NOT_REGISTERED,
        "Cannot call getFrameRange before playback handler is registered" );

    // If we don't use last completed frame for max, we can cause the connection
    // to advance while paused and "skipping to end". There might be a way around
    // that behavior, but for now we just max at last completed.
    minFrameOut = m_input->getStartingFrameNumber();
    maxFrameOut = m_input->getLastCompletedFrameNumber();

    return HK_SUCCESS;
}

hkResult hkVdbPlaybackHandler::setCurrentFrame( hkUint32 frameNumber, hkVdbPlaybackFrameMode::Enum mode )
{
    hkVdbPlaybackFlags flags = m_state.m_targetFlags;
    flags.orWith( ( mode == hkVdbPlaybackFrameMode::FAST ) * hkVdbPlaybackFlags::FAST_REPLAY );
    flags.clear( ( mode == hkVdbPlaybackFrameMode::COMPLETE ) * hkVdbPlaybackFlags::FAST_REPLAY );

    // If we haven't changed frame, but our mode has changed, apply flags and redo current frame.
    if ( ( m_state.m_targetFrame == frameNumber ) && ( m_state.m_targetFlags != flags ) )
    {
        updateFlagsAndSignal( flags );
        return redoCurrentFrame();
    }
    // If the mode hasn't change, we intentionally consider this a no-op.
    // This is to ensure the framework doesn't do redundant work unless it's explicitly requested (redoCurrentFrame()).
    else if ( m_state.m_targetFrame == frameNumber )
    {
        return HK_SUCCESS;
    }
    // Else, proceed with updating the framework to the requested frame and mode
    else
    {
        HK_VDB_VERIFY_CONDITION_MSG(
            m_input,
            0xedb00040,
            hkVdbError::CMD_HANDLER_NOT_REGISTERED,
            "Cannot call setCurrentFrame before playback handler is registered" );

        if( resolveFrameArgs( frameNumber, frameNumber, true, frameNumber, frameNumber ).isFailure() ) return HK_FAILURE;
        updateFlagsAndSignal( flags );
        return setCurrentFrameInternal( frameNumber );
    }
}

hkResult hkVdbPlaybackHandler::redoCurrentFrame()
{
    HK_VDB_VERIFY_CONDITION_MSG(
        m_input,
        0x0, 
        hkVdbError::CMD_HANDLER_NOT_REGISTERED,
        "Cannot call setCurrentFrame before playback handler is registered" );

    m_state.m_targetFlags.clear( hkVdbPlaybackFlags::FRAME_ENDED );
    return setCurrentFrameInternal( m_state.m_targetFrame );
}

hkResult hkVdbPlaybackHandler::setCurrentFrameInternal( hkUint32 frameNumber )
{
    // Set our buffer playhead
#ifdef REPORT_FRAME_CHANGES
    HK_REPORT( "User" );
#endif
    if( m_bufferPlayhead->set( frameNumber, BufferPlayhead::START ).isFailure() ) return HK_FAILURE;

    // Process the frame we set our selves to
    if( m_bufferPlayhead->processFrame( PlayheadProcessingFlags::INCLUSIVE | PlayheadProcessingFlags::EXCLUSIVE ).isFailure() ) return HK_FAILURE;

    // Since we processed this frame, indicate that
    updateFlagsAndSignal( m_state.m_targetFlags | hkVdbPlaybackFlags::FRAME_ENDED );

    // When the user explicitly sets the frame, we need to reset the playback stopwatch
    // because they've interrupted automatic playback.
    return resetPlaybackStopwatch();
}

hkResult hkVdbPlaybackHandler::saveReplay( const char* filepath, hkInt64 startFrame, hkInt64 endFrame ) const
{
    hkVdbFileConnection fileConnection( filepath, hkFileSystem::ACCESS_WRITE );
    return saveReplay( fileConnection, startFrame, endFrame );
}

hkResult hkVdbPlaybackHandler::saveReplay( hkVdbConnection& connection, hkInt64 startFrameIn, hkInt64 endFrameIn ) const
{
    HK_VDB_VERIFY_CONDITION_MSG(
        m_input && m_persistentCache && m_setupHandler,
        0xedb00041,
        hkVdbError::CMD_HANDLER_NOT_REGISTERED,
        "Cannot call saveReplay before playback handler is registered" );

    HK_VDB_VERIFY_CONDITION_MSG(
        m_setupHandler->isCapableOf( hkVdbCapabilities::SAVE_CMD_INPUT_BUFFER ),
        0xedb00125,
        hkVdbError::UNSUPPORTED_VERSION,
        "The system is not capable of saving a replay in this version" );

    // Try once to connect if we aren't already connected.
    if ( connection.getState() != hkVdbConnectionState::CONNECTED )
    {
        HK_VDB_VERIFY_REPORTER_OPERATION( connection.connect(), connection, hkVdbError::CONNECTION_ERROR );
        HK_VDB_VERIFY_CONDITION_MSG(
            connection.getState() == hkVdbConnectionState::CONNECTED,
            0xedb00042,
            hkVdbError::CONNECTION_ERROR,
            "Could not establish connection to save replay" );
    }

    hkUint32 startFrame, endFrame;
    {
        // Resolve params against our input buffer
        if ( resolveFrameArgs( startFrameIn, endFrameIn, false, startFrame, endFrame ).isFailure() ) return HK_FAILURE;

        HK_VDB_VERIFY_CONDITION_MSG(
            startFrame <= endFrame,
            0xedb00044,
            hkVdbError::INVALID_ARGUMENTS,
            "startFrame must be less than or equal to endFrame; resolved frames are [" << startFrame << ", " << endFrame << "]" );
    }

    // Determine our cached frame to persist.
    // Some systems read from the cache to save necessary data to file (Eg. Object Handler)
    // while other systems rely on maintaining current data in the persistent cache as frames
    // are evicted (Eg. Display Handler).
    hkUint32 cacheFrame;
    if ( startFrame > m_input->getStartingFrameNumber() )
    {
        // We can cache the frame before our desired start.
        cacheFrame = ( startFrame - 1 );
    }
    else
    {
        // There's no frames before the desired start, we need to use
        // the startFrame's cache and advance the persistent cache accordingly.
        // This means *technically* we are saving from startFrame + 1 -> endFrame.
        cacheFrame = startFrame;
        startFrame++;

        // The below won't practically happen, but a good check none-the-less.
        HK_VDB_VERIFY_CONDITION_MSG(
            startFrame <= endFrame,
            0xedb00160,
            hkVdbError::INVALID_STATE,
            "A degenerate hkVdbCmdInput makes saving impossible" );
    }

    
    
    
    
    // "Catch-up" our persistent state to where we need to start from.
    hkRefPtr<hkVdbPersistentStateCache> persistentCache = m_persistentCache;
    {
        // Create a temporary cache that we will advance to our starting frame
        persistentCache = m_persistentCache->clone();

        // Now advance this temp cache up to start frame by feeding more frames from the input buffer
        for ( hkVdbCmdInput::Iterator frameIt = m_input->getIterator();
            m_input->isValid( frameIt );
            frameIt = m_input->getNext( frameIt ) )
        {
            const hkVdbFrame* frame = m_input->getValue( frameIt );

            // Check that our frame number is in range
            if ( frame->getFrameNumber() >= startFrame )
            {
                break;
            }

            // Persist our input buffer frame
            HK_VDB_VERIFY_OPERATION_MSG(
                persistentCache->persistEvictedFrame( *frame ),
                0xedb00161,
                hkVdbError::SAVE_ERROR,
                "Unable to persist frame from circular buffer" );
        }

        // Persist our cache frame
        HK_VDB_VERIFY_OPERATION_MSG(
            persistentCache->persistCacheFrame( *m_cache, cacheFrame ),
            0xedb00162,
            hkVdbError::SAVE_ERROR,
            "Unable to persist frame from cache" );
    }

    // Connect if we aren't already connected
    if ( connection.getState() != hkVdbConnectionState::CONNECTED )
    {
        HK_VDB_VERIFY_REPORTER_OPERATION( connection.connect(), connection, hkVdbError::CONNECTION_ERROR );
    }

    // Verify we are ready to write
    hkVdbSignalResult result;
    hkStreamWriter* writer = connection.getWriter();
    HK_VDB_VERIFY_REPORTER_CONDITION( writer && writer->isOk(), connection, hkVdbError::CONNECTION_WRITE_ERROR );

    // Ensure that our writer always writes all the bits we need
    hkVdbBlockingWriter blockingWriter( *writer );
    writer = &blockingWriter;

    // Persistent State section
    hkVdbPersistentStateCache::Iterator persistentCmdIt = persistentCache->getIterator();
    if ( persistentCache->isValid( persistentCmdIt ) )
    {
        hkVdbInternalHandler::writeDebugTagCmd( *writer, m_input->getProtocol(), "Persistent State" );
        do
        {
            const hkVdbCmd* cmd = persistentCache->getValue( persistentCmdIt );
            
            
            
            
            hkVdbInternalHandler::inplaceModifyStreamDebugTag( *const_cast<hkVdbCmd*>( cmd ), m_input->getProtocol() );
            HK_ON_DEBUG( int written = ) writer->write( cmd, cmd->getCmdSize() );
            HK_ASSERT_NO_MSG( 0x22440916, hkUint32( written ) == cmd->getCmdSize() );
            persistentCmdIt = persistentCache->getNext( persistentCmdIt );
        } while ( persistentCache->isValid( persistentCmdIt ) );
    }
    HK_VDB_VERIFY_REPORTER_CONDITION( writer->isOk(), connection, hkVdbError::CONNECTION_WRITE_ERROR );

    // Cmd Buffer section
    hkVdbCmdInput::Iterator cmdInputFrameIt = m_input->getIterator( startFrame );
    if ( m_input->isValid( cmdInputFrameIt, true ) )
    {
        hkVdbInternalHandler::writeDebugTagCmd( *writer, m_input->getProtocol(), "Cmd Buffer" );
        do
        {
            const hkVdbFrame* frame = m_input->getValue( cmdInputFrameIt );

            // Check that our frame number is in range
            if ( frame->getFrameNumber() > endFrame )
            {
                break;
            }

            // Write our the frame cmds
            for ( hkVdbFrame::Iterator cmdIt = frame->getIterator();
                frame->isValid( cmdIt );
                cmdIt = frame->getNext( cmdIt ) )
            {
                const hkVdbCmd* cmd = frame->getValue( cmdIt );

                
                
                
                
                hkVdbInternalHandler::inplaceModifyStreamDebugTag( *const_cast<hkVdbCmd*>( cmd ), m_input->getProtocol() );
                HK_ON_DEBUG( int written = ) writer->write( cmd, cmd->getCmdSize() );
                HK_ASSERT_NO_MSG( 0x22440916, hkUint32( written ) == cmd->getCmdSize() );
            }

            cmdInputFrameIt = m_input->getNext( cmdInputFrameIt );
        } while ( m_input->isValid( cmdInputFrameIt, true ) );
    }
    HK_VDB_VERIFY_REPORTER_CONDITION( writer->isOk(), connection, hkVdbError::CONNECTION_WRITE_ERROR );

    writer->flush();

    return result.isSuccess() ? HK_SUCCESS : HK_FAILURE; 
}

hkResult hkVdbPlaybackHandler::startRecording( const char* filepath )
{
    HK_VDB_VERIFY_CONDITION_MSG(
        m_client,
        0xedb00045,
        hkVdbError::CMD_HANDLER_NOT_REGISTERED,
        "Cannot call startRecording before playback handler is registered" );

    // This is a bit wonky in that this will call hkVdbClient::connectFile
    // which will do some stuff, and then call back into saveReplay( lastFrame, lastFrame ).
    // However, it's the best way to ensure no matter how the client starts recording
    // (either by looking at the handlers and seeing this function where they'd expect it
    // OR by just setting a file connection on the hkVdbClient), everything works.
    return m_client->connectFile( filepath, hkVdbConnectionUse::SINK );
}

hkResult hkVdbPlaybackHandler::startRecording( hkVdbConnection& connection )
{
    HK_VDB_VERIFY_CONDITION_MSG(
        m_client,
        0xedb00118,
        hkVdbError::CMD_HANDLER_NOT_REGISTERED,
        "Cannot call startRecording before playback handler is registered" );

    // (see notes in startRecording( filepath ))
    return m_client->setConnection( &connection, hkVdbConnectionUse::SINK );
}

hkResult hkVdbPlaybackHandler::stopRecording()
{
    HK_VDB_VERIFY_CONDITION_MSG(
        m_client,
        0xedb00046,
        hkVdbError::CMD_HANDLER_NOT_REGISTERED,
        "Cannot call stopRecording before playback handler is registered" );

    return m_client->disconnect( hkVdbConnectionUse::SINK );
}

hkResult hkVdbPlaybackHandler::syncToConnection()
{
    HK_VDB_VERIFY_CONDITION_MSG(
        m_client,
        0xedb00119,
        hkVdbError::CMD_HANDLER_NOT_REGISTERED,
        "Cannot call syncToConnection() before playback handler is registered" );

    HK_VDB_VERIFY_CONDITION_MSG(
        m_client->getConnection(),
        0xedb00120,
        hkVdbError::CMD_HANDLER_NOT_REGISTERED,
        "Cannot call syncToConnection() if the client does not have an APP connection" );

    return syncToConnection( *m_client->getConnection() );
}

hkResult hkVdbPlaybackHandler::syncToConnection( hkVdbConnection& connection )
{
    // Read everything we can from the connection
    {
        // See if we can estimate the time to read the full stream
        
        //hkReal estTime = 0;
        //if ( hkStreamReader* reader = connection.getReader() )
        //{
        //  if ( hkSeekableStreamReader* seekableReader = reader->isSeekTellSupported() )
        //  {
        //      int here = seekableReader->tell();
        //      if ( seekableReader->seek( 0, hkSeekableStreamReader::STREAM_END ).isSuccess() )
        //      {
        //          const int bytes = ( seekableReader->tell() - here );
        //          estTime = bytes * 2.7e-8f;
        //      }
        //      seekableReader->seek( 0, hkSeekableStreamReader::STREAM_SET );
        //  }
        //}

        // Do our read with progress reporting
        ConnectionPlayhead& playhead = *m_connectionPlayhead;
        hkResult result = m_progressReporter->workTillCompleted(
            "Loading frames",
            [&playhead, &connection]( hkVdbSignalResult& warningResult, hkVdbSignalResult& errorResult, hkStringBuf& workMsgOut )
            {
                const hkVdbFrame* _frameOut;
                const hkVdbCmd* _cmdOut;

                do
                {
                    if ( playhead.play( connection, _frameOut, _cmdOut ).isFailure() )
                    {
                        errorResult.signalError( playhead );
                        _cmdOut = HK_NULL;
                    }

                // Load a frame at a time to avoid slowing down the system with many work update calls
                } while ( _cmdOut && ( _cmdOut->getType() != hkVdbCmdType::STEP ) );

                // No more cmds, no more work to do...connection is depleted.
                return ( _cmdOut != HK_NULL );
            } );

        HK_VDB_VERIFY_REPORTER_OPERATION( result, *m_connectionPlayhead, hkVdbError::CMD_HANDLER_ERROR );
    }

    // Catch up the buffer playhead (this will force a dispatch to the system)
    hkInt64 connectionFrame = m_connectionPlayhead->getFrameNumber();
    if ( connectionFrame >= 0 )
    {
        if ( m_bufferPlayhead->set( hkUint32( connectionFrame ), BufferPlayhead::END ).isFailure() ) return HK_FAILURE;
    }

    // Reset stopwatch, we have interrupted normal playback
    return resetPlaybackStopwatch();
}

hkVdbPlayhead hkVdbPlaybackHandler::getPlayhead( hkVdbPlayheadType::Enum type ) const
{
    switch ( type )
    {
        case hkVdbPlayheadType::BUFFER: return *m_bufferPlayhead;
        case hkVdbPlayheadType::CONNECTION: return *m_connectionPlayhead;
        case hkVdbPlayheadType::PROCESSED: return *m_processedPlayhead;
        default: HK_ASSERT( 0x22441097, false, "Unknown playhead type" ); return hkVdbPlayhead( *m_input );
    }
}

hkResult hkVdbPlaybackHandler::processPlaybackInfoCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader )
{
    

    HK_ASSERT_NOT_IMPLEMENTED( 0 );

    //return updateStateChangeAndSignal();
    //return updateFrameChangeAndSignal();
    return HK_SUCCESS;
}

void hkVdbPlaybackHandler::onDisconnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection )
{
    if ( use == hkVdbConnectionUse::APPLICATION )
    {
        if ( m_playbackStopwatch.isRunning() )
        {
            m_playbackStopwatch.stop();
        }
    }
    else if ( use == hkVdbConnectionUse::SINK )
    {
        updateFlagsAndSignal( m_state.m_targetFlags & ~hkVdbPlaybackFlags::RECORDING );
    }
}

void hkVdbPlaybackHandler::onStepCompletedSignal( const hkVdbFrame& frame, const hkVdbCmd& cmd )
{
    const hkInt64 nextFrame = getNextFrame( frame.getFrameNumber() );
    hkUint32 minFrame, maxFrame;
    getFrameRange( minFrame, maxFrame );
    updateFrameAndSignal( hkUint32( hkMath::clamp( nextFrame, minFrame, maxFrame ) ) );
}

void hkVdbPlaybackHandler::onFrameEvictedSignal( hkVdbCmdInput& input, const hkVdbFrame& frame )
{
    // If process ticking is going to cause the buffer playhead to become invalid (due to evicting the frame)
    // advance it into safe territory.  This is so that anybody listening to the frame updates and relying
    // on replay flags can access the soon-to-be-evicted frame's data in order to determine how to move to
    // the earliest valid frame.
    if ( frame.getFrameNumber() == m_state.m_targetFrame )
    {
        // Don't allow listeners to not update thinking this is FAST_REPLAY since data is going away.
        // Note: FAST_REPLAY is not very likely during normal playback (in which eviction occurs).
        hkVdbPlaybackFlags flags = m_state.m_targetFlags;
        updateFlagsAndSignal( flags & ~hkVdbPlaybackFlags::FAST_REPLAY );
        {
            setCurrentFrameInternal( m_state.m_targetFrame + 1 );
        }
        updateFlagsAndSignal( flags );
    }
}

void hkVdbPlaybackHandler::onCmdDispatchingSignal( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbSignalResult& result )
{
    HK_VDB_VERIFY_SIGNALLED_REPORTER_OPERATION(
        updateReplayFlagAndSignal(),
        *this,
        result );
}

void hkVdbPlaybackHandler::onCmdDispatchedSignal( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbSignalResult& result )
{
    m_processedPlayhead->update( &frame, &cmd );
    HK_VDB_VERIFY_SIGNALLED_REPORTER_OPERATION(
        updateProcessedAvailableFlagAndSignal(),
        *this,
        result );
}

hkResult hkVdbPlaybackHandler::processSaveReplayInfoCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader )
{
    

    HK_ASSERT_NOT_IMPLEMENTED( 0 );

    
    
    

    return HK_SUCCESS;
}

hkResult hkVdbPlaybackHandler::getNextCmd(
    hkVdbConnection* connection,
    const hkVdbFrame*& frameOut,
    const hkVdbCmd*& cmdOut )
{
    HK_VDB_VERIFY_CONDITION(
        m_input,
        0xedb00047,
        hkVdbError::CMD_HANDLER_NOT_REGISTERED );

    // We always consume more data from our connection. This supports throttling playback.
    // And in the synchronized stepping case (Eg. network connection), the connection will wait for STEP_ACKs before advancing.
    if ( connection )
    {
        const hkVdbFrame* _frameOut;
        const hkVdbCmd* _cmdOut;
        HK_VDB_VERIFY_REPORTER_OPERATION( m_connectionPlayhead->play( *connection, _frameOut, _cmdOut ), *m_connectionPlayhead, hkVdbError::CMD_HANDLER_ERROR );
    }

    // Grab our next frame from the buffered playhead if we are playing
    if ( m_state.m_targetState == hkVdbPlaybackState::PLAYING )
    {
        HK_VDB_VERIFY_REPORTER_OPERATION( m_bufferPlayhead->play( frameOut, cmdOut ), *m_bufferPlayhead, hkVdbError::CMD_HANDLER_ERROR );
    }
    else
    {
        frameOut = HK_NULL;
        cmdOut = HK_NULL;
    }

    checkPlayheads( frameOut, cmdOut );

    return HK_SUCCESS;
}

// Change if-branch below to handle possible divide by zero


//HK_COMPILE_TIME_ASSERT( HK_VDB_SERVER_FRAME_RATE == 0 );

hkBool32 hkVdbPlaybackHandler::shouldThrottlingConnection( const hkVdbFrame& frame )
{
    HK_ON_DEBUG( hkBool32 connected = ( m_client->getConnection() && ( m_client->getConnection()->getState() == hkVdbConnectionState::CONNECTED ) ); )
    HK_ASSERT(
        0x22440958,
        !connected ||
        ( m_playbackStopwatch.isRunning() == ( m_state.m_targetState == hkVdbPlaybackState::PLAYING ) ),
        "Desync in playback stopwatch state" );

#ifdef REPORT_THROTTLING_CHANGES
    bool prevThrottle = m_state.m_targetFlags.anyIsSet( hkVdbPlaybackFlags::THROTTLING );
#endif

    bool throttle;
    if (// It's enabled by our flags
        m_state.m_targetFlags.anyIsSet( hkVdbPlaybackFlags::THROTTLE_FRAMERATE ) &&
        // And we are currently running our playback stopwatch
        m_playbackStopwatch.isRunning() )
    {
        // Just for debugging
        HK_ON_DEBUG( const hkReal elapsedMs = ( m_playbackStopwatch.isRunning() ) ? ( m_playbackStopwatch.getElapsedSeconds() * 1000 ) : 0; )
        HK_ON_DEBUG( elapsedMs; )

        float throttleTime;
        if ( m_state.m_targetFrameRate == HK_VDB_SERVER_FRAME_RATE )
        {
            const bool fwd = ( m_state.m_targetDirection == hkVdbPlaybackDirection::FORWARD );
            const bool bwd = ( m_state.m_targetDirection == hkVdbPlaybackDirection::BACKWARD );

            // Note: we compare against end/start of frame depending on direction
            throttleTime = hkMath::abs( fwd * frame.getEndTime() + bwd * frame.getStartTime() - m_playbackStartTime );
        }
        else
        {
            // Note: m_targetFrameRate can't be zero because HK_VDB_SERVER_FRAME_RATE == 0
            // fps < n / elapsed
            // elapsed * fps < n
            // elapsed < n * ( 1 / fps )
            throttleTime = ( hkMath::abs( hkInt64( frame.getFrameNumber() ) - m_playbackStartFrame ) * ( 1 / m_state.m_targetFrameRate ) * 1000 );
        }

        throttle = ( ( m_playbackStopwatch.getElapsedSeconds() * 1000 ) < throttleTime );
    }
    else
    {
        throttle = false;
    }

#ifdef REPORT_THROTTLING_CHANGES
    if ( prevThrottle != throttle )
    {
        hkStringBuf buf;
        buf.printf( "ThrottlingChange:[%i -> %i]", prevThrottle, throttle );
        HK_REPORT( buf );
        HK_REPORT( "" );
    }
#endif

    return throttle;
}

hkResult hkVdbPlaybackHandler::resetPlaybackStopwatch()
{
    if ( m_state.m_targetState == hkVdbPlaybackState::PAUSED )
    {
        if ( m_playbackStopwatch.isRunning() )
        {
            m_playbackStopwatch.stop();
        }
    }
    else
    {
        const hkVdbFrame* frame;
        {
            if ( m_bufferPlayhead->isValid() )
            {
                frame = m_bufferPlayhead->getFrame();
            }
            else
            {
                frame = m_connectionPlayhead->getFrame();
            }
        }

        if ( frame )
        {
            m_playbackStartTime =
                ( m_state.m_targetDirection == hkVdbPlaybackDirection::FORWARD ) ?
                frame->getStartTime() :
                frame->getEndTime();
            m_playbackStartFrame = frame->getFrameNumber();
        }
        else
        {
            m_playbackStartTime = 0;
            m_playbackStartFrame = 0;
        }

        // Reset our m_inv_ticks_per_second and start
        new ( &m_playbackStopwatch ) hkStopwatch();
        m_playbackStopwatch.start();
    }

    return HK_SUCCESS;
}

hkResult hkVdbPlaybackHandler::updateStateAndSignal( hkVdbPlaybackState::Enum state )
{
    if ( state == m_state.m_targetState )
    {
        return HK_SUCCESS;
    }

    hkVdbSignalResult result;
    {
        // Update
        m_state.m_previousState = m_state.m_targetState;
        m_state.m_targetState = state;

        // Create info for signal with no change to other properties
        PlaybackInfo info = m_state;
        info.m_previousDirection = info.m_targetDirection;
        info.m_previousFlags = info.m_targetFlags;
        info.m_previousFrame = info.m_targetFrame;
        info.m_previousFrameRate = info.m_targetFrameRate;
        m_playbackInfoReceivedInternal.fire( info, result );
        resetPlaybackStopwatch();
    }
    HK_VDB_VERIFY_SIGNAL_RESULT( result );

    
#ifdef REPORT_STATE_CHANGES
    hkStringBuf buf;
    buf.printf( "StateChange:[%i -> %i]", m_state.m_previousState, m_state.m_targetState );
    HK_REPORT( buf );
    HK_REPORT( "" );
#endif

    return HK_SUCCESS;
}

hkResult hkVdbPlaybackHandler::updateDirectionAndSignal( hkVdbPlaybackDirection::Enum dir )
{
    if ( dir == m_state.m_targetDirection )
    {
        return HK_SUCCESS;
    }

    hkVdbSignalResult result;
    {
        // Update
        m_state.m_previousDirection = m_state.m_targetDirection;
        m_state.m_targetDirection = dir;

        // Reset our playback stopwatch
        resetPlaybackStopwatch();

        // Create info for signal with no change to other properties
        PlaybackInfo info = m_state;
        info.m_previousState = info.m_targetState;
        info.m_previousFlags = info.m_targetFlags;
        info.m_previousFrame = info.m_targetFrame;
        info.m_previousFrameRate = info.m_targetFrameRate;
        m_playbackInfoReceivedInternal.fire( info, result );
    }
    HK_VDB_VERIFY_SIGNAL_RESULT( result );

    
#ifdef REPORT_DIRECTION_CHANGES
    hkStringBuf buf;
    buf.printf( "DirectionChange:[%i -> %i]", m_state.m_previousDirection, m_state.m_targetDirection );
    HK_REPORT( buf );
    HK_REPORT( "" );
#endif

    return HK_SUCCESS;
}

hkResult hkVdbPlaybackHandler::updateFlagsAndSignal( hkVdbPlaybackFlags flags )
{
    if ( flags == m_state.m_targetFlags )
    {
        return HK_SUCCESS;
    }

    hkVdbSignalResult result;
    {
        // Update
        m_state.m_previousFlags = m_state.m_targetFlags;
        m_state.m_targetFlags = flags;

        // Reset playback if we changed throttle flag
        if ( m_state.m_previousFlags.anyIsSet( hkVdbPlaybackFlags::THROTTLE_FRAMERATE ) !=
            m_state.m_targetFlags.anyIsSet( hkVdbPlaybackFlags::THROTTLE_FRAMERATE ) )
        {
            resetPlaybackStopwatch();
        }

        // Create info for signal with no change to other properties
        PlaybackInfo info = m_state;
        info.m_previousState = info.m_targetState;
        info.m_previousDirection = info.m_targetDirection;
        info.m_previousFrame = info.m_targetFrame;
        info.m_previousFrameRate = info.m_targetFrameRate;
        m_playbackInfoReceivedInternal.fire( info, result );
    }

    // Users can't reject implementation flags
    
    //flags.xorWith( m_state.m_previousFlags );
    //HK_ASSERT( 0x22440987, flags.anyIsSet( hkVdbPlaybackFlags::USER_FLAGS_MASK ) ^)
    //if ( flags.noneIsSet( hkVdbPlaybackFlags::IMPLEMENTATION_FLAGS_MASK ) )
    {
        HK_VDB_VERIFY_SIGNAL_RESULT( result );
    }

    
#ifdef REPORT_FLAG_CHANGES
    hkStringBuf buf;
    buf.printf( "FlagsChange:[%x -> %x]", m_state.m_previousFlags, m_state.m_targetFlags );
    HK_REPORT( buf );
    HK_REPORT( "" );
#endif

    return HK_SUCCESS;
}

hkResult hkVdbPlaybackHandler::updateFrameRateAndSignal( hkReal fps )
{
    if ( fps == m_state.m_targetFrameRate )
    {
        return HK_SUCCESS;
    }

    hkVdbSignalResult result;
    {
        // Update
        m_state.m_previousFrameRate = m_state.m_targetFrameRate;
        m_state.m_targetFrameRate = fps;

        // Reset playback since our fps changed
        resetPlaybackStopwatch();

        // Create info for signal with no change to other properties
        PlaybackInfo info = m_state;
        info.m_previousState = info.m_targetState;
        info.m_previousDirection = info.m_targetDirection;
        info.m_previousFlags = info.m_targetFlags;
        info.m_previousFrame = info.m_targetFrame;
        m_playbackInfoReceivedInternal.fire( info, result );
    }
    HK_VDB_VERIFY_SIGNAL_RESULT( result );

    
#ifdef REPORT_FRAME_RATE_CHANGES
    hkStringBuf buf;
    buf.printf( "FrameRateChange:[%f -> %f]", m_state.m_previousFrameRate, m_state.m_targetFrameRate );
    HK_REPORT( buf );
    HK_REPORT( "" );
#endif

    return HK_SUCCESS;
}

hkResult hkVdbPlaybackHandler::updateFrameAndSignal( hkUint32 frameNumber )
{
    if ( frameNumber == m_state.m_targetFrame )
    {
        return HK_SUCCESS;
    }

    // Update and signal
    {
        hkVdbSignalResult result;
        // End previous frame
        HK_VDB_VERIFY_SIGNALLED_REPORTER_OPERATION(
            updateFlagsAndSignal( m_state.m_targetFlags | hkVdbPlaybackFlags::FRAME_ENDED ),
            *this,
            result );

        // Update frame
        m_state.m_previousFrame = m_state.m_targetFrame;
        m_state.m_targetFrame = frameNumber;

        // Clear frame ended since we've switched to the new frame
        // We don't signal in this case because the signal from frame X to Y is the "begin"
        m_state.m_targetFlags.clear( hkVdbPlaybackFlags::FRAME_ENDED );

        // Send replay signal so subscribers can respond to frame change accordingly
        HK_VDB_VERIFY_SIGNALLED_REPORTER_OPERATION(
            updateReplayFlagAndSignal(),
            *this,
            result );

        // Create info for signal with no change to other properties
        PlaybackInfo info = m_state;
        info.m_previousState = info.m_targetState;
        info.m_previousDirection = info.m_targetDirection;
        info.m_previousFlags = info.m_targetFlags;
        info.m_previousFrameRate = info.m_targetFrameRate;
        m_playbackInfoReceivedInternal.fire( info, result );
        HK_VDB_VERIFY_SIGNAL_RESULT( result );
    }

    
#ifdef REPORT_FRAME_CHANGES
    hkStringBuf buf;
    buf.printf( "FrameChange:[%i -> %i]", m_state.m_previousFrame, m_state.m_targetFrame );
    HK_REPORT( buf );
    HK_REPORT( "" );
#endif

    return HK_SUCCESS;
}

hkResult hkVdbPlaybackHandler::updateReplayFlagAndSignal()
{
    bool isReplaying =
        ( m_bufferPlayhead->isValid() &&
        ( *m_bufferPlayhead <= *m_processedPlayhead ) &&
            // Note: during caching, we start with m_processedPlayhead < m_bufferPlayhead and exit when
            // m_bufferPlayhead == m_processedPlayhead. We are moving m_processedPlayhead -> m_bufferPlayhead.
            // We may need to ack the final cmd step in this case so don't signify we are replaying.
            m_state.m_targetFlags.noneIsSet( hkVdbPlaybackFlags::CACHING ) );

    hkVdbPlaybackFlags newFlags = m_state.m_targetFlags;
    newFlags.orWith( isReplaying * hkVdbPlaybackFlags::REPLAYING );
    newFlags.andNotWith( !isReplaying * hkVdbPlaybackFlags::REPLAYING );

    // Note: users cannot reject implementation flags with signal results
    return updateFlagsAndSignal( newFlags );
}

hkResult hkVdbPlaybackHandler::updateProcessedAvailableFlagAndSignal()
{
    if ( m_processedPlayhead->isValid() )
    {
        hkVdbPlaybackFlags newFlags = m_state.m_targetFlags;

        bool availableCmdsProcessed = ( *m_processedPlayhead == *m_connectionPlayhead );
        newFlags.orWith( availableCmdsProcessed * hkVdbPlaybackFlags::AVAILABLE_CMDS_PROCESSED );
        newFlags.andNotWith( !availableCmdsProcessed * hkVdbPlaybackFlags::AVAILABLE_CMDS_PROCESSED );

        // Note: users cannot reject implementation flags with signal results
        return updateFlagsAndSignal( newFlags );
    }
    return HK_SUCCESS;
}

void hkVdbPlaybackHandler::checkPlayheads( const hkVdbFrame* frame, const hkVdbCmd* cmd )
{
#ifdef HK_DEBUG
    
    
    
    
    
    HK_ASSERT( 0x22440967, *m_processedPlayhead <= *m_connectionPlayhead, "Processed playhead is too far" );
    Playhead inputPlayhead( *this );
    inputPlayhead.m_frameIt = m_input->getIterator( m_input->m_processedFrames );
    inputPlayhead.m_cmdIt = m_input->isValid( inputPlayhead.m_frameIt, true ) ? m_input->getValue( inputPlayhead.m_frameIt )->getIterator( m_input->getValue( inputPlayhead.m_frameIt )->getNumCmds() - 1 ) : hkVdbFrame::InvalidIterator;
    HK_ASSERT( 0x22441121,
        // Either we haven't processed anything here yet (which can happen during initialization)
        !m_processedPlayhead->isValid() ||
        // Or we've processed *no more than* the input has processed
        ( *m_processedPlayhead <= inputPlayhead ) ||
        // Or (minor exception) we are at the edge case of the input being at the end of a frame and processed is at the very start of the next
        // this occurs due to when signalling occurs and is okay.
        ( inputPlayhead.isValid() && ( inputPlayhead.getFrameNumber() + 1 ) == m_processedPlayhead->getFrameNumber() && ( inputPlayhead.getCmd()->getType() == hkVdbCmdType::STEP ) ),
        // Hitting this could mean a dangerous inconsistency between systems has occurred.
        "Failed to dispatch all cmds to framework" );
#endif
#ifdef REPORT_PLAYHEADS
    if ( cmd )
    {
        hkStringBuf bufferSel;
        bufferSel.printf( "%s", cmd == m_bufferPlayhead->getCmd() ? "* " : "" );
        hkStringBuf connectionSel;
        connectionSel.printf( "%s", cmd != m_bufferPlayhead->getCmd() ? "* " : "" );

        hkStringBuf buf;
        Playhead nextPlayhead( *this );
        nextPlayhead.m_frameIt = m_bufferPlayhead->m_nextFrameIt;
        nextPlayhead.m_cmdIt = m_bufferPlayhead->m_nextCmdIt;
        buf.printf( "%sBuffer:[%i,%i -> %i,%i]", bufferSel.cString(), m_bufferPlayhead->getFrameNumber(), m_bufferPlayhead->getCmdIndex(), nextPlayhead.getFrameNumber(), nextPlayhead.getCmdIndex() );
        HK_REPORT( buf );
        buf.printf( "Processed:[%i,%i]", m_processedPlayhead->getFrameNumber(), m_processedPlayhead->getCmdIndex() );
        HK_REPORT( buf );
        buf.printf( "%sConnection:[%i,%i]", connectionSel.cString(), m_connectionPlayhead->getFrameNumber(), m_connectionPlayhead->getCmdIndex() );
        HK_REPORT( buf );
        HK_REPORT( "" );
    }
#endif
}

hkResult hkVdbPlaybackHandler::resolveFrameArgs(
    hkInt64 startFrameIn,
    hkInt64 endFrameIn,
    hkBool32 includeMargins,
    hkUint32& startFrameOut,
    hkUint32& endFrameOut ) const
{
    hkUint32 minFrame, maxFrame;
    bool succeeded;
    if ( includeMargins )
    {
        succeeded = getFrameRange( minFrame, maxFrame ).isSuccess();
    }
    else
    {
        
        succeeded = m_input->getFrameRange( minFrame, maxFrame ).isSuccess();
    }

    if ( startFrameIn < 0 )
    {
        startFrameOut = minFrame;
    }
    else if ( startFrameIn < minFrame )
    {
        succeeded = false;
    }
    else
    {
        startFrameOut = hkUint32( startFrameIn );
    }

    if ( endFrameIn < 0 )
    {
        endFrameOut = maxFrame;
    }
    else if ( endFrameIn > maxFrame )
    {
        succeeded = false;
    }
    else
    {
        endFrameOut = hkUint32( endFrameIn );
    }

    HK_VDB_VERIFY_CONDITION_MSG(
        succeeded,
        0xedb00052,
        hkVdbError::INVALID_ARGUMENTS,
        "The following frame values [" << startFrameIn << ", " << endFrameIn << "] could not be resolved into the range [" << minFrame << ", " << maxFrame << "]" );

    // NOTE: going backwards is okay for some callers
    return succeeded ? HK_SUCCESS : HK_FAILURE;
}

//////////////////////////////////////////////////////////////////////////
// hkVdbPlayhead
//////////////////////////////////////////////////////////////////////////

hkVdbPlayhead::hkVdbPlayhead( hkVdbCmdInput& input ) :
    m_frameIt( hkVdbCmdInput::InvalidIterator ),
    m_cmdIt( hkVdbFrame::InvalidIterator ),
    m_input( &input )
{}

hkInt64 hkVdbPlayhead::compare( const hkVdbPlayhead& other ) const
{
    // Note: if frame number is -1, then it will result in comparison that places non-neg
    // evaluations *earlier* than valid ones, which is what we want.
    hkInt64 frameCmp = ( getFrameNumber() - other.getFrameNumber() );
    return frameCmp + ( bool( frameCmp == 0 ) * ( m_cmdIt - other.m_cmdIt ) );
}

const hkVdbFrame* hkVdbPlayhead::getFrame() const
{
    if ( m_input->isValid( m_frameIt, true ) )
    {
        return m_input->getValue( m_frameIt );
    }

    return HK_NULL;
}

const hkVdbCmd* hkVdbPlayhead::getCmd() const
{
    if ( m_input->isValid( m_frameIt, true ) )
    {
        const hkVdbFrame* frame = m_input->getValue( m_frameIt );
        if ( frame->isValid( m_cmdIt ) )
        {
            return frame->getValue( m_cmdIt );
        }
    }

    return HK_NULL;
}

hkInt64 hkVdbPlayhead::getFrameNumber() const
{
    if ( const hkVdbFrame* frame = getFrame() )
    {
        return frame->getFrameNumber();
    }
    return -1;
}

hkInt64 hkVdbPlayhead::getCmdIndex() const
{
    if ( const hkVdbCmd* cmd = getCmd() )
    {
        return cmd->getIndex();
    }
    return -1;
}

//////////////////////////////////////////////////////////////////////////
// Playhead
//////////////////////////////////////////////////////////////////////////

Playhead::Playhead( hkVdbPlaybackHandler& handler ) :
    hkVdbPlayhead( *handler.m_input ),
    hkVdbDefaultErrorReporter( &s_debugLog ),
    m_handler( handler )
{}

Playhead::Playhead( const Playhead& other ) :
    hkVdbPlayhead( *other.m_handler.m_input ),
    hkVdbDefaultErrorReporter( &s_debugLog ),
    m_handler( other.m_handler )
{
    this->operator =( other );
}

hkResult Playhead::processFrame( PlayheadProcessingFlags flags )
{
    hkVdbCmdInput* input = m_handler.m_input;
    hkVdbCmdDispatcher* dispatcher = m_handler.m_dispatcher;
    

    if ( input->isValid( m_frameIt, true ) )
    {
        const hkVdbFrame* frame = input->getValue( m_frameIt );
        hkVdbFrame::Iterator lastCmdIt = m_cmdIt;
        while ( true )
        {
            // Check to see if we've run-off-the-end (no step cmd available)
            if ( !frame->isValid( m_cmdIt ) )
            {
                m_cmdIt = lastCmdIt;
                break;
            }

            // Get our cmd and potentially dispatch it
            const hkVdbCmd* cmd = frame->getValue( m_cmdIt );
            if (// 1) If we are a step cmd, make sure we are allowed to dispatch
                ( ( cmd->getType() != hkVdbCmdType::STEP ) || flags.anyIsSet( PlayheadProcessingFlags::STEP ) ) &&
                    // a) We are on the current cmd and we are allowed to dispatch
                    ( ( lastCmdIt == m_cmdIt ) && flags.anyIsSet( PlayheadProcessingFlags::INCLUSIVE ) ||
                    // b) We are on a later cmd and we are allowed to dispatch
                    ( lastCmdIt != m_cmdIt ) && flags.anyIsSet( PlayheadProcessingFlags::EXCLUSIVE ) ) )
            {
                HK_VDB_VERIFY_REPORTER_OPERATION(
                    dispatcher->dispatch( *frame, *cmd, input->getProtocol() ),
                    *dispatcher,
                    hkVdbError::CMD_DISPATCH_ERROR );
            }

            // Check for step cmd
            if ( frame->getValue( m_cmdIt )->getType() == hkVdbCmdType::STEP )
            {
                break;
            }

            // Advance
            lastCmdIt = m_cmdIt;
            m_cmdIt = frame->getNext( m_cmdIt );
        }
    }
    return HK_SUCCESS;
}

hkResult Playhead::advance()
{
    hkVdbCmdInput* input = m_handler.m_input;

    if ( input->isValid( m_frameIt, true ) )
    {
        // Note: a reverse wrap around will give us an invalidly high number.
        const hkUint32 nextFrame = hkUint32( m_handler.getNextFrame( getFrameNumber() ) );
        m_frameIt = input->getIterator( nextFrame );
        m_cmdIt = ( input->isValid( m_frameIt, true ) ) ? input->getValue( m_frameIt )->getIterator() : hkVdbFrame::InvalidIterator;
    }

    return HK_SUCCESS;
}

hkBool32 Playhead::isValid() const
{
    hkVdbCmdInput* input = m_handler.m_input;

    if ( input->isValid( m_frameIt, true ) )
    {
        const hkVdbFrame* frame = input->getValue( m_frameIt );
        return frame->isValid( m_cmdIt );
    }

    return false;
}

Playhead Playhead::peek() const
{
    if ( isValid() )
    {
        const hkVdbFrame* frame = m_handler.m_input->getValue( m_frameIt );

        Playhead nextPlayhead( m_handler );
        nextPlayhead.m_cmdIt = frame->getNext( m_cmdIt );
        if ( frame->isValid( nextPlayhead.m_cmdIt ) )
        {
            // We are still in the same frame.
            nextPlayhead.m_frameIt = m_frameIt;
            return nextPlayhead;
        }
        else
        {
            // Try to advance a frame.
            nextPlayhead.m_frameIt = m_handler.m_input->getNext( m_frameIt );
            if ( m_handler.m_input->isValid( nextPlayhead.m_frameIt, true ) )
            {
                frame = m_handler.m_input->getValue( nextPlayhead.m_frameIt );
                nextPlayhead.m_cmdIt = frame->getIterator();
                if ( frame->isValid( nextPlayhead.m_cmdIt ) )
                {
                    return nextPlayhead;
                }
            }
        }
    }

    // Return invalid playhead
    return Playhead( m_handler );
}

PlayheadDiff Playhead::operator -( const Playhead& other ) const
{
    PlayheadDiff diff;
    diff.m_numFrames = ( getFrameNumber() - other.getFrameNumber() );
    diff.m_numCmds = ( diff.m_numFrames == 0 ) ? ( getCmdIndex() - other.getCmdIndex() ) : 0;
    return diff;
}

hkResult Playhead::set( const hkVdbFrame* frame, const hkVdbCmd* cmd )
{
    hkVdbCmdInput* input = m_handler.m_input;

    if ( frame && cmd )
    {
        m_frameIt = input->getIterator( frame->getFrameNumber() );
        if ( input->isValid( m_frameIt, true ) )
        {
            HK_ASSERT_NO_MSG( 0x22440969, input->getValue( m_frameIt ) == frame );
            m_cmdIt = frame->getIterator( cmd->getIndex() );
            if ( frame->isValid( m_cmdIt ) )
            {
                // Keep our iterators
                return HK_SUCCESS;
            }
        }
    }

    // Not valid, clear iterators
    m_frameIt = hkVdbCmdInput::InvalidIterator;
    m_cmdIt = hkVdbFrame::InvalidIterator;
    return HK_SUCCESS;
}

//////////////////////////////////////////////////////////////////////////
// MaxPlayhead
//////////////////////////////////////////////////////////////////////////

MaxPlayhead::MaxPlayhead( hkVdbPlaybackHandler& handler ) :
    Playhead( handler )
{}

MaxPlayhead::MaxPlayhead( const Playhead& other ) :
    Playhead( other )
{}

hkBool32 MaxPlayhead::update( const hkVdbFrame* frame, const hkVdbCmd* cmd )
{
    hkVdbCmdInput* input = m_handler.m_input;

    // Update frame if it's later.
    // Note: <= is important because we need to update m_cmd in the == case
    const hkInt64 currentFrameNumber = getFrameNumber();
    if ( frame && cmd )
    {
        const hkUint32 updatedFrameNumber = frame->getFrameNumber();
        if ( currentFrameNumber <= updatedFrameNumber )
        {
            m_frameIt = input->getIterator( updatedFrameNumber );
            HK_ASSERT(
                0x22440973,
                ( updatedFrameNumber == getFrameNumber() ) && ( updatedFrameNumber <= ( currentFrameNumber + 1 ) ),
                "Skipped a frame" );

            // Clear the current cmdIt if we've changed frames
            if ( currentFrameNumber != updatedFrameNumber )
            {
                m_cmdIt = hkVdbFrame::InvalidIterator;
            }

            // Update cmd if it's later
            const hkInt64 currentCmdIndex = getCmdIndex();
            const hkUint32 updatedCmdIndex = cmd->getIndex();
            if ( currentCmdIndex < updatedCmdIndex )
            {
                m_cmdIt = frame->getIterator( updatedCmdIndex );
                HK_ASSERT(
                    0x22440974,
                    // Note: This can happen if a cmd had no handler; but is generally a really good
                    // check. You can disable this assert if you know it's okay *or* register a no-op handler.
                    ( updatedCmdIndex == getCmdIndex() ) && ( updatedCmdIndex == ( currentCmdIndex + 1 ) ),
                    "Skipped a cmd" );
                return true;
            }
        }
    }

    return false;
}

hkResult MaxPlayhead::tryAdvanceTo( const Playhead& other, hkBool32& reachedOtherOut )
{
    Playhead& self = *this;
    HK_ASSERT( 0x22441122, other.isValid(), "You should not pass an invalid playhead to MaxPlayhead::tryAdvanceTo" );

    // See if we've already reached other
    if ( self >= other )
    {
        reachedOtherOut = true;
        return HK_SUCCESS;
    }
    else
    {
        reachedOtherOut = false;
    }

    // Check that we have been initialized, this could be false if we just started
    if ( HK_VERY_UNLIKELY( !isValid() ) )
    {
        if ( const hkVdbFrame* firstFrame = m_handler.m_input->begin() )
        {
            hkVdbFrame::Iterator cmdIt = firstFrame->getIterator();
            if ( firstFrame->isValid( cmdIt ) )
            {
                set( firstFrame, firstFrame->getValue( cmdIt ) );
            }
        }
    }

    // If we failed above, we can't proceed
    if ( HK_VERY_UNLIKELY( !isValid() ) )
    {
        HK_VDB_SIGNAL_ERROR_MSG( 0xedb00121, hkVdbError::FRAME_SYNC_ERROR, "Cannot set frames when there's nothing in the buffer" );
        return HK_FAILURE;
    }

    // Else, continue with advance
    PlayheadDiff diff = ( other - self );
    if ( diff.m_numFrames > 0 )
    {
        // We are manually moving our processed playhead forward towards our buffer playhead
        hkVdbPlaybackDirection::Enum prevDirection = m_handler.m_state.m_targetDirection;
        m_handler.m_state.m_targetDirection = hkVdbPlaybackDirection::FORWARD;

        // Enter blocking processing loop
        hkVdbPlaybackHandler& handler = m_handler;
        PlayheadProcessingFlags flags = ( PlayheadProcessingFlags::EXCLUSIVE | PlayheadProcessingFlags::STEP );

        m_handler.m_progressReporter->workTillCompleted(
            "Processing frames",
            [&self, &other, &handler, &flags]( hkVdbSignalResult& warningResult, hkVdbSignalResult& errorResult, hkStringBuf& workMsgOut )
            {
                if ( other > self )
                {
                    handler.updateFlagsAndSignal( handler.m_state.m_targetFlags | hkVdbPlaybackFlags::CACHING );
                    self.processFrame( flags );
                    self.advance();
                    hkInt64 nextFrameToProcess = self.getFrameNumber();
                    if ( nextFrameToProcess == -1 )
                    {
                        // No work left to do
                        return false;
                    }

                    flags.orWith( PlayheadProcessingFlags::INCLUSIVE );

                    // We did work
                    return true;
                }

                // No work left to do
                return false;
            },
            hkUint32( diff.m_numFrames ) );

        // Update our state
        m_handler.updateFlagsAndSignal( m_handler.m_state.m_targetFlags & ~hkVdbPlaybackFlags::CACHING );
        
        
        
        
        
        
        
        
        m_handler.m_state.m_targetDirection = prevDirection;

        // Determine results and clamp if needed.
        // self.advance() takes us over the frame boundary. If other was *on* the frame boundary, we can go from
        // ( other > self ) to ( other < self ).  Additionally, if buffer is also pointing to the *last* cmd,
        // the advance will take us passed our last frame and thus invalidate us.
        if ( !self.isValid() || ( self >= other ) )
        {
            self = other;
            reachedOtherOut = true;
        }
    }

    HK_ASSERT( 0x22441123, isValid(), "MaxPlayhead::tryAdvanceTo should not invalidate itself" );
    return HK_SUCCESS;
}

//////////////////////////////////////////////////////////////////////////
// BufferPlayhead
//////////////////////////////////////////////////////////////////////////

BufferPlayhead::BufferPlayhead( hkVdbPlaybackHandler& owner ) :
    Playhead( owner ),
    m_nextFrameIt( hkVdbCmdInput::InvalidIterator ),
    m_nextCmdIt( hkVdbFrame::InvalidIterator ),
    m_atBeginningPlayingBackwards( false )
{}

BufferPlayhead::BufferPlayhead( const Playhead& other ) :
    Playhead( other ),
    m_nextFrameIt( other.m_frameIt ),
    m_nextCmdIt( other.m_cmdIt ),
    m_atBeginningPlayingBackwards( false )
{}


hkResult BufferPlayhead::processFrame( PlayheadProcessingFlags flags )
{
    HK_ASSERT( 0x22441084, flags.noneIsSet( PlayheadProcessingFlags::STEP ), "Should not step from buffered playhead replay" );

    hkVdbCmdInput* input = m_handler.m_input;
    hkVdbCmdDispatcher* dispatcher = m_handler.m_dispatcher;
    

    if ( isValid() )
    {
        const hkVdbFrame* frame;
        const hkVdbCmd* cmd;
        hkBool32 firstRun = true;

        do
        {
            // Play the next cmd from the buffer
            play( frame, cmd );
            m_handler.checkPlayheads( frame, cmd );

            // We want to stay in this frame, so go back to last processed when we reach the end
            if ( !cmd || ( cmd->getType() == hkVdbCmdType::STEP ) )
            {
                m_nextFrameIt = m_frameIt;
                m_nextCmdIt = m_cmdIt;
                break;
            }

            if ( firstRun && flags.anyIsSet( PlayheadProcessingFlags::INCLUSIVE ) ||
                !firstRun && flags.anyIsSet( PlayheadProcessingFlags::EXCLUSIVE ) )
            {
                HK_VDB_VERIFY_REPORTER_OPERATION(
                    dispatcher->dispatch( *frame, *cmd, input->getProtocol() ),
                    *dispatcher,
                    hkVdbError::CMD_DISPATCH_ERROR );
            }

        } while ( true );
    }

    return HK_SUCCESS;
}

hkResult BufferPlayhead::advance()
{
    hkResult result = Playhead::advance();
    m_nextFrameIt = m_frameIt;
    m_nextCmdIt = m_nextCmdIt;
    return result;
}

Playhead BufferPlayhead::peek() const
{
    Playhead nextPlayhead( m_handler );
    nextPlayhead.m_frameIt = m_nextFrameIt;
    nextPlayhead.m_cmdIt = m_nextCmdIt;
    return nextPlayhead;
}

hkResult BufferPlayhead::play( const hkVdbFrame*& frameOut, const hkVdbCmd*& cmdOut )
{
    hkVdbCmdInput* input = m_handler.m_input;

    // If we aren't valid, start consuming from our the next valid playhead
    if ( !isValid() )
    {
        if ( HK_VERY_LIKELY( m_handler.m_processedPlayhead->isValid() ) )
        {
            // Get the next unprocessed playhead.
            Playhead next = m_handler.m_processedPlayhead->peek();
            m_frameIt = m_handler.m_processedPlayhead->m_frameIt;
            m_cmdIt = m_handler.m_processedPlayhead->m_cmdIt + 1; // Not in replay, so add 1
            m_nextFrameIt = next.m_frameIt;
            m_nextCmdIt = next.m_cmdIt;

            // Our processed playhead only moves forward, but if we are moving backward
            // we need to advance backward from where we currently are.
            if ( m_handler.m_state.m_targetDirection == hkVdbPlaybackDirection::BACKWARD )
            {
                m_nextFrameIt = m_handler.m_input->getPrevious( m_frameIt );
                m_nextCmdIt = m_handler.m_input->isValid( m_nextFrameIt ) ? m_handler.m_input->getValue( m_nextFrameIt )->getIterator() : hkVdbFrame::InvalidIterator;
            }
        }
        else if ( m_handler.m_connectionPlayhead->isValid() )
        {
            HK_ASSERT( 0x22441128, !m_input->isFull(), "Some input data won't be processed" );

            // I've we've read into the input buffer, just pick up from the beginning.
            // This really only happens during initialization.
            m_frameIt = m_input->getIterator();
            m_cmdIt =
                m_input->isValid( m_frameIt, true ) ?
                m_input->getValue( m_frameIt )->getIterator() :
                hkVdbFrame::InvalidIterator;
            m_nextFrameIt = m_frameIt;
            m_nextCmdIt = m_cmdIt;
        }
        else
        {
            frameOut = HK_NULL;
            cmdOut = HK_NULL;
            return HK_SUCCESS;
        }
    }

    // Check for holding due to backwards play.
    if ( m_atBeginningPlayingBackwards )
    {
        // See if we are still playing backwards.
        m_atBeginningPlayingBackwards = ( m_handler.m_state.m_targetDirection == hkVdbPlaybackDirection::BACKWARD );

        // If we are, just hold here by returning no frame/cmd.
        if ( m_atBeginningPlayingBackwards )
        {
            frameOut = HK_NULL;
            cmdOut = HK_NULL;
            return HK_SUCCESS;
        }
        // Otherwise, we've changed directions, update our iterators and start moving forward.
        else
        {
            m_nextFrameIt = input->getIterator();
            HK_VDB_VERIFY_CONDITION( input->isValid( m_nextFrameIt, true ), 0xedb00104, hkVdbError::FRAME_SYNC_ERROR );
            const hkVdbFrame* frame = input->getValue( m_nextFrameIt );
            m_nextCmdIt = frame->getIterator();
            HK_VDB_VERIFY_CONDITION( frame->isValid( m_nextCmdIt ), 0xedb00105, hkVdbError::FRAME_SYNC_ERROR );
        }
    }

    // Check for holding due to throttling.
    const hkVdbFrame* nextFrame = ( input->isValid( m_nextFrameIt, true ) ? input->getValue( m_nextFrameIt ) : HK_NULL );
    if ( nextFrame && m_handler.shouldThrottlingConnection( *nextFrame ) )
    {
        frameOut = HK_NULL;
        cmdOut = HK_NULL;
        return HK_SUCCESS;
    }

    // Set our current iters
    m_frameIt = m_nextFrameIt;
    m_cmdIt = m_nextCmdIt;

    // Grab the current frame if valid
    const hkVdbFrame* frame = nextFrame;
    if ( frame && frame->isValid( m_cmdIt ) )
    {
        // Get current data
        frameOut = frame;
        cmdOut = frameOut->getValue( m_cmdIt );

        // Compute our next iters
        m_nextCmdIt = frameOut->getNext( m_cmdIt );
        if ( !frameOut->isValid( m_nextCmdIt ) )
        {
            // If we were at the end playing backwards, we don't want to ack the step and release the connection
            // to give us a new frame.
            if ( ( frame->getFrameNumber() >= input->getLastCompletedFrameNumber() ) &&
                ( m_handler.m_state.m_targetDirection == hkVdbPlaybackDirection::BACKWARD ) )
            {
                frameOut = HK_NULL;
                cmdOut = HK_NULL;
            }

            // Advance to the next frame
            // Note: a reverse wrap around will give us an invalidly high number.
            const hkUint32 nextFrameNumber = hkUint32( m_handler.getNextFrame( frame->getFrameNumber() ) );
            HK_VDB_FAIL_IF_OPERATION_FAILED( setNoSignal( nextFrameNumber, START ) );
        }
    }
    else
    {
        frameOut = HK_NULL;
        cmdOut = HK_NULL;
    }

    // Note: we rely on onStepCompletedSignal to signal the frame change.
    // This is so that the frame change always happens *after* all commands are processed from the frame.
    // This maintains consistency of signals between this code path and the setCurrentFrame codepath.
    return HK_SUCCESS;
}

hkResult BufferPlayhead::set( hkUint32 frameNumber, InterFrameLoc loc )
{
    // Check if we are already at the frame and loc.
    if ( ( getFrameNumber() == frameNumber ) &&
        ( ( loc == START ) == ( getFrame()->getIterator() == m_cmdIt ) ) )
    {
        return HK_SUCCESS;
    }

    // If we haven't been initialized, we are implicitly starting from the last processed frame
    if ( !isValid() )
    {
        *this = *m_handler.m_processedPlayhead;
    }

    // Now set ourselves to be at the desired frame
    if ( setNoSignal( frameNumber, loc ).isSuccess() )
    {
        // We are *at* frameNumber in the set case, so update our current iters.
        m_frameIt = m_nextFrameIt;
        m_cmdIt = m_nextCmdIt;

        // Ensure our processed playhead is advanced to our buffer playhead.
        // This returns true if processed >= buffered when the function returns.
        hkBool32 reachedBufferPlayhead;
        HK_VDB_VERIFY_REPORTER_OPERATION(
            m_handler.m_processedPlayhead->tryAdvanceTo( *m_handler.m_bufferPlayhead, reachedBufferPlayhead ),
            *m_handler.m_processedPlayhead,
            hkVdbError::FRAME_SYNC_ERROR );

        // If we didn't reach the buffer playhead, adjust our buffer playhead.
        // This can happen if someone cancels our attempt via the progress reporter api.
        if ( !reachedBufferPlayhead )
        {
            *m_handler.m_bufferPlayhead = *m_handler.m_processedPlayhead;
        }

        // Update and signal changes.
        // We use getFrameNumber() because it may have changed during tryAdvanceTo (if user cancelled sync).
        hkInt64 finalFrameNumber = m_handler.m_bufferPlayhead->getFrameNumber();
        HK_VDB_VERIFY_CONDITION( finalFrameNumber >= 0, 0xedb00137, hkVdbError::FRAME_SYNC_ERROR );
        return m_handler.updateFrameAndSignal( hkUint32( finalFrameNumber ) );
    }

    return HK_FAILURE;
}

hkResult BufferPlayhead::setNoSignal( hkUint32 frameNumber, InterFrameLoc loc )
{
    hkVdbCmdInput* input = m_handler.m_input;

    // Clear this field, we'll assign it again if it's true before exiting the function.
    // But if we've switched directions and have a valid iterator, we need to make sure it's cleared.
    m_atBeginningPlayingBackwards = false;

    // Get the frame and cmd from our input
    m_nextFrameIt = input->getIterator( frameNumber );
    if ( input->isValid( m_nextFrameIt, true ) )
    {
        const hkVdbFrame* frame = input->getValue( m_nextFrameIt );
        m_nextCmdIt = frame->getIterator();
        if ( frame->isValid( m_nextCmdIt ) )
        {
            // Get our playhead location
            Playhead nextPlayhead = peek();

            // Skip to end if requested
            if ( loc == END ) nextPlayhead.processFrame( PlayheadProcessingFlags::NONE );

            // Update next iters and return
            m_nextFrameIt = nextPlayhead.m_frameIt;
            m_nextCmdIt = nextPlayhead.m_cmdIt;
            return HK_SUCCESS;
        }
    }

    // We've run out of frames from the input
    m_nextFrameIt = hkVdbCmdInput::InvalidIterator;
    m_nextCmdIt = hkVdbFrame::InvalidIterator;
    m_atBeginningPlayingBackwards = ( m_handler.m_state.m_targetDirection == hkVdbPlaybackDirection::BACKWARD );
    return HK_SUCCESS;
}

hkBool32 BufferPlayhead::isValid() const
{
    return Playhead::isValid() || m_atBeginningPlayingBackwards;
}

//////////////////////////////////////////////////////////////////////////
// ConnectionPlayhead
//////////////////////////////////////////////////////////////////////////

ConnectionPlayhead::ConnectionPlayhead( hkVdbPlaybackHandler& handler ) :
    Playhead( handler )
{}

ConnectionPlayhead::ConnectionPlayhead( const Playhead& other ) :
    Playhead( other )
{}

hkResult ConnectionPlayhead::play( hkVdbConnection& connection, const hkVdbFrame*& frameOut, const hkVdbCmd*& cmdOut )
{
    hkVdbCmdInput* input = m_handler.m_input;

    HK_VDB_VERIFY_CONDITION(
        input,
        0xedb00103,
        hkVdbError::CMD_HANDLER_NOT_REGISTERED );

    const hkBool32 bufferPlayheadIsValid = m_handler.m_bufferPlayhead->isValid();
    HK_ON_DEBUG( const hkBool32 processedPlayheadIsValid = m_handler.m_processedPlayhead->isValid(); )

    // Only read-ahead if an advance on the input buffer *wouldn't* invalidate the buffer playhead.
    // Note: because many listeners act on FRAME_END rather than frame changed, we -1 to the buffer playhead frame number.
    // Note: processed playhead is always >= buffer playhead and valid (after first cmd is processed).
    if ( !bufferPlayheadIsValid || !input->isFull() || ( ( m_handler.m_bufferPlayhead->getFrameNumber() - 1 ) > input->getStartingFrameNumber() ) )
    {
        HK_VDB_VERIFY_REPORTER_OPERATION(
            input->readNext( connection, frameOut, cmdOut ),
            *input,
            hkVdbError::CMD_BUFFER_ERROR );

        // Store our max playhead location so that we can be compared with other
        // playheads effectively.
        if ( frameOut && cmdOut )
        {
            HK_ASSERT( 0x22440962, frameOut->getFrameNumber() >= getFrameNumber(), "The connection has returned an older frame" );
            HK_ASSERT( 0x22440963, ( frameOut->getFrameNumber() > getFrameNumber() ) || ( cmdOut->getIndex() > getCmdIndex() ), "The connection has returned an older cmd" );
            m_frameIt = input->getIterator( frameOut->getFrameNumber() );
            m_cmdIt = frameOut->getIterator( cmdOut->getIndex() );
        }

        HK_ASSERT(
            0x22441039,
            ( bool( bufferPlayheadIsValid ) == bool( m_handler.m_bufferPlayhead->isValid() ) ) && ( bool( processedPlayheadIsValid ) == bool( m_handler.m_processedPlayhead->isValid() ) ),
            "Read-ahead invalidated a playhead" );
    }
    else
    {
        frameOut = HK_NULL;
        cmdOut = HK_NULL;
    }

    return HK_SUCCESS;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
