// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Visualize/hkServerObjectHandler.h>
#include <Common/Visualize/hkVisualDebuggerCmdType.h>

#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbCmdHandler.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbPlaybackHandler.h>
#include <VisualDebugger/VdbServices/System/Cache/hkVdbCache.h>

struct hkVdbCmd;
#ifdef HK_VDB_USE_HK_SERIALIZE
namespace hkSerialize { class Load; }
#else
namespace hkVdbSerialize { class TypeLoad; }
#endif

//////////////////////////////////////////////////////////////////////////
/// A handler for object-inspection-related cmds.
//////////////////////////////////////////////////////////////////////////
class HK_EXPORT_VDB hkVdbObjectHandler : public hkVdbCmdHandler<hkVdbCmdHandlerType::OBJECT>
{
public:
    typedef hkInt32 TypeId;
    typedef hkUint64 ObjectId;
    typedef hkUint64 ConnectionId;

    HK_DECLARE_CLASS( hkVdbObjectHandler, New );
    hkVdbObjectHandler();
    ~hkVdbObjectHandler();

    //
    // hkVdbCmdHandler
    //

    virtual hkResult registerSelf( hkVdbClient& client ) HK_OVERRIDE;
    virtual hkResult unregisterSelf( hkVdbClient& client ) HK_OVERRIDE;
    virtual void onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection ) HK_OVERRIDE;

    //
    // Helpers
    //

    /// Returns the Type registered for the given id.
    const hkReflect::Type* getObjectType( TypeId typeId ) const;

    //
    // Signals
    //

    
    struct ObjectsAddedCmd;
    struct ObjectsUpdatedCmd;
    struct ObjectsRemovedCmd;
    struct DisposeObjectsCmd;

    /// Base class for all object-related cmd data.
    struct ObjectsCmd
    {
        HK_DECLARE_CLASS( ObjectsCmd, New );
        ObjectsCmd() : m_cmdType( hkVdbCmdType::INVALID ) {}

        /// The cmd type.
        
        hkVdbCmdType::Enum m_cmdType;

        /// The ids of the objects to be added/updated/removed.
        hkArrayView<ObjectId> m_ids;

        /// Returns the cmd as an added objects cmd if that is the type; null otherwise.
        HK_INLINE const ObjectsAddedCmd* asObjectsAddedCmd() const;

        /// Returns the cmd as an updated objects cmd if that is the type; null otherwise.
        HK_INLINE const ObjectsUpdatedCmd* asObjectsUpdatedCmd() const;

        /// Returns the cmd as a removed objects cmd if that is the type; null otherwise.
        HK_INLINE const ObjectsRemovedCmd* asObjectsRemovedCmd() const;

        /// Returns the cmd as a disposed objects cmd if that is the type; null otherwise.
        HK_INLINE const DisposeObjectsCmd* asDisposeObjectsCmd() const;
    };

    /// Info about updated objects.
    struct ObjectsUpdatedCmd : public ObjectsCmd
    {
        HK_DECLARE_CLASS( ObjectsUpdatedCmd, New );
        ObjectsUpdatedCmd() { m_cmdType = hkVdbCmdType::UPDATE_OBJECTS; }

        /// The data for the objects.
        hkArrayView<hkVdbReflect::Var> m_objects;
    };

    /// Info about newly added objects.
    struct ObjectsAddedCmd : public ObjectsUpdatedCmd
    {
        HK_DECLARE_CLASS( ObjectsAddedCmd, New );
        ObjectsAddedCmd() { m_cmdType = hkVdbCmdType::ADD_OBJECTS; }

        /// Objects flags.
        hkObjectFlags m_flags;

        /// The tag (usually from process/viewer) associated with the objects.
        int m_tag;
    };

    /// Info about removed objects (the server has released them).
    struct ObjectsRemovedCmd : public ObjectsCmd
    {
        HK_DECLARE_CLASS( ObjectsRemovedCmd, New );
        ObjectsRemovedCmd() { m_cmdType = hkVdbCmdType::REMOVE_OBJECTS; }
    };

    /// Info about disposed objects (they are being removed from cmd input buffer).
    struct DisposeObjectsCmd : public ObjectsUpdatedCmd
    {
        HK_DECLARE_CLASS( DisposeObjectsCmd, New );
        DisposeObjectsCmd() { m_cmdType = hkVdbCmdType::CLIENT_INTERNAL; }
    };

    struct ConnectIdsCmd;
    struct DisconnectIdsCmd;
    struct SetConnectionIdsCmd;

    /// Base class for all object-connectivity-related cmd data.
    struct ConnectivityCmd
    {
        HK_DECLARE_CLASS( ConnectivityCmd, New );
        ConnectivityCmd() : m_cmdType( hkVdbCmdType::INVALID ) {}

        /// The cmd type.
        hkVdbCmdType::Enum m_cmdType;

        /// The id from which to connect/disconnect ids (id -> ids).
        ConnectionId m_id;

        /// The ids involved in the connectivity (id -> ids).
        hkArrayView<ConnectionId> m_ids;

        /// Returns the cmd as a connect ids cmd if that is the type; null otherwise.
        HK_INLINE const ConnectIdsCmd* asConnectIdsCmd() const;

        /// Returns the cmd as a disconnect ids cmd if that is the type; null otherwise.
        HK_INLINE const DisconnectIdsCmd* asDisconnectIdsCmd() const;

        /// Returns the cmd as a set connection ids cmd if that is the type; null otherwise.
        HK_INLINE const SetConnectionIdsCmd* asSetConnectionIdsCmd() const;
    };

    /// Ids are being connected.
    struct ConnectIdsCmd : public ConnectivityCmd
    {
        HK_DECLARE_CLASS( ConnectIdsCmd, New );
        ConnectIdsCmd() { m_cmdType = hkVdbCmdType::CONNECT_OBJECTS; }

        /// The tag (usually from process/viewer) associated with the connection.
        int m_tag;
    };

    /// Ids are being disconnected.
    struct DisconnectIdsCmd : public ConnectivityCmd
    {
        HK_DECLARE_CLASS( DisconnectIdsCmd, New );
        DisconnectIdsCmd() { m_cmdType = hkVdbCmdType::DISCONNECT_OBJECTS; }
    };

    /// And ids connections are being set.
    struct SetConnectionIdsCmd : public ConnectIdsCmd
    {
        HK_DECLARE_CLASS( SetConnectionIdsCmd, New );
        SetConnectionIdsCmd() { m_cmdType = hkVdbCmdType::CUSTOM; }
    };

    HK_DECLARE_SIGNAL( ObjectTypeRegisteredSignal, hkSignal3< TypeId, const hkReflect::Type&, hkVdbSignalResult& > );
    /// Fired when a new object type is registered.
    ObjectTypeRegisteredSignal m_objectTypeRegistered;

    HK_DECLARE_SIGNAL( ObjectsCmdReceivedSignal, hkSignal2< const ObjectsCmd&, hkVdbSignalResult& > );
    /// Fired when object-related cmds are received/processed.
    ObjectsCmdReceivedSignal m_objectsCmdReceived;

    HK_DECLARE_SIGNAL( ConnectivityCmdReceivedSignal, hkSignal2< const ConnectivityCmd&, hkVdbSignalResult& > );
    /// Fired when object-connectivity-related cmds are received/processed.
    ConnectivityCmdReceivedSignal m_connectivityCmdReceived;
    

    //
    // Internal use
    //

    void onStepStartedSignal( const hkVdbFrame& frame, const hkVdbCmd& cmd );
    void onPlaybackInfoReceivedSignalInternal( const hkVdbPlaybackHandler::PlaybackInfo& info, hkVdbSignalResult& result );
    void onEntryDeallocatedSignal( const hkVdbCachedObjects& cache, const hkVdbCachedObject& object );
    void onSparseDataDeallocatedSignal( const hkVdbCachedObjects& cache, const hkVdbCachedObject& object, const hkReflect::Var data );
    void onPersistCacheFrameSignal( const hkVdbCache& cache, hkUint32 frame, hkVdbLocalOStream& dataWriter, hkVdbSignalResult& result );
    hkResult processRegisterObjectTypeCmd( const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader );
    hkResult processAddOrUpdateObjectsCmd( const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader );
    hkResult processRemoveObjectsCmd( int protocol, hkVdbLocalIStream& dataReader );
    hkResult processConnectOrDisconnectObjectsCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader );

protected:
    struct AddObjectInfo
    {
        HK_DECLARE_CLASS( AddObjectInfo, New, Pod );
        void init() { new ( &m_data ) hkVdbReflect::Var(); }
        int m_tag;
        hkUint64 m_id;
        hkVdbReflect::Var m_data;
        hkObjectFlags m_flags;
        HK_INLINE hkBool32 operator<( const AddObjectInfo& other ) const
        {
            return
                ( m_tag < other.m_tag ) ||
                ( m_flags < other.m_flags );
        }
    };
    static void processAddObjectBatches(
        hkArray<AddObjectInfo>& addObjectInfos,
        void ( processAddObjectBatchFunc )(
            hkArrayView<hkUint64> ids,
            hkArrayView<hkVdbReflect::Var> datas,
            hkObjectFlags flags,
            int tag,
            void* handle ),
        void* handle = HK_NULL );

    struct CacheAction
    {
        enum Enum
        {
            CREATE,
            ACCESS,
            ACCESS_OR_CREATE_TRANSIENT,
            ACCESS_OR_CREATE,
        };
    };
    template<
        typename CachedEntryTypeCinfo,
        typename CachedEntryType,
        typename CachedEntriesType,
        typename IdType>
    void convertToUids(
        CachedEntriesType& cachedEntries,
        CacheAction::Enum cacheAction,
        hkUint32 frame,
        const hkArrayView<IdType>& ids,
        hkLocalArray<IdType>& uidsOut,
        hkLocalArray<CachedEntryType*>& entriesOut )
    {
        for ( int i = 0; i < ids.getSize(); i++ )
        {
            IdType id = ids[i];
            CachedEntryType* entry;
            IdType uid;
#ifndef HK_VDB_DISABLE_OBJECT_CMD_CACHING
            switch ( cacheAction )
            {
                case CacheAction::ACCESS_OR_CREATE_TRANSIENT:
                case CacheAction::ACCESS_OR_CREATE:
                {
                    if ( CachedEntryType* object = cachedEntries.accessEntry( id ) )
                    {
                        if ( object->getFrameRemoved() == -1 )
                        {
                            entry = object;
                            uid = object->getUid();
                            break;
                        }
                    }
                    // else fall through to CREATE
                }
                case CacheAction::CREATE:
                {
                    CachedEntryTypeCinfo cinfo(
                        -1,
                        id,
                        frame );
                    entry = &cachedEntries.createEntry( cinfo );
                    uid = entry->getUid();
                    if ( cacheAction == CacheAction::ACCESS_OR_CREATE_TRANSIENT )
                    {
                        m_transientEntryStreamIds.pushBack( id );
                        HK_ON_DEBUG( if ( !m_transientObjectsFrame ) m_transientObjectsFrame = frame; )
                        HK_ASSERT( 0x22441208, frame > m_transientObjectsClearedFrame, "Cleared transients too early" );
                    }
                    break;
                }
                case CacheAction::ACCESS:
                {
                    if ( CachedEntryType* object = cachedEntries.accessEntry( id ) )
                    {
                        entry = object;
                        uid = object->getUid();
                    }
                    else
                    {
                        
                        
                        entry = HK_NULL;
                        uid = id;
                        
                        
                    }
                    break;
                }
            }
#else
            entry = HK_NULL;
            uid = id;
#endif
            entriesOut.pushBack( entry );
            uidsOut.pushBack( uid );
        }
    }

    hkResult addOrUpdateObjectsInternal(
        hkVdbCmdType::Enum type,
        const hkArrayView<ObjectId>& ids,
        const hkArrayView<hkVdbReflect::Var>& objects,
        hkObjectFlags flags,
        int tag );

    hkResult removeObjectsInternal(
        const hkArrayView<ObjectId>& ids );

    hkResult connectOrDisconnectInternal(
        hkVdbCmdType::Enum type,
        ConnectionId id,
        const hkArrayView<ConnectionId>& ids,
        hkConnectivityFlags flags,
        int tag );

    void clearTypes();
    void clearTransientEntries();

    void updateObjectsToFrame( hkUint32 frame, hkVdbSignalResult& result );

    // Avoid allocations during dispatch by having these declared and just reusing them
    ObjectsAddedCmd m_objectsCmd;
    ConnectIdsCmd m_connectivityCmd;

    hkUint32 m_lastPlaybackFrame;
    HK_DEBUG_ONLY_MEMBER( hkUint32, m_transientObjectsFrame );
    HK_DEBUG_ONLY_MEMBER( hkUint32, m_transientObjectsClearedFrame );
    HK_DEBUG_ONLY_MEMBER( hkUint64, m_totalObjectCount );
    HK_DEBUG_ONLY_MEMBER( hkUint64, m_updatedObjectCount );

#ifdef HK_VDB_USE_HK_SERIALIZE
    hkSerialize::Load* m_objectLoader;
#else
    hkVdbSerialize::TypeLoad* m_typeLoader;
#endif

    hkRefPtr<hkVdbCmdInput> m_input;
    hkRefPtr<hkVdbCache> m_cache;
    hkRefPtr<hkVdbProgressReporter> m_progressReporter;
    hkRefPtr<hkVdbPlaybackHandler> m_playbackHandler;
    hkBool32 m_cachingFrames;
    hkBool32 m_haveDataFromCachingFrames;
    struct TypesInfo
    {
        TypesInfo(
            TypeId id = 0,
            const hkReflect::Type* serverType = HK_NULL,
            const hkReflect::Type* clientType = HK_NULL ) :
            m_id( id ),
            m_serverType( serverType ),
            m_clientType( clientType )
        {}
        TypeId m_id;
        const hkReflect::Type* m_serverType;
        const hkReflect::Type* m_clientType;
    };
    hkVdbMap<TypeId, TypesInfo> m_typeIdToInfo;
    hkVdbMap<const hkReflect::Type*, TypesInfo> m_clientTypeToInfo;
    hkArray<hkUint64> m_transientEntryStreamIds;

    friend class hkVdbInternalHandler;
    friend class hkVdbPersistentStateCache;
    friend class InternalServerObjectSerializer;
};

#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbObjectHandler.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
