// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbServices/hkVdbServices.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbObjectHandler.h>

#include <Common/Base/Container/LocalArray/hkLocalArray.h>
#include <Common/Base/Reflect/Core/Detail/hkReflectTypeDetail.h>
#include <Common/Base/Reflect/Builder/hkTypeBuilder.h>
#include <Common/Base/Serialize/Core/hkSerializeCore.h>
#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Visualize/Serialize/hkPersistableReadFormat.h>
#include <Common/Visualize/hkVisualDebuggerCmd.h>

#include <VisualDebugger/VdbServices/hkVdbClient.h>
#include <VisualDebugger/VdbServices/System/Cache/hkVdbCache.h>
#include <VisualDebugger/VdbServices/System/Cache/hkVdbPersistentStateCache.h>
#include <VisualDebugger/VdbServices/System/hkVdbProgressReporter.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdInput.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdDispatcher.h>

#ifndef HK_VDB_USE_HASH_MAP
#include <Common/Base/Container/PointerMap/hkMap.h>
#include <Common/Base/Container/PointerMap/hkMap.hxx>
template class HK_EXPORT_COMMON hkMapBase<hkVdbObjectHandler::TypeId, const hkReflect::Type*>;
template class HK_EXPORT_COMMON hkMap<hkVdbObjectHandler::TypeId, const hkReflect::Type*>;
template class HK_EXPORT_COMMON hkMapBase<hkReflect::Type*, hkUlong>;
template class HK_EXPORT_COMMON hkMap<hkReflect::Type*, hkUlong>;
#endif

#define DEBUG_LOG_IDENTIFIER "vdb.Cmd.Handler.Object"
#include <Common/Base/System/Log/hkLog.hxx>

//#define REPORT_OBJECT_COUNTS
//#define PRINT_REPLAY_IDS
//#define STRICT_UPDATE_POLICY
//#define ENABLE_BREAK_ON_DEBUG_IDS

#ifdef HK_VDB_ENABLE_OBJECT_CMD_PROFILING
#include <Common/Base/System/Stopwatch/hkSystemClock.h>
#endif

#ifndef HK_VDB_DISABLE_OBJECT_CMD_SIGNALLING
#define DO_SIGNAL( x ) x
#else
#define DO_SIGNAL( x )
#endif

// If this assert hits, change reinterpret cast in onCachedObjectDataDeallocatedSignal
HK_COMPILE_TIME_ASSERT( sizeof( hkReflect::Var ) == sizeof( hkVdbReflect::Var ) );

class InternalServerObjectSerializer : public hkServerObjectSerializer
{
public:
    InternalServerObjectSerializer(
        hkVdbOStream* outStream,
        hkVdbMap<const hkReflect::Type*, hkVdbObjectHandler::TypesInfo>& clientTypeToInfo ) :
        hkServerObjectSerializer( outStream, HK_NULL ),
        m_clientTypeToInfo( clientTypeToInfo )
    {
        m_outStreamLock = new hkCriticalSection( 1000 );
    }
    virtual int addObjects(
        const hkArrayView<ObjectId>& ids,
        const hkArrayView<hkReflect::Var>& objects,
        hkObjectFlags flags,
        int tag ) HK_OVERRIDE
    {
        using namespace hkReflect;

        int objectDataSize = 0;
        hkLocalArray<ObjectId> addedIds( ids.getSize() );
        hkLocalArray<hkReflect::Var> addedObjects( ids.getSize() );
        hkLocalArray<int> addedTypeIds( ids.getSize() );
        for ( int i = 0; i < ids.getSize(); i++ )
        {
            ObjectId id = ids[i];
            Var vdbObject = objects[i];
            hkVdbObjectHandler::TypesInfo info = m_clientTypeToInfo.getWithDefault( vdbObject.getType(), hkVdbObjectHandler::TypesInfo() );
            if ( info.m_clientType )
            {
                objectDataSize += hkVdbSerialize::getObjectSize( vdbObject );
                addedIds.pushBack( id );
                addedObjects.pushBack( vdbObject );
                addedTypeIds.pushBack( info.m_id );
            }
        }

        if ( objectDataSize > 0 )
        {
            writeAddOrUpdateObjectsCmdHeader(
                true,
                addedIds.getSize(),
                objectDataSize,
                &flags );

            for ( int i = 0; i < addedIds.getSize(); i++ )
            {
                ObjectId id = addedIds[i];
                hkVdbReflect::Var obj = addedObjects[i];

                writeAddOrUpdateObjectsCmdObjectStart( true, id, obj, addedTypeIds[i] );

#ifdef HK_VDB_USE_HK_SERIALIZE
#error Need to impl this case for saving
#else
                hkVdbSerialize::reserializeObject(
                    *m_outStream->getStreamWriter(),
                    obj );
#endif
            }

            writeAddOrUpdateObjectsCmdFooter( true, &tag );
        }

        return objectDataSize;
    }
    using hkServerObjectSerializer::addObjects;
    ~InternalServerObjectSerializer()
    {
        delete m_outStreamLock;
    }
    hkVdbMap<const hkReflect::Type*, hkVdbObjectHandler::TypesInfo>& m_clientTypeToInfo;
};

namespace
{
    hkResult handleRegisterObjectTypeCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
#ifndef HK_VDB_DISABLE_OBJECT_CMD_READING
        hkVdbObjectHandler* handler = reinterpret_cast< hkVdbObjectHandler* >( userHandle );
        return handler->processRegisterObjectTypeCmd( cmd, protocol, dataReader );
#else
        return HK_SUCCESS;
#endif
    }

    hkResult handleAddOrUpdateObjectsCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
#ifndef HK_VDB_DISABLE_OBJECT_CMD_READING
        hkVdbObjectHandler* handler = reinterpret_cast< hkVdbObjectHandler* >( userHandle );
        return handler->processAddOrUpdateObjectsCmd( cmd, protocol, dataReader );
#else
        return HK_SUCCESS;
#endif
    }

    hkResult handleRemoveObjectsCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
#ifndef HK_VDB_DISABLE_OBJECT_CMD_READING
        hkVdbObjectHandler* handler = reinterpret_cast< hkVdbObjectHandler* >( userHandle );
        return handler->processRemoveObjectsCmd( protocol, dataReader );
#else
        return HK_SUCCESS;
#endif
    }

    hkResult handleConnectOrDisconnectObjectsCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
#ifndef HK_VDB_DISABLE_OBJECT_CMD_READING
        hkVdbObjectHandler* handler = reinterpret_cast< hkVdbObjectHandler* >( userHandle );
        return handler->processConnectOrDisconnectObjectsCmd( cmd.getType(), protocol, dataReader );
#else
        return HK_SUCCESS;
#endif
    }

#ifdef ENABLE_BREAK_ON_DEBUG_IDS
    const hkUint64 s_id = 0;
    const hkUint64 s_uid = 0;
    const hkVdbCmdType::Enum s_types[] =
    {
        hkVdbCmdType::INVALID
    };
    template<typename T>
    void breakOnId( hkVdbCmdType::Enum cmdType, const T& idOrIds );
    template<>
    void breakOnId( hkVdbCmdType::Enum cmdType, const hkUint64& id )
    {
        hkBool32 enabled = ( ( HK_COUNT_OF( s_types ) == 1 ) && ( s_types[0] == hkVdbCmdType::INVALID ) );
        for ( int i = 0; !enabled && ( i < HK_COUNT_OF( s_types ) ); i++ )
        {
            enabled = ( s_types[i] == cmdType );
        }

        if ( enabled )
        {
            if ( s_uid && ( id == s_uid ) )
            {
                HK_BREAKPOINT( 0 );
            }
            if ( s_id && ( hkVdbCmdId::extractStreamId( id ) == s_id ) )
            {
                HK_BREAKPOINT( 0 );
            }
        }
    }
    template<>
    void breakOnId( hkVdbCmdType::Enum cmdType, const hkArrayView<hkUint64>& ids )
    {
        for each ( hkUint64 id in ids )
        {
            breakOnId( cmdType, id );
        }
    }
    template<>
    void breakOnId( hkVdbCmdType::Enum cmdType, const hkLocalArray<hkUint64>& ids )
    {
        hkArrayView<hkUint64> idsView = ids;
        breakOnId( cmdType, idsView );
    }
#define BREAK_ON_ID(cmdType, idOrIds) breakOnId(cmdType, idOrIds)
#else
#define BREAK_ON_ID(cmdType, idOrIds)
#endif

#ifdef HK_VDB_ENABLE_OBJECT_CMD_PROFILING

    hkUint64 s_objectCmdReadTickAcc = 0;
    hkUint64 s_objectCmdReadTick = 0;
    hkUint64 s_objectCmdProcessTickAcc = 0;
    hkUint64 s_objectCmdProcessTick = 0;
    hkUint64 s_objectCmdReportTickAcc = 0;
    hkUint64 s_objectCmdReportTick = 0;
    void checkAndReportObjectCmdTime( const hkVdbPlaybackHandler::PlaybackInfo& info, hkVdbSignalResult& )
    {
        if ( info.flagWasSet( hkVdbPlaybackFlags::FRAME_ENDED ) )
        {
            hkUint64 tick = hkSystemClock::getTickCounter();
            s_objectCmdReportTickAcc += ( tick - s_objectCmdReportTickAcc );
            if ( hkSystemClock::secondsFromTicks( s_objectCmdReportTickAcc ) > 1.0f )
            {
                HK_REPORT2(
                    0x0,
                    "Object Cmd Read: " << hkSystemClock::secondsFromTicks( s_objectCmdReadTickAcc ) << "s\t\t"
                    "Process: " << hkSystemClock::secondsFromTicks( s_objectCmdProcessTickAcc ) << "s" );
                s_objectCmdReadTickAcc = 0;
                s_objectCmdProcessTickAcc = 0;
                s_objectCmdReportTickAcc = 0;
            }
        }
    }

#define START_OBJECT_CMD_READ() s_objectCmdReadTick = hkSystemClock::getTickCounter()
#define END_OBJECT_CMD_READ() s_objectCmdReadTickAcc += ( hkSystemClock::getTickCounter() - s_objectCmdReadTick )
#define START_OBJECT_CMD_PROCESS() s_objectCmdProcessTick = hkSystemClock::getTickCounter()
#define END_OBJECT_CMD_PROCESS() s_objectCmdProcessTickAcc += ( hkSystemClock::getTickCounter() - s_objectCmdProcessTick )

#else
#define START_OBJECT_CMD_READ()
#define END_OBJECT_CMD_READ()
#define START_OBJECT_CMD_PROCESS()
#define END_OBJECT_CMD_PROCESS()
#endif
};

hkVdbObjectHandler::hkVdbObjectHandler() :
    hkVdbCmdHandler<hkVdbCmdHandlerType::OBJECT>( &s_debugLog ),
    m_lastPlaybackFrame( hkUint32( -1 ) )
    HK_ON_DEBUG( , m_transientObjectsFrame( 0 ) )
    HK_ON_DEBUG( , m_transientObjectsClearedFrame( 0 ) )
    HK_ON_DEBUG( , m_totalObjectCount( 0 ) )
    HK_ON_DEBUG( , m_updatedObjectCount( 0 ) )
#ifdef HK_VDB_USE_HK_SERIALIZE
    , m_objectLoader( HK_NULL )
#else
    , m_typeLoader( HK_NULL )
#endif
    , m_cachingFrames( false )
    , m_haveDataFromCachingFrames( false )
{}

hkVdbObjectHandler::~hkVdbObjectHandler()
{
    // Clear the objects cache as these may rely on our allocated types.
    // This is necessary because we will get destructed before the cache clears out any lingering entries.
    // Note: this is a blanket clear which assumes no one else is adding cached objects that need to stick around.
    if ( m_cache ) m_cache->objects().clear();
    clearTypes();
#ifdef HK_VDB_USE_HK_SERIALIZE
    delete m_objectLoader;
#endif
}

hkResult hkVdbObjectHandler::registerSelf( hkVdbClient& client )
{
    using namespace hkVdbCmdType;

    bool succeeded = true;
    hkVdbCmdDispatcher& dispatcher = client.getCmdDispatcher();

    succeeded &= ( dispatcher.registerHandler( REGISTER_OBJECT_TYPE, handleRegisterObjectTypeCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( ADD_OBJECTS, handleAddOrUpdateObjectsCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( UPDATE_OBJECTS, handleAddOrUpdateObjectsCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( REMOVE_OBJECTS, handleRemoveObjectsCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( CONNECT_OBJECTS, handleConnectOrDisconnectObjectsCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( DISCONNECT_OBJECTS, handleConnectOrDisconnectObjectsCmdFunc, this, this ).isSuccess() );

    m_input = &client.getCmdInput();
    m_cache = &client.getCache();
    m_progressReporter = &client.getProgressReporter();
    m_playbackHandler = client.getCmdHandler<hkVdbPlaybackHandler>();
    m_cachingFrames = m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::CACHING );
    m_haveDataFromCachingFrames = false;

    HK_SUBSCRIBE_TO_SIGNAL( client.m_stepStarted, this, hkVdbObjectHandler );
    HK_SUBSCRIBE_TO_SIGNAL( m_cache->objects().m_entryDeallocated, this, hkVdbObjectHandler );
    HK_SUBSCRIBE_TO_SIGNAL( m_cache->objects().m_sparseDataDeallocated, this, hkVdbObjectHandler );
    HK_SUBSCRIBE_TO_SIGNAL( client.getPersistentStateCache().m_persistCacheFrame, this, hkVdbObjectHandler );
    HK_SUBSCRIBE_TO_SIGNAL( m_playbackHandler->m_playbackInfoReceivedInternal, this, hkVdbObjectHandler );

    // We haven't processed any frames from the input yet, so we are still invalid.
    m_lastPlaybackFrame = hkUint32( -1 );

    HK_VDB_VERIFY_REPORTER_CONDITION( succeeded, dispatcher, hkVdbError::CMD_HANDLER_REGISTRATION_ERROR );

#ifdef HK_VDB_ENABLE_OBJECT_CMD_PROFILING
    m_playbackHandler->m_playbackInfoReceivedInternal.subscribe( checkAndReportObjectCmdTime );
#endif

    return HK_SUCCESS;
}

hkResult hkVdbObjectHandler::unregisterSelf( hkVdbClient& client )
{
    using namespace hkVdbCmdType;

    bool succeeded = true;
    hkVdbCmdDispatcher& dispatcher = client.getCmdDispatcher();

    succeeded &= ( dispatcher.unregisterHandler( REGISTER_OBJECT_TYPE ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( ADD_OBJECTS ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( UPDATE_OBJECTS ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( REMOVE_OBJECTS ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( CONNECT_OBJECTS ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( DISCONNECT_OBJECTS ).isSuccess() );

    client.m_stepStarted.unsubscribeAll( this );
    m_cache->objects().m_entryDeallocated.unsubscribeAll( this );
    m_cache->objects().m_sparseDataDeallocated.unsubscribeAll( this );
    client.getPersistentStateCache().m_persistCacheFrame.unsubscribeAll( this );
    m_playbackHandler->m_playbackInfoReceivedInternal.unsubscribeAll( this );

    m_input = HK_NULL;
    // Need this ptr during dtor to ensure it gets cleared before we deallocate types used in cached objects
    // m_cache = HK_NULL;
    m_progressReporter = HK_NULL;
    m_playbackHandler = HK_NULL;

    HK_VDB_VERIFY_REPORTER_CONDITION( succeeded, dispatcher, hkVdbError::CMD_HANDLER_REGISTRATION_ERROR );

#ifdef HK_VDB_ENABLE_OBJECT_CMD_PROFILING
    m_playbackHandler->m_playbackInfoReceivedInternal.unsubscribe( checkAndReportObjectCmdTime );
#endif

    return HK_SUCCESS;
}

void hkVdbObjectHandler::onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection )
{
    if ( use == hkVdbConnectionUse::APPLICATION )
    {
        m_objectsCmd.m_ids.clear();
        m_objectsCmd.m_cmdType = hkVdbCmdType::INVALID;
        m_connectivityCmd.m_ids.clear();
        m_connectivityCmd.m_cmdType = hkVdbCmdType::INVALID;
        HK_ON_DEBUG( m_transientObjectsFrame = 0 );
        HK_ON_DEBUG( m_transientObjectsClearedFrame = 0 );
        HK_ON_DEBUG( m_totalObjectCount = 0 );
        HK_ON_DEBUG( m_updatedObjectCount = 0 );
        m_transientEntryStreamIds.clear();
        clearTypes();
#ifdef HK_VDB_USE_HK_SERIALIZE
        delete m_objectLoader;
        m_objectLoader = HK_NULL;
#endif
        m_lastPlaybackFrame = hkUint32( -1 );
    }
}

const hkReflect::Type* hkVdbObjectHandler::getObjectType( TypeId typeId ) const
{
    return m_typeIdToInfo.getWithDefault( typeId, TypesInfo() ).m_clientType;
}

void hkVdbObjectHandler::onStepStartedSignal( const hkVdbFrame& frame, const hkVdbCmd& cmd )
{
    // Note: Null check necessary because persistent cached uses this class w/o registering
    if ( m_playbackHandler && m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::REPLAYING ) )
    {
        // We don't want to clear transients during replaying, they are from the live stream.
        return;
    }

    clearTransientEntries();
}

void hkVdbObjectHandler::onPlaybackInfoReceivedSignalInternal( const hkVdbPlaybackHandler::PlaybackInfo& info, hkVdbSignalResult& result )
{
    // When we start caching frames, we note this and wait until end of caching to signal listeners
    // of objects that were processed.  This greatly reduces the overall load on the listeners.
    if ( info.flagWasSet( hkVdbPlaybackFlags::CACHING ) )
    {
        HK_ASSERT( 0x22441401, !(m_cachingFrames || m_haveDataFromCachingFrames), "Didn't properly apply state after previously caching frames" );
        m_cachingFrames = true;
        m_haveDataFromCachingFrames = false;
    }

    // During fast replay and caching, no need to update all our OI info, wait till we've completed before we do full update.
    if ( info.m_targetFlags.anyIsSet( hkVdbPlaybackFlags::FAST_REPLAY | hkVdbPlaybackFlags::CACHING ) )
    {
        return;
    }

    // We need to replay from the cache when finish a frame while replaying, when we stop replaying, or if we have data from caching frames.
    if ( info.flagWasSet( hkVdbPlaybackFlags::FRAME_ENDED ) || info.flagWasCleared( hkVdbPlaybackFlags::REPLAYING ) || m_haveDataFromCachingFrames )
    {
        if ( info.m_targetFlags.anyIsSet( hkVdbPlaybackFlags::REPLAYING ) || info.flagWasCleared( hkVdbPlaybackFlags::REPLAYING ) || m_haveDataFromCachingFrames )
        {
            updateObjectsToFrame( info.m_targetFrame, result );
        }
        else
        {
            m_lastPlaybackFrame = info.m_targetFrame;
#if defined( REPORT_OBJECT_COUNTS ) && defined( HK_DEBUG )
            hkStringBuf buf;
            buf.printf( "\n\tTotal Object Count: %llu\n", m_totalObjectCount );
            buf.appendPrintf( "\tUpdated Object Count: %llu\n", m_updatedObjectCount );
            HK_REPORT2( 0, buf );
#endif
            HK_ON_DEBUG( m_updatedObjectCount = 0 );
        }
    }

    // Clear these fields as we will have signaled for frame-cached object data above.
    m_cachingFrames = false;
    m_haveDataFromCachingFrames = false;
}

void hkVdbObjectHandler::onEntryDeallocatedSignal( const hkVdbCachedObjects& cache, const hkVdbCachedObject& object )
{
    // Pass our initial data to the sparse data dealloc, so we send a signal for it.
    onSparseDataDeallocatedSignal( cache, object, object.getData( 0 ) );
}

void hkVdbObjectHandler::onSparseDataDeallocatedSignal( const hkVdbCachedObjects& cache, const hkVdbCachedObject& object, const hkReflect::Var data )
{
    HK_ASSERT(
        0x22441214,
        m_cache && &m_cache->objects() == &cache,
        "Should not be changing cache after registration or listening to signals after unregistration" );

    // Listeners will only care about the object data going away.
    
    
    hkVdbReflect::Var vdbData = data;
    if ( vdbData.isValid() )
    {
        hkUint64 uid = object.getUid();

        m_objectsCmd.m_cmdType = hkVdbCmdType::CLIENT_INTERNAL; // signifies dispose cmd
        m_objectsCmd.m_ids = hkArrayViewT::fromSingleObject<ObjectId>( uid );
        m_objectsCmd.m_objects = hkArrayViewT::make<hkVdbReflect::Var>( &vdbData, 1 );

        hkVdbSignalResult result;
        DO_SIGNAL( m_objectsCmdReceived.fire( m_objectsCmd, result ); )
        if ( result != hkVdbError::NONE )
        {
            
        }
    }
}

void hkVdbObjectHandler::onPersistCacheFrameSignal( const hkVdbCache& cache, hkUint32 frame, hkVdbLocalOStream& dataWriter, hkVdbSignalResult& result )
{
    InternalServerObjectSerializer serializer( &dataWriter, m_clientTypeToInfo );

    // Go over objects and create addObject and connection cmds.
    hkVdbCachedObjects& objects = cache.objects();
    hkLocalArray<hkVdbObjectHandler::AddObjectInfo> addedObjects( objects.getNumEntries() );
    for ( hkVdbCachedObjects::Iterator iter = objects.getIterator();
        objects.isValid( iter );
        iter = objects.getNext( iter ) )
    {
        const hkVdbCachedObject& object = *objects.getValue( iter );

        if ( object.isAdded( frame ) )
        {
            const hkUint64 streamId = object.getStreamId();
            hkReflect::Var data = object.getData( frame );

            // Add object cmd batching
            if ( data )
            {
                hkVdbObjectHandler::AddObjectInfo& addedObject = addedObjects.expandOne();
                addedObject.init();
                addedObject.m_tag = object.getTag();
                addedObject.m_id = streamId;
                addedObject.m_data = data;
                addedObject.m_flags = object.getFlags();
            }

            // Connection cmds
            if ( const hkArray<hkUint64>* connections = object.getConnections( frame ) )
            {
                hkLocalArray<hkUint64> connectionStreamIds( connections->getSize() );
                for ( int i = 0; i < connections->getSize(); i++ ) connectionStreamIds.pushBack( hkVdbCmdId::extractStreamId( ( *connections )[i] ) );
                serializer.connect( streamId, connectionStreamIds, object.getTag() );
            }
        }
    }

    // Add object cmds
    if ( addedObjects.getSize() )
    {
        auto processAddObjectBatchFunc = [](
            hkArrayView<hkUint64> ids,
            hkArrayView<hkVdbReflect::Var> datas,
            hkObjectFlags flags,
            int tag,
            void* handle )
        {
            hkServerObjectSerializer* serializer = reinterpret_cast< hkServerObjectSerializer* >( handle );
            serializer->addObjects(
                ids,
                datas,
                flags,
                tag );
        };
        hkVdbObjectHandler::processAddObjectBatches(
            addedObjects,
            processAddObjectBatchFunc,
            &serializer );
    }
}

hkResult hkVdbObjectHandler::processRegisterObjectTypeCmd( const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader )
{
    using namespace hkSerialize;
    using namespace hkVdbCmdUtils;

    // Note: Null check necessary because persistent cached uses this class w/o registering
    if ( m_playbackHandler && m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::REPLAYING ) )
    {
        // Shouldn't need to re-register object types
        return HK_SUCCESS;
    }

#ifdef HK_VDB_USE_HK_SERIALIZE
    // Initialize a new object loader if needed
    if ( !m_objectLoader )
    {
        m_objectLoader = new hkSerialize::Load();
    }
#else
    // Initialize a type loader if needed
    if ( !m_typeLoader )
    {
        m_typeLoader = new hkVdbSerialize::TypeLoad();
    }
#endif

    START_OBJECT_CMD_READ();

    hkUint32 typeId = dataReader.read32u();
#ifdef HK_VDB_USE_HK_SERIALIZE
    const hkReflect::Type* serverType = hkDynCast( m_objectLoader->toVar( dataReader.getStreamReader() ) );
    const hkReflect::Type* clientType = serverType;
#else
    const hkReflect::Type* serverType;
    const hkVdbReflect::Type* clientType = hkVdbSerialize::deserializeType(
        *dataReader.getStreamReader(),
        *m_typeLoader,
        HK_NULL,
        HK_NULL,
        &serverType );
#endif

    END_OBJECT_CMD_READ();

    HK_VDB_VERIFY_CONDITION_MSG(
        clientType,
        0xedb00143,
        hkVdbError::OBJECT_SERIALIZATION_ERROR,
        "Runtime type for \"" << typeId << "\" is not available" );

    HK_VDB_VERIFY_CONDITION( dataReader.isOk(), 0xedb00144, hkVdbError::CMD_HANDLER_READ_ERROR );

    // Update object type registry
    {
        // Get previous info
        TypesInfo prevInfo = m_typeIdToInfo.getWithDefault( typeId, TypesInfo() );

        // Set our new info
        TypesInfo newInfo( typeId, serverType, clientType );
        hkVdbMap<TypeId, const hkReflect::Type*>::Iterator iter = m_typeIdToInfo.findOrInsertKey( typeId, TypesInfo() );
        m_typeIdToInfo.setValue( iter, newInfo );

        // Signal
        hkVdbSignalResult result;
        DO_SIGNAL( m_objectTypeRegistered.fire( typeId, *clientType, result ); )

        // If the type was rejected, then we need to restore prevType
        TypesInfo infoToCleanup = prevInfo;
        if ( result.isFailure() )
        {
            infoToCleanup = newInfo;
            if ( !prevInfo.m_clientType )
            {
                // We had no previous
                m_typeIdToInfo.remove( iter );
            }
            else
            {
                // Restore previous
                m_typeIdToInfo.setValue( iter, prevInfo );
            }
        }
        else
        {
            // Note: prevInfo.m_clientType could be null if this is first registration
            m_clientTypeToInfo.remove( prevInfo.m_clientType );
            HK_ON_DEBUG( hkBool32 newKey = ) m_clientTypeToInfo.insert( newInfo.m_clientType, newInfo );
            HK_ASSERT( 0x22441346, newKey, "This type should not already be registred" );
        }

        // Cleanup evicted type.
        if ( infoToCleanup.m_clientType )
        {
            m_clientTypeToInfo.remove( infoToCleanup.m_clientType );
            hk::DeleteTypeInfo::deleteType( const_cast<hkReflect::Type*>( infoToCleanup.m_clientType ) );
        }
    }

    return HK_SUCCESS;
}

hkResult hkVdbObjectHandler::processAddOrUpdateObjectsCmd( const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader )
{
    using namespace hkVdbCmdType;

    // Note: Null check necessary because persistent cached uses this class w/o registering
    if ( m_playbackHandler && m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::REPLAYING ) )
    {
        // Object state replay is handled by the listening to frame change signals from the hkVdbPlaybackHandler
        return HK_SUCCESS;
    }

    START_OBJECT_CMD_READ();

    // Read objects
    
    
    
    
    
    

    hkObjectFlags flags = 0;
    if ( cmd.getType() == ADD_OBJECTS )
    {
        flags = dataReader.read8u();
    }

    hkInt32 numObjects = dataReader.read32();
    HK_ON_DEBUG( m_totalObjectCount += ( cmd.getType() == ADD_OBJECTS ) * numObjects );
    HK_ON_DEBUG( m_updatedObjectCount += ( cmd.getType() == UPDATE_OBJECTS ) * numObjects );
    hkLocalArray<hkUint64> objectIds( numObjects );
    hkLocalArray<hkVdbReflect::Var> objects( numObjects );
    hkUint32 badTypeId = 0;
    hkUint64 badObjectId = 0;
    for ( int i = 0; i < numObjects; i++ )
    {
        hkUint64 objectId = dataReader.read64u();

#ifdef HK_VDB_USE_HK_SERIALIZE
        hkUint32 typeId = 0;
        hkVdbReflect::Var object = m_objectLoader->toVar( dataReader.getStreamReader() );
#else
        hkUint32 typeId = dataReader.read32u();

        const hkReflect::Type* type = getObjectType( typeId );
        if ( !type )
        {
            if ( !badTypeId )
            {
                badTypeId = typeId;
            }
            continue;
        }

        hkVdbReflect::Var object = hkVdbSerialize::deserializeObject(
            *dataReader.getStreamReader(),
            type );
#endif

        if ( !object )
        {
            if ( !badTypeId && !badObjectId )
            {
                badTypeId = typeId;
                badObjectId = objectId;
            }
            continue;
        }

        objectIds.pushBack( objectId );
        objects.pushBack( object );
    }

    int tag = -1;
    if ( cmd.getType() == ADD_OBJECTS )
    {
        tag = dataReader.read32();
    }

    END_OBJECT_CMD_READ();

    HK_VDB_VERIFY_CONDITION( dataReader.isOk(), 0xedb00145, hkVdbError::CMD_HANDLER_READ_ERROR );

    addOrUpdateObjectsInternal(
        cmd.getType(),
        objectIds,
        objects,
        flags,
        tag );

    HK_VDB_VERIFY_CONDITION_MSG(
        badObjectId == 0,
        0xedb00164,
        hkVdbError::OBJECT_SERIALIZATION_ERROR,
        "Failed to deserialize object with type \"" << badTypeId << "\" and id \"" << badObjectId );

    HK_VDB_VERIFY_CONDITION_MSG(
        badTypeId == 0,
        0xedb00140,
        hkVdbError::OBJECT_TYPE_NOT_REGISTERED,
        "Cannot add/update object with type \"" << badTypeId << "\"; that type hasn't been registered" );

    return HK_SUCCESS;
}

hkResult hkVdbObjectHandler::processRemoveObjectsCmd( int protocol, hkVdbLocalIStream& dataReader )
{
    // Note: Null check necessary because persistent cached uses this class w/o registering
    if ( m_playbackHandler && m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::REPLAYING ) )
    {
        // Object state replay is handled by the listening to frame change signals from the hkVdbPlaybackHandler
        return HK_SUCCESS;
    }

    START_OBJECT_CMD_READ();

    // Read object ids
    hkInt32 numObjects = dataReader.read32();
    HK_ON_DEBUG( m_totalObjectCount -= numObjects );
    hkLocalArray<hkUint64> objectIds( numObjects );
    for ( int i = 0; i < numObjects; i++ )
    {
        hkUint64 objectId = dataReader.read64u();
        objectIds.pushBack( objectId );
    }

    END_OBJECT_CMD_READ();

    HK_VDB_VERIFY_CONDITION( dataReader.isOk(), 0xedb00146, hkVdbError::CMD_HANDLER_READ_ERROR );

    removeObjectsInternal( objectIds );

    return HK_SUCCESS;
}

hkResult hkVdbObjectHandler::processConnectOrDisconnectObjectsCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader )
{
    // Note: Null check necessary because persistent cached uses this class w/o registering
    if ( m_playbackHandler && m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::REPLAYING ) )
    {
        // Object state replay is handled by the listening to frame change signals from the hkVdbPlaybackHandler
        return HK_SUCCESS;
    }

    START_OBJECT_CMD_READ();

    hkUint64 fromId = dataReader.read64u();
    hkInt32 numConnections = dataReader.read32();
    hkLocalArray<hkUint64> toIds( numConnections );
    for ( int i = 0; i < numConnections; i++ )
    {
        hkUint64 toId = dataReader.read64u();
        toIds.pushBack( toId );
    }

    int tag = 0;
    hkConnectivityFlags flags = 0;
    int tagOrFlags = dataReader.read32();
    if ( type == hkVdbCmdType::CONNECT_OBJECTS )
    {
        tag = tagOrFlags;
    }
    else
    {
        flags = hkUint8( tagOrFlags );
    }

    END_OBJECT_CMD_READ();

    HK_VDB_VERIFY_CONDITION( dataReader.isOk(), 0xedb00147, hkVdbError::CMD_HANDLER_READ_ERROR );

    connectOrDisconnectInternal(
        type,
        fromId,
        toIds,
        flags,
        tag );

    return HK_SUCCESS;
}

hkResult hkVdbObjectHandler::addOrUpdateObjectsInternal(
    hkVdbCmdType::Enum type,
    const hkArrayView<ObjectId>& ids,
    const hkArrayView<hkVdbReflect::Var>& objects,
    hkObjectFlags flags,
    int tag )
{
    HK_ASSERT_NO_MSG( 0x0, ids.getSize() == objects.getSize() );

    if ( !m_input )
    {
        HK_ASSERT( 0x22441158, false, "Need to be registered so we can create custom storage in input stream" );
        return HK_FAILURE;
    }

    hkUint32 frame = m_input->getProcessingFrameNumber();

    // Convert to uids
    BREAK_ON_ID( type, ids );
    hkLocalArray<ObjectId> uids( ids.getSize() );
    hkLocalArray<hkVdbCachedObject*> entries( ids.getSize() );
    convertToUids<hkVdbCachedObjectCinfo>(
        m_cache->objects(),
#ifdef STRICT_UPDATE_POLICY
        ( type == hkVdbCmdType::ADD_OBJECTS ) ?
        flags.anyIsSet( hkObjectFlags::TRANSIENT ) ?
            CacheAction::ACCESS_OR_CREATE_TRANSIENT :
            CacheAction::ACCESS_OR_CREATE :
            CacheAction::ACCESS,
#else
        flags.anyIsSet( hkObjectFlags::TRANSIENT ) ?
            CacheAction::ACCESS_OR_CREATE_TRANSIENT :
            CacheAction::ACCESS_OR_CREATE,
#endif
        frame,
        ids,
        uids,
        entries );

    // Update cache
    BREAK_ON_ID( type, uids );
    for ( int i = 0; i < entries.getSize(); i++ )
    {
        if ( hkVdbCachedObject* entry = entries[i] )
        {
            if ( type == hkVdbCmdType::ADD_OBJECTS )
            {
                if ( HK_VERY_LIKELY( entry->getTag() == -1 ) )
                {
                    entry->registerDataAdded( frame, tag, objects[i], flags );
                }
                else
                {
                    
                    
                    
                    
                    
                    
                    HK_WARN_ONCE( 0x22441366, "Some objects are being added twice, "
                        "these are treated as updates and tag and flag information are discarded" );
                    entry->registerDataUpdate( frame, objects[i] );
                    type = hkVdbCmdType::UPDATE_OBJECTS;
                }
            }
            else
            {
                if ( HK_VERY_UNLIKELY( entry->getTag() == -1 ) )
                {
                    HK_WARN_ONCE( 0x22441405, "Some objects are being updated without being added, "
                        "these objects will receive default tag and flags" );
                    entry->registerDataAdded( frame, 0, objects[i], hkObjectFlags::DEFAULT );
                    type = hkVdbCmdType::ADD_OBJECTS;
                }
                else
                {
                    entry->registerDataUpdate( frame, objects[i] );
                }
            }
        }
        else
        {
            HK_ASSERT( 0x22441344, false, "Could not find cached entry for id, this will leak memory" );
        }
    }

    // Signal.
    // If we are caching frames, note we have data.
    // We'll wait till the end and signal an update with current state from object cache.
    if ( m_cachingFrames )
    {
        m_haveDataFromCachingFrames = true;
    }
    else
    {
        hkVdbSignalResult result;
        m_objectsCmd.m_cmdType = type;
        m_objectsCmd.m_ids = uids;
        m_objectsCmd.m_objects = hkArrayViewT::make<hkVdbReflect::Var>( reinterpret_cast< hkVdbReflect::Var* >( objects.begin() ), objects.getSize() );
        m_objectsCmd.m_flags = flags;
        m_objectsCmd.m_tag = tag;
        DO_SIGNAL( m_objectsCmdReceived.fire( m_objectsCmd, result ); )
        
        HK_VDB_VERIFY_SIGNAL_RESULT( result );
    }

    return HK_SUCCESS;
}

hkResult hkVdbObjectHandler::removeObjectsInternal( const hkArrayView<ObjectId>& ids )
{
    if ( !m_input )
    {
        HK_ASSERT( 0x22441159, false, "Need to be registered so we can create custom storage in input stream" );
        return HK_FAILURE;
    }

    hkUint32 frame = m_input->getProcessingFrameNumber();

    // Convert to uids
    BREAK_ON_ID( hkVdbCmdType::REMOVE_OBJECTS, ids );
    hkLocalArray<ObjectId> uids( ids.getSize() );
    hkLocalArray<hkVdbCachedObject*> entries( ids.getSize() );
    convertToUids<hkVdbCachedObjectCinfo>(
        m_cache->objects(),
        CacheAction::ACCESS,
        frame,
        ids,
        uids,
        entries );

    // Update cache
    BREAK_ON_ID( hkVdbCmdType::REMOVE_OBJECTS, uids );
    for ( int i = 0; i < entries.getSize(); i++ )
    {
        if ( hkVdbCachedObject* entry = entries[i] )
        {
            if ( !entry->isAdded( frame ) )
            {
                HK_WARN_ONCE( 0x22441367, "Some objects are being removed twice" );
                return HK_FAILURE;
            }
            entry->registerDataRemoved( frame );
        }
    }

    // Signal.
    // If we are caching frames, note we have data.
    // We'll wait till the end and signal an update with current state from object cache.
    if ( m_cachingFrames )
    {
        m_haveDataFromCachingFrames = true;
    }
    else
    {
        hkVdbSignalResult result;
        m_objectsCmd.m_cmdType = hkVdbCmdType::REMOVE_OBJECTS;
        m_objectsCmd.m_ids = uids;
        DO_SIGNAL( m_objectsCmdReceived.fire( m_objectsCmd, result ); )
        HK_VDB_VERIFY_SIGNAL_RESULT( result );
    }

    return HK_SUCCESS;
}

hkResult hkVdbObjectHandler::connectOrDisconnectInternal(
    hkVdbCmdType::Enum type,
    ConnectionId id,
    const hkArrayView<ConnectionId>& ids,
    hkConnectivityFlags flags,
    int tag )
{
    if ( !m_input )
    {
        HK_ASSERT( 0x22441160, false, "Need to be registered so we can create custom storage in input stream" );
        return HK_FAILURE;
    }

    
    
    
    
    

    hkUint32 frame = m_input->getProcessingFrameNumber();

    // Convert to uids
    BREAK_ON_ID( type, id );
    BREAK_ON_ID( type, ids );
    hkLocalArray<hkUint64> fromUids( ids.getSize() );
    hkLocalArray<hkUint64> selfUid( 1 );
    hkLocalArray<hkVdbCachedObject*> selfEntry( 1 );
    convertToUids<hkVdbCachedObjectCinfo>(
        m_cache->objects(),
        ( type == hkVdbCmdType::CONNECT_OBJECTS ) ? CacheAction::ACCESS_OR_CREATE : CacheAction::ACCESS,
        frame,
        hkArrayViewT::fromSingleObject( id ),
        selfUid,
        selfEntry );
    hkLocalArray<hkUint64> toUids( ids.getSize() );
    hkLocalArray<hkVdbCachedObject*> toEntries( ids.getSize() );
    convertToUids<hkVdbCachedObjectCinfo>(
        m_cache->objects(),
        ( type == hkVdbCmdType::CONNECT_OBJECTS ) ? CacheAction::ACCESS_OR_CREATE : CacheAction::ACCESS,
        frame,
        ids,
        toUids,
        toEntries );

    // Update cache and get resolved set of uids
    BREAK_ON_ID( type, selfUid );
    BREAK_ON_ID( type, toUids );
    hkLocalArray<hkUint64>* fromUidsPtr = &fromUids;
    hkLocalArray<hkUint64>* toUidsPtr = &toUids;
    hkLocalArray<hkUint64> resolvedFromUidsStorage( ids.getSize() );
    hkLocalArray<hkUint64> resolvedToUidsStorage( ids.getSize() );
    if ( hkVdbCachedObject* entry = selfEntry[0] )
    {
        if ( type == hkVdbCmdType::CONNECT_OBJECTS )
        {
            // resolvedToUidsStorage will contain toUids that weren't already connected
            entry->registerConnect( frame, toUids, tag, resolvedToUidsStorage );
            toUidsPtr = &resolvedToUidsStorage;
        }
        else
        {
            if ( toUids.getSize() )
            {
                // resolvedToUidsStorage will contain toUids that weren't already disconnected
                entry->registerDisconnect( frame, toUids, resolvedToUidsStorage );
                toUidsPtr = &resolvedToUidsStorage;
            }
            else
            {
                // resolvedFrom/ToUidsStorage will contain incoming/outgoing connections as appropriate
                // for the value of flags.
                entry->registerDisconnectAll( frame, flags, resolvedFromUidsStorage, resolvedToUidsStorage );
                BREAK_ON_ID( type, resolvedFromUidsStorage );
                BREAK_ON_ID( type, resolvedToUidsStorage );
                fromUidsPtr = &resolvedFromUidsStorage;
                toUidsPtr = &resolvedToUidsStorage;
            }
        }
    }

    // Signal.
    // If we are caching frames, note we have data.
    // We'll wait till the end and signal an update with current state from object cache.
    if ( m_cachingFrames )
    {
        m_haveDataFromCachingFrames = true;
    }
    else
    {
        hkVdbSignalResult result;

        // First, incoming
        for ( int i = 0; i < fromUidsPtr->getSize(); i++ )
        {
            m_connectivityCmd.m_cmdType = type;
            m_connectivityCmd.m_id = ( *fromUidsPtr )[i];
            m_connectivityCmd.m_ids = selfUid;
            m_connectivityCmd.m_tag = tag;
            DO_SIGNAL( m_connectivityCmdReceived.fire( m_connectivityCmd, result ); )
        }

        // Then, outgoing
        if ( toUidsPtr->getSize() )
        {
            m_connectivityCmd.m_cmdType = type;
            m_connectivityCmd.m_id = selfUid[0];
            m_connectivityCmd.m_ids = ( *toUidsPtr );
            m_connectivityCmd.m_tag = tag;
            DO_SIGNAL( m_connectivityCmdReceived.fire( m_connectivityCmd, result ); )
        }

        HK_VDB_VERIFY_SIGNAL_RESULT( result );
    }

    return HK_SUCCESS;
}

void hkVdbObjectHandler::clearTypes()
{
    // Cleanup allocated types.
    for ( hkVdbMap<TypeId, TypesInfo>::Iterator iter = m_typeIdToInfo.getIterator();
        m_typeIdToInfo.isValid( iter );
        iter = m_typeIdToInfo.getNext( iter ) )
    {
        TypesInfo& info = m_typeIdToInfo.getValue( iter );
        hk::DeleteTypeInfo::deleteType( const_cast< hkReflect::Type* >( info.m_clientType ) );
    }
    m_typeIdToInfo.clear();
    m_clientTypeToInfo.clear();
#ifndef HK_VDB_USE_HK_SERIALIZE
    delete m_typeLoader;
    m_typeLoader = HK_NULL;
#endif
}

void hkVdbObjectHandler::clearTransientEntries()
{
    if ( m_transientEntryStreamIds.getSize() )
    {
        // We don't expect transient objects to *for sure* be added/connected.
        // They may have been removed by the user, may never have been connected up, etc.
        // The whole concept is simply a convenience to avoid the necessity of that cleanup.
        hkDisableError disable22441209( 0x22441209 );

        HK_ASSERT(
            0x22441207,
            !m_transientObjectsFrame || ( m_transientObjectsFrame == ( m_input->getProcessingFrameNumber() - 1 ) ),
            "Skipped a transient objects clearing frame" );
        HK_ON_DEBUG( m_transientObjectsClearedFrame = m_transientObjectsFrame );
        HK_ON_DEBUG( m_transientObjectsFrame = 0 );
        removeObjectsInternal( m_transientEntryStreamIds );
        for ( int i = 0; i < m_transientEntryStreamIds.getSize(); i++ )
        {
            hkUint64 id = m_transientEntryStreamIds[i];
            connectOrDisconnectInternal( hkVdbCmdType::DISCONNECT_OBJECTS, id, hkArrayView<ConnectionId>(), hkConnectivityFlags::ALL, 0 );
        }
        m_transientEntryStreamIds.clear();
    }
}

void hkVdbObjectHandler::processAddObjectBatches(
    hkArray<AddObjectInfo>& addObjectInfos,
    void ( processAddObjectBatchFunc )(
        hkArrayView<hkUint64> ids,
        hkArrayView<hkVdbReflect::Var> datas,
        hkObjectFlags flags,
        int tag,
        void* handle ),
    void* handle )
{
    hkAlgorithm::insertionSort( addObjectInfos.begin(), addObjectInfos.getSize() );
    int currentTag = addObjectInfos[0].m_tag;
    hkObjectFlags currentFlags = addObjectInfos[0].m_flags;
    hkLocalArray<hkUint64> currentIds( addObjectInfos.getSize() );
    hkLocalArray<hkVdbReflect::Var> currentObjects( addObjectInfos.getSize() );

    int i = 0;
    while ( true )
    {
        if ( ( i == addObjectInfos.getSize() ) ||
            ( currentTag != addObjectInfos[i].m_tag ) ||
            ( currentFlags != addObjectInfos[i].m_flags ) )
        {
            if ( currentIds.getSize() )
            {
                processAddObjectBatchFunc(
                    currentIds,
                    currentObjects,
                    currentFlags,
                    currentTag,
                    handle
                );
                currentIds.clear();
                currentObjects.clear();
            }

            if ( i < addObjectInfos.getSize() )
            {
                currentTag = addObjectInfos[i].m_tag;
                currentFlags = addObjectInfos[i].m_flags;
            }
            else
            {
                break;
            }
        }

        AddObjectInfo& addObjectInfo = addObjectInfos[i];
        currentIds.pushBack( addObjectInfo.m_id );
        currentObjects.pushBack( addObjectInfo.m_data );
        i++;
    }
}

void hkVdbObjectHandler::updateObjectsToFrame( hkUint32 frame, hkVdbSignalResult& result )
{
    if ( !m_cache || !m_progressReporter )
    {
        HK_VDB_SIGNAL_ERROR( 0xedb00150, hkVdbError::CMD_HANDLER_NOT_REGISTERED );
        result.signalError( *this );
        return;
    }

    // No-op, avoid clearing transients, etc.
    // Note: We don't do anything in response to redoCurrentFrame, so this is an okay early-out.
    //       If we start supporting changes here due to toggling viewers, will need to revisit.
    if ( m_lastPlaybackFrame == frame )
    {
        return;
    }

    // Determine start/end frames
    hkUint32 startFrame = m_lastPlaybackFrame;
    const hkUint32 endFrame = frame;
    m_lastPlaybackFrame = frame;

    // Clear previous transients (if any)
    clearTransientEntries();

    
    
    
    hkLocalArray<AddObjectInfo> addedObjects( m_cache->objects().getNumEntries() / 3 );
    hkLocalArray<hkUint64> updatedObjectIds( m_cache->objects().getNumEntries() / 3 );
    hkLocalArray<hkVdbReflect::Var> updatedObjectDatas( m_cache->objects().getNumEntries() / 3 );
    hkLocalArray<hkUint64> removedObjectIds( m_cache->objects().getNumEntries() / 3 );
    hkLocalArray<hkVdbReflect::Var> removedObjectDatas( m_cache->objects().getNumEntries() / 3 );

    // If we are updating after a long cache, notify of our work.
    // It's likely all this data will be new to any listeners.
    hkUint64 progressId = hkUint64( -1 );
    if ( m_haveDataFromCachingFrames )
    {
        // This will be a high-bar, not all objects will have connections
        progressId = m_progressReporter->startWorkLoad( m_cache->objects().getNumEntries() * 2, 0, false );
        m_progressReporter->setWorkMessage( progressId, "Updating Objects" );
    }

    // Update objects
    hkVdbCachedObjects& objects = m_cache->objects();
    for ( hkVdbCachedObjects::Iterator iter = objects.getIterator();
        objects.isValid( iter );
        iter = objects.getNext( iter ) )
    {
        const hkVdbCachedObject& object = *objects.getValue( iter );
        const hkUint64 id = object.getUid();
        hkReflect::Var data;
        hkReflect::Var prevData;
        const hkArray<hkUint64>* connections = HK_NULL;
        const hkArray<hkUint64>* prevConnections = HK_NULL;

        
        if ( object.isAdded( startFrame ) )
        {
            prevData = object.getData( startFrame );
            prevConnections = object.getConnections( startFrame );
        }
        if ( object.isAdded( endFrame ) )
        {
            data = object.getData( endFrame );
            connections = object.getConnections( endFrame );
        }

        // Object data update
        if ( prevData != data )
        {
            if ( data )
            {
                if ( prevData )
                {
                    updatedObjectIds.pushBack( id );
                    updatedObjectDatas.pushBack( data );
                }
                else
                {
                    AddObjectInfo& addedObject = addedObjects.expandOne();
                    addedObject.init();
                    addedObject.m_tag = object.getTag();
                    addedObject.m_id = id;
                    addedObject.m_data = data;
                    addedObject.m_flags = object.getFlags();
                }
            }
            else
            {
                removedObjectIds.pushBack( id );
                removedObjectDatas.pushBack( data );
            }
        }

        // Object connectivity
        if ( prevConnections != connections )
        {
            m_connectivityCmd.m_cmdType = hkVdbCmdType::CUSTOM; // CUSTOM indicates SetConnectionIdsCmd for asSetConnectionIdsCmd()
            m_connectivityCmd.m_id = id;
            m_connectivityCmd.m_tag = object.getTag();
            
            m_connectivityCmd.m_ids = connections ? ( connections->operator hkArrayView<ConnectionId>() ) : hkArrayView<ConnectionId>();
            DO_SIGNAL( m_connectivityCmdReceived.fire( m_connectivityCmd, result ); )
            if ( progressId != hkUint64( -1 ) ) m_progressReporter->didWorkFor( progressId );

#ifdef PRINT_REPLAY_IDS
            hkStringBuf buf;
            buf.appendPrintf( "[%llx] connected to:\n", id );
            for ( int i = 0; i < m_connectivityCmd.m_ids.getSize(); i++ )
            {
                hkUint64 toId = m_connectivityCmd.m_ids[i];
                buf.appendPrintf( "\t[%llx]\n", toId );
            }
            buf.setLength( hkMath::max2( buf.getLength() - 1, 0 ) );
            HK_REPORT2( 0x0, buf.cString() );
#endif
        }
    }

    
    
    

    // Send signals. Order matters!
    // If we are going from one cached entry to another cached entry that share the *same* streamId,
    // we need to ensure the remove happens first before the add.

    // Removed objects
    {
        m_objectsCmd.m_cmdType = hkVdbCmdType::REMOVE_OBJECTS;
        m_objectsCmd.m_ids = removedObjectIds;
        m_objectsCmd.m_objects = removedObjectDatas;

        if ( m_objectsCmd.m_ids.getSize() )
        {
            DO_SIGNAL( m_objectsCmdReceived.fire( m_objectsCmd, result ); )
            if ( progressId != hkUint64( -1 ) ) m_progressReporter->didWorkFor( progressId );
        }

#ifdef PRINT_REPLAY_IDS
        hkStringBuf buf;
        for ( int i = 0; i < m_objectsCmd.m_ids.getSize(); i++ )
        {
            hkUint64 id = m_objectsCmd.m_ids[i];
            buf.appendPrintf( "[%llx] removed\n", id );
        }
        buf.setLength( hkMath::max2( buf.getLength() - 1, 0 ) );
        HK_REPORT2( 0x0, buf.cString() );
#endif
    }

    // Added objects
    if ( addedObjects.getSize() )
    {
        struct FuncArgs { hkVdbObjectHandler& handler; hkVdbSignalResult& result; hkUint64 progressIdArg; };
        FuncArgs args = { *this, result, progressId };
        auto processAddObjectBatchFunc = [](
            hkArrayView<hkUint64> ids,
            hkArrayView<hkVdbReflect::Var> datas,
            hkObjectFlags flags,
            int tag,
            void* handle )
        {
            FuncArgs* args = reinterpret_cast< FuncArgs* >( handle );
            args->handler.m_objectsCmd.m_cmdType = hkVdbCmdType::ADD_OBJECTS;
            args->handler.m_objectsCmd.m_ids = ids;
            args->handler.m_objectsCmd.m_objects = datas;
            args->handler.m_objectsCmd.m_flags = flags;
            args->handler.m_objectsCmd.m_tag = tag;
            DO_SIGNAL( args->handler.m_objectsCmdReceived.fire( args->handler.m_objectsCmd, args->result ); )
            if ( args->progressIdArg != hkUint64( -1 ) ) args->handler.m_progressReporter->didWorkFor( args->progressIdArg );

#ifdef PRINT_REPLAY_IDS
            hkStringBuf buf;
            for ( int i = 0; i < handler->m_objectsCmd.m_ids.getSize(); i++ )
            {
                hkUint64 id = handler->m_objectsCmd.m_ids[i];
                buf.appendPrintf( "[%llx] added with tag [%x] and flags [%x]\n", id, handler->m_objectsCmd.m_tag, handler->m_objectsCmd.m_flags );
            }
            buf.setLength( hkMath::max2( buf.getLength() - 1, 0 ) );
            HK_REPORT2( 0x0, buf.cString() );
#endif
        };
        processAddObjectBatches(
            addedObjects,
            processAddObjectBatchFunc,
            &args );
    }

    // Updated objects
    {
        m_objectsCmd.m_cmdType = hkVdbCmdType::UPDATE_OBJECTS;
        m_objectsCmd.m_ids = updatedObjectIds;
        m_objectsCmd.m_objects = updatedObjectDatas;

        if ( m_objectsCmd.m_ids.getSize() )
        {
            DO_SIGNAL( m_objectsCmdReceived.fire( m_objectsCmd, result ); )
            if ( progressId != hkUint64( -1 ) ) m_progressReporter->didWorkFor( progressId );
        }

#ifdef PRINT_REPLAY_IDS
        hkStringBuf buf;
        for ( int i = 0; i < m_objectsCmd.m_ids.getSize(); i++ )
        {
            hkUint64 id = m_objectsCmd.m_ids[i];
            buf.appendPrintf( "[%llx] updated\n", id );
        }
        buf.setLength( hkMath::max2( buf.getLength() - 1, 0 ) );
        HK_REPORT2( 0x0, buf.cString() );
#endif
    }

    if ( progressId != hkUint64( -1 ) ) m_progressReporter->endWorkLoad( progressId );
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
