// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbServices/hkVdbServices.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbInternalHandler.h>

#include <Common/Visualize/Serialize/hkPersistableReadFormat.h>
#include <Common/Visualize/hkVisualDebuggerCmd.h>

#include <VisualDebugger/VdbServices/hkVdbClient.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdDispatcher.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbStatsHandler.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbPlaybackHandler.h>

#define DEBUG_LOG_IDENTIFIER "vdb.Cmd.Handler.Internal"
#include <Common/Base/System/Log/hkLog.hxx>

using namespace hkVdbCmdUtils;

namespace
{
    hkResult handleProcessInternalCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
        hkVdbInternalHandler* handler = reinterpret_cast< hkVdbInternalHandler* >( userHandle );
        return handler->processInternalCmd( frame, cmd, protocol, dataReader, processInfo );
    }
};

hkVdbInternalHandler::hkVdbInternalHandler() :
    hkVdbCmdHandler<hkVdbCmdHandlerType::INVALID>( &s_debugLog )
{}

hkResult hkVdbInternalHandler::registerSelf( hkVdbClient& client )
{
    using namespace hkVdbCmdType;

    bool succeeded = true;
    hkVdbCmdDispatcher& dispatcher = client.getCmdDispatcher();

    succeeded &= ( dispatcher.registerHandler( CLIENT_INTERNAL, handleProcessInternalCmdFunc, this, this ).isSuccess() );

    m_dispatcher = &dispatcher;
    m_statsHandler = client.getCmdHandler<hkVdbStatsHandler>();
    m_playbackHandler = client.getCmdHandler<hkVdbPlaybackHandler>();

    HK_VDB_VERIFY_REPORTER_CONDITION( succeeded, dispatcher, hkVdbError::CMD_HANDLER_REGISTRATION_ERROR );

    return HK_SUCCESS;
}

hkResult hkVdbInternalHandler::unregisterSelf( hkVdbClient& client )
{
    using namespace hkVdbCmdType;

    bool succeeded = true;
    hkVdbCmdDispatcher& dispatcher = client.getCmdDispatcher();

    succeeded &= ( dispatcher.unregisterHandler( CLIENT_INTERNAL ).isSuccess() );

    m_dispatcher = HK_NULL;
    m_statsHandler = HK_NULL;
    m_playbackHandler = HK_NULL;

    HK_VDB_VERIFY_REPORTER_CONDITION( succeeded, dispatcher, hkVdbError::CMD_HANDLER_REGISTRATION_ERROR );

    return HK_SUCCESS;
}

void hkVdbInternalHandler::onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection ) {}

hkResult hkVdbInternalHandler::writeDispatchCmd( hkStreamWriter& writer, const hkVdbCmd& cmd, int protocol )
{
    using namespace hkVdbCmdType;
    const int serverType = patchType<Direction::CLIENT_TO_SERVER>( protocol, CLIENT_INTERNAL );
    const int packetSize = s8 + s8 + s32 + cmd.getCmdSize();
    hkVdbOStream ostream( &writer );
    ostream.write32u( packetSize );
    ostream.write8u( hkUint8( serverType ) );
    ostream.write8u( hkVdbInternalCmdType::DISPATCH_CMD );
    ostream.write32( hkVersionReporter::m_protocolVersion );
    ostream.writeRaw( &cmd, cmd.getCmdSize() );
    return HK_SUCCESS;
}

hkResult hkVdbInternalHandler::readDispatchCmd( const hkVdbCmd& cmd, int protocol, hkArray<hkInt8>& cmdBufferOut, int& protocolOut )
{
    hkVdbLocalIStream reader( cmd.getData(), cmd.getDataSize() );
    if ( skipAndVerifyHeader( cmd, protocol, reader, hkVdbInternalCmdType::DISPATCH_CMD ).isFailure() ) return HK_FAILURE;

    // Get the dispatch protocol
    protocolOut = reader.read32();

    // Read our cmd into a buffer with space for meta-data
    const int dispatchCmdOffset = s8 /*internal cmd type*/ + s32 /*dispatchProtocol we just read*/;
    const hkVdbCmd* cmdToDispatchSrc = reinterpret_cast< const hkVdbCmd* >( cmd.getData( dispatchCmdOffset ) );
    const int dispatchCmdSizeWithMeta = ( cmdToDispatchSrc->getCmdSize() + hkVdbCmd::getMetaDataSize() );
    cmdBufferOut.setSize( dispatchCmdSizeWithMeta );
    reader.readRaw( cmdBufferOut.begin(), cmdToDispatchSrc->getCmdSize() );

    // Patch the type
    hkVdbCmd* cmdToDispatch = reinterpret_cast< hkVdbCmd* >( cmdBufferOut.begin() );
    const hkUint8 serverType = cmdToDispatch->getServerType();
    const hkUint8 type = hkUint8( hkVdbCmdType::patchType<hkVdbCmdType::Direction::SERVER_TO_CLIENT>( protocolOut, serverType ) );
    cmdToDispatch->setType( hkVdbCmdType::Enum( type ) );

    // Set index to the same as source cmd (we don't exist in the input buffer)
    cmdToDispatch->setIndex( cmd.getIndex() );
    return HK_SUCCESS;
}

hkResult hkVdbInternalHandler::writeDebugTagCmd( hkStreamWriter& writer, int protocol, const char* tag )
{
    using namespace hkVdbCmdType;
    const int serverType = patchType<Direction::CLIENT_TO_SERVER>( protocol, CLIENT_INTERNAL );
    const int numCharacters = ( hkString::strLen( tag ) + 1 ); // include null terminator
    const int packetSize = s8 + s8 + s8 + ( s8 * numCharacters );
    hkVdbOStream ostream( &writer );
    ostream.write32u( packetSize );
    ostream.write8u( hkUint8( serverType ) );
    ostream.write8u( hkVdbInternalCmdType::DEBUG_TAG );
    ostream.write8( '>' );
    ostream.writeString( tag );
    return HK_SUCCESS;
}

hkResult hkVdbInternalHandler::readDebugTagCmd( const hkVdbCmd& cmd, int protocol, hkStringBuf& bufOut )
{
    hkVdbLocalIStream reader( cmd.getData(), cmd.getDataSize() );
    if( skipAndVerifyHeader( cmd, protocol, reader, hkVdbInternalCmdType::DEBUG_TAG ).isFailure()) return HK_FAILURE;
    reader.readString( bufOut );
    return HK_SUCCESS;
}

hkResult hkVdbInternalHandler::inplaceModifyStreamDebugTag( hkVdbCmd& cmd, int protocol )
{
    hkVdbLocalIStream reader( cmd.getData(), cmd.getDataSize() );
    if( skipAndVerifyHeader( cmd, protocol, reader, hkVdbInternalCmdType::DEBUG_TAG ).isFailure() ) return HK_FAILURE;
    const char* tag = reinterpret_cast< const char* >( cmd.getData( reader.tell() ) );
    if ( ( hkString::strLen( tag ) > 0 ) && ( tag[0] == '>' ) )
    {
        const_cast<char*>( tag )[0] = '<';
        return HK_SUCCESS;
    }
    else
    {
        return HK_FAILURE;
    }
}

hkResult hkVdbInternalHandler::writeUnpersistReadFormatCmd( hkStreamWriter& writer, int protocol, const hkArray<hkInt8>& persistentState, hkVdbCmdHandlerType::Enum handlerType )
{
    using namespace hkVdbCmdType;
    if ( persistentState.getSize() )
    {
        const int serverType = patchType<Direction::CLIENT_TO_SERVER>( protocol, CLIENT_INTERNAL );
        const int packetSize = s8 + s8 + s8 + persistentState.getSize();
        hkVdbOStream ostream( &writer );
        ostream.write32u( packetSize );
        ostream.write8u( hkUint8( serverType ) );
        ostream.write8u( hkVdbInternalCmdType::UNPERSIST_READ_FORMAT );
        ostream.write8u( hkUint8( handlerType ) );
        ostream.writeRaw( persistentState.begin(), persistentState.getSize() );
    }
    return HK_SUCCESS;
}

hkResult hkVdbInternalHandler::readUnpersistReadFormatCmd( const hkVdbCmd& cmd, int protocol, hkArray<hkInt8>& persistentStateOut, hkVdbCmdHandlerType::Enum& handlerTypeOut )
{
    hkVdbLocalIStream reader( cmd.getData(), cmd.getDataSize() );
    if ( skipAndVerifyHeader( cmd, protocol, reader, hkVdbInternalCmdType::UNPERSIST_READ_FORMAT ).isFailure() ) return HK_FAILURE;
    handlerTypeOut = hkVdbCmdHandlerType::Enum( reader.read8u() );
    const int internalDataSize = ( cmd.getDataSize() - reader.getStreamReader()->isSeekTellSupported()->tell() );
    persistentStateOut.setSize( internalDataSize );
    reader.readRaw( persistentStateOut.begin(), internalDataSize );
    return HK_SUCCESS;
}


// Update the below code
HK_COMPILE_TIME_ASSERT( hkVdbCmd::META_DATA_SIZE == ( s8 + s32 ) );
hkResult hkVdbInternalHandler::writeMetaData( hkStreamWriter& writer, hkInt16 cmdIdx )
{
    using namespace hkVdbCmdType;
    hkVdbOStream ostream( &writer );
    ostream.write8u( CLIENT_INTERNAL );
    ostream.write32u( cmdIdx );
    return HK_SUCCESS;
}

hkResult hkVdbInternalHandler::processInternalCmd(
    const hkVdbFrame& frame,
    const hkVdbCmd& cmd,
    int protocol,
    hkVdbLocalIStream& dataReader,
    hkVdbProcessInfo* processInfo )
{
    HK_ASSERT_NO_MSG( 0x22440869, cmd.getType() == hkVdbCmdType::CLIENT_INTERNAL );
    const hkVdbInternalCmdType::Enum internalType = hkVdbInternalCmdType::Enum( dataReader.read8u() );

    switch ( internalType )
    {
        case hkVdbInternalCmdType::UNPERSIST_READ_FORMAT:
        {
            // Note: Should not redo these cmds during replay
            
            if ( m_playbackHandler && m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::REPLAYING ) ) return HK_SUCCESS;

            hkVdbCmdHandlerType::Enum handlerType;
            hkArray<hkInt8> persistentState;
            HK_VDB_VERIFY_OPERATION_MSG(
                readUnpersistReadFormatCmd( cmd, protocol, persistentState, handlerType ),
                0xedb00113,
                hkVdbError::CMD_HANDLER_READ_ERROR,
                "Failed to read UNPERSIST_READ_FORMAT cmd" );

            switch ( handlerType )
            {
                case hkVdbCmdHandlerType::STATS:
                {
                    if ( !m_statsHandler ) return HK_FAILURE;
                    hkRefPtr<hkPersistableReadFormat> persistableReadFormat = hkRefNew<hkPersistableReadFormat>( new hkPersistableReadFormat() );
                    persistableReadFormat->unpersist( persistentState, false );
                    m_statsHandler->setLoadedReader( persistableReadFormat );
                    break;
                }
                default:
                {
                    HK_ASSERT( 0x22440919, false, "Do not recognize read format command handler target" );
                    return HK_FAILURE;
                }
            }
            break;
        }
        case hkVdbInternalCmdType::DEBUG_TAG:
        {
            // no-op, just for debug strings in the movie file
            return HK_SUCCESS;
        }
        case hkVdbInternalCmdType::DISPATCH_CMD:
        {
            // Read the cmd
            int dispatchProtocol;
            hkArray<hkInt8> dispatchCmdBuffer;
            HK_VDB_VERIFY_OPERATION_MSG(
                hkVdbInternalHandler::readDispatchCmd( cmd, protocol, dispatchCmdBuffer, dispatchProtocol ),
                0xedb00139,
                hkVdbError::CMD_HANDLER_READ_ERROR,
                "Internal dispatch cmd read failure" );

            // Dispatch it
            const hkVdbCmd* dispatchCmd = reinterpret_cast< hkVdbCmd* >( dispatchCmdBuffer.begin() );
            HK_ASSERT( 0x22441132, dispatchCmd->isOk(), "Cmd is not ok" );
            m_dispatcher->dispatch( frame, *dispatchCmd, dispatchProtocol, processInfo );
            break;
        }
        case hkVdbInternalCmdType::CUSTOM:
        {
            
            break;
        }
        default:
        {
            HK_ASSERT( 0x22440870, false, "Do not recognize internal command" );
            return HK_FAILURE;
        }

        //////////////////////////////////////////////////////////////////////////
        // Deprecated commands
        //////////////////////////////////////////////////////////////////////////
        case hkVdbInternalCmdType::LOAD_VERSIONED_STAT_MAPS:
        {
            // Note: Should not redo these cmds during replay
            
            if ( m_playbackHandler && m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::REPLAYING ) ) return HK_SUCCESS;
            HK_ASSERT( 0x22440918, protocol < hkProtocolVersion::VDB_2017_1, "This internal command is deprecated" );
            if ( !m_statsHandler ) return HK_FAILURE;
            const int internalDataSize = ( cmd.getDataSize() - s8 );
            hkArray<hkInt8> statsMaps( internalDataSize );
            dataReader.readRaw( statsMaps.begin(), internalDataSize );
            if ( m_statsHandler->loadStatsMaps( statsMaps ).isFailure() ) return HK_FAILURE;
            break;
        }
    }

    HK_VDB_VERIFY_CONDITION_MSG( dataReader.isOk(), 0xedb00098, hkVdbError::CMD_HANDLER_READ_ERROR, "Corruption after internal cmd read" );

    return HK_SUCCESS;
}

hkResult hkVdbInternalHandler::skipAndVerifyHeader(
    const hkVdbCmd& cmd,
    int protocol,
    hkVdbLocalIStream& reader,
    hkVdbInternalCmdType::Enum expectedInternalType )
{
    if ( cmd.getType() != hkVdbCmdType::CLIENT_INTERNAL ) return HK_FAILURE;
    const hkVdbInternalCmdType::Enum internalType = hkVdbInternalCmdType::Enum( reader.read8u() );
    return ( internalType == expectedInternalType ) ? HK_SUCCESS : HK_FAILURE;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
