// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbServices/hkVdbServices.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbFileHandler.h>

#include <VisualDebugger/VdbServices/hkVdbClient.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdDispatcher.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbPlaybackHandler.h>

#define DEBUG_LOG_IDENTIFIER "vdb.Cmd.Handler.File"
#include <Common/Base/System/Log/hkLog.hxx>

namespace
{
    hkResult handleFileInfoCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
        hkVdbFileHandler* handler = reinterpret_cast< hkVdbFileHandler* >( userHandle );
        return handler->processFileInfoCmd( cmd, protocol, dataReader );
    }

    hkResult handleDeprecatedSnapshotCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
        hkVdbFileHandler* handler = reinterpret_cast< hkVdbFileHandler* >( userHandle );
        return handler->processDeprecatedSnapshotCmd( cmd, protocol, dataReader );
    }
};

hkVdbFileHandler::hkVdbFileHandler() :
    hkVdbCmdHandler<hkVdbCmdHandlerType::FILE>( &s_debugLog )
{}

hkResult hkVdbFileHandler::registerSelf( hkVdbClient& client )
{
    using namespace hkVdbCmdType;

    bool succeeded = true;
    hkVdbCmdDispatcher& dispatcher = client.getCmdDispatcher();

    succeeded &= ( dispatcher.registerHandler( SEND_FILE, handleFileInfoCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( DEPRECATED_WORLD_SNAPSHOT, handleDeprecatedSnapshotCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( DEPRECATED_MEMORY_SNAPSHOT, handleDeprecatedSnapshotCmdFunc, this, this ).isSuccess() );

    m_playbackHandler = client.getCmdHandler<hkVdbPlaybackHandler>();

    HK_VDB_VERIFY_REPORTER_CONDITION( succeeded, dispatcher, hkVdbError::CMD_HANDLER_REGISTRATION_ERROR );

    return HK_SUCCESS;
}

hkResult hkVdbFileHandler::unregisterSelf( hkVdbClient& client )
{
    using namespace hkVdbCmdType;

    bool succeeded = true;
    hkVdbCmdDispatcher& dispatcher = client.getCmdDispatcher();

    succeeded &= ( dispatcher.unregisterHandler( SEND_FILE ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( DEPRECATED_WORLD_SNAPSHOT ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( DEPRECATED_MEMORY_SNAPSHOT ).isSuccess() );

    m_playbackHandler = HK_NULL;

    HK_VDB_VERIFY_REPORTER_CONDITION( succeeded, dispatcher, hkVdbError::CMD_HANDLER_REGISTRATION_ERROR );

    return HK_SUCCESS;
}

void hkVdbFileHandler::onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection ) {}

hkResult hkVdbFileHandler::processFileInfoCmd( const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader )
{
    using namespace hkVdbCmdType;

    
    
    if ( m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::REPLAYING | hkVdbPlaybackFlags::CACHING ) )
    {
        // During replay, ignore file send commands
        return HK_SUCCESS;
    }

    hkSeekableStreamReader* reader = dataReader.getStreamReader()->isSeekTellSupported();
    HK_VDB_VERIFY_CONDITION( reader, 0xedb00080, hkVdbError::CMD_HANDLER_READ_ERROR );

    //
    // Read file info first
    //

    const int startInfo = reader->tell();

    hkStringBuf typeNameStringStorage;
    dataReader.readString( typeNameStringStorage );
    m_fileInfo.m_typeName = typeNameStringStorage;

    hkStringBuf extensionsStringStorage;
    {
        m_fileInfo.m_extensions.clear();
        dataReader.readString( extensionsStringStorage );
        char* extension = const_cast<char*>( extensionsStringStorage.cString() );
        do
        {
            // Consume spaces and '.'s
            while (
                ( extension != HK_NULL ) &&
                ( ( *extension == ' ' ) || ( *extension == '.' ) ) )
            {
                extension++;
            }

            if ( extension != HK_NULL )
            {
                // Push back the extension
                m_fileInfo.m_extensions.pushBack( extension );

                // Finish this string and find the next one
                extension = const_cast< char* >( hkString::strStr( extension, "," ) );
                if ( extension )
                {
                    *extension = '\0';
                    extension++;
                }
            }
        } while ( extension != HK_NULL );

        m_fileInfo.m_tag = dataReader.read32();
    }

    const hkUint32 fileDataSize = dataReader.read32u();

    const int endInfo = reader->tell();
    const int fileInfoSize = ( endInfo - startInfo );
    HK_ASSERT( 0x22441001, ( hkInt64( fileInfoSize ) + fileDataSize ) == cmd.getDataSize(), "File size computation mismatch" );

    //
    // Reinterpret end of cmd data block as the data to avoid another copy.
    //
    
    
    

    m_fileInfo.m_data.setDataUserFree(
        // Point to raw cmd data so we avoid a copy
        reinterpret_cast< hkInt8* >( const_cast< void* >( cmd.getData() ) ) + fileInfoSize,
        fileDataSize,
        fileDataSize );

    HK_VDB_VERIFY_CONDITION( dataReader.isOk(), 0xedb00054, hkVdbError::CMD_HANDLER_READ_ERROR );

    // Signal
    
    
    hkVdbSignalResult result;
    m_fileInfoReceived.fire( m_fileInfo, result );
    HK_VDB_VERIFY_SIGNAL_RESULT( result );

    return HK_SUCCESS;
}

hkResult hkVdbFileHandler::processDeprecatedSnapshotCmd( const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader )
{
    using namespace hkVdbCmdType;

    
    
    if ( m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::REPLAYING | hkVdbPlaybackFlags::CACHING ) )
    {
        // During replay, ignore file send commands
        return HK_SUCCESS;
    }

    const hkVdbCmdType::Enum type = cmd.getType();
    switch ( type )
    {
        case DEPRECATED_WORLD_SNAPSHOT:
        {
            // Copy/pasted from old vdb (ish)
            {
                /*hkUint8 bytesInPointer = */dataReader.read8u();
                hkUint8 isXml = dataReader.read8u();

                // These two values are unused, keep reading them for backwards compatibility.
                /*hkUint8 reusePaddingOptimization = */dataReader.read8u();
                /*hkUint8 emptyBaseClassOptimization = */dataReader.read8u();

                // Snapshot itself
                int snapShotSize = dataReader.read32();
                if ( snapShotSize > 0 )
                {
                    m_fileInfo.m_tag = 0;
                    m_fileInfo.m_typeName = "Havok World Snapshot";
                    m_fileInfo.m_extensions.clear();
                    m_fileInfo.m_extensions.pushBack( isXml && ( protocol > 10100 ) ? "xml" : "hkt" );
                    m_fileInfo.m_data.clearAndDeallocate(); // ensure we stop pointing to previous
                    m_fileInfo.m_data.setSize( snapShotSize );
                    dataReader.readRaw( m_fileInfo.m_data.begin(), snapShotSize );
                    HK_VDB_VERIFY_CONDITION( dataReader.isOk(), 0xedb00135, hkVdbError::CMD_HANDLER_READ_ERROR );

                    // Signal
                    hkVdbSignalResult result;
                    m_fileInfoReceived.fire( m_fileInfo, result );
                    HK_VDB_VERIFY_SIGNAL_RESULT( result );
                }
            }
            break;
        }
        case DEPRECATED_MEMORY_SNAPSHOT:
        {
            // Old hkVdbCommandWriter header stuff
            
            
            hkUint8 isFinalPacket = dataReader.read8u();
            hkUint32 contentSize = ( cmd.getDataSize() - sizeof( isFinalPacket ) );

            // If this is our first packet, init
            if ( m_fileInfo.m_tag != -2 )
            {
                m_fileInfo.m_data.clearAndDeallocate(); // ensure we stop pointing to previous
                m_fileInfo.m_tag = -2;
                m_fileInfo.m_typeName = "Havok Memory Snapshot";
                m_fileInfo.m_extensions.clear();
                m_fileInfo.m_extensions.pushBack( "hkmem" );
            }

            // Get data from this packet
            hkInt8* dest = m_fileInfo.m_data.expandBy( contentSize );
            dataReader.readRaw( dest, contentSize );
            HK_VDB_VERIFY_CONDITION( dataReader.isOk(), 0xedb00135, hkVdbError::CMD_HANDLER_READ_ERROR );

            // Signal
            if ( isFinalPacket )
            {
                hkVdbSignalResult result;
                m_fileInfoReceived.fire( m_fileInfo, result );
                m_fileInfo.m_tag = 0;
                HK_VDB_VERIFY_SIGNAL_RESULT( result );
            }
            break;
        }
        default:
        {
            HK_VDB_SIGNAL_ERROR_MSG( 0xedb00134, hkVdbError::CMD_UNKNOWN, "Unknown snapshot cmd" );
            return HK_FAILURE;
        }
    }

    return HK_SUCCESS;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
