// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Types/Color/hkColor.h>
#include <Common/Base/Types/Geometry/hkGeometry.h>

#include <Common/Visualize/hkVisualDebuggerCmdType.h>
#include <Common/Visualize/hkDebugDisplayHandler.h>
#include <Common/Visualize/Shape/hkDisplayGeometry.h>
#include <Common/Visualize/Shape/hkDisplayGeometryFlags.h>

#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbCmdHandler.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbPlaybackHandler.h>
#include <VisualDebugger/VdbServices/System/Cache/hkVdbCache.h>

class hkVdbCmdOutput;
class hkVdbCmdInput;
class hkVdbPlaybackHandler;
class hkVdbProcessHandler;

//////////////////////////////////////////////////////////////////////////
/// A handler for display-related cmds.
//////////////////////////////////////////////////////////////////////////
class HK_EXPORT_VDB hkVdbDisplayHandler : public hkVdbCmdHandler<hkVdbCmdHandlerType::DISPLAY>
{
public:
    inline operator hkReferencedObject*( ) { return static_cast< hkReferencedObject* >( static_cast< hkVdbCmdHandler<hkVdbCmdHandlerType::DISPLAY>* >( this ) ); }
    inline operator const hkReferencedObject*( ) const { return static_cast< const hkReferencedObject* >( static_cast< const hkVdbCmdHandler<hkVdbCmdHandlerType::DISPLAY>* >( this ) ); }
    HK_DECLARE_CLASS( hkVdbDisplayHandler, New );
    hkVdbDisplayHandler();

    //
    // hkVdbCmdHandler interface
    //

    virtual hkResult waitForCompletion() HK_OVERRIDE;
    virtual hkResult registerSelf( hkVdbClient& client ) HK_OVERRIDE;
    virtual hkResult unregisterSelf( hkVdbClient& client ) HK_OVERRIDE;
    virtual void onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection ) HK_OVERRIDE;

    //
    // Control and Query
    //

    
    /// Get debug display options or null if it has not been set.
    /// These options can be set by the server or client, but are cleared on new connections.
    HK_INLINE const hkDebugDisplayHandler::Options* getDisplayOptions() const { return m_haveDisplayOptionsBeenSet ? &m_displayOptions : HK_NULL; }

    /// Set debug display options.
    hkResult setDisplayOptions( const hkDebugDisplayHandler::Options& options );
    

    //
    // Signals
    //

    
    
    
    struct AddGeometryCmd;
    struct InstanceGeometryCmd;
    struct UpdateGeometryCmd;
    struct RemoveGeometryCmd;
    struct DisposeGeometryCmd;

    /// Base class for all geometry-related cmd data.
    struct GeometryCmd
    {
        HK_DECLARE_CLASS( GeometryCmd, New );
        GeometryCmd() : m_cmdType( hkVdbCmdType::INVALID ) {}

        /// The cmd type.
        hkVdbCmdType::Enum m_cmdType;

        /// The id associated with the geometry.
        hkUint64 m_id;

        /// Returns the cmd as an add geometry cmd if that is the type; null otherwise.
        HK_INLINE const AddGeometryCmd* asAddCmd() const;

        /// Returns the cmd as an instance geometry cmd if that is the type; null otherwise.
        HK_INLINE const InstanceGeometryCmd* asInstanceCmd() const;

        /// Returns the cmd as an update geometry cmd if that is the type; null otherwise.
        HK_INLINE const UpdateGeometryCmd* asUpdateCmd() const;

        /// Returns the cmd as a remove geometry cmd if that is the type; null otherwise.
        HK_INLINE const RemoveGeometryCmd* asRemoveCmd() const;

        /// Returns the cmd as a dispose geometry cmd if that is the type; null otherwise.
        HK_INLINE const DisposeGeometryCmd* asDisposeCmd() const;
    };

    /// New geometry has been added.
    struct AddGeometryCmd : public GeometryCmd
    {
        HK_DECLARE_CLASS( AddGeometryCmd, New );

        /// The tag (usually from process/viewer) associated with the geometry.
        hkInt32 m_tag;

        /// The added geometry data.
        hkArray<hkDisplayGeometry*> m_geometry;

        /// The initial geometry transform.
        hkMatrix4 m_transform;

        /// Extended information.
        struct Ex
        {
            HK_DECLARE_CLASS( Ex, New );

            /// The desired color for the geometry.
            hkColor::Argb m_color;

            /// Geometry flags that determine behavior and give optimization hints.
            hkDisplayGeometryFlags m_flags;
        };

        /// Gets extended information if available.
        HK_INLINE const Ex* getExtended() const;

    protected:
        Ex m_extended;
        friend class hkVdbDisplayHandler;
    };

    /// Geometry has been instanced.
    struct InstanceGeometryCmd : public GeometryCmd
    {
        HK_DECLARE_CLASS( InstanceGeometryCmd, New );
        InstanceGeometryCmd() { m_cmdType = hkVdbCmdType::INSTANCE_GEOMETRY; }

        /// The tag (usually from process/viewer) associated with the geometry.
        hkInt32 m_tag;

        /// The id for the source geometry to be instanced.
        hkUint64 m_sourceGeometryId;

        /// The initial geometry transform.
        hkMatrix4 m_transform;
    };

    /// Geometry has had some properties updated.
    /// This cmd can occur even on removed geometries (replay is achieved by
    /// changing visibility of removed geometries).
    /// Note: What can be updated is often restricted by hkDisplayGeometryFlags.
    struct UpdateGeometryCmd : public GeometryCmd
    {
        HK_DECLARE_CLASS( UpdateGeometryCmd, New );
        UpdateGeometryCmd() { m_cmdType = hkVdbCmdType::INVALID; }

        /// Returns a transform if updated; null otherwise.
        HK_INLINE const hkMatrix4* getTransform() const;

        /// Returns a color if updated; null otherwise.
        HK_INLINE const hkColor::Argb* getColor() const;

        /// Returns a transparency alpha if updated; null otherwise.
        HK_INLINE const hkReal* getAlpha() const;

        /// Returns an array of vertices if updated; null otherwise.
        HK_INLINE const hkArray<hkVector4>* getVerts() const;

        /// Returns an array of particle transforms if updated; null otherwise.
        HK_INLINE const hkArray<hkTransform>* getParticleTransforms() const;

        /// Returns display flag bits to set if updated; null otherwise.
        HK_INLINE const hkDisplayGeometryFlags* getSetFlagBits() const;

        /// Returns display flag bits to clear if updated; null otherwise.
        HK_INLINE const hkDisplayGeometryFlags* getClearFlagBits() const;

        /// Returns particle system update id if updated; null otherwise.
        HK_INLINE const int* getParticleUpdateId() const;

    protected:
        UpdateGeometryCmd( const UpdateGeometryCmd& other) {}
        void operator=( const UpdateGeometryCmd& other ) {}
        enum UpdateBits
        {
            TRANSFORM = ( 1 << 0 ),
            COLOR = ( 1 << 1 ),
            ALPHA = ( 1 << 2 ),
            VERTS = ( 1 << 3 ),
            PARTICLE_TRANSFORMS = ( 1 << 4 ),
            SET_FLAGS = ( 1 << 5 ),
            CLEAR_FLAGS = ( 1 << 6 ),
            // This should stay at end
            SENTINEL = ( 1 << 7 )
        };
        hkFlags<UpdateBits, hkUint8> m_updateBits;
        hkArray<hkVector4> m_newVertices;
        hkMatrix4 m_transform;
        hkColor::Argb m_color;
        hkReal m_alpha;
        hkDisplayGeometryFlags m_setFlagBits;
        hkDisplayGeometryFlags m_clearFlagBits;
        hkArray<hkTransform> m_newParticleTransforms;
        int m_particleUpdateId;
        friend class hkVdbDisplayHandler;
    };

    /// Geometry has been removed (the server has released it).
    /// Listeners should note that the system will still update
    /// properties on a removed object. This is how the replay
    /// system works for this type of cmd. When this signal is
    /// received, the implementation should hide the object but
    /// keep the resources around. During replay, visibility
    /// flags are set and cleared as needed.
    /// The DisposeGeometryCmd will occur when the framework will
    /// no longer reference the geometry.
    struct RemoveGeometryCmd : public GeometryCmd
    {
        HK_DECLARE_CLASS( RemoveGeometryCmd, New );
        RemoveGeometryCmd() { m_cmdType = hkVdbCmdType::REMOVE_GEOMETRY; }
    };

    /// Geometry has been disposed (is no longer in input buffer).
    /// The framework will no longer reference this geometry; it
    /// is safe to free resources.
    struct DisposeGeometryCmd : public GeometryCmd
    {
        HK_DECLARE_CLASS( DisposeGeometryCmd, New );
        DisposeGeometryCmd() { m_cmdType = hkVdbCmdType::CLIENT_INTERNAL; }
    };

    /// For advanced use.
    /// Storage and helper class for any GeometryCmd.
    /// Can be used to create a uniform storage container (like an array).
    struct AnyGeometryCmd : public GeometryCmd
    {
        AnyGeometryCmd() { m_cmdType = hkVdbCmdType::INVALID; }
        ~AnyGeometryCmd();
        AnyGeometryCmd& operator =( const GeometryCmd& cmd );
    private:
        HK_ALIGN16( char m_pad[sizeof( UpdateGeometryCmd )] );
    };

    struct DisplayGeometryCmd;
    struct Display1PtCmd;
    struct Display2PtCmd;
    struct Display3PtCmd;
    struct DisplayTextCmd;
    struct Display3dTextCmd;

    /// Base class for all immediate-mode-display-related cmd data.
    /// This data should only be drawn for a single frame.
    struct DisplayCmd
    {
        HK_DECLARE_CLASS( DisplayCmd, New );
        DisplayCmd() : m_cmdType( hkVdbCmdType::INVALID ) {}

        /// The cmd type.
        hkVdbCmdType::Enum m_cmdType;

        /// The id associated with the display data.
        hkUint64 m_id;

        /// The tag (usually from process/viewer) associated with the display data.
        hkInt32 m_tag;

        /// The color to be used to draw the display data.
        hkColor::Argb m_color;

        /// Returns the cmd as a display geometry cmd if that is the type; null otherwise.
        HK_INLINE const DisplayGeometryCmd* asGeometryCmd() const;

        /// Returns the cmd as a display 1pt cmd if that is the type; null otherwise.
        HK_INLINE const Display1PtCmd* as1PtCmd() const;

        /// Returns the cmd as a display 2pt cmd if that is the type; null otherwise.
        HK_INLINE const Display2PtCmd* as2PtCmd() const;

        /// Returns the cmd as a display 3pt cmd if that is the type; null otherwise.
        HK_INLINE const Display3PtCmd* as3PtCmd() const;

        /// Returns the cmd as a display text cmd if that is the type; null otherwise.
        HK_INLINE const DisplayTextCmd* asTextCmd() const;

        /// Returns the cmd as a display 3d text cmd if that is the type; null otherwise.
        HK_INLINE const Display3dTextCmd* as3dTextCmd() const;
    };

    /// A cmd to display geometry.
    struct DisplayGeometryCmd : public DisplayCmd
    {
        HK_DECLARE_CLASS( DisplayGeometryCmd, New );
        DisplayGeometryCmd() { m_cmdType = hkVdbCmdType::DISPLAY_GEOMETRY; }

        /// The geometry data to be displayed.
        hkArray<hkDisplayGeometry*> m_geometry;

        /// The display geometry transform.
        hkMatrix4 m_transform;
    };

    /// A cmd to display 1 point.
    struct Display1PtCmd : public DisplayCmd
    {
        HK_DECLARE_CLASS( Display1PtCmd, New );

        /// The position of the point to be displayed.
        hkVector4 m_pt1;

        /// Extended information.
        struct Ex
        {
            HK_DECLARE_CLASS( Ex, New );

            /// The style in which the point is to be drawn.
            hk1PointDisplayStyle::Enum m_style;

            /// The scale at which the point is to be drawn.
            hkReal m_scale;
        };

        /// Gets extended information if available.
        HK_INLINE const Ex* getExtended() const;

    protected:
        Ex m_extended;
        friend class hkVdbDisplayHandler;
    };

    /// A cmd to display 2 points.
    struct Display2PtCmd : public DisplayCmd
    {
        HK_DECLARE_CLASS( Display2PtCmd, New );

        //@{
        /// The position of the points to be displayed.
        hkVector4 m_pt1;
        hkVector4 m_pt2;
        //@}

        /// Extended information.
        struct Ex
        {
            HK_DECLARE_CLASS( Ex, New );

            /// The style in which the points are to be drawn.
            hk2PointDisplayStyle::Enum m_style;

            /// The thickness with which the points are to be drawn.
            hkReal m_thickness;
        };

        /// Gets extended information if available.
        HK_INLINE const Ex* getExtended() const;

    protected:
        Ex m_extended;
        friend class hkVdbDisplayHandler;
    };

    /// A cmd to display 3 points.
    struct Display3PtCmd : public DisplayCmd
    {
        HK_DECLARE_CLASS( Display3PtCmd, New );
        Display3PtCmd() { m_cmdType = hkVdbCmdType::DISPLAY_3_POINTS; }

        //@{
        /// The position of the points to be displayed.
        hkVector4 m_pt1;
        hkVector4 m_pt2;
        hkVector4 m_pt3;
        //@}

        
        
        
        
        
        
        
        
        
    };

    /// A cmd to display text wherever, but usually near visuals.
    struct DisplayTextCmd : public DisplayCmd
    {
        HK_DECLARE_CLASS( DisplayTextCmd, New );
        DisplayTextCmd() { m_cmdType = hkVdbCmdType::DISPLAY_TEXT; }

        /// Text to display.
        const char* m_text;
    };

    /// A cmd to display text placed in 3d.
    struct Display3dTextCmd : public DisplayTextCmd
    {
        HK_DECLARE_CLASS( Display3dTextCmd, New );
        Display3dTextCmd() { m_cmdType = hkVdbCmdType::DISPLAY_3D_TEXT; }

        /// The 3d position of the text to display.
        hkVector4 m_position;
    };

    struct UpdateCameraCmd;
    struct RemoveCameraCmd;

    /// Base class for all camera-related cmd data.
    struct CameraCmd
    {
        HK_DECLARE_CLASS( CameraCmd, New );
        CameraCmd() : m_cmdType( hkVdbCmdType::INVALID ) {}

        /// The cmd type.
        hkVdbCmdType::Enum m_cmdType;

        /// The name of the camera.
        const char* m_name;

        /// Returns the cmd as an update camera cmd if that is the type; null otherwise.
        HK_INLINE const UpdateCameraCmd* asUpdateCmd() const;

        /// Returns the cmd as a remove camera cmd if that is the type; null otherwise.
        HK_INLINE const RemoveCameraCmd* asRemoveCmd() const;
    };

    /// A cmd to update (or add) camera settings.
    struct UpdateCameraCmd : public CameraCmd
    {
        HK_DECLARE_CLASS( UpdateCameraCmd, New );
        UpdateCameraCmd() { m_cmdType = hkVdbCmdType::UPDATE_CAMERA; }

        /// Camera from position.
        hkVector4 m_from;

        /// Camera to position.
        hkVector4 m_to;

        /// Camera's up vector.
        hkVector4 m_up;

        /// Camera's near clipping plane.
        hkReal m_nearPlane;

        /// Camera's far clipping plane.
        hkReal m_farPlane;

        /// Camera's field of view.
        hkReal m_fov;
    };

    /// A cmd to remove a previously updated/added camera.
    struct RemoveCameraCmd : public CameraCmd
    {
        HK_DECLARE_CLASS( RemoveCameraCmd, New );
        RemoveCameraCmd() { m_cmdType = hkVdbCmdType::REMOVE_CAMERA; }
    };

    HK_DECLARE_SIGNAL( DisplayOptionsSetSignal, hkSignal2< const hkDebugDisplayHandler::Options&, hkVdbSignalResult& > );
    /// Fired when the server/client sets the display options.
    DisplayOptionsSetSignal m_displayOptionsSet;

    HK_DECLARE_SIGNAL( GeometryCmdReceivedSignal, hkSignal2< const GeometryCmd&, hkVdbSignalResult& > );
    /// Fired when a geometry cmd is received/processed.
    GeometryCmdReceivedSignal m_geometryCmdReceived;

    HK_DECLARE_SIGNAL( DisplayCmdReceivedSignal, hkSignal2< const DisplayCmd&, hkVdbSignalResult& > );
    /// Fired when an immediate-mode-display cmd is received/processed.
    DisplayCmdReceivedSignal m_displayCmdReceived;

    HK_DECLARE_SIGNAL( FlushDisplaySignal, hkSignal1< bool > );
    /// Fired when the server has requested the immediate-mode-display be flushed.
    /// Bool indicates if the currently rendering display should be cleared or just added to.
    FlushDisplaySignal m_flushDisplay;

    HK_DECLARE_SIGNAL( CameraCmdReceivedSignal, hkSignal2< const CameraCmd&, hkVdbSignalResult& > );
    /// Fired when a camera cmd is received/processed.
    CameraCmdReceivedSignal m_cameraCmdReceived;
    

    //
    // Internal use
    //

    void onPlaybackInfoReceivedSignalInternal( const hkVdbPlaybackHandler::PlaybackInfo& info, hkVdbSignalResult& result );
    void onEntryDeallocatedSignal( const hkVdbCachedGeometries& cache, const hkVdbCachedGeometry& geometry );
    hkResult processSetDisplayOptionsCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader );
    hkResult processAddGeometryCmd( hkUint32 frameNumber, hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader );
    hkResult processDeprecatedAddGeometryPartCmd( hkUint32 frameNumber, int protocol, hkVdbLocalIStream& dataReader );
    hkResult processInstanceGeometryCmd( hkUint32 frameNumber, hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader );
    hkResult processUpdateGeometryCmd( hkUint32 frameNumber, hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader );
    hkResult processRemoveGeometryCmd( hkUint32 frameNumber, hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader );
    hkResult processDisplayGeometryCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader );
    hkResult processDisplay1PtCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader );
    hkResult processDisplay2PtCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader );
    hkResult processDisplay3PtCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader );
    hkResult processDisplayTextCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader );
    hkResult processDisplay3dTextCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader );
    hkResult processDisplayFlushCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader );
    hkResult processUpdateCameraCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader );
    hkResult processRemoveCameraCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader );

protected:

    void updateGeometriesToFrame( hkUint32 frame, hkVdbSignalResult& result );
    void displayIds( const hkVdbCachedGeometry& geom, int frame );

    // Avoid allocations during dispatch by having these declared and just reusing them
    hkDebugDisplayHandler::Options m_displayOptions;
    AddGeometryCmd m_addGeomCmd;
    InstanceGeometryCmd m_instanceGeomCmd;
    UpdateGeometryCmd m_updateGeomCmd;
    RemoveGeometryCmd m_removeGeomCmd;
    DisposeGeometryCmd m_disposeGeomCmd;
    DisplayGeometryCmd m_displayGeomCmd;
    Display1PtCmd m_display1PtCmd;
    Display2PtCmd m_display2PtCmd;
    Display3PtCmd m_display3PtCmd;
    DisplayTextCmd m_displayTextCmd;
    Display3dTextCmd m_display3dTextCmd;
    UpdateCameraCmd m_updateCameraCmd;
    RemoveCameraCmd m_removeCameraCmd;

    hkBool32 m_haveDisplayOptionsBeenSet;
    hkUint32 m_lastPlaybackFrame;
    hkUint64 m_queuedDisplayGeometriesId;
    hkArray<hkDisplayGeometry*> m_queuedDisplayGeometries;

    hkRefPtr<hkVdbCmdOutput> m_output;
    hkRefPtr<hkVdbCache> m_cache;
    hkRefPtr<hkVdbPlaybackHandler> m_playbackHandler;
    hkRefPtr<hkVdbProcessHandler> m_processHandler;

    friend class hkVdbPlaybackHandler;
};

#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbDisplayHandler.inl>

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
