// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbServices/hkVdbServices.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbDisplayHandler.h>

#include <Common/Visualize/hkDebugDisplayHandler.h>

#include <VisualDebugger/VdbServices/hkVdbClient.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdInput.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdOutput.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdDispatcher.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbPlaybackHandler.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbProcessHandler.h>

#define DEBUG_LOG_IDENTIFIER "vdb.Cmd.Handler.Display"
#include <Common/Base/System/Log/hkLog.hxx>

//#define DISPLAY_GEOM_IDS




#define READ_ID_FIRST( _ID, _STREAM ) \
    if ( protocol >= hkProtocolVersion::SUPPORTS_CIRCULAR_BUFFER ) { _ID = _STREAM.read64u(); } HK_MUST_END_WITH_SEMICOLON
#define READ_ID_LATER( _ID, _STREAM ) \
    if ( protocol < hkProtocolVersion::SUPPORTS_CIRCULAR_BUFFER ) { _ID = _STREAM.read64u(); } HK_MUST_END_WITH_SEMICOLON

#define READ_TAG_LAST( _TAG, _STREAM ) \
    if ( protocol >= hkProtocolVersion::VDB_2017_1 ) { _TAG = _STREAM.read32(); } HK_MUST_END_WITH_SEMICOLON
#define READ_TAG_EARLIER( _TAG, _STREAM ) \
    if ( protocol < hkProtocolVersion::VDB_2017_1 ) { _TAG = _STREAM.read32(); } HK_MUST_END_WITH_SEMICOLON
#define CONSUME_TAG( _STREAM ) \
    if ( protocol < hkProtocolVersion::VDB_2017_1 ) { _STREAM.read32(); } HK_MUST_END_WITH_SEMICOLON






#define READ_STRING( _STRBUF, _STREAM ) \
    HK_MULTILINE_MACRO_BEGIN \
        if ( protocol >= hkProtocolVersion::VDB_2017_1 ) { _STREAM.readString(_STRBUF); } \
        else \
        { \
            unsigned int length = _STREAM.read16u(); \
            hkInplaceArray<char, 128> buffer; \
            buffer.setSize( length + 1 ); \
            _STREAM.readRaw( buffer.begin(), length ); \
            buffer[length] = '\0'; \
            _STRBUF = &buffer[0]; \
        } \
    HK_MULTILINE_MACRO_END

// We check if we are replaying first so that we don't waste hash-table looks ups.
// When we aren't replaying we *know* the tag is enabled otherwise we wouldn't
// be receiving the signal!
#define RETURN_IF_FILTERED_DURING_REPLAY( _TAG ) \
    if ( m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::REPLAYING ) || \
         !m_output->canFlush() ) \
    { \
        hkBool32 _isSelected; \
        HK_VDB_VERIFY_OPERATION( m_processHandler->isProcessSelected( _TAG, _isSelected ), 0xedb00063, hkVdbError::CMD_HANDLER_READ_ERROR ); \
        if ( !_isSelected ) return HK_SUCCESS; \
    }

namespace
{
    hkResult handleSetDisplayOptionsCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
        hkVdbDisplayHandler* handler = reinterpret_cast< hkVdbDisplayHandler* >( userHandle );
        return handler->processSetDisplayOptionsCmd( cmd.getType(), protocol, dataReader );
    }

    hkResult handleAddGeometryCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
        hkVdbDisplayHandler* handler = reinterpret_cast< hkVdbDisplayHandler* >( userHandle );
        return handler->processAddGeometryCmd( frame.getFrameNumber(), cmd.getType(), protocol, dataReader );
    }

    hkResult handleInstanceGeometryCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
        hkVdbDisplayHandler* handler = reinterpret_cast< hkVdbDisplayHandler* >( userHandle );
        return handler->processInstanceGeometryCmd( frame.getFrameNumber(), cmd.getType(), protocol, dataReader );
    }

    hkResult handleUpdateGeometryCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
        hkVdbDisplayHandler* handler = reinterpret_cast< hkVdbDisplayHandler* >( userHandle );
        return handler->processUpdateGeometryCmd( frame.getFrameNumber(), cmd.getType(), protocol, dataReader );
    }

    hkResult handleRemoveGeometryCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
        hkVdbDisplayHandler* handler = reinterpret_cast< hkVdbDisplayHandler* >( userHandle );
        return handler->processRemoveGeometryCmd( frame.getFrameNumber(), cmd.getType(), protocol, dataReader );
    }

    hkResult handleDisplayGeometryCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
        hkVdbDisplayHandler* handler = reinterpret_cast< hkVdbDisplayHandler* >( userHandle );
        return handler->processDisplayGeometryCmd( cmd.getType(), protocol, dataReader );
    }

    hkResult handleDisplay1PtCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
        hkVdbDisplayHandler* handler = reinterpret_cast< hkVdbDisplayHandler* >( userHandle );
        return handler->processDisplay1PtCmd( cmd.getType(), protocol, dataReader );
    }

    hkResult handleDisplay2PtCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
        hkVdbDisplayHandler* handler = reinterpret_cast< hkVdbDisplayHandler* >( userHandle );
        return handler->processDisplay2PtCmd( cmd.getType(), protocol, dataReader );
    }

    hkResult handleDisplay3PtCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
        hkVdbDisplayHandler* handler = reinterpret_cast< hkVdbDisplayHandler* >( userHandle );
        return handler->processDisplay3PtCmd( cmd.getType(), protocol, dataReader );
    }

    hkResult handleDisplayTextCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
        hkVdbDisplayHandler* handler = reinterpret_cast< hkVdbDisplayHandler* >( userHandle );
        return handler->processDisplayTextCmd( cmd.getType(), protocol, dataReader );
    }

    hkResult handleDisplay3dTextCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
        hkVdbDisplayHandler* handler = reinterpret_cast< hkVdbDisplayHandler* >( userHandle );
        return handler->processDisplay3dTextCmd( cmd.getType(), protocol, dataReader );
    }

    hkResult handleDisplayFlushCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
        hkVdbDisplayHandler* handler = reinterpret_cast< hkVdbDisplayHandler* >( userHandle );
        return handler->processDisplayFlushCmd( cmd.getType(), protocol, dataReader );
    }

    hkResult handleUpdateCameraCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
        hkVdbDisplayHandler* handler = reinterpret_cast< hkVdbDisplayHandler* >( userHandle );
        return handler->processUpdateCameraCmd( cmd.getType(), protocol, dataReader );
    }

    hkResult handleRemoveCameraCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
        hkVdbDisplayHandler* handler = reinterpret_cast< hkVdbDisplayHandler* >( userHandle );
        return handler->processRemoveCameraCmd( cmd.getType(), protocol, dataReader );
    }

    hkResult handleDeprecatedAddGeometryPartCmdFunc( const hkVdbFrame& frame, const hkVdbCmd& cmd, int protocol, hkVdbLocalIStream& dataReader, hkVdbProcessInfo* processInfo, void* userHandle )
    {
        hkVdbDisplayHandler* handler = reinterpret_cast< hkVdbDisplayHandler* >( userHandle );
        return handler->processDeprecatedAddGeometryPartCmd( frame.getFrameNumber(), protocol, dataReader );
    }
};

hkVdbDisplayHandler::hkVdbDisplayHandler() :
    hkVdbCmdHandler<hkVdbCmdHandlerType::DISPLAY>( &s_debugLog ),
    m_haveDisplayOptionsBeenSet( false ),
    m_lastPlaybackFrame( 0 ),
    m_queuedDisplayGeometriesId( 0 )
{}

hkResult hkVdbDisplayHandler::waitForCompletion()
{
    
    return hkVdbCmdHandler<hkVdbCmdHandlerType::DISPLAY>::waitForCompletion();
}

hkResult hkVdbDisplayHandler::registerSelf( hkVdbClient& client )
{
    using namespace hkVdbCmdType;

    bool succeeded = true;
    hkVdbCmdDispatcher& dispatcher = client.getCmdDispatcher();

    // Setup
    succeeded &= ( dispatcher.registerHandler( SET_DISPLAY_OPTIONS, handleSetDisplayOptionsCmdFunc, this, this ).isSuccess() );

    // Persistent Geometry
    succeeded &= ( dispatcher.registerHandler( ADD_GEOMETRY, handleAddGeometryCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( ADD_GEOMETRY_EX, handleAddGeometryCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( DEPRECATED_ADD_GEOMETRY_PART, handleDeprecatedAddGeometryPartCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( DEPRECATED_ADD_GEOMETRY_FINAL, handleAddGeometryCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( INSTANCE_GEOMETRY, handleInstanceGeometryCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( UPDATE_GEOMETRY_TRANSFORM, handleUpdateGeometryCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( UPDATE_GEOMETRY_TRANSFORM_EX, handleUpdateGeometryCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( UPDATE_PARTICLE_TRANSFORMS, handleUpdateGeometryCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( UPDATE_GEOMETRY_VERTICES, handleUpdateGeometryCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( SET_GEOMETRY_COLOR, handleUpdateGeometryCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( SET_GEOMETRY_ALPHA, handleUpdateGeometryCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( SET_GEOMETRY_FLAG_BITS, handleUpdateGeometryCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( CLEAR_GEOMETRY_FLAG_BITS, handleUpdateGeometryCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( REMOVE_GEOMETRY, handleRemoveGeometryCmdFunc, this, this ).isSuccess() );

    // Temporary Display
    succeeded &= ( dispatcher.registerHandler( DISPLAY_GEOMETRY, handleDisplayGeometryCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( DISPLAY_POINT, handleDisplay1PtCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( DISPLAY_POINT_EX, handleDisplay1PtCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( DISPLAY_2_POINTS, handleDisplay2PtCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( DISPLAY_2_POINTS_EX, handleDisplay2PtCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( DISPLAY_3_POINTS, handleDisplay3PtCmdFunc, this, this ).isSuccess() );
    
    succeeded &= ( dispatcher.registerHandler( DISPLAY_TEXT, handleDisplayTextCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( DISPLAY_3D_TEXT, handleDisplay3dTextCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( FLUSH_DISPLAY, handleDisplayFlushCmdFunc, this, this ).isSuccess() );

    // Camera
    succeeded &= ( dispatcher.registerHandler( UPDATE_CAMERA, handleUpdateCameraCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( REMOVE_CAMERA, handleRemoveCameraCmdFunc, this, this ).isSuccess() );

    // Deprecated
    succeeded &= ( dispatcher.registerHandler( DEPRECATED_ADD_RIGID_GEOMETRY, handleAddGeometryCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( DEPRECATED_ADD_RIGID_GEOMETRY_EX, handleAddGeometryCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( DEPRECATED_ADD_MORPHING_GEOMETRY, handleAddGeometryCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( DEPRECATED_ADD_MORPHING_GEOMETRY_EX, handleAddGeometryCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( DEPRECATED_ADD_FIXED_GEOMETRY, handleAddGeometryCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( DEPRECATED_ADD_FIXED_GEOMETRY_EX, handleAddGeometryCmdFunc, this, this ).isSuccess() );
    succeeded &= ( dispatcher.registerHandler( DEPRECATED_DISPLAY_GEOMETRY, handleDisplayGeometryCmdFunc, this, this ).isSuccess() );

    // Hold on to useful objects
    m_output = &client.getCmdOutput();
    m_cache = &client.getCache();
    m_playbackHandler = client.getCmdHandler<hkVdbPlaybackHandler>();
    m_processHandler = client.getCmdHandler<hkVdbProcessHandler>();

    HK_SUBSCRIBE_TO_SIGNAL( m_playbackHandler->m_playbackInfoReceivedInternal, this, hkVdbDisplayHandler );
    HK_SUBSCRIBE_TO_SIGNAL( m_cache->geometries().m_entryDeallocated, this, hkVdbDisplayHandler );

    m_lastPlaybackFrame = client.getCmdInput().getCurrentFrameNumber();
    m_queuedDisplayGeometriesId = 0;

    HK_VDB_VERIFY_REPORTER_CONDITION( succeeded, dispatcher, hkVdbError::CMD_HANDLER_REGISTRATION_ERROR );

    return HK_SUCCESS;
}

hkResult hkVdbDisplayHandler::unregisterSelf( hkVdbClient& client )
{
    using namespace hkVdbCmdType;

    bool succeeded = true;
    hkVdbCmdDispatcher& dispatcher = client.getCmdDispatcher();

    // Setup
    succeeded &= ( dispatcher.unregisterHandler( SET_DISPLAY_OPTIONS ).isSuccess() );

    // Persistent Geometry
    succeeded &= ( dispatcher.unregisterHandler( ADD_GEOMETRY ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( ADD_GEOMETRY_EX ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( DEPRECATED_ADD_GEOMETRY_PART ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( DEPRECATED_ADD_GEOMETRY_FINAL ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( INSTANCE_GEOMETRY ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( UPDATE_GEOMETRY_TRANSFORM ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( UPDATE_GEOMETRY_TRANSFORM_EX ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( UPDATE_GEOMETRY_VERTICES ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( SET_GEOMETRY_COLOR ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( SET_GEOMETRY_ALPHA ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( SET_GEOMETRY_FLAG_BITS ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( CLEAR_GEOMETRY_FLAG_BITS ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( REMOVE_GEOMETRY ).isSuccess() );

    // Temporary Display
    succeeded &= ( dispatcher.unregisterHandler( DISPLAY_GEOMETRY ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( DISPLAY_POINT ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( DISPLAY_POINT_EX ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( DISPLAY_2_POINTS ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( DISPLAY_2_POINTS_EX ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( DISPLAY_3_POINTS ).isSuccess() );
    
    succeeded &= ( dispatcher.unregisterHandler( DISPLAY_TEXT ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( DISPLAY_3D_TEXT ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( FLUSH_DISPLAY ).isSuccess() );

    // Camera
    succeeded &= ( dispatcher.unregisterHandler( UPDATE_CAMERA ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( REMOVE_CAMERA ).isSuccess() );

    // Deprecated
    succeeded &= ( dispatcher.unregisterHandler( DEPRECATED_ADD_RIGID_GEOMETRY ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( DEPRECATED_ADD_RIGID_GEOMETRY_EX ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( DEPRECATED_ADD_MORPHING_GEOMETRY ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( DEPRECATED_ADD_MORPHING_GEOMETRY_EX ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( DEPRECATED_ADD_FIXED_GEOMETRY ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( DEPRECATED_ADD_FIXED_GEOMETRY_EX ).isSuccess() );
    succeeded &= ( dispatcher.unregisterHandler( DEPRECATED_DISPLAY_GEOMETRY ).isSuccess() );

    m_playbackHandler->m_playbackInfoReceivedInternal.unsubscribeAll( this );
    m_cache->geometries().m_entryDeallocated.unsubscribeAll( this );

    // Release objects
    m_output = HK_NULL;
    m_cache = HK_NULL;
    m_playbackHandler = HK_NULL;
    m_processHandler = HK_NULL;

    HK_VDB_VERIFY_REPORTER_CONDITION( succeeded, dispatcher, hkVdbError::CMD_HANDLER_REGISTRATION_ERROR );

    return HK_SUCCESS;
}

void hkVdbDisplayHandler::onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection )
{
    if ( use == hkVdbConnectionUse::APPLICATION )
    {
        // Reset to defaults in case an outside observer grabs our current display options and they haven't
        // been set yet by our new connection.
        m_displayOptions = hkDebugDisplayHandler::Options();
        m_haveDisplayOptionsBeenSet = false;
        m_lastPlaybackFrame = 0;
        m_queuedDisplayGeometriesId = 0;
        m_queuedDisplayGeometries.clear();
    }
}

hkResult hkVdbDisplayHandler::setDisplayOptions( const hkDebugDisplayHandler::Options& options )
{
    m_displayOptions = options;
    m_haveDisplayOptionsBeenSet = true;

    hkVdbSignalResult result;
    m_displayOptionsSet.fire( m_displayOptions, result );
    HK_VDB_VERIFY_SIGNAL_RESULT( result );

    return HK_SUCCESS;
}

void hkVdbDisplayHandler::onPlaybackInfoReceivedSignalInternal( const hkVdbPlaybackHandler::PlaybackInfo& info, hkVdbSignalResult& result )
{
    // We need to replay from the cache when finish a frame while replaying AND when we stop replaying.
    if ( info.flagWasSet( hkVdbPlaybackFlags::FRAME_ENDED ) || info.flagWasCleared( hkVdbPlaybackFlags::REPLAYING ) )
    {
        if ( info.m_targetFlags.anyIsSet( hkVdbPlaybackFlags::REPLAYING ) || info.flagWasCleared( hkVdbPlaybackFlags::REPLAYING ) )
        {
            updateGeometriesToFrame( info.m_targetFrame, result );
        }
        else
        {
            m_lastPlaybackFrame = info.m_targetFrame;
#ifdef DISPLAY_GEOM_IDS
            for ( hkVdbCache::Iterator iter = m_cache->getIterator();
                m_cache->isValid( iter );
                iter = m_cache->getNext( iter ) )
            {
                const hkVdbCachedGeometry& geom = *m_cache->getValue( iter );
                displayIds( geom, info.m_targetFrame );
            }
#endif
        }
    }
}

void hkVdbDisplayHandler::onEntryDeallocatedSignal( const hkVdbCachedGeometries& cache, const hkVdbCachedGeometry& geometry )
{
    HK_ASSERT( 0x22440924, &m_cache->geometries() == &cache, "Should not be changing cache after registration or listening to signals after unregistration" );

    // We only want to send signals if we are currently registered
    if ( m_cache )
    {
        m_disposeGeomCmd.m_id = geometry.getUid();

        hkVdbSignalResult result;
        m_geometryCmdReceived.fire( m_disposeGeomCmd, result );
        if ( result != hkVdbError::NONE )
        {
            
        }
    }
}

hkResult hkVdbDisplayHandler::processSetDisplayOptionsCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader )
{
    // Read in options, start with defaults just to be safe.
    hkDebugDisplayHandler::Options displayOptions;
    {
        dataReader.readQuadVector4( displayOptions.m_up );
        displayOptions.m_unitSize = dataReader.readFloat32();
        dataReader.readAabb( displayOptions.m_displayBounds );
    }

    HK_VDB_VERIFY_CONDITION( dataReader.isOk(), 0xedb00109, hkVdbError::CMD_HANDLER_READ_ERROR );

    HK_VDB_FAIL_IF_OPERATION_FAILED( setDisplayOptions( displayOptions ) );

    return HK_SUCCESS;
}

hkResult hkVdbDisplayHandler::processAddGeometryCmd( hkUint32 frameNumber, hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader )
{
    using namespace hkVdbCmdType;

    
    
    if ( m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::REPLAYING ) )
    {
        // Geometry state replay is handled by the listening to frame change signals from the hkVdbPlaybackHandler
        return HK_SUCCESS;
    }

    m_addGeomCmd.m_cmdType = type;

    // Note: We force the deprecated geometry commands to have extended info even though no color was provided.
    // We simply use a default of LIGHTGREY. This is unfortunate, but there's no way around it...and it's deprecated.
    // It's unlikely to impact much because we used to set the color almost always.
    switch ( type )
    {
        case DEPRECATED_ADD_RIGID_GEOMETRY:
        case DEPRECATED_ADD_RIGID_GEOMETRY_EX:
        case DEPRECATED_ADD_GEOMETRY_FINAL:
        {
            m_addGeomCmd.m_extended.m_color = hkColor::LIGHTGREY;
            m_addGeomCmd.m_extended.m_flags = hkDisplayGeometryFlags::DEFAULT;
            break;
        }
        case DEPRECATED_ADD_FIXED_GEOMETRY:
        case DEPRECATED_ADD_FIXED_GEOMETRY_EX:
        {
            m_addGeomCmd.m_extended.m_color = hkColor::LIGHTGREY;
            m_addGeomCmd.m_extended.m_flags = hkDisplayGeometryFlags::DEFAULT_FIXED;
            break;
        }
        case DEPRECATED_ADD_MORPHING_GEOMETRY:
        case DEPRECATED_ADD_MORPHING_GEOMETRY_EX:
        {
            m_addGeomCmd.m_extended.m_color = hkColor::LIGHTGREY;
            m_addGeomCmd.m_extended.m_flags =
                ( hkDisplayGeometryFlags::PICKABLE |
                    hkDisplayGeometryFlags::VISIBLE |
                    hkDisplayGeometryFlags::MUTABLE_TRANSFORM |
                    hkDisplayGeometryFlags::MUTABLE_VERTICES );
            break;
        }
    }

    READ_ID_FIRST( m_addGeomCmd.m_id, dataReader );

    // Read the array of display geometries
    {
        int numDisplayGeometeries = dataReader.read32();
        m_addGeomCmd.m_geometry.setSize( numDisplayGeometeries );
        for ( int i = 0; i < numDisplayGeometeries; i++ )
        {
            dataReader.readDisplayGeometry( m_addGeomCmd.m_geometry[i] );
        }

        // Append any queued geometries from deprecated "parts" cmds
        m_addGeomCmd.m_geometry.append( m_queuedDisplayGeometries );
    }

    // Read transform but store as matrix
    {
        hkTransform t;
        dataReader.readTransform( t );
        m_addGeomCmd.m_transform.set( t );
    }

    READ_ID_LATER( m_addGeomCmd.m_id, dataReader );
    if ( HK_VERY_UNLIKELY( m_queuedDisplayGeometriesId && ( m_queuedDisplayGeometriesId != m_addGeomCmd.m_id ) ) )
    {
        HK_WARN_ONCE( 0x22441356, "Id for queued display geometries doesn't match id for final geometries" );
    }
    m_queuedDisplayGeometries.clear();

    if (( type == ADD_GEOMETRY_EX ) ||
        ( type == DEPRECATED_ADD_RIGID_GEOMETRY_EX ) ||
        ( type == DEPRECATED_ADD_FIXED_GEOMETRY_EX ) ||
        ( type == DEPRECATED_ADD_MORPHING_GEOMETRY_EX ) )
    {
        m_addGeomCmd.m_extended.m_color = dataReader.read32u();
        m_addGeomCmd.m_extended.m_flags = hkDisplayGeometryFlags( dataReader.read8u() );

        // The deprecated EX commands did not have available the same comprehensive set of geometry flags.
        // There was only PICKABLE. Instead, the command itself was responsible for dictating *some* of
        // these capabilities. Make sure the flags reflect the capabilities of the various deprecated EX cmds.
        switch ( type )
        {
            case DEPRECATED_ADD_RIGID_GEOMETRY_EX:
            {
                m_addGeomCmd.m_extended.m_flags.orWith( hkDisplayGeometryFlags::VISIBLE | hkDisplayGeometryFlags::MUTABLE_TRANSFORM );
                break;
            }
            case DEPRECATED_ADD_FIXED_GEOMETRY_EX:
            {
                m_addGeomCmd.m_extended.m_flags.orWith( hkDisplayGeometryFlags::VISIBLE );
                break;
            }
            case DEPRECATED_ADD_MORPHING_GEOMETRY_EX:
            {
                m_addGeomCmd.m_extended.m_flags.orWith( hkDisplayGeometryFlags::VISIBLE | hkDisplayGeometryFlags::MUTABLE_VERTICES );
                break;
            }
        }
    }

    // Read the display tag
    m_addGeomCmd.m_tag = dataReader.read32();

    HK_VDB_VERIFY_CONDITION( dataReader.isOk(), 0xedb00003, hkVdbError::CMD_HANDLER_READ_ERROR );

    // Grab already cached geometry from deprecated "parts" cmds (if created)
    hkVdbCachedGeometry* geometry = HK_NULL;
    if ( HK_VERY_UNLIKELY( type == DEPRECATED_ADD_GEOMETRY_FINAL ) )
    {
        geometry = m_cache->geometries().accessEntry( m_addGeomCmd.m_id );
        if ( geometry )
        {
            m_addGeomCmd.m_id = geometry->getUid();
            geometry->deprecatedPostInit(
                m_addGeomCmd.m_tag,
                m_addGeomCmd.m_transform,
                m_addGeomCmd.m_extended.m_color,
                m_addGeomCmd.m_extended.m_flags );
        }
    }

    // Cache geometry
    if ( HK_VERY_LIKELY( !geometry ) )
    {
        // Create a new cached geometry
        hkVdbCachedGeometryCinfo cinfo(
            m_addGeomCmd.m_tag,
            m_addGeomCmd.m_id,
            frameNumber,
            m_addGeomCmd.m_transform );

        // Set extended information
        if ( const AddGeometryCmd::Ex* ex = m_addGeomCmd.getExtended() )
        {
            cinfo.m_initialColor = ex->m_color;
            cinfo.m_initialFlags = ex->m_flags;
        }

        m_addGeomCmd.m_id = m_cache->geometries().createEntry( cinfo ).getUid();
    }

    hkVdbSignalResult result;
    if ( dataReader.isOk() )
    {
        m_geometryCmdReceived.fire( m_addGeomCmd, result );
    }
    else
    {
        result.signalError( 0xedb00057, hkVdbError::CMD_HANDLER_READ_ERROR );
    }

    // Delete the intermediate display geometries
    hkReferencedObject::removeReferences( m_addGeomCmd.m_geometry.begin(), m_addGeomCmd.m_geometry.getSize() );

    HK_VDB_VERIFY_SIGNAL_RESULT( result );
    return HK_SUCCESS;
}

hkResult hkVdbDisplayHandler::processDeprecatedAddGeometryPartCmd( hkUint32 frameNumber, int protocol, hkVdbLocalIStream& dataReader )
{
    using namespace hkVdbCmdType;

    if ( m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::REPLAYING ) )
    {
        // Geometry state replay is handled by the listening to frame change signals from the hkVdbPlaybackHandler
        return HK_SUCCESS;
    }

    HK_ASSERT( 0x22441355, protocol < hkProtocolVersion::CURRENT, "This command should no longer be sent to the client" );

    hkUint64 id = 0;
    READ_ID_FIRST( id, dataReader );
    if ( HK_VERY_UNLIKELY( m_queuedDisplayGeometriesId && ( m_queuedDisplayGeometriesId != id ) ) )
    {
        HK_WARN_ONCE( 0x22441359, "Queued display geometries were not properly finalized, id has changed" );
    }
    m_queuedDisplayGeometriesId = id;

    // Read the array of display geometries
    {
        int numDisplayGeometeries = dataReader.read32();
        hkDisplayGeometry** geometries = m_queuedDisplayGeometries.expandBy( numDisplayGeometeries );
        for ( int i = 0; i < numDisplayGeometeries; i++ )
        {
            dataReader.readDisplayGeometry( geometries[i] );
        }
    }

    HK_VDB_VERIFY_CONDITION( dataReader.isOk(), 0xedb00165, hkVdbError::CMD_HANDLER_READ_ERROR );

    // Cache Geometry so update commands will succeed
    // Note: Before we changed protocol to include an id...we didn't support updating geometry before finalizing.
    if ( id && !m_cache->geometries().accessEntry( id ) )
    {
        // Create a new cached geometry
        hkVdbCachedGeometryCinfo cinfo(
            -1,
            id,
            frameNumber,
            hkMatrix4::getIdentity() );

        // Set extended information
        cinfo.m_initialColor = hkColor::LIGHTGREY;
        cinfo.m_initialFlags = hkDisplayGeometryFlags::DEFAULT;

        m_cache->geometries().createEntry( cinfo );
    }

    return HK_SUCCESS;
}

hkResult hkVdbDisplayHandler::processInstanceGeometryCmd( hkUint32 frameNumber, hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader )
{
    if ( m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::REPLAYING ) )
    {
        // Geometry state replay is handled by the listening to frame change signals from the hkVdbPlaybackHandler
        return HK_SUCCESS;
    }

    READ_ID_FIRST( m_instanceGeomCmd.m_id, dataReader );

    // Find source geometry
    hkVdbCachedGeometry* sourceGeometry = HK_NULL;
    {
        m_instanceGeomCmd.m_sourceGeometryId = dataReader.read64u();
        sourceGeometry = m_cache->geometries().accessEntry( m_instanceGeomCmd.m_sourceGeometryId );
        if ( sourceGeometry )
        {
            m_instanceGeomCmd.m_sourceGeometryId = sourceGeometry->getUid();
        }
    }

    // Read transform but store as matrix
    {
        hkTransform t;
        dataReader.readTransform( t );
        m_instanceGeomCmd.m_transform.set( t );
    }

    READ_ID_LATER( m_instanceGeomCmd.m_id, dataReader );

    // Read the display tag
    m_instanceGeomCmd.m_tag = dataReader.read32();

    // Cache geometry
    {
        // Create a new cached geometry
        hkVdbCachedGeometryCinfo cinfo(
            m_instanceGeomCmd.m_tag,
            m_instanceGeomCmd.m_id,
            frameNumber,
            m_instanceGeomCmd.m_transform );

        if ( sourceGeometry )
        {
            cinfo.m_initialColor = sourceGeometry->getColor( frameNumber );
            cinfo.m_initialFlags = sourceGeometry->getFlags( frameNumber );
        }

        m_instanceGeomCmd.m_id = m_cache->geometries().createEntry( cinfo ).getUid();
    }

    HK_VDB_VERIFY_CONDITION( dataReader.isOk(), 0xedb00058, hkVdbError::CMD_HANDLER_READ_ERROR );

    // Signal
    hkVdbSignalResult result;
    m_geometryCmdReceived.fire( m_instanceGeomCmd, result );
    HK_VDB_VERIFY_SIGNAL_RESULT( result );

    return HK_SUCCESS;
}

hkResult hkVdbDisplayHandler::processUpdateGeometryCmd( hkUint32 frameNumber, hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader )
{
    using namespace hkVdbCmdType;

    if ( m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::REPLAYING ) )
    {
        // Geometry state replay is handled by the listening to frame change signals from the hkVdbPlaybackHandler
        return HK_SUCCESS;
    }

    m_updateGeomCmd.m_cmdType = type;

    
    
    
    // shouldSignal is *only* necessary to support DEPRECATED_ADD_GEOMETRY_PART and DEPRECATED_ADD_GEOMETRY_FINAL
    hkBool32 shouldSignal = true;
    switch ( type )
    {
        case SET_GEOMETRY_FLAG_BITS:
        case CLEAR_GEOMETRY_FLAG_BITS:
        {
            hkDisplayGeometryFlags& flagBits = ( type == SET_GEOMETRY_FLAG_BITS ) ? m_updateGeomCmd.m_setFlagBits : m_updateGeomCmd.m_clearFlagBits;
            m_updateGeomCmd.m_id = dataReader.read64u();
            flagBits = dataReader.read8u();

            if ( hkVdbCachedGeometry* geometry = m_cache->geometries().accessEntry( m_updateGeomCmd.m_id ) )
            {
                m_updateGeomCmd.m_id = geometry->getUid();
                if ( type == SET_GEOMETRY_FLAG_BITS )
                {
                    geometry->registerSetFlagBits( frameNumber, flagBits );
                }
                else
                {
                    geometry->registerClearFlagBits( frameNumber, flagBits );
                }
                shouldSignal = ( geometry->getTag() != -1 );
            }
            else
            {
                HK_WARN( 0x22441032, "Tried to set/clear flags on geometry with invalid ID" );
            }
            break;
        }
        case SET_GEOMETRY_COLOR:
        {
            READ_ID_FIRST( m_updateGeomCmd.m_id, dataReader );
            m_updateGeomCmd.m_color = dataReader.read32u();
            READ_ID_LATER( m_updateGeomCmd.m_id, dataReader );
            CONSUME_TAG( dataReader );

            if ( hkVdbCachedGeometry* geometry = m_cache->geometries().accessEntry( m_updateGeomCmd.m_id ) )
            {
                m_updateGeomCmd.m_id = geometry->getUid();
                geometry->registerColorUpdate( frameNumber, m_updateGeomCmd.m_color );
                shouldSignal = ( geometry->getTag() != -1 );
            }
            else
            {
                HK_WARN( 0x145c9460, "Tried to set color on geometry with invalid ID" );
            }
            break;
        }
        case SET_GEOMETRY_ALPHA:
        {
            READ_ID_FIRST( m_updateGeomCmd.m_id, dataReader );
            m_updateGeomCmd.m_alpha = hkMath::clamp( dataReader.readFloat32(), 0.0f, 1.0f );
            READ_ID_LATER( m_updateGeomCmd.m_id, dataReader );
            CONSUME_TAG( dataReader );

            if ( hkVdbCachedGeometry* geometry = m_cache->geometries().accessEntry( m_updateGeomCmd.m_id ) )
            {
                m_updateGeomCmd.m_id = geometry->getUid();
                hkColor::Argb currentColor = geometry->getColor( frameNumber );
                m_updateGeomCmd.m_color = hkColor::replaceAlpha( ( unsigned char ) ( m_updateGeomCmd.m_alpha * 255 ), currentColor );
                geometry->registerColorUpdate( frameNumber, m_updateGeomCmd.m_color );
                shouldSignal = ( geometry->getTag() != -1 );
            }
            else
            {
                HK_WARN( 0x6996972b, "Tried to set color on geometry with invalid ID" );
            }
            break;
        }
        case UPDATE_GEOMETRY_TRANSFORM:
        {
            READ_ID_FIRST( m_updateGeomCmd.m_id, dataReader );
            // Read transform but store as matrix
            {
                hkTransform t;
                dataReader.readTransform( t );
                m_updateGeomCmd.m_transform.set( t );
            }
            READ_ID_LATER( m_updateGeomCmd.m_id, dataReader );
            CONSUME_TAG( dataReader );

            // Add to the cached geometry's list of transforms
            if ( hkVdbCachedGeometry* geometry = m_cache->geometries().accessEntry( m_updateGeomCmd.m_id ) )
            {
                m_updateGeomCmd.m_id = geometry->getUid();
                geometry->registerTransformUpdate( frameNumber, m_updateGeomCmd.m_transform );
                shouldSignal = ( geometry->getTag() != -1 );
            }
            else
            {
                HK_WARN( 0x22441210, "Invalid geometry ID for update" );
            }
            break;
        }
        case UPDATE_GEOMETRY_TRANSFORM_EX:
        {
            // Read the object id
            m_updateGeomCmd.m_id = dataReader.read64u();

            // Read matrix
            dataReader.readMatrix( m_updateGeomCmd.m_transform );

            // Consume the display tag (no longer sent)
            CONSUME_TAG( dataReader );

            // Add to the cached geometry's list of transforms
            if ( hkVdbCachedGeometry* geometry = m_cache->geometries().accessEntry( m_updateGeomCmd.m_id ) )
            {
                m_updateGeomCmd.m_id = geometry->getUid();
                geometry->registerTransformUpdate( frameNumber, m_updateGeomCmd.m_transform );
                shouldSignal = ( geometry->getTag() != -1 );
            }
            else
            {
                HK_WARN( 0x22441211, "Invalid geometry ID for update" );
            }
            break;
        }
        case UPDATE_GEOMETRY_VERTICES:
        {
            m_updateGeomCmd.m_id = dataReader.read64u();

            // Get vertices (we deallocate to avoid stomping on memory given to geometry cache)
            m_updateGeomCmd.m_newVertices.clearAndDeallocate();
            m_updateGeomCmd.m_newVertices.setSize( dataReader.read32() );
            for ( int i = 0; i < m_updateGeomCmd.m_newVertices.getSize(); i++ )
            {
                dataReader.readQuadVector4( m_updateGeomCmd.m_newVertices[i] );
            }

            // Consume the display tag (no longer sent)
            CONSUME_TAG( dataReader );

            // Register with cache
            if ( hkVdbCachedGeometry* geometry = m_cache->geometries().accessEntry( m_updateGeomCmd.m_id ) )
            {
                m_updateGeomCmd.m_id = geometry->getUid();
                geometry->registerVerticesUpdate( frameNumber, m_updateGeomCmd.m_newVertices );
                shouldSignal = ( geometry->getTag() != -1 );
            }
            else
            {
                HK_WARN( 0x22441212, "Invalid geometry ID for update" );
            }
            break;
        }
        case UPDATE_PARTICLE_TRANSFORMS:
        {
            // Read the command
            m_updateGeomCmd.m_id = dataReader.read64u();
            m_updateGeomCmd.m_particleUpdateId = dataReader.read32();
            hkUint8 useOrientation = dataReader.read8u();
            hkUint32 numTransforms = dataReader.read32u();

            // Get transforms (we deallocate to avoid stomping on memory given to geometry cache)
            m_updateGeomCmd.m_newParticleTransforms.clearAndDeallocate();
            m_updateGeomCmd.m_newParticleTransforms.setSize( numTransforms );

            for ( hkUint32 i = 0; i < numTransforms; i++ )
            {
                // Read orientation
                if ( useOrientation )
                {
                    hkQuaternion q; dataReader.readArrayFloat32( &q.m_vec( 0 ), 4 );
                    m_updateGeomCmd.m_newParticleTransforms[i].setRotation( q );
                }
                else
                {
                    m_updateGeomCmd.m_newParticleTransforms[i].setIdentity();
                }

                // Read translation and color
                hkVector4 v; dataReader.readQuadVector4( v );
                *(hkColor::Argb*)&v( 3 ) = dataReader.read32u();
                m_updateGeomCmd.m_newParticleTransforms[i].setTranslation( v );
            }

            // Add to the cached geometry's list of transforms
            if ( hkVdbCachedGeometry* geometry = m_cache->geometries().accessEntry( m_updateGeomCmd.m_id ) )
            {
                m_updateGeomCmd.m_id = geometry->getUid();
                geometry->registerParticleTransformsUpdate( frameNumber, m_updateGeomCmd.m_newParticleTransforms );
                shouldSignal = ( geometry->getTag() != -1 );
            }
            else
            {
                HK_WARN( 0x22441210, "Invalid geometry ID for update" );
            }
            break;
        }
        default:
        {
            HK_VDB_SIGNAL_ERROR_MSG(
                0xedb00070,
                hkVdbError::CMD_UNKNOWN,
                "Unknown cmd type of " << type );
            return HK_FAILURE;
        }
    }

    HK_VDB_VERIFY_CONDITION( dataReader.isOk(), 0xedb00059, hkVdbError::CMD_HANDLER_READ_ERROR );

    if ( HK_VERY_LIKELY( shouldSignal ) )
    {
        hkVdbSignalResult result;
        m_geometryCmdReceived.fire( m_updateGeomCmd, result );
        HK_VDB_VERIFY_SIGNAL_RESULT( result );
    }

    return HK_SUCCESS;
}

hkResult hkVdbDisplayHandler::processRemoveGeometryCmd( hkUint32 frameNumber, hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader )
{
    if ( m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::REPLAYING ) )
    {
        // Geometry state replay is handled by the listening to frame change signals from the hkVdbPlaybackHandler
        return HK_SUCCESS;
    }

    // read the object id
    m_removeGeomCmd.m_id = dataReader.read64u();

    // consume the display tag (no longer sent)
    CONSUME_TAG( dataReader );

    // shouldSignal is *only* necessary to support DEPRECATED_ADD_GEOMETRY_PART and DEPRECATED_ADD_GEOMETRY_FINAL
    hkBool32 shouldSignal = true;
    if ( hkVdbCachedGeometry* geometry = m_cache->geometries().accessEntry( m_removeGeomCmd.m_id ) )
    {
        m_removeGeomCmd.m_id = geometry->getUid();
        geometry->registerRemoved( frameNumber );
        shouldSignal = ( geometry->getTag() != -1 );
    }
    else
    {
        HK_WARN( 0x7a7ad52f, "Invalid geometry ID for remove" );
    }

    HK_VDB_VERIFY_CONDITION( dataReader.isOk(), 0xedb00060, hkVdbError::CMD_HANDLER_READ_ERROR );

    if ( HK_VERY_LIKELY( shouldSignal ) )
    {
        hkVdbSignalResult result;
        m_geometryCmdReceived.fire( m_removeGeomCmd, result );
        HK_VDB_VERIFY_SIGNAL_RESULT( result );
    }

    return HK_SUCCESS;
}

hkResult hkVdbDisplayHandler::processDisplayGeometryCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader )
{
    using namespace hkVdbCmdType;

    if ( m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::CACHING ) )
    {
        // No need to process single frame cmds while caching
        return HK_SUCCESS;
    }

    // maybe read the object id
    READ_ID_FIRST( m_displayGeomCmd.m_id, dataReader );

    // read the array of display geometries
    {
        int numDisplayGeometries = dataReader.read32();
        m_displayGeomCmd.m_geometry.setSize( numDisplayGeometries );
        for ( int i = 0; i < numDisplayGeometries; i++ )
        {
            dataReader.readDisplayGeometry( m_displayGeomCmd.m_geometry[i] );
        }
    }

    if ( type == DEPRECATED_DISPLAY_GEOMETRY )
    {
        // Dep didn't include transform
        m_displayGeomCmd.m_transform.setIdentity();
    }
    else
    {
        // Read transform but store as matrix
        hkTransform t;
        dataReader.readTransform( t );
        m_displayGeomCmd.m_transform.set( t );
    }

    // read the color
    m_displayGeomCmd.m_color = dataReader.read32u();

    // maybe read the object id
    if ( protocol >= hkProtocolVersion::MORE_CMD_IDS_700 )
    {
        READ_ID_LATER( m_displayGeomCmd.m_id, dataReader );
    }

    // read the display tag
    m_displayGeomCmd.m_tag = dataReader.read32();
    RETURN_IF_FILTERED_DURING_REPLAY( m_displayGeomCmd.m_tag );

    hkVdbSignalResult result;
    if ( dataReader.isOk() )
    {
        m_displayCmdReceived.fire( m_displayGeomCmd, result );
    }
    else
    {
        result.signalError( 0xedb00062, hkVdbError::CMD_HANDLER_READ_ERROR );
    }

    // Delete the intermediate display geometries
    hkReferencedObject::removeReferences( m_displayGeomCmd.m_geometry.begin(), m_displayGeomCmd.m_geometry.getSize() );

    HK_VDB_VERIFY_SIGNAL_RESULT( result );
    return HK_SUCCESS;
}

hkResult hkVdbDisplayHandler::processDisplay1PtCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader )
{
    using namespace hkVdbCmdType;

    if ( m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::CACHING ) )
    {
        // No need to process single frame cmds while caching
        return HK_SUCCESS;
    }

    m_display1PtCmd.m_cmdType = type;

    // maybe read the object id
    READ_ID_FIRST( m_display1PtCmd.m_id, dataReader );

    dataReader.readQuadVector4( m_display1PtCmd.m_pt1 );

    if ( protocol >= hkProtocolVersion::VDB_2017_1 ) m_display1PtCmd.m_color = dataReader.read32u();

    if ( type == DISPLAY_POINT_EX )
    {
        m_display1PtCmd.m_extended.m_style = ( protocol >= hkProtocolVersion::VDB_2017_1 ) ? hk1PointDisplayStyle::Enum( dataReader.read8u() ) : hk1PointDisplayStyle::DOT;
        m_display1PtCmd.m_extended.m_scale = dataReader.readFloat32();
    }

    if ( protocol < hkProtocolVersion::VDB_2017_1 ) m_display1PtCmd.m_color = dataReader.read32u();

    // maybe read the object id
    if ( ( protocol >= hkProtocolVersion::MORE_CMD_IDS_700 ) || ( type == DISPLAY_POINT_EX ) )
    {
        READ_ID_LATER( m_display1PtCmd.m_id, dataReader );
    }

    // get the display tag
    m_display1PtCmd.m_tag = dataReader.read32();
    RETURN_IF_FILTERED_DURING_REPLAY( m_display1PtCmd.m_tag );

    HK_VDB_VERIFY_CONDITION( dataReader.isOk(), 0xedb00063, hkVdbError::CMD_HANDLER_READ_ERROR );

    hkVdbSignalResult result;
    m_displayCmdReceived.fire( m_display1PtCmd, result );
    HK_VDB_VERIFY_SIGNAL_RESULT( result );

    return HK_SUCCESS;
}

hkResult hkVdbDisplayHandler::processDisplay2PtCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader )
{
    using namespace hkVdbCmdType;

    if ( m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::CACHING ) )
    {
        // No need to process single frame cmds while caching
        return HK_SUCCESS;
    }

    m_display2PtCmd.m_cmdType = type;

    // maybe read the object id
    READ_ID_FIRST( m_display2PtCmd.m_id, dataReader );

    dataReader.readQuadVector4( m_display2PtCmd.m_pt1 );
    dataReader.readQuadVector4( m_display2PtCmd.m_pt2 );
    m_display2PtCmd.m_color = dataReader.read32u();

    if ( type == DISPLAY_2_POINTS_EX )
    {
        m_display2PtCmd.m_extended.m_style = hk2PointDisplayStyle::Enum( dataReader.read8u() );
        m_display2PtCmd.m_extended.m_thickness = dataReader.readFloat32();
    }

    // maybe read the object id
    if ( protocol >= hkProtocolVersion::MORE_CMD_IDS_700 )
    {
        READ_ID_LATER( m_display2PtCmd.m_id, dataReader );
    }

    // get the display tag
    m_display2PtCmd.m_tag = dataReader.read32();
    RETURN_IF_FILTERED_DURING_REPLAY( m_display2PtCmd.m_tag );

    HK_VDB_VERIFY_CONDITION( dataReader.isOk(), 0xedb00064, hkVdbError::CMD_HANDLER_READ_ERROR );

    hkVdbSignalResult result;
    m_displayCmdReceived.fire( m_display2PtCmd, result );
    HK_VDB_VERIFY_SIGNAL_RESULT( result );

    return HK_SUCCESS;
}

hkResult hkVdbDisplayHandler::processDisplay3PtCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader )
{
    if ( m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::CACHING ) )
    {
        // No need to process single frame cmds while caching
        return HK_SUCCESS;
    }

    m_display3PtCmd.m_id = dataReader.read64u();
    dataReader.readQuadVector4( m_display3PtCmd.m_pt1 );
    dataReader.readQuadVector4( m_display3PtCmd.m_pt2 );
    dataReader.readQuadVector4( m_display3PtCmd.m_pt3 );
    m_display3PtCmd.m_color = dataReader.read32u();

    // get the display tag
    m_display3PtCmd.m_tag = dataReader.read32();
    RETURN_IF_FILTERED_DURING_REPLAY( m_display3PtCmd.m_tag );

    HK_VDB_VERIFY_CONDITION( dataReader.isOk(), 0xedb00065, hkVdbError::CMD_HANDLER_READ_ERROR );

    hkVdbSignalResult result;
    m_displayCmdReceived.fire( m_display3PtCmd, result );
    HK_VDB_VERIFY_SIGNAL_RESULT( result );

    return HK_SUCCESS;
}

hkResult hkVdbDisplayHandler::processDisplayTextCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader )
{
    if ( m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::CACHING ) )
    {
        // No need to process single frame cmds while caching
        return HK_SUCCESS;
    }

    // maybe read the object id
    READ_ID_FIRST( m_displayTextCmd.m_id, dataReader );

    // read text
    hkStringBuf buff;
    READ_STRING( buff, dataReader );
    m_displayTextCmd.m_text = buff;

    // read the color information from the text output
    m_displayTextCmd.m_color = dataReader.read32u();

    // maybe read the object id
    if ( protocol >= hkProtocolVersion::MORE_CMD_IDS_700 )
    {
        READ_ID_LATER( m_displayTextCmd.m_id, dataReader );
    }

    // get the display tag
    m_displayTextCmd.m_tag = dataReader.read32();
    RETURN_IF_FILTERED_DURING_REPLAY( m_displayTextCmd.m_tag );

    HK_VDB_VERIFY_CONDITION( dataReader.isOk(), 0xedb00066, hkVdbError::CMD_HANDLER_READ_ERROR );

    hkVdbSignalResult result;
    m_displayCmdReceived.fire( m_displayTextCmd, result );
    HK_VDB_VERIFY_SIGNAL_RESULT( result );

    return HK_SUCCESS;
}

hkResult hkVdbDisplayHandler::processDisplay3dTextCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader )
{
    if ( m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::CACHING ) )
    {
        // No need to process single frame cmds while caching
        return HK_SUCCESS;
    }

    // maybe read the object id
    READ_ID_FIRST( m_display3dTextCmd.m_id, dataReader );

    // read text
    hkStringBuf buff;
    READ_STRING( buff, dataReader );
    m_display3dTextCmd.m_text = buff;

    // read the color information from the text output
    m_display3dTextCmd.m_color = dataReader.read32();

    // maybe read the object id
    if ( protocol >= hkProtocolVersion::MORE_CMD_IDS_700 )
    {
        READ_ID_LATER( m_display3dTextCmd.m_id, dataReader );
    }

    // get the display tag
    READ_TAG_EARLIER( m_display3dTextCmd.m_tag, dataReader );

    // get the position
    dataReader.readQuadVector4( m_display3dTextCmd.m_position );

    // get the display tag
    READ_TAG_LAST( m_display3dTextCmd.m_tag, dataReader );
    RETURN_IF_FILTERED_DURING_REPLAY( m_display3dTextCmd.m_tag );

    HK_VDB_VERIFY_CONDITION( dataReader.isOk(), 0xedb00067, hkVdbError::CMD_HANDLER_READ_ERROR );

    hkVdbSignalResult result;
    m_displayCmdReceived.fire( m_display3dTextCmd, result );
    HK_VDB_VERIFY_SIGNAL_RESULT( result );

    return HK_SUCCESS;
}

hkResult hkVdbDisplayHandler::processDisplayFlushCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader )
{
    if ( m_playbackHandler->getFlags().anyIsSet( hkVdbPlaybackFlags::CACHING | hkVdbPlaybackFlags::REPLAYING ) )
    {
        // No need to process single frame cmds while caching.
        // And during replay, flushing doesn't make much sense.
        return HK_SUCCESS;
    }

    
    
    
    bool clear = dataReader.read32();

    HK_VDB_VERIFY_CONDITION( dataReader.isOk(), 0xedb00087, hkVdbError::CMD_HANDLER_READ_ERROR );

    m_flushDisplay.fire( clear );

    return HK_SUCCESS;
}

hkResult hkVdbDisplayHandler::processUpdateCameraCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader )
{
    
    dataReader.readQuadVector4( m_updateCameraCmd.m_from );
    dataReader.readQuadVector4( m_updateCameraCmd.m_to );
    dataReader.readQuadVector4( m_updateCameraCmd.m_up );
    m_updateCameraCmd.m_nearPlane = dataReader.readFloat32();
    m_updateCameraCmd.m_farPlane = dataReader.readFloat32();
    m_updateCameraCmd.m_fov = dataReader.readFloat32();
    hkStringBuf buff;
    READ_STRING( buff, dataReader );
    m_updateCameraCmd.m_name = buff;

    HK_VDB_VERIFY_CONDITION( dataReader.isOk(), 0xedb00068, hkVdbError::CMD_HANDLER_READ_ERROR );

    hkVdbSignalResult result;
    m_cameraCmdReceived.fire( m_updateCameraCmd, result );
    HK_VDB_VERIFY_SIGNAL_RESULT( result );

    return HK_SUCCESS;
}

hkResult hkVdbDisplayHandler::processRemoveCameraCmd( hkVdbCmdType::Enum type, int protocol, hkVdbLocalIStream& dataReader )
{
    
    hkStringBuf buff;
    READ_STRING( buff, dataReader );
    m_removeCameraCmd.m_name = buff;

    HK_VDB_VERIFY_CONDITION( dataReader.isOk(), 0xedb00069, hkVdbError::CMD_HANDLER_READ_ERROR );

    hkVdbSignalResult result;
    m_cameraCmdReceived.fire( m_removeCameraCmd, result );
    HK_VDB_VERIFY_SIGNAL_RESULT( result );

    return HK_SUCCESS;
}

void hkVdbDisplayHandler::updateGeometriesToFrame( hkUint32 frame, hkVdbSignalResult& result )
{
    if ( !m_cache )
    {
        HK_VDB_SIGNAL_ERROR( 0xedb00102, hkVdbError::CMD_HANDLER_NOT_REGISTERED );
        result.signalError( *this );
        return;
    }

    // Set our start/end
    hkUint32 startFrame = m_lastPlaybackFrame;
    const hkUint32 endFrame = frame;
    m_lastPlaybackFrame = frame;

    // Update geometries
    hkVdbCachedGeometries& geometries = m_cache->geometries();
    for ( hkVdbCachedGeometries::Iterator iter = geometries.getIterator();
        geometries.isValid( iter );
        iter = geometries.getNext( iter ) )
    {
        const hkVdbCachedGeometry& geom = *geometries.getValue( iter );
        const hkUint64 id = geom.getUid();

        m_updateGeomCmd.m_cmdType = hkVdbCmdType::UPDATE_GEOMETRY;
        m_updateGeomCmd.m_id = id;
        m_updateGeomCmd.m_updateBits.clear();
        m_updateGeomCmd.m_setFlagBits.clear();
        m_updateGeomCmd.m_clearFlagBits.clear();

        const hkUint32 frameAdded = geom.getFrameAdded();
        const hkUint32 frameRemoved = geom.getFrameRemoved();
        const hkDisplayGeometryFlags startFlags = geom.getFlags( startFrame );
        const hkDisplayGeometryFlags flags = geom.getFlags( endFrame );
        bool canBeVisible;
        {
            // Make sure the geom is added.
            canBeVisible = geom.isAdded( endFrame );

            // Make sure it's marked visible.
            canBeVisible &= bool( flags.anyIsSet( hkDisplayGeometryFlags::VISIBLE ) );

            // Lastly, check that it's source is selected.
            if ( canBeVisible )
            {
                // And we enabled by selected processes which can change during playback.
                hkBool32 isSelected;
                if ( m_processHandler->isProcessSelected( geom.getTag(), isSelected ).isFailure() )
                {
                    HK_VDB_SIGNAL_ERROR( 0xedb00116, hkVdbError::CMD_HANDLER_ERROR );
                    result.signalError( *m_processHandler );
                    continue;
                }
                canBeVisible = isSelected;
            }
        }

        //
        // Determine visibility flag bits based on add/remove frames.
        //
        // We playing fwd
        if ( startFrame < endFrame )
        {
            // We check for crossing over FrameAdded or FrameRemoved going forward
            // Note: order of checks matters
            if ( ( startFrame < frameRemoved ) &&
                ( frameRemoved <= endFrame ) )
            {
                m_updateGeomCmd.m_clearFlagBits.orWith( hkDisplayGeometryFlags::VISIBLE );
            }
            else if ( ( startFrame < frameAdded ) &&
                ( frameAdded <= endFrame ) )
            {
                m_updateGeomCmd.m_setFlagBits.orWith( canBeVisible * hkDisplayGeometryFlags::VISIBLE );
            }
            else
            {
                m_updateGeomCmd.m_clearFlagBits.orWith( !canBeVisible * hkDisplayGeometryFlags::VISIBLE );
                m_updateGeomCmd.m_setFlagBits.orWith( canBeVisible * hkDisplayGeometryFlags::VISIBLE );
            }
        }
        // We are playing bwd
        else if ( startFrame > endFrame )
        {
            // We check for crossing over FrameAdded or FrameRemoved going backward
            // Note: order of checks matters
            if ( ( startFrame >= frameAdded ) &&
                ( frameAdded > endFrame ) )
            {
                m_updateGeomCmd.m_clearFlagBits.orWith( hkDisplayGeometryFlags::VISIBLE );
            }
            else if ( ( startFrame >= frameRemoved ) &&
                ( frameRemoved > endFrame ) )
            {
                m_updateGeomCmd.m_setFlagBits.orWith( canBeVisible * hkDisplayGeometryFlags::VISIBLE );
            }
            else
            {
                m_updateGeomCmd.m_clearFlagBits.orWith( !canBeVisible * hkDisplayGeometryFlags::VISIBLE );
                m_updateGeomCmd.m_setFlagBits.orWith( canBeVisible * hkDisplayGeometryFlags::VISIBLE );
            }
        }
        // We are redoing a frame, we need to allow viewer changes to affect visibility
        else
        {
            m_updateGeomCmd.m_clearFlagBits.orWith( !canBeVisible * hkDisplayGeometryFlags::VISIBLE );
            m_updateGeomCmd.m_setFlagBits.orWith( canBeVisible * hkDisplayGeometryFlags::VISIBLE );
        }

        //
        // If we changed flags based on add/remove frame, ensure that we always signal.
        // This is because we can't determine changes from the cached "flags" since they aren't
        // the only factor that determines visibility.
        //
        const bool forceSetFlagBits = m_updateGeomCmd.m_setFlagBits.get();
        const bool forceClearFlagBits = m_updateGeomCmd.m_clearFlagBits.get();

        //
        // Update the geometry to reflect the endFrame.
        //
        if ( ( frameAdded <= endFrame ) && ( endFrame < frameRemoved ) )
        {
            // Update flags.
            if ( startFlags != flags )
            {
                hkDisplayGeometryFlags newSetFlags = flags; newSetFlags.andNotWith( startFlags );
                hkDisplayGeometryFlags newClearFlags = startFlags; newClearFlags.andNotWith( flags );
                m_updateGeomCmd.m_setFlagBits.orWith( newSetFlags );
                m_updateGeomCmd.m_clearFlagBits.orWith( newClearFlags );
            }

            // There are some properties we only set if we are visible.
            // This is only okay because we apply these every frame.
            if ( m_updateGeomCmd.m_clearFlagBits.noneIsSet( hkDisplayGeometryFlags::VISIBLE ) )
            {
                // Update vertices, we always keep the last valid ones.
                // We only update if there's actually a change (for performance reasons).
                // And by convention keep the last valid verts we have.
                const hkArray<hkVector4>* startVerts = geom.getVertices( startFrame );
                const hkArray<hkVector4>* endVerts = geom.getVertices( endFrame );
                if ( ( startVerts != endVerts ) && endVerts )
                {
                    hkArray<hkVector4>& newVertsToShare = const_cast< hkArray<hkVector4>& > ( *endVerts );
                    m_updateGeomCmd.m_newVertices.setDataUserFree( newVertsToShare.begin(), newVertsToShare.getSize(), newVertsToShare.getCapacity() );
                    m_updateGeomCmd.m_updateBits.orWith( UpdateGeometryCmd::VERTS );
                }

                // Update transforms.
                if ( const hkArray<hkTransform>* endParticleTransforms = geom.getParticleTransforms( endFrame ) )
                {
                    const hkArray<hkTransform>* startParticleTransforms = geom.getParticleTransforms( startFrame );
                    if ( startParticleTransforms != endParticleTransforms )
                    {
                        hkArray<hkTransform>& newParticleTransformsToShare = const_cast<hkArray<hkTransform>&> (*endParticleTransforms);
                        m_updateGeomCmd.m_newParticleTransforms.setDataUserFree( newParticleTransformsToShare.begin(), newParticleTransformsToShare.getSize(), newParticleTransformsToShare.getCapacity() );
                        m_updateGeomCmd.m_updateBits.orWith( UpdateGeometryCmd::PARTICLE_TRANSFORMS );
                    }
                }
                // Note: Particles are always updated by PARTICLE_TRANSFORMS, not individually
                else if ( flags.noneIsSet( hkDisplayGeometryFlags::PARTICLE_TEMPLATE ) )
                {
                    m_updateGeomCmd.m_transform = geom.getTransform( endFrame );
                    m_updateGeomCmd.m_updateBits.orWith( UpdateGeometryCmd::TRANSFORM );
                }

                // Update color changes.
                m_updateGeomCmd.m_color = geom.getColor( endFrame );
                m_updateGeomCmd.m_updateBits.orWith( UpdateGeometryCmd::COLOR );
            }
        }

        displayIds( geom, endFrame );

        // Determine if flags have changed
#ifdef HK_DEBUG
        hkDisplayGeometryFlags newFlags = startFlags;
        newFlags.orWith( m_updateGeomCmd.m_setFlagBits );
        newFlags.andNotWith( m_updateGeomCmd.m_clearFlagBits );
        HK_ASSERT( 0x22441086, flags.allAreSet( newFlags.get( hkDisplayGeometryFlags::VISIBLE ) ), "Failed to get proper flags" );
#endif
        m_updateGeomCmd.m_updateBits.orWith(
            ( forceSetFlagBits || !bool( startFlags.allAreSet( m_updateGeomCmd.m_setFlagBits ) ) ) * ( UpdateGeometryCmd::SET_FLAGS ) );
        m_updateGeomCmd.m_updateBits.orWith(
            ( forceClearFlagBits || bool( startFlags.anyIsSet( m_updateGeomCmd.m_clearFlagBits ) ) ) * ( UpdateGeometryCmd::CLEAR_FLAGS ) );

        // Signal
        if ( m_updateGeomCmd.m_updateBits.get() )
        {
            m_geometryCmdReceived.fire( m_updateGeomCmd, result );
        }
    }
}

void hkVdbDisplayHandler::displayIds( const hkVdbCachedGeometry& geom, int frame )
{
#ifdef DISPLAY_GEOM_IDS

    const hkUint64 uid = geom.getUid();
    const hkUint64 id = geom.getStreamId();

    hkTransform t;
    geom.getTransform( frame ).get( t );
    hkStringBuf idstr; idstr.printf( "[0x%x/%i]", uid, id );

    m_display3dTextCmd.m_cmdType = hkVdbCmdType::DISPLAY_3D_TEXT;
    m_display3dTextCmd.m_id = uid;
    m_display3dTextCmd.m_tag = 0;
    m_display3dTextCmd.m_color = hkColor::WHITE;
    m_display3dTextCmd.m_text = idstr;
    m_display3dTextCmd.m_position = t.getTranslation();

    hkVdbSignalResult result_;
    m_displayCmdReceived.fire( m_display3dTextCmd, result_ );
#endif
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
