// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Types/hkSignalSlots.h>

#include <VisualDebugger/VdbServices/System/Connection/hkVdbConnection.h>

struct hkVdbSignalResult;
class hkVdbClient;

//////////////////////////////////////////////////////////////////////////
/// Cmd handler types, see individual cmd handler classes for impl details.
/// Note: Enum value determines registration order which is significant
/// because hkSignals calls happen last-registered-to-first.
//////////////////////////////////////////////////////////////////////////
namespace hkVdbCmdHandlerType
{
    enum Enum
    {
        CUSTOM,
        STATS,
        FILE,
        TEXT,
        DISPLAY,
        OBJECT,
        PROCESS,
        PLAYBACK,
        SETUP,

        NUM_TYPES,
        INVALID = NUM_TYPES
    };
}

//////////////////////////////////////////////////////////////////////////
/// For advanced use only.
/// Base class for hkVdbCmdHandler implementations.
//////////////////////////////////////////////////////////////////////////
template<hkVdbCmdHandlerType::Enum ImplType>
class hkVdbCmdHandler : public hkVdbDefaultErrorReporter
{
public:
    HK_DECLARE_CLASS( hkVdbCmdHandler, New );
    hkVdbCmdHandler( hkLog::Origin* log ) : hkVdbDefaultErrorReporter( log ) {}

    /// Retrieve the type from a static context.
    static hkVdbCmdHandlerType::Enum getStaticType() { return ImplType; }

    /// Retrieve the type.
    virtual hkVdbCmdHandlerType::Enum getType() const { return ImplType; }

    /// Wait for the handler to complete any work that it is
    /// executing in a background thread (not currently used).
    virtual hkResult waitForCompletion() { m_waitForCompletion.fire(); return HK_SUCCESS; }

    /// Called to allow cmd handler to register itself with an hkVdbClient
    /// This is a good place to hook into signals, etc.
    virtual hkResult registerSelf( hkVdbClient& client ) = 0;

    /// Called to allow cmd handler to unregister itself with an hkVdbClient.
    /// This should effectively undo registerSelf(...).
    virtual hkResult unregisterSelf( hkVdbClient& client ) = 0;

    /// Called when a new connection has been established.
    /// Handler's should think about how they need to respond to a new connection,
    /// especially an APPLICATION connection which is where most reset their internal state.
    virtual void onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection ) = 0;

    HK_DECLARE_SIGNAL( WaitForCompletionSignal, hkSignal0 );
    /// Signal fired when wait for completetion finishes (not currently used).
    WaitForCompletionSignal m_waitForCompletion;

private:

    virtual void dtorFwd() { this->~hkVdbCmdHandler(); }
    virtual void reportErrorCustomFwd( hkUint32 id, hkVdbError::Enum error, const char* msg ) const
    { this->reportErrorCustom( id, error, msg ); }

    friend class hkVdbAnyCmdHandler;
};

//////////////////////////////////////////////////////////////////////////
/// Type representing any cmd handler.
/// Allows for storing hkVdbCmdHandlers in a uniform array of pointers, etc.
//////////////////////////////////////////////////////////////////////////
class hkVdbAnyCmdHandler : public hkVdbCmdHandler<hkVdbCmdHandlerType::INVALID>
{
public:
    ~hkVdbAnyCmdHandler();

    //
    // hkVdbCmdHandler interface
    //

    virtual hkResult waitForCompletion() HK_OVERRIDE;
    virtual hkResult registerSelf( hkVdbClient& client ) HK_OVERRIDE;
    virtual hkResult unregisterSelf( hkVdbClient& client ) HK_OVERRIDE;
    virtual void onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection ) HK_OVERRIDE;

    /// Get the name of the class of the actual cmd handler.
    const char* getClassName() const;

    /// Dynamically cast to the specified handler type (if we are of that type).
    template<typename HandlerType>
    HandlerType* dynCast();

private:
    virtual void reportErrorCustom( hkUint32 id, hkVdbError::Enum error, const char* msg ) const HK_OVERRIDE;
    void checkImpl()
    {
        // This is just to test that we don't have a new function that must be implemented
        HK_ON_DEBUG( hkVdbAnyCmdHandler a; )
    }
    // Cannot construct these
    hkVdbAnyCmdHandler();
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
