// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Types/hkSignalSlots.h>

#include <VisualDebugger/VdbServices/System/Connection/hkVdbConnection.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmd.h>

class hkVdbCmdInput;
class hkVdbCache;
struct hkVdbFrame;
class hkVdbStatsHandler;
class hkVdbObjectHandler;
class PersistableObjectRegistry;

//////////////////////////////////////////////////////////////////////////
/// Stores/combines persistent cmds when a frame is evicted from the hkVdbCmdInput.
/// This class is necessary to ensure that data in the vdb cmd input can be played from
/// the beginning, even after cmds are evicted.
//////////////////////////////////////////////////////////////////////////

class HK_EXPORT_VDB hkVdbPersistentStateCache : public hkReferencedObject
{
public:

    HK_DECLARE_CLASS( hkVdbPersistentStateCache, New );

    /// Create a persistent cache which stores important information needed for replaying/saving
    /// the provided vdb cmd input.
    hkVdbPersistentStateCache( hkVdbCmdInput& input );

    /// Create a deep clone which can be modified w/o impacting the source cache.
    hkRefNew<hkVdbPersistentStateCache> clone() const;

    // dtor
    ~hkVdbPersistentStateCache();

    //
    // Controls
    //

    /// Clear all data from the persistent cache.
    void clear();

    //
    // Advanced use
    //

    /// Persist frame data from the cmds in the frame.
    /// Some cmds require this method to persist.
    HK_ALWAYS_INLINE hkResult persistEvictedFrame( const hkVdbFrame& frame )
    { onFrameEvictedSignal( *m_input, frame ); return HK_SUCCESS; }

    /// Persist data by constructing cmds from a cached frame.
    /// Some cmds require this method to persist.
    hkResult persistCacheFrame( const hkVdbCache& cache, hkUint32 frame );

    //
    // Iteration
    //

    typedef struct _Iterator
    {
        HK_DECLARE_CLASS( _Iterator, New );
        _Iterator( const void* ptrIn ) : ptr( ptrIn ), stage( 0 ), cmdIndex( 0 ) {}
        const void* ptr;
        int stage;
        hkInt16 cmdIndex;
        hkVdbStorage storage;
    } Iterator;
    Iterator getIterator() const;
    Iterator getNext( Iterator& i ) const;
    const hkVdbCmd* getValue( Iterator& i ) const;
    hkBool32 isValid( Iterator& i ) const;

    //
    // Signals
    //

    HK_DECLARE_SIGNAL( PersistCacheFrameSignal, hkSignal4< const hkVdbCache&, hkUint32, hkVdbLocalOStream&, hkVdbSignalResult& > );
    /// Fires when persistCacheFrame is called; allowing listeners to add data to the output stream for persisting.
    PersistCacheFrameSignal m_persistCacheFrame;

    //
    // Internal use
    //
    void onFrameEvictedSignal( hkVdbCmdInput& input, const hkVdbFrame& frame );
    void onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection );

HK_VDB_ON_HASH_MAP( protected: )

    /// A doubly-linked-circular list node which points to a vdb command.
    /// This allows us to maintain the order of the vdb commands (which is critical) while still
    /// being able to remove command when they are no longer needed in the persistent state.
    struct VdbCmdNode
    {
        HK_DECLARE_CLASS( VdbCmdNode, New );

        VdbCmdNode() { init(); }

        ~VdbCmdNode();

        /// Insert cmdNode after this node in the doubly-linked-circular list.
        void insertAfter( VdbCmdNode* cmdNode );

        /// Insert cmdNode before this node in the doubly-linked-circular list.
        void insertBefore( VdbCmdNode* cmdNode );

        /// Reset this node so it can be re-inserted elsewhere.
        /// This function removes this node from the doubly-linked-circular list.
        void reset();

        /// Initialize the node. Use reset() to reuse an updated node.
        HK_INLINE void init()
        {
            m_cmd = HK_NULL;
            m_previous = this;
            m_next = this;
            m_referenceCount = 0;
            m_fixed = false;
        }

        /// Starting with this cmd node, go through all cmd nodes and delete them.
        HK_INLINE void deleteAllReferencedMemory()
        {
            VdbCmdNode* cmdNode = m_next;
            while ( cmdNode != this )
            {
                VdbCmdNode* nextNode = cmdNode->m_next;
                HK_ON_DEBUG( cmdNode->m_referenceCount = 0 ); // avoid assert while cleaning up
                delete cmdNode;
                cmdNode = nextNode;
            }
            m_referencesTo.clearAndDeallocate();
        }

        /// Check the consistency of the doubly-linked-circular list.
        void checkConsistency();

        hkVdbCmd* m_cmd;
        VdbCmdNode* m_previous;
        VdbCmdNode* m_next;
        hkArray<VdbCmdNode*> m_referencesTo;
        int m_referenceCount;
        hkBool32 m_fixed;
    };

HK_VDB_ON_DEP_MAP( protected: )

    void persistCommand( const hkVdbCmd& cmd );

    void deleteUsedMemory();

    VdbCmdNode* registerPersistentCommand(
        hkVdbCmdType::Enum type,
        hkUint64 id,
        const hkVdbCmd* cmd,
        hkBool32 evictPreviousCmd = true );

    void addPersistentCommandReference(
        VdbCmdNode* fromNode,
        hkVdbCmdType::Enum type,
        hkUint64 id );

    template<typename HANDLER_TYPE>
    void persistReadFormat( const hkRefPtr<HANDLER_TYPE>& handler, int protocol, Iterator& i ) const
    {
        if ( ( i.storage.m_data.getSize() == 0 ) && handler )
        {
            if ( hkPersistableReadFormat* persistableReadFormat = ( hkPersistableReadFormat* ) handler->m_readFormat.val() )
            {
                hkArrayStreamWriter writer( &i.storage.m_data, hkArrayStreamWriter::ARRAY_BORROW );

                
                //HK_ASSERT(
                //  0x22440849,
                //  input.getProtocol() >= hkProtocolVersion::SUPPORTS_CIRCULAR_BUFFER,
                //  "The persistent state cache doesn't write out old stats formats correctly" );

                HK_ON_DEBUG( hkResult result = ) hkVdbInternalHandler::writeUnpersistReadFormatCmd(
                    writer,
                    protocol,
                    persistableReadFormat->persist(),
                    handler->getType() );
                HK_ASSERT( 0x22441011, result.isSuccess(), "Couldn't write persistent read format data" );
                HK_ON_DEBUG( result = ) hkVdbInternalHandler::writeMetaData( writer, i.cmdIndex );
                HK_ASSERT( 0x22441013, result.isSuccess(), "Couldn't write meta data" );
            }
        }
    }

    template<typename HANDLER_TYPE>
    void unpersistReadFormat( hkRefPtr<HANDLER_TYPE>& handler, const hkArrayBase<hkInt8>& persistentState ) const
    {
        hkRefPtr<hkPersistableReadFormat> persistableReadFormat = hkRefNew<hkPersistableReadFormat>( new hkPersistableReadFormat() );
        persistableReadFormat->unpersist( persistentState, true );
        HK_ASSERT( 0x22441010, !handler, "Should only get one of these commands and it should be *before* any other dependent cmds" );
        handler = hkRefNew<HANDLER_TYPE>( new HANDLER_TYPE( true ) );
        handler->setLoadedReader( persistableReadFormat );
    }

    hkRefPtr<hkVdbCmdInput> m_input;
    hkRefPtr<hkVdbStatsHandler> m_persistableStatsHandler;
    hkArray<char> m_persistentCacheCmdsStorage;
    hkArray<char> m_persistentStepCmdStorage;
    VdbCmdNode m_persistentEvictedDataStart;
#ifdef HK_VDB_USE_HASH_MAP
    using IdToNodeMap = hkVdbMap<hkUint64, VdbCmdNode*>;
#else
public:
    struct IdToNodeMap : public hkVdbMap<hkUint64, VdbCmdNode*>
    {
        IdToNodeMap() {}
        IdToNodeMap( const IdToNodeMap& src ) :
            IdToNodeMap()
        {}
        void operator=( const IdToNodeMap& src ) {}
    };
protected:
#endif
    hkVdbMap<hkVdbCmdType::Enum, IdToNodeMap > m_persistentDataMap;
    HK_DEBUG_ONLY_MEMBER( int, m_protocol );
};

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
