// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbServices/hkVdbServices.h>
#include <VisualDebugger/VdbServices/System/Cache/hkVdbPersistentStateCache.h>

#include <Common/Visualize/Serialize/hkPersistableReadFormat.h>
#include <Common/Visualize/hkVisualDebuggerCmd.h>
#include <Common/Visualize/hkVisualDebuggerCmdType.h>

#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdInput.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbStatsHandler.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbObjectHandler.h>
#include <VisualDebugger/VdbServices/System/Command/Handlers/hkVdbInternalHandler.h>
#include <VisualDebugger/VdbServices/System/Connection/hkVdbConnection.h>
#include <VisualDebugger/VdbServices/System/Utils/hkVdbCmdUtils.h>

#ifndef HK_VDB_USE_HASH_MAP
#include <Common/Base/Container/PointerMap/hkMap.h>
#include <Common/Base/Container/PointerMap/hkMap.hxx>
template class HK_EXPORT_COMMON hkMapBase<hkUint64, hkVdbPersistentStateCache::VdbCmdNode*>;
template class HK_EXPORT_COMMON hkMap<hkUint64, hkVdbPersistentStateCache::VdbCmdNode*>;
template class HK_EXPORT_COMMON hkMapBase<hkVdbCmdType::Enum, hkVdbPersistentStateCache::IdToNodeMap >;
template class HK_EXPORT_COMMON hkMap<hkVdbCmdType::Enum, hkVdbPersistentStateCache::IdToNodeMap >;
#endif



//#define ENABLE_EXTRA_PERSISTENT_NODE_CONSISTENCY_CHECKS

namespace
{
    enum IterStages
    {
        EVICTED_CMDS = 0,
        CACHE_CMDS,
        PERSIST_STATS_MAP_READER,
        STEP_CMD,
        COMPLETED
    };
};

hkVdbPersistentStateCache::hkVdbPersistentStateCache( hkVdbCmdInput& input ) :
    m_input( &input )
    HK_ON_DEBUG( , m_protocol( -1 ) )
{
    using namespace hkVdbCmdType;
    using namespace hkVdbCmdUtils;

    HK_SUBSCRIBE_TO_SIGNAL( input.m_frameEvicted, this, hkVdbPersistentStateCache );

    // Store a step cmd, which we always return as the last cmd during persistent state cache iteration
    {
        hkVdbLocalOStream ostream( m_persistentStepCmdStorage );
        const int packetSize = s8 + sF;
        ostream.write32u( packetSize );
        ostream.write8u( STEP );
        ostream.writeFloat32( HK_FLOAT_MIN );
    }
}

hkRefNew<hkVdbPersistentStateCache> hkVdbPersistentStateCache::clone() const
{
    hkRefNew<hkVdbPersistentStateCache> clonedCache = hkRefNew<hkVdbPersistentStateCache>( new hkVdbPersistentStateCache( *m_input ) );

    // Feed frames from the current cache so that we have that data
    for ( hkVdbPersistentStateCache::Iterator persistentCmdIt = getIterator();
        isValid( persistentCmdIt );
        persistentCmdIt = getNext( persistentCmdIt ) )
    {
        const hkVdbCmd* cmd = getValue( persistentCmdIt );
        clonedCache->persistCommand( *cmd );
    }

    // Grab signal subscriptions
    clonedCache->m_persistCacheFrame += m_persistCacheFrame;

    return clonedCache;
}

hkVdbPersistentStateCache::~hkVdbPersistentStateCache()
{
    m_input->m_frameEvicted.unsubscribeAll( this );
    deleteUsedMemory();
}

void hkVdbPersistentStateCache::clear()
{
    deleteUsedMemory();
    m_persistentEvictedDataStart.init();
    HK_ON_DEBUG( m_protocol = -1; )
}

void hkVdbPersistentStateCache::deleteUsedMemory()
{
    m_persistentEvictedDataStart.deleteAllReferencedMemory();
    m_persistableStatsHandler = HK_NULL;
    m_persistentDataMap.clearAndDeallocate();
}

hkVdbPersistentStateCache::Iterator hkVdbPersistentStateCache::getIterator() const
{
    _Iterator it( m_persistentEvictedDataStart.m_next );
    HK_ASSERT( 0x22440982, !isValid( it ) || getValue( it )->getType() == hkVdbCmdType::SEND_SERVER_INFO, "The first cmd must set the protocol" );
    return it;
}

hkVdbPersistentStateCache::Iterator hkVdbPersistentStateCache::getNext( Iterator& i ) const
{
    if ( !isValid( i ) )
    {
        HK_ASSERT( 0x22440806, false, "Iterator is no longer valid" );
        return i;
    }

    // Current stage
    int stage = i.stage;

    // We are moving, clear any temp data created/used in getValue()
    i.storage.m_data.clear();

    // Determine next node/stage
    const void* nextPtr;
    int nextStage;
    switch ( i.stage )
    {
        case IterStages::EVICTED_CMDS:
        {
            const VdbCmdNode* node = reinterpret_cast< const VdbCmdNode* >( i.ptr );
            nextPtr = node->m_next;
            nextStage = stage;
            break;
        }
        case IterStages::CACHE_CMDS:
        {
            const hkVdbCmd* node = reinterpret_cast< const hkVdbCmd* >( i.ptr );
            nextPtr = reinterpret_cast< const hkVdbCmd* >( hkAddByteOffset( node, node->getCmdSize() ) );
            nextStage = stage;
            break;
        }
        case IterStages::PERSIST_STATS_MAP_READER:
        case IterStages::STEP_CMD:
        {
            nextPtr = HK_NULL;
            nextStage = ( stage + 1 );
            break;
        }
        default:
        {
            HK_ASSERT( 0x22440907, false, "Invalid iterator stage" );
            nextPtr = HK_NULL;
            nextStage = COMPLETED;
            break;
        }
    }

    // Validate next stage to see if we need to advance to a later stage
    switch ( nextStage )
    {
        case IterStages::EVICTED_CMDS:
        {
            // Check validity
            if ( nextPtr != &m_persistentEvictedDataStart )
            {
                break;
            }

            // If invalid, fall-through to validate next stage
            nextPtr = reinterpret_cast< const hkVdbCmd* >( m_persistentCacheCmdsStorage.begin() );
            nextStage = IterStages::CACHE_CMDS;
        }
        case IterStages::CACHE_CMDS:
        {
            // Check validity
            if ( nextPtr && ( nextPtr >= m_persistentCacheCmdsStorage.begin() ) && ( nextPtr < m_persistentCacheCmdsStorage.end() ) )
            {
                break;
            }

            // If invalid, fall-through to validate next stage
            nextPtr = HK_NULL;
            nextStage = IterStages::PERSIST_STATS_MAP_READER;
        }
        case IterStages::PERSIST_STATS_MAP_READER:
        {
            // Check validity
            if ( m_persistableStatsHandler )
            {
                break;
            }

            // If invalid, fall-through to validate next stage
            nextPtr = reinterpret_cast< const hkVdbCmd* >( m_persistentStepCmdStorage.begin() );
            nextStage = IterStages::STEP_CMD;
        }
        case IterStages::STEP_CMD:
        {
            // Check validity
            if ( nextPtr && ( nextPtr >= m_persistentStepCmdStorage.begin() ) && ( nextPtr < m_persistentStepCmdStorage.end() ) )
            {
                break;
            }

            // If invalid, fall-through to validate next stage
            nextPtr = HK_NULL;
            nextStage = IterStages::COMPLETED;
        }
        case IterStages::COMPLETED:
        {
            // Always valid
            break;
        }
        default:
        {
            HK_ASSERT_NO_MSG( 0x22440909, false );
            break;
        }
    }

    i.ptr = nextPtr;
    i.stage = nextStage;
    i.cmdIndex++;

    return i;
}

const hkVdbCmd* hkVdbPersistentStateCache::getValue( Iterator& i ) const
{
    using namespace hkVdbCmdType;

    HK_ASSERT( 0x22440808, isValid( i ), "Iterator is no longer valid" );

    switch ( i.stage )
    {
        case IterStages::EVICTED_CMDS:
        {
            return reinterpret_cast< const VdbCmdNode* >( i.ptr )->m_cmd;
        }
        case IterStages::CACHE_CMDS:
        {
            return reinterpret_cast< const hkVdbCmd* >( i.ptr );
        }
        case IterStages::PERSIST_STATS_MAP_READER:
        {
            persistReadFormat( m_persistableStatsHandler, m_input->getProtocol(), i );
            return reinterpret_cast< const hkVdbCmd* >( i.storage.m_data.begin() );
        }
        case IterStages::STEP_CMD:
        {
            return reinterpret_cast< const hkVdbCmd* >( i.ptr );
        }
        case IterStages::COMPLETED:
        {
            return HK_NULL;
        }
    }

    HK_ASSERT( 0x22440908, false, "Invalid iterator stage" );
    return HK_NULL;
}

hkBool32 hkVdbPersistentStateCache::isValid( Iterator& i ) const
{
    return
        // We have not yet completed.
        ( i.stage != COMPLETED ) &&
        // We have data to iterate over, if we don't have data then
        // the first iterator we return will fail this condition.
        ( i.ptr != &m_persistentEvictedDataStart );
}

hkResult hkVdbPersistentStateCache::persistCacheFrame( const hkVdbCache& cache, hkUint32 frame )
{
    // We can't persist the state of early command protocol frames without a lot of extra work.
    
    
    if ( m_input->getProtocol() < hkProtocolVersion::SUPPORTS_CIRCULAR_BUFFER )
    {
        return HK_FAILURE;
    }

    m_persistentCacheCmdsStorage.clear();
    hkVdbLocalOStream writer( m_persistentCacheCmdsStorage );

    
    
    

    hkVdbSignalResult result;
    m_persistCacheFrame.fire( cache, frame, writer, result );
    return result;
}

void hkVdbPersistentStateCache::onFrameEvictedSignal( hkVdbCmdInput& input, const hkVdbFrame& frame )
{
    // We can't persist the state of early command protocol frames without a lot of extra work.
    
    
    if ( input.getProtocol() < hkProtocolVersion::SUPPORTS_CIRCULAR_BUFFER )
    {
        return;
    }

    HK_ASSERT( 0x22440920, m_input == &input, "Should not be changing input after ctor" );

    for ( hkVdbFrame::Iterator iter = frame.getIterator();
        frame.isValid( iter );
        iter = frame.getNext( iter ) )
    {
        const hkVdbCmd* cmd = frame.getValue( iter );
        persistCommand( *cmd );
    }
}

void hkVdbPersistentStateCache::onConnectedSignal( hkVdbConnectionUse::Enum use, hkVdbConnection& connection )
{
    if ( use == hkVdbConnectionUse::APPLICATION )
    {
        clear();
    }
}

namespace
{
    using namespace hkVdbCmdType;

    static hkVdbCmdType::Enum GeometryCommands[] =
    {
        ADD_GEOMETRY,
        ADD_GEOMETRY_EX,
        INSTANCE_GEOMETRY,
        UPDATE_GEOMETRY,
        UPDATE_GEOMETRY_TRANSFORM,
        UPDATE_GEOMETRY_TRANSFORM_EX,
        UPDATE_PARTICLE_TRANSFORMS,
        UPDATE_GEOMETRY_VERTICES,
        SET_GEOMETRY_COLOR,
        SET_GEOMETRY_ALPHA,
        SET_GEOMETRY_FLAG_BITS,
        CLEAR_GEOMETRY_FLAG_BITS,
        REMOVE_GEOMETRY,
        // Deprecated
        DEPRECATED_ADD_RIGID_GEOMETRY,
        DEPRECATED_ADD_RIGID_GEOMETRY_EX,
        DEPRECATED_ADD_MORPHING_GEOMETRY,
        DEPRECATED_ADD_MORPHING_GEOMETRY_EX,
        DEPRECATED_ADD_FIXED_GEOMETRY,
        DEPRECATED_ADD_FIXED_GEOMETRY_EX
    };

    static hkVdbCmdType::Enum SourceGeometryCommands[] =
    {
        ADD_GEOMETRY,
        ADD_GEOMETRY_EX,
        // Note: the server does allow adding a geometry instance *from* another instance.
        INSTANCE_GEOMETRY,
        REMOVE_GEOMETRY,
        // Deprecated
        DEPRECATED_ADD_RIGID_GEOMETRY,
        DEPRECATED_ADD_RIGID_GEOMETRY_EX,
        DEPRECATED_ADD_MORPHING_GEOMETRY,
        DEPRECATED_ADD_MORPHING_GEOMETRY_EX,
        DEPRECATED_ADD_FIXED_GEOMETRY,
        DEPRECATED_ADD_FIXED_GEOMETRY_EX
    };

    // You may need to update the above arrays with any new geometry commands
    // OR you may need to update the processing of persistent commands (see persistentCommand)
    
    
    HK_COMPILE_TIME_ASSERT( hkVdbCmdType::MAX_SERVER_TO_CLIENT_CMD == 50 );
}

void hkVdbPersistentStateCache::persistCommand( const hkVdbCmd& cmd )
{
    using namespace hkVdbCmdType;

    if ( !hkVdbCmdUtils::isPersistentCmd( cmd ) )
    {
        return;
    }

    // Note: not all persistent commands have ids
    hkUint64 id;
    hkUint64 referenceId;
    hkVdbCmdUtils::extractUniqueIds( cmd, id, referenceId );

#ifdef HK_DEBUG
    if ( cmd.getType() == hkVdbCmdType::SEND_SERVER_INFO )
    {
        hkVdbLocalIStream protocolReader( cmd.getData(), cmd.getDataSize() );
        m_protocol = protocolReader.read32();
    }
    else
    {
        // If this assert hits, it means that when our persistent state is read back later
        // it will output commands that the caller won't know how to patch types on.
        HK_ASSERT( 0x22440981, m_protocol != -1, "Protocol version has not been persisted yet" );
    }
#endif

    // Process internal cmds (should only happen when saving from a saved file)
    
    if ( HK_VERY_UNLIKELY( cmd.getType() == CLIENT_INTERNAL ) )
    {
        if ( id == hkVdbInternalCmdType::UNPERSIST_READ_FORMAT )
        {
            hkVdbCmdHandlerType::Enum handlerType;
            hkArray<hkInt8> persistentState;
            if ( hkVdbInternalHandler::readUnpersistReadFormatCmd(
                cmd,
                m_input->getProtocol(),
                persistentState,
                handlerType ).isFailure() )
            {
                HK_ASSERT( 0x22441014, false, "Could not read UNPERSIST_READ_FORMAT cmd" );
            }

            switch ( handlerType )
            {
                case hkVdbCmdHandlerType::STATS:
                {
                    unpersistReadFormat( m_persistableStatsHandler, persistentState );
                    break;
                }
                default:
                {
                    HK_ASSERT( 0x22441015, false, "Do not recognize persistent read format's handler target" );
                }
            }
        }
        else
        {
            registerPersistentCommand( cmd.getType(), id, &cmd );
        }
    }
    // Process the evicted stats maps commands with our internal stats handler
    // to collect read format types, these don't get stored directly in the normal command stream.
    else if ( cmd.getType() == SEND_STATS_MAPS )
    {
        if ( !m_persistableStatsHandler ) m_persistableStatsHandler = hkRefNew<hkVdbStatsHandler>( new hkVdbStatsHandler( true ) );
        hkVdbLocalIStream cmdStream( cmd.getData(), cmd.getDataSize() );
        m_persistableStatsHandler->processStatsMapsCmd( hkUint32( -1 ), cmd, m_input->getProtocol(), cmdStream );
    }
    // Purge persistent storage for display commands for this id.
    else if ( cmd.getType() == REMOVE_GEOMETRY )
    {
        for ( int i = 0; i < HK_COUNT_OF( GeometryCommands ); i++ )
        {
            registerPersistentCommand( GeometryCommands[i], id, HK_NULL );
        }
    }
    // Need to keep all set/clear flag bits commands, they are stateful, reliant on implementation defaults,
    // and potentially order-sensitive with other cmds.
    else if ( ( cmd.getType() == SET_GEOMETRY_FLAG_BITS ) || ( cmd.getType() == CLEAR_GEOMETRY_FLAG_BITS ) )
    {
        registerPersistentCommand( cmd.getType(), id, &cmd, false );
    }
    // Default registration of the persistent command.
    else
    {
        VdbCmdNode* registeredNode = registerPersistentCommand( cmd.getType(), id, &cmd );

        // Track dependencies from instance commands
        if ( cmd.getType() == INSTANCE_GEOMETRY )
        {
            for ( int i = 0; i < HK_COUNT_OF( SourceGeometryCommands ); i++ )
            {
                addPersistentCommandReference( registeredNode, SourceGeometryCommands[i], referenceId );
            }
        }
    }
}

hkVdbPersistentStateCache::VdbCmdNode* hkVdbPersistentStateCache::registerPersistentCommand(
    hkVdbCmdType::Enum type,
    hkUint64 id,
    const hkVdbCmd* cmd,
    hkBool32 evictPreviousCmd )
{
    hkVdbMap<hkVdbCmdType::Enum, IdToNodeMap >::Iterator infoIt =
        m_persistentDataMap.findOrInsertKey( type, IdToNodeMap() );

    IdToNodeMap& info = m_persistentDataMap.getValue( infoIt );
    IdToNodeMap::Iterator cmdIt = info.findOrInsertKey( id, HK_NULL );

    hkBool32 evicted = false;
    VdbCmdNode* oldNode = info.getValue( cmdIt );
    if ( oldNode && evictPreviousCmd )
    {
        // We will be removing from map or we will be evicting in which case this will toggle back to true.
        oldNode->m_fixed = false;

        // Evict former resident if it has no references.
        if ( oldNode->m_referenceCount == 0 )
        {
            oldNode->reset();
            evicted = true;
        }
    }

    // Set new resident
    if ( cmd )
    {
        // Lazily create node if needed
        VdbCmdNode* newNode = ( evicted ) ? oldNode : new VdbCmdNode();

        // Copy our cmd
        
        
        
        
        newNode->m_cmd = cmd->copy();

        // Effectively inserts at the "end" of the doubly linked list
        m_persistentEvictedDataStart.insertBefore( newNode );

        // Update info map
        newNode->m_fixed = true;
        info.setValue( cmdIt, newNode );

        return newNode;
    }
    else
    {
        info.remove( cmdIt );

        if ( evicted )
        {
            delete oldNode;
        }

        return HK_NULL;
    }
}

void hkVdbPersistentStateCache::addPersistentCommandReference(
    VdbCmdNode* fromNode,
    hkVdbCmdType::Enum type,
    hkUint64 id )
{
    hkVdbMap<hkVdbCmdType::Enum, IdToNodeMap >::Iterator infoIt =
        m_persistentDataMap.findOrInsertKey( type, IdToNodeMap() );

    IdToNodeMap& info = m_persistentDataMap.getValue( infoIt );

    if ( VdbCmdNode* toNode = info.getWithDefault( id, HK_NULL ) )
    {
        fromNode->m_referencesTo.pushBack( toNode );
        toNode->m_referenceCount++;
    }
}

hkVdbPersistentStateCache::VdbCmdNode::~VdbCmdNode()
{
    HK_ASSERT( 0x22440852, m_referenceCount == 0, "Deleting a referenced node" );

    if ( m_cmd )
    {
        m_cmd->destroyCopy();
        HK_ON_DEBUG( m_cmd = HK_NULL );
    }

#ifdef HK_DEBUG
    m_previous = HK_NULL;
    m_next = HK_NULL;
#endif
}

void hkVdbPersistentStateCache::VdbCmdNode::insertAfter( VdbCmdNode* cmdNode )
{
    VdbCmdNode* next = m_next;

    // Update me
    m_next = cmdNode;

    // Update incoming cmd
    cmdNode->m_previous = this;
    cmdNode->m_next = next;

    // Update next
    next->m_previous = cmdNode;

    checkConsistency();
}

void hkVdbPersistentStateCache::VdbCmdNode::insertBefore( VdbCmdNode* cmdNode )
{
    VdbCmdNode* previous = m_previous;

    // Update me
    m_previous = cmdNode;

    // Update incoming cmd
    cmdNode->m_previous = previous;
    cmdNode->m_next = this;

    // Update previous
    previous->m_next = cmdNode;

    checkConsistency();
}

void hkVdbPersistentStateCache::VdbCmdNode::reset()
{
    HK_ASSERT( 0x22440851, m_referenceCount == 0, "Resetting a referenced node" );

    // Cleanup references
    {
        for ( int i = 0; i < m_referencesTo.getSize(); i++ )
        {
            VdbCmdNode* referencedNode = m_referencesTo[i];
            referencedNode->m_referenceCount--;
            if ( ( referencedNode->m_referenceCount == 0 ) && !( referencedNode->m_fixed ) )
            {
                // Delayed eviction, this node is no longer referenced.
                referencedNode->reset();
                delete referencedNode;
            }
        }
        m_referencesTo.clear();
    }

    // Cleanup cmd
    if ( m_cmd )
    {
        m_cmd->destroyCopy();
        m_cmd = HK_NULL;
    }

    // Evict me from linked list
    m_previous->m_next = m_next;
    m_next->m_previous = m_previous;

#ifdef HK_DEBUG_SLOW
    m_previous->checkConsistency();
    m_next->checkConsistency();
#endif

    m_previous = this;
    m_next = this;
}

void hkVdbPersistentStateCache::VdbCmdNode::checkConsistency()
{
#if defined(HK_DEBUG) || defined(ENABLE_EXTRA_PERSISTENT_NODE_CONSISTENCY_CHECKS)
    VdbCmdNode* cmdNode = m_next;
    HK_ASSERT_NO_MSG( 0x22440831, cmdNode );
#endif
#ifdef ENABLE_EXTRA_PERSISTENT_NODE_CONSISTENCY_CHECKS
    while ( cmdNode != this )
    {
        if ( hkVdbCmd* cmd = cmdNode->m_cmd )
        {
            HK_ASSERT_NO_MSG( 0x22440832, cmd->isOk() );
        }

        VdbCmdNode* nextNode = cmdNode->m_next;
        HK_ASSERT_NO_MSG( 0x22440834, nextNode );
        HK_ASSERT_NO_MSG( 0x22440835, nextNode->m_previous == cmdNode );
        cmdNode = nextNode;
    }
#endif
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
