// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#pragma once

#include <Common/Base/Container/BitField/hkBitField.h>
#include <Common/Base/Types/Color/hkColor.h>
#include <Common/Base/Types/hkSignalSlots.h>

#include <Common/Visualize/Shape/hkDisplayGeometryFlags.h>
#include <Common/Visualize/hkVisualDebuggerCmd.h>
#include <Common/Visualize/hkServerObjectHandler.h>

#include <VisualDebugger/VdbServices/System/Connection/hkVdbConnection.h>
#include <VisualDebugger/VdbServices/System/Command/hkVdbCmdInput.h>

struct hkVdbFrame;
class hkVdbCmdInput;
typedef hkArray<hkInt8> hkAnyArray;
class hkVdbCache;



//////////////////////////////////////////////////////////////////////////
/// A cinfo describing cached entry to be created.
//////////////////////////////////////////////////////////////////////////
struct HK_EXPORT_VDB hkVdbCachedEntryCinfo
{
    HK_DECLARE_CLASS( hkVdbCachedEntryCinfo, New );

    hkVdbCachedEntryCinfo(
        int tag,
        hkUint64 streamId,
        hkUint32 frameAdded ) :
        m_tag( tag ),
        m_streamId( streamId ),
        m_frameAdded( frameAdded ),
        m_frameRemoved( hkUint32( -1 ) )
    {}

    /// The tag for the viewer/process which created this entry.
    int m_tag;

    /// The original id from the vdb cmd.
    hkUint64 m_streamId;

    /// The frame this entry was added.
    hkUint32 m_frameAdded;

    /// The frame this entry was removed (hkUint32(-1) if entry hasn't been removed).
    hkUint32 m_frameRemoved;
};

//////////////////////////////////////////////////////////////////////////
/// A base class providing shared functionality for any cached entry.
//////////////////////////////////////////////////////////////////////////
class HK_EXPORT_VDB hkVdbCachedEntryBase : protected hkVdbCachedEntryCinfo
{
public:
    HK_DECLARE_CLASS( hkVdbCachedEntryBase, New );

    /// Returns a unique id for this entry which won't conflict with another entry of the same type having the same streamId.
    /// Any number of removed entries can have the same streamId as other entries.
    hkUint64 getUid() const
    {
        return hkVdbCmdId::computePersistentId( m_frameAdded, m_streamId );
    }

    /// Return the original streamId for the entry.
    hkUint64 getStreamId() const { return m_streamId; }

    /// Return the streamId from the uid.
    static hkUint64 getStreamId( hkUint64 uid )
    {
        return hkVdbCmdId::extractStreamId( uid );
    }

    /// Return the process tag that created this entry.
    int getTag() const { return m_tag; }

    /// Returns if the entry is present on the given frame.
    hkBool32 isAdded( hkUint32 frame ) const { return ( ( m_frameAdded <= frame ) && ( frame < m_frameRemoved ) ); }

    /// Get the frame this entry was added.
    hkUint32 getFrameAdded() const { return m_frameAdded; }

    /// Get the frame this entry was removed (hkUint32(-1) if entry hasn't been removed).
    hkUint32 getFrameRemoved() const { return m_frameRemoved; }

    /// Register that the entry was removed on a particular frame.
    void registerRemoved( hkUint32 frame );

protected:

    hkVdbCachedEntryBase(
        hkVdbCmdInput& input,
        int entryIdx,
        hkVdbCachedEntryCinfo& cinfo ) :
        hkVdbCachedEntryCinfo( cinfo ),
        m_input( input ),
        m_entryIndex( entryIdx ),
        m_stateFlags( 0 )
    {}
    hkVdbCachedEntryBase(
        hkVdbCmdInput& input,
        int entryIdx,
        int tag,
        hkUint64 streamId,
        hkUint32 frameAdded ) :
        hkVdbCachedEntryCinfo( tag, streamId, frameAdded ),
        m_input( input ),
        m_entryIndex( entryIdx ),
        m_stateFlags( 0 )
    {}
    virtual ~hkVdbCachedEntryBase() {}

    /// Create sparse data for the cached entry.
    /// This is data which we don't place in the vdb frame's custom byte input (to be allocated every frame).
    template<typename T>
    T* createSparseData() const
    {
        
        T* sparseData = new T();
        m_sparseData.insert( sparseData, sparseData );
        return sparseData;
    }

    /// Get our custom frame bytes for the specified id
    hkInt8* accessCustomFrameBytes( hkUint32 id, hkUint32 frame, int elementSize ) const;

    /// Get our custom frame bytes for the specified id
    template<typename T>
    T* accessCustomFrameBytes( hkUint32 id, hkUint32 frame ) const
    {
        return reinterpret_cast< T* >( accessCustomFrameBytes( id, frame, sizeof( T ) ) );
    }

    template<typename T>
    const T& getCachedValue( hkUint32 id, hkUint32 frame, const T& initialValue ) const
    {
        if ( frame <= m_frameAdded )
        {
            return initialValue;
        }
        else if ( T* cachedValue = accessCustomFrameBytes<T>( id, frame ) )
        {
            return *cachedValue;
        }
        else if ( frame >= m_frameRemoved )
        {
            if ( T* finalCachedValue = accessCustomFrameBytes<T>( id, m_frameRemoved ) )
            {
                return *finalCachedValue;
            }
        }

        HK_ASSERT( 0x22441199, false, "Entry invalid for frame" );
        return initialValue;
    }

    
    
    
    
    
    

    template<typename T>
    void registerCachedValueUpdate( hkUint32 id, hkUint32 frame, T& initialValue, const T& value ) const
    {
        if ( frame == m_frameAdded )
        {
            initialValue = value;
        }
        else if ( T* cachedValue = accessCustomFrameBytes<T>( id, frame ) )
        {
            *cachedValue = value;
        }
        else
        {
            HK_ASSERT( 0x22440818, false, "No space reserved to store update" );
        }
    }

    template<typename T>
    void registerCachedFlagsSet( hkUint32 id, hkUint32 frame, T& initialFlags, const T& flagBits ) const
    {
        if ( frame == m_frameAdded )
        {
            initialFlags.orWith( flagBits );
        }
        else if ( T::StorageType* cachedFlags = accessCustomFrameBytes<T::StorageType>( id, frame ) )
        {
            ( *cachedFlags ) |= ( flagBits );
        }
        else
        {
            HK_ASSERT( 0x22441030, false, "No space reserved to store update" );
        }
    }

    template<typename T>
    void registerCachedFlagsClear( hkUint32 id, hkUint32 frame, T& initialFlags, const T& flagBits ) const
    {
        if ( frame == m_frameAdded )
        {
            initialFlags.andNotWith( flagBits );
        }
        else if ( T::StorageType* cachedFlags = accessCustomFrameBytes<T::StorageType>( id, frame ) )
        {
            ( *cachedFlags ) &= ~( flagBits );
        }
        else
        {
            HK_ASSERT( 0x22441030, false, "No space reserved to store update" );
        }
    }

    template<typename T>
    T* registerCachedSparseValueUpdate( hkUint32 id, hkUint32 frame, T*& initialValuePtr ) const
    {
        if ( frame == m_frameAdded )
        {
            if ( !initialValuePtr ) initialValuePtr = createSparseData<T>();
            return initialValuePtr;
        }
        else if ( T** cachedValues = accessCustomFrameBytes<T*>(
            id,
            frame ) )
        {
            T*& cachedValuesPtr = ( *cachedValues );
            cachedValuesPtr = createSparseData<T>();
            return cachedValuesPtr;
        }
        else
        {
            HK_ASSERT( 0x22440820, false, "No space reserved to store update" );
            return HK_NULL;
        }
    }

    void registerCachedSparseVarUpdate( hkUint32 id, hkUint32 frame, hkReflect::Var& initialVar, const hkReflect::Var& updatedVar ) const
    {
        if ( frame == m_frameAdded )
        {
            HK_ASSERT( 0x22441155, !m_sparseData.contains( initialVar.getAddress() ), "Var address has already been registered" );
            initialVar = updatedVar;
        }
        else if ( hkReflect::Var* cachedVars = accessCustomFrameBytes<hkReflect::Var>(
            id,
            frame ) )
        {
            ( *cachedVars ) = updatedVar;
            HK_VERIFY( 0x22441156, !updatedVar || m_sparseData.insert( updatedVar.getAddress(), updatedVar ), "Var address has already been registered" );
        }
        else
        {
            HK_ASSERT( 0x22440820, false, "No space reserved to store update" );
        }
    }

    /// Flags indicating various states of the entry.
    enum StateFlags
    {
        /// The entry is currently mapped to the streamId in the cache map.
        ENTRY_OWNS_STREAM_ID = 1 << 0,
    };

    hkVdbCmdInput& m_input;
    int m_entryIndex;
    hkFlags<StateFlags, hkInt8> m_stateFlags;
    
    
    mutable hkVdbMap<const void*, hkReflect::Var> m_sparseData;

    template<typename CachedEntryType>
    friend class hkVdbCachedEntries;
};

template<typename CachedEntryType>
class hkVdbCachedEntries;

//////////////////////////////////////////////////////////////////////////
/// A concrete cached entry base class of CachedEntryType.
/// Cached entry classes should inherit from this with their type
/// as a template parameter.
//////////////////////////////////////////////////////////////////////////
template<typename CachedEntryType>
class hkVdbCachedEntry : public hkVdbCachedEntryBase
{
public:
    HK_DECLARE_CLASS( hkVdbCachedEntry, New );

protected:

    /// Delete previously created sparse data.
    template<typename T>
    void deleteSparseData( T* sparseData )
    {
        
        if ( sparseData )
        {
            hkVdbMap<const void*, hkReflect::Var>::Iterator iter = m_sparseData.findKey( sparseData );
            HK_ASSERT_NO_MSG( 0x22440847, m_sparseData.isValid( iter ) );
            if ( m_sparseData.isValid( iter ) )
            {
                hkReflect::Var var = m_sparseData.getValue( iter );
                m_owner.m_sparseDataDeallocated.fire( m_owner, *static_cast<CachedEntryType*>( this ), var );
                var.destroy();
                m_sparseData.remove( iter );
            }
        }
    }

    hkVdbCachedEntry(
        hkVdbCachedEntries<CachedEntryType>& owner,
        hkVdbCmdInput& input,
        int entryIdx,
        hkVdbCachedEntryCinfo& cinfo ) :
        hkVdbCachedEntryBase( input, entryIdx, cinfo ),
        m_owner( owner )
    {}
    hkVdbCachedEntry(
        hkVdbCachedEntries<CachedEntryType>& owner,
        hkVdbCmdInput& input,
        int entryIdx,
        int tag,
        hkUint64 streamId,
        hkUint32 frameAdded ) :
        hkVdbCachedEntryBase( input, entryIdx, tag, streamId, frameAdded ),
        m_owner( owner )
    {}
    ~hkVdbCachedEntry()
    {
        for ( hkVdbMap<const void*, hkReflect::Var>::Iterator iter = m_sparseData.getIterator();
            m_sparseData.isValid( iter );
            iter = m_sparseData.getNext( iter ) )
        {
            hkReflect::Var& var = m_sparseData.getValue( iter );
            m_owner.m_sparseDataDeallocated.fire( m_owner, *static_cast<CachedEntryType*>( this ), var );
            var.destroy();
        }
    }

    hkVdbCachedEntries<CachedEntryType>& m_owner;

    friend class hkVdbCachedEntries<CachedEntryType>;
};

//////////////////////////////////////////////////////////////////////////
/// A container for cached entries of CachedEntryType.
/// Cached entry container classes should inherit from this with their
/// contained entry type as a template parameter.
//////////////////////////////////////////////////////////////////////////
template<typename CachedEntryType>
class hkVdbCachedEntries : public hkReferencedObject
{
public:

    HK_DECLARE_CLASS( hkVdbCachedEntries, New );

    hkVdbCachedEntries( hkVdbCache& owner, hkVdbCmdInput& input );
    ~hkVdbCachedEntries();

    /// Get the total number of cached entries.
    int getNumEntries() { return m_entries.getSize(); }

    template<typename CachedEntryCinfoType>
    CachedEntryType& createEntry( CachedEntryCinfoType& cinfo );

    /// Access a previously created "live" entry by streamId.
    /// This is a hash-table lookup.
    
    
    CachedEntryType* accessEntry( hkUint64 streamId );

    /// Search for a particular entry by its uid.
    /// This is a linear lookup.
    CachedEntryType* findEntryByUid( hkUint64 uid );

    /// Clear all cached entries.
    void clear();

    //
    // Iteration
    //

    typedef int Iterator;
    Iterator getIterator() const;
    Iterator getNext( Iterator i ) const;
    Iterator getPrevious( Iterator i ) const;
    const CachedEntryType* getValue( Iterator i ) const;
    hkBool32 isValid( Iterator i ) const;

    //
    // Signals
    //

    HK_DECLARE_SIGNAL( EntryDeallocatedSignal, hkSignal2< const hkVdbCachedEntries<CachedEntryType>&, const CachedEntryType& > );
    /// Is fired when an entry is deallocated (see documentation for more details).
    EntryDeallocatedSignal m_entryDeallocated;

    HK_DECLARE_SIGNAL( SparseDataDeallocatedSignal, hkSignal3< const hkVdbCachedEntries<CachedEntryType>&, const CachedEntryType&, const hkReflect::Var > );
    /// Is fired when an entrie's sparse data is deallocated (see documentation for more details).
    SparseDataDeallocatedSignal m_sparseDataDeallocated;

    //
    // Internal use
    //

    void onPreCustomStorageAllocSignal( const hkVdbFrame& frame, hkArray<hkVdbCmdInput::CustomStorageReserveInfo>::Temp& reserveInfos );
    void onPostCustomStorageAllocSignal( const hkVdbFrame& frame );
    void onPreCustomStorageDeallocSignal( const hkVdbFrame& frame );
    void markEntry( CachedEntryType& entry )
    {
        m_markedEntryIds.pushBack( entry.getUid() );
        m_markedEntries.pushBack( &entry );
    }

protected:

    Iterator getNextInternal( Iterator i ) const;
    void reserveCustomStorage( hkArray<hkVdbCmdInput::CustomStorageReserveInfo>::Temp& reserveInfos, int totalNumEntries );
    void processMarkedEntries( const hkVdbFrame& frame, hkArray<hkUint64>& markedEntryIds, hkArray<CachedEntryType*>& markedEntries ) {}
    void initializeCustomStorage( const hkVdbFrame& frame, hkArray<CachedEntryType*>& newEntries, int totalNumEntries );
    void cleanupCustomSparseStorage( const hkVdbFrame& frame, CachedEntryType* entry );

    hkVdbCache& m_owner;
    hkRefPtr<hkVdbCmdInput> m_input;
    hkArray<CachedEntryType*> m_entries;
    hkArray<CachedEntryType*> m_newEntries;
    hkBitField m_indicesPool;
    hkVdbMap<hkUint64, CachedEntryType*> m_entriesByStreamId;
    int m_numEntriesLastFrame;

private:
    hkArray<hkUint64> m_markedEntryIds;
    hkArray<CachedEntryType*> m_markedEntries;
};

//////////////////////////////////////////////////////////////////////////
/// A cinfo describing a cached geometry.
//////////////////////////////////////////////////////////////////////////
struct HK_EXPORT_VDB hkVdbCachedGeometryCinfo : public hkVdbCachedEntryCinfo
{
    HK_DECLARE_CLASS( hkVdbCachedGeometryCinfo, New );

    hkVdbCachedGeometryCinfo(
        int tag = -1,
        hkUint64 streamId = 0,
        hkUint32 frameAdded = 0,
        const hkMatrix4& initialTransform = hkMatrix4::getIdentity(),
        const hkColor::Argb initialColor = hkColor::NONE,
        const hkDisplayGeometryFlags initialFlags = hkDisplayGeometryFlags::DEFAULT ) :
        hkVdbCachedEntryCinfo( tag, streamId, frameAdded ),
        m_initialTransform( initialTransform ),
        m_initialColor( initialColor ),
        m_initialFlags( initialFlags )
    {}

    /// The initial transform for the geometry.
    hkMatrix4 m_initialTransform;

    /// The initial color for the geometry.
    hkColor::Argb m_initialColor;

    /// The initial flags for the geometry.
    hkDisplayGeometryFlags m_initialFlags;
};

//////////////////////////////////////////////////////////////////////////
/// A concrete cached entry representing geometry.
//////////////////////////////////////////////////////////////////////////
class HK_EXPORT_VDB hkVdbCachedGeometry : public hkVdbCachedEntry<hkVdbCachedGeometry>
{
public:
    HK_DECLARE_CLASS( hkVdbCachedGeometry, New );

    //
    // Query
    //

    /// Get the transform of this geometry at a particular frame.
    const hkMatrix4& getTransform( hkUint32 frame ) const;

    /// Get the color of this geometry at a particular frame.
    hkColor::Argb getColor( hkUint32 frame ) const;

    /// Gets the flags for this geometry at a particular frame.
    hkDisplayGeometryFlags getFlags( hkUint32 frame ) const;

    /// Get the vertices of this geometry at a particular frame (if any).
    const hkArray<hkVector4>* getVertices( hkUint32 frame ) const;

    /// Get the particle transforms of this geometry at a particular frame (if any).
    const hkArray<hkTransform>* getParticleTransforms( hkUint32 frame ) const;

    //
    // Registration/Updating
    //

    /// Register that the transform was updated on a particular frame.
    void registerTransformUpdate( hkUint32 frame, const hkMatrix4& transform );

    /// Register that the color was updated on a particular frame.
    void registerColorUpdate( hkUint32 frame, hkColor::Argb color );

    /// Register set flag bits on a particular frame.
    void registerSetFlagBits( hkUint32 frame, hkDisplayGeometryFlags flagBits );

    /// Register clear flag bits on a particular frame.
    void registerClearFlagBits( hkUint32 frame, hkDisplayGeometryFlags flagBits );

    /// Register that the vertices were updated on a particular frame.
    void registerVerticesUpdate( hkUint32 frame, hkArray<hkVector4>& newVertsToStore );

    /// Register that the particle transforms were updated on a particular frame.
    void registerParticleTransformsUpdate( hkUint32 frame, hkArray<hkTransform>& newTransformsToStore );

    //
    // For use by deprecated cmds
    //

    void deprecatedPostInit(
        int tag,
        const hkMatrix4& transform,
        hkColor::Argb color,
        hkDisplayGeometryFlags flags )
    {
        HK_ASSERT( 0x22441358, tag == -1, "Already initialized" );
        m_tag = tag;
        m_initialTransform = transform;
        m_initialColor = color;
        m_initialFlags = flags;
    }

protected:

    hkVdbCachedGeometry(
        hkVdbCachedGeometryCinfo& cinfo,
        hkVdbCachedEntries<hkVdbCachedGeometry>& owner,
        hkVdbCmdInput& input,
        int geometryCachedIndex ) :
        hkVdbCachedEntry<hkVdbCachedGeometry>( owner, input, geometryCachedIndex, cinfo ),
        m_initialTransform( cinfo.m_initialTransform ),
        m_initialColor( cinfo.m_initialColor ),
        m_initialFlags( cinfo.m_initialFlags ),
        m_initialVertices( HK_NULL ),
        m_initialParticleTransforms( HK_NULL )
    {}
    ~hkVdbCachedGeometry() {}

    hkMatrix4 m_initialTransform;
    hkColor::Argb m_initialColor;
    hkDisplayGeometryFlags m_initialFlags;
    
    hkArray<hkVector4>* m_initialVertices;
    hkArray<hkTransform>* m_initialParticleTransforms;

    friend class hkVdbCachedEntries<hkVdbCachedGeometry>;
};

//////////////////////////////////////////////////////////////////////////
/// A container of hkVdbCachedGeometry entries.
//////////////////////////////////////////////////////////////////////////
using hkVdbCachedGeometries = hkVdbCachedEntries<hkVdbCachedGeometry>;

//////////////////////////////////////////////////////////////////////////
/// A cinfo describing a cached object.
//////////////////////////////////////////////////////////////////////////
struct HK_EXPORT_VDB hkVdbCachedObjectCinfo : public hkVdbCachedEntryCinfo
{
    HK_DECLARE_CLASS( hkVdbCachedObjectCinfo, New );

    hkVdbCachedObjectCinfo(
        int tag = -1,
        hkUint64 streamId = 0,
        hkUint32 frameAdded = 0,
        hkReflect::Var data = hkReflect::Var(),
        hkObjectFlags flags = hkObjectFlags::DEFAULT,
        hkArray<hkUint64>* connections = HK_NULL ) :
        hkVdbCachedEntryCinfo( tag, streamId, frameAdded ),
        m_initialData( data ),
        m_flags( flags )
    {}

    /// Initial object data
    hkReflect::Var m_initialData;

    /// Object flags
    hkObjectFlags m_flags;

    /// Initial connections
    hkArray<hkUint64> m_initialConnections;
};

//////////////////////////////////////////////////////////////////////////
/// A concrete cached entry representing cached object data.
//////////////////////////////////////////////////////////////////////////
class HK_EXPORT_VDB hkVdbCachedObject : public hkVdbCachedEntry<hkVdbCachedObject>
{
public:
    HK_DECLARE_CLASS( hkVdbCachedObject, New );

    //
    // Query
    //

    /// Get the data for this object at a particular frame.
    const hkReflect::Var getData( hkUint32 frame ) const;

    /// Get the flags for this cached object
    hkObjectFlags getFlags() const { return m_flags; }

    /// Get the connections for this object at a particular frame.
    const hkArray<hkUint64>* getConnections( hkUint32 frame ) const;

    //
    // Registration/Updating
    //

    /// Register that data has been initialized on a particular frame.
    void registerDataAdded( hkUint32 frame, int tag, const hkVdbReflect::Var& dataToStore, hkObjectFlags flags );

    /// Register that the data was updated on a particular frame.
    void registerDataUpdate( hkUint32 frame, const hkVdbReflect::Var& dataToStore );

    /// Register that the data was removed
    void registerDataRemoved( hkUint32 frame );

    /// Register new connections for a particular frame.
    void registerConnect( hkUint32 frame, const hkArrayView<hkUint64>& connections, int tag, hkLocalArray<hkUint64>& addedConnectionsOut );

    /// Register new connections for a particular frame.
    void registerDisconnect( hkUint32 frame, const hkArrayView<hkUint64>& connections, hkLocalArray<hkUint64>& removedConnectionsOut );

    /// Register full disconnect on a particular frame.
    void registerDisconnectAll( hkUint32 frame, hkConnectivityFlags flags, hkLocalArray<hkUint64>& removedIncomingConnectionsOut, hkLocalArray<hkUint64>& removedOutgoingConnectionsOut );

protected:

    hkVdbCachedObject(
        hkVdbCachedObjectCinfo& cinfo,
        hkVdbCachedEntries<hkVdbCachedObject>& owner,
        hkVdbCmdInput& input,
        int objectCachedIndex ) :
        hkVdbCachedEntry<hkVdbCachedObject>( owner, input, objectCachedIndex, cinfo ),
        m_initialData( cinfo.m_initialData ),
        m_flags( cinfo.m_flags ),
        m_initialConnections( cinfo.m_initialConnections )
    {}
    ~hkVdbCachedObject()
    {
        m_initialData.destroy();
    }

    hkArray<hkUint64>* getOrCreateFrameConnections( hkUint32 frame );

    void updateParentConnections( hkUint64 parentId, const hkArrayBase<hkUint64>* addedChildIds, const hkArrayBase<hkUint64>* removedChildIds );

    void removeIfAbandoned();

    void checkConnectionConsistency();

    /// Initial object data
    hkReflect::Var m_initialData;

    /// Object flags
    hkObjectFlags m_flags;

    /// Initial connections
    hkArray<hkUint64> m_initialConnections;

    /// Parent connections
    hkArray<hkUint64> m_parentConnections;

    friend class hkVdbCachedEntries<hkVdbCachedObject>;
};

//////////////////////////////////////////////////////////////////////////
/// A container of hkVdbCachedObject entries.
//////////////////////////////////////////////////////////////////////////
using hkVdbCachedObjects = hkVdbCachedEntries<hkVdbCachedObject>;

//////////////////////////////////////////////////////////////////////////
/// A cache of entries.
/// Some of the entries' data lives in the vdb input back-end, this class (and hkVdbCachedEntry) simplifies
/// the retrieving and storage of extra information to be able to quickly zip around the data.
//////////////////////////////////////////////////////////////////////////
class HK_EXPORT_VDB hkVdbCache : public hkReferencedObject
{
public:

    HK_DECLARE_CLASS( hkVdbCache, New );

    /// Create a new cache with the provided vdb input back-end.
    hkVdbCache( hkVdbCmdInput& input );

    /// Access to cached geometries
    hkVdbCachedGeometries& geometries() const { return *m_geometriesCache; }

    /// Access to cached objects
    hkVdbCachedObjects& objects() const { return *m_objectsCache; }

    /// Clear the cache.
    void clear();

protected:

    hkRefPtr<hkVdbCachedGeometries> m_geometriesCache;
    hkRefPtr<hkVdbCachedObjects> m_objectsCache;
};


template<typename CachedEntryType>
template<typename CachedEntryCinfoType>
CachedEntryType& hkVdbCachedEntries<CachedEntryType>::createEntry( CachedEntryCinfoType& cinfo )
{
#ifdef HK_DEBUG
    if ( CachedEntryType* e = accessEntry( cinfo.m_streamId ) )
    {
        HK_ASSERT(
            0x217089c5,
            // Unfortunately, we do some hackery when saving the persistent state to preserve entries that rely on other entries.
            // We end up compressing several frames worth of data into frame zero and so this can trigger in that case.
            ( cinfo.m_frameAdded == 0 ) ||
            // Otherwise, we shouldn't be getting an add if we haven't removed the geometry already.
            ( e->m_frameRemoved <= cinfo.m_frameAdded ),
            "Creating an entry for a streamId which is already in use" );
    }
#endif

    int idx = m_entries.getSize();

    // Check our index pool for an available index slot
    if ( m_indicesPool.getSize() )
    {
        const int availableIdx = m_indicesPool.findFirstSet();
        if ( availableIdx != -1 )
        {
            m_indicesPool.clear( availableIdx );
            idx = availableIdx;
        }
        else
        {
            m_indicesPool.clearStorage();
        }
    }

    CachedEntryType* cachedEntry = new CachedEntryType(
        cinfo,
        *this,
        *m_input,
        idx );

    // Place our new entry in the correct place in the array
    if ( idx < m_entries.getSize() )
    {
        HK_ASSERT( 0x22440823, m_entries[idx] == HK_NULL, "Slot is occupied" );
        m_entries[idx] = cachedEntry;
    }
    else
    {
        m_entries.pushBack( cachedEntry );
    }

    // Branchless update to m_entriesByStreamId and state flags
    {
        hkVdbMap<hkUint64, CachedEntryType*>::Iterator iter = m_entriesByStreamId.findOrInsertKey( cinfo.m_streamId, cachedEntry );
        m_entriesByStreamId.getValue( iter )->m_stateFlags.clear( hkVdbCachedEntryBase::ENTRY_OWNS_STREAM_ID );
        cachedEntry->m_stateFlags.orWith( hkVdbCachedEntryBase::ENTRY_OWNS_STREAM_ID );
        m_entriesByStreamId.setValue( iter, cachedEntry );
    }

    // Note new geometry for later processing
    m_newEntries.pushBack( cachedEntry );

    return *cachedEntry;
}

template<typename CachedEntryType>
CachedEntryType* hkVdbCachedEntries<CachedEntryType>::accessEntry( hkUint64 streamId )
{
    return m_entriesByStreamId.getWithDefault( streamId, HK_NULL );
}

template<typename CachedEntryType>
CachedEntryType* hkVdbCachedEntries<CachedEntryType>::findEntryByUid( hkUint64 uid )
{
    for ( int i = 0; i < m_entries.getSize(); ++i )
    {
        hkVdbCachedEntryBase* e = m_entries[i];
        if ( e && ( e->getUid() == uid ) )
        {
            return static_cast<CachedEntryType*>( e );
        }
    }
    return HK_NULL;
}

template<typename CachedEntryType>
typename hkVdbCachedEntries<CachedEntryType>::Iterator hkVdbCachedEntries<CachedEntryType>::getIterator() const
{
    return getNextInternal( -1 );
}

template<typename CachedEntryType>
typename hkVdbCachedEntries<CachedEntryType>::Iterator hkVdbCachedEntries<CachedEntryType>::getNext( Iterator i ) const
{
    HK_ASSERT( 0x22440859, isValid( i ), "Iterator is no longer valid" );
    return getNextInternal( i );
}

template<typename CachedEntryType>
typename hkVdbCachedEntries<CachedEntryType>::Iterator hkVdbCachedEntries<CachedEntryType>::getNextInternal( Iterator i ) const
{
    i++;
    while ( ( i < m_entries.getSize() ) && !m_entries[i] ) i++;
    return i;
}

template<typename CachedEntryType>
typename hkVdbCachedEntries<CachedEntryType>::Iterator hkVdbCachedEntries<CachedEntryType>::getPrevious( Iterator i ) const
{
    HK_ASSERT( 0x22440860, isValid( i ), "Iterator is no longer valid" );
    i--;
    while ( ( i >= 0 ) && !m_entries[i] ) i--;
    return i;
}

template<typename CachedEntryType>
const CachedEntryType* hkVdbCachedEntries<CachedEntryType>::getValue( Iterator i ) const
{
    HK_ASSERT_NO_MSG( 0x22440861, m_entries[i] );
    return m_entries[i];
}

template<typename CachedEntryType>
hkBool32 hkVdbCachedEntries<CachedEntryType>::isValid( Iterator i ) const
{
    return ( ( i >= 0 ) && ( i < m_entries.getSize() ) );
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
