// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbServices/hkVdbServices.h>
#include <VisualDebugger/VdbServices/System/Cache/hkVdbCache.h>

#include <Common/Base/Algorithm/Sort/hkSort.h>
#include <Common/Base/Container/LocalArray/hkLocalBuffer.h>

#ifndef HK_VDB_USE_HASH_MAP
#include <Common/Base/Container/PointerMap/hkMap.h>
#include <Common/Base/Container/PointerMap/hkMap.hxx>
template class HK_EXPORT_COMMON hkMapBase<const void*, hkReflect::Var>;
template class HK_EXPORT_COMMON hkMap<const void*, hkReflect::Var>;
template class HK_EXPORT_COMMON hkMapBase<hkUint64, hkVdbCachedGeometry*>;
template class HK_EXPORT_COMMON hkMap<hkUint64, hkVdbCachedGeometry*>;
template class HK_EXPORT_COMMON hkMapBase<hkUint64, hkVdbCachedObject*>;
template class HK_EXPORT_COMMON hkMap<hkUint64, hkVdbCachedObject*>;
#endif



//#define ASSERT_NO_DUPS
//#define ASSERT_CONNECTION_CONSISTENCY

namespace
{
    
    // Geometry
    static const hkUint32 HK_VDB_TRANSFORM_CACHE_ID = 1;
    static const hkUint32 HK_VDB_COLOR_CACHE_ID = 2;
    static const hkUint32 HK_VDB_FLAGS_CACHE_ID = 3;
    static const hkUint32 HK_VDB_VERTEX_CACHE_ID = 4;
    static const hkUint32 HK_VDB_PARTICLE_TRANSFORM_CACHE_ID = 5;

    // Object
    static const hkUint32 HK_VDB_OBJECT_DATA_CACHE_ID = 101;
    static const hkUint32 HK_VDB_OBJECT_CONNECTIVITY_CACHE_ID = 102;

    void assertNoDups( hkArray<hkUint64>& results )
    {
#if defined(ASSERT_NO_DUPS) && defined(HK_DEBUG)
        hkVdbSet<hkUint64> ids; ids.reserve( results.getSize() );
        for ( int i = 0; i < results.getSize(); i++ )
        {
            hkUint64 id = results[i];
            HK_ASSERT( 0x0, ids.insert( id ), "Dup found" );
        }
#endif
    }

    // Could use a set, but there should be relatively few connections.
    // So this is probably faster in most cases (overall).
    hkBool32 addFrameConnections( hkArray<hkUint64>& resultInOut, const hkArrayView<hkUint64>& toAdd, hkLocalArray<hkUint64>* addedOut = HK_NULL )
    {
        hkBool32 added = false;
        if ( addedOut ) addedOut->reserve( addedOut->getSize() + toAdd.getSize() );
        for ( int i = 0; i < toAdd.getSize(); i++ )
        {
            hkUint64 id = toAdd[i];
            int idx = hkAlgorithm::findInsertionIndex( id, resultInOut.begin(), resultInOut.getSize(), hkAlgorithm::less<hkUint64>() );

            // Skip duplicates
            if ( ( idx < resultInOut.getSize() ) && ( resultInOut[idx] == id ) )
            {
                continue;
            }

            // Insert
            resultInOut.insertAt( idx, id );
            if ( addedOut ) addedOut->pushBack( id );
            added = true;
        }
        assertNoDups( resultInOut );
        return added;
    }
    hkBool32 subtractFrameConnections( hkArray<hkUint64>& resultInOut, const hkArrayView<hkUint64>& toSubtract, hkLocalArray<hkUint64>* subtractedOut = HK_NULL )
    {
        hkBool32 subtracted = false;
        if ( resultInOut.getSize() )
        {
            if ( subtractedOut ) subtractedOut->reserve( subtractedOut->getSize() + toSubtract.getSize() );
            for ( int i = 0; i < toSubtract.getSize(); i++ )
            {
                hkUint64 id = toSubtract[i];
                int idx = hkAlgorithm::binarySearch( id, resultInOut.begin(), resultInOut.getSize(), hkAlgorithm::less<hkUint64>() );

                // Remove if found
                if ( idx != -1 )
                {
                    resultInOut.removeAtAndCopy( idx );
                    if ( subtractedOut ) subtractedOut->pushBack( id );
                    subtracted = true;
                }
            }
            assertNoDups( resultInOut );
        }
        return subtracted;
    }

    // Steal ownership of the array's contents, but leave both pointing to the same data.
    template<typename T>
    void stealArrayOwnership( hkArray<T>& newOwner, hkArray<T>& oldOwner )
    {
        HK_ASSERT_NO_MSG( 0x22441296, 0 == ( oldOwner.getCapacityAndFlags()&hkArrayBase<T>::DONT_DEALLOCATE_FLAG ) || oldOwner.begin() == HK_NULL );
        newOwner.clearAndDeallocate();
        newOwner._setDataUnchecked( oldOwner.begin(), oldOwner.getSize(), oldOwner.getCapacityAndFlags() );
        oldOwner._setDataUnchecked( oldOwner.begin(), oldOwner.getSize(), oldOwner.getCapacityAndFlags() | hkArrayBase<T>::DONT_DEALLOCATE_FLAG );
#if 0
        // hkArray::steal()
        HK_ASSERT_NO_MSG( 0x0, 0 == ( a.m_capacityAndFlags&hkArrayBase<T>::DONT_DEALLOCATE_FLAG ) || a.m_data == HK_NULL );
        clearAndDeallocate();
        hkArrayBase<T>::m_data = a.m_data;
        hkArrayBase<T>::m_size = a.m_size;
        hkArrayBase<T>::m_capacityAndFlags = a.m_capacityAndFlags;
        a.m_capacityAndFlags |= hkArrayBase<T>::DONT_DEALLOCATE_FLAG;
#endif
    }
}

//////////////////////////////////////////////////////////////////////////
// hkVdbCachedEntryBase
//////////////////////////////////////////////////////////////////////////

#pragma region hkVdbCachedEntryBase

void hkVdbCachedEntryBase::registerRemoved( hkUint32 frame )
{
    HK_ASSERT( 0x4f1aea8e, ( m_frameRemoved == hkUint32( -1 ) ) || ( m_frameRemoved == frame ), "Removing a cached entry twice" );
    m_frameRemoved = frame;
}

hkInt8* hkVdbCachedEntryBase::accessCustomFrameBytes( hkUint32 id, hkUint32 frameNum, int elementSize ) const
{
    hkInt8* cachePtr = HK_NULL;

    
    
    
    

    if ( const hkVdbFrame* frame = m_input.getFrame( frameNum ) )
    {
        // Look for per-frame storage.
        int numCustomBytes;
        if ( hkInt8* customBytes = frame->getCustomBytes( id, numCustomBytes ) )
        {
            const int startingByteIndex = ( m_entryIndex * elementSize );
            if ( startingByteIndex < numCustomBytes )
            {
                cachePtr = &customBytes[startingByteIndex];
            }
        }
    }

    
    
    
    return cachePtr;
}

#pragma endregion

//////////////////////////////////////////////////////////////////////////
// hkVdbCachedEntries
//////////////////////////////////////////////////////////////////////////

#pragma region hkVdbCachedEntries

template<typename CachedEntryType>
hkVdbCachedEntries<CachedEntryType>::hkVdbCachedEntries(
    hkVdbCache& owner,
    hkVdbCmdInput& input ) :
    m_owner( owner ),
    m_input( &input ),
    m_numEntriesLastFrame( 0 )
{
    HK_SUBSCRIBE_TO_SIGNAL( input.m_preCustomStorageAlloc, this, hkVdbCachedEntries<CachedEntryType> );
    HK_SUBSCRIBE_TO_SIGNAL( input.m_postCustomStorageAlloc, this, hkVdbCachedEntries<CachedEntryType> );
    HK_SUBSCRIBE_TO_SIGNAL( input.m_preCustomStorageDealloc, this, hkVdbCachedEntries<CachedEntryType> );
}

template<typename CachedEntryType>
hkVdbCachedEntries<CachedEntryType>::~hkVdbCachedEntries()
{
    for ( int i = 0; i < m_entries.getSize(); i++ )
    {
        if ( CachedEntryType* entry = m_entries[i] )
        {
            m_entryDeallocated.fire( *this, *entry );
            delete entry;
        }
    }
    m_input->m_preCustomStorageAlloc.unsubscribeAll( this );
    m_input->m_postCustomStorageAlloc.unsubscribeAll( this );
    m_input->m_preCustomStorageDealloc.unsubscribeAll( this );
}

template<typename CachedEntryType>
void hkVdbCachedEntries<CachedEntryType>::clear()
{
    for ( int i = 0; i < m_entries.getSize(); i++ )
    {
        if ( CachedEntryType* entry = m_entries[i] )
        {
            m_entryDeallocated.fire( *this, *entry );
            delete entry;
        }
    }
    m_input->unreserveCustomStorage( HK_VDB_TRANSFORM_CACHE_ID );
    m_input->unreserveCustomStorage( HK_VDB_COLOR_CACHE_ID );
    m_input->unreserveCustomStorage( HK_VDB_FLAGS_CACHE_ID );
    m_input->unreserveCustomStorage( HK_VDB_VERTEX_CACHE_ID );
    m_input->unreserveCustomStorage( HK_VDB_PARTICLE_TRANSFORM_CACHE_ID );
    m_entries.clear();
    m_newEntries.clear();
    m_entriesByStreamId.clear();
    m_indicesPool.clearStorage();
    m_numEntriesLastFrame = 0;
}

template<typename CachedEntryType>
void hkVdbCachedEntries<CachedEntryType>::onPreCustomStorageAllocSignal( const hkVdbFrame& frame, hkArray<hkVdbCmdInput::CustomStorageReserveInfo>::Temp& reserveInfos )
{
    // Reserve more space if needed
    if ( m_numEntriesLastFrame != m_entries.getSize() )
    {
        reserveCustomStorage( reserveInfos, m_entries.getSize() );
        m_numEntriesLastFrame = m_entries.getSize();
    }
    // Process any marked entries
    if ( m_markedEntries.getSize() )
    {
        processMarkedEntries( frame, m_markedEntryIds, m_markedEntries );
        m_markedEntryIds.clear();
        m_markedEntries.clear();
    }
}

template<>
void hkVdbCachedEntries<hkVdbCachedGeometry>::reserveCustomStorage( hkArray<hkVdbCmdInput::CustomStorageReserveInfo>::Temp& reserveInfos, int numEntries )
{
    hkVdbCmdInput::CustomStorageReserveInfo* reserveInfosPtr = reserveInfos.expandBy( 5 );

    reserveInfosPtr[0].m_id = HK_VDB_TRANSFORM_CACHE_ID;
    reserveInfosPtr[0].m_bytes = ( numEntries * sizeof( hkMatrix4 ) );
    reserveInfosPtr[0].m_initOp = hkVdbCmdInput::COPY_FROM_PREVIOUS;

    reserveInfosPtr[1].m_id = HK_VDB_COLOR_CACHE_ID;
    reserveInfosPtr[1].m_bytes = ( numEntries * sizeof( hkColor::Argb ) );
    reserveInfosPtr[1].m_initOp = hkVdbCmdInput::COPY_FROM_PREVIOUS;

    reserveInfosPtr[2].m_id = HK_VDB_FLAGS_CACHE_ID;
    reserveInfosPtr[2].m_bytes = ( numEntries * sizeof( hkDisplayGeometryFlags::StorageType ) );
    reserveInfosPtr[2].m_initOp = hkVdbCmdInput::COPY_FROM_PREVIOUS;

    reserveInfosPtr[3].m_id = HK_VDB_VERTEX_CACHE_ID;
    reserveInfosPtr[3].m_bytes = ( numEntries * sizeof( hkArray<hkVector4>* ) );
    reserveInfosPtr[3].m_initOp = hkVdbCmdInput::COPY_FROM_PREVIOUS;

    reserveInfosPtr[4].m_id = HK_VDB_PARTICLE_TRANSFORM_CACHE_ID;
    reserveInfosPtr[4].m_bytes = (numEntries * sizeof( hkArray<hkTransform>* ));
    reserveInfosPtr[4].m_initOp = hkVdbCmdInput::COPY_FROM_PREVIOUS;
}

template<>
void hkVdbCachedEntries<hkVdbCachedObject>::reserveCustomStorage( hkArray<hkVdbCmdInput::CustomStorageReserveInfo>::Temp& reserveInfos, int numEntries )
{
    hkVdbCmdInput::CustomStorageReserveInfo* reserveInfosPtr = reserveInfos.expandBy( 2 );

    reserveInfosPtr[0].m_id = HK_VDB_OBJECT_DATA_CACHE_ID;
    reserveInfosPtr[0].m_bytes = ( numEntries * sizeof( hkReflect::Var ) );
    reserveInfosPtr[0].m_initOp = hkVdbCmdInput::COPY_FROM_PREVIOUS;

    reserveInfosPtr[1].m_id = HK_VDB_OBJECT_CONNECTIVITY_CACHE_ID;
    reserveInfosPtr[1].m_bytes = ( numEntries * sizeof( hkArray<hkUint64>* ) );
    reserveInfosPtr[1].m_initOp = hkVdbCmdInput::COPY_FROM_PREVIOUS;
}

template<typename CachedEntryType>
void hkVdbCachedEntries<CachedEntryType>::onPostCustomStorageAllocSignal( const hkVdbFrame& frame )
{
    // Copy initial data into frame input for new entries
    if ( m_newEntries.getSize() )
    {
        initializeCustomStorage( frame, m_newEntries, m_entries.getSize() );
        m_newEntries.clear();
    }
}

template<>
void hkVdbCachedEntries<hkVdbCachedGeometry>::initializeCustomStorage( const hkVdbFrame& frame, hkArray<hkVdbCachedGeometry*>& newEntries, int totalNumEntries )
{
    hkMatrix4* cachedTransforms = frame.getCustomObjects<hkMatrix4>(
        HK_VDB_TRANSFORM_CACHE_ID,
        m_entries.getSize() );

    hkColor::Argb* cachedColors = frame.getCustomObjects<hkColor::Argb>(
        HK_VDB_COLOR_CACHE_ID,
        m_entries.getSize() );

    hkDisplayGeometryFlags::StorageType* cachedFlags = frame.getCustomObjects<hkDisplayGeometryFlags::StorageType>(
        HK_VDB_FLAGS_CACHE_ID,
        m_entries.getSize() );

    hkArray<hkVector4>** cachedVertices = frame.getCustomObjects<hkArray<hkVector4>*>(
        HK_VDB_VERTEX_CACHE_ID,
        m_entries.getSize() );

    hkArray<hkTransform>** cachedParticleTransforms = frame.getCustomObjects<hkArray<hkTransform>*>(
        HK_VDB_PARTICLE_TRANSFORM_CACHE_ID,
        m_entries.getSize() );

    HK_ASSERT( 0x22440824, cachedTransforms && cachedColors && cachedVertices && cachedParticleTransforms, "Cache was not properly setup after onFrameCompletedSignal" );

    for ( int i = 0; i < newEntries.getSize(); i++ )
    {
        if ( hkVdbCachedGeometry* geometry = newEntries[i] )
        {
            const int idx = geometry->m_entryIndex;
            cachedTransforms[idx] = geometry->m_initialTransform;
            cachedColors[idx] = geometry->m_initialColor;
            cachedFlags[idx] = geometry->m_initialFlags;
            cachedVertices[idx] = geometry->m_initialVertices;
            cachedParticleTransforms[idx] = geometry->m_initialParticleTransforms;
        }
    }
}

template<>
void hkVdbCachedEntries<hkVdbCachedObject>::initializeCustomStorage( const hkVdbFrame& frame, hkArray<hkVdbCachedObject*>& newEntries, int totalNumEntries )
{
    hkReflect::Var* cachedData = frame.getCustomObjects<hkReflect::Var>(
        HK_VDB_OBJECT_DATA_CACHE_ID,
        m_entries.getSize() );

    hkArray<hkUint64>** cachedConnectivity = frame.getCustomObjects<hkArray<hkUint64>*>(
        HK_VDB_OBJECT_CONNECTIVITY_CACHE_ID,
        m_entries.getSize() );

    HK_ASSERT( 0x22441153, cachedData && cachedConnectivity, "Cache was not properly setup after onFrameCompletedSignal" );

    for ( int i = 0; i < newEntries.getSize(); i++ )
    {
        if ( hkVdbCachedObject* object = newEntries[i] )
        {
            const int idx = object->m_entryIndex;
            hkString::memSet4( &cachedData[idx], 0, sizeof( hkReflect::Var ) >> 2 );
            cachedData[idx] = object->m_initialData;
            cachedConnectivity[idx] = &object->m_initialConnections;
        }
    }
}

template<typename CachedEntryType>
void hkVdbCachedEntries<CachedEntryType>::onPreCustomStorageDeallocSignal( const hkVdbFrame& frame )
{
    const hkUint32 deallocatedFrameNumber = frame.getFrameNumber();

    hkBool32 shrinking = true;
    m_indicesPool.resize( 0, m_entries.getSize() );
    for ( int i = m_entries.getSize() - 1; i >= 0; i-- )
    {
        CachedEntryType* cachedEntry = m_entries[i];

        hkBool32 maybeDeleteSparseData =
            ( cachedEntry &&
            ( cachedEntry->m_frameAdded <= deallocatedFrameNumber ) );

        hkBool32 deallocateCachedEntry =
            ( cachedEntry &&
            ( cachedEntry->m_frameRemoved <= deallocatedFrameNumber ) );

        // Potentially delete sparse data.
        // If we are deallocating the entire entry, this gets cleaned up in the cached entry dtor.
        if ( maybeDeleteSparseData && !deallocateCachedEntry )
        {
            cleanupCustomSparseStorage( frame, cachedEntry );
        }

        // Potentially deallocate entire cached entry
        if ( deallocateCachedEntry )
        {
            HK_ASSERT( 0x22440826,
                cachedEntry->m_frameRemoved == deallocatedFrameNumber,
                "We missed a frame somewhere, which could lead to other major problems" );

            if ( cachedEntry->m_stateFlags.anyIsSet( hkVdbCachedEntryBase::ENTRY_OWNS_STREAM_ID ) )
            {
                HK_ASSERT( 0x22440827, m_entriesByStreamId.getWithDefault( cachedEntry->getStreamId(), HK_NULL ) == cachedEntry, "Flag not up-to-date" );
                m_entriesByStreamId.remove( cachedEntry->getStreamId() );
            }
#ifdef HK_DEBUG
            else
            {
                HK_ASSERT( 0x22440828, m_entriesByStreamId.getWithDefault( cachedEntry->getStreamId(), HK_NULL ) != cachedEntry, "Flag not up-to-date" );
            }
#endif

            m_entryDeallocated.fire( *this, *cachedEntry );
            delete cachedEntry;
            cachedEntry = HK_NULL;

            if ( shrinking )
            {
                m_entries.setSize( i );
            }
            else
            {
                m_entries[i] = HK_NULL;
            }
        }

        // Update our available indices
        m_indicesPool.assign( i, !cachedEntry );
        shrinking &= !cachedEntry;
    }

    if ( shrinking )
    {
        m_indicesPool.clearStorage();
    }
#ifdef HK_DEBUG_SLOW
    else
    {
        for ( int i = 0; i < m_entries.getSize(); i++ )
        {
            HK_ASSERT_NO_MSG( 0x22440829, !m_entries[i] == m_indicesPool.get( i ) );
        }
    }
#endif
}

template<>
void hkVdbCachedEntries<hkVdbCachedGeometry>::cleanupCustomSparseStorage( const hkVdbFrame& frame, hkVdbCachedGeometry* cachedGeometry )
{
    const hkUint32 deallocatedFrameNumber = frame.getFrameNumber();

    const hkArray<hkVector4>* deallocatedVertices = cachedGeometry->getVertices( deallocatedFrameNumber );
    const hkArray<hkVector4>* vertices = cachedGeometry->getVertices( deallocatedFrameNumber + 1 );
    if ( deallocatedVertices != vertices )
    {
        cachedGeometry->deleteSparseData( deallocatedVertices );
    }

    const hkArray<hkTransform>* deallocatedParticleTransforms = cachedGeometry->getParticleTransforms( deallocatedFrameNumber );
    const hkArray<hkTransform>* particleTransforms = cachedGeometry->getParticleTransforms( deallocatedFrameNumber + 1 );
    if ( deallocatedParticleTransforms != particleTransforms )
    {
        cachedGeometry->deleteSparseData( deallocatedParticleTransforms );
    }
}

template<>
void hkVdbCachedEntries<hkVdbCachedObject>::cleanupCustomSparseStorage( const hkVdbFrame& frame, hkVdbCachedObject* cachedObject )
{
    const hkUint32 deallocatedFrameNumber = frame.getFrameNumber();

    hkReflect::Var deallocatedVar = cachedObject->getData( deallocatedFrameNumber );
    const hkReflect::Var var = cachedObject->getData( deallocatedFrameNumber + 1 );
    if ( ( deallocatedVar != var ) && ( deallocatedVar != cachedObject->m_initialData ) )
    {
        cachedObject->deleteSparseData( deallocatedVar.getAddress() );
    }

    const hkArray<hkUint64>* deallocatedConnections = cachedObject->getConnections( deallocatedFrameNumber );
    const hkArray<hkUint64>* connections = cachedObject->getConnections( deallocatedFrameNumber + 1 );
    if ( ( deallocatedConnections != connections ) && ( deallocatedConnections != &cachedObject->m_initialConnections ) )
    {
        cachedObject->deleteSparseData( deallocatedConnections );
    }
}

#pragma endregion

//////////////////////////////////////////////////////////////////////////
// hkVdbCachedGeometry
//////////////////////////////////////////////////////////////////////////

#pragma region hkVdbCachedGeometry

const hkMatrix4& hkVdbCachedGeometry::getTransform( hkUint32 frame ) const
{
    return getCachedValue( HK_VDB_TRANSFORM_CACHE_ID, frame, m_initialTransform );
}

hkColor::Argb hkVdbCachedGeometry::getColor( hkUint32 frame ) const
{
    return getCachedValue( HK_VDB_COLOR_CACHE_ID, frame, m_initialColor );
}

hkDisplayGeometryFlags hkVdbCachedGeometry::getFlags( hkUint32 frame ) const
{
    return getCachedValue( HK_VDB_FLAGS_CACHE_ID, frame, m_initialFlags );
}

const hkArray<hkVector4>* hkVdbCachedGeometry::getVertices( hkUint32 frame ) const
{
    return getCachedValue( HK_VDB_VERTEX_CACHE_ID, frame, m_initialVertices );
}

const hkArray<hkTransform>* hkVdbCachedGeometry::getParticleTransforms( hkUint32 frame ) const
{
    return getCachedValue( HK_VDB_PARTICLE_TRANSFORM_CACHE_ID, frame, m_initialParticleTransforms );
}

void hkVdbCachedGeometry::registerTransformUpdate( hkUint32 frame, const hkMatrix4& transform )
{
    registerCachedValueUpdate( HK_VDB_TRANSFORM_CACHE_ID, frame, m_initialTransform, transform );
}

void hkVdbCachedGeometry::registerColorUpdate( hkUint32 frame, hkColor::Argb color )
{
    registerCachedValueUpdate( HK_VDB_COLOR_CACHE_ID, frame, m_initialColor, color );
}

void hkVdbCachedGeometry::registerSetFlagBits( hkUint32 frame, hkDisplayGeometryFlags flagBits )
{
    registerCachedFlagsSet( HK_VDB_FLAGS_CACHE_ID, frame, m_initialFlags, flagBits );
}

void hkVdbCachedGeometry::registerClearFlagBits( hkUint32 frame, hkDisplayGeometryFlags flagBits )
{
    registerCachedFlagsClear( HK_VDB_FLAGS_CACHE_ID, frame, m_initialFlags, flagBits );
}

void hkVdbCachedGeometry::registerVerticesUpdate( hkUint32 frame, hkArray<hkVector4>& newVertsToStore )
{
    if ( hkArray<hkVector4>* sparseStorage = registerCachedSparseValueUpdate( HK_VDB_VERTEX_CACHE_ID, frame, m_initialVertices ) )
    {
        stealArrayOwnership( *sparseStorage, newVertsToStore );
    }
}

void hkVdbCachedGeometry::registerParticleTransformsUpdate( hkUint32 frame, hkArray<hkTransform>& newTransformsToStore )
{
    if ( hkArray<hkTransform>* sparseStorage = registerCachedSparseValueUpdate( HK_VDB_PARTICLE_TRANSFORM_CACHE_ID, frame, m_initialParticleTransforms ) )
    {
        stealArrayOwnership( *sparseStorage, newTransformsToStore );
    }
}

#pragma endregion

//////////////////////////////////////////////////////////////////////////
// hkVdbCachedObject
//////////////////////////////////////////////////////////////////////////

#pragma region hkVdbCachedObject

const hkReflect::Var hkVdbCachedObject::getData( hkUint32 frame ) const
{
    return getCachedValue( HK_VDB_OBJECT_DATA_CACHE_ID, frame, m_initialData );
}

void hkVdbCachedObject::registerDataAdded( hkUint32 frame, int tag, const hkVdbReflect::Var& dataToStore, hkObjectFlags flags )
{
    
    
    HK_ASSERT( 0x22441155, getData( frame ) == HK_NULL, "Data already initialized for object; use registerDataUpdate instead" );
    HK_ASSERT( 0x22441291, ( m_tag == -1 ) || ( m_tag == tag ), "We don't support using the same id across different tags" );
    HK_ASSERT( 0x22441292, ( m_flags == hkObjectFlags::DEFAULT ) || ( m_flags == flags ), "We don't support using the same id with different flags" );
    m_tag = tag;
    m_flags = flags;
    registerDataUpdate( frame, dataToStore );
}

void hkVdbCachedObject::registerDataUpdate( hkUint32 frame, const hkVdbReflect::Var& dataToStore )
{
    
    
    registerCachedSparseVarUpdate( HK_VDB_OBJECT_DATA_CACHE_ID, frame, m_initialData, dataToStore );
}

void hkVdbCachedObject::registerDataRemoved( hkUint32 frame )
{
    
    
    registerCachedSparseVarUpdate( HK_VDB_OBJECT_DATA_CACHE_ID, frame, m_initialData, hkReflect::Var() );
    removeIfAbandoned();
}

const hkArray<hkUint64>* hkVdbCachedObject::getConnections( hkUint32 frame ) const
{
    return getCachedValue( HK_VDB_OBJECT_CONNECTIVITY_CACHE_ID, frame, &m_initialConnections );
}

void hkVdbCachedObject::registerConnect( hkUint32 frame, const hkArrayView<hkUint64>& connections, int tag, hkLocalArray<hkUint64>& addedConnectionsOut )
{
    
    if ( hkArray<hkUint64>* frameConnections = getOrCreateFrameConnections( frame ) )
    {
        
        
        
        addFrameConnections( *frameConnections, connections, &addedConnectionsOut );
        updateParentConnections( getUid(), &addedConnectionsOut, HK_NULL );
        checkConnectionConsistency();
    }
}

void hkVdbCachedObject::registerDisconnect( hkUint32 frame, const hkArrayView<hkUint64>& connections, hkLocalArray<hkUint64>& removedConnectionsOut )
{
    if ( hkArray<hkUint64>* frameConnections = getOrCreateFrameConnections( frame ) )
    {
        subtractFrameConnections( *frameConnections, connections, &removedConnectionsOut );
        updateParentConnections( getUid(), HK_NULL, &removedConnectionsOut );
        checkConnectionConsistency();
    }
}

void hkVdbCachedObject::registerDisconnectAll( hkUint32 frame, hkConnectivityFlags flags, hkLocalArray<hkUint64>& removedIncomingConnectionsOut, hkLocalArray<hkUint64>& removedOutgoingConnectionsOut )
{
    if ( flags.anyIsSet( hkConnectivityFlags::INCOMING_CONNECTIONS ) )
    {
        removedIncomingConnectionsOut.reserve( removedIncomingConnectionsOut.getSize() + m_parentConnections.getSize() );
        for ( int i = 0; i < m_parentConnections.getSize(); i++ )
        {
            hkUint64 myId = getUid();
            hkUint64 parentId = m_parentConnections[i];
            if ( hkVdbCachedObject* parentEntry = m_owner.accessEntry( hkVdbCmdId::extractStreamId( parentId ) ) )
            {
                if ( hkArray<hkUint64>* frameConnections = parentEntry->getOrCreateFrameConnections( frame ) )
                {
                    if ( subtractFrameConnections( *frameConnections, hkArrayViewT::fromSingleObject( myId ) ) )
                    {
                        removedIncomingConnectionsOut.pushBack( parentEntry->getUid() );
                        parentEntry->removeIfAbandoned();
                    }
                }
            }
        }
        m_parentConnections.clear();
        checkConnectionConsistency();
    }
    if ( flags.anyIsSet( hkConnectivityFlags::OUTGOING_CONNECTIONS ) )
    {
        if ( hkArray<hkUint64>* frameConnections = getOrCreateFrameConnections( frame ) )
        {
            updateParentConnections( getUid(), HK_NULL, frameConnections );
            removedOutgoingConnectionsOut.append( *frameConnections );
            frameConnections->clear();
            checkConnectionConsistency();
        }
    }
    removeIfAbandoned();
}

hkArray<hkUint64>* hkVdbCachedObject::getOrCreateFrameConnections( hkUint32 frame )
{
    const hkArray<hkUint64>* prevFrameConnections = getConnections( ( frame > m_frameAdded ) ? frame - 1 : m_frameAdded );
    hkArray<hkUint64>* frameConnections = const_cast< hkArray<hkUint64>* >( getConnections( frame ) );
    if ( prevFrameConnections == frameConnections )
    {
        hkArray<hkUint64>* initialConnections = &m_initialConnections;
        hkArray<hkUint64>* newConnections = registerCachedSparseValueUpdate( HK_VDB_OBJECT_CONNECTIVITY_CACHE_ID, frame, initialConnections );

        // In the initial connections case, these will be the same...until we move to another frame.
        // hkArray::append( this ) corrupts the memory; so avoid that here.
        if ( newConnections && ( newConnections != prevFrameConnections ) )
        {
            // Copy over current connections
            newConnections->append( *prevFrameConnections );
            assertNoDups( *newConnections );
        }
        return newConnections;
    }
    return frameConnections;
}

void hkVdbCachedObject::updateParentConnections( hkUint64 parentId, const hkArrayBase<hkUint64>* addedChildIds, const hkArrayBase<hkUint64>* removedChildIds )
{
    if ( addedChildIds && addedChildIds->getSize() )
    {
        for ( int i = 0; i < addedChildIds->getSize(); i++ )
        {
            hkUint64 childId = ( *addedChildIds )[i];
            if ( hkVdbCachedObject* childEntry = m_owner.accessEntry( hkVdbCmdId::extractStreamId( childId ) ) )
            {
                addFrameConnections( childEntry->m_parentConnections, hkArrayViewT::fromSingleObject( parentId ) );
            }
        }
    }
    if ( removedChildIds && removedChildIds->getSize() )
    {
        hkLocalArray<hkVdbCachedObject*> maybeAbandonedChildEntries( removedChildIds->getSize() );
        for ( int i = 0; i < removedChildIds->getSize(); i++ )
        {
            hkUint64 childId = ( *removedChildIds )[i];
            if ( hkVdbCachedObject* childEntry = m_owner.accessEntry( hkVdbCmdId::extractStreamId( childId ) ) )
            {
                hkBool32 subtracted = subtractFrameConnections( childEntry->m_parentConnections, hkArrayViewT::fromSingleObject( parentId ) );
                if ( subtracted ) maybeAbandonedChildEntries.pushBack( childEntry );
            }
        }
        for ( int i = 0; i < maybeAbandonedChildEntries.getSize(); i++ )
        {
            hkVdbCachedObject* childEntry = maybeAbandonedChildEntries[i];
            childEntry->removeIfAbandoned();
        }
        removeIfAbandoned();
    }
}

void hkVdbCachedObject::removeIfAbandoned()
{
    if ( m_frameRemoved == -1 )
    {
        const hkUint32 frame = m_input.getProcessingFrameNumber();
        const hkArray<hkUint64>* outgoingConnections = getConnections( frame );

        const hkBool32 noIncoming = ( m_parentConnections.getSize() == 0 );
        const hkBool32 noOutgoing = ( !outgoingConnections || ( outgoingConnections->getSize() == 0 ) );
        const hkBool32 noData = ( getData( frame ) == HK_NULL );
        if ( noIncoming && noOutgoing && noData )
        {
            registerRemoved( frame );
        }
    }
}

void hkVdbCachedObject::checkConnectionConsistency()
{
#ifdef ASSERT_CONNECTION_CONSISTENCY
    // Can't assert if we aren't in debug
#ifdef HK_DEBUG
    const hkUint32 frame = m_input.getProcessingFrameNumber();
    for ( hkVdbCachedObjects::Iterator iter = m_owner.getIterator();
        m_owner.isValid( iter );
        iter = m_owner.getNext( iter ) )
    {
        const hkVdbCachedObject* parent = m_owner.getValue( iter );
        if ( parent->isAdded( frame ) )
        {
            const hkArray<hkUint64>* childConnections = parent->getConnections( frame );
            for ( int childIt = 0; childIt < childConnections->getSize(); childIt++ )
            {
                const hkVdbCachedObject* child = m_owner.accessEntry( hkVdbCmdId::extractStreamId( ( *childConnections )[childIt] ) );
                HK_ASSERT_NO_MSG( 0x0, child );
                HK_ASSERT_NO_MSG( 0x0, child->m_parentConnections.indexOf( parent->getUid() ) != -1 );
            }
        }
    }
#endif
#endif
}

#pragma endregion

//////////////////////////////////////////////////////////////////////////
// hkVdbCache
//////////////////////////////////////////////////////////////////////////

#pragma region hkVdbCache

hkVdbCache::hkVdbCache( hkVdbCmdInput& input ) :
    m_geometriesCache( hkRefNew<hkVdbCachedGeometries>( new hkVdbCachedGeometries( *this, input ) ) ),
    m_objectsCache( hkRefNew<hkVdbCachedObjects>( new hkVdbCachedObjects( *this, input ) ) )
{}

void hkVdbCache::clear()
{
    m_geometriesCache->clear();
    m_objectsCache->clear();
}

#pragma endregion

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
