// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM     : WIN32 X64 UWP
// PRODUCT      : COMMON
// VISIBILITY   : PUBLIC
//
// ------------------------------------------------------TKBMS v1.0

#include <VisualDebugger/VdbServices/hkVdbServices.h>
#include <VisualDebugger/VdbServices/Plugin/hkVdbPluginManager.h>

#include <Common/Base/hkBase.h>
#include <Common/Base/Memory/System/Util/hkMemoryInitUtil.h>
#include <Common/Base/System/Io/FileSystem/hkFileSystem.h>
#include <Common/Base/System/Dll/DynamicLibrary/hkDynamicLibrary.h>

#include <VisualDebugger/VdbServices/Plugin/hkVdbPlugin.h>

#define DEBUG_LOG_IDENTIFIER "vdb.PluginManager"
#include <Common/Base/System/Log/hkLog.hxx>

hkVdbPluginManager::hkVdbPluginManager() :
    hkVdbDefaultErrorReporter( &s_debugLog )
{}

hkVdbPluginManager::~hkVdbPluginManager()
{
    unloadAll();
}

hkResult hkVdbPluginManager::load( const char* fullPath )
{
    hkHashMap<const char*, PluginInfo*>::Iterator iter = m_pluginInfos.findKey( fullPath );
    if ( m_pluginInfos.isValid( iter ) )
    {
        return HK_SUCCESS;
    }

    PluginInfo* info = new PluginInfo();
    info->m_dll = hkDynamicLibrary::load( fullPath );
    if ( info->m_dll )
    {
        if ( void* dllCreationFunc = info->m_dll->findSymbol( "createVdbPlugin" ) )
        {
            typedef hkRefNew<hkVdbPlugin>( *CreationFunc )();
            if ( hkRefPtr<hkVdbPlugin> plugin = reinterpret_cast< CreationFunc >( dllCreationFunc )( ) )
            {
                hkMemoryInitUtil::SyncInfo si;
                hkMemoryInitUtil::SyncInfo::getLocalInfo( si );
                HK_VDB_VERIFY_REPORTER_OPERATION( plugin->init( si, hkError::getInstancePtr() ), *plugin, hkVdbError::PLUGIN_ERROR );
                info->m_plugin = plugin;
                m_pluginInfos.insert( fullPath, info ); 
                return HK_SUCCESS;
            }
        }
    }

    HK_VDB_SIGNAL_ERROR_MSG( 0xedb00013, hkVdbError::PLUGIN_DLL_INVALID, "No valid dll could be found at \"" << fullPath << "\"" );
    delete info;
    return HK_FAILURE;
}

hkResult hkVdbPluginManager::loadAll( const char* dirPath, bool recurse )
{
    if ( hkFileSystem* fileSystem = hkFileSystem::getInstancePtr() )
    {
        bool validDllFound = false;

        for ( hkFileSystem::Iterator it( fileSystem, dirPath ); it.advance(); )
        {
            const hkFileSystem::Entry& entry = it.current();
            if ( entry.isDir() )
            {
                if ( recurse )
                {
                    it.recurseInto( entry.getPath() );
                }
            }
            else if ( hkString::endsWithCase( entry.getPath(), ".dll" ) )
            {
                validDllFound |= ( load( entry.getPath() ).isSuccess() );
            }
        }

        if ( validDllFound )
        {
            return HK_SUCCESS;
        }
    }

    HK_VDB_SIGNAL_ERROR_MSG( 0xedb00013, hkVdbError::IO_ERROR, "No valid dlls could be found" );
    return HK_FAILURE;
}

hkResult hkVdbPluginManager::unloadAll()
{
    bool succeeded = true;

    for ( hkHashMap<const char*, PluginInfo*>::Iterator iter = m_pluginInfos.getIterator();
        m_pluginInfos.isValid( iter );
        iter = m_pluginInfos.getNext( iter ) )
    {
        PluginInfo* pluginInfo = m_pluginInfos.getValue( iter );
        succeeded &= ( unloadInternal( pluginInfo ).isSuccess() );
    }
    m_pluginInfos.clear();

    return succeeded ? HK_SUCCESS : HK_FAILURE;
}

hkVdbPlugin* hkVdbPluginManager::getLoadedPlugin( const char* pluginName ) const
{
    for ( hkHashMap<const char*, PluginInfo*>::Iterator iter = m_pluginInfos.getIterator();
        m_pluginInfos.isValid( iter );
        iter = m_pluginInfos.getNext( iter ) )
    {
        const PluginInfo* pluginInfo = m_pluginInfos.getValue( iter );
        if ( hkString::strCmp( pluginName, pluginInfo->m_plugin->getName() ) == 0 )
        {
            return pluginInfo->m_plugin;
        }
    }
    return HK_NULL;
}

int hkVdbPluginManager::getLoadedPlugins( hkArray<hkVdbPlugin*>& pluginsOut ) const
{
    int prevCount = pluginsOut.getSize();

    for ( hkHashMap<const char*, PluginInfo*>::Iterator iter = m_pluginInfos.getIterator();
        m_pluginInfos.isValid( iter );
        iter = m_pluginInfos.getNext( iter ) )
    {
        const PluginInfo* pluginInfo = m_pluginInfos.getValue( iter );
        if ( pluginsOut.indexOf( pluginInfo->m_plugin ) == -1 )
        {
            pluginsOut.pushBack( pluginInfo->m_plugin );
        }
    }

    return pluginsOut.getSize() - prevCount;
}

hkResult hkVdbPluginManager::registerPlugins( hkVdbClient& client )
{
    hkVdbSignalResult result;

    for ( hkHashMap<const char*, PluginInfo*>::Iterator iter = m_pluginInfos.getIterator();
        m_pluginInfos.isValid( iter );
        iter = m_pluginInfos.getNext( iter ) )
    {
        PluginInfo* pluginInfo = m_pluginInfos.getValue( iter );
        if ( !pluginInfo->m_registered )
        {
            if ( pluginInfo->m_plugin->registerSelf( client ).isSuccess() )
            {
                pluginInfo->m_registered = true;
            }
            else
            {
                result.signalError( *pluginInfo->m_plugin );
            }
        }
    }

    HK_VDB_VERIFY_SIGNAL_RESULT( result );
    return HK_SUCCESS;
}

hkResult hkVdbPluginManager::unregisterPlugins( hkVdbClient& client )
{
    hkVdbSignalResult result;

    for ( hkHashMap<const char*, PluginInfo*>::Iterator iter = m_pluginInfos.getIterator();
        m_pluginInfos.isValid( iter );
        iter = m_pluginInfos.getNext( iter ) )
    {
        PluginInfo* pluginInfo = m_pluginInfos.getValue( iter );
        if ( pluginInfo->m_registered )
        {
            if ( pluginInfo->m_plugin->unregisterSelf( client ).isSuccess() )
            {
                pluginInfo->m_registered = false;
            }
            else
            {
                result.signalError( *pluginInfo->m_plugin );
            }
        }
    }

    HK_VDB_VERIFY_SIGNAL_RESULT( result );
    return HK_SUCCESS;
}

hkVdbPluginManager::PluginInfo::PluginInfo() :
    m_dll( HK_NULL ),
    m_plugin( HK_NULL ),
    m_registered( false )
{}

hkVdbPluginManager::PluginInfo::~PluginInfo()
{}

hkResult hkVdbPluginManager::unloadInternal( PluginInfo* pluginInfo )
{
    HK_VDB_VERIFY_CONDITION_MSG( !pluginInfo->m_registered, 0xedb00014, hkVdbError::PLUGIN_REGISTERED, "Cannot unload a plugin that is still registered" );
    HK_VDB_VERIFY_REPORTER_OPERATION( pluginInfo->m_plugin->quit(), *pluginInfo->m_plugin, hkVdbError::PLUGIN_ERROR );
    delete pluginInfo;
    return HK_SUCCESS;
}

/*
 * Havok SDK - Base file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
