// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM		: WIN32 X64
// PRODUCT		: COMMON
// VISIBILITY	: CLIENT
//
// ------------------------------------------------------TKBMS v1.0

using System.Text.RegularExpressions;
using System.Linq;
using System;
using System.Windows;
using System.Windows.Media;

// Root namespace for easier use with ToString extension
public enum StringFormat
{
    Default,
    Hexadecimal,
}

namespace HavokVisualDebugger
{
    public static class hkStringUtils
    {
        private static Regex SplitCamelCaseRegex1 = new Regex(@"(\P{Ll})(\P{Ll}\p{Ll})");
        private static Regex SplitCamelCaseRegex2 = new Regex(@"(\p{Ll})(\P{Ll})");

        public static string SplitCamelCase(string originalString)
        {
            string interimResult = SplitCamelCaseRegex1.Replace(originalString, "$1 $2");
            return SplitCamelCaseRegex2.Replace(interimResult, "$1 $2");
        }

        public static double GetFontHeight(
            double fontSize,
            FontFamily font)
        {
            double lineSpacing = (font != null) ? font.LineSpacing : (fontSize * 0.1);
            double fontHeight = fontSize * lineSpacing;
            // Ceiling to ensure uniform placement for other elements w/o stressing anti-aliasing
            return Math.Ceiling(fontHeight);
        }

        public static double GetFontWidth(
            string text,
            double fontSize,
            FontFamily font,
            FontStyle? style = null,
            FontWeight? weight = null,
            FontStretch? stretch = null)
        {
            if (font != null)
            {
                var formattedText = new FormattedText(
                    text,
                    System.Globalization.CultureInfo.CurrentUICulture,
                    System.Windows.FlowDirection.LeftToRight,
                    new Typeface(font,
                        style.HasValue ? style.Value : FontStyles.Normal,
                        weight.HasValue ? weight.Value : FontWeights.Normal,
                        stretch.HasValue ? stretch.Value : FontStretches.Normal),
                    fontSize,
                    Brushes.Black);
                return formattedText.Width;
            }
            else
            {
                return 0;
            }
        }

        public static string TrimHexStart(this string s)
        {
            return s.TrimStart(" &h0x#".ToCharArray());
        }

        public static string ToString<T>(this T any, StringFormat format)
        {
            switch (format)
            {
                case StringFormat.Hexadecimal:
                    return ToStringInternal(any, ToHexString);
                default:
                    System.Diagnostics.Debug.Assert(format == StringFormat.Default, "Unimplemented format");
                    return ToStringInternal(any, (o) => o.ToString());
            }
        }

        private static string ToStringInternal(System.Collections.IList list, Func<object, string> func)
        {
            return "[ " + string.Join(", ", list.Cast<object>().Select(o => ToStringInternal(o, func))) + " ]";
        }

        private static string ToStringInternal(Havok.Vdb.Object hkObject, Func<object, string> func)
        {
            return "{ " + string.Join(", ", hkObject.Properties.Values.Select(o => ToStringInternal(o, func))) + " }";
        }

        private static string ToStringInternal(Havok.Vdb.IReadOnlyProperty hkPropertyRO, Func<object, string> func)
        {
            return string.Join(", ", hkPropertyRO.Name + "=" + ToStringInternal(hkPropertyRO.Value, func));
        }

        private static string ToStringInternal<T>(T any, Func<object, string> func)
        {
            Type type = any.GetType();
            if (typeof(System.Collections.IList).IsAssignableFrom(type))
            {
                return ToStringInternal((System.Collections.IList)any, func);
            }
            else if (typeof(Havok.Vdb.Object).IsAssignableFrom(type))
            {
                return ToStringInternal((Havok.Vdb.Object)any, func);
            }
            else if (typeof(Havok.Vdb.IReadOnlyProperty).IsAssignableFrom(type))
            {
                return ToStringInternal((Havok.Vdb.IReadOnlyProperty)any, func);
            }
            else if (!hkReflect.IsBuiltinType(type))
            {
                try
                {
                    // Note: this is necessary for things like Havok.Vdb.Vector/Rotation,etc. because the system's ToString
                    // won't honor the hex setting we may be trying to apply.
                    System.Reflection.PropertyInfo[] properties = type.GetProperties(hkReflect.BindingFlagsEx.AllPropsFlag);
                    System.Reflection.FieldInfo[] fields = type.GetFields(hkReflect.BindingFlagsEx.AllFieldsFlag);
                    return
                        "{ " +

                            // Properties
                            string.Join(", ",
                                properties.Select(info =>
                                {
                                    try
                                    {
                                        object val = info.GetValue(any);
                                        return string.Join(", ", info.Name + "=" + ToStringInternal(val, func));
                                    }
                                    catch { }
                                    // same as backend when a property can't be read
                                    return "incomplete";
                                })) +

                            // Joiner
                            ((properties.Length != 0) ? ", " : "") +

                            // Fields
                            string.Join(", ",
                                fields.Select(info =>
                                {
                                    try
                                    {
                                        object val = info.GetValue(any);
                                        return string.Join(", ", info.Name + "=" + ToStringInternal(val, func));
                                    }
                                    catch { }
                                    // same as backend when a property can't be read
                                    return "incomplete";
                                })) +

                        " }";
                }
                catch { }
            }

            // Fallback
            return func(any);
        }

        private static string ToHexString(object any)
        {
            if (hkReflect.IsFloatingPoint(any))
            {
                try
                {
                    Double? d = hkMath.GetFloat(any);
                    any = BitConverter.DoubleToInt64Bits(d.Value);
                    // Fall through to integral conversion
                }
                catch { }
            }
            if (hkReflect.IsInteger(any))
            {
                try
                {
                    return String.Format("0x{0:x}", any);
                }
                catch { }
            }
            return any.ToString();
        }
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
