// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM		: WIN32 X64
// PRODUCT		: COMMON
// VISIBILITY	: CLIENT
//
// ------------------------------------------------------TKBMS v1.0

using System;
using System.Collections.Generic;
using System.Text.RegularExpressions;
using System.Linq;

namespace HavokVisualDebugger
{
    public class SearchFilter
    {
        public static readonly Regex FilterRegex =
            new Regex(@"(?x)                                                # Ignore pattern whitespace - spaces must be escaped in the rest of this expression!
                        (
                            [\s]*                                           # Optional whitespace
                            (?<leftContext>[\.\*\\\/]*)                     # Context for the left operand (parent/child relationship)
                            (?<leftOperand>[\w\.,]+)                        # Left operand
                            [\s]*                                           # Optional whitespace
                            (
                                (?<operator>[=<>!]+)                        # Operator
                                [\s]*                                       # Optional whitespace
                                (
                                    [""'](?<rightOperand>[\w\., ]+)[""']    # Right operand with quotes, allow spaces
                                    |
                                    (?<rightOperand>[\w\.,]+)               # Right operand, don't allow spaces
                                )
                            )?                                              # (operator right-operand) is optional, can just have a left operand on its own
                            (
                                [\s]*                                       # Optional whitespace
                                (?<boolean>AND|and|OR|or|&|&&|\||\|\|)*     # Boolean operation connecting multiple statements
                                [\s]*                                       # Optional whitespace
                            )
                        )+");

        public string Value { get; set; }
        public string[] PropertyPath { get; set; }
        public OperatorType Operator { get; set; }
        public BooleanOperator BooleanSuffix { get; set; }
        public RelativeSearchType RelativeSearch { get; set; }

        public enum BooleanOperator
        {
            Or,
            And,
            None
        };

        public enum OperatorType
        {
            LessThan,
            LessThanEqual,
            GreaterThan,
            GreaterThanEqual,
            Equal,
            NotEqual
        };

        public enum RelativeSearchType
        {
            This,
            Parent,
            Child,
            AnyParent,
            AnyChild
        };

        public enum FilterResult
        {
            Unevaluated,
            True,
            False,
            ParentTrue,
            ParentFalse,
            ChildTrue,
            ChildFalse
        };

        public SearchFilter()
        {
            PropertyPath = new string[] { };
            Value = "";
            Operator = OperatorType.Equal;
            BooleanSuffix = BooleanOperator.None;
            RelativeSearch = RelativeSearchType.This;
        }

        private static void AppendDefaultSearchFilters(string filter, List<SearchFilter> generatedFilters)
        {
            if (!String.IsNullOrEmpty(filter))
            {
                // Check Type
                {
                    SearchFilter fm = new SearchFilter()
                    {
                        // Note: "." makes it so we won't find and embedded type name
                        PropertyPath = ".TypeName".Split('.'),
                        Value = filter,
                        BooleanSuffix = BooleanOperator.Or
                    };
                    generatedFilters.Add(fm);
                }

                // Check Name
                {
                    SearchFilter fm = new SearchFilter()
                    {
                        // Note: "." makes it so we won't find and embedded name
                        PropertyPath = ".Name".Split('.'), // Don't search embedded props
                        Value = filter,
                        BooleanSuffix = BooleanOperator.Or
                    };
                    generatedFilters.Add(fm);
                }
            }
        }

        public static void GenerateSearchFilters(string filter, List<SearchFilter> generatedFilters)
        {
            MatchCollection matches = FilterRegex.Matches(filter);
            if (matches == null || matches.Count == 0)
            {
                AppendDefaultSearchFilters(filter, generatedFilters);
            }
            else
            {
                foreach (Match match in matches)
                {
                    try
                    {
                        int captures = match.Groups["leftOperand"].Captures.Count;
                        for (int i = 0; i < captures; i++)
                        {
                            SearchFilter fm = new SearchFilter();

                            if(match.Groups["leftOperand"].Captures.Count > i)
                            {
                                string leftOperand = match.Groups["leftOperand"].Captures[i].Value;
                                fm.PropertyPath = leftOperand.Split('.');
                                // If we have a specified path, place the implicit root at the beginning
                                // if it doesn't exist.  This is because we don't support partial subpath
                                // matching.
                                // Eg. "someProp.subProp" will only match [root].someProp.subProp not *.someProp.subProp
                                // Since a lot of our hierarchy is formed by connection ids (not embedded record properties), this seems completely sufficient
                                if ((fm.PropertyPath.Length > 1) && (fm.PropertyPath[0] != ""))
                                {
                                    fm.PropertyPath = new string[] { "" }.Concat(fm.PropertyPath).ToArray();
                                }
                            }
                            else
                            {
                                // No left operand, fail
                                return;
                            }

                            bool comparing = false;
                            if( match.Groups["rightOperand"].Captures.Count > i)
                            {
                                fm.Value = match.Groups["rightOperand"].Captures[i].Value;

                                if (match.Groups["operator"].Captures.Count > i)
                                {
                                    string binaryOp = match.Groups["operator"].Captures[i].Value;
                                    if (binaryOp == ">")
                                    {
                                        fm.Operator = OperatorType.GreaterThan;
                                    }
                                    else if (binaryOp == "=>" || binaryOp == ">=")
                                    {
                                        fm.Operator = OperatorType.GreaterThanEqual;
                                    }
                                    else if (binaryOp == "=<" || binaryOp == "<=")
                                    {
                                        fm.Operator = OperatorType.LessThanEqual;
                                    }
                                    else if (binaryOp == "<")
                                    {
                                        fm.Operator = OperatorType.LessThan;
                                    }
                                    else if (binaryOp == "=" || binaryOp == "==")
                                    {
                                        fm.Operator = OperatorType.Equal;
                                    }
                                    else if (binaryOp == "<>" || binaryOp == "!=" || binaryOp == "=!")
                                    {
                                        fm.Operator = OperatorType.NotEqual;
                                    }
                                    else
                                    {
                                        // default
                                        fm.Operator = OperatorType.Equal;
                                    }

                                    comparing = true;
                                }
                            }

                            if(!comparing)
                            {
                                // No direct comparison, collapse to defaults
                                AppendDefaultSearchFilters(filter, generatedFilters);
                            }
                            else
                            {
                                string booleanOp = (match.Groups["boolean"]?.Captures.Count > i) ? match.Groups["boolean"].Captures[i].Value : "";
                                if (booleanOp == "AND" || booleanOp == "&&" || booleanOp == "&" || booleanOp == "and")
                                {
                                    fm.BooleanSuffix = BooleanOperator.And;
                                }
                                else if (booleanOp == "OR" || booleanOp == "||" || booleanOp == "|" || booleanOp == "or")
                                {
                                    fm.BooleanSuffix = BooleanOperator.Or;
                                }
                                else
                                {
                                    fm.BooleanSuffix = BooleanOperator.None;
                                }

                                string relativeOperation = (match.Groups["leftContext"]?.Captures.Count > i) ? match.Groups["leftContext"].Captures[i].Value : "";
                                if (relativeOperation == "*/")
                                {
                                    fm.RelativeSearch = RelativeSearchType.AnyParent;
                                }
                                else if (relativeOperation == "/*")
                                {
                                    fm.RelativeSearch = RelativeSearchType.AnyChild;
                                }
                                else
                                {
                                    fm.RelativeSearch = RelativeSearchType.This;
                                }

                                generatedFilters.Add(fm);
                            }
                        }
                    }
                    catch
                    {
                        // bad user input, do nothing
                    }
                }
            }
        }

        public static int PathCompare(List<string> currentPropDir, string[] propPath)
        {
            if (propPath.Length == 1)
            {
                return 0;
            }
            else if (propPath.Length == 0)
            {
                return +1;
            }
            else
            {
                int i = 0;
                for (; i < currentPropDir.Count; i++)
                {
                    string dir = currentPropDir[i];
                    if ((i >= propPath.Length) || (dir != propPath[i]))
                    {
                        // Failure, will return +1
                        i = propPath.Length;
                        break;
                    }
                }
                return (i - (propPath.Length - 1));
            }
        }

        public static void PushPathDir(List<string> currentPropPath, string dir)
        {
            currentPropPath.Add(dir);
        }

        public static void PopPathDir(List<string> currentPropPath)
        {
            currentPropPath.RemoveAt(currentPropPath.Count - 1);
        }

        public static void PushPathDir(List<string> currentPropPath, object dir)
        {
            currentPropPath.Add(dir.ToString());
        }

        public static string NameFromPath(string[] propPath)
        {
            if (propPath != null)
            {
                return propPath[propPath.Length - 1];
            }
            return "";
        }

        public static bool Match(object value, SearchFilter filter)
        {
            bool result = false;
            if (value is string && filter.Operator == SearchFilter.OperatorType.Equal)
            {
                // do a "contains" for string equal
                result = value.ToString().IndexOf(filter.Value, StringComparison.OrdinalIgnoreCase) >= 0;
            }
            else
            {
                int compareResult;

                if (!hkMath.TryNumericCompare(value, filter.Value, out compareResult))
                {
                    compareResult = String.Compare(value.ToString(), filter.Value, StringComparison.OrdinalIgnoreCase);
                }

                switch (filter.Operator)
                {
                    case SearchFilter.OperatorType.LessThan:
                        {
                            result = (compareResult < 0);
                            break;
                        }
                    case SearchFilter.OperatorType.LessThanEqual:
                        {
                            result = (compareResult <= 0);
                            break;
                        }
                    case SearchFilter.OperatorType.GreaterThan:
                        {
                            result = (compareResult > 0);
                            break;
                        }
                    case SearchFilter.OperatorType.GreaterThanEqual:
                        {
                            result = (compareResult >= 0);
                            break;
                        }
                    case SearchFilter.OperatorType.NotEqual:
                        {
                            result = (compareResult != 0);
                            break;
                        }
                    default:
                    case SearchFilter.OperatorType.Equal:
                        {
                            result = (compareResult == 0);
                            break;
                        }
                }
            }

            return result;
        }

        public static bool CombineFilterResults(List<SearchFilter> filters, List<FilterResult> results, List<FilterResult> parentResults, List<List<FilterResult>> childResults)
        {
            System.Diagnostics.Debug.Assert((results == null || filters.Count == results.Count) &&
                                            (parentResults == null || filters.Count == parentResults.Count));

            bool combinedResult = false;

            BooleanOperator previousBooleanOperator = BooleanOperator.None;
            for (int i = 0; i < filters.Count; i++)
            {
                SearchFilter filter = filters[i];

                // Determine result based on filter type and data available (parent, current and child evaluations)
                bool result = false;
                switch (filter.RelativeSearch)
                {
                    case RelativeSearchType.Parent:
                    case RelativeSearchType.AnyParent:
                    {
                        result = (parentResults != null) ? (parentResults[i] == FilterResult.ParentTrue) : false;
                        break;
                    }
                    case RelativeSearchType.Child:
                    case RelativeSearchType.AnyChild:
                    {
                        // Loop through all children (there can be multiple) and see if any children have a matching filter
                        if( childResults == null )
                        {
                            result = false;
                        }
                        else
                        {
                            foreach (List<FilterResult> childFilterResults in childResults)
                            {
                                result |= (childFilterResults != null) ? (childFilterResults[i] == FilterResult.ChildTrue) : false;
                            }
                        }
                        break;
                    }
                    default:
                    case RelativeSearchType.This:
                    {
                        result = (results != null) ? (results[i] == FilterResult.True) : false;
                        break;
                    }
                }

                // Combine with previous operations
                switch (previousBooleanOperator)
                {
                    case BooleanOperator.And:
                    {
                        combinedResult &= result;
                        break;
                    }
                    case BooleanOperator.Or:
                    {
                        combinedResult |= result;
                        break;
                    }
                    default:
                    case BooleanOperator.None:
                    {
                        combinedResult = result;
                        break;
                    }
                }

                previousBooleanOperator = filter.BooleanSuffix;
            }

            return combinedResult;
        }

        public static List<FilterResult> DoFiltersMatch(List<SearchFilter> filters, Func<string[], object[]> GetPropertyValuesForPathFunc)
        {
            List<FilterResult> results = new List<FilterResult>();
            foreach (SearchFilter filter in filters)
            {
                object[] propertyValues = GetPropertyValuesForPathFunc(filter.PropertyPath);
                if ((propertyValues != null) && (propertyValues.Length != 0))
                {
                    bool result = false;
                    foreach (object propertyValue in propertyValues)
                    {
                        if ((propertyValue != null) && SearchFilter.Match(propertyValue, filter))
                        {
                            result = true;
                            break;
                        }
                    }

                    // Parent filter, update parent slot
                    if ((filter.RelativeSearch == SearchFilter.RelativeSearchType.Parent) || (filter.RelativeSearch == SearchFilter.RelativeSearchType.AnyParent))
                    {
                        results.Add(result ? FilterResult.ParentTrue : FilterResult.ParentFalse);
                    }
                    else if((filter.RelativeSearch == SearchFilter.RelativeSearchType.Child) || (filter.RelativeSearch == SearchFilter.RelativeSearchType.AnyChild))
                    {
                        results.Add(result ? FilterResult.ChildTrue : FilterResult.ChildFalse);
                    }
                    else
                    {
                        results.Add(result ? FilterResult.True : FilterResult.False);
                    }
                }
                else
                {
                    results.Add(FilterResult.False);
                }
            }

            System.Diagnostics.Debug.Assert(results.Count == filters.Count, "Expecting a result entry for every filter");
            return results;
        }

#if DEBUG
#if false
        static SearchFilter()
        {
            SearchFilter s = new SearchFilter();
            {
                SByte integer = 0x7f;
                s.Value = "0x7f";
                System.Diagnostics.Debug.Assert(Match(integer, s));
                s.Value = "127";
                System.Diagnostics.Debug.Assert(Match(integer, s));
                System.Diagnostics.Debug.Assert(!Match(integer - 1, s));
                System.Diagnostics.Debug.Assert(!Match(integer + 1, s));
            }
            {
                Int16 integer = 0x7fff;
                s.Value = "x7fff";
                System.Diagnostics.Debug.Assert(Match(integer, s));
                s.Value = "32767";
                System.Diagnostics.Debug.Assert(Match(integer, s));
                System.Diagnostics.Debug.Assert(!Match(integer - 1, s));
                System.Diagnostics.Debug.Assert(!Match(integer + 1, s));
            }
            {
                Int32 integer = 0x7fffffff;
                s.Value = "&h7fffffff";
                System.Diagnostics.Debug.Assert(Match(integer, s));
                s.Value = "2147483647";
                System.Diagnostics.Debug.Assert(Match(integer, s));
                System.Diagnostics.Debug.Assert(!Match(integer - 1, s));
                System.Diagnostics.Debug.Assert(!Match(integer + 1, s));
            }
            {
                Int64 integer = 0x7fffffffffffffff;
                s.Value = "7fffffffffffffff";
                System.Diagnostics.Debug.Assert(Match(integer, s));
                s.Value = "9223372036854775807";
                System.Diagnostics.Debug.Assert(Match(integer, s));
                System.Diagnostics.Debug.Assert(!Match(integer - 1, s));
                System.Diagnostics.Debug.Assert(!Match(integer + 1, s));
            }
            {
                Byte integer = 0xff;
                s.Value = "0xff";
                System.Diagnostics.Debug.Assert(Match(integer, s));
                s.Value = "255";
                System.Diagnostics.Debug.Assert(Match(integer, s));
                System.Diagnostics.Debug.Assert(!Match(integer - 1, s));
                System.Diagnostics.Debug.Assert(!Match(integer + 1, s));
            }
            {
                UInt16 integer = 0xffff;
                s.Value = "xffff";
                System.Diagnostics.Debug.Assert(Match(integer, s));
                s.Value = "65535";
                System.Diagnostics.Debug.Assert(Match(integer, s));
                System.Diagnostics.Debug.Assert(!Match(integer - 1, s));
                System.Diagnostics.Debug.Assert(!Match(integer + 1, s));
            }
            {
                UInt32 integer = 0xffffffff;
                s.Value = "&hffffffff";
                System.Diagnostics.Debug.Assert(Match(integer, s));
                s.Value = "4294967295";
                System.Diagnostics.Debug.Assert(Match(integer, s));
                System.Diagnostics.Debug.Assert(!Match(integer - 1, s));
                System.Diagnostics.Debug.Assert(!Match(integer + 1, s));
            }
            {
                UInt64 integer = 0xffffffffffffffff;
                s.Value = "ffffffffffffffff";
                System.Diagnostics.Debug.Assert(Match(integer, s));
                //s.Value = "‭‭‭‬‬‬";
                //System.Diagnostics.Debug.Assert(Match(integer, s));
                System.Diagnostics.Debug.Assert(!Match(integer - 1, s));
                System.Diagnostics.Debug.Assert(!Match(integer + 1, s));
            }
        }
#endif
#endif
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
