// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM		: WIN32 X64
// PRODUCT		: COMMON
// VISIBILITY	: CLIENT
//
// ------------------------------------------------------TKBMS v1.0

using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using System.Xml.Serialization;

namespace HavokVisualDebugger
{
    public static class hkReflect
    {
        public class BindingFlagsEx
        {
            public const BindingFlags AllPropsFlag = (BindingFlags.Instance | BindingFlags.Static | BindingFlags.Public | BindingFlags.NonPublic | BindingFlags.FlattenHierarchy);
            public const BindingFlags AllFieldsFlag = (BindingFlags.Instance | BindingFlags.Static | BindingFlags.Public | BindingFlags.NonPublic | BindingFlags.FlattenHierarchy);
        }

        public static bool IsBuiltinType(Type t)
        {
            if (t != null)
            {
                try
                {
                    return t.Module.ScopeName == "CommonLanguageRuntimeLibrary";
                }
                catch { }
            }
            return false;
        }

        public static bool IsBuiltin(object o)
        {
            if (o != null)
            {
                return IsBuiltinType(o.GetType());
            }
            return false;
        }

        public static bool IsFloatingPointType(Type t)
        {
            return
                (t == typeof(Single)) ||
                (t == typeof(Double)) ||
                (t == typeof(Decimal));
        }

        public static bool IsFloatingPoint(object o)
        {
            return
                (o is Single) ||
                (o is Double) ||
                (o is Decimal);
        }

        public static bool IsSignedIntegerType(Type t)
        {
            return
                (t == typeof(SByte)) ||
                (t == typeof(Int16)) ||
                (t == typeof(Int32)) ||
                (t == typeof(Int64));
        }

        public static bool IsSignedInteger(object o)
        {
            return
                (o is SByte) ||
                (o is Int16) ||
                (o is Int32) ||
                (o is Int64);
        }

        public static bool IsUnsignedIntegerType(Type t)
        {
            return
                (t == typeof(Byte)) ||
                (t == typeof(UInt16)) ||
                (t == typeof(UInt32)) ||
                (t == typeof(UInt64));
        }

        public static bool IsUnsignedInteger(object o)
        {
            return
                (o is Byte) ||
                (o is UInt16) ||
                (o is UInt32) ||
                (o is UInt64);
        }

        public static bool IsIntegerType(Type t)
        {
            return
                IsSignedIntegerType(t) ||
                IsUnsignedIntegerType(t);
        }

        public static bool IsInteger(object o)
        {
            return
                IsSignedInteger(o) ||
                IsUnsignedInteger(o);
        }

        public static bool IsNumericType(Type t)
        {
            return
                IsFloatingPointType(t) ||
                IsIntegerType(t);
        }

        public static bool IsNumeric(object o)
        {
            return
                IsFloatingPoint(o) ||
                IsInteger(o);
        }

        public static bool IsXmlIgnored(Type type, string propertyName)
        {
            return type.GetProperty(propertyName).GetCustomAttributes(false).Any(a => a is XmlIgnoreAttribute);
        }
    }

    public class hkToStringComparer : System.Collections.IComparer, IComparer<object>
    {
        public static hkToStringComparer CurrentCulture { get; private set; }
        public static hkToStringComparer CurrentCultureIgnoreCase { get; private set; }
        public static hkToStringComparer InvariantCulture { get; private set; }
        public static hkToStringComparer InvariantCultureIgnoreCase { get; private set; }
        public static hkToStringComparer Ordinal { get; private set; }
        public static hkToStringComparer OrdinalIgnoreCase { get; private set; }

        public Int32 Compare(Object x, Object y)
        {
            return
                m_stringComparer.Compare(
                    (x != null) ? x.ToString() : null,
                    (y != null) ? y.ToString() : null);
        }

        public hkToStringComparer(StringComparer stringComparer = null)
        {
            m_stringComparer = stringComparer ?? StringComparer.CurrentCulture;
        }

        static hkToStringComparer()
        {
            CurrentCulture = new hkToStringComparer(StringComparer.CurrentCulture);
            CurrentCultureIgnoreCase = new hkToStringComparer(StringComparer.CurrentCultureIgnoreCase);
            InvariantCulture = new hkToStringComparer(StringComparer.InvariantCulture);
            InvariantCultureIgnoreCase = new hkToStringComparer(StringComparer.InvariantCultureIgnoreCase);
            Ordinal = new hkToStringComparer(StringComparer.Ordinal);
            OrdinalIgnoreCase = new hkToStringComparer(StringComparer.OrdinalIgnoreCase);
        }

        private StringComparer m_stringComparer;
    }

    /// <summary>
    /// Restore read/write-able poperties of an object on dispose.
    /// </summary>
    public class hkRevertOnDispose : IDisposable
    {
        public hkRevertOnDispose(object o) :
            this(o, null)
        { }
        public hkRevertOnDispose(object o, params string[] propertyOrFieldNamesIn)
        {
            List<string> propertyOrFieldNames = (propertyOrFieldNamesIn != null) ? new List<string>(propertyOrFieldNamesIn) : null;
            PropertyInfo[] properties = (o != null) ? o.GetType().GetProperties(hkReflect.BindingFlagsEx.AllPropsFlag) : new PropertyInfo[] { };
            foreach (PropertyInfo property in properties)
            {
                if (property.CanRead && property.CanWrite && ((propertyOrFieldNames == null) || propertyOrFieldNames.Contains(property.Name)))
                {
                    try
                    {
                        object value = property.GetValue(o);
                        m_oldPropertyValues.Add(Tuple.Create(property, value));
                    }
                    catch { System.Diagnostics.Debug.Assert(false, "Failed to store property"); }
                }
            }
            FieldInfo[] fields = (o != null) ? o.GetType().GetFields(hkReflect.BindingFlagsEx.AllFieldsFlag) : new FieldInfo[] { };
            foreach (FieldInfo field in fields)
            {
                if ((propertyOrFieldNames == null) || propertyOrFieldNames.Contains(field.Name))
                {
                    try
                    {
                        object value = field.GetValue(o);
                        m_oldFieldValues.Add(Tuple.Create(field, value));
                    }
                    catch { System.Diagnostics.Debug.Assert(false, "Failed to store field"); }
                }
            }
            m_object = o;
        }

        public void Dispose()
        {
            foreach (Tuple<PropertyInfo, object> oldPropertyValue in m_oldPropertyValues)
            {
                try
                {
                    oldPropertyValue.Item1.SetValue(m_object, oldPropertyValue.Item2);
                }
                catch { System.Diagnostics.Debug.Assert(false, "Failed to revert property"); }
            }
            foreach (Tuple<FieldInfo, object> oldFieldValue in m_oldFieldValues)
            {
                try
                {
                    oldFieldValue.Item1.SetValue(m_object, oldFieldValue.Item2);
                }
                catch { System.Diagnostics.Debug.Assert(false, "Failed to revert property"); }
            }
        }

        public object Object { get { return m_object; } }

        private object m_object;
        private List<Tuple<PropertyInfo, object>> m_oldPropertyValues = new List<Tuple<PropertyInfo, object>>();
        private List<Tuple<FieldInfo, object>> m_oldFieldValues = new List<Tuple<FieldInfo, object>>();
    }

    /// <summary>
    /// Set a property and have it reset when this object is disposed.
    /// </summary>
    public class hkResetOnDispose : IDisposable
    {
        public hkResetOnDispose(object o, string propOrFieldName, object val) :
            this((Type)null, o, propOrFieldName, val)
        { }

        public hkResetOnDispose(Type staticType, string propOrFieldName, object val) :
            this(staticType, (object)null, propOrFieldName, val)
        { }

        public hkResetOnDispose(object o, string getterName, string setterName, object val) :
            this((Type)null, o, getterName, setterName, (object)null, val)
        { }

        public hkResetOnDispose(Type staticType, string getterName, string setterName, object val) :
            this(staticType, (object)null, getterName, setterName, (object)null, val)
        { }

        public hkResetOnDispose(object o, string getterName, string setterName, object accessor, object val) :
            this((Type)null, o, getterName, setterName, accessor, val)
        { }

        public hkResetOnDispose(Type staticType, string getterName, string setterName, object accessor, object val) :
            this(staticType, (object)null, getterName, setterName, accessor, val)
        { }

        protected hkResetOnDispose(Type type, object o, string propOrFieldName, object val)
        {
            if (o != null)
            {
                type = o.GetType();
            }
            else if (type == null)
            {
                return;
            }

            m_object = o;

            m_propInfo = type.GetProperty(propOrFieldName, hkReflect.BindingFlagsEx.AllPropsFlag);
            if (m_propInfo != null)
            {
                m_oldVal = m_propInfo.GetValue(m_object, null);
                m_propInfo.SetValue(m_object, val, null);
            }
            else
            {
                m_fieldInfo = type.GetField(propOrFieldName, hkReflect.BindingFlagsEx.AllFieldsFlag);
                m_oldVal = m_fieldInfo.GetValue(m_object);
                m_fieldInfo.SetValue(m_object, val);
            }
        }

        protected hkResetOnDispose(Type type, object o, string getterName, string setterName, object accessor, object val)
        {
            if (o != null)
            {
                type = o.GetType();
            }
            else if (type == null)
            {
                return;
            }

            m_object = o;
            m_methodAccessor = accessor;

            if (m_methodAccessor != null)
            {
                MethodInfo getterInfo = type.GetMethod(getterName, new Type[] { m_methodAccessor.GetType() });
                m_oldVal = getterInfo.Invoke(m_object, new object[] { m_methodAccessor });
                m_methodInfo = type.GetMethod(setterName, new Type[] { m_methodAccessor.GetType(), getterInfo.ReturnType });
                m_methodInfo.Invoke(m_object, new object[] { m_methodAccessor, val });
            }
            else
            {
                MethodInfo getterInfo = type.GetMethod(getterName, Type.EmptyTypes);
                m_oldVal = getterInfo.Invoke(m_object, null);
                m_methodInfo = type.GetMethod(setterName, new Type[] { getterInfo.ReturnType });
                m_methodInfo.Invoke(m_object, new object[] { val });
            }
        }

        public void Dispose()
        {
            if (m_propInfo != null)
            {
                m_propInfo.SetValue(m_object, m_oldVal, null);
            }
            else if (m_fieldInfo != null)
            {
                m_fieldInfo.SetValue(m_object, m_oldVal);
            }
            else if (m_methodInfo != null)
            {
                if (m_methodAccessor != null)
                {
                    m_methodInfo.Invoke(m_object, new object[] { m_methodAccessor, m_oldVal });
                }
                else
                {
                    m_methodInfo.Invoke(m_object, new object[] { m_oldVal });
                }
            }
        }

        public object Object { get { return m_object; } }

        private PropertyInfo m_propInfo;
        private FieldInfo m_fieldInfo;
        private MethodInfo m_methodInfo;
        private object m_object;
        private object m_oldVal;
        private object m_methodAccessor;
    }

    /// <summary>
    /// Conditionally set a property and have it reset when this object is disposed.
    /// </summary>
    public class hkConditionalResetOnDispose : hkResetOnDispose
    {
        public hkConditionalResetOnDispose(bool condition, object o, string propOrFieldName, object val) :
            base((Type)null, condition ? o : null, propOrFieldName, val)
        { }

        public hkConditionalResetOnDispose(bool condition, Type staticType, string propOrFieldName, object val) :
            base(condition ? staticType : null, (object)null, propOrFieldName, val)
        { }

        public hkConditionalResetOnDispose(bool condition, object o, string getterName, string setterName, object val) :
            base((Type)null, condition ? o : null, getterName, setterName, (object)null, val)
        { }

        public hkConditionalResetOnDispose(bool condition, Type staticType, string getterName, string setterName, object val) :
            base(condition ? staticType : null, (object)null, getterName, setterName, (object)null, val)
        { }

        public hkConditionalResetOnDispose(bool condition, object o, string getterName, string setterName, object accessor, object val) :
            base((Type)null, condition ? o : null, getterName, setterName, accessor, val)
        { }

        public hkConditionalResetOnDispose(bool condition, Type staticType, string getterName, string setterName, object accessor, object val) :
            base(condition ? staticType : null, (object)null, getterName, setterName, accessor, val)
        { }
    }

    /// <summary>
    /// Call a method on dispose.
    /// </summary>
    public class hkInvokeOnDispose : IDisposable
    {
        public hkInvokeOnDispose(object o, string methodName, object[] methodParams = null) :
            this((Type)null, o, methodName, methodParams, null)
        { }

        public hkInvokeOnDispose(Type staticType, string methodName, object[] methodParams = null) :
            this(staticType, null, methodName, methodParams, null)
        { }


        public hkInvokeOnDispose(object o, string methodName, object[] methodParams, Type[] methodParamTypes) :
            this((Type)null, o, methodName, methodParams, methodParamTypes)
        { }

        public hkInvokeOnDispose(Type staticType, string methodName, object[] methodParams, Type[] methodParamTypes) :
            this(staticType, null, methodName, methodParams, methodParamTypes)
        { }

        protected hkInvokeOnDispose(Type type, object o, string methodName, object[] methodParams, Type[] methodParamTypes)
        {
            if (o != null)
            {
                type = o.GetType();
            }
            else if (type == null)
            {
                return;
            }

            m_object = o;
            m_methodParams = methodParams;

            if (methodParamTypes != null)
            {
                m_methodInfo = type.GetMethod(methodName, methodParamTypes);
            }
            else if (methodParams != null)
            {
                m_methodInfo = type.GetMethod(methodName, (new List<object>(methodParams)).ConvertAll<Type>(p => (p == null) ? null : p.GetType()).ToArray());
            }
            else
            {
                m_methodInfo = type.GetMethod(methodName, Type.EmptyTypes);
            }
        }

        public void Dispose()
        {
            if (m_methodInfo != null)
            {
                m_methodInfo.Invoke(m_object, m_methodParams);
            }
        }

        public object Object { get { return m_object; } }

        private MethodInfo m_methodInfo;
        private object m_object;
        private object[] m_methodParams;
    }

    /// <summary>
    /// Conditionally call a method on dispose.
    /// </summary>
    public class hkConditionalInvokeOnDispose : hkInvokeOnDispose
    {
        public hkConditionalInvokeOnDispose(bool condition, object o, string methodName, object[] methodParams = null) :
            base((Type)null, condition ? o : null, methodName, methodParams, null)
        { }

        public hkConditionalInvokeOnDispose(bool condition, Type staticType, string methodName, object[] methodParams = null) :
            base(condition ? staticType : null, null, methodName, methodParams, null)
        { }

        public hkConditionalInvokeOnDispose(bool condition, object o, string methodName, object[] methodParams, Type[] methodParamTypes) :
            base((Type)null, condition ? o : null, methodName, methodParams, methodParamTypes)
        { }

        public hkConditionalInvokeOnDispose(bool condition, Type staticType, string methodName, object[] methodParams, Type[] methodParamTypes) :
            base(condition ? staticType : null, null, methodName, methodParams, methodParamTypes)
        { }
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
