// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM		: WIN32 X64
// PRODUCT		: COMMON
// VISIBILITY	: CLIENT
//
// ------------------------------------------------------TKBMS v1.0

using System;
using System.Collections.Generic;
using System.Windows;

namespace HavokVisualDebugger
{
    /// <summary>
    /// Flags which affect other cmd line arguments.
    /// </summary>
    [Flags]
    public enum hkCommandLineFlags
    {
        #region Debug

        /// <summary>
        /// Close after completing other cmd line operations
        /// </summary>
        AutoClose = (1 << 0),

        /// <summary>
        /// Sink files (save/record) will be human-readable text files
        /// </summary>
        WriteTextFile = (1 << 2)

        #endregion
    }

    public class hkCommandLine
    {
        /// <summary>
        /// Connection info, modified by -connect
        /// </summary>
        public static string ConnectAddress = "";
        public static string ConnectPort = "";

        /// <summary>
        /// File open info, modified by -open
        /// </summary>
        public static string OpenFileName = "";

        /// <summary>
        /// File load info, modified by -load
        /// </summary>
        public static string LoadFileName = "";

        /// <summary>
        /// Record info, modified by -record
        /// </summary>
        public static string RecordFileName = "";

        /// <summary>
        /// Whether to reset the settings file, modified by -resetSettings
        /// </summary>
        public static bool ResetSettings = false;

        /// <summary>
        /// Command line flags
        /// </summary>
        public static hkCommandLineFlags Flags;
    }

    public class hkCommandLineUtils
    {
        /// <summary>
        /// List of statically registered commands
        /// </summary>
        public static List<CommandLineCommandInfo> RegisteredCommands = new List<CommandLineCommandInfo>();

        public delegate string ParseCommandArgs(string[] args);

        /// <summary>
        /// Descriptor for a command
        /// </summary>
        public class CommandLineCommandInfo
        {
            public List<string> Names { get; set; }
            public int NumArgs { get; set; }
            public ParseCommandArgs ParseFunction { get; set; }
            public string HelpText { get; set; }
            public string SyntaxText { get; set; }

            public string ErrorText
            {
                get
                {
                    if (String.IsNullOrEmpty(_ErrorText))
                    {
                        return "'" + Names[0] + "' command expects " + NumArgs + " arguments.";
                    }
                    else
                    {
                        return _ErrorText;
                    }
                }

                set
                {
                    _ErrorText = value;
                }
            }
            private string _ErrorText = "";
        }

        /// <summary>
        /// Registers a command
        /// </summary>
        /// <param name="commandNames">List of command names.  The first name will be used in some default error text.
        /// All command names must start with a dash "-".  Recommended to include at least two command names, a long
        /// version and a short version (-help and -h).</param>
        /// <param name="numArgs">How many arguments this command expects.</param>
        /// <param name="errorString">The string to print when the user hasn't entered the correct amount of args</param>
        /// <param name="helpString">A string describing what the command does (displayed via -help)</param>
        /// <param name="syntaxString">A string describing the syntax of the command (displayed via -help)</param>
        /// <param name="parseFunc">A delegate function to be executed.  This does the actual work of the command.</param>
        public static void RegisterCommand(List<string> commandNames, int numArgs, string errorString, string helpString, string syntaxString, ParseCommandArgs parseFunc)
        {
            // Make sure command doesn't exist already
            foreach (string commandName in commandNames)
            {
                if (FindRegisteredCommand(commandName) != null)
                {
                    System.Diagnostics.Debug.Assert(false, "Command already exists.");
                    return;
                }

                if (!commandName.StartsWith("-"))
                {
                    System.Diagnostics.Debug.Assert(false, "Command must begin with a dash (-)");
                    return;
                }
            }

            // Create and register command
            CommandLineCommandInfo info = new CommandLineCommandInfo()
            {
                NumArgs = numArgs,
                ErrorText = errorString,
                HelpText = helpString,
                SyntaxText = syntaxString,
                ParseFunction = parseFunc
            };

            info.Names = new List<string>();
            info.Names.AddRange(commandNames);

            RegisteredCommands.Add(info);
        }

        /// <summary>
        /// // Find command in registered commands list by name
        /// </summary>
        public static CommandLineCommandInfo FindRegisteredCommand(string commandName)
        {
            foreach(CommandLineCommandInfo commandInfo in RegisteredCommands)
            {
                foreach (string commandInfoName in commandInfo.Names)
                {
                    if (commandInfoName == commandName)
                    {
                        return commandInfo;
                    }
                }
            }

            return null;
        }

        /// <summary>
        /// Register commands to the static list of commands.
        /// </summary>
        public static void RegisterCommands()
        {
            // Connect command
            RegisterCommand(new List<string> { "-connect", "-c" }, 1,
                "-connect argument requires a host:port argument (eg localhost:25001)",
                "Connects to a host machine with a specified port",
                "-connect host:port",
                delegate (string[] args)
                {
                    string[] pairedArgs = args[0].Split(':');
                    if (pairedArgs.Length == 2)
                    {
                        hkCommandLine.ConnectAddress = pairedArgs[0];
                        hkCommandLine.ConnectPort = pairedArgs[1];
                        int port;
                        if (!int.TryParse(hkCommandLine.ConnectPort, out port))
                        {
                            return "Invalid port of \"" + hkCommandLine.ConnectPort + "\" for -connect argument";
                        }
                    }
                    else
                    {
                        return "-connect argument requires a host:port argument (eg localhost:25001)";
                    }
                    return "";
                });

            // Open command
            RegisterCommand(new List<string> { "-open", "-o" }, 1,
                "-open argument requires the name of a file to open (eg -open c:\\movie.hkm)",
                "Opens a VDB movie file",
                "-open filepath",
                delegate (string[] args)
                {
                    string arg = args[0];
                    if (System.IO.File.Exists(arg))
                    {
                        hkCommandLine.OpenFileName = arg;
                    }
                    else
                    {
                        return "The file provided for '-open' does not exist.";
                    }
                    return "";
                });

            // Load command
            RegisterCommand(new List<string> { "-load", "-l" }, 1,
                "-load argument requires the name of a file to load (eg -load c:\\movie.hkm)",
                "Loads a VDB movie file (opens, then reads and caches all frames)",
                "-load filepath",
                delegate (string[] args)
                {
                    string arg = args[0];
                    if (System.IO.File.Exists(arg))
                    {
                        hkCommandLine.LoadFileName = arg;
                    }
                    else
                    {
                        return "The file provided for '-load' does not exist.";
                    }
                    return "";
                });

            // Convert command
            RegisterCommand(new List<string> { "-convert", "-v" }, 1,
                "-convert argument requires the name of a file to convert (eg -convert c:\\movie.hkm)",
                "Converts a VDB movie file (opens, reads and caches all frames, and saves to human-readable text file)",
                "-convert filepath",
                delegate (string[] args)
                {
                    string arg = args[0];
                    if (System.IO.File.Exists(arg))
                    {
                        hkCommandLine.LoadFileName = arg;
                        hkCommandLine.Flags |= ( hkCommandLineFlags.WriteTextFile | hkCommandLineFlags.AutoClose );
                    }
                    else
                    {
                        return "The file provided for '-convert' does not exist.";
                    }
                    return "";
                });

            // Record command
            RegisterCommand(new List<string> { "-record", "-r" }, 1,
                "-record argument requires the name of a file to record to (eg -record c:\\movie.hkm)",
                "Records a VDB movie file from source defined by other command line option such as connect/open/load",
                "-record filepath",
                delegate (string[] args)
                {
                    string arg = args[0];
                    hkCommandLine.RecordFileName = arg;
                    return "";
                });

            // Auto close
            RegisterCommand(new List<string> { "-autoclose", "-a" }, 0,
                "-autoclose does not have any arguments",
                "Causes the tool to close after it's completed other command line options",
                "-autoclose",
                delegate (string[] args)
                {
                    hkCommandLine.Flags |= hkCommandLineFlags.AutoClose;
                    return "";
                });

            // Write human-readable text file
            RegisterCommand(new List<string> { "-writetext", "-w" }, 0,
                "-writetext does not have any arguments",
                "The tool will use a human-readable text format for recording file (if -record isn't specified a default name will be used)",
                "-writetext",
                delegate (string[] args)
                {
                    hkCommandLine.Flags |= hkCommandLineFlags.WriteTextFile;
                    return "";
                });

            // Delete settings file
            RegisterCommand(new List<string> { "-resetSettings", "-s" }, 0,
                "-resetSettings does not have any arguments",
                "Resets the VDB settings file to default settings.",
                "-resetSettings",
                delegate (string[] args)
                {
                    Properties.Settings.Default.Reset();
                    Properties.Settings.Default.Save();
                    VdbSettings.ForceLoad();
                    return "";
                });

            // Print system-specific diagnostic information
            RegisterCommand(new List<string> { "-diag", "-d" }, 0,
                "-diag does not have any arguments",
                "Prints a diagnostic file to help troubleshoot UI problems",
                "-diag",
                delegate (string[] args)
                {
                    string fileName = "diag" + System.DateTime.Now.ToFileTime() + ".txt";
                    try
                    {
                        using (System.IO.StreamWriter file = new System.IO.StreamWriter(fileName))
                        {
                            // Print out theme
                            file.WriteLine("Current Theme:");
                            file.WriteLine(hkDebugUtils.GetThemeName());
                            file.WriteLine("\n\n");

                            file.WriteLine("Current DPI scaling:");
                            file.WriteLine("\tScaling Factor 1: "  + hkDebugUtils.GetScalingFactor1());
                            file.WriteLine("\tScaling Factor 2: " + hkDebugUtils.GetScalingFactor2());
                            file.WriteLine("\n\n");

                            // Print out SystemParameters
                            file.WriteLine("SystemParameters:");
                            Type type = typeof(SystemParameters);
                            foreach (System.Reflection.PropertyInfo propInfo in type.GetProperties())
                            {
                                string name = propInfo.Name;
                                if (name.EndsWith("Key"))
                                {
                                    continue;
                                }
                                object propVal = propInfo.GetValue(null);
                                file.WriteLine("{0}={1}", name, propVal);
                            }
                        }
                    }
                    catch
                    {
                        return "Could not access " + fileName;
                    }

                    return "";
                });

            // Help command
            RegisterCommand(new List<string> { "-help", "-h", "-?" }, 0, "", "Displays this help list", "-help",
                delegate (string[] args)
                {
                    string helpStr = "Syntax: \n";

                    // Determine padding for the syntax text
                    int maxSyntaxLength = 1;
                    foreach (CommandLineCommandInfo commandInfo in RegisteredCommands)
                    {
                        maxSyntaxLength = Math.Max(commandInfo.SyntaxText.Length, maxSyntaxLength);
                    }

                    // Construct help string
                    foreach (CommandLineCommandInfo commandInfo in RegisteredCommands)
                    {
                        helpStr += "\t" + commandInfo.SyntaxText.PadRight(maxSyntaxLength) + "  (" + commandInfo.HelpText + ")\n";
                    }

                    WriteStringToConsole(helpStr, false);
                    return null;
                });
        }

        /// <summary>
        /// Given a list of command line arguments, parses and executes the commands.
        /// </summary>
        /// <param name="args">List of command line args</param>
        /// <returns>
        /// An empty string if program should continue execution
        /// A null string if the program should halt
        /// An error string if the program should halt and print the error</returns>
        public static string ParseCommandLineArgs(string[] args)
        {
            // No arguments, early-out
            if (args.Length == 0)
            {
                return "";
            }

            RegisterCommands();

            // The following is true when a user associates an extension with the Vdb and
            // double-clicks on it.  Only one argument will be present: the file path.
            // This makes the assumption that all command line flags start with a "-"
            if ((args.Length == 1) && !args[0].StartsWith("-"))
            {
                // Redirect to open command
                args = new string[] { "-open", args[0] };
            }

            // Parse arguments
            for (int i = 0; i < args.Length; i++)
            {
                string arg = args[i];
                CommandLineCommandInfo commandInfo = FindRegisteredCommand(arg);
                if (commandInfo != null)
                {
                    List<string> commandArgs = new List<string>();
                    if (commandInfo.NumArgs == -1)
                    {
                        while (((i + 1) < args.Length) && !args[i + 1].StartsWith("-"))
                        {
                            string commandArg = args[++i];
                            commandArgs.Add(commandArg);
                        }
                    }
                    else
                    {
                        for (int j = 0; j < commandInfo.NumArgs; j++)
                        {
                            if ((i + 1) >= args.Length)
                            {
                                return commandInfo.ErrorText;
                            }

                            string commandArg = args[++i];
                            if (commandArg.StartsWith("-"))
                            {
                                return commandInfo.ErrorText;
                            }

                            commandArgs.Add(commandArg);
                        }
                    }

                    string result = commandInfo.ParseFunction(commandArgs.ToArray());

                    // Early-out if command failed
                    if (result != "")
                    {
                        return result;
                    }
                }
                else
                {
                    return "Unknown argument " + arg;
                }
            }

            return "";
        }

        /// <summary>
        /// Prints a string to console
        /// </summary>
        public static void WriteStringToConsole(string outputString, bool isError)
        {
            string errorString = isError ? "Error: " : "";
            Console.WriteLine(errorString + outputString);
        }

        public static void Apply(VdbViewModel vm)
        {
            // Note: order matters

            // Setup a recording target if requested
            if (hkCommandLine.Flags.HasFlag(hkCommandLineFlags.WriteTextFile))
            {
                // Explicitly defined filename
                if (!String.IsNullOrEmpty(hkCommandLine.RecordFileName))
                {
                    vm.QueueAutoStartDebugRecording(hkCommandLine.RecordFileName);
                }
                // Derived from file source
                else if (!String.IsNullOrEmpty(hkCommandLine.LoadFileName))
                {
                    vm.QueueAutoStartDebugRecording(hkCommandLine.LoadFileName + ".txt");
                }
                else if (!String.IsNullOrEmpty(hkCommandLine.OpenFileName))
                {
                    vm.QueueAutoStartDebugRecording(hkCommandLine.OpenFileName + ".txt");
                }
                else if (!String.IsNullOrEmpty(hkCommandLine.ConnectAddress) && !String.IsNullOrEmpty(hkCommandLine.ConnectPort))
                {
                    vm.QueueAutoStartDebugRecording("IP_" + hkCommandLine.ConnectAddress.Replace(".", "_") + "_" + hkCommandLine.ConnectPort + ".hkm.txt");
                }
            }
            else if (!String.IsNullOrEmpty(hkCommandLine.RecordFileName))
            {
                vm.QueueAutoStartRecording(hkCommandLine.RecordFileName);
            }

            // Setup our connecting source if requested
            if (!String.IsNullOrEmpty(hkCommandLine.LoadFileName))
            {
                vm.DoConnectToFileName(hkCommandLine.LoadFileName, true);
            }
            else if (!String.IsNullOrEmpty(hkCommandLine.OpenFileName))
            {
                vm.DoConnectToFileName(hkCommandLine.OpenFileName, false);
            }
            else if (!String.IsNullOrEmpty(hkCommandLine.ConnectAddress) && !String.IsNullOrEmpty(hkCommandLine.ConnectPort))
            {
                vm.AutoReconnectEnabled = true;
                vm.MachineAddress = hkCommandLine.ConnectAddress;
                vm.MachinePort = hkCommandLine.ConnectPort;
                UInt16 port;
                if (UInt16.TryParse(vm.MachinePort, out port))
                {
                    vm.Client.AttemptAutoReconnect = new Tuple<string, UInt16>(vm.MachineAddress, port);
                }
                else
                {
                    System.Diagnostics.Debug.Assert(false, "Didn't fail on parse properly");
                    Application.Current.Dispatcher.InvokeAsync(() =>
                    {
                        Application.Current.Shutdown();
                    });
                }
            }

            // Hook up our callbacks to close when done if requested
            if (hkCommandLine.Flags.HasFlag(hkCommandLineFlags.AutoClose))
            {
                // Don't allow any prompts to interrupt us
                vm.AllowAutoPrompts = false;

                if (!String.IsNullOrEmpty(hkCommandLine.LoadFileName) || !String.IsNullOrEmpty(hkCommandLine.OpenFileName))
                {
                    // Close when we are done with file
                    int stage = 0;
                    vm.Client.ListenTo<Havok.Vdb.ConnectionHandler, Havok.Vdb.Client>(
                        nameof(Havok.Vdb.Client.FileReadCompleted),
                        delegate (Havok.Vdb.Client sender)
                        {
                            stage = 1;
                        });
                    vm.Client.ListenTo<Havok.Vdb.ConnectionHandler, Havok.Vdb.Client>(
                        nameof(Havok.Vdb.Client.FileProcessingCompleted),
                        delegate (Havok.Vdb.Client sender)
                        {
                            if (stage == 1)
                            {
                                Application.Current.Dispatcher.InvokeAsync(() =>
                                {
                                    Application.Current.Shutdown();
                                });
                            }
                        });
                }
                else if (!String.IsNullOrEmpty(hkCommandLine.ConnectAddress) && !String.IsNullOrEmpty(hkCommandLine.ConnectPort))
                {
                    // Close when we have completed connection or failed to connect
                    int stage = 0;
                    vm.Client.ListenTo<Havok.Vdb.ConnectionHandler, Havok.Vdb.Client>(
                        nameof(Havok.Vdb.Client.Connecting),
                        delegate (Havok.Vdb.Client sender)
                        {
                            stage = 1;
                        });
                    vm.Client.ListenTo<Havok.Vdb.ConnectionHandler, Havok.Vdb.Client>(
                        nameof(Havok.Vdb.Client.Connected),
                        delegate (Havok.Vdb.Client sender)
                        {
                            if (stage == 1)
                            {
                                Application.Current.Dispatcher.InvokeAsync(() =>
                                {
                                    Application.Current.Shutdown();
                                });
                            }
                        });
                    vm.Client.ListenTo<Havok.Vdb.ConnectionHandler, Havok.Vdb.Client>(
                        nameof(Havok.Vdb.Client.Disconnecting),
                        delegate (Havok.Vdb.Client sender)
                        {
                            if (stage == 1)
                            {
                                Application.Current.Dispatcher.InvokeAsync(() =>
                                {
                                    Application.Current.Shutdown();
                                });
                            }
                        });
                }
            }
        }
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
