// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM		: WIN32 X64
// PRODUCT		: COMMON
// VISIBILITY	: CLIENT
//
// ------------------------------------------------------TKBMS v1.0

using System;
using System.Linq;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;

namespace HavokVisualDebugger
{
    /// <summary>
    /// A custom observable collection with the following properties which doesn't use the Reset action.
    /// </summary>
    public class hkNonResettableObservableCollection<T> : ObservableCollection<T>
    {
        public hkNonResettableObservableCollection() : base() { }
        public hkNonResettableObservableCollection(List<T> list) : base(list) { }
        public hkNonResettableObservableCollection(IEnumerable<T> collection) : base(collection) { }

        protected override void ClearItems()
        {
            // Base class impl sends a reset which means we don't have proper collection information
            if (Count > 0)
            {
                // One at a time because some listeners don't support range add/removes.
                for (int i = Items.Count - 1; i >= 0; i--)
                {
                    T item = Items[i];
                    NotifyCollectionChangedEventArgs args =
                        new NotifyCollectionChangedEventArgs(
                            NotifyCollectionChangedAction.Remove,
                            new List<T> { item },
                            i);
                    Items.RemoveAt(i);
                    OnCollectionChanged(args);
                }
            }
        }
    }

    /// <summary>
    /// A custom observable collection with the following properties:
    /// 1) Reset action never occurs, instead a clear causes a Remove action for all items in the collection
    /// 2) Add/Remove,All/Range is available and will use a -1 starting index for notifications (as objects are pooled from across the collection)
    /// Note: this is not safe with all UI controls as they don't all support range removal/adding
    /// </summary>
    public class hkObservableCollection<T> : hkNonResettableObservableCollection<T>
    {
        public hkObservableCollection() : base() { }
        public hkObservableCollection(List<T> list) : base(list) { }
        public hkObservableCollection(IEnumerable<T> collection) : base(collection) { }

        public void AddRange(IEnumerable<T> collection)
        {
            List<T> added = new List<T>(collection.Count());
            foreach (T item in collection)
            {
                Items.Add(item);
                added.Add(item);
            }
            NotifyCollectionChangedEventArgs args =
                new NotifyCollectionChangedEventArgs(
                    NotifyCollectionChangedAction.Add,
                    added,
                    -1);
            OnCollectionChanged(args);
        }
        public bool RemoveAll(T item)
        {
            if (RemoveAllNoNotifyInternal(item) > 0)
            {
                NotifyCollectionChangedEventArgs args =
                    new NotifyCollectionChangedEventArgs(
                        NotifyCollectionChangedAction.Remove,
                        new List<T> { item },
                        -1);
                OnCollectionChanged(args);
                return true;
            }
            else
            {
                return false;
            }
        }
        public bool RemoveAll(Predicate<T> criteria)
        {
            List<T> removed = RemoveAllNoNotifyInternal(criteria);
            if (removed.Count > 0)
            {
                NotifyCollectionChangedEventArgs args =
                    new NotifyCollectionChangedEventArgs(
                        NotifyCollectionChangedAction.Remove,
                        removed,
                        -1);
                OnCollectionChanged(args);
                return true;
            }
            else
            {
                return false;
            }
        }
        public bool RemoveRange(IEnumerable<T> collection)
        {
            List<T> removedItems = new List<T>(collection.Count());
            foreach (T item in collection)
            {
                if (RemoveAllNoNotifyInternal(item) > 0)
                {
                    removedItems.Add(item);
                }
            }
            NotifyCollectionChangedEventArgs args =
                new NotifyCollectionChangedEventArgs(
                    NotifyCollectionChangedAction.Remove,
                    removedItems,
                    -1);
            OnCollectionChanged(args);
            return (removedItems.Count > 0);
        }
        protected int RemoveAllNoNotifyInternal(T item)
        {
            int removed = 0;
            while (Items.Remove(item)) removed++;
            return removed;
        }
        protected List<T> RemoveAllNoNotifyInternal(Predicate<T> criteria)
        {
            List<T> removed = new List<T>(Count);
            for (int i = Items.Count - 1; i >= 0; i--)
            {
                T item = Items[i];
                bool? result = criteria?.Invoke(item);
                if (result.HasValue && result.Value)
                {
                    Items.RemoveAt(i);
                    removed.Add(item);
                }
            }
            return removed.Distinct().ToList();
        }
        protected override void ClearItems()
        {
            // Base class impl sends a reset which means we don't have proper collection information
            if (Count > 0)
            {
                NotifyCollectionChangedEventArgs args =
                    new NotifyCollectionChangedEventArgs(
                        NotifyCollectionChangedAction.Remove,
                        new List<T>(Items),
                        0);
                Items.Clear();
                OnCollectionChanged(args);
            }
        }
    }

    public class hkCollectionUtils
    {
        public static void InsertIntoLimitedStringCollection(ObservableCollection<string> collection, string value, int maxCollectionSize)
        {
            if (!String.IsNullOrEmpty(value) && !collection.Contains(value))
            {
                if (collection.Count >= maxCollectionSize)
                {
                    collection.RemoveAt(collection.Count - 1);
                }
                collection.Insert(0, value);
            }
        }

        public static void InitializeLimitedStringCollection(ObservableCollection<string> dst, StringCollection src, int maxCollectionSize)
        {
            System.Diagnostics.Debug.Assert(dst != null && src != null, "Invalid parameters");
            if (dst != null && src != null)
            {
                foreach (string srcString in src)
                {
                    if (dst.Count >= maxCollectionSize)
                    {
                        break;
                    }
                    dst.Insert(0, srcString);
                }
            }
        }

        public static void InitializeLimitedStringCollectionFromSettings(ObservableCollection<string> collection, string savedPropertySetting, int maxCollectionSize)
        {
            InitializeLimitedStringCollection(collection, VdbSettings.GetValue<StringCollection>(savedPropertySetting), maxCollectionSize);
        }
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
