// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM		: WIN32 X64
// PRODUCT		: COMMON
// VISIBILITY	: CLIENT
//
// ------------------------------------------------------TKBMS v1.0

// Define this to debug XAML exceptions (it's quite slow though)
//#define XAML_DEBUG_SLOW

// Define this to use the render surface.  Can be handy to undefine if needing to just debug the UI
#define DISPLAY_RENDER_SURFACE

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Diagnostics;
using System.Windows;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Interop;
using System.Windows.Media.Animation;
using System.Windows.Navigation;

namespace HavokVisualDebugger
{
    /// <summary>
    /// Interaction logic for MainWindow.xaml
    /// </summary>
    public partial class MainWindow : Window
    {
        public VdbViewModel ViewModel { get; set; }
        private bool StartedUp = false;

        private GridLength ContentGrid_InspectorSizeStored = new GridLength(0);
        private GridLength ContentGrid_LogViewerSizeStored = new GridLength(0);
        private GridLength ContentGrid_DiagnosticsSizeStored = new GridLength(0);
        private GridLength ContentGrid_ViewportSizeStored = new GridLength(0);
        private GridLength ContentGrid_ViewportWidthSizeStored = new GridLength(0);

        // When this key is pressed, it toggles a debug mode
        // When it's held, the debug mode turns on until it's released and some debug actions can be performed
        private Key s_debugKey = Key.OemQuestion;
        private bool s_debugKeyHeld = false;

        public MainWindow()
        {
#if XAML_DEBUG_SLOW
            System.Diagnostics.PresentationTraceSources.Refresh();
            System.Diagnostics.PresentationTraceSources.MarkupSource.Switch.Level = System.Diagnostics.SourceLevels.All;
            System.Diagnostics.PresentationTraceSources.MarkupSource.Listeners.Add(new System.Diagnostics.DefaultTraceListener());
#endif
            ViewModel = App.ViewModel;
            DataContext = App.ViewModel;

            try
            {
                InitializeComponent();
            }
            catch (Exception e)
            {
                hkDebugUtils.DebugXamlException(e);
            }

            Keyboard.AddPreviewKeyDownHandler(
                Application.Current.MainWindow,
                (s, e) =>
                {
                    if (e.Key == s_debugKey)
                    {
                        if (e.IsRepeat)
                        {
                            s_debugKeyHeld = true;
                            ViewModel.PerformDebugActionsEnabled = true;
                            ViewModel.IsDebugViewEnabled = true;
                        }
                    }
                });
            Keyboard.AddPreviewKeyUpHandler(
                Application.Current.MainWindow,
                (s, e) =>
                {
                    if (e.Key == s_debugKey)
                    {
                        if (s_debugKeyHeld)
                        {
                            s_debugKeyHeld = false;
                            ViewModel.PerformDebugActionsEnabled = false;
                            ViewModel.IsDebugViewEnabled = false;
                        }
                        else
                        {
                            ViewModel.IsDebugViewEnabled = !ViewModel.IsDebugViewEnabled;
                        }
                    }
                });
            KeyEventHandler focusOnSelection = delegate(object sender, KeyEventArgs e)
            {
                if (e.Key == Key.F)
                {
                    if (ViewModel.CameraModeIsTrackballModeMaya)
                    {
                        ViewModel.DefaultSelectionView(null);
                    }
                }
                else if (e.Key == Key.Z)
                {
                    if (ViewModel.CameraModeIsTrackballModeMax)
                    {
                        ViewModel.DefaultSelectionView(null);
                    }
                }
            };
            Keyboard.AddPreviewKeyUpHandler(VdbRenderSurface, focusOnSelection);
            Keyboard.AddPreviewKeyUpHandler(VdbObjectInspectionWidgetUi.VdbObjectTreeViewUi, focusOnSelection);

            WindowInteropHelper interopHelper = new WindowInteropHelper(this);
            System.Windows.Forms.Screen currentScreen = System.Windows.Forms.Screen.FromHandle(interopHelper.Handle);
            {
                this.MaxHeight = currentScreen.WorkingArea.Height;

                double windowWidth = VdbSettings.GetValue<double>(nameof(Properties.Settings.VdbUi_WindowWidth));
                double windowHeight = VdbSettings.GetValue<double>(nameof(Properties.Settings.VdbUi_WindowHeight));
                if (windowHeight > 10 && windowWidth > 10)
                {
                    this.Width = windowWidth;
                    this.Height = windowHeight;
                }
                else
                {
                    // If stored settings get too low, just restore to a percentage of screen size
                    this.Width = currentScreen.WorkingArea.Width * 0.8f;
                    this.Height = currentScreen.WorkingArea.Height * 0.8f;
                }

                double windowTop = VdbSettings.GetValue<double>(nameof(Properties.Settings.VdbUi_WindowTop));
                double windowLeft = VdbSettings.GetValue<double>(nameof(Properties.Settings.VdbUi_WindowLeft));
                if (windowTop >= 0 && windowLeft >= 0)
                {
                    if (this.Top > (currentScreen.WorkingArea.Y + currentScreen.WorkingArea.Height))
                    {
                        this.Top = currentScreen.WorkingArea.Y;
                    }

                    if (this.Left > (currentScreen.WorkingArea.X + currentScreen.WorkingArea.Width))
                    {
                        this.Left = currentScreen.WorkingArea.X;
                    }
                }
                else
                {
                    this.Top = currentScreen.WorkingArea.Y;
                    this.Left = currentScreen.WorkingArea.X;
                }
            }

            this.WindowState = VdbSettings.GetValue<WindowState>(nameof(Properties.Settings.VdbUi_WindowMaximized));
            if (this.WindowState == WindowState.Minimized)
            {
                this.WindowState = WindowState.Normal;
            }

            ContentGrid_InspectorSizeStored = VdbSettings.GetValue<GridLength>(nameof(Properties.Settings.VdbUi_InspectionSize));
            ContentGrid_LogViewerSizeStored = VdbSettings.GetValue<GridLength>(nameof(Properties.Settings.VdbUi_LogViewerSize));
            ContentGrid_DiagnosticsSizeStored = VdbSettings.GetValue<GridLength>(nameof(Properties.Settings.VdbUi_DiagnosticsSize));
            ContentGrid_ViewportSizeStored = VdbSettings.GetValue<GridLength>(nameof(Properties.Settings.VdbUi_ViewportHeightSize));
            ContentGrid_ViewportWidthSizeStored = VdbSettings.GetValue<GridLength>(nameof(Properties.Settings.VdbUi_ViewportWidthSize));
            ViewportRowDefinition.Height = ContentGrid_ViewportSizeStored;
            ViewportColumnDefinition.Width = ContentGrid_ViewportWidthSizeStored;

            // Apply app settings
            ViewModel.InitializeTheme(VdbSettings.GetValue<string>(nameof(Properties.Settings.VdbUi_ColorTheme)));
            ViewModel.FontSize = VdbSettings.GetValue<double>(nameof(Properties.Settings.VdbUi_FontSize));
            ViewModel.IsAllowingTelemetry = VdbSettings.GetValue<bool>(nameof(Properties.Settings.VdbSettings_AllowTelemetry));

            this.SourceInitialized += (s, e) =>
            {
#if DISPLAY_RENDER_SURFACE
                StartedUp = true;
                ViewModel.StartProcessing(VdbRenderSurface);
#endif
                // Apply command line after settings, so that settings don't squash command line options
                VdbSettings.ApplyTo(ViewModel);
                hkCommandLineUtils.Apply(ViewModel);
            };

            this.Loaded += (s,e) =>
            {
                // Some settings are applied directly to main window *after* it's loaded.
                VdbSettings.ApplyTo(this);
            };

            this.Closing += (s, e) =>
            {
#if DISPLAY_RENDER_SURFACE
                // Aren't guaranteed to get OnSourceInitialized before OnClosing (the app may be shutting down early)
                // so need to track if processing was actually started
                if (StartedUp)
                {
                    ViewModel.StopProcessing();
                }
#endif
            };

            
            
            //this.Closed += delegate (object sender, EventArgs e)
            //{
            //    VdbModel.VdbClient.Client.Dispose();
            //};

            // App shortcuts that are modifiable by the user.
            
            
            (new List<VdbShortcutViewModel>(ViewModel.VdbCommands.CommandShortcuts)).ForEach(s =>
            {
                if (s.InputBinding != null)
                {
                    InputBindings.Add(s.InputBinding);
                }
            });

            // App shortcuts that are not modifiable by the user (don't show up in shortcuts menu).
            
            {
                Tuple<string, ICommand>[] appKeyToCommands =
                {
                    Tuple.Create("Esc", ViewModel.VdbCommands.UnselectAllCommand)
                };

                foreach (Tuple<string, ICommand> appKeyToCommand in appKeyToCommands)
                {
                    try
                    {
                        KeyGestureConverter kgc = new KeyGestureConverter();
                        InputGesture gesture = (kgc.ConvertFromString(appKeyToCommand.Item1) as InputGesture);
                        InputBindings.Add(new InputBinding(appKeyToCommand.Item2, gesture));
                    }
                    catch
                    {
                        System.Diagnostics.Debug.Assert(false, "Failed to apply shortcut");
                    }
                }
            }

            hkTelemetry.Provider.SessionEnded += Provider_SessionEnded;

            hkMouseDoubleClickDetector d = new hkMouseDoubleClickDetector();
            d.Hook(VdbRenderSurface, delegate (object sender, MouseButtonEventArgs e)
            {
                Point p = e.GetPosition(sender as IInputElement);

                // Apply DPI scaling to cursor position
                PresentationSource source = PresentationSource.FromVisual(this);
                double dpiScale = 1.0f;
                if (source != null)
                {
                    dpiScale = source.CompositionTarget.TransformToDevice.M11;
                }

                Tuple<Havok.Vdb.RenderObject, Havok.Vdb.Vector, Havok.Vdb.Vector> hit = ViewModel.Client.RenderSurface.HitTest((int)(p.X * dpiScale), (int)(p.Y * dpiScale));
                if (hit != null)
                {
                    ViewModel.Selection.SelectedRenderObject = hit.Item1;
                    ViewModel.Client.RenderSurface.SetLocalPointOfInterest(hit.Item1, hit.Item3);
                }
                else
                {
                    ViewModel.Selection.SelectedRenderObject = null;
                    // Note: We don't clear POI in this case.
                    //       This matches back-end behavior and is so that POIs can stick to objects
                    //       and you can watch them over frames w/o requiring selection to be maintained.
                    // ViewModel.Client.RenderSurface.ClearPointOfInterest();
                }
            });
        }

        private void Provider_SessionEnded()
        {
            string uiDimensions = "(" + Width.ToString("F2") + " x " + Height.ToString("F2") + ")";
            string workingArea;
            string screenResolution;
            bool isPrimaryMonitor;
            {
                WindowInteropHelper interopHelper = new WindowInteropHelper(this);
                System.Windows.Forms.Screen currentScreen = System.Windows.Forms.Screen.FromHandle(interopHelper.Handle);
                workingArea = "(" + currentScreen.WorkingArea.Width + " x " + currentScreen.WorkingArea.Height + ")";
                screenResolution = "(" + currentScreen.Bounds.Width + " x " + currentScreen.Bounds.Height + ")";
                isPrimaryMonitor = currentScreen.Primary;
            }

            hkTelemetry.Provider.SizeInfo(uiDimensions, screenResolution, workingArea, isPrimaryMonitor, System.Windows.Forms.Screen.AllScreens.Length);
        }

        private void RightContentGrid_PreviewMouseLeftButtonDown(object sender, MouseButtonEventArgs e)
        {
            if (!IsNavPanelPinned)
            {
                ResetCurrentlySelectedPanel();
            }
        }

        private bool ToggleNavPanel(bool isPanelSelected)
        {
            ResetCurrentlySelectedPanel();
            bool shouldPanelHide = !isPanelSelected;

            // Unpin the panel if we've toggled this panel off (all other panels should already be off care of the reset)
            IsNavPanelPinned = IsNavPanelPinned && shouldPanelHide;

            return shouldPanelHide;
        }

        private void File_Click(object sender, RoutedEventArgs e)
        {
            IsFilePanelSelected = ToggleNavPanel(IsFilePanelSelected);
        }

        private void Connect_Click(object sender, RoutedEventArgs e)
        {
            IsConnectPanelSelected = ToggleNavPanel(IsConnectPanelSelected);
        }

        private void CameraBtn_Click(object sender, RoutedEventArgs e)
        {
            IsCameraPanelSelected = ToggleNavPanel(IsCameraPanelSelected);
        }

        private void ViewersBtn_Click(object sender, RoutedEventArgs e)
        {
            IsViewersPanelSelected = ToggleNavPanel(IsViewersPanelSelected);
        }

        private void SettingsBtn_Click(object sender, RoutedEventArgs e)
        {
            IsSettingsPanelSelected = ToggleNavPanel(IsSettingsPanelSelected);
        }

        private void KeyboardShortcutsBtn_Click(object sender, RoutedEventArgs e)
        {
            IsKeyboardPanelSelected = ToggleNavPanel(IsKeyboardPanelSelected);
        }

        private void HelpBtn_Click(object sender, RoutedEventArgs e)
        {
            IsHelpPanelSelected = ToggleNavPanel(IsHelpPanelSelected);
        }

        #region Attached Properties

        public bool IsConnectPanelSelected
        {
            get { return (bool)GetValue(IsConnectPanelSelectedProperty); }
            set { SetValue(IsConnectPanelSelectedProperty, value); }
        }

        // Using a DependencyProperty as the backing store for IsConnectPanelSelected.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty IsConnectPanelSelectedProperty =
            DependencyProperty.Register("IsConnectPanelSelected", typeof(bool), typeof(MainWindow), new PropertyMetadata(false));

        public bool IsSettingsPanelSelected
        {
            get { return (bool)GetValue(IsSettingsPanelSelectedProperty); }
            set { SetValue(IsSettingsPanelSelectedProperty, value); }
        }

        // Using a DependencyProperty as the backing store for IsSettingsPanelSelected.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty IsSettingsPanelSelectedProperty =
            DependencyProperty.Register("IsSettingsPanelSelected", typeof(bool), typeof(MainWindow), new PropertyMetadata(false));

        public bool IsKeyboardPanelSelected
        {
            get { return (bool)GetValue(IsKeyboardPanelSelectedProperty); }
            set { SetValue(IsKeyboardPanelSelectedProperty, value); }
        }

        // Using a DependencyProperty as the backing store for IsKeyboardPanelSelected.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty IsKeyboardPanelSelectedProperty =
            DependencyProperty.Register("IsKeyboardPanelSelected", typeof(bool), typeof(MainWindow), new PropertyMetadata(false));

        public bool IsHelpPanelSelected
        {
            get { return (bool)GetValue(IsHelpPanelSelectedProperty); }
            set { SetValue(IsHelpPanelSelectedProperty, value); }
        }

        // Using a DependencyProperty as the backing store for IsHelpPanelSelected.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty IsHelpPanelSelectedProperty =
            DependencyProperty.Register("IsHelpPanelSelected", typeof(bool), typeof(MainWindow), new PropertyMetadata(false));

        public bool IsViewersPanelSelected
        {
            get { return (bool)GetValue(IsViewersPanelSelectedProperty); }
            set { SetValue(IsViewersPanelSelectedProperty, value); }
        }

        // Using a DependencyProperty as the backing store for IsViewersPanelSelected.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty IsViewersPanelSelectedProperty =
            DependencyProperty.Register("IsViewersPanelSelected", typeof(bool), typeof(MainWindow), new PropertyMetadata(false));

        public bool IsCameraPanelSelected
        {
            get { return (bool)GetValue(IsCameraPanelSelectedProperty); }
            set { SetValue(IsCameraPanelSelectedProperty, value); }
        }

        // Using a DependencyProperty as the backing store for IsCameraPanelSelected.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty IsCameraPanelSelectedProperty =
            DependencyProperty.Register("IsCameraPanelSelected", typeof(bool), typeof(MainWindow), new PropertyMetadata(false));

        public bool IsFilePanelSelected
        {
            get { return (bool)GetValue(IsFilePanelSelectedProperty); }
            set { SetValue(IsFilePanelSelectedProperty, value); }
        }

        // Using a DependencyProperty as the backing store for IsFilePanelSelected.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty IsFilePanelSelectedProperty =
            DependencyProperty.Register("IsFilePanelSelected", typeof(bool), typeof(MainWindow), new PropertyMetadata(false));

        public bool IsFilterSelected
        {
            get { return (bool)GetValue(IsFilterSelectedProperty); }
            set { SetValue(IsFilterSelectedProperty, value); }
        }

        // Using a DependencyProperty as the backing store for IsFilterSelected.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty IsFilterSelectedProperty =
            DependencyProperty.Register("IsFilterSelected", typeof(bool), typeof(MainWindow), new PropertyMetadata(false));


        public GridLength ContentGridInspectorSize
        {
            get { return (GridLength)GetValue(ContentGridInspectorSizeProperty); }
            set { SetValue(ContentGridInspectorSizeProperty, value); }
        }

        // Using a DependencyProperty as the backing store for ContentGridInspectorSize.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty ContentGridInspectorSizeProperty =
            DependencyProperty.Register("ContentGridInspectorSize", typeof(GridLength), typeof(MainWindow), new PropertyMetadata(new GridLength(0)));


        public GridLength ContentGridLogViewerSize
        {
            get { return (GridLength)GetValue(ContentGridLogViewerSizeProperty); }
            set { SetValue(ContentGridLogViewerSizeProperty, value); }
        }

        // Using a DependencyProperty as the backing store for ContentGridLogViewerSize.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty ContentGridLogViewerSizeProperty =
            DependencyProperty.Register("ContentGridLogViewerSize", typeof(GridLength), typeof(MainWindow), new PropertyMetadata(new GridLength(0)));


        public GridLength ContentGridDiagnosticsSize
        {
            get { return (GridLength)GetValue(ContentGridDiagnosticsSizeProperty); }
            set { SetValue(ContentGridDiagnosticsSizeProperty, value); }
        }

        // Using a DependencyProperty as the backing store for ContentGridDiagnosticsSize.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty ContentGridDiagnosticsSizeProperty =
            DependencyProperty.Register("ContentGridDiagnosticsSize", typeof(GridLength), typeof(MainWindow), new PropertyMetadata(new GridLength(0)));


        public GridLength ViewportGridWidthSize
        {
            get { return (GridLength)GetValue(ViewportGridWidthSizeProperty); }
            set { SetValue(ViewportGridWidthSizeProperty, value); }
        }

        // Using a DependencyProperty as the backing store for ViewportGridWidthSize.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty ViewportGridWidthSizeProperty =
            DependencyProperty.Register("ViewportGridWidthSize", typeof(GridLength), typeof(MainWindow), new PropertyMetadata(new GridLength(0)));


        public bool IsNavPanelPinned
        {
            get { return (bool)GetValue(IsNavPanelPinnedProperty); }
            set { SetValue(IsNavPanelPinnedProperty, value); }
        }

        // Using a DependencyProperty as the backing store for IsNavPanelPinned.  This enables animation, styling, binding, etc...
        public static readonly DependencyProperty IsNavPanelPinnedProperty =
            DependencyProperty.Register("IsNavPanelPinned", typeof(bool), typeof(MainWindow), new PropertyMetadata(false));

        #endregion

        public void ResetCurrentlySelectedPanel()
        {
            IsConnectPanelSelected = false;
            IsCameraPanelSelected = false;
            IsViewersPanelSelected = false;
            IsHelpPanelSelected = false;
            IsSettingsPanelSelected = false;
            IsFilePanelSelected = false;
            IsKeyboardPanelSelected = false;
        }

        private void Window_SizeChanged(object sender, SizeChangedEventArgs e)
        {
            VdbSettings.SetValue(nameof(Properties.Settings.VdbUi_WindowWidth), e.NewSize.Width);
            VdbSettings.SetValue(nameof(Properties.Settings.VdbUi_WindowHeight), e.NewSize.Height);
            VdbSettings.SetValue(nameof(Properties.Settings.VdbUi_WindowMaximized), this.WindowState);
        }

        private void Window_LocationChanged(object sender, EventArgs e)
        {
            VdbSettings.SetValue(nameof(Properties.Settings.VdbUi_WindowTop), this.Top);
            VdbSettings.SetValue(nameof(Properties.Settings.VdbUi_WindowLeft), this.Left);
        }

        private void Root_Drop(object sender, DragEventArgs e)
        {
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
            {
                string[] files = (string[])e.Data.GetData(DataFormats.FileDrop);

                if (ViewModel.PerformDebugActionsEnabled)
                {
                    ViewModel.QueueAutoStartDebugRecording(files[0] + ".txt");
                }

                ViewModel.DoConnectToFileName(files[0], Keyboard.IsKeyDown(Key.LeftCtrl));
            }
        }

        private void VdbObjectInspectionWidget_IsVisibleChanged(object sender, DependencyPropertyChangedEventArgs e)
        {
            if ( (bool)e.NewValue )
            {
                ContentGridInspectorSize = ContentGrid_InspectorSizeStored;
                ViewportGridWidthSize = ContentGrid_ViewportWidthSizeStored;
            }
            else
            {
                ContentGrid_InspectorSizeStored = InspectorColumnDef.Width;
                ContentGrid_ViewportWidthSizeStored = ViewportColumnDefinition.Width;

                ContentGridInspectorSize = new GridLength(0);
                ViewportGridWidthSize = new GridLength(0);
            }

            // Save setting
            SaveContentGridLayout();
        }

        private void LogGrid_IsVisibleChanged(object sender, DependencyPropertyChangedEventArgs e)
        {
            ViewModel.UnreadLogCount = (ViewModel.RemoteLogCollection.Count + ViewModel.LocalLogCollection.Count);

            // Update visual tree - hack to prevent framerate issues when log isn't visible
            {
                bool isVisible = (bool)e.NewValue;
                if (!isVisible)
                {
                    LogGrid.Children.Remove(LogWidget);
                }
                else if (!LogGrid.Children.Contains(LogWidget))
                {
                    LogGrid.Children.Add(LogWidget);
                }
            }

            System.Windows.Controls.RowDefinition logSplitterRow = FindRowDefinitionByName(RightContentGrid, "LogSplitterRow");
            System.Windows.Controls.RowDefinition logPanelRow = FindRowDefinitionByName(RightContentGrid, "LogPanelRow");
            if ((bool)e.NewValue)
            {
                // Create new row definitions
                {
                    System.Windows.Controls.RowDefinition splitterRowDef = new System.Windows.Controls.RowDefinition();
                    splitterRowDef.Height = GridLength.Auto;
                    splitterRowDef.Name = "LogSplitterRow";

                    System.Windows.Controls.RowDefinition panelRowDef = new System.Windows.Controls.RowDefinition();
                    {
                        panelRowDef.Name = "LogPanelRow";

                        // setup binding to height
                        Binding panelHeightBinding = new Binding();
                        panelHeightBinding.ElementName = "Root";
                        panelHeightBinding.Path = new PropertyPath("ContentGridLogViewerSize");
                        panelHeightBinding.Mode = BindingMode.TwoWay;
                        BindingOperations.SetBinding(panelRowDef, System.Windows.Controls.RowDefinition.HeightProperty, panelHeightBinding);
                    }

                    RightContentGrid.RowDefinitions.Add(splitterRowDef);
                    RightContentGrid.RowDefinitions.Add(panelRowDef);
                }

                ContentGridLogViewerSize = ContentGrid_LogViewerSizeStored;
            }
            else
            {
                ContentGrid_LogViewerSizeStored = logPanelRow.Height;
                ContentGridLogViewerSize = new GridLength(0);

                RightContentGrid.RowDefinitions.Remove(logSplitterRow);
                RightContentGrid.RowDefinitions.Remove(logPanelRow);
            }

            SetDiagLogRows();
            SaveContentGridLayout();
        }


        private void DiagnosticsGrid_IsVisibleChanged(object sender, DependencyPropertyChangedEventArgs e)
        {
            System.Windows.Controls.RowDefinition diagnosticsSplitterRow = FindRowDefinitionByName(RightContentGrid, "DiagnosticsSplitterRow");
            System.Windows.Controls.RowDefinition diagnosticsPanelRow = FindRowDefinitionByName(RightContentGrid, "DiagnosticsPanelRow");
            if ((bool)e.NewValue)
            {
                if (diagnosticsSplitterRow != null)
                {
                    RightContentGrid.RowDefinitions.Remove(diagnosticsSplitterRow);
                }

                if (diagnosticsPanelRow != null)
                {
                    RightContentGrid.RowDefinitions.Remove(diagnosticsPanelRow);
                }

                // Create new row definitions
                {
                    System.Windows.Controls.RowDefinition splitterRowDef = new System.Windows.Controls.RowDefinition();
                    splitterRowDef.Height = GridLength.Auto;
                    splitterRowDef.Name = "DiagnosticsSplitterRow";

                    System.Windows.Controls.RowDefinition panelRowDef = new System.Windows.Controls.RowDefinition();
                    {
                        panelRowDef.Name = "DiagnosticsPanelRow";

                        // setup binding to height
                        Binding panelHeightBinding = new Binding();
                        panelHeightBinding.ElementName = "Root";
                        panelHeightBinding.Path = new PropertyPath("ContentGridDiagnosticsSize");
                        panelHeightBinding.Mode = BindingMode.TwoWay;
                        BindingOperations.SetBinding(panelRowDef, System.Windows.Controls.RowDefinition.HeightProperty, panelHeightBinding);
                    }

                    RightContentGrid.RowDefinitions.Add(splitterRowDef);
                    RightContentGrid.RowDefinitions.Add(panelRowDef);
                }

                ContentGridDiagnosticsSize = ContentGrid_DiagnosticsSizeStored;
            }
            else
            {
                ContentGrid_DiagnosticsSizeStored = diagnosticsPanelRow.Height;
                ContentGridDiagnosticsSize = new GridLength(0);

                RightContentGrid.RowDefinitions.Remove(diagnosticsSplitterRow);
                RightContentGrid.RowDefinitions.Remove(diagnosticsPanelRow);
            }

            SetDiagLogRows();
            SaveContentGridLayout();
        }

        private void SetDiagLogRows()
        {
            int logIndex = 1;
            System.Windows.Controls.Grid.SetRow(LogSplitter, logIndex);
            System.Windows.Controls.Grid.SetRow(LogGrid, logIndex+1);

            int diagIndex = (LogGrid.Visibility == Visibility.Visible) ? 3 : 1;
            System.Windows.Controls.Grid.SetRow(DiagnosticsSplitter, diagIndex);
            System.Windows.Controls.Grid.SetRow(DiagnosticsGrid, diagIndex + 1);
        }

        private void SaveContentGridLayout()
        {
            double totalHeight = ViewportRowDefinition.Height.Value;
            totalHeight += ContentGridLogViewerSize.Value;
            totalHeight += ContentGridDiagnosticsSize.Value;

            if (ContentGridLogViewerSize.Value > 1)
            {
                double logViewerRatio = ContentGridLogViewerSize.Value / totalHeight;
                ContentGrid_LogViewerSizeStored = new GridLength(logViewerRatio, GridUnitType.Star);
                ContentGridLogViewerSize = ContentGrid_LogViewerSizeStored;
            }
            else if(ContentGrid_LogViewerSizeStored.Value > 1)
            {
                double logViewerRatio = ContentGrid_LogViewerSizeStored.Value / totalHeight;
                ContentGrid_LogViewerSizeStored = new GridLength(logViewerRatio, GridUnitType.Star);
            }

            if (ContentGridDiagnosticsSize.Value > 1)
            {
                double diagnosticsRatio = ContentGridDiagnosticsSize.Value / totalHeight;
                ContentGrid_DiagnosticsSizeStored = new GridLength(diagnosticsRatio, GridUnitType.Star);
                ContentGridDiagnosticsSize = ContentGrid_DiagnosticsSizeStored;
            }
            else if (ContentGrid_DiagnosticsSizeStored.Value > 1)
            {
                double diagnosticsRatio = ContentGrid_DiagnosticsSizeStored.Value / totalHeight;
                ContentGrid_DiagnosticsSizeStored = new GridLength(diagnosticsRatio, GridUnitType.Star);
            }

            double viewportRatio = ViewportRowDefinition.Height.Value / totalHeight;
            ViewportRowDefinition.Height = new GridLength(viewportRatio, GridUnitType.Star);

            VdbSettings.SetValue(nameof(Properties.Settings.VdbUi_LogViewerSize), ContentGrid_LogViewerSizeStored);
            VdbSettings.SetValue(nameof(Properties.Settings.VdbUi_DiagnosticsSize), ContentGrid_DiagnosticsSizeStored);
            VdbSettings.SetValue(nameof(Properties.Settings.VdbUi_ViewportHeightSize), ViewportRowDefinition.Height);
            VdbSettings.SetValue(nameof(Properties.Settings.VdbUi_InspectionSize), ContentGrid_InspectorSizeStored);
            VdbSettings.SetValue(nameof(Properties.Settings.VdbUi_ViewportWidthSize), ViewportColumnDefinition.Width);
        }

        private System.Windows.Controls.RowDefinition FindRowDefinitionByName(System.Windows.Controls.Grid grid, string name)
        {
            foreach (System.Windows.Controls.RowDefinition rowDef in grid.RowDefinitions)
            {
                if (rowDef.Name == name)
                {
                    return rowDef;
                }
            }
            return null;
        }

        private void DockablePanelNav_Click(object sender, RoutedEventArgs e)
        {
            if (!IsNavPanelPinned)
            {
                ResetCurrentlySelectedPanel();
            }
        }

        private void ClosePanelButton_Click(object sender, RoutedEventArgs e)
        {
            ResetCurrentlySelectedPanel();
            IsNavPanelPinned = false;
        }
    }

    public class GridLengthAnimation : AnimationTimeline
    {
        public override Type TargetPropertyType
        {
            get
            {
                return typeof(GridLength);
            }
        }

        protected override Freezable CreateInstanceCore()
        {
            return new GridLengthAnimation();
        }

        static GridLengthAnimation()
        {
            FromProperty = DependencyProperty.Register("From", typeof(GridLength), typeof(GridLengthAnimation));
            ToProperty = DependencyProperty.Register("To", typeof(GridLength), typeof(GridLengthAnimation));
        }
        public static readonly DependencyProperty FromProperty;
        public GridLength From
        {
            get
            {
                return (GridLength)GetValue(FromProperty);
            }
            set
            {
                SetValue(FromProperty, value);
            }
        }
        public static readonly DependencyProperty ToProperty;
        public GridLength To
        {
            get
            {
                return (GridLength)GetValue(ToProperty);
            }
            set
            {
                SetValue(ToProperty, value);
            }
        }

        public override object GetCurrentValue(object defaultOriginValue, object defaultDestinationValue, AnimationClock animationClock)
        {
            double fromVal = ((GridLength)GetValue(FromProperty)).Value;
            double toVal = ((GridLength)GetValue(ToProperty)).Value;

            
            if (fromVal > toVal)
            {
                return new GridLength((1 - animationClock.CurrentProgress.Value) * (fromVal - toVal) + toVal, GridUnitType.Pixel);
            }
            else
            {
                return new GridLength(animationClock.CurrentProgress.Value * (toVal - fromVal) + fromVal, GridUnitType.Pixel);
            }
        }
    }

    public class VdbBrowserRedirectHyperlink : Hyperlink
    {
        public VdbBrowserRedirectHyperlink()
        {
            RequestNavigate += OnRequestNavigate;
        }

        private void OnRequestNavigate(object sender, RequestNavigateEventArgs e)
        {
            try
            {
                System.Diagnostics.Process.Start(new ProcessStartInfo(e.Uri.AbsoluteUri));
                e.Handled = true;
            }
            catch { }
        }
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
