// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM		: WIN32 X64
// PRODUCT		: COMMON
// VISIBILITY	: CLIENT
//
// ------------------------------------------------------TKBMS v1.0



// This will enable some stats on the disposed object registry to show up when pressing the debug key.
// The wrapper only has this enabled in DEBUG.
//#define SHOW_OBJECT_INSPECTION_DISPOSED_OBJECT_REGISTRY_STATS
// This will enable some stats on object inspection to show up when pressing the debug key (can be very slow).
//#define SHOW_OBJECT_INSPECTION_STATS

using Microsoft.Win32;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.ComponentModel;
using System.IO;
using System.Reflection;
using System.Windows;
using System.Windows.Data;
using System.Windows.Media;
using System.Linq;
using static HavokVisualDebugger.VdbModel;

namespace HavokVisualDebugger
{
    public class VdbViewModel : ViewModelNotifyPropertyChanged
    {
        public event Action<VdbViewModel> AutoRefreshPaused;
        public event Action<VdbViewModel> AutoRefreshUnPaused;
        public event Action<VdbViewModel> AutoRefreshTick;
        public event Action<VdbViewModel> FramesChanging;

        public VdbClient Client { get; private set; }
        public VdbSelection Selection { get; private set; }
        internal void UnselectAll(object o)
        {
            Selection.ClearSelected();
        }

        #region VdbViewModel Initialization

        public VdbViewModel()
        {
            AllowAutoPrompts = true;
            Client = new VdbClient();
            RegisteredServerActionProcessesCollection = new VdbProcessCollection(this);
            DefaultProcessesCollection = new VdbProcessCollection(this);
            CategoryProcessCollection = new hkNonResettableObservableCollection<VdbProcessesViewModel>();
            RegisteredCamerasCollection = new ObservableCollection<VdbCameraViewModel>();
            RegisteredWidgetsCollection = new ObservableCollection<VdbWidgetViewModel>();
            ObjectPropertyOptions = new VdbObjectPropertiesOptions(this);
            Selection = new VdbSelection(this);

            TimerViewModelDataFlat = new ObservableCollection<TimerNodeViewModel>();
            TimerViewModelView = new ObservableCollection<TimerNodeViewModel>();
            TimerNodeManager = new VdbTimerNodeManager(this, TimerViewModelView, TimerViewModelDataFlat, TimerPathToViewModel);

            VdbCommands = new VdbViewModelCommands(this);

            InitializeListeners();
            InitializeObjectColors();
            InitializeViewerInfo();
            InitializeLogViewers();
            InitializeObjectInspection();
            InitializeConnectionInfo();
            InitializeAppSettings();

            // Post InitializeObjectInspection, so we have tree to listen to
            VdbObjectFilterBaseLayer = new VdbFixedFilterLayerViewModel(Selection);
            VdbObjectFilterLayers = new VdbMutableFilterLayerCollection(this, Client.RenderSurface, VdbObjectFilterBaseLayer, CategoryProcessCollection);

            DebugViewportDisplayObjects = new ObservableCollection<DebugDisplayObjects>();
            DebugViewportDisplayObjects.Add(new DebugDisplayObjects(() => "")); // Spacer to get passed wpf UI toolkit handle
            DebugViewportDisplayObjects.Add(new DebugDisplayObjects(() => String.Format("Update: {0:0.####}", (Client != null) ? Client.RenderSurface.UpdateFrameMs : 0.0)));
            DebugViewportDisplayObjects.Add(new DebugDisplayObjects(() => String.Format("Draw: {0:0.####}", (Client != null) ? Client.RenderSurface.DrawFrameMs : 0.0)));
#if SHOW_OBJECT_INSPECTION_STATS
            DebugViewportDisplayObjects.Add(new DebugDisplayObjects(() => String.Format("ObjectCount: {0}", (VdbObjectTree != null) ? VdbObjectTree.Count : 0)));
            DebugViewportDisplayObjects.Add(new DebugDisplayObjects(() => String.Format("ObjectVmCount: {0}", (VdbObjectViewModels != null) ? VdbObjectViewModels.Count : 0)));
            DebugViewportDisplayObjects.Add(new DebugDisplayObjects(() =>
            {
                int total = 0;
                int max = 0;
                int count = 0;
                if (VdbObjectTree != null)
                {
                    foreach (VdbObjectMultiParentTreeItem item in VdbObjectTree.Values)
                    {
                        int parents = item.PathsToRoot;
                        total += parents;
                        max = Math.Max(max, parents);
                    }
                    count = VdbObjectTree.Values.Count;
                }
                return String.Format("PathsToRoot Total{0}: Avg:{1} Max:{2}", total, count > 0 ? total / count : 0, max);
            }));
#endif
#if SHOW_OBJECT_INSPECTION_DISPOSED_OBJECT_REGISTRY_STATS
            DebugViewportDisplayObjects.Add(new DebugDisplayObjects(() => String.Format("DisposedObjectEntryCount: {0}", (Client != null) ? Client.GetSubInterop(nameof(Havok.Vdb.Client.ObjectHandler)).GetProperty<int>(nameof(Havok.Vdb.ObjectHandler.DisposedObjectEntryCount)) : 0)));
            DebugViewportDisplayObjects.Add(new DebugDisplayObjects(() => String.Format("DisposedObjectCount: {0}", (Client != null) ? Client.GetSubInterop(nameof(Havok.Vdb.Client.ObjectHandler)).GetProperty<int>(nameof(Havok.Vdb.ObjectHandler.DisposedObjectCount)) : 0)));
#endif

            CompositionTarget.Rendering += delegate
            {
                UpdateDebugViewportDisplayObjs();
            };

            AutoRefreshTick += delegate
            {
                using (Client.AcquireLock())
                {
                    Havok.Vdb.PerfStatsReceivedEventArgs perfStats = Client.GetSubInterop(nameof(Havok.Vdb.Client.StatsHandler)).
                        GetProperty<Havok.Vdb.PerfStatsReceivedEventArgs>(nameof(Havok.Vdb.StatsHandler.PerfStats));
                    UpdateTimers(perfStats);
                }
            };

            FramesChanging += delegate
            {
                
                // If we are going to be changing frame then clear out the search (see VDB-1018).
                VdbObjectQuery = null;
                Selection.ClearHighlightedAndSelected();
            };

            RenderWpfUi = true;
            PauseOnScrub = true;

            DiscoveredServers = new ObservableCollection<VdbServerInfoViewModel>();
            Client.DiscoveredServers.CollectionChanged += delegate (object sender, NotifyCollectionChangedEventArgs e)
            {
                if( e.Action == NotifyCollectionChangedAction.Reset )
                {
                    DiscoveredServers.Clear();
                    return;
                }

                if( e.NewItems != null )
                {
                    foreach(Havok.Vdb.ServerInfo serverInfo in e.NewItems)
                    {
                        DiscoveredServers.Add(new VdbServerInfoViewModel(serverInfo, this));
                    }
                }

                if( e.OldItems != null)
                {
                    foreach (Havok.Vdb.ServerInfo serverInfo in e.OldItems)
                    {
                        int serverToRemove = -1;
                        for(int i = 0; i < DiscoveredServers.Count; i++ )
                        {
                            VdbServerInfoViewModel discoveredInfo = DiscoveredServers[i];
                            if (discoveredInfo.ServerInfo == serverInfo )
                            {
                                serverToRemove = i;
                                break;
                            }
                        }

                        if( serverToRemove >= 0)
                        {
                            DiscoveredServers.RemoveAt(serverToRemove);
                        }
                    }
                }
            };

            DiscoveredServersViewSource = new CollectionViewSource() { Source = DiscoveredServers };
            DiscoveredServersView.Filter = IsDiscoveredServerIncluded;

            // Keep current machine (localhost,etc) servers at the top
            DiscoveredServersView.SortDescriptions.Add(new SortDescription(nameof(VdbServerInfoViewModel.IsLocalHost), ListSortDirection.Descending));
        }

        public void InitializeResourceProperties()
        {
            IconFont = (Application.Current.FindResource("IconFont") as FontFamily);
            TextFont = (Application.Current.FindResource("TextFont") as FontFamily);
            MonospaceFont = (Application.Current.FindResource("MonospaceFont") as FontFamily);
        }

        private void Cleanup()
        {
            // Don't let the UI hold onto cleaned-up OI data.
            InitializeObjectInspection();

            
            RegisteredServerActionProcessesCollection.Clear();
            foreach(VdbProcessesViewModel categoryProc in CategoryProcessCollection)
            {
                categoryProc.Processes.Dispose();
            }
            CategoryProcessCollection.Clear();

            DefaultProcessesCollection.Clear();
            RegisteredCamerasCollection.Clear();
            RemoteLogCollection.Clear();
            RemoteLogCollectionView.Clear();
            Selection.ClearHighlightedAndSelected();

            TimerNodeManager.ClearTimers();
        }

        private void InitializeConnectionInfo()
        {
            System.Diagnostics.Debug.Assert(MachineAddresses == null);
            System.Diagnostics.Debug.Assert(MachinePorts == null);
            System.Diagnostics.Debug.Assert(MachineDiscoveryPorts == null);
            System.Diagnostics.Debug.Assert(RecentFiles == null);

            // Initialize addresses and ports
            MachineAddresses = new ObservableCollection<string>();
            hkCollectionUtils.InitializeLimitedStringCollectionFromSettings(MachineAddresses, nameof(Properties.Settings.VdbConnection_MachineAddresses), MaxMachineAddressEntries);
            MachineAddresses.CollectionChanged += MachineAddresses_CollectionChanged;

            MachinePorts = new ObservableCollection<string>();
            hkCollectionUtils.InitializeLimitedStringCollectionFromSettings(MachinePorts, nameof(Properties.Settings.VdbConnection_MachinePorts), MaxMachinePortEntries);
            MachinePorts.CollectionChanged += MachinePorts_CollectionChanged;

            MachineDiscoveryPorts = new ObservableCollection<string>();
            hkCollectionUtils.InitializeLimitedStringCollectionFromSettings(MachineDiscoveryPorts, nameof(Properties.Settings.VdbConnection_MachineDiscoveryPorts), MaxMachinePortEntries);
            MachineDiscoveryPorts.CollectionChanged += MachinePorts_CollectionChanged;

            RecentFiles = new ObservableCollection<string>();
            hkCollectionUtils.InitializeLimitedStringCollectionFromSettings(RecentFiles, nameof(Properties.Settings.VdbSettings_RecentFiles), MaxRecentFiles);
            RecentFiles.CollectionChanged += RecentFiles_CollectionChanged;
        }

        
        private void InitializeListeners()
        {
            // These will have been filled out on ctor of the Client
            NotifyClientPropertiesChanged();

            hkTelemetry.Provider.SessionStarted += Provider_SessionStarted;
            hkTelemetry.Provider.SessionEnded += Provider_SessionEnded;

            // Listen to events
            Client.ListenTo<Havok.Vdb.ConnectionHandler, Havok.Vdb.Client>(
                nameof(Havok.Vdb.Client.Connecting),
                delegate (Havok.Vdb.Client sender)
                {
                    m_completedFirstStep = false;
                    NotifyConnectionPropertiesChanged();
                    PerformAutoStartRecording();
                    PerformAutoStartDebugRecording();
                });
            Client.ListenTo<Havok.Vdb.ConnectionHandler, Havok.Vdb.Client>(
                nameof(Havok.Vdb.Client.Connected),
                delegate (Havok.Vdb.Client sender)
                {
                    Cleanup();
                    NotifyConnectionPropertiesChanged();
                    // Our min/max/current frame, etc. will have changed on fresh connection
                    NotifyPlaybackPropertiesChanged();
                    // It's unclear if this should belong here or in the underlying framework.
                    // I could see use cases where you want to ensure that you pause *immediately* on connect
                    // and therefore the playback state/direction, etc. aren't reset by a new connection.
                    // But for now, it's a bit weird not to...so make sure we are playing forward.
                    Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).
                        SetPropertyAsync(nameof(Havok.Vdb.PlaybackHandler.PlaybackDirection),
                        Havok.Vdb.PlaybackDirection.Forward);
                    Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).
                        SetPropertyAsync(nameof(Havok.Vdb.PlaybackHandler.PlaybackState),
                        Havok.Vdb.PlaybackState.Playing);
                    // Below is a behavior choice, if we uncomment then we will never try to reconnect
                    // after establishing the first connection. This can be preferred if you don't want
                    // to lose circular buffer to a reconnection.
                    // Client.AttemptAutoReconnect = null;
                    PerformAutoSync();

                    // Begin a telemetry session
                    hkTelemetry.Provider.OnSessionStarted();
                });
            Client.ListenTo<Havok.Vdb.ConnectionHandler, Havok.Vdb.Client>(
                nameof(Havok.Vdb.Client.Disconnecting),
                delegate (Havok.Vdb.Client sender)
                {
                    NotifyConnectionPropertiesChanged();
                    // Unfortunately, atm, we'll get a disconnect for the NoOpConnection we have
                    // ResetAutos();
                });
            Client.ListenTo<Havok.Vdb.ConnectionHandler, Havok.Vdb.Client>(
                nameof(Havok.Vdb.Client.Disconnected),
                delegate (Havok.Vdb.Client sender)
                {
                    
                    NotifyConnectionPropertiesChanged();

                    // End a telemetry session
                    if (SessionStartTime != null)
                    {
                        // Don't send telemetry info if user disabled telemetry gathering
                        // Keep gathering though, in case they turn it back on.
                        if (IsAllowingTelemetry)
                        {
                            hkTelemetry.Provider.OnSessionEnded();
                        }

                        SessionStartTime = null;
                    }
                });
            Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).
                ListenTo<Havok.Vdb.PlaybackCmdReceivedHandler, Havok.Vdb.PlaybackHandler, Havok.Vdb.PlaybackCmdReceivedArgs>(
                    nameof(Havok.Vdb.PlaybackHandler.PlaybackCmdReceived),
                    delegate (Havok.Vdb.PlaybackHandler sender, Havok.Vdb.PlaybackCmdReceivedArgs args)
                    {
                        NotifyPlaybackPropertiesChanged(args);

                        
                        
                        
                        
                        
                        
                        if (!m_completedFirstStep && (args.PreviousFrame < args.TargetFrame))
                        {
                            VdbSettings.ApplyToSession(this);
                            m_completedFirstStep = true;
                        }

                        // Paused during playback
                        if ((args.TargetState == Havok.Vdb.PlaybackState.Paused) && !IsAutoRefreshPaused)
                        {
                            AutoRefreshTick?.Invoke(this);
                        }
                    });
            VdbClientInterop processHandlerInterop = Client.GetSubInterop(nameof(Havok.Vdb.Client.ProcessHandler));
            processHandlerInterop.
                ListenTo<Havok.Vdb.ProcessRegisteredEventHandler, Havok.Vdb.ProcessHandler, Havok.Vdb.ProcessRegisteredEventArgs>(
                    nameof(Havok.Vdb.ProcessHandler.ProcessRegistered),
                    delegate (Havok.Vdb.ProcessHandler sender, Havok.Vdb.ProcessRegisteredEventArgs args)
                    {
                        VdbProcessViewModel processVm = new VdbProcessViewModel(processHandlerInterop, args.Tag);
                        if (processVm.IsFileCommandViewer)
                        {
                            RegisteredServerActionProcessesCollection.Add(processVm);
                        }
                        else
                        {
                            VdbProcessesViewModel categoryProcesses = CategoryProcessCollection.FirstOrDefault(x => x.Category == processVm.Category);
                            if (categoryProcesses == null)
                            {
                                categoryProcesses = new VdbProcessesViewModel(processVm.Category, this);
                                CategoryProcessCollection.Add(categoryProcesses);
                            }

                            categoryProcesses.Processes.Add(processVm);
                        }
                    });
            processHandlerInterop.
                ListenTo<Havok.Vdb.ProcessSelectionChangedEventHandler, Havok.Vdb.ProcessHandler, Havok.Vdb.ProcessSelectionChangedEventArgs>(
                    nameof(Havok.Vdb.ProcessHandler.ProcessSelectionChanged),
                    delegate (Havok.Vdb.ProcessHandler sender, Havok.Vdb.ProcessSelectionChangedEventArgs args)
                    {
                        VdbProcessViewModel processVm = null;
                        foreach (VdbProcessesViewModel categoryProc in CategoryProcessCollection )
                        {
                            processVm = categoryProc.FindProcessByTag(args.Tag);
                            if( processVm != null )
                            {
                                break;
                            }
                        }

                        if (processVm != null)
                        {
                            if (!m_completedFirstStep)
                            {
                                DefaultProcessesCollection.Add(processVm);
                            }
                            VdbSettings.SetCollectionItemEnabled(Properties.Settings.Default.VdbEnabledViewers, processVm.Name, args.Selected);
                            processVm.NotifySelectionChanged();
                        }
                    });
            Client.GetSubInterop(nameof(Havok.Vdb.Client.SetupHandler)).
                ListenTo<Havok.Vdb.SetupHandlerEventHandler, Havok.Vdb.SetupHandler, Havok.Vdb.SetupHandlerEventArgs>(
                    nameof(Havok.Vdb.SetupHandler.ServerInfoReceived),
                    delegate (Havok.Vdb.SetupHandler sender, Havok.Vdb.SetupHandlerEventArgs args)
                    {
                        NotifyServerPropertiesChanged();
                    });
            Client.GetSubInterop(nameof(Havok.Vdb.Client.StatsHandler)).
                ListenTo<Havok.Vdb.MemStatsReceivedEventHandler, Havok.Vdb.StatsHandler, Havok.Vdb.MemStatsReceivedEventArgs>(
                    nameof(Havok.Vdb.StatsHandler.MemStatsReceived),
                    delegate (Havok.Vdb.StatsHandler sender, Havok.Vdb.MemStatsReceivedEventArgs args)
                    {
                        NotifyPropertyChanged(nameof(MemStats));
                    });
            Client.GetSubInterop(nameof(Havok.Vdb.Client.StatsHandler)).
                ListenTo<Havok.Vdb.FrameStatsReceivedEventHandler, Havok.Vdb.StatsHandler, Havok.Vdb.FrameStatsReceivedEventArgs>(
                    nameof(Havok.Vdb.StatsHandler.FrameStatsReceived),
                    delegate (Havok.Vdb.StatsHandler sender, Havok.Vdb.FrameStatsReceivedEventArgs args)
                    {
                        // find jitter rate
                        if( SessionLastFrameStats != null )
                        {
                            // see if the current frame elapsed seconds is more than 3 std deviations away from the last frame mean
                            double mean = SessionLastFrameStats.Mean;
                            double stdDev = SessionLastFrameStats.StandardDeviation;
                            double threeStdDevs = (mean + (3 * stdDev));
                            if (args.LastStepElapsedSeconds > threeStdDevs)
                            {
                                if (SessionJitterFramePrevious > 0)
                                {
                                    // keep rolling average
                                    SessionJitterFrameDifferenceSum += (CurrentFrame - SessionJitterFramePrevious);
                                    SessionJitterJitterSum += args.LastStepElapsedSeconds - threeStdDevs;
                                }
                                SessionJitterFramePrevious = CurrentFrame;
                                SessionJitterDifferenceCount++;
                            }
                        }

                        SessionLastFrameStats = args;
                        NotifyPropertyChanged(nameof(FrameStats));
                    });
            Client.GetSubInterop(nameof(Havok.Vdb.Client.StatsHandler)).
              ListenTo<Havok.Vdb.PerfStatsReceivedEventHandler, Havok.Vdb.StatsHandler, Havok.Vdb.PerfStatsReceivedEventArgs>(
                  nameof(Havok.Vdb.StatsHandler.PerfStatsReceived),
                  delegate (Havok.Vdb.StatsHandler sender, Havok.Vdb.PerfStatsReceivedEventArgs args)
                  {
                      // Early out
                      if (!IsDiagnosticsShown || (LineGraphWidget == null) || (!LineGraphWidget.IsSelected && !IsStatsTextSelected))
                      {
                          return;
                      }

                      UpdateTimers(args);
                  });
            Client.GetSubInterop(nameof(Havok.Vdb.Client.DisplayHandler)).
                ListenTo<Havok.Vdb.DisplayOptionsSetEventHandler, Havok.Vdb.DisplayHandler, Havok.Vdb.DisplayOptionsSetEventArgs>(
                    nameof(Havok.Vdb.DisplayHandler.DisplayOptionsSet),
                    delegate (Havok.Vdb.DisplayHandler sender, Havok.Vdb.DisplayOptionsSetEventArgs args)
                    {
                        NotifyPropertyChanged(nameof(DisplayWorldUp));
                    });
            Client.GetSubInterop(nameof(Havok.Vdb.Client.DisplayHandler)).
                ListenTo<Havok.Vdb.CameraEventHandler, Havok.Vdb.DisplayHandler, Havok.Vdb.CameraEventArgs>(
                    nameof(Havok.Vdb.DisplayHandler.CameraAdded),
                    delegate (Havok.Vdb.DisplayHandler sender, Havok.Vdb.CameraEventArgs args)
                    {
                        VdbCameraViewModel cameraVm = new VdbCameraViewModel(Client.RenderSurface, args.Name, this);

                        // Place cameras in the following order:
                        //  "*" prefixed,
                        //  recorded cameras,
                        //  all other cameras
                        string name = cameraVm.Name.Trim();
                        if (name.StartsWith("*") ||
                            hkConstants.RecordedCameraNames.Contains(name))
                        {
                            VdbCameraViewModel lastAsteriskCvm = RegisteredCamerasCollection.LastOrDefault(cvm => cvm.Name.Trim().StartsWith("*"));
                            int idx = (lastAsteriskCvm != null) ? (RegisteredCamerasCollection.IndexOf(lastAsteriskCvm) + 1) : 0;
                            if (hkConstants.RecordedCameraNames.Contains(name))
                            {
                                VdbCameraViewModel lastRecordedCvm = RegisteredCamerasCollection.LastOrDefault(cvm => hkConstants.RecordedCameraNames.Contains(cvm.Name.Trim()));
                                idx = (lastRecordedCvm != null) ? (RegisteredCamerasCollection.IndexOf(lastRecordedCvm) + 1) : idx;
                            }
                            RegisteredCamerasCollection.Insert(idx, cameraVm);
                        }
                        else
                        {
                            RegisteredCamerasCollection.Add(cameraVm);
                        }

                        // Determine if it should be selected.
                        if (CameraConnectModeCameraName && (VdbSettings.GetValue<string>(nameof(Properties.Settings.Default.VdbCamera_SelectedCamera)) == args.Name))
                        {
                            cameraVm.IsSelected = true;
                        }
                        else if (CameraConnectModeDefaultCamera && (RegisteredCamerasCollection.IndexOf(cameraVm) == 0))
                        {
                            cameraVm.IsSelected = true;
                        }
                        else
                        {
                            cameraVm.IsSelected = false;
                        }
                    });
            Client.GetSubInterop(nameof(Havok.Vdb.Client.DisplayHandler)).
                ListenTo<Havok.Vdb.CameraEventHandler, Havok.Vdb.DisplayHandler, Havok.Vdb.CameraEventArgs>(
                    nameof(Havok.Vdb.DisplayHandler.CameraRemoved),
                    delegate (Havok.Vdb.DisplayHandler sender, Havok.Vdb.CameraEventArgs args)
                    {
                        VdbCameraViewModel foundDisplay = null;
                        foreach (VdbCameraViewModel display in RegisteredCamerasCollection)
                        {
                            if (display.Name == args.Name)
                            {
                                foundDisplay = display;
                                break;
                            }
                        }

                        if (foundDisplay != null)
                        {
                            RegisteredCamerasCollection.Remove(foundDisplay);
                        }
                    });
            Client.GetSubInterop(nameof(Havok.Vdb.Client.FileHandler)).
                ListenTo<Havok.Vdb.FileCmdReceivedHandler, Havok.Vdb.FileHandler, Havok.Vdb.FileCmdReceivedArgs>(
                    nameof(Havok.Vdb.FileHandler.FileCmdReceived),
                    delegate (Havok.Vdb.FileHandler sender, Havok.Vdb.FileCmdReceivedArgs args)
                    {
                        string fileName = DoFileSave(
                            string.Format(HavokVisualDebugger.Properties.Resources.File_SaveFile, args.TypeName),
                            FilterFromExtensions(args.TypeName, args.Extensions));

                        if (fileName != null)
                        {
                            using (var fileStream = File.Create(fileName))
                            {
                                args.MemoryStream.Seek(0, SeekOrigin.Begin);
                                args.MemoryStream.CopyTo(fileStream);
                            }
                        }
                    });
            Client.GetSubInterop(nameof(Havok.Vdb.Client.TextHandler)).
                ListenTo<Havok.Vdb.TextCmdReceivedHandler, Havok.Vdb.TextHandler, Havok.Vdb.TextCmdReceivedArgs>(
                    nameof(Havok.Vdb.TextHandler.TextCmdReceived),
                    delegate (Havok.Vdb.TextHandler sender, Havok.Vdb.TextCmdReceivedArgs args)
                    {
                        VdbLogViewModel logEntry = new VdbLogViewModel(args.Text, args.Level, CurrentBufferFrame, args.Source, CurrentFrame, args.Tag, args.Id);
                        if (args.Source == Havok.Vdb.LogSource.Server)
                        {
                            if (RemoteLogCollection.Count >= _MaxLogLimit)
                            {
                                RemoteLogCollection.RemoveAt(0);
                            }
                            RemoteLogCollection.Add(logEntry);
                        }
                        else
                        {
                            if (LocalLogCollection.Count >= _MaxLogLimit)
                            {
                                LocalLogCollection.RemoveAt(0);
                            }
                            LocalLogCollection.Add(logEntry);
                        }
                    });
            Client.GetSubInterop(nameof(Havok.Vdb.Client.ObjectHandler)).
                ListenTo<Havok.Vdb.ObjectsChangedEventHandler, Havok.Vdb.ObjectHandler, Havok.Vdb.ObjectsChangedEventArgs>(
                    nameof(Havok.Vdb.ObjectHandler.ObjectsChanged),
                    delegate (Havok.Vdb.ObjectHandler sender, Havok.Vdb.ObjectsChangedEventArgs args)
                    {
                        foreach (Havok.Vdb.ObjectsChangedEvent evt in args.ObjectsChangedEvents)
                        {
                            // Updated evt is by-far the most common evt type, so we check for this first.
                            Havok.Vdb.ObjectsUpdatedEvent updatedEvt = (evt as Havok.Vdb.ObjectsUpdatedEvent);
                            if (updatedEvt != null)
                            {
                                VdbObjectTree.UpdateObjects(updatedEvt);
                                continue;
                            }

                            // SetConnectivity is more common than other types during replay from buffer, so we check it second.
                            Havok.Vdb.ObjectSetConnectivityEvent setConnectivityEvt = (evt as Havok.Vdb.ObjectSetConnectivityEvent);
                            if (setConnectivityEvt != null)
                            {
                                VdbObjectTree.SetObjectConnectivity(setConnectivityEvt);
                                continue;
                            }

                            // Check for all other evt types
                            Havok.Vdb.ObjectsAddedEvent addedEvt = (evt as Havok.Vdb.ObjectsAddedEvent);
                            if (addedEvt != null)
                            {
                                VdbObjectTree.AddObjects(addedEvt);
                                continue;
                            }
                            Havok.Vdb.ObjectsRemovedEvent removedEvt = (evt as Havok.Vdb.ObjectsRemovedEvent);
                            if (removedEvt != null)
                            {
                                VdbObjectTree.RemoveObjects(removedEvt);
                                continue;
                            }
                            Havok.Vdb.ObjectConnectedEvent connectedEvt = (evt as Havok.Vdb.ObjectConnectedEvent);
                            if (connectedEvt != null)
                            {
                                VdbObjectTree.ConnectObject(connectedEvt);
                                continue;
                            }
                            Havok.Vdb.ObjectDisconnectedEvent disconnectedEvt = (evt as Havok.Vdb.ObjectDisconnectedEvent);
                            if (disconnectedEvt != null)
                            {
                                VdbObjectTree.DisconnectObject(disconnectedEvt);
                                continue;
                            }

                            // Unhandled!
                            System.Diagnostics.Debug.Assert(false, "Unhandled object evt type");
                        }
                    });
            Client.GetSubInterop(nameof(Havok.Vdb.Client.RenderSurface)).
                ListenTo<Havok.Vdb.HasRenderObjectsChangedEventHandler, Havok.Vdb.RenderSurface, bool>(
                    nameof(Havok.Vdb.RenderSurface.HasRenderObjectsChanged),
                    delegate (Havok.Vdb.RenderSurface sender, bool hasRenderObjects)
                    {
                        NotifyPropertyChanged(nameof(HasRenderObjects));
                    });
        }

        private void InitializeObjectColors()
        {
            _ObjectColors.Clear();
            Type t = typeof(Colors);
            foreach (PropertyInfo propInfo in t.GetProperties(BindingFlags.Public | BindingFlags.Static))
            {
                if (propInfo.PropertyType == typeof(Color))
                {
                    _ObjectColors.Add((Color)propInfo.GetValue(null));
                }
            }
        }

        public void StartProcessing(object renderSurface)
        {
            Client.ClientTicked += () =>
            {
                // Due to throttling and read-ahead, the cmd buffer can fill even if nothing else is happening.
                // Always refresh cmd buffer numbers.
                NotifyPropertyChanged(nameof(MinFrame));
                NotifyPropertyChanged(nameof(MaxFrame));
                NotifyPropertyChanged(nameof(BufferFrameCount));

                // Check for render camera changes
                NotifyIfRenderServerCameraChanged();
            };

            Client.Errored += delegate (Exception e)
            {
                System.Diagnostics.Debug.Assert(false);
                VdbLogViewModel logEntry = new VdbLogViewModel(e.Message, Havok.Vdb.LogLevel.Error, CurrentBufferFrame, Havok.Vdb.LogSource.Client, CurrentFrame, 0, 0);
                LocalLogCollection.Add(logEntry);
            };

            Client.StartProcessing(renderSurface, IsMSAAEnabled);

            // Our camera properties (bound to UI) aren't valid until the render surface is initialized.
            // Therefore, we must notify of changes here to get up-to-date initial values from the back-end.
            NotifyPropertyChanged(nameof(CameraNearPlane));
            NotifyPropertyChanged(nameof(CameraFarPlane));
            NotifyPropertyChanged(nameof(CameraFieldOfView));
            NotifyPropertyChanged(nameof(CameraPerspectiveMode));
            NotifyPropertyChanged(nameof(CameraUp));
            NotifyPropertyChanged(nameof(CameraUpOnXAxis));
            NotifyPropertyChanged(nameof(CameraUpOnYAxis));
            NotifyPropertyChanged(nameof(CameraUpOnZAxis));
            NotifyPropertyChanged(nameof(CameraUpIsNegated));
            NotifyPropertyChanged(nameof(CameraLeftHandedCoordinateSystem));
            NotifyPropertyChanged(nameof(CameraLookSensitivity));
            NotifyPropertyChanged(nameof(CameraMaximumFov));
            NotifyPropertyChanged(nameof(CameraMinimumFov));
            NotifyPropertyChanged(nameof(CameraMinimumPlane));
        }

        public void StopProcessing()
        {
            Client.StopProcessing();
        }

        public bool HasShortcut(string shortcut, out string existingShortcut)
        {
            existingShortcut = "";
            foreach (VdbShortcutViewModel vdbShortcut in VdbCommands.CommandShortcuts)
            {
                if (shortcut == vdbShortcut.Shortcut)
                {
                    existingShortcut = vdbShortcut.ShortcutName;
                    return true;
                }
            }

            return false;
        }

        public VdbViewModelCommands VdbCommands { get; private set; }
        public bool RenderWpfUi { get; set; }

        #endregion

        #region VdbViewModel Telemetry
        private DateTime? SessionStartTime;
        private int SessionNumRecordActions;
        private List<long> SessionSavedMovieSizes = new List<long>();
        private List<long> SessionSavedMovieLengths = new List<long>();
        private double SessionJitterFrameDifferenceSum = 0;
        private double SessionJitterJitterSum = 0;
        private double SessionJitterDifferenceCount = 0;
        private double SessionJitterFramePrevious = 0;
        private Havok.Vdb.FrameStatsReceivedEventArgs SessionLastFrameStats;
        private string SessionConnectionType = "";
        private string SessionMovieFileName = "";
        private long SessionMovieStartFrame = 0;
        private int SessionCameraNegativeXClicks, SessionCameraNegativeYClicks, SessionCameraNegativeZClicks;
        private int SessionCameraPositiveXClicks, SessionCameraPositiveYClicks, SessionCameraPositiveZClicks;
        private int SessionCameraDefaultClicks, SessionCameraFitCurrentClicks, SessionCameraDefaultSelectionClicks, SessionCameraFitCurrentSelectionClicks;
        private int SessionInspectionCollapseClicks, SessionNumTreeFilterOperations;
        private int SessionNumPlay, SessionNumPlayForward, SessionNumPlayBackward, SessionNumPause, SessionNumGoToBegin, SessionNumGoToEnd;
        private int SessionNumOneFrameForward, SessionNumOneFrameBackward, SessionNumSaveMovie;
        private int SessionWireframeToggles, SessionOutlineToggles, SessionRandomizeColorsToggles, SessionSelectionVisualizationToggles, SessionLockLightToCameraToggles, SessionBackfaceCullingChanges, SessionClearColorChanges;
        private void Provider_SessionStarted()
        {
            SessionStartTime = DateTime.Now;
            SessionNumRecordActions = 0;
            SessionSavedMovieSizes.Clear();
            SessionSavedMovieLengths.Clear();
            SessionConnectionType = ConnectedSource.ToString();
            SessionCameraNegativeXClicks = 0;
            SessionCameraNegativeYClicks = 0;
            SessionCameraNegativeZClicks = 0;
            SessionCameraPositiveXClicks = 0;
            SessionCameraPositiveYClicks = 0;
            SessionCameraPositiveZClicks = 0;
            SessionCameraDefaultClicks = 0;
            SessionCameraFitCurrentClicks = 0;
            SessionCameraDefaultSelectionClicks = 0;
            SessionCameraFitCurrentSelectionClicks = 0;
            SessionNumPlay = 0;
            SessionNumPlayForward = 0;
            SessionNumPlayBackward = 0;
            SessionNumPause = 0;
            SessionNumGoToBegin = 0;
            SessionNumGoToEnd = 0;
            SessionNumOneFrameForward = 0;
            SessionNumOneFrameBackward = 0;
            SessionNumSaveMovie = 0;
            SessionWireframeToggles = 0;
            SessionOutlineToggles = 0;
            SessionRandomizeColorsToggles = 0;
            SessionSelectionVisualizationToggles = 0;
            SessionLockLightToCameraToggles = 0;
            SessionBackfaceCullingChanges = 0;
            SessionClearColorChanges = 0;
            SessionLastFrameStats = null;
            SessionJitterDifferenceCount = 0;
            SessionJitterFrameDifferenceSum = 0;
            SessionJitterJitterSum = 0;
            SessionJitterFramePrevious = 0;
            SessionInspectionCollapseClicks = 0;
            if (!IsRecording) SessionMovieFileName = "";
            SessionMovieStartFrame = 0;
            SessionNumTreeFilterOperations = 0;

            foreach (VdbWidgetViewModel vm in RegisteredWidgetsCollection)
            {
                vm.ResetToggles();
            }
        }

        private void Provider_SessionEnded()
        {
            // Send general telemetry data
            if (SessionStartTime != null)
            {
                TimeSpan sessionLength = DateTime.Now.Subtract(SessionStartTime.Value);
                hkTelemetry.Provider.GeneralInfo(sessionLength.TotalSeconds, BuildDate, ClientProtocol, ClientSdkVersion, System.Threading.Thread.CurrentThread.CurrentCulture.EnglishName);
            }

            // Send viewer telemetry data
            string enabledViewers = "";
            foreach (VdbProcessViewModel process in GetAllProcesses())
            {
                if (process.IsSelected)
                {
                    enabledViewers += process.Name + ", ";
                }
            }
            enabledViewers = enabledViewers.Trim();
            hkTelemetry.Provider.ViewerInfo(enabledViewers);

            // Send performance telemetry data
            if (FrameStats != null)
            {
                SessionJitterFrameDifferenceSum += (FrameStats.FrameCount - SessionJitterFramePrevious);
                double avgFrameJitterDifferences = SessionJitterFrameDifferenceSum / SessionJitterDifferenceCount;
                double avgJitterDifferences = SessionJitterJitterSum / SessionJitterDifferenceCount;

                hkTelemetry.Provider.PerformanceInfo(FrameStats.Mean, FrameStats.Max, FrameStats.StandardDeviation,
                    avgFrameJitterDifferences, avgJitterDifferences,
                    Client.RenderSurface.DrawFrameMs, Client.RenderSurface.UpdateFrameMs, FrameStats.FrameCount,
                    FrameStats.StepMean, FrameStats.StepMax, FrameStats.StepStandardDeviation);
            }

            // Send connection telemetry data
            
            hkTelemetry.Provider.ConnectionInfo(IsServerDiscoveryEnabled, AutoReconnectEnabled, DiscoveredServers.Count, MachineAddresses.Count,
                MachinePorts.Count, SessionConnectionType, SessionNumRecordActions, SessionSavedMovieSizes, SessionSavedMovieLengths);

            // Send camera telemetry data
            hkTelemetry.Provider.CameraInfo(RegisteredCamerasCollection.Count, CameraMovementMode, CameraUp.ToString(), CameraNearPlane,
                CameraFarPlane, CameraFieldOfView, CameraPerspectiveMode.ToString(), CameraLookSensitivity,
                SessionCameraNegativeXClicks, SessionCameraNegativeYClicks, SessionCameraNegativeZClicks,
                SessionCameraPositiveXClicks, SessionCameraPositiveYClicks, SessionCameraPositiveZClicks,
                SessionCameraDefaultClicks, SessionCameraFitCurrentClicks/*, SessionCameraDefaultSelectionClicks, SessionCameraFitCurrentSelectionClicks*/);

            // Send playback telemetry data
            hkTelemetry.Provider.PlaybackInfo(SessionNumPlay, SessionNumPlayForward, SessionNumPlayBackward,
                SessionNumPause, SessionNumGoToBegin, SessionNumGoToEnd,
                SessionNumOneFrameForward, SessionNumOneFrameBackward, SessionNumSaveMovie,
                IsThrottlingToServerFrameRate, ThrottlingFrameRate);

            // Send log viewer telemetry data
            hkTelemetry.Provider.LogViewerInfo(LocalLogCollection.Count, RemoteLogCollection.Count, MaxLogLimit);

            // Send render info telemetry data
            {
                VdbWidgetViewModel orientation = GetWidgetByName("Orientation");
                VdbWidgetViewModel origin = GetWidgetByName("Origin");
                VdbWidgetViewModel statsBarGraph = GetWidgetByName("Bar Graph");
                VdbWidgetViewModel depStatsBarGraph = GetWidgetByName("Deprecated Bar Graph");
                VdbWidgetViewModel statsLineGraph = GetWidgetByName("Line Graph");
                VdbWidgetViewModel status = GetWidgetByName("Status Info");
                VdbWidgetViewModel grid = GetWidgetByName("Grid");

                System.Diagnostics.Debug.Assert(orientation != null);
                System.Diagnostics.Debug.Assert(origin != null);
                System.Diagnostics.Debug.Assert(statsBarGraph != null);
                System.Diagnostics.Debug.Assert(depStatsBarGraph != null);
                System.Diagnostics.Debug.Assert(statsLineGraph != null);
                System.Diagnostics.Debug.Assert(status != null);
                System.Diagnostics.Debug.Assert(grid != null);

                hkTelemetry.Provider.RenderInfo(
                    orientation == null ? false : orientation.IsSelected,
                    orientation == null ? 0 : orientation.SessionNumToggles,
                    origin == null ? false : origin.IsSelected,
                    origin == null ? 0 : origin.SessionNumToggles,
                    // No longer have this widget, it's been combined with line graph
                    false, //statsText == null ? false : statsText.IsSelected,
                    0, //statsText == null ? 0 : statsText.SessionNumToggles,
                    statsBarGraph == null ? false : statsBarGraph.IsSelected,
                    statsBarGraph == null ? 0 : statsBarGraph.SessionNumToggles,
                    depStatsBarGraph == null ? false : depStatsBarGraph.IsSelected,
                    depStatsBarGraph == null ? 0 : depStatsBarGraph.SessionNumToggles,
                    statsLineGraph == null ? false : statsLineGraph.IsSelected,
                    statsLineGraph == null ? 0 : statsLineGraph.SessionNumToggles,
                    status == null ? false : status.IsSelected,
                    status == null ? 0 : status.SessionNumToggles,
                    grid == null ? false : grid.IsSelected,
                    grid == null ? 0 : grid.SessionNumToggles,
                    RenderWireframe,
                    SessionWireframeToggles,
                    RenderOutlineFaces,
                    SessionOutlineToggles,
                    RandomizedColors,
                    SessionRandomizeColorsToggles,
                    LockLightToCamera,
                    SessionLockLightToCameraToggles,
                    BackfaceCullingMode.ToString(),
                    SessionBackfaceCullingChanges,
                    SelectedClearColor.ToString(),
                    SessionClearColorChanges);
            }

            // Send shortcut telemetry data
            string shortcutString = "";
            foreach (VdbShortcutViewModel vdbShortcut in VdbCommands.CommandShortcuts)
            {
                shortcutString += "(" + vdbShortcut.ShortcutName + ": " + vdbShortcut.Shortcut + ") ";
            }
            hkTelemetry.Provider.ShortcutInfo(shortcutString);

            // Send inspection info data
            string inspectionQueries = String.Join(",", VdbObjectQueries);
            hkTelemetry.Provider.InspectorInfo(inspectionQueries, VdbObjectQueries.Count, SessionNumTreeFilterOperations, SessionInspectionCollapseClicks, (VdbObjectTree != null) ? VdbObjectTree.Count : -1, MaxVdbObjectModelContainerChildren);
        }
        #endregion

        #region VdbViewModel Connection
        internal void ConnectToMachineOrDisconnect(object o)
        {
            if (ConnectToNetworkActionString == HavokVisualDebugger.Properties.Resources.Connection_Disconnect)
            {
                DisconnectNetwork(o);
            }
            else
            {
                ConnectToMachine(o);
            }
        }
        internal void ConnectToMachine(object o)
        {
            if (bool.TrueString.Equals(o)) QueueAutoStartDebugRecording(GetDefaultDebugFileName());
            UInt16 port;
            if (UInt16.TryParse(MachinePort, out port))
            {
                Client.Call(nameof(Havok.Vdb.Client.ConnectToMachine), MachineAddress, port);
                if (AutoReconnectEnabled)
                {
                    Client.AttemptAutoReconnect = new Tuple<string, UInt16>(MachineAddress, port);
                }
            }
        }
        internal void ConnectToFileWithPrompt(object o)
        {
            DoConnectToFileWithPrompt(o, false);
        }
        internal void ConnectToFileWithPromptAndSyncToConnection(object o)
        {
            DoConnectToFileWithPrompt(o, true);
        }
        internal void ConnectToFile(object o)
        {
            string providedString = (o as string);
            DoConnectToFileName(providedString ?? FileName, false);
        }
        internal void ConnectToFileAndSyncToConnection(object o)
        {
            string providedString = (o as string);
            DoConnectToFileName(providedString ?? FileName, true);
        }
        internal void DisconnectNetwork(object o)
        {
            // If user is disconnecting from the network, stop initiating auto reconnect attempts
            if ((Client.AttemptAutoReconnect != null) && (ConnectedSource == Havok.Vdb.ConnectedSource.Network))
            {
                Client.AttemptAutoReconnect = null;
            }
            FileName = "";
            Client.Call(nameof(Havok.Vdb.Client.Disconnect));
        }
        internal void SelectDiscoveredServer(object o)
        {
            VdbServerInfoViewModel info = (VdbServerInfoViewModel)o;
            MachineAddress = info.DisplayAddress;
            MachinePort = info.DisplayPort;
            DisconnectNetwork(null);
            ConnectToMachine(null);
        }
        internal void GoToFrame(object o)
        {
            
            IsCurrentFrameFocused = true;
        }

        public bool IsCurrentFrameFocused
        {
            get
            {
                return _IsCurrentFrameFocused;
            }

            set
            {
                _IsCurrentFrameFocused = value;
                NotifyPropertyChanged(nameof(IsCurrentFrameFocused));
            }
        }
        private bool _IsCurrentFrameFocused = false;

        public Havok.Vdb.ConnectedSource ConnectedSource { get { return Client.GetProperty<Havok.Vdb.ConnectedSource>(nameof(Havok.Vdb.Client.ConnectedSource)); } }
        public Havok.Vdb.ConnectedState ConnectedState { get { return Client.GetProperty<Havok.Vdb.ConnectedState>(nameof(Havok.Vdb.Client.ConnectedState)); } }
        public bool IsDisconnected { get { return !IsConnecting && !IsConnected; } }
        public bool IsConnecting { get { return (ConnectedState == Havok.Vdb.ConnectedState.Connecting); } }
        public bool IsConnected { get { return (ConnectedState == Havok.Vdb.ConnectedState.Connected); } }
        public bool IsSyncedToConnection { get { return ((Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).GetProperty<Havok.Vdb.PlaybackFlags>(nameof(Havok.Vdb.PlaybackHandler.PlaybackFlags)) & (Havok.Vdb.PlaybackFlags.AvailableCommandsProcessed)) != 0); } }
        public bool IsServerDiscoveryEnabled
        {
            get { return Client.IsServerDiscoveryEnabled; }
            set
            {
                if (IsServerDiscoveryEnabled != value)
                {
                    Client.IsServerDiscoveryEnabled = value;
                    NotifyPropertyChanged();

                    VdbSettings.SetValue(nameof(Properties.Settings.VdbConnection_ServerDiscoveryEnabled), value);
                }
            }
        }

        private ObservableCollection<VdbServerInfoViewModel> DiscoveredServers;
        private CollectionViewSource DiscoveredServersViewSource;
        public ICollectionView DiscoveredServersView
        {
            get
            {
                return DiscoveredServersViewSource.View;
            }
        }

        public bool IsDiscoveredServerIncluded(object o)
        {
            VdbServerInfoViewModel server = o as VdbServerInfoViewModel;
            if (server == null)
            {
                return false;
            }

            if (!HasServerDiscoveryFilter)
            {
                return true;
            }


            if ((server.Name.IndexOf(ServerDiscoveryFilter, StringComparison.CurrentCultureIgnoreCase) != -1) ||
               (server.DisplayName.IndexOf(ServerDiscoveryFilter, StringComparison.CurrentCultureIgnoreCase) != -1) ||
               (server.DisplayPort.IndexOf(ServerDiscoveryFilter, StringComparison.CurrentCultureIgnoreCase) != -1) ||
               (server.DisplayAddress.IndexOf(ServerDiscoveryFilter, StringComparison.CurrentCultureIgnoreCase) != -1))
            {
                return true;
            }

            return false;
        }

        public string ServerDiscoveryFilter
        {
            get { return _ServerDiscoveryFilter; }
            set
            {
                if (_ServerDiscoveryFilter != value)
                {
                    _ServerDiscoveryFilter = value;

                    NotifyPropertyChanged();
                    NotifyPropertyChanged(nameof(HasServerDiscoveryFilter));

                    if (DiscoveredServersView.CanFilter && DiscoveredServersView.SourceCollection != null)
                    {
                        DiscoveredServersView.Refresh();
                    }
                }
            }
        }
        private string _ServerDiscoveryFilter;

        public bool HasServerDiscoveryFilter
        {
            get
            {
                return !String.IsNullOrEmpty(ServerDiscoveryFilter);
            }
        }

        public String ConnectToNetworkActionString
        {
            get
            {
                return
                    ((ConnectedSource == Havok.Vdb.ConnectedSource.Network) && !IsDisconnected) ?
                    HavokVisualDebugger.Properties.Resources.Connection_Disconnect :
                    HavokVisualDebugger.Properties.Resources.Connection_Connect;
            }
        }

        public string ServerDiscoveryActionString
        {
            get
            {
                return
                    (Client.IsServerDiscoveryEnabled) ?
                    HavokVisualDebugger.Properties.Resources.Connection_DisableServerDiscovery :
                    HavokVisualDebugger.Properties.Resources.Connection_EnableServerDiscovery;
            }
        }

        public string ConnectedStateString
        {
            get
            {
                return
                    (ConnectedState == Havok.Vdb.ConnectedState.Connected) ?
                        HavokVisualDebugger.Properties.Resources.Connection_CurrentlyConnected:
                    (ConnectedState == Havok.Vdb.ConnectedState.Connecting) ?
                        HavokVisualDebugger.Properties.Resources.Connection_AttemptingToConnect:
                    (ConnectedState == Havok.Vdb.ConnectedState.Disconnected) ?
                        HavokVisualDebugger.Properties.Resources.Connection_CurrentlyDisconnected :
                    (IsRecording) ?
                        HavokVisualDebugger.Properties.Resources.Connection_Recording :
                    HavokVisualDebugger.Properties.Resources.Connection_Unknown;
            }
        }

        /// <summary>
        /// Machine Port
        /// </summary>
		public string MachinePort
        {
            get { return _MachinePort; }
            set
            {
                // Trim before checking to ensure no errant whitespaces occur
                value = value.Trim();
                if (_MachinePort != value)
                {
                    _MachinePort = value;
                    if (Client.AttemptAutoReconnect != null)
                    {
                        ConnectToMachine(this);
                    }

                    // Add the new query to the list of used queries
                    hkCollectionUtils.InsertIntoLimitedStringCollection(MachinePorts, value, MaxMachinePortEntries);

                    VdbSettings.SetValue(nameof(Properties.Settings.VdbConnection_Port), value);
                    NotifyPropertyChanged();
                    NotifyPropertyChanged(nameof(IsMachineInfoValid));
                }
            }
        }
        private string _MachinePort = "";
        public ObservableCollection<string> MachinePorts { get; set; }

        /// <summary>
        /// Machine Discovery Port
        /// </summary>
        public string MachineDiscoveryPort
        {
            get { return Client.ServerDiscoveryPort.ToString(); }
            set
            {
                // Trim before checking to ensure no errant whitespaces occur
                value = value.Trim();
                if (!MachineDiscoveryPort.Equals(value))
                {
                    UInt16 port;
                    if (UInt16.TryParse(value, out port))
                    {
                        Client.ServerDiscoveryPort = port;

                        // Add the new query to the list of used queries
                        hkCollectionUtils.InsertIntoLimitedStringCollection(MachineDiscoveryPorts, value, MaxMachinePortEntries);

                        VdbSettings.SetValue(nameof(Properties.Settings.VdbConnection_DiscoveryPort), value);
                        NotifyPropertyChanged();
                    }
                }
            }
        }
        public ObservableCollection<string> MachineDiscoveryPorts { get; set; }

        private const int MaxMachinePortEntries = 20;

        private void MachinePorts_CollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            StringCollection portsCollection = null;
            if (sender == MachinePorts)
            {
                portsCollection = Properties.Settings.Default.VdbConnection_MachinePorts;
            }
            else if (sender == MachineDiscoveryPorts)
            {
                portsCollection = Properties.Settings.Default.VdbConnection_MachineDiscoveryPorts;
            }
            else
            {
                System.Diagnostics.Debug.Assert(false, "Unknown ports collection");
                return;
            }

            if (e.NewItems != null)
            {
                foreach (string newItem in e.NewItems)
                {
                    VdbSettings.SetCollectionItemEnabled(portsCollection, newItem, true);
                }
            }
            else if (e.OldItems != null)
            {
                foreach (string oldItem in e.OldItems)
                {
                    VdbSettings.SetCollectionItemEnabled(portsCollection, oldItem, false);
                }
            }
        }

        internal void DeleteMachinePortItem(object o = null)
        {
            string portToDelete = o as string;
            if (MachinePorts.Contains(portToDelete))
            {
                MachinePorts.Remove(portToDelete);
                
                NotifyPropertyChanged(nameof(MachinePorts));
            }
        }
        internal void DeleteMachineDiscoveryPortItem(object o = null)
        {
            string portToDelete = o as string;
            if (MachineDiscoveryPorts.Contains(portToDelete))
            {
                MachineDiscoveryPorts.Remove(portToDelete);
                
                NotifyPropertyChanged(nameof(MachineDiscoveryPorts));
            }
        }

        /// <summary>
        /// Machine Address
        /// </summary>
        public string MachineAddress
        {
            get { return _MachineAddress; }
            set
            {
                // Trim before checking to ensure no errant whitespaces occur
                value = value.Trim();
                if (_MachineAddress != value)
                {
					_MachineAddress = value;
					if (Client.AttemptAutoReconnect != null)
                    {
                        ConnectToMachine(this);
                    }

                    // Add the new query to the list of used queries
                    hkCollectionUtils.InsertIntoLimitedStringCollection(MachineAddresses, value, MaxMachineAddressEntries);

                    VdbSettings.SetValue(nameof(Properties.Settings.VdbConnection_Address), value);
                    NotifyPropertyChanged();
                    NotifyPropertyChanged(nameof(IsMachineInfoValid));
                }
            }
        }
        private string _MachineAddress = "";
        public ObservableCollection<string> MachineAddresses { get; set; }
        const int MaxMachineAddressEntries = 20;

        private void MachineAddresses_CollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            if (e.NewItems != null)
            {
                foreach (string newItem in e.NewItems)
                {
                    VdbSettings.SetCollectionItemEnabled(Properties.Settings.Default.VdbConnection_MachineAddresses, newItem, true);
                }
            }
            else if (e.OldItems != null)
            {
                foreach (string oldItem in e.OldItems)
                {
                    VdbSettings.SetCollectionItemEnabled(Properties.Settings.Default.VdbConnection_MachineAddresses, oldItem, false);
                }
            }
        }

        internal void DeleteMachineAddressItem(object o = null)
        {
            string addressToDelete = o as string;
            if (MachineAddresses.Contains(addressToDelete))
            {
                MachineAddresses.Remove(addressToDelete);
                NotifyPropertyChanged(nameof(MachineAddresses));
            }
        }

        public string FileName
        {
            get { return _FileName; }
            set
            {
                if (_FileName != value)
                {
                    _FileName = value;
                    NotifyPropertyChanged();
                }
            }
        }
        private string _FileName = "";
        public bool AutoReconnectEnabled
        {
            get
            {
                return _AutoReconnectEnabled;
            }
            set
            {
                if (_AutoReconnectEnabled != value)
                {
                    _AutoReconnectEnabled = value;
                    if (_AutoReconnectEnabled)
                    {
                        // Don't report network connection timeouts
                        Havok.Vdb.Error.SetIsEnabled(0xedb00159, false);
                    }
                    else
                    {
                        Client.AttemptAutoReconnect = null;
                        // Start reporting network connection timeouts
                        Havok.Vdb.Error.SetIsEnabled(0xedb00159, true);
                    }
                    VdbSettings.SetValue(nameof(Properties.Settings.VdbConnection_AutoReconnect), value);
                    NotifyPropertyChanged();
                }
            }
        }
        bool _AutoReconnectEnabled;
        public bool IsMachineInfoValid
        {
            get
            {
                int port;
                return
                    !String.IsNullOrEmpty(MachinePort) &&
                    !String.IsNullOrEmpty(MachineAddress) &&
                    int.TryParse(MachinePort, out port);
            }
        }

        public string ConnectionName
        {
            get
            {
                Havok.Vdb.ConnectedSource source = Client.GetProperty<Havok.Vdb.ConnectedSource>(nameof(Havok.Vdb.Client.ConnectedSource));
                if (source == Havok.Vdb.ConnectedSource.Network)
                {
                    _ConnectionName = "(" + MachineAddress + " : " + MachinePort + ")";
                }
                else if (source == Havok.Vdb.ConnectedSource.File)
                {
                    _ConnectionName = "(" + FileName + ")";
                }

                return _ConnectionName;
            }
        }
        private string _ConnectionName = "";

        private static int MaxRecentFiles = 10;
        public ObservableCollection<string> RecentFiles { get; set; }

        private void RecentFiles_CollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            if (e.NewItems != null)
            {
                foreach (string newItem in e.NewItems)
                {
                    VdbSettings.SetCollectionItemEnabled(Properties.Settings.Default.VdbSettings_RecentFiles, newItem, true);
                }
            }
            else if (e.OldItems != null)
            {
                foreach (string oldItem in e.OldItems)
                {
                    VdbSettings.SetCollectionItemEnabled(Properties.Settings.Default.VdbSettings_RecentFiles, oldItem, false);
                }
            }
        }

        internal void OpenRecentFile(object o)
        {
            string fileName = o as string;
            if( String.IsNullOrEmpty(fileName) )
            {
                return;
            }

            ConnectToFile(fileName);
        }

        #endregion

        #region VdbViewModel Playback Handling
        public bool HasPlaybackData { get { return (BufferFrameCount > 0); } }
        public bool IsReplaying { get { return ((Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).GetProperty<Havok.Vdb.PlaybackFlags>(nameof(Havok.Vdb.PlaybackHandler.PlaybackFlags)) & (Havok.Vdb.PlaybackFlags.Replaying)) != 0); } }
        public bool IsRecording { get { return ((Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).GetProperty<Havok.Vdb.PlaybackFlags>(nameof(Havok.Vdb.PlaybackHandler.PlaybackFlags)) & (Havok.Vdb.PlaybackFlags.Recording)) != 0); } }
        public String RecordActionString { get { return IsRecording ? HavokVisualDebugger.Properties.Resources.MenuHeader_StopRecording : HavokVisualDebugger.Properties.Resources.MenuHeader_StartRecording; } }
        internal void ToggleRecording(object o)
        {
            if (VdbCommands.StopRecordingCommand.CanExecute(o))
            {
                VdbCommands.StopRecordingCommand.Execute(o);
            }
            else if (VdbCommands.StartRecordingCommand.CanExecute(0))
            {
                VdbCommands.StartRecordingCommand.Execute(o);
            }
            else
            {
                System.Diagnostics.Debug.Assert(false, "We expect to be able to execute one or the other command");
            }
        }

        internal void StartRecording(object o)
        {
            string fileName = DoFileSave();
            if (fileName != null)
            {
                if (bool.TrueString.Equals(o))
                {
                    Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).Call(nameof(Havok.Vdb.PlaybackHandler.StartDebugRecording), fileName);
                }
                else
                {
                    Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).Call(nameof(Havok.Vdb.PlaybackHandler.StartRecording), fileName);
                }
                SessionNumRecordActions++;
                SessionMovieFileName = fileName;
                SessionMovieStartFrame = CurrentFrame;
            }
        }

        internal void StopRecording(object o)
        {
            Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).Call(nameof(Havok.Vdb.PlaybackHandler.StopRecording));

            System.IO.FileInfo fileInfo = new System.IO.FileInfo(SessionMovieFileName);
            SessionSavedMovieSizes.Add(fileInfo.Length);
            SessionSavedMovieLengths.Add(CurrentFrame - SessionMovieStartFrame);
        }

        internal void SaveReplay(object o)
        {
            string fileName = DoFileSave();
            if (fileName != null)
            {
                if (bool.TrueString.Equals(o))
                {
                    Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).Call(nameof(Havok.Vdb.PlaybackHandler.SaveDebugReplay), fileName);
                }
                else
                {
                    Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).Call(nameof(Havok.Vdb.PlaybackHandler.SaveReplay), fileName);
                }
            }
        }

        internal void SkipToStart(object o)
        {
            CurrentFrame = MinFrame;
            SessionNumGoToBegin++;
        }

        internal void StepBwd(object o)
        {
            CurrentFrame = (CurrentFrame - 1);
            Pause(o);
            SessionNumOneFrameBackward++;
            SessionNumPause--;
        }

        internal void PlayBwd(object o)
        {
            FramesChanging?.Invoke(this);
            Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).SetPropertyAsync(nameof(Havok.Vdb.PlaybackHandler.PlaybackState), Havok.Vdb.PlaybackState.Playing);
            Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).SetPropertyAsync(nameof(Havok.Vdb.PlaybackHandler.PlaybackDirection), Havok.Vdb.PlaybackDirection.Backward);
            SessionNumPlayBackward++;
        }

        internal void TogglePlaying(object o)
        {
            if (PlaybackState == Havok.Vdb.PlaybackState.Playing)
            {
                Pause(o);
            }
            else
            {
                Play(o);
            }
        }

        internal void PlayFwd(object o)
        {
            FramesChanging?.Invoke(this);
            Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).SetPropertyAsync(nameof(Havok.Vdb.PlaybackHandler.PlaybackState), Havok.Vdb.PlaybackState.Playing);
            Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).SetPropertyAsync(nameof(Havok.Vdb.PlaybackHandler.PlaybackDirection), Havok.Vdb.PlaybackDirection.Forward);
            SessionNumPlayForward++;
        }

        internal void StepFwd(object o)
        {
            CurrentFrame = (CurrentFrame + 1);
            Pause(o);
            SessionNumOneFrameForward++;
            SessionNumPause--;
        }

        internal void AdvanceOne(object o)
        {
            FramesChanging?.Invoke(this);
            Havok.Vdb.PlaybackDirection? direction = (o as Havok.Vdb.PlaybackDirection?);
            if (direction.HasValue)
            {
                Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).CallAsync(nameof(Havok.Vdb.PlaybackHandler.AdvanceOne), direction.Value);
            }
            else
            {
                Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).CallAsync(nameof(Havok.Vdb.PlaybackHandler.AdvanceOne));
            }
        }

        internal void AdvanceOneFwd(object o)
        {
            AdvanceOne(Havok.Vdb.PlaybackDirection.Forward);
        }

        internal void SkipToEnd(object o)
        {
            CurrentFrame = MaxFrame;
            SessionNumGoToEnd++;
        }

        internal void SyncToConnection(object o)
        {
            Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).CallAsync(nameof(Havok.Vdb.PlaybackHandler.SyncToConnection));
        }

        public bool IsThrottlingToServerFrameRate
        {
            get
            {
                return Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).GetProperty<double>(nameof(Havok.Vdb.PlaybackHandler.PlaybackFrameRate)) == Havok.Vdb.PlaybackFrameRates.ServerFrameRate;
            }
            set
            {
                if (value)
                {
                    Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).SetPropertyAsync(nameof(Havok.Vdb.PlaybackHandler.PlaybackFrameRate), Havok.Vdb.PlaybackFrameRates.ServerFrameRate);
                }
                else
                {
                    Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).SetPropertyAsync(nameof(Havok.Vdb.PlaybackHandler.PlaybackFrameRate), _ThrottlingFrameRate);
                }
                NotifyPropertyChanged();
            }
        }

        public double ThrottlingFrameRate
        {
            get
            {
                return _ThrottlingFrameRate;
            }
            set
            {
                _ThrottlingFrameRate = value;
                if (!IsThrottlingToServerFrameRate)
                {
                    Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).SetPropertyAsync(nameof(Havok.Vdb.PlaybackHandler.PlaybackFrameRate), _ThrottlingFrameRate);
                }
                NotifyPropertyChanged();
            }
        }
        private double _ThrottlingFrameRate = 60;

        internal void Play(object o = null)
        {
            FramesChanging?.Invoke(this);
            Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).SetPropertyAsync(nameof(Havok.Vdb.PlaybackHandler.PlaybackState), Havok.Vdb.PlaybackState.Playing);
            SessionNumPlay++;
        }

        internal void PlayTo(object o)
        {
            Int64? oAsI = (o as Int64?);
            if (oAsI.HasValue)
            {
                FramesChanging?.Invoke(this);
                Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).CallAsync(nameof(Havok.Vdb.PlaybackHandler.PlayTo), oAsI.Value);
            }
        }

        public bool IsPaused { get { return (HasPlaybackData && (PlaybackState == Havok.Vdb.PlaybackState.Paused)); } }
        internal void Pause(object o = null)
        {
            Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).SetPropertyAsync(nameof(Havok.Vdb.PlaybackHandler.PlaybackState), Havok.Vdb.PlaybackState.Paused);
            SessionNumPause++;
        }

        public Havok.Vdb.PlaybackState PlaybackState { get { return Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).GetProperty<Havok.Vdb.PlaybackState>(nameof(Havok.Vdb.PlaybackHandler.PlaybackState)); } }

        public bool IsPlayingBackward
        {
            get { return (HasPlaybackData && (PlaybackDirection == Havok.Vdb.PlaybackDirection.Backward)); }
        }

        public bool IsPlayingForward
        {
            get { return (HasPlaybackData && (PlaybackDirection == Havok.Vdb.PlaybackDirection.Forward)); }
        }

        public bool IsAutoRefreshPaused
        {
            get { return _IsAutoRefreshPaused; }
            set
            {
                if (_IsAutoRefreshPaused != value)
                {
                    _IsAutoRefreshPaused = value;
                    if (value)
                    {
                        AutoRefreshPaused?.Invoke(this);
                    }
                    else
                    {
                        AutoRefreshUnPaused?.Invoke(this);
                        if (PlaybackState == Havok.Vdb.PlaybackState.Paused)
                        {
                            AutoRefreshTick?.Invoke(this);
                        }
                    }
                    NotifyPropertyChanged();
                }
            }
        }
        private bool _IsAutoRefreshPaused;

        public bool IsDraggingPlaybackThumb
        {
            get
            {
                return _IsDraggingPlaybackThumb;
            }
            set
            {
                if (_IsDraggingPlaybackThumb != value)
                {
                    _IsDraggingPlaybackThumb = value;
                    if (value)
                    {
                        Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).Call(nameof(Havok.Vdb.PlaybackHandler.SuspendExpensiveFrameUpdates));
                        _CurrentFramePropertyHook = _DelayNonProcessedFrameSets;
                        IsAutoRefreshPaused = true;
                    }
                    else
                    {
                        Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).Call(nameof(Havok.Vdb.PlaybackHandler.ResumeNormalFrameUpdates));
                        if (_CurrentFramePropertyHook == _DelayNonProcessedFrameSets)
                        {
                            _CurrentFramePropertyHook = null;
                            if (_DelayedNonProcessedFrameSet >= 0)
                            {
                                CurrentFrame = _DelayedNonProcessedFrameSet;
                                _DelayedNonProcessedFrameSet = -1;
                            }
                        }
                        IsAutoRefreshPaused = false;
                    }
                    NotifyPropertyChanged();
                }
            }
        }

        private Int64 _DelayNonProcessedFrameSets(Int64 potentialFrameSet)
        {
            // Getter
            if (potentialFrameSet < 0)
            {
                if (_DelayedNonProcessedFrameSet >= 0)
                {
                    return _DelayedNonProcessedFrameSet;
                }
                else
                {
                    return Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).GetProperty<Int64>(nameof(Havok.Vdb.PlaybackHandler.CurrentFrame));
                }
            }
            // Setter
            else
            {
                if (potentialFrameSet >= MaxProcessedFrame)
                {
                    _DelayedNonProcessedFrameSet = potentialFrameSet;
                }
                else
                {
                    _DelayedNonProcessedFrameSet = -1;
                    Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).SetPropertyAsync(nameof(Havok.Vdb.PlaybackHandler.CurrentFrame), potentialFrameSet);
                }
                return -1;
            }
        }
        private Int64 _DelayedNonProcessedFrameSet = -1;
        private bool _IsDraggingPlaybackThumb;

        public Havok.Vdb.PlaybackDirection PlaybackDirection { get { return Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).GetProperty<Havok.Vdb.PlaybackDirection>(nameof(Havok.Vdb.PlaybackHandler.PlaybackDirection)); } }

        public Int64 CurrentBufferFrame
        {
            get { return (CurrentFrame - MinFrame); }
            set { CurrentFrame = (MinFrame + value); }
        }
        public Int64 MaxProcessedBufferFrame { get { return (MaxProcessedFrame - MinFrame); } }
        public Int64 BufferFrameCount { get { return (MaxFrame - MinFrame); } }

        public Int64 CurrentFrame
        {
            get
            {
                if (_CurrentFramePropertyHook != null)
                {
                    return _CurrentFramePropertyHook(-1);
                }
                else
                {
                    return Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).GetProperty<Int64>(nameof(Havok.Vdb.PlaybackHandler.CurrentFrame));
                }
            }
            set
            {
                if (_CurrentFramePropertyHook != null)
                {
                    if (value >= 0)
                    {
                        _CurrentFramePropertyHook(value);
                    }
                }
                else
                {
                    Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).SetPropertyAsync(nameof(Havok.Vdb.PlaybackHandler.CurrentFrame), value);
                }

                if (CurrentFrame != value)
                {
                    FramesChanging?.Invoke(this);
                }
            }
        }
        private delegate Int64 CurrentFramePropertyHook(Int64 potentialFrameSet);
        private CurrentFramePropertyHook _CurrentFramePropertyHook;
        public Int64 MaxProcessedFrame { get { return Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).GetProperty<Int64>(nameof(Havok.Vdb.PlaybackHandler.MaxProcessedFrame)); } }
        public Int64 MinFrame { get { return Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).GetProperty<Int64>(nameof(Havok.Vdb.PlaybackHandler.MinFrame)); } }
        public Int64 MaxFrame { get { return Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).GetProperty<Int64>(nameof(Havok.Vdb.PlaybackHandler.MaxFrame)); } }

        public DoubleCollection PlaybackTicks
        {
            get
            {
                double frameTickInterval = PlaybackTickFrequency;
                DoubleCollection c = new DoubleCollection();
                for (int i = 0; i <= 20; i++)
                {
                    c.Add(MinFrame + (i * frameTickInterval));
                }

                return c;
            }
        }

        public double PlaybackTickFrequency
        {
            get
            {
                return Math.Ceiling((double)(MaxFrame - MinFrame) / 20);
            }
        }

        public bool PauseOnScrub { get; set; }

        #endregion

        #region VdbViewModel Process Handling
        public VdbProcessCollection RegisteredServerActionProcessesCollection { get; private set; }
        public VdbProcessCollection DefaultProcessesCollection { get; private set; }
        public hkNonResettableObservableCollection<VdbProcessesViewModel> CategoryProcessCollection { get; private set; }

        public IEnumerable<VdbProcessViewModel> GetAllProcesses()
        {
            // Clear selected non-default processes
            foreach (VdbProcessesViewModel categoryProc in CategoryProcessCollection)
            {
                foreach (VdbProcessViewModel process in categoryProc.Processes)
                {
                    yield return process;
                }
            }
        }

        internal void ResetViewers(object o)
        {
            // Clear selected non-default processes
            foreach (VdbProcessViewModel process in GetAllProcesses())
            {
                process.IsSelected = false;
            }

            // Clear the settings in case no connection established yet
            // Need to do this so that users can clear their previously selected viewers before a connection is established.
            // Use case: If a viewer causes a crash on connection, there's no way to turn it off since the user doesn't have access
            // to the viewer list until connection and the tool will automatically turn on all viewers, so need a way to clear the
            // "selected" viewers list, as we do below.
            Properties.Settings.Default.VdbEnabledViewers.Clear();
        }

        internal void ResetViewersToDefault(object o)
        {
            // Clear selected non-default processes
            foreach (VdbProcessViewModel process in GetAllProcesses())
            {
                process.IsSelected = DefaultProcessesCollection.Contains(process);
            }
        }
        #endregion

        #region VdbViewModel Setup Handling
        public Havok.Vdb.SetupInfo ServerInfo { get { return Client.GetSubInterop(nameof(Havok.Vdb.Client.SetupHandler)).GetProperty<Havok.Vdb.SetupHandlerEventArgs>(nameof(Havok.Vdb.SetupHandler.SetupInfo)).ServerInfo; } }
        public Havok.Vdb.SetupInfo ClientInfo { get { return Client.GetSubInterop(nameof(Havok.Vdb.Client.SetupHandler)).GetProperty<Havok.Vdb.SetupHandlerEventArgs>(nameof(Havok.Vdb.SetupHandler.SetupInfo)).ClientInfo; } }
        #endregion

        #region VdbViewModel Stats Handling
        
        
        
        
        public Havok.Vdb.MemStatsReceivedEventArgs MemStats { get { return Client.GetSubInterop(nameof(Havok.Vdb.Client.StatsHandler)).GetProperty<Havok.Vdb.MemStatsReceivedEventArgs>(nameof(Havok.Vdb.StatsHandler.MemStats)); } }
        public Havok.Vdb.FrameStatsReceivedEventArgs FrameStats { get { return Client.GetSubInterop(nameof(Havok.Vdb.Client.StatsHandler)).GetProperty<Havok.Vdb.FrameStatsReceivedEventArgs>(nameof(Havok.Vdb.StatsHandler.FrameStats)); } }

        public ObservableCollection<TimerNodeViewModel> TimerViewModelView { get; set; }
        public ObservableCollection<TimerNodeViewModel> TimerViewModelDataFlat { get; set; }
        public Dictionary<string, List<TimerNodeViewModel>> TimerPathToViewModel = new Dictionary<string, List<TimerNodeViewModel>>();
        public VdbTimerNodeManager TimerNodeManager;

        public bool IsDiagnosticsShown
        {
            get
            {
                return VdbSettings.GetValue<bool>(nameof(Properties.Settings.VdbUi_DiagnosticsShown));
            }

            set
            {
                VdbSettings.SetValue(nameof(Properties.Settings.VdbUi_DiagnosticsShown), value);
                NotifyPropertyChanged();
            }
        }

        internal void ToggleDiagnosticsPanelShown(object o = null)
        {
            IsDiagnosticsShown = !IsDiagnosticsShown;
        }

        public void DeselectAllTimers(object o = null)
        {
            TimerNodeManager.DeselectAllTimers();
        }

        public bool ShowSelectedTimersOnly
        {
            get { return _ShowSelectedTimersOnly; }
            set
            {
                if (_ShowSelectedTimersOnly != value)
                {
                    _ShowSelectedTimersOnly = value;
                    TimerNodeManager.ToggleShowSelectedTimersOnly();
                    NotifyPropertyChanged();
                }
            }
        }
        private bool _ShowSelectedTimersOnly = false;

        public string TimerFilter
        {
            get { return _TimerFilter; }
            set
            {
                if (_TimerFilter != value)
                {
                    _TimerFilter = value;
                    if( TimerNodeManager.UpdateTimerFilter(value) )
                    {
                        NotifyPropertyChanged(nameof(TimerViewModelView));
                    }

                    NotifyPropertyChanged();
                    NotifyPropertyChanged(nameof(HasTimerFilter));
                }
            }
        }
        private string _TimerFilter;

        public bool HasTimerFilter
        {
            get
            {
                return !String.IsNullOrEmpty(TimerFilter);
            }
        }

        private void UpdateTimers(Havok.Vdb.PerfStatsReceivedEventArgs args)
        {
            using (Client.AcquireLock())
            {
                if (TimerNodeManager.UpdateTimers(args))
                {
                    NotifyPropertyChanged(nameof(TimerViewModelView));
                }
            }
        }

        public double TimerMinY
        {
            get
            {
                VdbWidgetViewModel widget = GetWidgetByName("Line Graph");
                return (widget != null) ? widget.GetOption<double>("MinY") : 0;
            }
            set
            {
                VdbWidgetViewModel widget = GetWidgetByName("Line Graph");
                if (widget != null)
                {
                    double oldMinY = widget.SetOption("MinY", value);
                    if (oldMinY != value)
                    {
                        VdbSettings.SetValue(nameof(Properties.Settings.VdbWidgets_TimerMinY), value);
                        NotifyPropertyChanged();
                    }
                }
            }
        }

        public double TimerMaxY
        {
            get
            {
                VdbWidgetViewModel widget = GetWidgetByName("Line Graph");
                return (widget != null) ? widget.GetOption<double>("MaxY") : 0;
            }
            set
            {
                VdbWidgetViewModel widget = GetWidgetByName("Line Graph");
                if (widget != null)
                {
                    double oldMinY = widget.SetOption("MaxY", value);
                    if (oldMinY != value)
                    {
                        VdbSettings.SetValue(nameof(Properties.Settings.VdbWidgets_TimerMaxY), value);
                        NotifyPropertyChanged();
                    }
                }
            }
        }

        #endregion

        #region VdbViewModel Camera Handling

        #region VdbViewModel Registered Cameras
        public ObservableCollection<VdbCameraViewModel> RegisteredCamerasCollection { get; private set; }

        public bool IsCameraLockedToServer
        {
            get { return (Client.RenderSurface.RenderCamera != null); }
        }

        internal void UnlockCamera(object o = null)
        {
            Client.RenderSurface.RenderCamera = null;
            NotifyIfRenderServerCameraChanged();
        }

        internal void SwapToCamera1(object o)
        {
            SwapToCamera(1);
        }
        internal void SwapToCamera2(object o)
        {
            SwapToCamera(2);
        }
        internal void SwapToCamera3(object o)
        {
            SwapToCamera(3);
        }
        internal void SwapToCamera4(object o)
        {
            SwapToCamera(4);
        }
        internal void SwapToCamera5(object o)
        {
            SwapToCamera(5);
        }
        private void SwapToCamera(int camera)
        {
            int cameraIndex = camera - 1;
            if(cameraIndex < RegisteredCamerasCollection.Count )
            {
                RegisteredCamerasCollection[cameraIndex].IsSelected = true;
            }
        }

        internal void NotifyIfRenderServerCameraChanged()
        {
            if (m_prevFrameCameraLock != Client.RenderSurface.RenderCamera)
            {
                (new List<VdbCameraViewModel>(RegisteredCamerasCollection))
                    .FindAll(c => (c.Name == m_prevFrameCameraLock) || (c.Name == Client.RenderSurface.RenderCamera))
                    .ForEach(c => c.NotifySelectionChanged());
                NotifyPropertyChanged(nameof(IsCameraLockedToServer));
                m_prevFrameCameraLock = Client.RenderSurface.RenderCamera;
            }
        }
        private string m_prevFrameCameraLock = null;

        #endregion

        #region VdbViewModel Camera Properties
        public double CameraMinimumPlane
        {
            get { return (double)GetCameraPropertyValueBackend("Minimum Plane", 0.001); }
        }

        public double CameraMinimumFov
        {
            get { return (double)GetCameraPropertyValueBackend("Minimum Fov", 0.1); }
        }

        public double CameraMaximumFov
        {
            get { return (double)GetCameraPropertyValueBackend("Maximum Fov", 179.0); }
        }

        public double CameraNearPlane
        {
            get { return (double)GetCameraPropertyValueBackend("Near Plane", 0.01); }
            set
            {
                // Clamp value
                value = Math.Max(CameraMinimumPlane, value);
                SetCameraPropertyValueBackend("Near Plane", value);
                NotifyPropertyChanged();
            }
        }

        public double CameraFarPlane
        {
            get { return (double)GetCameraPropertyValueBackend("Far Plane", 10000.0); }
            set
            {
                value = Math.Max(CameraMinimumPlane + CameraNearPlane, value);
                SetCameraPropertyValueBackend("Far Plane", value);
                NotifyPropertyChanged();
            }
        }

        public double CameraFieldOfView
        {
            get { return (double)GetCameraPropertyValueBackend("Field of View", 45.0); }
            set
            {
                // Clamp value
                value = hkMath.Clamp(value, CameraMinimumFov, CameraMaximumFov);
                SetCameraPropertyValueBackend("Field of View", value);
                NotifyPropertyChanged();
            }
        }

        public bool CameraPerspectiveMode
        {
            get { return (bool)GetCameraPropertyValueBackend("Perspective Mode", true); }
            set { SetCameraPropertyValueBackend("Perspective Mode", value); NotifyPropertyChanged(); }
        }
        internal void TogglePerspectiveMode(object o)
        {
            CameraPerspectiveMode = !CameraPerspectiveMode;
        }

        public Havok.Vdb.Vector CameraUp
        {
            get { return (Havok.Vdb.Vector)GetCameraPropertyValueBackend("Up Axis", new Havok.Vdb.Vector(0, 1, 0)); }
            set
            {
                _BackupDisplayWorldUp = value;
                SetCameraPropertyValueBackend("Up Axis", value);
                NotifyPropertyChanged();
                NotifyPropertyChanged(nameof(CameraUpIsNegated));
                NotifyPropertyChanged(nameof(CameraUpOnXAxis));
                NotifyPropertyChanged(nameof(CameraUpOnYAxis));
                NotifyPropertyChanged(nameof(CameraUpOnZAxis));
            }
        }

        public Havok.Vdb.Vector DisplayWorldUp
        {
            get
            {
                VdbClientInterop displayHandler = Client.GetSubInterop(nameof(Havok.Vdb.Client.DisplayHandler));
                Havok.Vdb.DisplayOptions options = displayHandler.GetProperty<Havok.Vdb.DisplayOptions>(nameof(Havok.Vdb.DisplayHandler.DisplayOptions));
                if (options != null)
                {
                    return options.Up;
                }
                else
                {
                    return _BackupDisplayWorldUp;
                }
            }
        }
        private Havok.Vdb.Vector _BackupDisplayWorldUp = new Havok.Vdb.Vector(0, 1, 0);

        public bool CameraUpOnXAxis
        {
            get
            {
                Havok.Vdb.Vector cameraUp = CameraUp;
                double absX = Math.Abs(cameraUp.X);
                double absY = Math.Abs(cameraUp.Y);
                double absZ = Math.Abs(cameraUp.Z);
                return (absX >= absY) && (absX >= absZ);
            }
            set
            {
                if (value)
                {
                    CameraUp = new Havok.Vdb.Vector(1.0f - 2.0f * Convert.ToSingle(CameraUpIsNegated), 0.0f, 0.0f);
                }
                else
                {
                    System.Diagnostics.Debug.Assert(CameraUpOnYAxis || CameraUpOnZAxis, "Only set to true, dependent bools are updated");
                }
            }
        }
        internal void SetCameraXUpAxis(object o)
        {
            CameraUpOnXAxis = true;
        }

        public bool CameraUpOnYAxis
        {
            get
            {
                Havok.Vdb.Vector cameraUp = CameraUp;
                double absX = Math.Abs(cameraUp.X);
                double absY = Math.Abs(cameraUp.Y);
                double absZ = Math.Abs(cameraUp.Z);
                return (absY >= absX) && (absY >= absZ);
            }
            set
            {
                if (value)
                {
                    CameraUp = new Havok.Vdb.Vector(0.0f, 1.0f - 2.0f * Convert.ToSingle(CameraUpIsNegated), 0.0f);
                }
                else
                {
                    System.Diagnostics.Debug.Assert(CameraUpOnXAxis || CameraUpOnZAxis, "Only set to true, dependent bools are updated");
                }
            }
        }
        internal void SetCameraYUpAxis(object o)
        {
            CameraUpOnYAxis = true;
        }

        public bool CameraUpOnZAxis
        {
            get
            {
                Havok.Vdb.Vector cameraUp = CameraUp;
                double absX = Math.Abs(cameraUp.X);
                double absY = Math.Abs(cameraUp.Y);
                double absZ = Math.Abs(cameraUp.Z);
                return (absZ >= absX) && (absZ >= absY);
            }
            set
            {
                if (value)
                {
                    CameraUp = new Havok.Vdb.Vector(0.0f, 0.0f, 1.0f - 2.0f * Convert.ToSingle(CameraUpIsNegated));
                }
                else
                {
                    System.Diagnostics.Debug.Assert(CameraUpOnXAxis || CameraUpOnYAxis, "Only set to true, dependent bools are updated");
                }
            }
        }
        internal void SetCameraZUpAxis(object o)
        {
            CameraUpOnZAxis = true;
        }

        public bool CameraUpIsNegated
        {
            get
            {
                Havok.Vdb.Vector cameraUp = CameraUp;
                return (cameraUp.X < 0) || (cameraUp.Y < 0) || (cameraUp.Z < 0);
            }
            set
            {
                if (CameraUpIsNegated != value)
                {
                    // Unlock first so that our cameraUp is updated.
                    UnlockCamera(null);
                    Havok.Vdb.Vector cameraUp = CameraUp;
                    CameraUp = new Havok.Vdb.Vector(
                        CameraUpOnXAxis ? -cameraUp.X : 0,
                        CameraUpOnYAxis ? -cameraUp.Y : 0,
                        CameraUpOnZAxis ? -cameraUp.Z : 0);
                }
            }
        }

        public bool CameraLeftHandedCoordinateSystem
        {
            get { return (bool)GetCameraPropertyValueBackend("Left Handed Coordinate System", false); }
            set { SetCameraPropertyValueBackend("Left Handed Coordinate System", value); NotifyPropertyChanged(); }
        }
        internal void ToggleCoordinateSystem(object o)
        {
            CameraLeftHandedCoordinateSystem = !CameraLeftHandedCoordinateSystem;
        }

        public double CameraLookSensitivity
        {
            get { return (double)GetCameraPropertyValueBackend("Look Sensitivity", 0.0); }
            set { SetCameraPropertyValueBackend("Look Sensitivity", value); NotifyPropertyChanged(); }
        }

        #endregion

        #region VdbViewModel Camera Controller Modes

        public string CameraMovementMode
        {
            get
            {
                return _CameraMovementMode;
            }
            set
            {
                if (_CameraMovementMode != value)
                {
                    SetCameraMode(value);
                }
            }
        }
        string _CameraMovementMode;

        public bool CameraModeIsFlyMode
        {
            get { return (_CameraMovementMode == "Fly"); }
            set { SetCameraMode("Fly"); }
        }

        public bool CameraModeIsFlyModeInvertedY
        {
            get { return (_CameraMovementMode == "Fly Inverted Y"); }
            set { SetCameraMode("Fly Inverted Y"); }
        }

        public bool CameraModeIsTrackballModeMax
        {
            get { return (_CameraMovementMode == "Max"); }
            set { SetCameraMode("Max"); }
        }

        public bool CameraModeIsTrackballModeMaya
        {
            get { return (_CameraMovementMode == "Maya"); }
            set { SetCameraMode("Maya"); }
        }

        public bool CameraIsLookingAtPOI
        {
            get { return Client.RenderSurface.CameraFlags.HasFlag(Havok.Vdb.CameraFlags.LookAtPrimaryPointOfInterest); }
            set
            {
                if (CameraIsLookingAtPOI != value)
                {
                    UnlockCamera(null);
                    if (value)
                    {
                        Client.RenderSurface.CameraFlags |= Havok.Vdb.CameraFlags.LookAtPrimaryPointOfInterest;
                        CameraIsLookingAtGeometryCenter = false;
                    }
                    else
                    {
                        Client.RenderSurface.CameraFlags &= ~Havok.Vdb.CameraFlags.LookAtPrimaryPointOfInterest;
                    }
                    VdbSettings.SetValue(nameof(Properties.Settings.VdbCamera_LookAtPrimaryPointOfInterest), value);
                    NotifyPropertyChanged();
                }
            }
        }

        public bool CameraIsLookingAtNothing
        {
            get { return !CameraIsLookingAtPOI && !CameraIsLookingAtGeometryCenter; }
            set
            {
                if( value )
                {
                    CameraIsLookingAtGeometryCenter = false;
                    CameraIsLookingAtPOI = false;
                }
            }
        }

        internal void ToggleCameraIsLookingAtPOI(object o)
        {
            CameraIsLookingAtPOI = !CameraIsLookingAtPOI;
        }

        public bool CameraIsKeepingRelativeToPOI
        {
            get { return Client.RenderSurface.CameraFlags.HasFlag(Havok.Vdb.CameraFlags.KeepRelativeToPrimaryPointOfInterest); }
            set
            {
                if (CameraIsKeepingRelativeToPOI != value)
                {
                    UnlockCamera(null);
                    if (value)
                    {
                        Client.RenderSurface.CameraFlags |= Havok.Vdb.CameraFlags.KeepRelativeToPrimaryPointOfInterest;
                        CameraIsKeepingRelativeToGeometryCenter = false;
                    }
                    else
                    {
                        Client.RenderSurface.CameraFlags &= ~Havok.Vdb.CameraFlags.KeepRelativeToPrimaryPointOfInterest;
                    }
                    VdbSettings.SetValue(nameof(Properties.Settings.VdbCamera_KeepRelativeToPrimaryPointOfInterest), value);
                    NotifyPropertyChanged();
                }
            }
        }
        internal void ToggleCameraIsKeepingRelativeToPOI(object o)
        {
            CameraIsKeepingRelativeToPOI = !CameraIsKeepingRelativeToPOI;
        }

        public bool CameraIsLookingAtGeometryCenter
        {
            get { return Client.RenderSurface.CameraFlags.HasFlag(Havok.Vdb.CameraFlags.LookAtPrimaryGeometryCenter); }
            set
            {
                if (CameraIsLookingAtGeometryCenter != value)
                {
                    UnlockCamera(null);
                    if (value)
                    {
                        Client.RenderSurface.CameraFlags |= Havok.Vdb.CameraFlags.LookAtPrimaryGeometryCenter;
                        CameraIsLookingAtPOI = false;
                    }
                    else
                    {
                        Client.RenderSurface.CameraFlags &= ~Havok.Vdb.CameraFlags.LookAtPrimaryGeometryCenter;
                    }
                    VdbSettings.SetValue(nameof(Properties.Settings.VdbCamera_LookAtGeometryCenter), value);
                    NotifyPropertyChanged();
                }
            }
        }

        internal void ToggleCameraIsLookingAtGeometryCenter(object o)
        {
            CameraIsLookingAtGeometryCenter = !CameraIsLookingAtGeometryCenter;
        }

        public bool CameraIsKeepingRelativeToGeometryCenter
        {
            get { return Client.RenderSurface.CameraFlags.HasFlag(Havok.Vdb.CameraFlags.KeepRelativeToPrimaryGeometryCenter); }
            set
            {
                if (CameraIsKeepingRelativeToGeometryCenter != value)
                {
                    UnlockCamera(null);
                    if (value)
                    {
                        Client.RenderSurface.CameraFlags |= Havok.Vdb.CameraFlags.KeepRelativeToPrimaryGeometryCenter;
                        CameraIsKeepingRelativeToPOI = false;
                    }
                    else
                    {
                        Client.RenderSurface.CameraFlags &= ~Havok.Vdb.CameraFlags.KeepRelativeToPrimaryGeometryCenter;
                    }
                    VdbSettings.SetValue(nameof(Properties.Settings.VdbCamera_KeepRelativeToGeometryCenter), value);
                    NotifyPropertyChanged();
                }
            }
        }
        internal void ToggleCameraIsKeepingRelativeToGeometryCenter(object o)
        {
            CameraIsKeepingRelativeToGeometryCenter = !CameraIsKeepingRelativeToGeometryCenter;
        }

        public bool CameraIsKeepingRelativeToNothing
        {
            get { return !CameraIsKeepingRelativeToGeometryCenter && !CameraIsKeepingRelativeToPOI; }
            set
            {
                if( value )
                {
                    CameraIsKeepingRelativeToPOI = false;
                    CameraIsKeepingRelativeToGeometryCenter = false;
                }
            }
        }

        #endregion

        #region VdbViewModel Camera Views

        internal void DefaultWorldView(object o)
        {
            CameraUp = DisplayWorldUp;
            Client.RenderSurface.FitCameraToTarget(Havok.Vdb.FitCameraTarget.World, Havok.Vdb.FitCameraOperation.UnitCubeFromAndTo);
            SessionCameraDefaultClicks++;
        }

        internal void FitWorld(object o)
        {
            Client.RenderSurface.FitCameraToTarget(Havok.Vdb.FitCameraTarget.World, Havok.Vdb.FitCameraOperation.DefaultTo);
            SessionCameraFitCurrentClicks++;
        }

        internal void DefaultSelectionView(object o)
        {
            Client.RenderSurface.FitCameraToTarget(Havok.Vdb.FitCameraTarget.Selection, Havok.Vdb.FitCameraOperation.DefaultFromAndTo);
            SessionCameraDefaultSelectionClicks++;
        }

        internal void FitSelection(object o)
        {
            Client.RenderSurface.FitCameraToTarget(Havok.Vdb.FitCameraTarget.Selection, Havok.Vdb.FitCameraOperation.DefaultTo);
            SessionCameraFitCurrentSelectionClicks++;
        }

        internal void PositiveXView(object o)
        {
            Client.RenderSurface.FitCameraToTarget(Havok.Vdb.FitCameraTarget.World, Havok.Vdb.FitCameraOperation.PositiveX);
            SessionCameraPositiveXClicks++;
        }

        internal void NegativeXView(object o)
        {
            Client.RenderSurface.FitCameraToTarget(Havok.Vdb.FitCameraTarget.World, Havok.Vdb.FitCameraOperation.NegativeX);
            SessionCameraNegativeXClicks++;
        }

        internal void PositiveYView(object o)
        {
            Client.RenderSurface.FitCameraToTarget(Havok.Vdb.FitCameraTarget.World, Havok.Vdb.FitCameraOperation.PositiveY);
            SessionCameraPositiveYClicks++;
        }

        internal void NegativeYView(object o)
        {
            Client.RenderSurface.FitCameraToTarget(Havok.Vdb.FitCameraTarget.World, Havok.Vdb.FitCameraOperation.NegativeY);
            SessionCameraNegativeYClicks++;
        }
        internal void PositiveZView(object o)
        {
            Client.RenderSurface.FitCameraToTarget(Havok.Vdb.FitCameraTarget.World, Havok.Vdb.FitCameraOperation.PositiveZ);
            SessionCameraPositiveZClicks++;
        }

        internal void NegativeZView(object o)
        {
            Client.RenderSurface.FitCameraToTarget(Havok.Vdb.FitCameraTarget.World, Havok.Vdb.FitCameraOperation.NegativeZ);
            SessionCameraNegativeZClicks++;
        }

        #endregion

        #region VdbViewModel Camera Connection Modes

        public bool CameraConnectModeCameraName
        {
            get { return VdbSettings.GetValue<bool>(nameof(Properties.Settings.VdbCamera_ConnectModeCameraName)); }
            set
            {
                if(CameraConnectModeCameraName != value)
                {
                    VdbSettings.SetValue(nameof(Properties.Settings.VdbCamera_ConnectModeCameraName), value);
                    NotifyPropertyChanged();
                }
            }
        }

        public bool CameraConnectModeDefaultCamera
        {
            get { return VdbSettings.GetValue<bool>(nameof(Properties.Settings.VdbCamera_ConnectModeDefaultCamera)); }
            set
            {
                if (CameraConnectModeDefaultCamera != value)
                {
                    VdbSettings.SetValue(nameof(Properties.Settings.VdbCamera_ConnectModeDefaultCamera), value);
                    NotifyPropertyChanged();
                }
            }
        }

        public bool CameraConnectModeNoCameraChange
        {
            get { return VdbSettings.GetValue<bool>(nameof(Properties.Settings.VdbCamera_ConnectModeNoChange)); }
            set
            {
                if (CameraConnectModeNoCameraChange != value)
                {
                    VdbSettings.SetValue(nameof(Properties.Settings.VdbCamera_ConnectModeNoChange), value);
                    NotifyPropertyChanged();
                }
            }
        }

        #endregion

        #region VdbViewModel Camera Internal Functions

        private Havok.Vdb.Property FindCameraProperty(IReadOnlyCollection<Havok.Vdb.Property> cameraProperties, string propName)
        {
            Havok.Vdb.Property foundProp = null;
            if (cameraProperties != null)
            {
                foreach (Havok.Vdb.Property prop in cameraProperties)
                {
                    if (prop.Name == propName)
                    {
                        foundProp = prop;
                        break;
                    }
                }
                System.Diagnostics.Debug.Assert(foundProp != null, "Could not find property named \"" + propName + "\"");
            }
            return foundProp;
        }

        private object GetCameraPropertyValueBackend(IReadOnlyCollection<Havok.Vdb.Property> cameraProperties, string propName, object defaultVal = null)
        {
            Havok.Vdb.Property prop = FindCameraProperty(cameraProperties, propName);
            return (prop != null) ? prop.Value : defaultVal;
        }
        private object GetCameraPropertyValueBackend(string propName, object defaultVal = null)
        {
            return
                GetCameraPropertyValueBackend(
                    (Client.RenderSurface.RenderCameraController != null) ?
                        Client.RenderSurface.RenderCameraController.Options :
                        null,
                    propName,
                    defaultVal);
        }

        private void SetCameraPropertyValueBackend(IReadOnlyCollection<Havok.Vdb.Property> cameraProperties, string propName, object value)
        {
            if (propName != "Left Handed Coordinate System" && propName != "Perspective Mode")
            {
                UnlockCamera(null);
            }

            Havok.Vdb.Property prop = FindCameraProperty(cameraProperties, propName);
            if (prop != null)
            {
                prop.Value = value;
            }
        }
        private void SetCameraPropertyValueBackend(string propName, object value)
        {
            SetCameraPropertyValueBackend(
                (Client.RenderSurface.RenderCameraController != null) ?
                    Client.RenderSurface.RenderCameraController.Options :
                    null,
                propName,
                value);
        }

        internal void SetCameraMode(string newCameraMode)
        {
            if (_CameraMovementMode == newCameraMode)
            {
                return;
            }

            // Set first
            _CameraMovementMode = newCameraMode;

            // Now check for inverted, which in the back end still uses "fly" mode
            if (CameraModeIsFlyModeInvertedY)
            {
                newCameraMode = "Fly";
            }

            // Find camera in list of controllers
            if (Client.RenderSurface.CameraControllers != null)
            {
                foreach (Havok.Vdb.CameraController cameraController in Client.RenderSurface.CameraControllers)
                {
                    if (cameraController.Name == newCameraMode)
                    {
                        // Grab current controller properties.
                        Havok.Vdb.Vector upAxis = CameraUp;
                        double nearPlane = CameraNearPlane;
                        double farPlane = CameraFarPlane;
                        double fov = CameraFieldOfView;
                        bool isPerspectiveMode = CameraPerspectiveMode;
                        bool isLeftHanded = CameraLeftHandedCoordinateSystem;
                        double lookSensitivity = CameraLookSensitivity;

                        // Set the new controller
                        Client.RenderSurface.RenderCameraController = cameraController;

                        // Set controller properties from previous controller
                        {
                            SetCameraPropertyValueBackend(cameraController.Options, "Invert Y", CameraModeIsFlyModeInvertedY);
                            SetCameraPropertyValueBackend(cameraController.Options, "Up Axis", upAxis);
                            SetCameraPropertyValueBackend(cameraController.Options, "Near Plane", nearPlane);
                            SetCameraPropertyValueBackend(cameraController.Options, "Far Plane", farPlane);
                            SetCameraPropertyValueBackend(cameraController.Options, "Field of View", fov);
                            SetCameraPropertyValueBackend(cameraController.Options, "Perspective Mode", isPerspectiveMode);
                            SetCameraPropertyValueBackend(cameraController.Options, "Left Handed Coordinate System", isLeftHanded);
                            SetCameraPropertyValueBackend(cameraController.Options, "Look Sensitivity", lookSensitivity);
                        }

                        // Listen to any back-end changes
                        if (_CurrentCameraController != null)
                        {
                            foreach (Havok.Vdb.Property prop in _CurrentCameraController.Options)
                            {
                                prop.PropertyChanged -= CameraProperty_PropertyChanged;
                            }
                        }
                        _CurrentCameraController = cameraController;
                        if (_CurrentCameraController != null)
                        {
                            foreach (Havok.Vdb.Property prop in _CurrentCameraController.Options)
                            {
                                prop.PropertyChanged += CameraProperty_PropertyChanged;
                            }
                        }

                        break;
                    }
                }

                VdbSettings.SetValue(nameof(Properties.Settings.VdbCamera_Controller), _CameraMovementMode);
            }

            UnlockCamera(null);
            NotifyPropertyChanged(nameof(CameraMovementMode));
            NotifyPropertyChanged(nameof(CameraModeIsFlyMode));
            NotifyPropertyChanged(nameof(CameraModeIsFlyModeInvertedY));
            NotifyPropertyChanged(nameof(CameraModeIsTrackballModeMax));
            NotifyPropertyChanged(nameof(CameraModeIsTrackballModeMaya));
        }
        private Havok.Vdb.CameraController _CurrentCameraController;

        private void CameraProperty_PropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            object propValue = GetCameraPropertyValueBackend(e.PropertyName);
            switch (e.PropertyName)
            {
                case "Up Axis":
                {
                    try
                    {
                        Havok.Vdb.Vector upVector = (Havok.Vdb.Vector)propValue;

                        // Find dominant axis
                        int axis = 0;
                        {
                            float[] abs = {
                                Math.Abs(upVector.X),
                                Math.Abs(upVector.Y),
                                Math.Abs(upVector.Z) };

                            float max = 0;
                            for (int i = 0; i < abs.Length; i++)
                            {
                                if (abs[i] > max)
                                {
                                    max = abs[i];
                                    axis = i;
                                }
                            }

                            // Default to y
                            if (max == 0)
                            {
                                upVector.Y = 1.0f;
                                axis = 1;
                            }
                        }

                        // Clamp vector to that axis with proper sign
                        Havok.Vdb.Vector clampedUpVector = new Havok.Vdb.Vector(
                            Convert.ToSingle(axis == 0) * Math.Sign(upVector.X),
                            Convert.ToSingle(axis == 1) * Math.Sign(upVector.Y),
                            Convert.ToSingle(axis == 2) * Math.Sign(upVector.Z));

                        // We only work with clamped values, send this back to the backend
                        if (!clampedUpVector.Equals(upVector))
                        {
                            SetCameraPropertyValueBackend("Up Axis", clampedUpVector);
                        }
                        // Else we have our clamped value in the backend
                        else
                        {
                            int[] upArray = new int[] { (int)clampedUpVector.X, (int)clampedUpVector.Y, (int)clampedUpVector.Z };
                            VdbSettings.SetValue(nameof(Properties.Settings.VdbCamera_Up), upArray);

                            NotifyPropertyChanged(nameof(CameraUp));
                            NotifyPropertyChanged(nameof(CameraUpOnXAxis));
                            NotifyPropertyChanged(nameof(CameraUpOnYAxis));
                            NotifyPropertyChanged(nameof(CameraUpOnZAxis));
                            NotifyPropertyChanged(nameof(CameraUpIsNegated));
                        }
                    }
                    catch
                    {
                        System.Diagnostics.Debug.Assert(false, "Exception while computing camera up vector");
                    }
                    break;
                }
                case "Near Plane":
                {
                    VdbSettings.SetValue(nameof(Properties.Settings.VdbCamera_NearPlane), propValue);
                    NotifyPropertyChanged(nameof(CameraNearPlane));
                    break;
                }
                case "Far Plane":
                {
                    VdbSettings.SetValue(nameof(Properties.Settings.VdbCamera_FarPlane), propValue);
                    NotifyPropertyChanged(nameof(CameraFarPlane));
                    break;
                }
                case "Field of View":
                {
                    VdbSettings.SetValue(nameof(Properties.Settings.VdbCamera_FieldOfView), propValue);
                    NotifyPropertyChanged(nameof(CameraFieldOfView));
                    break;
                }
                case "Perspective Mode":
                {
                    VdbSettings.SetValue(nameof(Properties.Settings.VdbCamera_PerspectiveMode), propValue);
                    NotifyPropertyChanged(nameof(CameraPerspectiveMode));
                    break;
                }
                case "Left Handed Coordinate System":
                {
                    VdbSettings.SetValue(nameof(Properties.Settings.VdbCamera_LeftHandedCoordinateSystem), propValue);
                    NotifyPropertyChanged(nameof(CameraLeftHandedCoordinateSystem));
                    break;
                }
                case "Look Sensitivity":
                {
                    VdbSettings.SetValue(nameof(Properties.Settings.VdbCamera_LookSensitivity), propValue);
                    NotifyPropertyChanged(nameof(CameraLookSensitivity));
                    break;
                }
                case "Invert Y":
                {
                    // Currently captured by our *mode* but when it becomes general for all controller, we'll need to add something here.
                    break;
                }
                default:
                {
                    System.Diagnostics.Debug.Assert(false, "Unknown/unexpected property changed");
                    break;
                }
            }
        }

        #endregion

        #endregion

        #region VdbViewModel Text Handling

        void InitializeLogViewers()
        {
            MaxLogLimit = VdbSettings.GetValue<int>(nameof(Properties.Settings.VdbLog_MaxLogLimit)).ToString();

            LocalLogCollection = new ObservableCollection<VdbLogViewModel>();
            RemoteLogCollection = new ObservableCollection<VdbLogViewModel>();
            RemoteLogCollectionView = new ObservableCollection<VdbLogViewModel>();
            LocalLogCollectionView = new ObservableCollection<VdbLogViewModel>();

            RemoteTimer = new System.Windows.Threading.DispatcherTimer(new TimeSpan(0, 0, 0, 0, 50), System.Windows.Threading.DispatcherPriority.DataBind,
                delegate (object sender, EventArgs e)
                {
                    // Update Remote Log unread counts
                    NotifyPropertyChanged(nameof(UnreadRemoteLogCount));
                    NotifyPropertyChanged(nameof(UnreadRemoteLogCountValue));
                    NotifyPropertyChanged(nameof(HasUnreadRemoteLogs));

                    // Update general unread counts
                    NotifyPropertyChanged(nameof(UnreadLogCount));
                    NotifyPropertyChanged(nameof(UnreadLogCountValue));
                    NotifyPropertyChanged(nameof(HasUnreadLogs));

                    RemoteTimer.IsEnabled = false;

                }, Application.Current.Dispatcher);

            LocalTimer = new System.Windows.Threading.DispatcherTimer(new TimeSpan(100), System.Windows.Threading.DispatcherPriority.DataBind,
                delegate (object sender, EventArgs e)
                {
                    // Update Remote Log unread counts
                    NotifyPropertyChanged(nameof(UnreadLocalLogCount));
                    NotifyPropertyChanged(nameof(UnreadLocalLogCountValue));
                    NotifyPropertyChanged(nameof(HasUnreadLocalLogs));

                    // Update general unread counts
                    NotifyPropertyChanged(nameof(UnreadLogCount));
                    NotifyPropertyChanged(nameof(UnreadLogCountValue));
                    NotifyPropertyChanged(nameof(HasUnreadLogs));

                    LocalTimer.IsEnabled = false;

                }, Application.Current.Dispatcher);


            LocalLogCollection.CollectionChanged += delegate (object sender, NotifyCollectionChangedEventArgs e)
            {
                if (e.Action == NotifyCollectionChangedAction.Reset)
                {
                    LocalLogCollectionView.Clear();
                }

                if (e.NewItems != null)
                {
                    foreach (VdbLogViewModel logVm in e.NewItems)
                    {
                        if (IsLogEntryIncluded(logVm))
                        {
                            LocalLogCollectionView.Add(logVm);
                        }
                    }
                }

                if (e.OldItems != null)
                {
                    foreach (VdbLogViewModel logVm in e.OldItems)
                    {
                        int idx = LocalLogCollectionView.IndexOf(logVm);
                        if (idx >= 0)
                        {
                            LocalLogCollectionView.RemoveAt(idx);
                        }
                    }
                }

                m_localLogCount -= (e != null && e.OldItems != null) ? e.OldItems.Count : 0;
                LocalTimer.IsEnabled = true;
            };

            RemoteLogCollection.CollectionChanged += delegate (object sender, NotifyCollectionChangedEventArgs e)
            {
                if( e.Action == NotifyCollectionChangedAction.Reset )
                {
                    RemoteLogCollectionView.Clear();
                }

                if (e.NewItems != null)
                {
                    foreach (VdbLogViewModel logVm in e.NewItems)
                    {
                        if (IsLogEntryIncluded(logVm))
                        {
                            RemoteLogCollectionView.Add(logVm);
                        }
                    }
                }

                if (e.OldItems != null)
                {
                    foreach (VdbLogViewModel logVm in e.OldItems)
                    {
                        int idx = RemoteLogCollectionView.IndexOf(logVm);
                        if (idx >= 0)
                        {
                            RemoteLogCollectionView.RemoveAt(idx);
                        }
                    }
                }

                m_remoteLogCount -= (e != null && e.OldItems != null) ? e.OldItems.Count : 0;
                RemoteTimer.IsEnabled = true;
            };

            StringCollection localFilters = VdbSettings.GetValue<StringCollection>(nameof(Properties.Settings.VdbLog_LocalFiltersEnabled));
            {
                IsLocalWarningSelected = localFilters.Contains(nameof(IsLocalWarningSelected));
                IsLocalInfoSelected = localFilters.Contains(nameof(IsLocalInfoSelected));
                IsLocalErrorSelected = localFilters.Contains(nameof(IsLocalErrorSelected));
            }

            StringCollection remoteFilters = VdbSettings.GetValue<StringCollection>(nameof(Properties.Settings.VdbLog_RemoteFiltersEnabled));
            {
                IsRemoteWarningSelected = remoteFilters.Contains(nameof(IsRemoteWarningSelected));
                IsRemoteInfoSelected = remoteFilters.Contains(nameof(IsRemoteInfoSelected));
                IsRemoteErrorSelected = remoteFilters.Contains(nameof(IsRemoteErrorSelected));
                IsRemoteMessageServerDisabled = remoteFilters.Contains(nameof(IsRemoteMessageServerDisabled));
            }
        }

        private System.Windows.Threading.DispatcherTimer RemoteTimer;
        private System.Windows.Threading.DispatcherTimer LocalTimer;

        public ObservableCollection<VdbLogViewModel> RemoteLogCollection { get; private set; }
        public ObservableCollection<VdbLogViewModel> RemoteLogCollectionView { get; private set; }

        public ObservableCollection<VdbLogViewModel> LocalLogCollection { get; private set; }
        public ObservableCollection<VdbLogViewModel> LocalLogCollectionView { get; private set; }

        public String MaxLogLimit
        {
            get { return _MaxLogLimit.ToString(); }
            set
            {
                if (MaxLogLimit != value)
                {
                    int oldValue = _MaxLogLimit;
                    if( Int32.TryParse(value, out _MaxLogLimit) )
                    {
                        if (_MaxLogLimit < oldValue)
                        {
                            // Shrinking, update collections now
                            while (RemoteLogCollection.Count > _MaxLogLimit)
                            {
                                RemoteLogCollection.RemoveAt(0);
                            }

                            while (LocalLogCollection.Count > _MaxLogLimit)
                            {
                                LocalLogCollection.RemoveAt(0);
                            }
                        }

                        VdbSettings.SetValue(nameof(Properties.Settings.VdbLog_MaxLogLimit), _MaxLogLimit);
                        NotifyPropertyChanged();
                    }
                }
            }
        }
        private int _MaxLogLimit = 0;

        public bool HasUnreadLogs { get { return (UnreadLogCount > 0); } }
        public bool HasUnreadLocalLogs { get { return (UnreadLocalLogCount > 0); } }
        public bool HasUnreadRemoteLogs { get { return (UnreadRemoteLogCount > 0); } }

        private int m_logCount = 0;
        private int m_localLogCount = 0;
        private int m_remoteLogCount = 0;

        public int UnreadLogCount
        {
            get { return UnreadLocalLogCount + UnreadRemoteLogCount; }
            set
            {
                m_logCount = value;
                NotifyPropertyChanged();
                NotifyPropertyChanged(nameof(UnreadLogCountValue));
                NotifyPropertyChanged(nameof(HasUnreadLogs));
                NotifyPropertyChanged(nameof(HasUnreadLocalLogs));
                NotifyPropertyChanged(nameof(HasUnreadRemoteLogs));
            }
        }

        public int UnreadLocalLogCount
        {
            get { return LocalLogCollection.Count - m_localLogCount; }
            set
            {
                m_localLogCount = value;
                NotifyPropertyChanged();
                NotifyPropertyChanged(nameof(UnreadLocalLogCountValue));
                NotifyPropertyChanged(nameof(HasUnreadLogs));
                NotifyPropertyChanged(nameof(HasUnreadLocalLogs));
            }
        }

        public int UnreadRemoteLogCount
        {
            get { return RemoteLogCollection.Count - m_remoteLogCount; }
            set
            {
                m_remoteLogCount = value;
                NotifyPropertyChanged();
                NotifyPropertyChanged(nameof(UnreadRemoteLogCountValue));
                NotifyPropertyChanged(nameof(HasUnreadLogs));
                NotifyPropertyChanged(nameof(HasUnreadRemoteLogs));
            }
        }

        public string UnreadLogCountValue
        {
            get
            {
                if (UnreadLogCount > 99)
                {
                    return "99+";
                }

                return UnreadLogCount.ToString();
            }
        }

        public string UnreadLocalLogCountValue
        {
            get
            {
                if (UnreadLocalLogCount > 99)
                {
                    return "99+";
                }

                return UnreadLocalLogCount.ToString();
            }
        }

        public string UnreadRemoteLogCountValue
        {
            get
            {
                if (UnreadRemoteLogCount > 99)
                {
                    return "99+";
                }

                return UnreadRemoteLogCount.ToString();
            }
        }

        private void NotifyLocalFiltersChanged()
        {
            LocalLogCollectionView.Clear();
            foreach (VdbLogViewModel logVm in LocalLogCollection)
            {
                if (IsLogEntryIncluded(logVm))
                {
                    LocalLogCollectionView.Add(logVm);
                }
            }

            NotifyPropertyChanged(nameof(UnreadLocalLogCount));
            NotifyPropertyChanged(nameof(UnreadLocalLogCountValue));
            NotifyPropertyChanged(nameof(HasUnreadLocalLogs));

            StringCollection localFilters = new StringCollection();
            {
                if (IsLocalWarningSelected) localFilters.Add(nameof(IsLocalWarningSelected));
                if (IsLocalInfoSelected) localFilters.Add(nameof(IsLocalInfoSelected));
                if (IsLocalErrorSelected) localFilters.Add(nameof(IsLocalErrorSelected));
                VdbSettings.SetValue(nameof(Properties.Settings.VdbLog_LocalFiltersEnabled), localFilters);
            }
        }

        private void NotifyRemoteFiltersChanged()
        {
            RemoteLogCollectionView.Clear();
            foreach (VdbLogViewModel logVm in RemoteLogCollection)
            {
                if (IsLogEntryIncluded(logVm))
                {
                    RemoteLogCollectionView.Add(logVm);
                }
            }

            NotifyPropertyChanged(nameof(UnreadRemoteLogCount));
            NotifyPropertyChanged(nameof(UnreadRemoteLogCountValue));
            NotifyPropertyChanged(nameof(HasUnreadRemoteLogs));

            StringCollection remoteFilters = new StringCollection();
            {
                if (IsRemoteWarningSelected) remoteFilters.Add(nameof(IsRemoteWarningSelected));
                if (IsRemoteInfoSelected) remoteFilters.Add(nameof(IsRemoteInfoSelected));
                if (IsRemoteErrorSelected) remoteFilters.Add(nameof(IsRemoteErrorSelected));
                if (IsRemoteMessageServerDisabled) remoteFilters.Add(nameof(IsRemoteMessageServerDisabled));
                VdbSettings.SetValue(nameof(Properties.Settings.VdbLog_RemoteFiltersEnabled), remoteFilters);
            }
        }

        private bool _IsLocalErrorSelected = true;
        public bool IsLocalErrorSelected
        {
            get { return _IsLocalErrorSelected; }
            set { _IsLocalErrorSelected = value; NotifyPropertyChanged(); NotifyLocalFiltersChanged(); }
        }

        private bool _IsLocalWarningSelected = true;
        public bool IsLocalWarningSelected
        {
            get { return _IsLocalWarningSelected; }
            set { _IsLocalWarningSelected = value; NotifyPropertyChanged(); NotifyLocalFiltersChanged(); }
        }

        private bool _IsLocalInfoSelected = true;
        public bool IsLocalInfoSelected
        {
            get { return _IsLocalInfoSelected; }
            set { _IsLocalInfoSelected = value; NotifyPropertyChanged(); NotifyLocalFiltersChanged(); }
        }

        private bool _IsRemoteErrorSelected = true;
        public bool IsRemoteErrorSelected
        {
            get { return _IsRemoteErrorSelected; }
            set { _IsRemoteErrorSelected = value; NotifyPropertyChanged(); NotifyRemoteFiltersChanged(); }
        }

        private bool _IsRemoteWarningSelected = true;
        public bool IsRemoteWarningSelected
        {
            get { return _IsRemoteWarningSelected; }
            set { _IsRemoteWarningSelected = value; NotifyPropertyChanged(); NotifyRemoteFiltersChanged(); }
        }

        private bool _IsRemoteInfoSelected = true;
        public bool IsRemoteInfoSelected
        {
            get { return _IsRemoteInfoSelected; }
            set { _IsRemoteInfoSelected = value; NotifyPropertyChanged(); NotifyRemoteFiltersChanged(); }
        }

        private bool _IsRemoteMessageServerDisabled = true;
        public bool IsRemoteMessageServerDisabled
        {
            get { return _IsRemoteMessageServerDisabled; }
            set { _IsRemoteMessageServerDisabled = value; NotifyPropertyChanged(); NotifyRemoteFiltersChanged(); }
        }

        internal void ClearLogs(object o)
        {
            bool localLog = (bool)o;
            ObservableCollection<VdbLogViewModel> collection = localLog ? LocalLogCollection : RemoteLogCollection;
            collection.Clear();
        }

        internal void CopyLogs(object o)
        {
            bool localLog = (bool)o;
            ObservableCollection<VdbLogViewModel> collection = localLog ? LocalLogCollection : RemoteLogCollection;
            Controls.VdbLogGridWidget.Copy(collection as System.Collections.IEnumerable);
        }

        /// <summary>
        /// Return true if log entry is included in the view, false to filter.
        /// </summary>
        public bool IsLogEntryIncluded(object o)
        {
            VdbLogViewModel logEntry = o as VdbLogViewModel;
            if( logEntry == null )
            {
                return false;
            }

            bool includeEntry = false;
            if( logEntry.LogSource == Havok.Vdb.LogSource.Client )
            {
                bool isInfo = IsLocalInfoSelected && (
                    (logEntry.LogLevelNative == Havok.Vdb.LogLevel.Debug) ||
                    (logEntry.LogLevelNative == Havok.Vdb.LogLevel.Info) ||
                    (logEntry.LogLevelNative == Havok.Vdb.LogLevel.Dev));

                includeEntry = isInfo ||
                 ((logEntry.LogLevelNative == Havok.Vdb.LogLevel.Warning) && IsLocalWarningSelected) ||
                 ((logEntry.LogLevelNative == Havok.Vdb.LogLevel.Error) && IsLocalErrorSelected);

                if (!string.IsNullOrEmpty(LocalLogFilterText))
                {
                    includeEntry &= (logEntry.LogText.IndexOf(LocalLogFilterText, StringComparison.CurrentCultureIgnoreCase) >= 0) ||
                                    (logEntry.LogTimestamp.IndexOf(LocalLogFilterText, StringComparison.CurrentCultureIgnoreCase) >= 0);
                }
            }
            else
            {
                bool isInfo = IsRemoteInfoSelected && (
                    (logEntry.LogLevelNative == Havok.Vdb.LogLevel.Debug) ||
                    (logEntry.LogLevelNative == Havok.Vdb.LogLevel.Info) ||
                    (logEntry.LogLevelNative == Havok.Vdb.LogLevel.Dev) );

                includeEntry = (isInfo ||
                 ((logEntry.LogLevelNative == Havok.Vdb.LogLevel.Warning) && IsRemoteWarningSelected) ||
                 ((logEntry.LogLevelNative == Havok.Vdb.LogLevel.Error) && IsRemoteErrorSelected)) &&
                 (!logEntry.IsServerDisabled || (logEntry.IsServerDisabled && IsRemoteMessageServerDisabled));

                if (!string.IsNullOrEmpty(RemoteLogFilterText))
                {
                    includeEntry &= (logEntry.LogText.IndexOf(RemoteLogFilterText, StringComparison.CurrentCultureIgnoreCase) >= 0) ||
                                    (logEntry.LogTimestamp.IndexOf(RemoteLogFilterText, StringComparison.CurrentCultureIgnoreCase) >= 0);
                }
            }

            return includeEntry;
        }

        public bool IsLogShown
        {
            get
            {
                return VdbSettings.GetValue<bool>(nameof(Properties.Settings.VdbUi_LogShown));
            }

            set
            {
                VdbSettings.SetValue(nameof(Properties.Settings.VdbUi_LogShown), value);
                NotifyPropertyChanged();
            }
        }
        internal void ToggleLogPanelShown(object o)
        {
            IsLogShown = !IsLogShown;
        }

        internal void GoToLogEntryFrame(object o = null)
        {
            VdbLogViewModel logVm = o as VdbLogViewModel;
            if ((logVm != null))
            {
                // Check to see if the log frame is still in the buffer
                if (logVm.AbsoluteFrame > MinFrame && logVm.AbsoluteFrame < MaxFrame)
                {
                    // go to that frame
                    Pause();
                    CurrentFrame = logVm.AbsoluteFrame;
                }
            }
        }

        public string RemoteLogFilterText
        {
            get
            {
                return _RemoteLogFilterText;
            }

            set
            {
                if(_RemoteLogFilterText != value )
                {
                    _RemoteLogFilterText = value;
                    NotifyRemoteFiltersChanged();
                    NotifyPropertyChanged();
                }
            }
        }
        private string _RemoteLogFilterText;

        public string LocalLogFilterText
        {
            get
            {
                return _LocalLogFilterText;
            }

            set
            {
                if (_LocalLogFilterText != value)
                {
                    _LocalLogFilterText = value;
                    NotifyLocalFiltersChanged();
                    NotifyPropertyChanged();
                }
            }
        }
        private string _LocalLogFilterText;

        #endregion

        #region VdbViewModel Error Handling
        
        public int Error { get; }
        public int ErrorId { get; }
        public String ErrorMessage { get; }
        #endregion

        #region VdbViewModel Rendering

        #region VdbViewModel Debug
        public delegate string DebugDisplayToString();
        public ObservableCollection<DebugDisplayObjects> DebugViewportDisplayObjects { get; private set; }
        public class DebugDisplayObjects : ViewModelNotifyPropertyChanged
        {
            public DebugDisplayObjects(DebugDisplayToString func) { ToStringFunc = func; }
            private DebugDisplayToString ToStringFunc;

            public void Update() { NotifyPropertyChanged(nameof(ObservableString)); }

            public string ObservableString
            {
                get { return ToStringFunc(); }
                set { NotifyPropertyChanged(); }
            }
        }
        public void UpdateDebugViewportDisplayObjs()
        {
            foreach(DebugDisplayObjects o in DebugViewportDisplayObjects)
            {
                o.Update();
            }
        }
        #endregion

        public bool HasRenderObjects { get { return Client.RenderSurface.HasRenderObjects; } }
        public ObservableCollection<VdbWidgetViewModel> RegisteredWidgetsCollection { get; private set; }
        public VdbWidgetViewModel LineGraphWidget { get; private set; }
        public VdbWidgetViewModel DeprecatedBarGraphWidget { get; private set; }
        public VdbWidgetViewModel BarGraphWidget { get; private set; }
        public VdbWidgetViewModel StatusInfoWidget { get; private set; }

        public bool IsStatsTextSelected
        {
            get { return _IsStatsTextSelected; }
            set
            {
                if (_IsStatsTextSelected != value)
                {
                    _IsStatsTextSelected = value;
                    NotifyPropertyChanged();
                }
            }
        }
        public bool _IsStatsTextSelected;

        public void SaveSelectedGeometries(object o = null)
        {
            string filePath = DoFileSave(
                Properties.Resources.File_SaveSelectedGeometries,
                " (*.hkt)|*.hkt");
            Client.RenderSurface.SaveSelectedObjects(filePath);
        }

        public void SaveHighlightedGeometries(object o = null)
        {
            string filePath = DoFileSave(
                Properties.Resources.File_SaveHighlightedGeometries,
                " (*.hkt)|*.hkt");
            Client.RenderSurface.SaveHighlightedObjects(filePath);
        }

        public void SaveAllGeometries(object o = null)
        {
            string filePath = DoFileSave(
                Properties.Resources.File_SaveAllGeometries,
                " (*.hkt)|*.hkt");
            Client.RenderSurface.SaveAllObjects(filePath);
        }

        public void SaveGeometry(object o)
        {
            Havok.Vdb.RenderObject ro = (o as Havok.Vdb.RenderObject);
            if (ro != null)
            {
                string filePath = DoFileSave(
                    Properties.Resources.File_SaveGeometry,
                    " (*.hkt)|*.hkt");
                Client.RenderSurface.SaveObject(filePath, ro);
            }
        }

        public void SetDiagnosticWidgets()
        {
            LineGraphWidget = GetWidgetByName("Line Graph");
            DeprecatedBarGraphWidget = GetWidgetByName("Deprecated Bar Graph");
            BarGraphWidget = GetWidgetByName("Bar Graph");
            StatusInfoWidget = GetWidgetByName("Status Info");

            NotifyPropertyChanged(nameof(LineGraphWidget));
            NotifyPropertyChanged(nameof(DeprecatedBarGraphWidget));
            NotifyPropertyChanged(nameof(BarGraphWidget));
            NotifyPropertyChanged(nameof(StatusInfoWidget));
        }

        private VdbWidgetViewModel GetWidgetByName(string name)
        {
            foreach (VdbWidgetViewModel widget in RegisteredWidgetsCollection)
            {
                if (widget.Name == name)
                {
                    return widget;
                }
            }
            return null;
        }

        private void ToggleWidgetByName(string name)
        {
            VdbWidgetViewModel widgetVm = GetWidgetByName(name);
            if (widgetVm != null)
            {
                widgetVm.IsSelected = !widgetVm.IsSelected;
            }
        }

        internal void ToggleGrid(object o)
        {
            ToggleWidgetByName("Grid");
        }

        internal void ToggleUnitAxis(object o)
        {
            ToggleWidgetByName("Orientation");
        }

        internal void ToggleStatBarGraph(object o)
        {
            ToggleWidgetByName("Bar Graph");
        }

        internal void ToggleStatus(object o)
        {
            ToggleWidgetByName("Status");
        }

        public bool RenderWireframe
        {
            get
            {
                return ((Client.RenderSurface.RenderFlags & (Havok.Vdb.RenderFlags.Wireframe)) != 0);
            }
            set
            {
                if (RenderWireframe != value)
                {
                    if (value)
                    {
                        Client.RenderSurface.RenderFlags = (Client.RenderSurface.RenderFlags | Havok.Vdb.RenderFlags.Wireframe);
                    }
                    else
                    {
                        Client.RenderSurface.RenderFlags = (Client.RenderSurface.RenderFlags & ~Havok.Vdb.RenderFlags.Wireframe);
                    }

                    VdbSettings.SetValue(nameof(Properties.Settings.VdbRender_Wireframe), value);
                    NotifyPropertyChanged();

                    SessionWireframeToggles++;
                }
            }
        }

        internal void ToggleWireframe(object o)
        {
            RenderWireframe = !RenderWireframe;
        }

        public bool RenderOutlineFaces
        {
            get
            {
                return ((Client.RenderSurface.RenderFlags & (Havok.Vdb.RenderFlags.OutlineFaces)) != 0);
            }
            set
            {
                if (RenderOutlineFaces != value)
                {
                    if (value)
                    {
                        Client.RenderSurface.RenderFlags = (Client.RenderSurface.RenderFlags | Havok.Vdb.RenderFlags.OutlineFaces);
                    }
                    else
                    {
                        Client.RenderSurface.RenderFlags = (Client.RenderSurface.RenderFlags & ~Havok.Vdb.RenderFlags.OutlineFaces);
                    }

                    VdbSettings.SetValue(nameof(Properties.Settings.VdbRender_OutlineFaces), value);
                    NotifyPropertyChanged();

                    SessionOutlineToggles++;
                }
            }
        }

        internal void ToggleOutlineFaces(object o)
        {
            RenderOutlineFaces = !RenderOutlineFaces;
        }

        public bool RandomizedColors
        {
            get
            {
                return ((Client.RenderSurface.RenderFlags & (Havok.Vdb.RenderFlags.RandomizedColors)) != 0);
            }
            set
            {
                if (RandomizedColors != value)
                {
                    if (value)
                    {
                        Client.RenderSurface.RenderFlags = (Client.RenderSurface.RenderFlags | Havok.Vdb.RenderFlags.RandomizedColors);
                    }
                    else
                    {
                        Client.RenderSurface.RenderFlags = (Client.RenderSurface.RenderFlags & ~Havok.Vdb.RenderFlags.RandomizedColors);
                    }

                    VdbSettings.SetValue(nameof(Properties.Settings.VdbRender_RandomizeColors), value);
                    NotifyPropertyChanged();

                    SessionRandomizeColorsToggles++;
                }
            }
        }

        public bool SelectionVisualization
        {
            get
            {
                return
                    ((Client.RenderSurface.DrawFlags & (Havok.Vdb.DrawFlags.VisualizeDisplaySelection)) != 0) ||
                    ((Client.RenderSurface.RenderFlags & (Havok.Vdb.RenderFlags.VisualizeGeometrySelection)) != 0);
            }
            set
            {
                if (SelectionVisualization != value)
                {
                    if (value)
                    {
                        Client.RenderSurface.DrawFlags = (Client.RenderSurface.DrawFlags | Havok.Vdb.DrawFlags.VisualizeDisplaySelection);
                        Client.RenderSurface.RenderFlags = (Client.RenderSurface.RenderFlags | Havok.Vdb.RenderFlags.VisualizeGeometrySelection);
                    }
                    else
                    {
                        Client.RenderSurface.DrawFlags = (Client.RenderSurface.DrawFlags & ~Havok.Vdb.DrawFlags.VisualizeDisplaySelection);
                        Client.RenderSurface.RenderFlags = (Client.RenderSurface.RenderFlags & ~Havok.Vdb.RenderFlags.VisualizeGeometrySelection);
                    }

                    VdbSettings.SetValue(nameof(Properties.Settings.VdbRender_SelectionVisualization), value);
                    NotifyPropertyChanged();

                    SessionSelectionVisualizationToggles++;
                }
            }
        }

        public bool LockLightToCamera
        {
            get
            {
                return Client.RenderSurface.LockLightToCamera;
            }
            set
            {
                if (LockLightToCamera != value)
                {
                    Client.RenderSurface.LockLightToCamera = value;

                    VdbSettings.SetValue(nameof(Properties.Settings.VdbRender_LockLightToCamera), value);
                    NotifyPropertyChanged();

                    SessionLockLightToCameraToggles++;
                }
            }
        }

        public Color SelectedClearColor
        {
            get
            {
                return Client.RenderSurface.ClearColor;
            }
            set
            {
                if (SelectedClearColor != value)
                {
                    Client.RenderSurface.ClearColor = value;

                    VdbSettings.SetValue(nameof(Properties.Settings.VdbRender_ClearColor), value);
                    NotifyPropertyChanged();

                    SessionClearColorChanges++;
                }
            }
        }

        public enum BackfaceCullingModeOptions
        {
            CullingNone,
            CullingClockwise,
            CullingCounterClockwise
        }

        public BackfaceCullingModeOptions BackfaceCullingMode
        {
            get
            {
                bool cullingOn = ((Client.RenderSurface.RenderFlags & (Havok.Vdb.RenderFlags.BackfaceCulling)) != 0);
                if( cullingOn )
                {
                    bool counterClockwiseCulling = ((Client.RenderSurface.RenderFlags & (Havok.Vdb.RenderFlags.CullmodeCcw)) != 0);
                    return counterClockwiseCulling ? BackfaceCullingModeOptions.CullingCounterClockwise : BackfaceCullingModeOptions.CullingClockwise;
                }
                else
                {
                    return BackfaceCullingModeOptions.CullingNone;
                }
            }
            set
            {
                if (BackfaceCullingMode != value)
                {
                    if (value == BackfaceCullingModeOptions.CullingNone)
                    {
                        Client.RenderSurface.RenderFlags = (Client.RenderSurface.RenderFlags & ~Havok.Vdb.RenderFlags.BackfaceCulling);
                    }
                    else
                    {
                        // turn on backface culling
                        Client.RenderSurface.RenderFlags = (Client.RenderSurface.RenderFlags | Havok.Vdb.RenderFlags.BackfaceCulling);

                        // set the backface culling mode
                        if (value == BackfaceCullingModeOptions.CullingCounterClockwise)
                        {
                            Client.RenderSurface.RenderFlags = (Client.RenderSurface.RenderFlags | Havok.Vdb.RenderFlags.CullmodeCcw);
                        }
                        else
                        {
                            Client.RenderSurface.RenderFlags = (Client.RenderSurface.RenderFlags & ~Havok.Vdb.RenderFlags.CullmodeCcw);
                        }
                    }

                    VdbSettings.SetValue(nameof(Properties.Settings.VdbRender_BackfaceCullingMode), value.ToString());
                    NotifyPropertyChanged();

                    SessionBackfaceCullingChanges++;
                }
            }
        }

        internal void ToggleRandomizedColors(object o)
        {
            RandomizedColors = !RandomizedColors;
        }

        internal void ToggleSelectionVisualization(object o)
        {
            SelectionVisualization = !SelectionVisualization;
        }

        internal void ToggleLockLightToCamera(object o)
        {
            LockLightToCamera = !LockLightToCamera;
        }

        internal void ResetLighting(object o)
        {
            LockLightToCamera = false;
            Client.RenderSurface.ResetLighting();
        }

        public ObservableCollection<Color> ObjectColors
        {
            get { return _ObjectColors; }
        }
        ObservableCollection<Color> _ObjectColors = new ObservableCollection<Color>();
        #endregion

        #region VdbViewModel Viewer Information

        public class ViewerInfos
        {
            public ViewerInfos() { ViewerInformation = new List<VdbModel.ViewerInformation>(); }
            public List<VdbModel.ViewerInformation> ViewerInformation { get; set; }
        }

        public void InitializeViewerInfo()
        {
            string viewerDatabasePath = hkResourceUtils.GetPackedResourcePath() + "Resources/Viewers/viewerInformation.json";
            string json = hkResourceUtils.ReadPackResource(viewerDatabasePath);
            ViewerInfos infos = hkResourceUtils.ParseJsonString<ViewerInfos>(json);

            foreach (VdbModel.ViewerInformation info in infos.ViewerInformation)
            {
                // Try given path
                if (!string.IsNullOrEmpty(info.ImagePath) && !System.IO.File.Exists(info.ImagePath))
                {
                    // Try resource path
                    info.ImagePath = hkResourceUtils.GetPackedResourcePath() + info.ImagePath;
                }
                else
                {
                    info.ImagePath = null;
                }

                if (info.LongDescription.StartsWith("@"))
                {
                    string resourcePath = info.LongDescription.Substring(1);
                    object resourceObject = HavokVisualDebugger.Properties.Resources.ResourceManager.GetObject(resourcePath);
                    if (resourceObject != null)
                    {
                        info.LongDescription = resourceObject.ToString();
                    }
                    else
                    {
                        info.LongDescription = "";
                    }
                }

                if (info.ShortDescription.StartsWith("@"))
                {
                    string resourcePath = info.ShortDescription.Substring(1);
                    object resourceObject = HavokVisualDebugger.Properties.Resources.ResourceManager.GetObject(resourcePath);
                    if (resourceObject != null)
                    {
                        info.ShortDescription = resourceObject.ToString();
                    }
                    else
                    {
                        info.ShortDescription = "";
                    }
                }

                foreach(string viewerName in info.ViewerNames)
                {
                    if (!VdbModel.ViewerInformation.VdbProcessInformationDictionary.ContainsKey(viewerName) && viewerName != "")
                    {
                        VdbModel.ViewerInformation.VdbProcessInformationDictionary.Add(viewerName, info);
                    }
                }
            }
        }

        #endregion

        #region VdbViewModel Object Inspection

        #region VdbViewModel Object Inspection Initialize

        private void InitializeObjectInspection()
        {
            if (VdbObjectQueries == null)
            {
                VdbObjectQueries = new ObservableCollection<string>();
            }
            else
            {
                VdbObjectQueries.CollectionChanged -= VdbObjectQueries_CollectionChanged;
                VdbObjectQueries.Clear();
            }
            hkCollectionUtils.InitializeLimitedStringCollectionFromSettings(VdbObjectQueries, nameof(Properties.Settings.VdbFilterQueries), MaxFilterEntries);
            VdbObjectQueries.CollectionChanged += VdbObjectQueries_CollectionChanged;

            if (CulledVdbObjectModelContainers == null)
            {
                CulledVdbObjectModelContainers = new ObservableCollection<VdbObjectViewModel>();
                CulledVdbObjectModelContainers.CollectionChanged += (s, e) =>
                {
                    // Notify if we went between zero and non-zero
                    if ((CulledVdbObjectModelContainers.Count == 0)
                        || ((e.NewItems != null) && (CulledVdbObjectModelContainers.Count == e.NewItems.Count)))
                    {
                        NotifyPropertyChanged(nameof(IsVdbObjectTreeCulled));
                    }
                };
            }
            else
            {
                CulledVdbObjectModelContainers.Clear();
            }

            if (_VdbObjectTree == null)
            {
                _VdbObjectTree = new VdbObjectMultiParentTree(this);
            }
            else
            {
                (_VdbObjectTree.Root.Children as INotifyCollectionChanged).CollectionChanged -= VdbObjectTree_Root_CollectionChanged;
                _VdbObjectTree.Dispose();
            }
            (_VdbObjectTree.Root.Children as INotifyCollectionChanged).CollectionChanged += VdbObjectTree_Root_CollectionChanged;

            CreateFilteredVdbObjectTreeView();
        }

        private void VdbObjectQueries_CollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            if (e.NewItems != null)
            {
                foreach (string newQuery in e.NewItems)
                {
                    VdbSettings.SetCollectionItemEnabled(Properties.Settings.Default.VdbFilterQueries, newQuery, true);
                }
            }
            else if (e.OldItems != null)
            {
                foreach (string oldQuery in e.OldItems)
                {
                    VdbSettings.SetCollectionItemEnabled(Properties.Settings.Default.VdbFilterQueries, oldQuery, false);
                }
            }
        }

        private void VdbObjectTree_Root_CollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            // Notify if we went between zero and non-zero
            if ((_VdbObjectTree.Root.Children.Count == 0)
                || ((e.NewItems != null) && (_VdbObjectTree.Root.Children.Count == e.NewItems.Count)))
            {
                NotifyPropertyChanged(nameof(IsVdbObjectTreeEmpty));
            }
        }

        #endregion

        #region VdbViewModel Object Inspection VdbObjectQuery

        public string VdbObjectQuery
        {
            get
            {
                return _VdbObjectQuery;
            }
            set
            {
                string sanitizedValue = (value != null) ? value.Trim() : "";
                if (_VdbObjectQuery != sanitizedValue)
                {
                    _VdbObjectQuery = sanitizedValue;

                    // Add the new query to the list of used queries
                    hkCollectionUtils.InsertIntoLimitedStringCollection(VdbObjectQueries, value, MaxFilterEntries);

                    // Create search filters from user input
                    {
                        Filters.Clear();
                        SearchFilter.GenerateSearchFilters(_VdbObjectQuery, Filters);
                    }

                    // Pause if we have filters (see VDB-1018)
                    if (Filters.Count > 0)
                    {
                        Pause(null);
                    }

                    CreateFilteredVdbObjectTreeView();
                    NotifyPropertyChanged(nameof(VdbObjectQueries));
                    NotifyPropertyChanged();
                }
            }
        }
        private string _VdbObjectQuery = "";
        public ObservableCollection<string> VdbObjectQueries { get; set; }
        public ReadOnlyCollection<VdbObjectViewModel> VdbObjectQueryResults { get; private set; }
        internal List<SearchFilter> Filters = new List<SearchFilter>();

        internal void DeleteQueryItem(object o = null)
        {
            string queryToDelete = o as string;
            if (VdbObjectQueries.Contains(queryToDelete))
            {
                VdbObjectQueries.Remove(queryToDelete);
                NotifyPropertyChanged(nameof(VdbObjectQueries));
            }
        }

        #endregion

        #region VdbViewModel Object Inspection VdbObjectTree

        public VdbObjectViewModel VdbObjectTreeView
        {
            get { return _VdbObjectTreeView; }
            private set
            {
                if (_VdbObjectTreeView != value)
                {
                    if (_VdbObjectTreeView != null) (_VdbObjectTreeView.Children as INotifyCollectionChanged).CollectionChanged -= VdbObjectTreeView_CollectionChanged;
                    _VdbObjectTreeView = value;
                    if (_VdbObjectTreeView != null) (_VdbObjectTreeView.Children as INotifyCollectionChanged).CollectionChanged += VdbObjectTreeView_CollectionChanged;
                    NotifyPropertyChanged(nameof(IsVdbObjectTreeViewEmpty));
                    NotifyPropertyChanged();
                }
            }
        }
        private VdbObjectViewModel _VdbObjectTreeView;
        private void VdbObjectTreeView_CollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            // Notify if we went between zero and non-zero
            if ((_VdbObjectTreeView.Children.Count == 0)
                || ((e.NewItems != null) && (_VdbObjectTreeView.Children.Count == e.NewItems.Count)))
            {
                NotifyPropertyChanged(nameof(IsVdbObjectTreeViewEmpty));
            }
        }
        public bool IsVdbObjectTreeCulled { get { return CulledVdbObjectModelContainers.Count > 0; } }
        public bool IsVdbObjectTreeEmpty { get { return (VdbObjectTree == null) || (VdbObjectTree.Root.Children.Count == 0); } }
        public bool IsVdbObjectTreeViewEmpty { get { return (VdbObjectTreeView == null) || (VdbObjectTreeView.Children.Count == 0); } }
        public bool IsVdbObjectTreeMenuShown
        {
            get
            {
                return _IsVdbObjectTreeMenuShown;
            }
            set
            {
                if (_IsVdbObjectTreeMenuShown != value)
                {
                    _IsVdbObjectTreeMenuShown = value;
                    NotifyPropertyChanged();
                }
            }
        }
        private bool _IsVdbObjectTreeMenuShown;
        public bool IsVdbObjectTreeViewShown
        {
            get
            {
                return VdbSettings.GetValue<bool>(nameof(Properties.Settings.VdbUi_InspectionShown));
            }
            set
            {
                VdbSettings.SetValue(nameof(Properties.Settings.VdbUi_InspectionShown), value);
                NotifyPropertyChanged();
            }
        }
        internal void ToggleVdbObjectTreeViewShown(object o)
        {
            IsVdbObjectTreeViewShown = !IsVdbObjectTreeViewShown;
        }

        public void CreateFilteredVdbObjectTreeView()
        {
            SessionNumTreeFilterOperations++;

            
            
            
            
            // Generate our new tree view.
            foreach (VdbObjectViewModel ovm in VdbObjectViewModels.Values) ovm.Dispose();
            VdbObjectViewModels.Clear();
            CulledVdbObjectModelContainers.Clear();
            List<VdbObjectViewModel> matchingTreeObjectVms;
            // Note: MaxVdbObjectModelContainerChildren can be adjusted if a select object isn't in the visible tree
            MaxVdbObjectModelContainerChildren = 1000;
            VdbObjectTreeView = AddToVdbObjectTreeView(VdbObjectTree.Root, out matchingTreeObjectVms);
            
            
            VdbObjectQueryResults = matchingTreeObjectVms.AsReadOnly();

            // Update our highlights/selections
            {
                // Clear all
                Selection.ClearHighlightedAndSelected();

                // If we have a filtered tree view, then highlight all objects in the tree view.
                if ((VdbObjectTreeView != null) && VdbObjectTreeView.IsFiltered)
                {
                    Selection.HighlightedObjects.AddRange(VdbObjectViewModels.Values);
                }

                // Set the selected node to be the first.
                Selection.SelectedObject =
                    (matchingTreeObjectVms.Count > 0) ?
                    matchingTreeObjectVms[0] :
                    null;
            }
        }

        internal VdbObjectViewModel AddToVdbObjectTreeView(VdbObjectModel objectTemplateM, out List<VdbObjectViewModel> matchingTreeObjectVms)
        {
            
            
            
            VdbObjectViewModel treeObjectVm = objectTemplateM.CreateTreeView(
                this,
                Filters,
                VdbObjectViewModels,
                out matchingTreeObjectVms);
            return treeObjectVm;
        }
        internal bool RemoveFromVdbObjectTreeView(VdbObjectModel objectTemplateM)
        {
            bool removed = false;
            VdbObjectViewModel treeObjectVm;
            if (VdbObjectViewModels.TryGetValue(objectTemplateM, out treeObjectVm))
            {
                VdbObjectViewModels.Remove(objectTemplateM);
                treeObjectVm.Dispose();
                foreach (VdbObjectModel childTemplateM in objectTemplateM.Children)
                {
                    RemoveFromVdbObjectTreeView(childTemplateM);
                }
                removed = true;
            }
            return removed;
        }
        internal void QueryVdbObjectTree(object o = null)
        {
            CreateFilteredVdbObjectTreeView();
        }
        internal void CollapseVdbObjectTree(object o)
        {
            VdbObjectTreeView.CollapseAll();
            SessionInspectionCollapseClicks++;
        }
        internal void GoToNextTreeObjectCommand(object o)
        {
            GoToTreeObject(true);
        }
        internal void GoToPrevTreeObjectCommand(object o)
        {
            GoToTreeObject(false);
        }
        internal void GoToTreeObject(bool next)
        {
            VdbObjectViewModel selectedObjectVm = Selection.SelectedObject;
            if (selectedObjectVm != null && selectedObjectVm.Parent != null)
            {
                // Find child index
                int indexInParent = -1;
                for (int i = 0; i < selectedObjectVm.Parent.Children.Count; i++)
                {
                    if (selectedObjectVm == selectedObjectVm.Parent.Children[i])
                    {
                        indexInParent = i;
                        break;
                    }
                }

                if ((indexInParent >= 0) && (selectedObjectVm.Parent.Children.Count > 1))
                {
                    if (next)
                    {
                        if (((indexInParent + 1) >= selectedObjectVm.Parent.Children.Count) && (selectedObjectVm.Parent.Children.Count > 0))
                        {
                            Selection.SelectedObject = selectedObjectVm.Parent.Children[0];
                        }
                        else
                        {
                            Selection.SelectedObject = selectedObjectVm.Parent.Children[indexInParent + 1];
                        }
                    }
                    else
                    {
                        if (((indexInParent - 1) >= 0) && (selectedObjectVm.Parent.Children.Count > 0))
                        {
                            Selection.SelectedObject = selectedObjectVm.Parent.Children[indexInParent - 1];
                        }
                        else
                        {
                            Selection.SelectedObject = selectedObjectVm.Parent.Children[selectedObjectVm.Parent.Children.Count - 1];
                        }
                    }
                }
            }
        }

        internal VdbObjectMultiParentTree VdbObjectTree { get { return _VdbObjectTree; } }
        private VdbObjectMultiParentTree _VdbObjectTree;
        internal Dictionary<VdbObjectModel, VdbObjectViewModel> VdbObjectViewModels = new Dictionary<VdbObjectModel, VdbObjectViewModel>();
        internal ObservableCollection<VdbObjectViewModel> CulledVdbObjectModelContainers;
        internal int MaxVdbObjectModelContainerChildren;
        internal int MaxFilterEntries = 20;
        public VdbObjectPropertiesOptions ObjectPropertyOptions { get; private set; }

        #endregion

        #region VdbViewModel Object Inspection VdbObjectFilters

        public VdbFixedFilterLayerViewModel VdbObjectFilterBaseLayer
        {
            get { return _VdbObjectFilterBaseLayer; }
            private set
            {
                if (_VdbObjectFilterBaseLayer != value)
                {
                    _VdbObjectFilterBaseLayer = value;
                    NotifyPropertyChanged();
                }
            }
        }
        private VdbFixedFilterLayerViewModel _VdbObjectFilterBaseLayer;
        public VdbMutableFilterLayerCollection VdbObjectFilterLayers { get; private set; }
        public bool IsSetModeOnAllLayersPanelShown
        {
            get
            {
                return _IsSetModeOnAllLayersPanelShown;
            }
            set
            {
                if (_IsSetModeOnAllLayersPanelShown != value)
                {
                    _IsSetModeOnAllLayersPanelShown = value;
                    NotifyPropertyChanged();
                }
            }
        }
        private bool _IsSetModeOnAllLayersPanelShown = false;
        public bool IsFilterLayerPanelShown
        {
            get
            {
                return VdbSettings.GetValue<bool>(nameof(Properties.Settings.VdbUi_FilterLayerPanelShown));
            }
            set
            {
                if (IsFilterLayerPanelShown != value)
                {
                    VdbSettings.SetValue(nameof(Properties.Settings.VdbUi_FilterLayerPanelShown), value);
                    NotifyPropertyChanged();
                }
            }
        }
        public bool IsFilterLayerMenuShown
        {
            get
            {
                return _IsFilterLayerMenuShown;
            }
            set
            {
                if (_IsFilterLayerMenuShown != value)
                {
                    _IsFilterLayerMenuShown = value;
                    NotifyPropertyChanged();
                }
            }
        }
        private bool _IsFilterLayerMenuShown;

        #endregion

        #endregion

        #region VdbViewModel Runtime Info

        public Havok.Vdb.Capabilities Capabilities { get { return Client.GetSubInterop(nameof(Havok.Vdb.Client.SetupHandler)).GetProperty<Havok.Vdb.SetupHandlerEventArgs>(nameof(Havok.Vdb.SetupHandler.SetupInfo)).Capabilities; } }
        public bool CapabilitiesCanSave {  get { return Capabilities.HasFlag(Havok.Vdb.Capabilities.SaveCommandInputBuffer); } }

        public int ClientProtocol { get { Havok.Vdb.SetupInfo info = ClientInfo; return (info != null) ? info.Protocol : -1; } }
        public int ClientMinProtocol { get { Havok.Vdb.SetupInfo info = ClientInfo; return (info != null) ? info.MinProtocol : -1; } }
        public int ClientSdkVersion { get { Havok.Vdb.SetupInfo info = ClientInfo; return (info != null) ? info.SdkVersion : -1; } }
        public string ClientPlatformString { get { Havok.Vdb.SetupInfo info = ClientInfo; return (info != null) ? info.PlatformString : ""; } }
        public string ClientCompilerString { get { Havok.Vdb.SetupInfo info = ClientInfo; return (info != null) ? info.CompilerString : ""; } }
        public int ClientPointerSize { get { Havok.Vdb.SetupInfo info = ClientInfo; return (info != null) ? info.PointerSize : -1; } }
        public bool ClientIsLittleEndian { get { Havok.Vdb.SetupInfo info = ClientInfo; return (info != null) ? info.IsLittleEndian : false; } }
        public int ServerProtocol { get { Havok.Vdb.SetupInfo info = ServerInfo; return (info != null) ? info.Protocol : -1; } }
        public int ServerMinProtocol { get { Havok.Vdb.SetupInfo info = ServerInfo; return (info != null) ? info.MinProtocol : -1; } }
        public int ServerSdkVersion { get { Havok.Vdb.SetupInfo info = ServerInfo; return (info != null) ? info.SdkVersion : -1; } }
        public string ServerPlatformString { get { Havok.Vdb.SetupInfo info = ServerInfo; return (info != null) ? info.PlatformString : ""; } }
        public string ServerCompilerString { get { Havok.Vdb.SetupInfo info = ServerInfo; return (info != null) ? info.CompilerString : ""; } }
        public int ServerPointerSize { get { Havok.Vdb.SetupInfo info = ServerInfo; return (info != null) ? info.PointerSize : -1; } }
        public bool ServerIsLittleEndian { get { Havok.Vdb.SetupInfo info = ServerInfo; return (info != null) ? info.IsLittleEndian : false; } }

		public string BuildDate
        {
            get
            {
                DateTime buildDate = Assembly.GetExecutingAssembly().GetLinkerTime();
                return buildDate.ToShortDateString() + "  " + buildDate.ToShortTimeString();
            }
        }

        public int PointReleaseChangeList { get { return hkConstants.PointReleaseChangeList; } }
        public bool IsPointRelease { get { return PointReleaseChangeList > 0; } }
        public string ClientVersion { get { return ClientSdkVersion.ToString() + (IsPointRelease ? " CL" + PointReleaseChangeList.ToString() : ""); } }

        public bool IsPointReleaseWarningShown
        {
            get
            {
                int dismissedChangeList = VdbSettings.GetValue<int>(nameof(Properties.Settings.VdbSettings_PointReleaseWarningDismissedChangelist));
                return (dismissedChangeList != PointReleaseChangeList);
            }

            set
            {
                if (IsPointReleaseWarningShown != value)
                {
                    VdbSettings.SetValue(nameof(Properties.Settings.VdbSettings_PointReleaseWarningDismissedChangelist), PointReleaseChangeList);
                    NotifyPropertyChanged();
                }
            }
        }

        #endregion

        #region VdbViewModel Private Methods and Properties
        public bool IsDebugViewEnabled
        {
            get { return _DebugViewEnabled; }
            set
            {
                if (_DebugViewEnabled != value)
                {
                    _DebugViewEnabled = value;
                    {
                        // Note: we don't go through normal property changed mechanics
                        // as that would require all nodes to get hooked into the view model property
                        // changed pipeline, which would be very expensive.
                        foreach (VdbObjectViewModel ovm in VdbObjectViewModels.Values) ovm.UpdateObjectProperties();
                    }
                    NotifyPropertyChanged();
                }
            }
        }
        private bool _DebugViewEnabled;
        public bool PerformDebugActionsEnabled;
        void NotifyConnectionPropertiesChanged()
        {
            NotifyPropertyChanged(nameof(ConnectedSource));
            NotifyPropertyChanged(nameof(ConnectedState));
            NotifyPropertyChanged(nameof(IsConnecting));
            NotifyPropertyChanged(nameof(IsConnected));
            NotifyPropertyChanged(nameof(IsDisconnected));
            NotifyPropertyChanged(nameof(ConnectToNetworkActionString));
            NotifyPropertyChanged(nameof(ServerDiscoveryActionString));
            NotifyPropertyChanged(nameof(ConnectedStateString));
            NotifyPropertyChanged(nameof(ConnectionName));

            foreach( VdbServerInfoViewModel serverInfo in DiscoveredServers )
            {
                serverInfo.Update();
            }
        }
        void NotifyPlaybackPropertiesChanged(Havok.Vdb.PlaybackCmdReceivedArgs args = null)
        {
            if ((args == null) || (args.PreviousState != args.TargetState))
            {
                NotifyPropertyChanged(nameof(PlaybackState));
                NotifyPropertyChanged(nameof(IsPaused));
            }

            if ((args == null) || (args.PreviousDirection != args.TargetDirection))
            {
                NotifyPropertyChanged(nameof(PlaybackDirection));
                NotifyPropertyChanged(nameof(IsPlayingBackward));
                NotifyPropertyChanged(nameof(IsPlayingForward));
            }

            if ((args == null) || (args.PreviousFlags != args.TargetFlags))
            {
                if ((args == null) || args.FlagChanged(Havok.Vdb.PlaybackFlags.Recording))
                {
                    NotifyPropertyChanged(nameof(IsRecording));
                    NotifyPropertyChanged(nameof(RecordActionString));
                    NotifyPropertyChanged(nameof(ConnectedStateString));
                }
                if ((args == null) || args.FlagChanged(Havok.Vdb.PlaybackFlags.AvailableCommandsProcessed))
                {
                    NotifyPropertyChanged(nameof(IsSyncedToConnection));
                }
            }

            if ((args == null) || (args.PreviousFrame != args.TargetFrame))
            {
                NotifyPropertyChanged(nameof(CurrentFrame));
                NotifyPropertyChanged(nameof(MaxProcessedFrame));
                NotifyPropertyChanged(nameof(MinFrame));
                NotifyPropertyChanged(nameof(MaxFrame));
                NotifyPropertyChanged(nameof(CurrentBufferFrame));
                NotifyPropertyChanged(nameof(MaxProcessedBufferFrame));
                NotifyPropertyChanged(nameof(BufferFrameCount));
                NotifyPropertyChanged(nameof(PlaybackTickFrequency));
                NotifyPropertyChanged(nameof(PlaybackTicks));
                NotifyPropertyChanged(nameof(HasPlaybackData));
            }
        }
        void NotifyClientPropertiesChanged()
        {
            NotifyPropertyChanged(nameof(ClientInfo));
            NotifyPropertyChanged(nameof(ClientProtocol));
            NotifyPropertyChanged(nameof(ClientMinProtocol));
            NotifyPropertyChanged(nameof(ClientSdkVersion));
            NotifyPropertyChanged(nameof(ClientPlatformString));
            NotifyPropertyChanged(nameof(ClientCompilerString));
            NotifyPropertyChanged(nameof(ClientPointerSize));
            NotifyPropertyChanged(nameof(ClientIsLittleEndian));
        }
        void NotifyServerPropertiesChanged()
        {
            NotifyPropertyChanged(nameof(ServerInfo));
            NotifyPropertyChanged(nameof(ServerProtocol));
            NotifyPropertyChanged(nameof(ServerMinProtocol));
            NotifyPropertyChanged(nameof(ServerSdkVersion));
            NotifyPropertyChanged(nameof(ServerPlatformString));
            NotifyPropertyChanged(nameof(ServerCompilerString));
            NotifyPropertyChanged(nameof(ServerPointerSize));
            NotifyPropertyChanged(nameof(ServerIsLittleEndian));
            NotifyPropertyChanged(nameof(Capabilities)); // capabilities depend on server
            NotifyPropertyChanged(nameof(CapabilitiesCanSave));
        }

        private string FilterFromExtensions(string typeName, IList<string> extensions)
        {
            if (extensions.Count != 0)
            {
                return
                    // Type name formatted for display
                    typeName +
                    // Extensions formatted for display
                    " (*." + string.Join(", *.", extensions) + ")|" +
                    // Extension patterns for file system
                    "*." + string.Join(";*.", extensions);
            }
            else
            {
                // Typename that matches any file
                return typeName + " (*.*)|*.*";
            }
        }

        private string DoFileSave(string title, string filter)
        {
            SaveFileDialog dialog = new SaveFileDialog();
            dialog.Title = title;
            dialog.Filter = filter;
            dialog.FilterIndex = 1;
            dialog.ValidateNames = true;
            dialog.RestoreDirectory = true;
            dialog.OverwritePrompt = true;

            bool? result = dialog.ShowDialog();
            if (result.HasValue && result.Value && !string.IsNullOrEmpty(dialog.FileName))
            {
                return dialog.FileName;
            }

            return null;
        }

        private string DoFileSave()
        {
            return DoFileSave(
                string.Format(Properties.Resources.File_SaveFile, Properties.Resources.File_HavokMovie),
                Properties.Resources.File_HavokMovie + " (*.hkm)|*.hkm");
        }

        private string DoFileOpen(string title, string filter)
        {
            OpenFileDialog dialog = new OpenFileDialog();
            dialog.Title = title;
            dialog.Filter = filter;
            dialog.FilterIndex = 1;
            dialog.ValidateNames = true;
            dialog.RestoreDirectory = true;
            dialog.CheckFileExists = true;
            dialog.CheckPathExists = true;

            bool? result = dialog.ShowDialog();
            if (result.HasValue && result.Value && !string.IsNullOrEmpty(dialog.FileName))
            {
                hkCollectionUtils.InsertIntoLimitedStringCollection(RecentFiles, dialog.FileName, MaxRecentFiles);
                return dialog.FileName;
            }

            return null;
        }

        private string DoFileOpen(string title)
        {
            return DoFileOpen(
                string.Format(title, Properties.Resources.File_HavokMovie),
                Properties.Resources.File_HavokMovie + " (*.hkm)|*.hkm");
        }

        private bool DoConnectToFileWithPrompt(object o, bool syncToConnection)
        {
            string fileName = DoFileOpen(syncToConnection ? Properties.Resources.File_LoadFile : Properties.Resources.File_OpenFile);
            if (bool.TrueString.Equals(o) && (fileName != null)) QueueAutoStartDebugRecording(System.IO.Path.ChangeExtension(fileName, ".hkm.txt"));
            return DoConnectToFileName(fileName, syncToConnection);
        }

        internal bool DoConnectToFileName(string fileName, bool syncToConnection)
        {
            if (fileName != null)
            {
                FileName = fileName;
                if (syncToConnection) QueueAutoSync();
                Client.CallAsync((o) => SkipToStart(null), nameof(Havok.Vdb.Client.ConnectToFile), fileName);
                hkCollectionUtils.InsertIntoLimitedStringCollection(RecentFiles, fileName, MaxRecentFiles);
                return true;
            }
            return false;
        }

        private string GetDefaultDebugFileName()
        {
            Havok.Vdb.ConnectedSource source = Client.GetProperty<Havok.Vdb.ConnectedSource>(nameof(Havok.Vdb.Client.ConnectedSource));
            if (source == Havok.Vdb.ConnectedSource.Network)
            {
                return System.Environment.CurrentDirectory + System.IO.Path.DirectorySeparatorChar + MachineAddress + "_" + MachinePort + ".hkm.txt";
            }
            else if (source == Havok.Vdb.ConnectedSource.File)
            {
                return System.IO.Path.ChangeExtension(FileName, ".hkm.txt");
            }
            else
            {
                System.Diagnostics.Debug.Assert(false);
                return "Unknown";
            }
        }

        private bool PerformAutoSync()
        {
            if (m_autoSync)
            {
                Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).CallAsync(nameof(Havok.Vdb.PlaybackHandler.SyncToConnection));
                m_autoSync = false;
                return true;
            }
            return false;
        }

        public void QueueAutoSync() { m_autoSync = true; }

        private bool PerformAutoStartRecording()
        {
            if (m_autoRecordFileName != null)
            {
                Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).CallAsync(nameof(Havok.Vdb.PlaybackHandler.StartRecording), m_autoRecordFileName);
                m_autoRecordFileName = null;
                return true;
            }
            return false;
        }

        public void QueueAutoStartRecording(string filename) { m_autoRecordFileName = filename; }

        private bool PerformAutoStartDebugRecording()
        {
            if (m_autoDebugRecordFileName != null)
            {
                Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).CallAsync(nameof(Havok.Vdb.PlaybackHandler.StartDebugRecording), m_autoDebugRecordFileName);
                m_autoDebugRecordFileName = null;
                return true;
            }
            return false;
        }

        public void QueueAutoStartDebugRecording(string filename) { m_autoDebugRecordFileName = filename; }

        private void ResetAutos()
        {
            m_autoSync = false;
            m_autoRecordFileName = null;
            m_autoDebugRecordFileName = null;
        }

        internal void TurnOnWorldInspectionViewer(object o)
        {
            // Clear selected non-default processes
            foreach (VdbProcessViewModel process in GetAllProcesses())
            {
                if (process.Name.Contains("World Inspector"))
                {
                    process.IsSelected = true;
                }
            }
        }

        private bool m_autoSync;
        private string m_autoRecordFileName;
        private string m_autoDebugRecordFileName;
        private bool m_completedFirstStep;
        #endregion

        #region VdbViewModel App Settings

        public bool AllowAutoPrompts { get; set; }

        private string _MaximizedButtonContent = "\xE71A";
        public string MaximizedButtonContent
        {
            get { return _MaximizedButtonContent; }
            set
            {
                _MaximizedButtonContent = value;
                NotifyPropertyChanged();
            }
        }

        private void InitializeAppSettings()
        {
            Themes = new ObservableCollection<VdbThemeViewModel>();

            Themes.Add(new VdbThemeViewModel(new ResourceDictionary()
            {
                Source = hkResourceUtils.GetThemeUri("DarkTheme")
            }));

            Themes.Add(new VdbThemeViewModel(new ResourceDictionary()
            {
                Source = hkResourceUtils.GetThemeUri("LightTheme")
            }));

            Themes.Add(new VdbThemeViewModel(new ResourceDictionary()
            {
                Source = hkResourceUtils.GetThemeUri("BlueTheme")
            }));

            SupportedFontSizes = new ObservableCollection<double>
            {
                12,15
            };

            NewContentManager = new VdbNewContentManager(this);
        }

        public ObservableCollection<VdbThemeViewModel> Themes { get; private set; }
        public ObservableCollection<double> SupportedFontSizes { get; private set; }

        public VdbThemeViewModel CurrentTheme
        {
            get
            {
                return m_currentTheme;
            }

            set
            {
                // Remove current theme
                if (m_currentTheme != null)
                {
                    Application.Current.Resources.MergedDictionaries.Remove(m_currentTheme.Dictionary);
                }

                m_currentTheme = value;

                // Apply new theme to dictionary
                Application.Current.Resources.MergedDictionaries.Add(m_currentTheme.Dictionary);

                // Update settings
                VdbSettings.SetValue(nameof(Properties.Settings.VdbUi_ColorTheme), m_currentTheme.Name);

                NotifyPropertyChanged();
            }
        }
        private VdbThemeViewModel m_currentTheme;

        public void InitializeTheme(string themeName)
        {
            VdbThemeViewModel foundTheme = GetThemeByName(themeName);
            if (foundTheme != null)
            {
                CurrentTheme = foundTheme;
                return;
            }

            // No theme found, set default
            foundTheme = GetThemeByName("Dark Theme");
            if (foundTheme != null)
            {
                CurrentTheme = foundTheme;
                return;
            }

            // No default found, use first theme
            if (Themes.Count > 0)
            {
                CurrentTheme = Themes[0];
                return;
            }

            // No themes found at all.  Something went wrong!
            System.Diagnostics.Debug.Assert(false, "No themes found.");
        }

        public VdbThemeViewModel GetThemeByName(string themeName)
        {
            foreach (VdbThemeViewModel theme in Themes)
            {
                if (theme.Name == themeName)
                {
                    return theme;
                }
            }

            return null;
        }

        public FontFamily IconFont
        { get; private set; }

        public FontFamily TextFont
        { get; private set; }

        public FontFamily MonospaceFont
        { get; private set; }

        public double FontSize
        {
            get { return GetBaseFontSize(); }
            set
            {
                value = hkMath.Clamp(value, 10, 20);

                if (GetBaseFontSize() != value)
                {
                    SetFontSize(value);
                    NotifyPropertyChanged();
                }
            }
        }

        public bool IsMSAAEnabled
        {
            get { return VdbSettings.GetValue<bool>(nameof(Properties.Settings.VdbSettings_MSAA_Enabled)); }
            set { VdbSettings.SetValue(nameof(Properties.Settings.VdbSettings_MSAA_Enabled), value); }
        }

        private double GetBaseFontSize()
        {
            object fontSize;
            ResourceDictionary fontDictionary = hkResourceUtils.FindResourceDictionary("ContentFontSize", out fontSize);
            if (fontDictionary == null)
            {
                return 12;
            }

            return (double)fontSize;
        }

        private void SetFontSize(double newFontSize)
        {
            object fontSize;
            ResourceDictionary fontDictionary = hkResourceUtils.FindResourceDictionary("ContentFontSize", out fontSize);
            if (fontDictionary == null)
            {
                return;
            }

            double difference = (double)fontSize - newFontSize;
            double proportionalDifference = newFontSize / GetBaseFontSize();

            // Ensure the font dictionaries get updated on the UI thread.
            // If they get updated on a separate thread, the resources may try be accessed
            // in between the remove/add.
            Application.Current.Dispatcher.BeginInvoke(System.Windows.Threading.DispatcherPriority.Normal,
                new Action(() =>
                {
                    SetSubtractiveFontResourceSize(fontDictionary, "VeryLargeContentFontSize", difference);
                    SetSubtractiveFontResourceSize(fontDictionary, "LargeContentFontSize", difference);
                    SetSubtractiveFontResourceSize(fontDictionary, "ContentFontSize", difference);
                    SetSubtractiveFontResourceSize(fontDictionary, "MediumContentFontSize", difference);
                    SetSubtractiveFontResourceSize(fontDictionary, "SmallContentFontSize", difference);
                    SetSubtractiveFontResourceSize(fontDictionary, "IconFontSize", difference);
                    SetProportionalResourceSize(fontDictionary, "ContentPaneWidth", proportionalDifference);

                    // Update settings
                    VdbSettings.SetValue(nameof(Properties.Settings.VdbUi_FontSize), newFontSize);

                    // Redo our tree since it sets it's width/height based on font sizes
                    CreateFilteredVdbObjectTreeView();
                }));

            OnFontSizeChanged(newFontSize);
        }

        public delegate void FontSizeChanged(double newSize);
        public event FontSizeChanged OnFontSizeChanged = delegate { };

        private void SetSubtractiveFontResourceSize(ResourceDictionary fontDictionary, string resourceName, double sizeDifference)
        {
            double fontSize = (double)fontDictionary[resourceName];
            fontDictionary.Remove(resourceName);
            fontDictionary.Add(resourceName, fontSize - sizeDifference);
        }

        private void SetProportionalResourceSize(ResourceDictionary fontDictionary, string resourceName, double proportionalDifference)
        {
            double size = (double)fontDictionary[resourceName];
            fontDictionary.Remove(resourceName);
            fontDictionary.Add(resourceName, size*proportionalDifference);
        }

        public VdbNewContentManager NewContentManager { get; set; }

        public string EulaDocument { get { return Properties.Resources.EULA; } }
        public string ThirdPartyNoticesDocument { get { return Properties.Resources.Third_Party_Notices; } }

        public bool IsLegalNoticeShown
        {
            get { return _IsLegalNoticeShown; }
            set
            {
                if (_IsLegalNoticeShown != value)
                {
                    _IsLegalNoticeShown = value;
                    NotifyPropertyChanged();
                }
            }
        }
        private bool _IsLegalNoticeShown;

        public bool IsAllowingTelemetry
        {
            get { return _IsAllowingTelemetry; }
            set
            {
                if(_IsAllowingTelemetry != value)
                {
                    _IsAllowingTelemetry = value;
                    NotifyPropertyChanged();

                    VdbSettings.SetValue(nameof(Properties.Settings.VdbSettings_AllowTelemetry), value);
                }
            }
        }
        private bool _IsAllowingTelemetry;

        public void LaunchHelpDocs(object o = null)
        {
            // Create embedded help file
            hkResourceUtils.EmbeddedResourceToFile("Docs", "Havok_VDB_Docs.chm");

            // Try to launch the .chm file
            try
            {
                System.Diagnostics.Process.Start("Havok_VDB_Docs.chm");
            }
            catch(Exception)
            {
            }
        }

        #endregion
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
