// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM		: WIN32 X64
// PRODUCT		: COMMON
// VISIBILITY	: CLIENT
//
// ------------------------------------------------------TKBMS v1.0

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;

namespace HavokVisualDebugger
{
    public class TimerNodeViewModel : ViewModelNotifyPropertyChanged
    {
        #region TimerNodeViewModel Initialization
        public TimerNodeViewModel(VdbViewModel rootViewModel, Havok.Vdb.ITimerNode node, bool includeChildren, Dictionary<string, List<TimerNodeViewModel>> timerPathToViewModel, ObservableCollection<TimerNodeViewModel> root, ObservableCollection<TimerNodeViewModel> view)
        {
            m_rootViewModel = rootViewModel;
            Parent = null;
            IsMatch = false;
            IsColorPressed = false;
            IsExpanded = false;
            ShowSelectedOnly = false;

            // Update internal data
            UpdateData(node);
            UpdateDictionaryCache(timerPathToViewModel);

            root.Add(this);

            TimerView = view;

            // Create child trees
            TimerChildren = new ObservableCollection<TimerNodeViewModel>();
            if (includeChildren)
            {
                foreach (Havok.Vdb.ITimerNode childNode in node.Children)
                {
                    TimerNodeViewModel vm = new TimerNodeViewModel(m_rootViewModel, childNode, includeChildren, timerPathToViewModel, root, view);
                    AddChild(vm);
                }
            }

            NotifyPropertyChanged(nameof(TimerChildren));
        }

        private void UpdateDictionaryCache(Dictionary<string, List<TimerNodeViewModel>> timerPathToViewModel)
        {
            // Update dictionary for quick access to this view model
            string fullNodePath = GetNodeFullPath(Path, Name);

            if (timerPathToViewModel.ContainsKey(fullNodePath))
            {
                List<TimerNodeViewModel> viewModelList = timerPathToViewModel[fullNodePath];
                viewModelList.Add(this);
            }
            else
            {
                List<TimerNodeViewModel> viewModelList = new List<TimerNodeViewModel>();
                viewModelList.Add(this);
                timerPathToViewModel.Add(fullNodePath, viewModelList);
            }
        }

        public void AddChild(TimerNodeViewModel child)
        {
            child.Parent = this;

            // Keep a max thread count
            VdbTimerNodeManager.MaxNumThreads = Math.Max(child.ThreadData.Count, VdbTimerNodeManager.MaxNumThreads);
            TimerChildren.Add(child);
        }
        #endregion

        #region TimerNodeViewModel View Commands and Properties
        public void UpdateData(Havok.Vdb.ITimerNode node)
        {
            // Clear thread data
            if (ThreadData == null)
            {
                ThreadData = new List<float>();
                ThreadDataCount = new List<uint>();

                // Init thread data
                foreach (float threadData in node.ThreadData)
                {
                    ThreadData.Add(threadData);
                }

                foreach (uint threadDataCount in node.ThreadDataCount)
                {
                    ThreadDataCount.Add(threadDataCount);
                }

                Name = node.Name;
                Path = node.Path;
                _NodeColor = node.Color;
                _IsEnabled = (node.Enabled & Havok.Vdb.ITimerNode.EnabledFlags.StatLineGraph) != 0;
            }
            else
            {
                // Early out
                if (TimerView != null && !TimerView.Contains(this))
                {
                    return;
                }

                while (node.ThreadData.Count >= ThreadData.Count)
                {
                    ThreadData.Add(0);
                    ThreadDataCount.Add(0);
                }

                // Reinit thread data
                for (int index = 0; index < node.ThreadData.Count; index++)
                {
                    ThreadData[index] = node.ThreadData[index];
                    ThreadDataCount[index] = node.ThreadDataCount[index];
                }
            }

            TimerValueSum = node.ValueSum;
            TimerCountSum = node.CountSum;

            NotifyThreadTimerProperties();
        }

        private void NotifyThreadTimerProperties()
        {
            NotifyPropertyChanged(nameof(ThreadData));
            NotifyPropertyChanged(nameof(ThreadDataCount));
            NotifyPropertyChanged(nameof(TimerTotalChildCallCount));
        }

        public void FilterTree(string filter)
        {
            IsMatch = IsNodeMatch(filter);

            foreach (TimerNodeViewModel childNode in TimerChildren)
            {
                childNode.FilterTree(filter);
            }
        }

        private bool IsNodeMatch(string filter)
        {
            return Name.IndexOf(filter, StringComparison.OrdinalIgnoreCase) >= 0;
        }

        public void DeselectAll()
        {
            IsEnabled = false;
            foreach (TimerNodeViewModel childNode in TimerChildren)
            {
                childNode.DeselectAll();
            }
        }

        public bool UpdateIsSelectedNode()
        {
            if (IsEnabled)
            {
                NotifyPropertyChanged(nameof(DisplayName));
                return true;
            }
            return false;
        }

        public void GetSelectedNodes(List<TimerNodeViewModel> selectedNodes, bool includeChildren)
        {
            if( UpdateIsSelectedNode() && !selectedNodes.Contains(this))
            {
                selectedNodes.Add(this);
            }

            if (includeChildren)
            {
                foreach (TimerNodeViewModel childNode in TimerChildren)
                {
                    childNode.GetSelectedNodes(selectedNodes, includeChildren);
                }
            }
        }

        public static bool NodesAreEqual(TimerNodeViewModel nodeA, TimerNodeViewModel nodeB)
        {
            return (nodeA.Path == nodeB.Path) && (nodeA.Name == nodeB.Name);
        }

        public static string GetNodeFullPath(string path, string name)
        {
            return path + name;
        }

        public ObservableCollection<TimerNodeViewModel> TimerChildren { get; set; }
        public TimerNodeViewModel Parent
        {
            get { return _Parent; }
            set
            {
                if( _Parent != value )
                {
                    _Parent = value;
                    NotifyPropertyChanged();
                }
            }
        }
        private TimerNodeViewModel _Parent;

        public string DisplayName
        {
            get
            {
                string name = String.IsNullOrEmpty(_Name) ? "Unknown" : _Name;
                if (((IsEnabled && ShowSelectedOnly) || IsMatch) && !String.IsNullOrEmpty(Path))
                {
                    return Path + name;
                }
                return name;
            }
        }

        public System.Windows.Thickness PaddingMargin
        {
            get
            {
                System.Windows.Thickness margin = new System.Windows.Thickness(ParentPadding * 15, 0, 0, 0);
                return margin;
            }
        }

        public string Name
        {
            get { return _Name; }
            set
            {
                if (_Name != value)
                {
                    _Name = value;
                    NotifyPropertyChanged();
                    NotifyPropertyChanged(nameof(DisplayName));
                }
            }
        }
        private string _Name;

        public string Path
        {
            get { return _Path; }
            set
            {
                if (_Path != value)
                {
                    _Path = value;
                    NotifyPropertyChanged();
                }
            }
        }
        private string _Path;

        public bool ShowSelectedOnly
        {
            get { return _ShowSelectedOnly; }
            set
            {
                if(_ShowSelectedOnly != value)
                {
                    _ShowSelectedOnly = value;
                    NotifyPropertyChanged();
                }
            }
        }
        private bool _ShowSelectedOnly;

        public bool IsColorPressed
        {
            get { return _IsColorPressed; }
            set
            {
                _IsColorPressed = value;
                NotifyPropertyChanged();
            }
        }

        public System.Windows.Media.Color NodeColor
        {
            get { return _NodeColor; }
            set
            {
                if( _NodeColor != value )
                {
                    _NodeColor = value;

                    using (m_rootViewModel.Client.AcquireLock())
                    {
                        Havok.Vdb.ITimerNode timerNode = GetTimerNode();
                        timerNode.Color = value;
                    }

                    NotifyPropertyChanged();
                }
            }
        }
        private System.Windows.Media.Color _NodeColor;

        public bool IsMatch
        {
            get { return _IsMatch; }
            set
            {
                if (_IsMatch != value)
                {
                    _IsMatch = value;
                    NotifyPropertyChanged();
                }
            }
        }

        public bool IsEnabled
        {
            get { return _IsEnabled; }
            set
            {
                if (_IsEnabled != value)
                {
                    _IsEnabled = value;

                    using (m_rootViewModel.Client.AcquireLock())
                    {
                        Havok.Vdb.ITimerNode timerNode = GetTimerNode();
                        if (value)
                        {
                            timerNode.Enabled |= Havok.Vdb.ITimerNode.EnabledFlags.StatLineGraph;
                        }
                        else
                        {
                            timerNode.Enabled &= ~Havok.Vdb.ITimerNode.EnabledFlags.StatLineGraph;
                        }
                    }

                    NotifyPropertyChanged();
                }
            }
        }
        private bool _IsEnabled;

        private Havok.Vdb.ITimerNode GetTimerNode()
        {
            System.Diagnostics.Debug.Assert(m_rootViewModel.Client.HasLock(), "Must have acquired lock on VdbClient before accessing perfStats here.");

            Havok.Vdb.PerfStatsReceivedEventArgs perfStats = m_rootViewModel.Client.GetSubInterop(nameof(Havok.Vdb.StatsHandler)).
                        GetProperty<Havok.Vdb.PerfStatsReceivedEventArgs>(nameof(Havok.Vdb.StatsHandler.PerfStats));

            string fullNodePath = GetNodeFullPath(Path, Name);
            return perfStats.RootTimerNode.FindOrCreateChild(fullNodePath);
        }

        public bool IsExpanded
        {
            get
            {
                if (Parent != null)
                {
                    return _IsExpanded && Parent.IsExpanded;
                }
                return _IsExpanded;
            }
            set
            {
                if (_IsExpanded != value)
                {
                    _IsExpanded = value;
                    NotifyPropertyChanged();

                    using (m_rootViewModel.Client.AcquireLock())
                    {
                        NotifyChildrenOfNodeExpansion(_IsExpanded, true);
                    }
                }

                // Expand all the way up to the root.
                if (_IsExpanded && Parent != null && !ShowSelectedOnly)
                {
                    Parent.IsExpanded = true;
                }
            }
        }

        private void NotifyChildrenOfNodeExpansion(bool parentExpanded, bool root)
        {
            if (!root)
            {
                NotifyPropertyChanged(nameof(IsExpanded));
                NotifyThreadTimerProperties();

                if (TimerView != null)
                {
                    int thisIndexInView = TimerView.IndexOf(this);

                    if (!parentExpanded)
                    {
                        if (thisIndexInView >= 0)
                        {
                            TimerView.RemoveAt(thisIndexInView);
                        }
                    }
                    else
                    {
                        if (thisIndexInView < 0 && Parent != null && Parent.IsExpanded)
                        {
                            int parentIndex = TimerView.IndexOf(Parent);
                            if (parentIndex >= 0)
                            {
                                TimerView.Insert(parentIndex + 1, this);
                                UpdateData(GetTimerNode());
                            }
                        }
                    }
                }
            }

            if (TimerChildren != null)
            {
                for (int i = TimerChildren.Count - 1; i >= 0; i--)
                {
                    TimerNodeViewModel node = TimerChildren[i];
                    node.NotifyChildrenOfNodeExpansion(parentExpanded, false);
                }
            }
        }

        public float TimerValueSum
        {
            get { return _ValueSum; }
            set
            {
                if (_ValueSum != value)
                {
                    _ValueSum = value;
                    NotifyPropertyChanged();
                }
            }
        }
        private float _ValueSum;

        public uint TimerCountSum
        {
            get { return _CountSum; }
            set
            {
                if (_CountSum != value)
                {
                    _CountSum = value;
                    NotifyPropertyChanged();
                }
            }
        }
        private uint _CountSum;

        public uint TimerTotalChildCallCount
        {
            get { return _TimerTotalChildCallCount; }
            set
            {
                if (_TimerTotalChildCallCount != value)
                {
                    _TimerTotalChildCallCount = value;
                    NotifyPropertyChanged();
                }
            }
        }
        private uint _TimerTotalChildCallCount;

        public int ParentPadding
        {
            get { return (Parent == null) ? 0 : 1 + Parent.ParentPadding; }
        }

        public List<float> ThreadData { get; private set; }
        public List<uint> ThreadDataCount { get; private set; }

        #endregion

        #region TimerNodeViewModel Private Methods and Properties

        private bool _IsMatch;
        private bool _IsExpanded;
        private bool _IsColorPressed;
        private VdbViewModel m_rootViewModel;
        public ObservableCollection<TimerNodeViewModel> TimerView;

        #endregion
    }

    public class VdbTimerNodeManager
    {
        // Presented to the user
        public ObservableCollection<TimerNodeViewModel> NodeView;

        // Backend - used to store and construct NodeView
        public ObservableCollection<TimerNodeViewModel> NodeViewModel;

        // Hashmap of full paths to nodes in the view
        public Dictionary<string, List<TimerNodeViewModel>> PathToViewModel;

        // Back reference
        public VdbViewModel RootViewModel;

        public VdbTimerNodeManager(VdbViewModel rootViewModel, ObservableCollection<TimerNodeViewModel> view, ObservableCollection<TimerNodeViewModel> viewModel, Dictionary<string, List<TimerNodeViewModel>> pathToViewModel)
        {
            RootViewModel = rootViewModel;
            NodeView = view;
            NodeViewModel = viewModel;
            PathToViewModel = pathToViewModel;
        }

        public static int MaxNumThreads = 0;

        public void ToggleShowSelectedTimersOnly()
        {
            if (RootViewModel.ShowSelectedTimersOnly)
            {
                List<TimerNodeViewModel> nodes = new List<TimerNodeViewModel>();
                lock (NodeView)
                {
                    foreach (TimerNodeViewModel vm in NodeView)
                    {
                        vm.GetSelectedNodes(nodes, true);
                    }
                }

                UpdateTimerView(nodes);
            }
            else
            {
                // Construct from the full node set
                UpdateTimerView(NodeViewModel);
            }
        }

        public void DeselectAllTimers()
        {
            lock (NodeView)
            {
                foreach (TimerNodeViewModel vm in NodeView)
                {
                    vm.DeselectAll();
                }
            }
        }

        public void ClearTimers()
        {
            NodeView.Clear();
            NodeViewModel.Clear();
            PathToViewModel.Clear();
            RootViewModel.ShowSelectedTimersOnly = false;
        }

        public bool UpdateTimers(Havok.Vdb.PerfStatsReceivedEventArgs perfStats)
        {
            System.Diagnostics.Debug.Assert(perfStats != null, "perfStats cannot be null here.  Caller can use StatsHandler subinterop to grab PerfStats, but access and usage must be contained in a client lock.");
            System.Diagnostics.Debug.Assert(RootViewModel.Client.HasLock(), "Must have acquired lock on VdbClient before accessing perfStats here.");

            if (NodeViewModel.Count == 0)
            {
                // full rebuild
                PathToViewModel.Clear();
                NodeViewModel.Clear();
                MaxNumThreads = 0;
                foreach (Havok.Vdb.ITimerNode node in perfStats.RootTimerNode.Children)
                {
                    
                    
                    new TimerNodeViewModel(RootViewModel, node, true, PathToViewModel, NodeViewModel, NodeView);
                }

                return UpdateTimerView(NodeViewModel);
            }
            else
            {
                int maxNumThreads = 0;
                int perfStatsNumChildren = GetNumTimerChildren(perfStats.RootTimerNode, ref maxNumThreads);

                // update timers
                foreach (Havok.Vdb.ITimerNode node in perfStats.RootTimerNode.Children)
                {
                    UpdateTimerData(node);
                }

                if (MaxNumThreads != maxNumThreads)
                {
                    MaxNumThreads = maxNumThreads;
                    return true;
                }
            }

            return false;
        }

        private int GetNumTimerChildren(Havok.Vdb.ITimerNode timerNode, ref int maxNumThreads)
        {
            int numSubChildren = 1;
            foreach (Havok.Vdb.ITimerNode childNode in timerNode.Children)
            {
                maxNumThreads = Math.Max(childNode.ThreadData.Count, maxNumThreads);
                numSubChildren += GetNumTimerChildren(childNode, ref maxNumThreads);
            }
            return numSubChildren;
        }

        private void UpdateTimerData(Havok.Vdb.ITimerNode timerNode)
        {
            string fullPath = TimerNodeViewModel.GetNodeFullPath(timerNode.Path, timerNode.Name);

            // If the timer path doesn't exist, insert it
            if (!PathToViewModel.ContainsKey(fullPath))
            {
                InsertNewTimerNode(fullPath, timerNode);
                return;
            }

            List<TimerNodeViewModel> nodeLists = PathToViewModel[fullPath];
            foreach (TimerNodeViewModel nodeVm in nodeLists)
            {
                nodeVm.UpdateData(timerNode);
            }

            // Update child timers
            foreach (Havok.Vdb.ITimerNode childNode in timerNode.Children)
            {
                UpdateTimerData(childNode);
            }
        }

        private void InsertNewTimerNode(string path, Havok.Vdb.ITimerNode timerNode)
        {
            // Find parent timer
            int lastPathIdx = -1;
            TimerNodeViewModel parent = null;
            string currentPath = path;
            while ((lastPathIdx = currentPath.LastIndexOf('/')) != -1)
            {
                currentPath = currentPath.Substring(0, lastPathIdx);
                if (PathToViewModel.ContainsKey(currentPath))
                {
                    parent = PathToViewModel[currentPath][0];
                    break;
                }
            }

            // Create node
            lock (NodeView)
            {
                TimerNodeViewModel newTimerNode = new TimerNodeViewModel(RootViewModel, timerNode, true, PathToViewModel, NodeViewModel, NodeView);
                if (parent != null)
                {
                    // Add node to data
                    {
                        // Find node
                        int dataParentIndex = NodeViewModel.IndexOf(parent);
                        if (dataParentIndex != -1)
                        {
                            parent.AddChild(newTimerNode);
                            //System.Diagnostics.Debug.Print("Adding node " + TimerNodeViewModel.GetNodeFullPath(newTimerNode.Path, newTimerNode.Name));
                        }
                        else
                        {
                            System.Diagnostics.Debug.Assert(false, "Adding new timer node with a parent that is not found in the current list.");
                        }
                    }

                    // Add node to view
                    if (parent.IsExpanded)
                    {
                        int dataParentViewIndex = 0;
                        for (; dataParentViewIndex < NodeView.Count; dataParentViewIndex++)
                        {
                            TimerNodeViewModel viewNode = NodeView[dataParentViewIndex];
                            if (TimerNodeViewModel.NodesAreEqual(viewNode, parent))
                            {
                                break;
                            }
                        }

                        if (dataParentViewIndex < NodeView.Count)
                        {
                            List<TimerNodeViewModel> nodes = new List<TimerNodeViewModel>();
                            if (RootViewModel.ShowSelectedTimersOnly)
                            {
                                newTimerNode.GetSelectedNodes(nodes, false);
                            }
                            else
                            {
                                nodes.Add(newTimerNode);
                            }

                            foreach (TimerNodeViewModel childNode in nodes)
                            {
                                NodeView.Insert(dataParentViewIndex + 1, childNode);
                            }
                        }
                    }
                }
                else
                {
                    // Root node
                    if (!path.Contains('/'))
                    {
                        NodeView.Insert(NodeView.Count - 1, newTimerNode);
                    }
                    else
                    {
                        System.Diagnostics.Debug.Assert(false, "Adding new timer node with a parent that is not found in the current list.");
                    }
                }

            }
        }

        private bool UpdateTimerView(ICollection<TimerNodeViewModel> collection)
        {
            lock (NodeView)
            {
                NodeView.Clear();
                foreach (TimerNodeViewModel vm in collection)
                {
                    bool addNodeToView = false;
                    if (RootViewModel.ShowSelectedTimersOnly)
                    {
                        if (vm.UpdateIsSelectedNode())
                        {
                            if (!String.IsNullOrEmpty(RootViewModel.TimerFilter))
                            {
                                if (vm.IsMatch)
                                {
                                    addNodeToView = true;
                                }
                            }
                            else
                            {
                                addNodeToView = true;
                            }
                        }
                    }
                    else if (!String.IsNullOrEmpty(RootViewModel.TimerFilter))
                    {
                        if (vm.IsMatch)
                        {
                            addNodeToView = true;
                        }
                    }
                    else if (vm.Parent == null || vm.Parent.IsExpanded)
                    {
                        addNodeToView = true;
                    }

                    if (addNodeToView)
                    {
                        
                        //TimerNodeViewModel newNode = new TimerNodeViewModel(vm, TimerPathToViewModel);
                        NodeView.Add(vm);
                    }
                }

                return true;
            }
        }

        public bool UpdateTimerFilter(string value)
        {
            // Filter here
            foreach (TimerNodeViewModel vm in NodeViewModel)
            {
                vm.FilterTree(value);
            }

            return UpdateTimerView(NodeViewModel);
        }
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
