// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM		: WIN32 X64
// PRODUCT		: COMMON
// VISIBILITY	: CLIENT
//
// ------------------------------------------------------TKBMS v1.0

using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Configuration;
using System.Linq;

namespace HavokVisualDebugger
{
    static class VdbSettings
    {
        static VdbSettings()
        {
            ForceLoad();

            WriteEnabled = true;
            foreach (SettingsProperty property in Properties.Settings.Default.Properties)
            {
                if (property.PropertyType == typeof(StringCollection) && (!HasValue(property.Name) || Properties.Settings.Default[property.Name] == null))
                {
                    Properties.Settings.Default[property.Name] = new StringCollection();
                }
            }
        }

        static public bool WriteEnabled { get; set; }

        public static bool IsCollectionItemEnabled(StringCollection collection, string itemName)
        {
            if (collection != null)
            {
                // Find the item
                foreach (string enabledItem in collection)
                {
                    if (enabledItem == itemName)
                    {
                        // Item is marked enabled
                        return true;
                    }
                }
            }

            return false;
        }

        public static void SetCollectionItemEnabled(StringCollection collection, string itemName, bool enabled)
        {
            bool isCurrentlyEnabled = VdbSettings.IsCollectionItemEnabled(collection, itemName);
            if (isCurrentlyEnabled == enabled)
            {
                // Settings match, early out
                return;
            }

            // Make sure a setting exists
            if (collection == null)
            {
                collection = new StringCollection();
            }

            if (enabled)
            {
                // Add viewer
                collection.Add(itemName);
            }
            else
            {
                // Remove viewer
                collection.Remove(itemName);
            }

            // Set and Save settings
            Properties.Settings.Default.Save();
        }

        static public T GetValue<T>(string name)
        {
            if (HasValue(name))
            {
                return (T)Properties.Settings.Default[name];
            }

            // Not found, return default
            return default(T);
        }

        static public bool HasValue(string name)
        {
            foreach (SettingsPropertyValue property in Properties.Settings.Default.PropertyValues)
            {
                if (property.Name == name && property.PropertyValue != null)
                {
                    return true;
                }
            }

            return false;
        }

        static public void SetValue<T>(string setting, T value)
        {
            if( !HasValue(setting) )
            {
                System.Diagnostics.Debug.Assert(false, "Expected a valid setting but none existed.  If a new command was added, make sure to add a corresponding user setting!");
                return;
            }

            try
            {
                if (!value.Equals((T)Properties.Settings.Default[setting]))
                {
                    Properties.Settings.Default[setting] = value;
                    Properties.Settings.Default.Save();
                }
            }
            catch
            {
            }
        }

        public static void UpdateSettingsToNewVersion(int oldVersion, int newVersion)
        {
            // Early out on no change
            if ((oldVersion == newVersion) || (oldVersion == -1))
            {
                return;
            }

            
        }

        public static void ApplyTo(MainWindow mainWindow)
        {
            // This is the soonest we have a client, apply any versioning to our settings
            {
                int version = -1;
                if (VdbSettings.HasValue(nameof(Properties.Settings.VdbSettingsVersion)))
                {
                    version = VdbSettings.GetValue<int>(nameof(Properties.Settings.VdbSettingsVersion));
                }
                UpdateSettingsToNewVersion(version, mainWindow.ViewModel.ClientSdkVersion);
            }

            if (mainWindow.ViewModel.AllowAutoPrompts)
            {
                // Disclosures/Legal
                {
                    float legalStatementAckVersion = VdbSettings.GetValue<float>(nameof(Properties.Settings.VdbApp_LegalStatementAckVersion));
                    if (legalStatementAckVersion < hkConstants.LegalStatementVersion)
                    {
                        mainWindow.ViewModel.IsLegalNoticeShown = true;
                        VdbSettings.SetValue(nameof(Properties.Settings.VdbApp_LegalStatementAckVersion), hkConstants.LegalStatementVersion);
                    }
                }

                // What's new
                {
                    int whatsNewAckVersion = VdbSettings.GetValue<int>(nameof(Properties.Settings.VdbApp_WhatsNewAckVersion));
                    if (whatsNewAckVersion < mainWindow.ViewModel.ClientSdkVersion)
                    {
                        mainWindow.ViewModel.NewContentManager.IsNewContentPanelShown = true;
                    }
                }

                // Tooltips
                {
                    
                }
            }
        }

        
        public static void ApplyTo(VdbViewModel vm)
        {
            using (new hkResetOnDispose(typeof(VdbSettings), nameof(WriteEnabled), false))
            {
                // Connection settings
                {
                    vm.AutoReconnectEnabled = VdbSettings.GetValue<bool>(nameof(Properties.Settings.VdbConnection_AutoReconnect));
                    vm.MachineAddress = VdbSettings.GetValue<string>(nameof(Properties.Settings.VdbConnection_Address));
                    vm.MachinePort = VdbSettings.GetValue<string>(nameof(Properties.Settings.VdbConnection_Port));
                    if (String.IsNullOrEmpty(vm.MachinePort)) vm.MachinePort = Havok.Vdb.Client.CONNECT_DEFAULT_SERVER_PORT.ToString();
                    vm.MachineDiscoveryPort = VdbSettings.GetValue<string>(nameof(Properties.Settings.VdbConnection_DiscoveryPort));
                    if (String.IsNullOrEmpty(vm.MachineDiscoveryPort)) vm.MachineDiscoveryPort = Havok.Vdb.Client.ADVERTISE_DEFAULT_SERVER_PORT.ToString();
                    vm.IsServerDiscoveryEnabled = VdbSettings.GetValue<bool>(nameof(Properties.Settings.VdbConnection_ServerDiscoveryEnabled));

                    // When first opening up the VDB, consider it to be trying to connect if auto-reconnect is enabled.
                    UInt16 port;
                    if (vm.AutoReconnectEnabled && UInt16.TryParse(vm.MachinePort, out port))
                    {
                        vm.Client.AttemptAutoReconnect = new Tuple<string, UInt16>(vm.MachineAddress, port);
                    }
                }

                // Render settings
                {
                    // Widgets
                    {
                        foreach (Havok.Vdb.RenderWidget widget in vm.Client.RenderSurface.Widgets)
                        {
                            // If we have our own control, uninstall the widget-version.
                            if (widget.Name == "Stats Text")
                            {
                                widget.Installed = false;
                            }
                            else
                            {
                                vm.RegisteredWidgetsCollection.Add(new VdbWidgetViewModel(widget));
                            }
                        }

                        foreach (VdbWidgetViewModel widget in vm.RegisteredWidgetsCollection)
                        {
                            widget.IsSelected = VdbSettings.IsCollectionItemEnabled(Properties.Settings.Default.VdbEnabledWidgets, widget.Name);
                        }

                        vm.SetDiagnosticWidgets();

                        vm.TimerMinY = VdbSettings.GetValue<double>(nameof(Properties.Settings.VdbWidgets_TimerMinY));
                        vm.TimerMaxY = VdbSettings.GetValue<double>(nameof(Properties.Settings.VdbWidgets_TimerMaxY));
                    }

                    // Visualizations
                    vm.RandomizedColors = VdbSettings.GetValue<bool>(nameof(Properties.Settings.VdbRender_RandomizeColors));
                    vm.SelectionVisualization = VdbSettings.GetValue<bool>(nameof(Properties.Settings.VdbRender_SelectionVisualization));
                    vm.RenderWireframe = VdbSettings.GetValue<bool>(nameof(Properties.Settings.VdbRender_Wireframe));
                    vm.RenderOutlineFaces = VdbSettings.GetValue<bool>(nameof(Properties.Settings.VdbRender_OutlineFaces));
                    vm.LockLightToCamera = VdbSettings.GetValue<bool>(nameof(Properties.Settings.VdbRender_LockLightToCamera));

                    // Backface culling
                    string backfaceCulling = VdbSettings.GetValue<string>(nameof(Properties.Settings.Default.VdbRender_BackfaceCullingMode));
                    VdbViewModel.BackfaceCullingModeOptions backfaceCullingOptions;
                    if( Enum.TryParse<VdbViewModel.BackfaceCullingModeOptions>(backfaceCulling, out backfaceCullingOptions) )
                    {
                        vm.BackfaceCullingMode = backfaceCullingOptions;
                    }

                    // Camera
                    {
                        // Note: we set camera mode first so that we listen for property value changes on the backend controller
                        string mode = VdbSettings.GetValue<string>(nameof(Properties.Settings.VdbCamera_Controller));
                        vm.SetCameraMode(mode);

                        Int32[] upVector = VdbSettings.GetValue<Int32[]>(nameof(Properties.Settings.VdbCamera_Up));
                        vm.CameraUp = new Havok.Vdb.Vector(upVector[0], upVector[1], upVector[2]);
                        vm.CameraNearPlane = VdbSettings.GetValue<double>(nameof(Properties.Settings.VdbCamera_NearPlane));
                        vm.CameraFarPlane = VdbSettings.GetValue<double>(nameof(Properties.Settings.VdbCamera_FarPlane));
                        vm.CameraFieldOfView = VdbSettings.GetValue<double>(nameof(Properties.Settings.VdbCamera_FieldOfView));
                        vm.CameraPerspectiveMode = VdbSettings.GetValue<bool>(nameof(Properties.Settings.VdbCamera_PerspectiveMode));
                        vm.CameraLookSensitivity = VdbSettings.GetValue<double>(nameof(Properties.Settings.VdbCamera_LookSensitivity));
                        vm.CameraLeftHandedCoordinateSystem = VdbSettings.GetValue<bool>(nameof(Properties.Settings.VdbCamera_LeftHandedCoordinateSystem));
                        vm.CameraIsLookingAtPOI = VdbSettings.GetValue<bool>(nameof(Properties.Settings.VdbCamera_LookAtPrimaryPointOfInterest));
                        vm.CameraIsLookingAtGeometryCenter = VdbSettings.GetValue<bool>(nameof(Properties.Settings.VdbCamera_LookAtGeometryCenter));
                        vm.CameraIsKeepingRelativeToPOI = VdbSettings.GetValue<bool>(nameof(Properties.Settings.VdbCamera_KeepRelativeToPrimaryPointOfInterest));
                        vm.CameraIsKeepingRelativeToGeometryCenter = VdbSettings.GetValue<bool>(nameof(Properties.Settings.VdbCamera_KeepRelativeToGeometryCenter));
                    }

                    System.Windows.Media.Color savedColor = VdbSettings.GetValue<System.Windows.Media.Color>(nameof(Properties.Settings.VdbRender_ClearColor));
                    {
                        // Make sure color is fully opaque
                        savedColor.A = 255;
                        vm.SelectedClearColor = savedColor;
                    }
                }

                vm.ObjectPropertyOptions.HexFormatEnabled = VdbSettings.GetValue<bool>(nameof(Properties.Settings.VdbOptions_HexFormatEnabled));

                // Timer settings
                {
#if false
                    
                    
                    Havok.Vdb.PerfStatsReceivedEventArgs perfStats = vm.Client.GetSubInterop(nameof(Havok.Vdb.Client.StatsHandler)).
                        GetProperty<Havok.Vdb.PerfStatsReceivedEventArgs>(nameof(Havok.Vdb.StatsHandler.PerfStats));
#endif

                }

                // Layers
                {
                    vm.IsFilterLayerPanelShown = VdbSettings.GetValue<bool>(nameof(Properties.Settings.VdbUi_FilterLayerPanelShown));

                    // Get these before we dispose/clear; because that will change this setting.
                    StringCollection layersCollection = VdbSettings.GetValue<StringCollection>(nameof(Properties.Settings.VdbLayers));
                    List<string> layersList = layersCollection.Cast<string>().ToList();

                    // Clear current layers
                    foreach (VdbMutableFilterLayerViewModel layer in vm.VdbObjectFilterLayers.ToArray())
                    {
                        layer.Dispose();
                    }
                    vm.VdbObjectFilterLayers.Clear();

                    // Add saved layers (in reverse order because new layers always go to the front and we saved them in sequential order).
                    for (int i = layersList.Count - 1; i >= 0; i--)
                    {
                        string layerStream = layersList[i];
                        vm.VdbObjectFilterLayers.CreateNewLayerFromStream(layerStream);
                    }
                }

            }
        }

        public static void ApplyToSession(VdbViewModel vm)
        {
            using (new hkResetOnDispose(typeof(VdbSettings), nameof(WriteEnabled), false))
            {
                // Viewers
                foreach (VdbProcessesViewModel categoryProc in vm.CategoryProcessCollection)
                {
                    List<VdbProcessViewModel> enabledProcesses = new List<VdbProcessViewModel>(categoryProc.Processes).FindAll(p => Properties.Settings.Default.VdbEnabledViewers.Contains(p.Name));
                    enabledProcesses.ForEach(p => p.IsSelected = true);
                }
            }
        }

        /// <summary>
        /// The Properties.Settings file lazy loads - this means that it won't load its contents until at least
        /// one .Default setting has been accessed.  This can have bad consequences depending on when the
        /// file is loaded.  Forcing it to load at a specific time can alleviate those issues.
        /// </summary>
        public static void ForceLoad()
        {
            // This is needed because if the locally copied settings file is deleted it can cause a crash.
            // This is only relevant in a debug setting where Visual Studio copies the settings files to a local directory.
            // While the reload doesn't necessarily load the properties, it will recreate the file if it's not there
            // (hence why we need to access a property below to force the lazy load).
            Properties.Settings.Default.Reload();

            // Pick an arbitrary property to access to force a load
            Properties.Settings.Default.VdbUi_WindowWidth.GetType();
        }
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
