// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM		: WIN32 X64
// PRODUCT		: COMMON
// VISIBILITY	: CLIENT
//
// ------------------------------------------------------TKBMS v1.0

using System;
using System.Linq;
using System.Collections.Generic;
using System.Collections.Specialized;

namespace HavokVisualDebugger
{
    public class VdbSelection : ViewModelNotifyPropertyChanged, IDisposable
    {
        public VdbSelection(VdbViewModel vm)
        {
            m_viewModel = vm;
            SelectedObjectProperties = new VdbObjectPropertiesViewModel(vm, this);
            SelectedObjects = new hkObservableCollection<VdbObjectViewModel>();
            SelectedObjects.CollectionChanged += HighlightedOrSelectedVdbObjects_CollectionChanged;
            HighlightedObjects = new hkObservableCollection<VdbObjectViewModel>();
            HighlightedObjects.CollectionChanged += HighlightedOrSelectedVdbObjects_CollectionChanged;
            m_selectionChangedInteropDelegate = m_viewModel.Client.GetSubInterop(nameof(Havok.Vdb.Client.RenderSurface)).
                ListenTo<Havok.Vdb.SelectionChangedEventHandler, Havok.Vdb.RenderSurface, Havok.Vdb.SelectionChangedEventArgs>(
                    nameof(Havok.Vdb.RenderSurface.SelectionChanged),
                    delegate (Havok.Vdb.RenderSurface sender, Havok.Vdb.SelectionChangedEventArgs args)
                    {
                        RenderSurface_SelectionChanged(sender, args);
                    });

            
            System.Diagnostics.Debug.Assert(
                m_viewModel.Client.RenderSurface.SelectionFlags.HasFlag(Havok.Vdb.SelectionFlags.ModifySelectionOnGrabbingGeometry),
                "We assume that our selection paradigm in the backend is doing clears when changing the selection based on interaction");
            System.Diagnostics.Debug.Assert(
                m_viewModel.Client.RenderSurface.SelectionFlags.HasFlag(Havok.Vdb.SelectionFlags.IgnoreUnhighlightedSelections),
                "We assume that our selection paradigm in the backend is ignoring unhighlighted selections");
        }

        public void Dispose()
        {
            // clear back-end?
            ClearHighlightedFrontend();
            ClearSelectedFrontend();
            _PreviousSelectedTreeItem = null;
            _PreviousSelectedObject = null;
            _PreviousSelectedRenderObject = null;
            SelectedObjectProperties.Dispose();
            SelectedObjectProperties = new VdbObjectPropertiesViewModel(m_viewModel, this);
            SelectedObjects.CollectionChanged -= HighlightedOrSelectedVdbObjects_CollectionChanged;
            SelectedObjects = new hkObservableCollection<VdbObjectViewModel>();
            HighlightedObjects.CollectionChanged -= HighlightedOrSelectedVdbObjects_CollectionChanged;
            HighlightedObjects = new hkObservableCollection<VdbObjectViewModel>();
            m_viewModel.Client.GetSubInterop(nameof(Havok.Vdb.Client.RenderSurface)).
                UnListenTo(
                    nameof(Havok.Vdb.RenderSurface.SelectionChanged),
                    m_selectionChangedInteropDelegate);
        }

        #region VdbSelection Public

        public VdbObjectMultiParentTreeItem SelectedTreeItem
        {
            get { return _SelectedTreeItem; }
            set
            {
                // This is a one to many relationship, so we need to always set
                // in case related objects *aren't* selected properly.
                // if (_SelectedTreeItem != value)
                {
                    SetSelected(value);
                }
                // Still need to clear other Selected properties on null set
                //else if (value == null)
                //{
                //    ClearSelected();
                //}
            }
        }
        private VdbObjectMultiParentTreeItem _SelectedTreeItem;
        private VdbObjectMultiParentTreeItem _PreviousSelectedTreeItem;
        public VdbObjectViewModel SelectedObject
        {
            get { return ((SelectedObjects != null) && (SelectedObjects.Count > 0)) ? SelectedObjects[0] : null; }
            set
            {
                if (SelectedObject != value)
                {
                    SetSelected(value);
                }
                // Still need to clear other Selected properties on null set
                else if (value == null)
                {
                    ClearSelected();
                }
            }
        }
        private VdbObjectViewModel _PreviousSelectedObject;
        public hkObservableCollection<VdbObjectViewModel> SelectedObjects { get; private set; }
        public int SelectedObjectsCount { get { return SelectedObjects.Count; } }
        public hkObservableCollection<VdbObjectViewModel> HighlightedObjects { get; private set; }
        public int HighlightedObjectsCount { get { return HighlightedObjects.Count; } }
        public Havok.Vdb.RenderObject SelectedRenderObject
        {
            get { return _SelectedRenderObject; }
            set
            {
                if ((_SelectedRenderObject != value) ||
                    ((_SelectedRenderObject != null) && (value != null) && (_SelectedRenderObject.Id != value.Id)))
                {
                    SetSelected(value);
                }
                // Still need to clear other Selected properties on null set
                else if (value == null)
                {
                    ClearSelected();
                }
            }
        }
        private Havok.Vdb.RenderObject _SelectedRenderObject;
        private Havok.Vdb.RenderObject _PreviousSelectedRenderObject;
        public VdbObjectPropertiesViewModel SelectedObjectProperties { get; private set; }
        public int SelectedRenderObjectsCount
        {
            get { return _SelectedRenderObjectsCount; }
            private set
            {
                if (_SelectedRenderObjectsCount != value)
                {
                    _SelectedRenderObjectsCount = value;
                    NotifyPropertyChanged();
                }
            }
        }
        private int _SelectedRenderObjectsCount;
        public int HighlightedRenderObjectsCount
        {
            get { return _HighlightedRenderObjectsCount; }
            private set
            {
                if (_HighlightedRenderObjectsCount != value)
                {
                    _HighlightedRenderObjectsCount = value;
                    NotifyPropertyChanged();
                }
            }
        }
        private int _HighlightedRenderObjectsCount;

        public void ClearHighlightedAndSelected(IEnumerable<UInt64> ids = null)
        {
            using (new hkResetOnDispose(this, nameof(SettingSelection), true))
            {
                if (ids != null)
                {
                    m_viewModel.Client.RenderSurface.ClearHighlightedAndSelectedObjects(
                        new List<Havok.Vdb.RenderObject>(
                            ids
                                .Select(id => m_viewModel.Client.RenderSurface.GetRenderObject(id))
                                .Where(ro => ro != null)));
                }
                else
                {
                    m_viewModel.Client.RenderSurface.ClearHighlightedAndSelectedObjects();
                }
                ClearSelectedFrontend(ids);
                ClearHighlightedFrontend(ids);
            }
        }
        public void ClearSelected(IEnumerable<UInt64> ids = null)
        {
            using (new hkResetOnDispose(this, nameof(SettingSelection), true))
            {
                if (ids != null)
                {
                    m_viewModel.Client.RenderSurface.ClearSelectedObjects(
                        new List<Havok.Vdb.RenderObject>(
                            ids
                                .Select(id => m_viewModel.Client.RenderSurface.GetRenderObject(id))
                                .Where(ro => ro != null)));
                }
                else
                {
                    m_viewModel.Client.RenderSurface.ClearSelectedObjects();
                }
                ClearSelectedFrontend(ids);
            }
        }
        public void ClearHighlighted(IEnumerable<UInt64> ids = null)
        {
            using (new hkResetOnDispose(this, nameof(SettingSelection), true))
            {
                if (ids != null)
                {
                    m_viewModel.Client.RenderSurface.ClearHighlightedObjects(
                        new List<Havok.Vdb.RenderObject>(
                            ids
                                .Select(id => m_viewModel.Client.RenderSurface.GetRenderObject(id))
                                .Where(ro => ro != null)));
                }
                else
                {
                    m_viewModel.Client.RenderSurface.ClearHighlightedObjects();
                }
                ClearHighlightedFrontend(ids);
            }
        }

        #endregion

        #region VdbSelection Private

        void HighlightedOrSelectedVdbObjects_CollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            System.Diagnostics.Debug.Assert(e.Action != NotifyCollectionChangedAction.Reset, "Unexpected; use an hkNonResettableObservableCollection");

            // If we are SettingSelection, we've already updated but we need to do notifies
            if (SettingSelection)
            {
                if (e.OldItems != null)
                {
                    _ObjectsToNotify.AddRange((e.OldItems as System.Collections.IEnumerable).Cast<VdbObjectViewModel>());
                }
                if (e.NewItems != null)
                {
                    System.Diagnostics.Debug.Assert(!e.NewItems.Contains(null), "We are not expecting null");
                    _ObjectsToNotify.AddRange((e.NewItems as System.Collections.IEnumerable).Cast<VdbObjectViewModel>());
                }
                return;
            }

            // Else, someone is using the SelectedObjects observable collection to affect our selection set
            using (new hkResetOnDispose(this, nameof(SettingSelection), true))
            {
                // Front-end
                // Deal with our SelectedObject changing
                if ((e.OldStartingIndex == 0) ||
                    (e.NewStartingIndex == 0))
                {
                    if (sender == SelectedObjects)
                    {
                        // If we are removing then the _PreviouslySelectedObject will be wrong since the removal occurred before this callback
                        // so the hkResetOnDispose's setting of SettingSelection won't have access to it.
                        if ((e.OldItems != null) && (e.OldItems.Count > 0))
                        {
                            _PreviousSelectedObject = (VdbObjectViewModel)e.OldItems[0];
                        }

                        SetRelatedSelectedFrontend(SelectedObject);
                    }
                }

                // Back-end
                if (e.OldItems != null)
                {
                    List<Havok.Vdb.VdbObject> displayMarkers = new List<Havok.Vdb.VdbObject>(e.OldItems.Count * 5);
                    foreach (VdbObjectViewModel unselectedObjectVm in e.OldItems)
                    {
                        if (unselectedObjectVm != null)
                        {
                            displayMarkers.AddRange(
                                unselectedObjectVm.CollectLeafwardDisplayables()
                                    .SelectMany(ovm => ovm.Object.ChildDisplayMarkers));
                        }
                    }
                    if (sender == SelectedObjects)
                    {
                        m_viewModel.Client.RenderSurface.ClearSelectedObjects(displayMarkers);
                        _ObjectsToNotify.AddRange((e.OldItems as System.Collections.IEnumerable).Cast<VdbObjectViewModel>());
                    }
                    else if (sender == HighlightedObjects)
                    {
                        m_viewModel.Client.RenderSurface.ClearHighlightedObjects(displayMarkers);
                    }
                    else
                    {
                        System.Diagnostics.Debug.Assert(false, "Unknown sender for HighlightedOrSelectedVdbObjects_CollectionChanged");
                    }
                }
                if (e.NewItems != null)
                {
                    System.Diagnostics.Debug.Assert(!e.NewItems.Contains(null), "We are not expecting null");
                    List<Havok.Vdb.VdbObject> displayMarkers = new List<Havok.Vdb.VdbObject>(e.NewItems.Count * 5);
                    foreach (VdbObjectViewModel selectedObjectVm in e.NewItems)
                    {
                        if (selectedObjectVm != null)
                        {
                            displayMarkers.AddRange(
                                selectedObjectVm.CollectLeafwardDisplayables()
                                    .SelectMany(ovm => ovm.Object.ChildDisplayMarkers));
                        }
                    }
                    if (sender == SelectedObjects)
                    {
                        m_viewModel.Client.RenderSurface.SelectObjects(displayMarkers);
                        _ObjectsToNotify.AddRange((e.NewItems as System.Collections.IEnumerable).Cast<VdbObjectViewModel>());
                    }
                    else if (sender == HighlightedObjects)
                    {
                        m_viewModel.Client.RenderSurface.HighlightObjects(displayMarkers);
                    }
                    else
                    {
                        System.Diagnostics.Debug.Assert(false, "Unknown sender for HighlightedOrSelectedVdbObjects_CollectionChanged");
                    }
                }

                // Extra notifies
                if (sender == SelectedObjects)
                {
                    NotifyPropertyChanged(nameof(SelectedObjectsCount));
                }
                else if (sender == HighlightedObjects)
                {
                    NotifyPropertyChanged(nameof(HighlightedObjectsCount));
                }
            }
        }
        void RenderSurface_SelectionChanged(Havok.Vdb.RenderSurface sender, Havok.Vdb.SelectionChangedEventArgs args)
        {
            // Always update our counts
            SelectedRenderObjectsCount = args.Selected.Count;
            HighlightedRenderObjectsCount = args.Highlighted.Count;

            // If we are SettingSelection, we've already updated
            if (SettingSelection) return;

            // Else, the back-end is triggering a selection change
            using (new hkResetOnDispose(this, nameof(SettingSelection), true))
            {
                if (args.RemovedFromSelected.Count > 0)
                {
                    ClearSelectedFrontend(args.RemovedFromSelected.Keys);
                }
                if (args.RemovedFromHighlighted.Count > 0)
                {
                    // If these are equal, we have a total clear (optimization)
                    if (args.Selected.Count == args.RemovedFromHighlighted.Count)
                    {
                        ClearHighlightedFrontend();
                    }
                    else
                    {
                        ClearHighlightedFrontend(args.RemovedFromHighlighted.Keys);
                    }
                }
                if (args.AddedToSelected.Count > 0)
                {
                    // If this is our first set, set our single-selection-properties
                    if (SelectedObjects.Count == 0)
                    {
                        ClearSelectedFrontend();
                        _SelectedRenderObject = FirstRenderObject(sender, args.AddedToSelected.Values);
                        _SelectedTreeItem = FirstTreeItem(_SelectedRenderObject, args.AddedToSelected.Keys);
                    }

                    // Add selected displayable and sub-displayable markers
                    IEnumerable<VdbObjectViewModel> displayables = AllDisplayables(args.AddedToSelected.Keys);
                    IEnumerable<VdbObjectViewModel> containingDisplayables = SetSelectedBackend(displayables);

                    // Note: O(n2), will be horrible if selected set grows large
                    SelectedObjects.AddRange(containingDisplayables.Where(ovm => !SelectedObjects.Contains(ovm)));
                }
                
                
                //if (args.AddedToHighlighted.Count > 0)
                //{}
            }
        }
        void SetSelected(VdbObjectMultiParentTreeItem selectedTreeItem)
        {
            using (new hkResetOnDispose(this, nameof(SettingSelection), true))
            {
                ClearSelected();
                if (selectedTreeItem != null)
                {
                    // Front-end
                    _SelectedTreeItem = selectedTreeItem;
                    VdbObjectViewModel objectVmTemplate = SetRelatedSelectedFrontend(selectedTreeItem);

                    // Back-end
                    if (objectVmTemplate != null)
                    {
                        VdbObjectViewModel containingDisplayable = SetSelectedBackend(objectVmTemplate);
                        if (containingDisplayable == objectVmTemplate) SetSelectedRenderObjectAndPOIInternal(objectVmTemplate.Object.ChildRenderObjects.FirstOrDefault());
                    }
                }
            }
        }
        VdbObjectViewModel SetRelatedSelectedFrontend(VdbObjectMultiParentTreeItem selectedTreeItem)
        {
            IEnumerable<VdbObjectModel> objectMs = selectedTreeItem.Objects;
            IEnumerable<VdbObjectViewModel> objectVms =
                objectMs
                    .Where(objectM => m_viewModel.VdbObjectViewModels.ContainsKey(objectM))
                    .Select(objectM => m_viewModel.VdbObjectViewModels[objectM]);
            VdbObjectViewModel objectVmTemplate = objectVms.FirstOrDefault();
            if (objectVmTemplate != null)
            {
                // Add objects first, then their containing displayables
                SelectedObjects.AddRange(objectVms);
                // Note: O(n2), will be horrible if selected set grows large
                SelectedObjects.AddRange(objectVms.Select(ovm => ovm.FindContainingDisplayable()).Where(ovm => (ovm != null) && !SelectedObjects.Contains(ovm)));
            }
            return objectVmTemplate;
        }
        void SetSelected(VdbObjectViewModel selectedObject)
        {
            using (new hkResetOnDispose(this, nameof(SettingSelection), true))
            {
                ClearSelected();
                if (selectedObject != null)
                {
                    // Front-end
                    SelectedObjects.Add(selectedObject);
                    SetRelatedSelectedFrontend(selectedObject);

                    // Back-end (and SelectedObjects front-end)
                    VdbObjectViewModel containingDisplayable = SetSelectedBackend(selectedObject);
                    if ((containingDisplayable != null) && (containingDisplayable != selectedObject)) SelectedObjects.Add(containingDisplayable);
                }
            }
        }
        VdbObjectViewModel SetSelectedBackend(VdbObjectViewModel selectedObject)
        {
            VdbObjectViewModel containingDisplayable = selectedObject.FindContainingDisplayable();
            if (containingDisplayable != null)
            {
                m_viewModel.Client.RenderSurface.SelectObjects(
                    new List<Havok.Vdb.VdbObject>(
                        containingDisplayable.CollectLeafwardDisplayables()
                            .SelectMany(ovm => ovm.Object.ChildDisplayMarkers)));
            }
            return containingDisplayable;
        }
        IEnumerable<VdbObjectViewModel> SetSelectedBackend(IEnumerable<VdbObjectViewModel> selectedObjects)
        {
            IEnumerable<VdbObjectViewModel> containingDisplayables = Enumerable.Empty<VdbObjectViewModel>();
            if ((selectedObjects != null) && (selectedObjects.Count() > 0))
            {
                containingDisplayables =
                    selectedObjects.Select(ovm => ovm.FindContainingDisplayable()).Where(ovm => (ovm != null));
                m_viewModel.Client.RenderSurface.SelectObjects(
                    new List<Havok.Vdb.VdbObject>(
                        containingDisplayables
                            .SelectMany(ovm => ovm.CollectLeafwardDisplayables())
                            .SelectMany(ovm => ovm.Object.ChildDisplayMarkers)));
            }
            return containingDisplayables;
        }
        void SetRelatedSelectedFrontend(VdbObjectViewModel selectedObject)
        {
            if ((selectedObject == null) ||
                (selectedObject.Object == null) ||
                !m_viewModel.VdbObjectTree.TryGetValue(selectedObject.Object.Id, out _SelectedTreeItem))
            {
                _SelectedTreeItem = null;
            }
            SetSelectedRenderObjectAndPOIInternal(((selectedObject != null) && (selectedObject.Object != null)) ? selectedObject.Object.ChildRenderObjects.FirstOrDefault() : null);
        }
        void SetSelected(Havok.Vdb.RenderObject renderObject)
        {
            using (new hkResetOnDispose(this, nameof(SettingSelection), true))
            {
                ClearSelected();
                if (renderObject != null)
                {
                    // Front-end
                    SetSelectedRenderObjectAndPOIInternal(renderObject);
                    IEnumerable<VdbObjectViewModel> displayables = SetRelatedSelectedFrontend(renderObject);

                    // Back-end
                    IEnumerable<VdbObjectViewModel> containingDisplayables = SetSelectedBackend(displayables);
                    if ((containingDisplayables == null) || (containingDisplayables.Count() == 0))
                    {
                        m_viewModel.Client.RenderSurface.SelectObject(renderObject);
                    }
                }
            }
        }
        void SetSelectedRenderObjectAndPOIInternal(Havok.Vdb.RenderObject renderObject)
        {
            // Note: the backend currently doesn't clear POIs on deselect, so we won't do it here either.
            _SelectedRenderObject = renderObject;
            if (renderObject != null)
            {
                m_viewModel.Client.RenderSurface.SetLocalPointOfInterest(renderObject, new Havok.Vdb.Vector(0, 0, 0));
            }
        }
        IEnumerable<VdbObjectViewModel> SetRelatedSelectedFrontend(Havok.Vdb.RenderObject renderObject)
        {
            IEnumerable<VdbObjectViewModel> displayables = (renderObject != null) ? AllDisplayables(new List<UInt64> { renderObject.Id }) : null;
            if ((displayables == null) ||
                (displayables.Count() == 0) ||
                !m_viewModel.VdbObjectTree.TryGetValue(displayables.First().Object.Id, out _SelectedTreeItem))
            {
                _SelectedTreeItem = null;
            }
            if ((displayables != null) && (displayables.Count() > 0))
            {
                SelectedObjects.AddRange(displayables);
            }
            else
            {
                SelectedObjects.Clear();
            }
            return displayables;
        }
        void ClearSelectedFrontend(IEnumerable<UInt64> ids = null)
        {
            if (ids != null)
            {
                if ((_SelectedTreeItem != null) && (_SelectedTreeItem.Objects.Count > 0) &&
                    _SelectedTreeItem.Objects[0].ChildDisplayMarkers.Any(ro => ids.Contains(ro.Id)))
                {
                    _SelectedTreeItem = null;
                }
                SelectedObjects.RemoveAll(o => ((o.Object != null) && o.Object.ChildDisplayMarkers.Any(ro => ids.Contains(ro.Id))));
                if ((_SelectedRenderObject != null) && ids.Contains(_SelectedRenderObject.Id))
                {
                    _SelectedRenderObject = null;
                }
            }
            else
            {
                _SelectedTreeItem = null;
                SelectedObjects.Clear();
                _SelectedRenderObject = null;
            }
        }
        void ClearHighlightedFrontend(IEnumerable<UInt64> ids = null)
        {
            if (ids != null)
            {
                HighlightedObjects.RemoveAll(o => ((o.Object != null) && o.Object.ChildDisplayMarkers.Any(ro => ids.Contains(ro.Id))));
            }
            else
            {
                HighlightedObjects.Clear();
            }
        }
        VdbObjectMultiParentTreeItem FirstTreeItem(Havok.Vdb.RenderObject firstRenderObject, ICollection<UInt64> ids)
        {
            UInt64 id;
            if (firstRenderObject != null)
            {
                id = firstRenderObject.Id;
            }
            else if (ids.Count > 0)
            {
                id = ids.First();
            }
            else
            {
                return null;
            }
            VdbObjectMultiParentTreeItem treeItem;
            if (m_viewModel.VdbObjectTree.TryGetValue(id, out treeItem))
            {
                return treeItem;
            }
            else
            {
                return null;
            }
        }
        Havok.Vdb.RenderObject FirstRenderObject(Havok.Vdb.RenderSurface surface, ICollection<Havok.Vdb.RenderObject> renderObjects)
        {
            Havok.Vdb.RenderObject renderObject = null;
            if (renderObjects.Count > 0)
            {
                renderObject =
                    renderObjects.FirstOrDefault(ro => surface.IsObjectGeometry(ro)) ??
                    renderObjects.First();
            }
            return renderObject;
        }
        List<VdbObjectModel> AllObjectMs(ICollection<UInt64> ids)
        {
            List<VdbObjectModel> objects = new List<VdbObjectModel>(ids.Count * 5);
            foreach (UInt64 id in ids)
            {
                VdbObjectMultiParentTreeItem selectedTreeItem;
                if (m_viewModel.VdbObjectTree.TryGetValue(id, out selectedTreeItem))
                {
                    objects.AddRange(selectedTreeItem.Objects);
                }
            }
            return objects;
        }
        IEnumerable<VdbObjectViewModel> AllDisplayables(ICollection<UInt64> ids)
        {
            List<VdbObjectModel> displayMarkers = AllObjectMs(ids);
            IEnumerable<VdbObjectViewModel> displayables =
                displayMarkers
                    .Select(objectM =>
                    {
                        VdbObjectModel displayableM = objectM.Parent;
                        VdbObjectViewModel displayableVm;
                        if ((displayableM != null) &&
                            m_viewModel.VdbObjectViewModels.TryGetValue(displayableM, out displayableVm))
                        {
                            return displayableVm;
                        }
                        else
                        {
                            return null;
                        }
                    })
                    .Where(objectVm => (objectVm != null));
            return displayables;
        }

        VdbViewModel m_viewModel;
        Delegate m_selectionChangedInteropDelegate;

        bool SettingSelection
        {
            get { lock(_SettingSelectionLock) return _SettingSelection; }
            set
            {
                lock (_SettingSelectionLock)
                {
                    if (_SettingSelection != value)
                    {
                        _SettingSelection = value;
                        if (value)
                        {
                            _PreviousSelectedTreeItem = SelectedTreeItem;
                            _PreviousSelectedObject = SelectedObject;
                            _PreviousSelectedRenderObject = SelectedRenderObject;
                        }
                        else
                        {
                            // Copy these in case someone tries to change selection *during* a notify
                            bool notifySelectedTreeItem = (_PreviousSelectedTreeItem != SelectedTreeItem);
                            VdbObjectViewModel selectedObject = SelectedObject;
                            bool notifySelectedObject = (_PreviousSelectedObject != selectedObject);
                            Havok.Vdb.RenderObject selectedRenderObject = SelectedRenderObject;
                            bool notifySelectedRenderObject = (_PreviousSelectedRenderObject != selectedRenderObject);
                            List<VdbObjectViewModel> objectsToNotify = new List<VdbObjectViewModel>(_ObjectsToNotify);
                            _ObjectsToNotify.Clear();

                            // Notify
                            // Note: order *matters* here
                            {
                                // 1) We do this first because *culled* ovms will add themselves to the visual tree
                                //    which enables code responding to changes to the "SelectedObject" to work with
                                //    a complete visual tree.
                                foreach (VdbObjectViewModel objectVm in objectsToNotify)
                                {
                                    if (objectVm != null) objectVm.NotifyIsSelectedChanged();
                                }

                                // 2) (2 & 3 could be reordered)
                                if (notifySelectedObject)
                                {
                                    // 2a) We do this before notify so that the panel is visible (and therefore visual tree
                                    //     objects are realized) for code responding to changes to the "SelectedObject"
                                    if (selectedObject != null)
                                    {
                                        // Show object inspection panel
                                        if (m_viewModel != null) m_viewModel.IsVdbObjectTreeViewShown = true;
                                    }

                                    NotifyPropertyChanged(nameof(SelectedObject));
                                }

                                // 3) (2 & 3 could be reordered)
                                if (notifySelectedRenderObject)
                                {
                                    // 3a) We do this before notify so that the panel is visible (and therefore visual tree
                                    //     objects are realized) for code responding to changes to the "SelectedRenderObject"
                                    if (selectedRenderObject != null)
                                    {
                                        // Show object inspection panel
                                        if (m_viewModel != null) m_viewModel.IsVdbObjectTreeViewShown = true;
                                    }

                                    NotifyPropertyChanged(nameof(SelectedRenderObject));
                                }

                                // 4) We do this last because it will have a "SelectedObject" and we want the visual
                                //    tree to be realized by the time this notify occurs (see #2).
                                if (notifySelectedTreeItem)
                                {
                                    NotifyPropertyChanged(nameof(SelectedTreeItem));
                                }
                            }
                        }
                    }
                }
            }
        }
        
        object _SettingSelectionLock = 0;
        bool _SettingSelection;
        List<VdbObjectViewModel> _ObjectsToNotify = new List<VdbObjectViewModel>();

        #endregion
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
