// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM		: WIN32 X64
// PRODUCT		: COMMON
// VISIBILITY	: CLIENT
//
// ------------------------------------------------------TKBMS v1.0

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.ComponentModel;
using static HavokVisualDebugger.VdbModel;

namespace HavokVisualDebugger
{
    public class VdbProcessesViewModel : ViewModelNotifyPropertyChanged
    {
        #region VdbProcessesViewModel Initialization
        public VdbProcessesViewModel(string category, VdbViewModel owner)
        {
            Category = category;
            Processes = new VdbProcessCollection(owner);
        }
        #endregion

        public VdbProcessCollection Processes { get; set; }
        public string Category { get; set; }

        public VdbProcessViewModel FindProcessByTag(int tag)
        {
            foreach (VdbProcessViewModel vm in Processes)
            {
                if (vm.Tag == tag)
                {
                    return vm;
                }
            }

            return null;
        }

        public VdbProcessViewModel FindProcessByName(string name)
        {
            foreach (VdbProcessViewModel vm in Processes)
            {
                if (vm.Name == name)
                {
                    return vm;
                }
            }

            return null;
        }
    }

    public class VdbProcessViewModel : ViewModelNotifyPropertyChanged
    {
        #region VdbProcessViewModel Initialization
        public VdbProcessViewModel(VdbClientInterop processHandler, int tag)
        {
            m_handler = processHandler;
            m_tag = tag;

            if (IsFileCommandViewer)
            {
                ProcessFileCommand = new ViewModelDelegateCommand(this, this.ProcessFileViewer);
            }
        }
        #endregion

        #region VdbProcessViewModel View Commands and Properties

        public string Name
        {
            get
            {
                VdbClientInterop process = m_handler.GetSubInterop(nameof(Havok.Vdb.ProcessHandler.GetProcess), m_tag);
                if (process.IsValid())
                {
                    return process.GetProperty<string>(nameof(Havok.Vdb.Process.Name));
                }
                else
                {
                    return "Process[" + m_tag + "]";
                }
            }
        }

        public string DisplayName
        {
            get
            {
                if (IsFileCommandViewer)
                {
                    return Name.Substring(1).Trim();
                }
                else
                {
                    return TrimCategoryPrefix(Name);
                }
            }
        }

        public string Category
        {
            get
            {
                return LookupStaticCategory(Name);
            }
        }

        private static string TrimCategoryPrefix(string viewerName)
        {
            List<string> prefixes = new List<string> { "NP ", "ND ", "AI - ", "Cloth " };
            int index = prefixes.FindIndex(x => viewerName.StartsWith(x));
            if (index >= 0)
            {
                return viewerName.Substring(prefixes[index].Length).Trim();
            }

            return viewerName;
        }

        private static string LookupStaticCategory(string viewerName)
        {
            string[] tokens = viewerName.Split(' ');
            string lowerViewerName = viewerName.ToLower();
            string lowerCasePrefix = tokens[0].ToLower();

            if (hkConstants.PrefixCategoryLookup.ContainsKey(lowerCasePrefix))
            {
                return hkConstants.PrefixCategoryLookup[lowerCasePrefix];
            }
            else if (hkConstants.ViewerCategoryLookup.ContainsKey(lowerViewerName))
            {
                return hkConstants.ViewerCategoryLookup[lowerViewerName];
            }
            else
            {
                return Properties.Resources.ViewerTooltip_CategoryCustom;
            }
        }

        public int Tag { get { return m_tag; } }

        public bool IsSelected
        {
            get
            {
                VdbClientInterop process = m_handler.GetSubInterop(nameof(Havok.Vdb.ProcessHandler.GetProcess), m_tag);
                if (process.IsValid())
                {
                    return process.GetProperty<bool>(nameof(Havok.Vdb.Process.Selected));
                }
                else
                {
                    return false;
                }
            }
            set
            {
                VdbClientInterop process = m_handler.GetSubInterop(nameof(Havok.Vdb.ProcessHandler.GetProcess), m_tag);
                if (process.IsValid())
                {
                    process.SetPropertyAsync(nameof(Havok.Vdb.Process.Selected), value);
                    // Note the below are handled in the selection changed event handler, which will call NotifySelectionChanged()
                    //VdbSettings.SetCollectionItemEnabled(Properties.Settings.VdbEnabledViewers, Name, value);
                    //NotifyPropertyChanged();
                }
                
            }
        }

        internal void NotifySelectionChanged()
        {
            NotifyPropertyChanged(nameof(IsSelected));
        }

        public bool IsImageValid
        {
            get
            {
                
                return !string.IsNullOrEmpty(ImagePath);
            }
        }

        public string ImagePath
        {
            get
            {
                if (VdbModel.ViewerInformation.VdbProcessInformationDictionary.ContainsKey(Name))
                {
                    return VdbModel.ViewerInformation.VdbProcessInformationDictionary[Name].ImagePath;
                }
                return null;
            }
        }

        public string TooltipDescription
        {
            get
            {
                if (VdbModel.ViewerInformation.VdbProcessInformationDictionary.ContainsKey(Name))
                {
                    return VdbModel.ViewerInformation.VdbProcessInformationDictionary[Name].LongDescription;
                }
                return HavokVisualDebugger.Properties.Resources.ViewerTooltip_NoDescription;
            }
        }

        public string ShortDescription
        {
            get
            {
                if (VdbModel.ViewerInformation.VdbProcessInformationDictionary.ContainsKey(Name))
                {
                    return VdbModel.ViewerInformation.VdbProcessInformationDictionary[Name].ShortDescription;
                }
                return "";
            }
        }

        public bool IsFileCommandViewer
        {
            get
            {
                return Name.StartsWith("*");
            }
        }

        public System.Windows.Input.ICommand ProcessFileCommand { get; private set; }

        internal void ProcessFileViewer(object o)
        {
            VdbClientInterop process = m_handler.GetSubInterop(nameof(Havok.Vdb.ProcessHandler.GetProcess), m_tag);
            if (process.IsValid())
            {
                process.SetPropertyAsync(nameof(Havok.Vdb.Process.Selected), true);
            }
        }

        #endregion

        #region VdbProcessViewModel Private Methods and Properties

        private VdbClientInterop m_handler;
        private int m_tag;

        #endregion
    }

    public class VdbProcessCollection : ObservableCollection<VdbProcessViewModel>, IDisposable
    {
        #region VdbProcessCollection Initialization
        public VdbProcessCollection(VdbViewModel owner)
        {
            m_owner = owner;

            // -1 so we get callbacks for frame we are initialized on
            m_currentFrame = -1;

            m_playbackReceivedDelegate =
                m_owner.Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).
                    ListenTo<Havok.Vdb.PlaybackCmdReceivedHandler, Havok.Vdb.PlaybackHandler, Havok.Vdb.PlaybackCmdReceivedArgs>(
                        nameof(Havok.Vdb.PlaybackHandler.PlaybackCmdReceived),
                        delegate (Havok.Vdb.PlaybackHandler sender, Havok.Vdb.PlaybackCmdReceivedArgs args)
                        {
                            if (args.TargetFrame > m_currentFrame)
                            {
                                // Process pending
                                foreach (VdbProcessViewModel process in m_processesAwaitingSelection)
                                {
                                    FireProcessEvent(process, ProcessEvent.ServerCreated);
                                }
                                foreach (VdbProcessViewModel process in m_processesAwaitingFirstStep)
                                {
                                    FireProcessEvent(process, ProcessEvent.FirstStep);
                                }

                                // Transfer through queues
                                m_processesAwaitingFirstStep.Clear();
                                m_processesAwaitingFirstStep.AddRange(m_processesAwaitingSelection);
                                m_processesAwaitingSelection.Clear();

                                // Advance
                                m_currentFrame = args.TargetFrame;
                            }
                        });

            m_clientConnectedDelegate =
                m_owner.Client.ListenTo<Havok.Vdb.ConnectionHandler, Havok.Vdb.Client>(
                    nameof(Havok.Vdb.Client.Connected),
                    delegate (Havok.Vdb.Client sender)
                    {
                        // -1 so we get callbacks for frame we are initialized on
                        m_currentFrame = -1;
                    });
        }

        public void Dispose()
        {
            if (m_playbackReceivedDelegate != null)
            {
                m_owner.Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).UnListenTo(nameof(Havok.Vdb.PlaybackHandler.PlaybackCmdReceived), m_playbackReceivedDelegate);
            }

            if (m_clientConnectedDelegate != null)
            {
                m_owner.Client.UnListenTo(nameof(Havok.Vdb.Client.Connected), m_clientConnectedDelegate);
            }

            this.Clear();
        }

        private Delegate m_playbackReceivedDelegate = null;
        private Delegate m_clientConnectedDelegate = null;
        private VdbViewModel m_owner;
        #endregion

        #region VdbProcessCollection Factory Methods and Commands

        public enum ProcessEvent
        {
            Added,
            ClientSelected,
            ServerCreated,
            FirstStep,
            Removed
        }

        public delegate void ProcessCallback(VdbProcessViewModel process, ProcessEvent evt);

        public void RegisterProcessCallback(string processName, ProcessCallback callback)
        {
            List<string> processNames;
            if (!m_processCallbackToNames.TryGetValue(callback, out processNames))
            {
                processNames = new List<string>();
                m_processCallbackToNames.Add(callback, processNames);
            }
            if (!processNames.Contains(processName))
            {
                processNames.Add(processName);
                List<ProcessCallback> callbacks;
                if (!m_processNameToCallbacks.TryGetValue(processName, out callbacks))
                {
                    callbacks = new List<ProcessCallback>();
                    m_processNameToCallbacks.Add(processName, callbacks);
                }
                System.Diagnostics.Debug.Assert(!callbacks.Contains(callback), "Callback registry inconsistency");
                callbacks.Add(callback);
            }
        }

        public void RegisterProcessCallback(ProcessCallback callback)
        {
            System.Diagnostics.Debug.Assert(!m_processCallbackToNames.ContainsKey(callback), "Automatically upgrading callback to generic for all");
            UnRegisterProcessCallback(callback);
            m_callbacksForAllProcesses.Add(callback);
        }

        public void UnRegisterProcessCallback(string processName, ProcessCallback callback)
        {
            System.Diagnostics.Debug.Assert(!m_callbacksForAllProcesses.Contains(callback), "Cannot unregister generic callback for specific processes");
            List<string> processNames;
            if (m_processCallbackToNames.TryGetValue(callback, out processNames))
            {
                System.Diagnostics.Debug.Assert(m_processNameToCallbacks.ContainsKey(processName), "Callback registry inconsistency");
                bool removed = m_processNameToCallbacks[processName].Remove(callback);
                removed &= processNames.Remove(processName);
                System.Diagnostics.Debug.Assert(removed, "Callback registry inconsistency");
                if (processNames.Count == 0)
                {
                    m_processCallbackToNames.Remove(callback);
                }
            }
        }

        public void UnRegisterProcessCallback(ProcessCallback callback)
        {
            if (m_callbacksForAllProcesses.Remove(callback))
            {
                System.Diagnostics.Debug.Assert(!m_processCallbackToNames.ContainsKey(callback), "Callback registry inconsistency");
                return;
            }
            List<string> processNames;
            if (m_processCallbackToNames.TryGetValue(callback, out processNames))
            {
                foreach (string processName in processNames)
                {
                    System.Diagnostics.Debug.Assert(m_processNameToCallbacks.ContainsKey(processName), "Callback registry inconsistency");
                    bool removed = m_processNameToCallbacks[processName].Remove(callback);
                    System.Diagnostics.Debug.Assert(removed, "Callback registry inconsistency");
                }
                m_processCallbackToNames.Remove(callback);
            }
        }

        #endregion

        #region VdbProcessCollection Private Methods and Properties

        protected override void ClearItems()
        {
            foreach (VdbProcessViewModel process in this)
            {
                OnProcessRemoved(process);
            }
            base.ClearItems();
        }

        protected override void OnCollectionChanged(NotifyCollectionChangedEventArgs e)
        {
            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                case NotifyCollectionChangedAction.Remove:
                case NotifyCollectionChangedAction.Replace:
                case NotifyCollectionChangedAction.Move:
                {
                    if (e.Action != NotifyCollectionChangedAction.Add)
                    {
                        foreach (VdbProcessViewModel process in e.OldItems)
                        {
                            OnProcessRemoved(process);
                        }
                    }
                    if (e.Action != NotifyCollectionChangedAction.Remove)
                    {
                        foreach (VdbProcessViewModel process in e.NewItems)
                        {
                            OnProcessAdded(process);
                        }
                    }
                    break;
                }
                // Note NotifyCollectionChangedAction.Reset happens after clearing items and doesn't provide them here, so we catch that in ClearItems() override
            }
            base.OnCollectionChanged(e);
        }

        private void OnProcessAdded(VdbProcessViewModel process)
        {
            process.PropertyChanged += Process_PropertyChanged;
            FireProcessEvent(process, ProcessEvent.Added);
        }

        private void OnProcessRemoved(VdbProcessViewModel process)
        {
            process.PropertyChanged -= Process_PropertyChanged;
            m_processesAwaitingSelection.RemoveAll(p => (p.Name == process.Name));
            m_processesAwaitingFirstStep.RemoveAll(p => (p.Name == process.Name));
            FireProcessEvent(process, ProcessEvent.Removed);
        }

        private void FireProcessEvent(VdbProcessViewModel process, ProcessEvent evt)
        {
            if (process != null)
            {
                List<ProcessCallback> callbacks;
                if (m_processNameToCallbacks.TryGetValue(process.Name, out callbacks))
                {
                    foreach (ProcessCallback callback in callbacks)
                    {
                        callback?.Invoke(process, evt);
                    }
                }
                foreach (ProcessCallback callback in m_callbacksForAllProcesses)
                {
                    callback?.Invoke(process, evt);
                }
            }
        }

        private void Process_PropertyChanged(Object sender, PropertyChangedEventArgs e)
        {
            VdbProcessViewModel process = (sender as VdbProcessViewModel);
            if ((e.PropertyName == nameof(VdbProcessViewModel.IsSelected)) && process.IsSelected)
            {
                m_processesAwaitingSelection.Add(process);
                FireProcessEvent(process, ProcessEvent.ClientSelected);
            }
        }

        private Dictionary<string, List<ProcessCallback>> m_processNameToCallbacks = new Dictionary<string, List<ProcessCallback>>();
        private Dictionary<ProcessCallback, List<string>> m_processCallbackToNames = new Dictionary<ProcessCallback, List<string>>();
        private List<ProcessCallback> m_callbacksForAllProcesses = new List<ProcessCallback>();
        private List<VdbProcessViewModel> m_processesAwaitingSelection = new List<VdbProcessViewModel>();
        private List<VdbProcessViewModel> m_processesAwaitingFirstStep = new List<VdbProcessViewModel>();
        private Int64 m_currentFrame = -1;

        #endregion
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
