// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM		: WIN32 X64
// PRODUCT		: COMMON
// VISIBILITY	: CLIENT
//
// ------------------------------------------------------TKBMS v1.0

#if DEBUG
// Disable all Object Inspection
//#define DISABLE_OBJECT_INSPECTION
// There are many legitimate reasons these hit, but it can be helpful in tracking down some problems.
// An example legitimate case would be an hknpBody getting removed from the world.
// 1) the world inspector viewer will call disconnectAll(bodyId) <- disconnecting from the world parent and all child ids including dispObjId
// 2) the shape viewer will call disconnectAll(dispObjId) <- which previously was connected to hknpBody
// However, both are necessary for the case of turning on/off each viewer independently.
//#define ASSERT_ON_REDUNDANT_OI_COMMANDS
// These are a misuse of the apis and can produce incorrect results; but aren't catestophic.
// Need to explore these in more detail. Currently, the inter-frame add/removes from the manifold system
// cause these to trigger, revisit after fixing that.
//#define ASSERT_ON_MISORDERED_OI_COMMANDS
// This can be helpful to break on id.
// Generally conditional breakpoints in C# are *very* slow.
//#define BREAK_ON_ID
#endif

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.ComponentModel;
using System.Linq;


namespace HavokVisualDebugger
{
    public class BreakOnId
    {
        const UInt64 Id = UInt64.MaxValue;
        const int DebugUid = -1;

        [System.Diagnostics.ConditionalAttribute("BREAK_ON_ID")]
        static public void Check(UInt64 idToCheck, UInt64 id = UInt64.MaxValue)
        {
            if (id != UInt64.MaxValue)
            {
                if (idToCheck == id)
                {
                    System.Diagnostics.Debug.Assert(false);
                }
            }
            else if (Id != UInt64.MaxValue)
            {
#if BREAK_ON_ID
                if (idToCheck == Id)
                {
                    System.Diagnostics.Debug.Assert(false);
                }
#endif
            }
        }
        [System.Diagnostics.ConditionalAttribute("BREAK_ON_ID")]
        static public void Check(int debugUidToCheck, int debugUid = -1)
        {
            if (debugUid != -1)
            {
                if (debugUidToCheck == debugUid)
                {
                    System.Diagnostics.Debug.Assert(false);
                }
            }
            else if (DebugUid != -1)
            {
#if BREAK_ON_ID
                if (debugUidToCheck == debugUid)
                {
                    System.Diagnostics.Debug.Assert(false);
                }
#endif
            }
        }
        [System.Diagnostics.ConditionalAttribute("BREAK_ON_ID")]
        static public void Check(VdbObjectMultiParentTreeItem item, UInt64 id = UInt64.MaxValue, int debugUid = -1)
        {
            if (item != null)
            {
#if BREAK_ON_ID
                Check(item.m_id, id);
#endif

#if DEBUG
                if (item.Objects.Count > 0) Check(item.Objects[0].DebugUid, debugUid);
#endif
            }
        }
        [System.Diagnostics.ConditionalAttribute("BREAK_ON_ID")]
        static public void Check(VdbObjectModel objectM, UInt64 id = UInt64.MaxValue, int debugUid = -1)
        {
            if (objectM != null)
            {
                Check(objectM.Id, id);
#if DEBUG
                Check(objectM.DebugUid, debugUid);
#endif
            }
        }
        [System.Diagnostics.ConditionalAttribute("BREAK_ON_ID")]
        static public void Check(VdbObjectViewModel objectVm, UInt64 id = UInt64.MaxValue, int debugUid = -1)
        {
            if (objectVm != null)
            {
                Check(objectVm.Object, id, debugUid);
            }
        }
    }

    public class VdbObjectMultiParentTreeModifiedEvent
    {
        public VdbObjectMultiParentTreeModifiedEvent(
            int numAdded,
            int numRemoved,
            int numConnected,
            int numDisconnected)
        {
            NumAdded = numAdded;
            NumRemoved = numRemoved;
            NumConnected = numConnected;
            NumDisconnected = numDisconnected;
        }
        public int NumAdded { get; private set; }
        public int NumRemoved { get; private set; }
        public int NumConnected { get; private set; }
        public int NumDisconnected { get; private set; }
    }
    public delegate void VdbObjectMultiParentTreeModifiedEventHandler(object sender, VdbObjectMultiParentTreeModifiedEvent e);

    public class VdbObjectMultiParentTreeItemsUpdatedEvent
    {
        public VdbObjectMultiParentTreeItemsUpdatedEvent(int numUpdated)
        {
            NumUpdated = numUpdated;
        }
        public int NumUpdated { get; private set; }
    }
    public delegate void VdbObjectMultiParentTreeItemsUpdatedEventHandler(object sender, VdbObjectMultiParentTreeItemsUpdatedEvent e);

    public class VdbObjectMultiParentTree :
        // SortedList<UInt64, VdbObjectMultiParentTreeItem>,
        Dictionary<UInt64, VdbObjectMultiParentTreeItem>,
        IDisposable
    {
        public event VdbObjectMultiParentTreeModifiedEventHandler TreeModified;
        public event VdbObjectMultiParentTreeItemsUpdatedEventHandler TreeItemsUpdated;

        public VdbObjectMultiParentTree(VdbViewModel viewModel)
        {
            m_viewModel = viewModel;
            m_rootTreeItem = new VdbObjectMultiParentTreeRoot(this, new ObservableCollection<VdbObjectModel>());
        }
        public void Dispose()
        {
            foreach (VdbObjectMultiParentTreeItem treeItem in Values)
            {
                treeItem.Dispose();
            }
            Clear();
            m_rootTreeItem = new VdbObjectMultiParentTreeRoot(this, new ObservableCollection<VdbObjectModel>());
        }

        public VdbObjectModel Root { get { return m_rootTreeItem.Root; } }

        public void AddObjects(Havok.Vdb.ObjectsAddedEvent evt)
        {
#if !DISABLE_OBJECT_INSPECTION
            int prevCount = Count;
            for (int i = 0; i < evt.Ids.Count; i++)
            {
                int tag = evt.Tag;
                UInt64 id = evt.Ids[i];
                AssertIdNotAdded(id);
                Havok.Vdb.VdbObject obj = evt.Objects[i];
                AssertIdConsistent(id, obj);
                VdbObjectMultiParentTreeItem objectTreeItem = EnsureObject(id);
                objectTreeItem.Added(tag, obj);
            }
            if (prevCount != Count) TreeModified?.Invoke(this, new VdbObjectMultiParentTreeModifiedEvent(Count - prevCount, 0, 0, 0));
#endif
        }
        public void UpdateObjects(Havok.Vdb.ObjectsUpdatedEvent evt)
        {
#if !DISABLE_OBJECT_INSPECTION
            int updated = 0;
            for (int i = 0; i < evt.Ids.Count; i++)
            {
                UInt64 id = evt.Ids[i];
                Havok.Vdb.VdbObject obj = evt.Objects[i];

                VdbObjectMultiParentTreeItem objectTreeItem;
                if (TryGetValue(id, out objectTreeItem))
                {
                    AssertIdConsistent(id, obj);
                    objectTreeItem.Updated(obj);
                    updated++;
                }
                else
                {
                    AssertIdNotAdded();
                }
            }
            if (updated > 0) TreeItemsUpdated?.Invoke(this, new VdbObjectMultiParentTreeItemsUpdatedEvent(updated));
#endif
        }
        public void RemoveObjects(Havok.Vdb.ObjectsRemovedEvent evt)
        {
#if !DISABLE_OBJECT_INSPECTION
            int prevCount = Count;
            for (int i = 0; i < evt.Ids.Count; i++)
            {
                UInt64 id = evt.Ids[i];

                VdbObjectMultiParentTreeItem objectTreeItem;
                if (TryGetValue(id, out objectTreeItem))
                {
                    objectTreeItem.Removed();
                }
                else
                {
                    AssertIdNotAdded();
                }
            }
            if (prevCount != Count) TreeModified?.Invoke(this, new VdbObjectMultiParentTreeModifiedEvent(0, Count - prevCount, 0, 0));
#endif
        }

        public void ConnectObject(Havok.Vdb.ObjectConnectedEvent evt)
        {
#if !DISABLE_OBJECT_INSPECTION
            int prevCount = Count;
            {
                VdbObjectMultiParentTreeItem objectTreeItem = EnsureObject(evt.FromId);
                objectTreeItem.Connected(new List<UInt64>(evt.ToIds).ConvertAll(
                    idToConvert =>
                    {
                        return EnsureObject(idToConvert);
                    }));
            }
            if (prevCount != Count) TreeModified?.Invoke(this, new VdbObjectMultiParentTreeModifiedEvent(0, 0, Count - prevCount, 0));
#endif
        }
        public void DisconnectObject(Havok.Vdb.ObjectDisconnectedEvent evt)
        {
#if !DISABLE_OBJECT_INSPECTION
            int prevCount = Count;
            {
                VdbObjectMultiParentTreeItem objectTreeItem;
                if (TryGetValue(evt.FromId, out objectTreeItem))
                {
                    objectTreeItem.Disconnected(new List<UInt64>(evt.ToIds).ConvertAll(
                        idToConvert =>
                        {
                            VdbObjectMultiParentTreeItem itemFromId;
                            if (TryGetValue(idToConvert, out itemFromId))
                            {
                                return itemFromId;
                            }
                            AssertIdNotAdded();
                            return null;
                        }));
                }
                else
                {
                    AssertIdNotAdded();
                }
            }
            if (prevCount != Count) TreeModified?.Invoke(this, new VdbObjectMultiParentTreeModifiedEvent(0, 0, 0, Count - prevCount));
#endif
        }
        public void SetObjectConnectivity(Havok.Vdb.ObjectSetConnectivityEvent evt)
        {
#if !DISABLE_OBJECT_INSPECTION
            int prevCount = Count;
            {
                VdbObjectMultiParentTreeItem objectTreeItem = EnsureObject(evt.FromId);
                objectTreeItem.SetConnected(new List<UInt64>(evt.ToIds).ConvertAll(
                    idToConvert =>
                    {
                        return EnsureObject(idToConvert);
                    }));
            }
            int diff = (Count - prevCount);
            if (diff != 0) TreeModified?.Invoke(this, new VdbObjectMultiParentTreeModifiedEvent(0, 0, diff > 0 ? diff : 0, diff < 0 ? diff : 0));
#endif
        }

        private VdbObjectMultiParentTreeItem EnsureObject(UInt64 id)
        {
            VdbObjectMultiParentTreeItem treeItem;
            if (!TryGetValue(id, out treeItem))
            {
                treeItem = (id == 0) ? m_rootTreeItem : new VdbObjectMultiParentTreeItem(this, new VdbObjectModel(m_viewModel, -1, id, null));
                Add(id, treeItem);
            }
            return treeItem;
        }
        private void EnsureObjects(IList<UInt64> ids)
        {
            (new List<UInt64>(ids)).ForEach(
                id =>
                {
                    EnsureObject(id);
                });
        }

        private class VdbObjectMultiParentTreeRoot : VdbObjectMultiParentTreeItem
        {
            private class VdbObjectModelRoot : VdbObjectModel
            {
                public VdbObjectModelRoot(VdbViewModel viewModel, ObservableCollection<VdbObjectModel> rootObjectMs) :
                    base(viewModel, -1, 0, null)
                {
                    // Do normal add child mechanism.
                    // This is for the purposes of setting rootObjects correctly; we will be pointing to them
                    // directly for children backing store.
                    foreach (VdbObjectModel rootObjectM in rootObjectMs)
                    {
                        AddChild(rootObjectM);
                    }

                    // Set the backing store
                    {
                        m_childMsRW = rootObjectMs;
                        Children = new ReadOnlyObservableCollection<VdbObjectModel>(m_childMsRW);
                    }

                    
                    {
                        //IsRootObjectContainer = true;
                        //IsVisible = false;
                    }
                }
            }

            public VdbObjectMultiParentTreeRoot(VdbObjectMultiParentTree tree, ObservableCollection<VdbObjectModel> rootObjectMs) :
                base(tree, new VdbObjectModelRoot(tree.m_viewModel, rootObjectMs))
            {}

            public VdbObjectModel Root { get { return m_objects[0]; } }
        }

        [System.Diagnostics.ConditionalAttribute("ASSERT_ON_REDUNDANT_OI_COMMANDS")]
        private void AssertIdNotAdded()
        {
            System.Diagnostics.Debug.Assert(false, "Id has not been added");
        }
        [System.Diagnostics.ConditionalAttribute("ASSERT_ON_REDUNDANT_OI_COMMANDS")]
        private void AssertIdNotAdded(UInt64 id)
        {
            if (ContainsKey(id))
            {
                System.Diagnostics.Debug.Assert(false, "Id has already been added");
            }
        }
        private void AssertIdConsistent(UInt64 id, Havok.Vdb.VdbObject obj)
        {
            if (id != obj.Id)
            {
                System.Diagnostics.Debug.Assert(false, "Ids are inconsistent");
            }
        }

        internal VdbViewModel m_viewModel;
        private VdbObjectMultiParentTreeRoot m_rootTreeItem;
    }

    
    
    
    public class VdbObjectMultiParentTreeItem : IDisposable
    {
        public VdbObjectMultiParentTreeItem(VdbObjectMultiParentTree tree, VdbObjectModel firstObjectM)
        {
            m_tree = tree;
            m_objects = new List<VdbObjectModel>();
            m_objects.Add(firstObjectM);
            m_id = firstObjectM.Id;
        }
        private void ModelDisposed()
        {
            // Never dispose of the root tree item, which is fixed for future connections.
            if (m_id != 0)
            {
                System.Diagnostics.Debug.Assert(m_tree.ContainsKey(m_id) && (m_tree[m_id] == this));
                m_tree.Remove(m_id);
            }
            Dispose();
        }
        public void Dispose()
        {
            // Never dispose of the root tree item, which is fixed for future connections.
            if (m_id != 0)
            {
                foreach (VdbObjectModel objectM in m_objects)
                {
                    objectM.Dispose();
                }
                m_objects.Clear();
            }
        }
        public int PathsToRoot { get { return m_objects.Count; } }
        public ReadOnlyCollection<VdbObjectModel> Objects { get { return m_objects.AsReadOnly(); } }

        public void Added(int tag, Havok.Vdb.VdbObject obj)
        {
            foreach (VdbObjectModel objectM in m_objects)
            {
                AssertNotAdded(objectM);
                objectM.Tag = tag;
                objectM.Object = obj;
            }
        }
        public void Updated(Havok.Vdb.VdbObject obj)
        {
            foreach (VdbObjectModel objectM in m_objects)
            {
                AssertAdded(objectM);
                objectM.Object = obj;
            }
        }
        public void Removed()
        {
            foreach (VdbObjectModel objectM in m_objects)
            {
                AssertAdded(objectM);
                objectM.Object = null;
            }
            DisposeAbandoned();
        }

        public void Connected(VdbObjectMultiParentTreeItem treeItem)
        {
            if (treeItem != null)
            {
                System.Diagnostics.Debug.Assert(this != treeItem);
                AssertNotConnected(treeItem);
                foreach (VdbObjectModel objectM in m_objects)
                {
                    objectM.AddChild(treeItem.NewObjectConnection());
                }
                AssertConnected(treeItem);
            }
        }
        public void Connected(IList<VdbObjectMultiParentTreeItem> treeItems)
        {
            foreach (VdbObjectMultiParentTreeItem treeItem in treeItems)
            {
                Connected(treeItem);
            }
        }
        public void SetConnected(List<VdbObjectMultiParentTreeItem> treeItems)
        {
            // Disconnect
            
            List<VdbObjectModel> childMs = new List<VdbObjectModel>();
            Dictionary<UInt64, VdbObjectMultiParentTreeItem> disconnectedTreeItems = new Dictionary<UInt64, VdbObjectMultiParentTreeItem>();
            foreach (VdbObjectModel objectM in m_objects)
            {
                for (int i = objectM.Children.Count - 1; i >= 0; i--)
                {
                    VdbObjectModel childM = objectM.Children[i];
                    if (treeItems.Find(t => t.m_id == childM.Id) == null)
                    {
                        UInt64 childIdToRemove = childM.Id;
                        objectM.RemoveChildAt(i);
                        if (!disconnectedTreeItems.ContainsKey(childIdToRemove))
                        {
                            VdbObjectMultiParentTreeItem childTreeItem;
                            if (m_tree.TryGetValue(childIdToRemove, out childTreeItem))
                            {
                                System.Diagnostics.Debug.Assert(this != childTreeItem);
                                disconnectedTreeItems.Add(childIdToRemove, childTreeItem);
                            }
                            else
                            {
                                
                                
                            }
                        }
                    }
                    else
                    {
                        childMs.Add(childM);
                    }
                }
            }
            // Connect
            foreach (VdbObjectMultiParentTreeItem treeItem in treeItems)
            {
                if (childMs.Find(om => om.Id == treeItem.m_id) == null)
                {
                    Connected(treeItem);
                }
            }
            // Purge abandoned
            DisposeAbandoned();
            foreach (VdbObjectMultiParentTreeItem treeItem in disconnectedTreeItems.Values)
            {
                treeItem.DisposeAbandoned();
            }
        }
        public void Disconnected(VdbObjectMultiParentTreeItem treeItem)
        {
            if (treeItem != null)
            {
                System.Diagnostics.Debug.Assert(this != treeItem);
                AssertConnected(treeItem);
                List<VdbObjectModel> candidateMsForRemoval = new List<VdbObjectModel>(treeItem.m_objects);
                foreach (VdbObjectModel objectM in m_objects)
                {
                    for (int i = candidateMsForRemoval.Count - 1; i >= 0; i--)
                    {
                        VdbObjectModel candidateM = candidateMsForRemoval[i];
                        if (objectM.RemoveChild(candidateM))
                        {
                            candidateMsForRemoval.RemoveAt(i);
                            break;
                        }
                    }
                }
                DisposeAbandoned();
                treeItem.DisposeAbandoned();
                AssertNotConnected(treeItem);
            }
        }
        public void Disconnected(IList<VdbObjectMultiParentTreeItem> treeItems)
        {
            foreach (VdbObjectMultiParentTreeItem treeItem in treeItems)
            {
                Disconnected(treeItem);
            }
        }

        
        private void DisposeAbandoned()
        {
            if (m_id != 0)
            {
                for (int i = m_objects.Count - 1; i >= 0; i--)
                {
                    VdbObjectModel objectM = m_objects[i];
                    bool isAbandoned = (objectM.Parent == null) && (objectM.Children.Count == 0);
                    bool isSoleRemainingDataSource = (m_objects.Count == 1) && (m_objects[0].Object != null);
                    if (isAbandoned && !isSoleRemainingDataSource)
                    {
                        m_objects[i].Dispose();
                        m_objects.RemoveAt(i);
                    }
                }
                if (m_objects.Count == 0)
                {
                    ModelDisposed();
                }
            }
        }
        private VdbObjectModel NewObjectConnection()
        {
            System.Diagnostics.Debug.Assert(m_objects.Count > 0);
            if (m_objects[0].Parent == null)
            {
                return m_objects[0];
            }
            else
            {
                VdbObjectModel newConnection = new VdbObjectModel(m_objects[0]);
                m_objects.Add(newConnection);
                return newConnection;
            }
        }

        [System.Diagnostics.Conditional("ASSERT_ON_REDUNDANT_OI_COMMANDS")]
        private void AssertConnected(VdbObjectMultiParentTreeItem treeItem)
        {
            foreach (VdbObjectModel objectM in m_objects)
            {
                int connectedCount = 0;
                foreach (VdbObjectModel childM in treeItem.m_objects)
                {
                    if (childM.Parent == objectM)
                    {
                        connectedCount++;
                    }
                }
                if (connectedCount != 1)
                {
                    System.Diagnostics.Debug.Assert(false, "Tree item is not uniquely connected");
                }
            }
        }
        [System.Diagnostics.Conditional("ASSERT_ON_REDUNDANT_OI_COMMANDS")]
        private void AssertNotConnected(VdbObjectMultiParentTreeItem treeItem)
        {
            foreach (VdbObjectModel objectM in m_objects)
            {
                bool connected = false;
                foreach (VdbObjectModel childM in treeItem.m_objects)
                {
                    if (childM.Parent == objectM)
                    {
                        connected = true;
                        break;
                    }
                }
                if (connected)
                {
                    System.Diagnostics.Debug.Assert(false, "Tree item is connected");
                }
            }
        }
        [System.Diagnostics.Conditional("ASSERT_ON_MISORDERED_OI_COMMANDS")]
        static private void AssertConnected(VdbObjectModel parentM, VdbObjectModel childM)
        {
            System.Diagnostics.Debug.Assert(parentM.Children.Contains(childM), "Child om is not connected");
        }
        [System.Diagnostics.Conditional("ASSERT_ON_MISORDERED_OI_COMMANDS")]
        static private void AssertNotConnected(VdbObjectModel parentM, VdbObjectModel childM)
        {
            System.Diagnostics.Debug.Assert(!parentM.Children.Contains(childM), "Child om is connected");
        }
        [System.Diagnostics.Conditional("ASSERT_ON_MISORDERED_OI_COMMANDS")]
        static private void AssertAdded(VdbObjectModel objectM)
        {
            System.Diagnostics.Debug.Assert(objectM.Object != null, "Object is not added");
        }
        [System.Diagnostics.Conditional("ASSERT_ON_MISORDERED_OI_COMMANDS")]
        static private void AssertNotAdded(VdbObjectModel objectM)
        {
            System.Diagnostics.Debug.Assert(objectM.Object == null, "Object is added");
        }
#if false // useful stop on id methods for debugging
        [System.Diagnostics.Conditional("DEBUG")]
        static private void StopOnId(UInt64 id)
        {
            System.Diagnostics.Debug.Assert((s_stopId == 0) || (s_stopId != id), "Stop");
        }
        [System.Diagnostics.Conditional("DEBUG")]
        static private void StopOnId(VdbObjectModel objectM)
        {
            System.Diagnostics.Debug.Assert((s_stopId == 0) || (objectM == null) || (s_stopId != objectM.Id), "Stop");
        }
        [System.Diagnostics.Conditional("DEBUG")]
        static private void StopOnId(Havok.Vdb.VdbObject vdbObject)
        {
            System.Diagnostics.Debug.Assert((s_stopId == 0) || (vdbObject == null) || (s_stopId != vdbObject.Id), "Stop");
        }
        [System.Diagnostics.Conditional("DEBUG")]
        static private void StopOnId(VdbObjectMultiParentTreeItem treeItem)
        {
            System.Diagnostics.Debug.Assert((s_stopId == 0) || (treeItem == null) || (s_stopId != treeItem.m_id), "Stop");
        }
#if DEBUG
        static private UInt64 s_stopId = 0;
#endif
#endif

        private VdbObjectMultiParentTree m_tree;
#if BREAK_ON_ID
        public UInt64 m_id;
#else
        private UInt64 m_id;
#endif
        protected List<VdbObjectModel> m_objects;
    }

    public class VdbObjectModel : IDisposable
    {
        public event Action Disposed;
        public event Action<VdbObjectModel, bool> VisibilityChanged;
        public event Action<VdbObjectModel, Havok.Vdb.VdbObject> ObjectSet;
        public event Action<VdbObjectModel, Havok.Vdb.VdbObject> ObjectUpdated;
        public event Action<VdbObjectModel, Havok.Vdb.VdbObject> ObjectCleared;
        public event Action<VdbObjectModel, Havok.Vdb.VdbObject> ObjectTypeChanged;

        
        
        
        
        
        
        
        
        
        public VdbObjectModel(VdbObjectModel templateM)
        {
            m_viewModel = templateM.m_viewModel;
            m_tag = templateM.m_tag;
            m_id = templateM.m_id;
            _IObject = templateM._IObject;
            _Flags = templateM._Flags;
            System.Diagnostics.Debug.Assert((_IObject == null) || (_IObject.Id == m_id));
            Children = new ReadOnlyObservableCollection<VdbObjectModel>(m_childMsRW);
            foreach (VdbObjectModel childTemplate in templateM.Children)
            {
                AddChild(new VdbObjectModel(childTemplate));
            }
        }
        public VdbObjectModel(VdbViewModel viewModel, int tag, UInt64 id, Havok.Vdb.VdbObject iobject)
        {
            m_viewModel = viewModel;
            m_tag = tag;
            m_id = id;
            _IObject = iobject;
            if (iobject != null)
            {
                _Flags |= iobject.Flags;
            }
            System.Diagnostics.Debug.Assert((_IObject == null) || (_IObject.Id == m_id));
            Children = new ReadOnlyObservableCollection<VdbObjectModel>(m_childMsRW);
        }

        public void Dispose()
        {
            Disposed?.Invoke();
        }

        public void AddChild(VdbObjectModel childM)
        {
            // Some issue with the AddChild(B1) during SetConnnectivity not hooking up the B1 model to B1 view model during this call...
            System.Diagnostics.Debug.Assert(childM.Parent == null);
            childM.Parent = this;
            m_childMsRW.Add(childM);
        }
        public bool RemoveChild(VdbObjectModel childM)
        {
            if (m_childMsRW.Remove(childM))
            {
                childM.Parent = null;
                return true;
            }
            return false;
        }
        public bool RemoveChildAt(int idx)
        {
            if (idx < m_childMsRW.Count)
            {
                m_childMsRW[idx].Parent = null;
                m_childMsRW.RemoveAt(idx);
                return true;
            }
            return false;
        }

        internal VdbObjectViewModel CreateTreeView(
            VdbViewModel owner,
            List<SearchFilter> filters,
            Dictionary<VdbObjectModel, VdbObjectViewModel> treeObjectVms,
            out List<VdbObjectViewModel> matchingTreeObjectVms,
            bool collectRelatedNodes = true)
        {
            // Note: we cannot avoid calling CreateTreeViewRecursive even if we don't have filters because
            // the function will enforce the maxNodeCount as well.  Even though it would be slightly faster to do:
            // return new VdbObjectViewModel(owner, this);
            matchingTreeObjectVms = new List<VdbObjectViewModel>();
            return CreateTreeViewRecursive(owner, filters, null, null, treeObjectVms, matchingTreeObjectVms, collectRelatedNodes);
        }

        private VdbObjectViewModel CreateTreeViewRecursive(
            VdbViewModel owner,
            List<SearchFilter> filters,
            List<SearchFilter.FilterResult> parentResults,
            List<SearchFilter.FilterResult> returnResults,
            Dictionary<VdbObjectModel, VdbObjectViewModel> treeObjectVms,
            List<VdbObjectViewModel> matchingTreeObjectVms,
            bool collectRelatedNodes)
        {
            // Get our results
            List<SearchFilter.FilterResult> results;
            {
                if (IsVisible)
                {
                    
                    results = SearchFilter.DoFiltersMatch(filters, GetPropertyValuesForPath);
                    if (returnResults != null) returnResults.AddRange(results);
                }
                else
                {
                    // If we aren't visible, don't waste cpu on property matching
                    results = filters.ConvertAll(f => SearchFilter.FilterResult.False);
                    if (returnResults != null) returnResults.AddRange(results);
                    return null;
                }
            }

            // DFS - gather children
            
            
            
            List<VdbObjectViewModel> childObjectVms = new List<VdbObjectViewModel>();
            List<List<SearchFilter.FilterResult>> childResults = new List<List<SearchFilter.FilterResult>>(m_childMsRW.Count);
            for (int i = 0; i < m_childMsRW.Count; i++)
            {
                VdbObjectModel childM = m_childMsRW[i];
                childResults.Add(new List<SearchFilter.FilterResult>());

                VdbObjectViewModel childObjectVm = childM.CreateTreeViewRecursive(
                    owner,
                    filters,
                    results,
                    childResults[i],
                    treeObjectVms,
                    matchingTreeObjectVms,
                    collectRelatedNodes);

                if (childObjectVm != null) childObjectVms.Add(childObjectVm);
            }

            // At this point parent, self and child relationships should be evaluated.  Combine all of them.
            bool filterExists = (filters.Count > 0);
            bool filterMatch = (!filterExists || SearchFilter.CombineFilterResults(filters, results, parentResults, (childResults.Count == 0) ? null : childResults));

            // Include all our children if we have a decendent that matched and we *aren't* a container.
            bool decendantMatched = (childObjectVms.Count > 0);
            if (collectRelatedNodes && !IsRootObjectContainer && (filterMatch || decendantMatched))
            {
                childObjectVms = CollectDescendants(owner, treeObjectVms);
            }

            // If any children were included, should include the parent, otherwise check if filter applies to self
            if (filterMatch || decendantMatched)
            {
                VdbObjectViewModel treeObjectVm = new VdbObjectViewModel(owner, this, childObjectVms)
                {
                    IsEmphasized = (filterExists && filterMatch),
                    IsExpanded = (IsRootObjectContainer || (decendantMatched && filterExists)),
                    IsFiltered = filterExists
                };
                try
                {
                    treeObjectVms.Add(this, treeObjectVm);
                    if (treeObjectVm.IsEmphasized) matchingTreeObjectVms.Add(treeObjectVm);
                    return treeObjectVm;
                }
                catch
                {
                    // treeObjectVms should not contain "this"...but avoid a crash if we have a programming error.
                    System.Diagnostics.Debug.Assert(false);
                    treeObjectVm.Dispose();
                    return null;
                }
            }
            else
            {
                return null;
            }
        }
        private List<VdbObjectViewModel> CollectDescendants(
            VdbViewModel owner,
            Dictionary<VdbObjectModel, VdbObjectViewModel> treeObjectVms)
        {
            List<VdbObjectViewModel> allChildVms = new List<VdbObjectViewModel>();
            for (int i = 0; i < m_childMsRW.Count; i++)
            {
                VdbObjectModel childM = m_childMsRW[i];
                if (childM.IsVisible)
                {
                    VdbObjectViewModel childVm;
                    if (!treeObjectVms.TryGetValue(childM, out childVm))
                    {
                        System.Diagnostics.Debug.Assert(!childM.IsRootObjectContainer, "Containers should have already been added in depth-first tree recursion");
                        List<VdbObjectViewModel> decendentVms = childM.CollectDescendants(owner, treeObjectVms);
                        childVm = new VdbObjectViewModel(owner, childM, decendentVms)
                        {
                            IsEmphasized = false,
                            IsExpanded = false,
                            IsFiltered = true
                        };
                        try
                        {
                            treeObjectVms.Add(childM, childVm);
                        }
                        catch
                        {
                            // treeObjectVms should not contain "childM"...but avoid a crash if we have a programming error.
                            System.Diagnostics.Debug.Assert(false);
                            childVm.Dispose();
                            continue;
                        }
                    }
                    allChildVms.Add(childVm);
                }
            }
            return allChildVms;
        }

        private static void GatherAllPropertyValuesForName(object cliObject, List<string> currentDir, string[] propPath, List<object> matchingPropertiesOut)
        {
            int pathCompare = SearchFilter.PathCompare(currentDir, propPath);
            if (pathCompare > 0)
            {
                return;
            }

            // If we are *at* our prop, get the propname for matching at this level
            string propName = (pathCompare == 0) ? SearchFilter.NameFromPath(propPath) : null;

            // Havok.Vdb.VdbObject processing
            Havok.Vdb.VdbObject vdbObject = (cliObject as Havok.Vdb.VdbObject);
            if (vdbObject != null)
            {
                if (propName != null)
                {
                    // Allow a special case for the Id of the Havok.Vdb.VdbObject type
                    if (propName.Equals("Id", StringComparison.OrdinalIgnoreCase))
                    {
                        matchingPropertiesOut.Add(vdbObject.Id);
                    }
                }

                // Note: no return because this is *also* an Havok.Vdb.Object
                // return;
            }

            // Havok.Vdb.Object processing
            Havok.Vdb.Object hkObject = (cliObject as Havok.Vdb.Object);
            if (hkObject != null)
            {
                if (propName != null)
                {
                    // Allow a special case for the type of the Havok.Vdb.Object type
                    if (propName.Equals("TypeName", StringComparison.OrdinalIgnoreCase))
                    {
                        matchingPropertiesOut.Add(hkObject.TypeName);
                    }
                }

                // Find properties by name in properties set; recursing as necessory for embedded records.
                
                foreach (Havok.Vdb.IReadOnlyProperty prop in hkObject.Properties.Values)
                {
                    if ((propName != null) && prop.Name.Equals(propName, StringComparison.OrdinalIgnoreCase))
                    {
                        matchingPropertiesOut.Add(prop.Value);
                    }
                    else
                    {
                        SearchFilter.PushPathDir(currentDir, prop.Name);
                        GatherAllPropertyValuesForName(prop.Value, currentDir, propPath, matchingPropertiesOut);
                        SearchFilter.PopPathDir(currentDir);
                    }
                }

                return;
            }

            object[] cliObjectsArray = (cliObject as object[]);
            if (cliObjectsArray != null)
            {
                for (int i = 0; i < cliObjectsArray.Length; i++)
                {
                    object cliSubObject = cliObjectsArray[i];
                    if ((propName != null) && i.ToString().Equals(propName, StringComparison.OrdinalIgnoreCase))
                    {
                        matchingPropertiesOut.Add(cliSubObject);
                    }
                    else
                    {
                        SearchFilter.PushPathDir(currentDir, i);
                        GatherAllPropertyValuesForName(cliSubObject, currentDir, propPath, matchingPropertiesOut);
                        SearchFilter.PopPathDir(currentDir);
                    }
                }

                return;
            }
        }

        public object[] GetPropertyValuesForPath(string[] propPath)
        {
            // Recurse to pick up any embedded records
            List<string> currentDir = new List<string>();
            List<object> matchingProperties = new List<object>();
            // Push the root dir
            SearchFilter.PushPathDir(currentDir, "");
            GatherAllPropertyValuesForName(_IObject, currentDir, propPath, matchingProperties);
            return matchingProperties.ToArray();
        }

        public Havok.Vdb.VdbObject Object
        {
            get { return _IObject; }
            set
            {
                if (_IObject != value)
                {
                    bool wasNull;
                    IntPtr prevNativeType;
                    if (_IObject != null)
                    {
                        wasNull = false;
                        prevNativeType = _IObject.Type;
                    }
                    else
                    {
                        wasNull = true;
                        prevNativeType = IntPtr.Zero;
                    }
                    bool wasVisible = IsVisible;

                    if (value != null)
                    {
                        if ((value.Flags & Havok.Vdb.ObjectFlags.Update) == 0)
                        {
                            _Flags = value.Flags;
                        }
                        else
                        {
                            _Flags |= value.Flags;
                        }
                    }
                    // else, we can't set flags to None in this case because a connection can prevent
                    // the object from clearing the cache, so that if/when it gets re-added, it's treated
                    // as an update and we need to be sure we keep the old flags (see warning 0x22441366
                    // in hkVdbObjectHandler).

                    _IObject = value;

                    if (wasNull != (_IObject == null))
                    {
                        if (_IObject != null)
                        {
                            ObjectSet?.Invoke(this, value);
                        }
                        else
                        {
                            ObjectCleared?.Invoke(this, value);
                        }
                    }
                    else if ((_IObject != null) && (_IObject.Type != prevNativeType))
                    {
                        ObjectTypeChanged?.Invoke(this, value);
                    }
                    ObjectUpdated?.Invoke(this, value);
                    if (wasVisible != IsVisible)
                    {
                        VisibilityChanged?.Invoke(this, IsVisible);
                    }
                }
            }
        }
        private Havok.Vdb.VdbObject _IObject;
        public Havok.Vdb.ObjectFlags Flags { get { return _Flags; } }
        private Havok.Vdb.ObjectFlags _Flags = Havok.Vdb.ObjectFlags.None;
        public VdbObjectModel Parent
        {
            get { return _Parent; }
            protected set
            {
                if (_Parent != value)
                {
                    _Parent = value;
                }
            }
        }
        private VdbObjectModel _Parent;
        public ReadOnlyObservableCollection<VdbObjectModel> Children { get; protected set; }
        public IEnumerable<Havok.Vdb.VdbObject> ChildDisplayMarkers { get { return m_childMsRW.Where(c => ((c.Flags & Havok.Vdb.ObjectFlags.Displayable) != 0)).Select(d => d.Object); } }
        public IEnumerable<Havok.Vdb.RenderObject> ChildRenderObjects { get { return ChildDisplayMarkers.Where(marker => m_viewModel.Client.RenderSurface.IsObjectGeometry(marker)).Select(marker => m_viewModel.Client.RenderSurface.GetObjectRenderObject(marker)); } }
        public int Tag { get { return m_tag; } internal set { m_tag = value; } }
        public UInt64 Id { get { System.Diagnostics.Debug.Assert((_IObject == null) || (_IObject.Id == m_id)); return m_id; } }
        public bool IsRootObjectContainer {  get { return ((Flags & Havok.Vdb.ObjectFlags.RootObjectContainer) != 0); } }
        public bool IsHighlighted { get { return m_childMsRW.Any(c => m_viewModel.Client.RenderSurface.IsObjectHighlighted(c.Object)); } }
        public bool IsSelected { get { return m_childMsRW.Any(c => m_viewModel.Client.RenderSurface.IsObjectSelected(c.Object)); } }
        public bool IsDisplayable { get { return m_childMsRW.Any(c => ((c.Flags & Havok.Vdb.ObjectFlags.Displayable) != 0)); } }
        public bool IsVisible { get { return (_IObject == null) || ((Flags & Havok.Vdb.ObjectFlags.Visible) != 0); } }

        protected VdbViewModel m_viewModel;
        protected int m_tag;
        protected UInt64 m_id;
        protected ObservableCollection<VdbObjectModel> m_childMsRW = new ObservableCollection<VdbObjectModel>();
#if DEBUG
        public int DebugUid = s_uids++;
        private static int s_uids = 0;
#endif
    }

    public class VdbObjectViewModel : ViewModelNotifyPropertyChanged, IDisposable
    {
#region Initialization
        public VdbObjectViewModel(VdbViewModel viewModel, VdbObjectModel objectM, IEnumerable<VdbObjectViewModel> childVms)
        {
            m_viewModel = viewModel;
            DisplayHeight = hkStringUtils.GetFontHeight(m_viewModel.FontSize, m_viewModel.TextFont);
            m_objectM = objectM;
            if (m_objectM != null)
            {
                m_isRootObjectContainer = m_objectM.IsRootObjectContainer;
                m_objectM.Disposed += ModelDisposed;
                m_objectM.VisibilityChanged += VdbObjectModel_VisibilityChanged;
                m_objectM.ObjectSet += VdbObjectModel_ObjectSet;
                m_objectM.ObjectCleared += VdbObjectModel_ObjectCleared;
                m_objectM.ObjectTypeChanged += VdbObjectModel_ObjectTypeChanged;
                (m_objectM.Children as INotifyCollectionChanged).CollectionChanged += VdbObjectModel_ChildrenChanged;
                UpdateObjectProperties(m_objectM.Object);
            }

            if (childVms != null)
            {
                foreach (VdbObjectViewModel childVm in childVms)
                {
                    AddChildVm(childVm);
                }
            }

            Children = new ReadOnlyObservableCollection<VdbObjectViewModel>(m_childrenVmsRW);
        }
        private void ModelDisposed()
        {
            Dispose();
            if (m_objectM != null)
            {
                m_viewModel.RemoveFromVdbObjectTreeView(m_objectM);
                if (IsCulled())
                {
                    m_viewModel.CulledVdbObjectModelContainers.Remove(this);
                }
            }
        }
        public void Dispose()
        {
            if (m_objectM != null)
            {
                m_objectM.Disposed -= ModelDisposed;
                m_objectM.VisibilityChanged -= VdbObjectModel_VisibilityChanged;
                m_objectM.ObjectSet -= VdbObjectModel_ObjectSet;
                m_objectM.ObjectCleared -= VdbObjectModel_ObjectCleared;
                m_objectM.ObjectTypeChanged -= VdbObjectModel_ObjectTypeChanged;
                (m_objectM.Children as INotifyCollectionChanged).CollectionChanged -= VdbObjectModel_ChildrenChanged;
            }
            
        }
#endregion

#region View Commands and Properties

        public VdbViewModel Owner { get { return m_viewModel; } }
        public VdbObjectViewModel Parent
        {
            get { return _Parent; }
            protected set
            {
                if (_Parent != value)
                {
                    _Parent = value;
                    System.Diagnostics.Debug.Assert(m_culledParent == null, "Remove from culled list");
                    NotifyPropertyChanged();
                }
            }
        }
        private VdbObjectViewModel _Parent;
        public ReadOnlyObservableCollection<VdbObjectViewModel> Children { get; protected set; }
        public VdbObjectModel Object { get { return m_objectM; } }
        public string TypeName { get; private set; }
        public string DisplayName { get; private set; }
        public double DisplayHeight { get; private set; }
        public double DisplayWidth { get; private set; }
        public override String ToString() { return DisplayName; }
        public string DisplayId { get; private set; }
        private UInt64? m_truncatedId;
        public bool IsEmphasized
        {
            get { return _IsEmphasized; }
            set
            {
                if (_IsEmphasized != value)
                {
                    _IsEmphasized = value;
                    NotifyPropertyChanged();
                }
            }
        }
        private bool _IsEmphasized;
        public bool IsSelected
        {
            get { return (m_viewModel.Selection.SelectedObjects.Contains(this)); }
            set
            {
                if (_NotifyingIsSelectedChanged) return;

                if (IsSelected != value)
                {
                    if (value)
                    {
                        m_viewModel.Selection.SelectedObject = this;
                    }
                    else
                    {
                        m_viewModel.Selection.SelectedObjects.Remove(this);
                    }
                }
            }
        }
        // Rather than have *all* ovms listen for selection changes, we allow VdbSelection to notify selected ovms.
        internal void NotifyIsSelectedChanged()
        {
            if (m_objectM != null)
            {
                if (IsSelected)
                {
                    // Just to be safe if this notify happens incorrectly, add a -= first.
                    m_objectM.ObjectUpdated -= VdbObjectModel_ObjectUpdated;
                    m_objectM.ObjectUpdated += VdbObjectModel_ObjectUpdated;

                    // Ensure we are not culled
                    EnsureNotCulled();
                }
                else
                {
                    m_objectM.ObjectUpdated -= VdbObjectModel_ObjectUpdated;
                }
            }
            using (new hkResetOnDispose(typeof(VdbObjectViewModel), nameof(_NotifyingIsSelectedChanged), true))
            {
                NotifyPropertyChanged(nameof(IsSelected));
            }
        }
        
        // Is needed because our tree view is not multi-select enabled and it will try and
        // change selection *during* notification (which we've already put in the state we want).
        private static bool _NotifyingIsSelectedChanged = false;
        public bool IsExpanded
        {
            get { return _IsExpanded; }
            set
            {
                if (_IsExpanded != value)
                {
                    _IsExpanded = value;
                    NotifyPropertyChanged();
                }

                if (_IsExpanded && _Parent != null)
                {
                    _Parent.IsExpanded = true;
                }
            }
        }
        private bool _IsExpanded;
        public bool IsFiltered { get; set; }
        public void CollapseAll(bool stopAtContainers = true)
        {
            if (m_childrenVmsRW.Count > 0)
            {
                bool hasNonContainerChild = true;
                foreach (VdbObjectViewModel childVm in m_childrenVmsRW)
                {
                    childVm.CollapseAll(stopAtContainers);
                    hasNonContainerChild = !childVm.m_isRootObjectContainer;
                }
                if (!stopAtContainers || hasNonContainerChild)
                {
                    IsExpanded = false;
                }
            }
        }
        public void ExpandParents()
        {
            if( Parent != null )
            {
                Parent.IsExpanded = true;
            }
        }

#endregion

#region Debugging
        public string DebugToolTip
        {
            get
            {
                if (m_objectM != null)
                {
                    string toolTip = String.Format("Id = 0x{0:X}", m_objectM.Id);
                    VdbObjectMultiParentTreeItem treeItem;
                    if (m_viewModel.VdbObjectTree.TryGetValue(m_objectM.Id, out treeItem))
                    {
                        toolTip += "\nPaths to Root = " + treeItem.PathsToRoot;
                    }
                    return toolTip;
                }
                else
                {
                    return "";
                }
            }
        }
#endregion

#region Private Methods and Properties
        private void VdbObjectModel_ObjectSet(VdbObjectModel objectM, Havok.Vdb.VdbObject vdbObject)
        {
            bool wasRootObjectContainer = m_isRootObjectContainer;
            m_isRootObjectContainer = (objectM.IsRootObjectContainer);
            InitFromObjectData(vdbObject);
            if (wasRootObjectContainer != m_isRootObjectContainer) ReSort();
            if (IsSelected) objectM.ObjectUpdated += VdbObjectModel_ObjectUpdated;
        }
        private void VdbObjectModel_ObjectCleared(VdbObjectModel objectM, Havok.Vdb.VdbObject vdbObject)
        {
            InitFromObjectData(vdbObject);
            objectM.ObjectUpdated -= VdbObjectModel_ObjectUpdated;
        }
        private void VdbObjectModel_ObjectTypeChanged(VdbObjectModel objectM, Havok.Vdb.VdbObject vdbObject)
        {
            UpdateObjectProperties(vdbObject);
        }
        private void VdbObjectModel_ObjectUpdated(VdbObjectModel objectM, Havok.Vdb.VdbObject vdbObject)
        {
            NotifyPropertyChanged(nameof(Object));
        }
        private void VdbObjectModel_VisibilityChanged(VdbObjectModel objectM, bool visibility)
        {
            // If the visibility of our model changes, just remove ourself from the tree and re-add.
            // This will ensure we are properly pruned or included.
            if (_Parent != null)
            {
                VdbObjectViewModel parent = _Parent;
                parent.RemoveChildM(m_objectM);
                parent.AddChildM(m_objectM);
                UpdateCulledStatusAfterRemoves();
            }
        }
        private void VdbObjectModel_ChildrenChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                case NotifyCollectionChangedAction.Remove:
                case NotifyCollectionChangedAction.Replace:
                case NotifyCollectionChangedAction.Move:
                {
                    if (e.Action != NotifyCollectionChangedAction.Add)
                    {
                        
                        foreach (VdbObjectModel objectM in e.OldItems)
                        {
                            RemoveChildM(objectM);
                        }
                        UpdateCulledStatusAfterRemoves();
                    }
                    if (e.Action != NotifyCollectionChangedAction.Remove)
                    {
                        foreach (VdbObjectModel objectM in e.NewItems)
                        {
                            AddChildM(objectM);
                        }
                    }
                    break;
                }
                case NotifyCollectionChangedAction.Reset:
                {
                    for (int i = m_childrenVmsRW.Count - 1; i >= 0; i--)
                    {
                        RemoveChildM(m_childrenVmsRW[i].Object);
                    }
                    UpdateCulledStatusAfterRemoves();
                    break;
                }
            }
        }
        private void InitFromObjectData(Havok.Vdb.VdbObject vdbObject)
        {
            UpdateObjectProperties(vdbObject);

            bool wasCulled = (m_culledChildrenVmsRW.Count > 0);
            bool isCulled = UpdateCulledStatusForAddingChildren(0);
            if (wasCulled != isCulled)
            {
                if (wasCulled)
                {
                    // Add anything culled back in
                    // Note: in practice this won't happen because it would require us to *first* be a container
                    // and then later have that flag cleared.  Since you can't update object flags after the object
                    // is added, this shouldn't actually happen.
                    m_viewModel.CulledVdbObjectModelContainers.Remove(this);
                    foreach (VdbObjectViewModel childVm in m_culledChildrenVmsRW)
                    {
                        childVm.m_culledParent = null;
                        AddChildVmUnchecked(childVm);
                    }
                    m_culledChildrenVmsRW.Clear();
                }
                else
                {
                    // Move children to culled
                    while (m_childrenVmsRW.Count > m_viewModel.MaxVdbObjectModelContainerChildren)
                    {
                        int idx = m_childrenVmsRW.Count - 1;
                        VdbObjectViewModel childVm = m_childrenVmsRW[idx];
                        RemoveChildVmAtUnchecked(idx);
                        childVm.m_culledParent = this;
                        m_culledChildrenVmsRW.Add(childVm);
                    }
                }
            }
        }
        private static readonly System.Text.RegularExpressions.Regex TypeNameMatchRegex = new System.Text.RegularExpressions.Regex(@"(hk[a-z]*)?([vV][dD][bB])?(?<typeName>[A-Z0-9]+[a-zA-Z0-9_]+)");
        private static Dictionary<string, string> m_cachedTypeNameDict = new Dictionary<string, string>();
        internal void UpdateObjectProperties(Havok.Vdb.VdbObject vdbObject = null)
        {
            Havok.Vdb.VdbObject prevVdbObject = (m_objectM != null) ? m_objectM.Object : null;
            string prevDebugToolTip = DebugToolTip;
            if (vdbObject == null) vdbObject = prevVdbObject;

            // TypeName
            string prevTypeName = TypeName;
            {
                TypeName = "Object";
                if (vdbObject != null)
                {
                    try
                    {
                        string typeName = vdbObject.TypeName;
                        string prettyTypeName;
                        if (!m_cachedTypeNameDict.TryGetValue(typeName, out prettyTypeName))
                        {
                            string[] types = vdbObject.TypeName.Split(new string[] { "::" }, StringSplitOptions.None);
                            string sanitizedType = types[types.Length - 1];

                            System.Text.RegularExpressions.Match match = TypeNameMatchRegex.Match(types[types.Length - 1]);
                            if (match.Success)
                            {
                                sanitizedType = match.Groups["typeName"].Value;
                            }

                            prettyTypeName = hkStringUtils.SplitCamelCase(sanitizedType);
                            m_cachedTypeNameDict.Add(typeName, prettyTypeName);
                        }
                        TypeName = prettyTypeName;
                    }
                    catch { TypeName = "Unknown"; }
                }
            }

            // DisplayId
            string prevDisplayId = DisplayId;
            {
                m_truncatedId = null;

                // Get id from VdbObjectModel (if available)
                if (m_objectM != null)
                {
                    // Note: this is hardcoded for how Physics packs values into it's object ids, but this should get moved to the name format regex stuff
                    //       *or* into a per-object treeId marker.
                    
                    m_truncatedId = (0xffff & m_objectM.Id);
                }

                // Format our DisplayId as desired by current settings
                if (m_isRootObjectContainer)
                {
                    // Don't include IDs for containers
                    DisplayId = "";
                }
                else if (m_truncatedId.HasValue)
                {
                    DisplayId = m_truncatedId.Value.ToString(m_viewModel.IsDebugViewEnabled ? StringFormat.Hexadecimal : StringFormat.Default);
                }
                else
                {
                    DisplayId = "";
                }
            }

            // DisplayName
            // This must be last because it depends on Name and DisplayId
            string prevDisplayName = DisplayName;
            {
                string nameString = null;
                {
                    Havok.Vdb.IReadOnlyProperty value;
                    if ((vdbObject != null) && vdbObject.Properties.TryGetValue("name", out value))
                    {
                        string namePropertyString = (value.Value as string);
                        if (namePropertyString != null)
                        {
                            if (!String.IsNullOrEmpty(namePropertyString))
                            {
                                nameString = " \"" + namePropertyString + "\"";
                            }
                        }
                        else if (value.Value != null)
                        {
                            nameString = " \"" + value.Value.ToString() + "\"";
                        }
                    }
                }

                string idString = null;
                {
                    string id = DisplayId;
                    if (!String.IsNullOrEmpty(id)) idString = ": " + id;
                }

                DisplayName =
                    TypeName +
                    ((nameString != null) ? nameString : "") +
                    ((idString != null) ? idString : "");
            }

            if (prevTypeName != TypeName) NotifyPropertyChanged(nameof(TypeName));
            if (prevDisplayName != DisplayName)
            {
                // Need to re-sort...this can be very slow, so we only do it when data is toggled.
                if (((prevVdbObject == null) != (vdbObject == null)) && (_Parent != null))
                {
                    _Parent.ReSort(this);
                }
                // Add a fudge factor for some control border/spacing accommodation.
                // Not ideal, but also this isn't that big of a deal, so we make it larg-ish
                DisplayWidth = hkStringUtils.GetFontWidth(DisplayName, m_viewModel.FontSize, m_viewModel.TextFont) + 15;
                NotifyPropertyChanged(nameof(DisplayName));
                NotifyPropertyChanged(nameof(DisplayWidth));
            }
            if (prevDisplayId != DisplayId) NotifyPropertyChanged(nameof(DisplayId));
            if (prevDebugToolTip != DebugToolTip) NotifyPropertyChanged(nameof(DebugToolTip));
        }

        private void AddChildM(VdbObjectModel childTemplateM)
        {
            if (childTemplateM != null)
            {
                List<VdbObjectViewModel> matchingTreeObjectVms;
                VdbObjectViewModel treeObjectVm = m_viewModel.AddToVdbObjectTreeView(childTemplateM, out matchingTreeObjectVms);
                AddChildVm(treeObjectVm);
            }
        }

        private void AddChildVmUnchecked(VdbObjectViewModel childVm)
        {
            if (childVm != null)
            {
                System.Diagnostics.Debug.Assert(childVm != this);
                System.Diagnostics.Debug.Assert(childVm.Parent == null);
                childVm.Parent = this;
                InsertSort(childVm);
                
                // This is *way* to slow with manifolds!  Because each of these crawls over all manifolds
                // and this will happen when you grab
                //if (IsFiltered)childVm.SetDisplayOpFromView(VdbObjectDisplayOp.Highlight);
                //if (IsFiltered && IsSelected) childVm.SetDisplayOpFromView(VdbObjectDisplayOp.Select);
            }
        }

        private void AddChildVm(VdbObjectViewModel childVm)
        {
            if (childVm != null)
            {
                if (UpdateCulledStatusForAddingChildren(1))
                {
                    childVm.m_culledParent = this;
                    m_culledChildrenVmsRW.Add(childVm);
                }
                else
                {
                    AddChildVmUnchecked(childVm);
                }
            }
        }

        private void RemoveChildM(VdbObjectModel childTemplateM)
        {
            if (childTemplateM != null)
            {
                Predicate<VdbObjectViewModel> removeAndClearParent =
                    childVm =>
                    {
                        if (childVm.m_objectM == childTemplateM)
                        {
                            childVm.m_culledParent = null;
                            return true;
                        }
                        else
                        {
                            return false;
                        }
                    };
                if (m_culledChildrenVmsRW.RemoveAll(removeAndClearParent) == 0)
                {
                    for (int i = m_childrenVmsRW.Count - 1; i >= 0; i--)
                    {
                        VdbObjectViewModel childVm = m_childrenVmsRW[i];
                        if (childVm.m_objectM == childTemplateM)
                        {
                            RemoveChildVmAtUnchecked(i);
                            m_viewModel.RemoveFromVdbObjectTreeView(childVm.m_objectM);
                            break;
                        }
                    }
                }
#if DEBUG
                else
                {
                    System.Diagnostics.Debug.Assert(
                        !m_childrenVmsRW.Any(childVm => childVm.m_objectM == childTemplateM),
                        "Child is in culled and normal collections");
                }
#endif
            }
        }

        private bool RemoveChildVmAtUnchecked(int idx)
        {
            if (idx < m_childrenVmsRW.Count)
            {
                VdbObjectViewModel childVm = m_childrenVmsRW[idx];
                m_viewModel.RemoveFromVdbObjectTreeView(childVm.m_objectM);
                m_childrenVmsRW.RemoveAt(idx);
                m_childrenVmsRW_sortedProxy.RemoveAt(idx);
                childVm.Parent = null;
                
                
                //child.SetDisplayOpFromView(VdbObjectDisplayOp.ClearHighlightedAndSelected);
                //child.SetDisplayOpFromView(VdbObjectDisplayOp.ClearHighlighted);
                
                
                //child.SetDisplayOpFromView(VdbObjectDisplayOp.ClearSelected);
                return true;
            }
            return false;
        }

        private void ReSort(VdbObjectViewModel childVm = null)
        {
            if (childVm != null)
            {
                bool removed = m_childrenVmsRW.Remove(childVm);
                removed &= m_childrenVmsRW_sortedProxy.Remove(childVm);
                System.Diagnostics.Debug.Assert(removed);
                InsertSort(childVm);
            }
            else
            {
                m_childrenVmsRW.Clear();
                m_childrenVmsRW_sortedProxy.Sort(GetComparer());
                foreach (var sortedChildVm in m_childrenVmsRW_sortedProxy)
                {
                    m_childrenVmsRW.Add(sortedChildVm);
                }
            }
        }

        private void InsertSort(VdbObjectViewModel childVm)
        {
            System.Diagnostics.Debug.Assert(!m_childrenVmsRW.Contains(childVm));
            System.Diagnostics.Debug.Assert(!m_childrenVmsRW_sortedProxy.Contains(childVm));
            int idx = m_childrenVmsRW_sortedProxy.BinarySearch(childVm, GetComparer());
            if (idx < 0) idx = ~idx;
            m_childrenVmsRW_sortedProxy.Insert(idx, childVm);
            m_childrenVmsRW.Insert(idx, childVm);
        }

        private bool IsCulled() { return (m_culledChildrenVmsRW.Count > 0); }
        private void EnsureNotCulled()
        {
            VdbObjectViewModel objectVm = this;
            while ((objectVm != null) && (objectVm != m_viewModel.VdbObjectTreeView))
            {
                if (objectVm.m_culledParent != null)
                {
                    VdbObjectViewModel _CulledParent = objectVm.m_culledParent;

                    // Increment this if needed, so we don't cull later.
                    m_viewModel.MaxVdbObjectModelContainerChildren = Math.Max(m_viewModel.MaxVdbObjectModelContainerChildren, _CulledParent.m_childrenVmsRW.Count + 1);

                    // Move the culled ovm into the visible set.
                    objectVm.m_culledParent = null;
                    bool removed = _CulledParent.m_culledChildrenVmsRW.Remove(objectVm);
                    System.Diagnostics.Debug.Assert(removed, "m_culledParent inconsistently set");
                    _CulledParent.UpdateCulledStatusAfterRemoves();
                    _CulledParent.AddChildVmUnchecked(objectVm);
                }
                objectVm = objectVm.Parent;
            }
        }

        private bool UpdateCulledStatusForAddingChildren(int numChildren)
        {
            if (IsCulled())
            {
                // Already culling
                return true;
            }
            else if (m_isRootObjectContainer && ((m_childrenVmsRW.Count + numChildren) > m_viewModel.MaxVdbObjectModelContainerChildren))
            {
                // Will be culling
                m_viewModel.CulledVdbObjectModelContainers.Add(this);
                return true;
            }
            else
            {
                // Not culling
                return false;
            }
        }

        private bool UpdateCulledStatusAfterRemoves()
        {
            bool culling = false;
            if (m_isRootObjectContainer)
            {
                // Steal from culled nodes if we have them
                while ((m_childrenVmsRW.Count < m_viewModel.MaxVdbObjectModelContainerChildren) && (m_culledChildrenVmsRW.Count > 0))
                {
                    int idx = m_culledChildrenVmsRW.Count - 1;
                    VdbObjectViewModel childVm = m_culledChildrenVmsRW[idx];
                    childVm.m_culledParent = null;
                    AddChildVmUnchecked(childVm);
                    m_culledChildrenVmsRW.RemoveAt(idx);
                }

                if (UpdateCulledStatusForAddingChildren(0))
                {
                    culling = true;
                }
                else
                {
                    // Remove ourselves if we are no longer culled
                    m_viewModel.CulledVdbObjectModelContainers.Remove(this);
                }
            }
            return culling;
        }

        internal VdbObjectViewModel FindContainingDisplayable()
        {
            VdbObjectViewModel parentVm = this;
            while (parentVm != null)
            {
                if ((parentVm.m_objectM != null) && (parentVm.m_objectM.IsDisplayable))
                {
                    break;
                }
                else
                {
                    parentVm = parentVm.Parent;
                }
            }
            return parentVm;
        }
        internal List<VdbObjectViewModel> CollectLeafwardDisplayables()
        {
            List<VdbObjectViewModel> displayables = new List<VdbObjectViewModel>(Children.Count * 5 + 1);
            if ((Object != null) && (Object.IsDisplayable))
            {
                displayables.Add(this);
            }
            // Recurse, but avoid recursing on a container as these are not considered owned by the ovm
            // and using them to set selection/highlighting can be very expensive
            if (!m_isRootObjectContainer)
            {
                foreach (VdbObjectViewModel child in Children)
                {
                    displayables.AddRange(child.CollectLeafwardDisplayables());
                }
            }
            return displayables;
        }
        internal IEnumerable<Havok.Vdb.RenderObject> GetRenderObjects(HashSet<UInt64> processedIds = null)
        {
            if (processedIds == null) processedIds = new HashSet<UInt64>();

            if (Object != null)
            {
                IEnumerable<Havok.Vdb.VdbObject> displayMarkers = Object.ChildDisplayMarkers;
                IEnumerable<Havok.Vdb.RenderObject> renderObjects = displayMarkers.
                    Where(marker => processedIds.Add(marker.Id)).
                    Select(marker => m_viewModel.Client.RenderSurface.GetObjectRenderObject(marker)).
                    Where(renderObject => (renderObject != null));
                return renderObjects;
            }
            else
            {
                return new List<Havok.Vdb.RenderObject>();
            }
        }

        private IComparer<object> GetComparer()
        {
            return
                (m_isRootObjectContainer) ?
                    (IComparer<object>) s_objectIdComparer :
                    hkToStringComparer.CurrentCultureIgnoreCase;
        }
        private class ObjectIdComparer : Comparer<object>
        {
            public override int Compare(object object1, object object2)
            {
                VdbObjectViewModel objectVm1 = (VdbObjectViewModel)object1;
                VdbObjectViewModel objectVm2 = (VdbObjectViewModel)object2;
                UInt64 id1 = objectVm1.m_truncatedId.HasValue ? objectVm1.m_truncatedId.Value : UInt64.MaxValue;
                UInt64 id2 = objectVm2.m_truncatedId.HasValue ? objectVm2.m_truncatedId.Value : UInt64.MaxValue;
                return (int)(id1 - id2);
            }
        }
        private static ObjectIdComparer s_objectIdComparer = new ObjectIdComparer();

        protected VdbViewModel m_viewModel;
        protected VdbObjectModel m_objectM;
        protected ObservableCollection<VdbObjectViewModel> m_childrenVmsRW = new ObservableCollection<VdbObjectViewModel>();
        protected List<VdbObjectViewModel> m_childrenVmsRW_sortedProxy = new List<VdbObjectViewModel>();
        protected List<VdbObjectViewModel> m_culledChildrenVmsRW = new List<VdbObjectViewModel>();
        protected VdbObjectViewModel m_culledParent;
        protected bool m_isRootObjectContainer;
#if DEBUG
        public int DebugUid = s_uids++;
        private static int s_uids = 0;
#endif
#endregion
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
