// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM		: WIN32 X64
// PRODUCT		: COMMON
// VISIBILITY	: CLIENT
//
// ------------------------------------------------------TKBMS v1.0

using System;
using System.Linq;
using System.Reflection;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;

namespace HavokVisualDebugger
{
    public class VdbObjectPropertiesOptions : ViewModelNotifyPropertyChanged
    {
        public VdbObjectPropertiesOptions(VdbViewModel vm)
        {
            m_viewModel = vm;
            m_viewModel.PropertyChanged += ViewModel_PropertyChanged;
        }

        private void ViewModel_PropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            System.Diagnostics.Debug.Assert(sender == m_viewModel);
            if (e.PropertyName == nameof(VdbViewModel.IsDebugViewEnabled))
            {
                NotifyPropertyChanged(nameof(HexFormatEnabled));
            }
        }

        public bool HexFormatEnabled
        {
            get
            {
                return
                    (m_stringFormat == StringFormat.Hexadecimal) ||
                    ((m_viewModel != null) && (m_viewModel.IsDebugViewEnabled));
            }
            set
            {
                if ((m_stringFormat == StringFormat.Hexadecimal) != value)
                {
                    m_stringFormat = (value) ? StringFormat.Hexadecimal : StringFormat.Default;
                    NotifyPropertyChanged();

                    VdbSettings.SetValue(nameof(Properties.Settings.VdbOptions_HexFormatEnabled), value);
                }
            }
        }

        public bool ShowRawPropertyValues
        {
            get { return _ShowRawPropertyValues; }
            set
            {
                if (_ShowRawPropertyValues != value)
                {
                    _ShowRawPropertyValues = value;
                    NotifyPropertyChanged();
                }
            }
        }
        private bool _ShowRawPropertyValues = false;


        public StringFormat GetStringFormat()
        {
            return HexFormatEnabled ? StringFormat.Hexadecimal : m_stringFormat;
        }

        private StringFormat m_stringFormat = StringFormat.Default;
        private VdbViewModel m_viewModel;
    }

    public class VdbObjectPropertiesViewModel : ViewModelNotifyPropertyChanged, IDisposable
    {
        public VdbObjectPropertiesViewModel(VdbViewModel vm, VdbSelection selection) :
            this(vm, selection, selection.SelectedObject, selection.SelectedRenderObject)
        {}
        public VdbObjectPropertiesViewModel(VdbViewModel vm, VdbSelection selection, VdbObjectViewModel objectVm, Havok.Vdb.RenderObject renderObject)
        {
            System.Diagnostics.Debug.Assert(vm != null);

            m_viewModel = vm;
            m_selection = selection;
            if (m_viewModel != null) m_viewModel.PropertyChanged += VdbViewModel_PropertyChanged;
            if (m_selection != null) m_selection.PropertyChanged += VdbSelection_PropertyChanged;
            Object = objectVm;
            ObjectProperties.CollectionChanged += (s, e) => { NotifyPropertyChanged(nameof(ObjectPropertiesCount)); };
            RenderObject = renderObject;

            m_propertyOptions = (m_viewModel != null) ? m_viewModel.ObjectPropertyOptions : null;
            if (m_propertyOptions != null) m_propertyOptions.PropertyChanged += Options_PropertyChanged;

            SaveRenderObjectCommand = new ViewModelDelegateCommand(
                this,
                (o) =>
                {
                    m_viewModel.SaveGeometry(RenderObject);
                },
                new List<string> {
                    nameof(RenderObject)
                },
                new List<Func<object, bool>> {
                delegate(object o) {
                    return (m_viewModel != null) && (RenderObject != null);
                }});

            ExpandAllCommand = new ViewModelDelegateCommand(
                this,
                (o) =>
                {
                    foreach (VdbObjectPropertyViewModel property in ObjectProperties.ToArray())
                    {
                        property.ExpandAll();
                    }
                },
                new List<string> {
                    nameof(ObjectPropertiesCount)
                },
                new List<Func<object, bool>> {
                    delegate(object o) {
                        return (ObjectProperties.Count > 0);
                }});

            CollapseAllCommand = new ViewModelDelegateCommand(
                this,
                (o) =>
                {
                    foreach (VdbObjectPropertyViewModel property in ObjectProperties.ToArray())
                    {
                        property.IsExpanded = false;
                    }
                },
                new List<string> {
                    nameof(ObjectPropertiesCount)
                },
                new List<Func<object, bool>> {
                    delegate(object o) {
                        return (ObjectProperties.Count > 0);
                }});
        }

        public void Dispose()
        {
            // Will unhook listeners
            Object = null;
            RenderObject = null;
            if (m_viewModel != null) m_viewModel.PropertyChanged -= VdbViewModel_PropertyChanged;
            if (m_selection != null) m_selection.PropertyChanged -= VdbSelection_PropertyChanged;
            if (m_propertyOptions != null) m_propertyOptions.PropertyChanged -= Options_PropertyChanged;
        }

        private void Options_PropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            // Update all values
            if (e.PropertyName == nameof(VdbObjectPropertiesOptions.HexFormatEnabled))
            {
                UpdateObjectProperties();
                NotifyPropertyChanged(nameof(RenderObjectId));
                NotifyPropertyChanged(nameof(RenderObjectTriangles));
                NotifyPropertyChanged(nameof(RenderObjectVertices));
            }
            else if (e.PropertyName == nameof(VdbObjectPropertiesOptions.ShowRawPropertyValues))
            {
                UpdateObjectProperties();
            }
        }

        public VdbObjectViewModel Object
        {
            get { return _Object; }
            set
            {
                if (_Object != value)
                {
                    if (_Object != null)
                    {
                        _Object.PropertyChanged -= VdbObjectViewModel_PropertyChanged;
                    }
                    if (value != null)
                    {
                        value.PropertyChanged += VdbObjectViewModel_PropertyChanged;
                    }
                    _Object = value;
                    UpdateObjectProperties();
                    NotifyPropertyChanged();
                }
            }
        }
        private VdbObjectViewModel _Object;

        public Havok.Vdb.RenderObject RenderObject
        {
            get { return _RenderObject; }
            set
            {
                if (_RenderObject != value)
                {
                    _RenderObject = value;
                    if (value == null)
                    {
                        // Hide the color panel if the render object has gone away.
                        IsRenderObjectColorPanelShown = false;
                    }
                    NotifyPropertyChanged();
                    NotifyPropertyChanged(nameof(RenderObjectColor));
                    NotifyPropertyChanged(nameof(RenderObjectResetColor));
                    NotifyPropertyChanged(nameof(RenderObjectId));
                    NotifyPropertyChanged(nameof(RenderObjectTriangles));
                    NotifyPropertyChanged(nameof(RenderObjectVertices));
                    NotifyPropertyChanged(nameof(RenderObjectEnableUserColor));
                    NotifyPropertyChanged(nameof(IsRenderObjectColorPanelShown));
                }
            }
        }
        private Havok.Vdb.RenderObject _RenderObject;

        public System.Windows.Media.Color RenderObjectColor
        {
            get
            {
                Havok.Vdb.RenderObject ro = RenderObject;
                if (ro == null)
                {
                    return System.Windows.Media.Color.FromArgb(0, 255, 255, 255);
                }

                // If user color is set to hkColor::NONE it won't be applied.
                if (ro.EnableUserColor && (ro.UserColor != System.Windows.Media.Color.FromArgb(0, 255, 255, 255)))
                {
                    return ro.UserColor;
                }
                else
                {
                    // If the user color isn't being applied, than the color is the same as the reset color.
                    return RenderObjectResetColor;
                }
            }
            set
            {
                Havok.Vdb.RenderObject ro = RenderObject;
                if (ro == null)
                {
                    return;
                }

                ro.EnableUserColor = true;
                ro.UserColor = value;

                NotifyPropertyChanged();
                NotifyPropertyChanged(nameof(RenderObjectEnableUserColor));
            }
        }

        public System.Windows.Media.Color RenderObjectResetColor
        {
            get
            {
                Havok.Vdb.RenderObject ro = RenderObject;
                if (ro == null)
                {
                    return System.Windows.Media.Color.FromArgb(0, 255, 255, 255);
                }

                if ((m_viewModel != null) && m_viewModel.RandomizedColors)
                {
                   return ro.RandomizedColor;
                }
                else
                {
                    return ro.ServerColor;
                }
            }
        }

        public bool IsRenderObjectColorPanelShown
        {
            get { return _IsRenderObjectColorPanelShown; }
            set
            {
                if (_IsRenderObjectColorPanelShown != value)
                {
                    _IsRenderObjectColorPanelShown = value;
                    NotifyPropertyChanged();
                }
            }
        }
        private bool _IsRenderObjectColorPanelShown;

        public string RenderObjectId
        {
            get
            {
                Havok.Vdb.RenderObject ro = RenderObject;
                if (ro == null)
                {
                    return "";
                }
                return ro.Id.ToString(GetStringFormat());
            }
        }

        public string RenderObjectTriangles
        {
            get
            {
                Havok.Vdb.RenderObject ro = RenderObject;
                if (ro == null)
                {
                    return "";
                }
                return ro.NumTriangles.ToString(GetStringFormat());
            }
        }

        public string RenderObjectVertices
        {
            get
            {
                Havok.Vdb.RenderObject ro = RenderObject;
                if (ro == null)
                {
                    return "";
                }
                return ro.NumVertices.ToString(GetStringFormat());
            }
        }

        public bool RenderObjectEnableUserColor
        {
            get
            {
                Havok.Vdb.RenderObject ro = RenderObject;
                return (ro != null) && ro.EnableUserColor;
            }
            set
            {
                if (RenderObjectEnableUserColor != value)
                {
                    _RenderObject.EnableUserColor = value;
                    NotifyPropertyChanged();
                    NotifyPropertyChanged(nameof(RenderObjectColor));
                }
            }
        }

        public ObservableCollection<VdbObjectPropertyViewModel> ObjectProperties { get { return m_objectPropertiesVm.Children; } }
        public int ObjectPropertiesCount { get { return m_objectPropertiesVm.Children.Count; } }

        public System.Windows.Input.ICommand SaveRenderObjectCommand { get; private set; }
        public System.Windows.Input.ICommand CollapseAllCommand { get; private set; }
        public System.Windows.Input.ICommand ExpandAllCommand { get; private set; }

        private void VdbViewModel_PropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            System.Diagnostics.Debug.Assert(sender == m_viewModel);
            switch (e.PropertyName)
            {
                case nameof(VdbViewModel.RandomizedColors):
                {
                    NotifyPropertyChanged(nameof(RenderObjectColor));
                    NotifyPropertyChanged(nameof(RenderObjectResetColor));
                    break;
                }
            }
        }

        private void VdbSelection_PropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            VdbSelection selection = (sender as VdbSelection);
            if (selection != null)
            {
                switch(e.PropertyName)
                {
                    case nameof(VdbSelection.SelectedObject):
                    case nameof(VdbSelection.SelectedRenderObject):
                    {
                        Object = selection.SelectedObject;
                        RenderObject = selection.SelectedRenderObject;
                        break;
                    }
                }
            }
        }

        private void VdbObjectViewModel_PropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            System.Diagnostics.Debug.Assert(_Object == sender);
            switch (e.PropertyName)
            {
                case nameof(VdbObjectViewModel.Object):
                {
                    UpdateObjectProperties();
                    break;
                }
            }
        }

        private void UpdateObjectProperties()
        {
            m_objectPropertiesVm.Set(
                "",
                ((_Object != null) && (_Object.Object != null)) ? _Object.Object.Object : null,
                GetStringFormat(),
                GetShouldDisplayRaw());
        }

        private bool GetShouldDisplayRaw()
        {
            return (m_propertyOptions != null) ? m_propertyOptions.ShowRawPropertyValues : false;
        }

        private StringFormat GetStringFormat()
        {
            return (m_propertyOptions != null) ? m_propertyOptions.GetStringFormat() : StringFormat.Default;
        }

        private VdbObjectPropertiesOptions m_propertyOptions = null;
        private VdbViewModel m_viewModel = null;
        private VdbSelection m_selection = null;
        private VdbObjectPropertyViewModel m_objectPropertiesVm = new VdbObjectPropertyViewModel(null, null);
    }

    public class VdbObjectPropertyViewModel : ViewModelNotifyPropertyChanged
    {
        public VdbObjectPropertyViewModel(VdbObjectPropertyViewModel parent, ObservableCollection<VdbObjectPropertyViewModel> rootFlattenedPropertyVms)
        {
            System.Diagnostics.Debug.Assert((parent == null) == (rootFlattenedPropertyVms == null), "Only root should have null flattened properties");
            Parent = parent;
            m_isExpandingAll = false;
            Children = new ObservableCollection<VdbObjectPropertyViewModel>();
            m_rootFlattenedPropertyVms = (rootFlattenedPropertyVms != null) ? rootFlattenedPropertyVms : Children;
        }

        public string Name
        {
            get { return _Name; }
            private set
            {
                if (_Name != value)
                {
                    _Name = value;
                    NotifyPropertyChanged();
                }
            }
        }
        private string _Name;

        public string Value
        {
            get { return _Value; }
            private set
            {
                if (_Value != value)
                {
                    _Value = value;
                    NotifyPropertyChanged();
                }
            }
        }
        private string _Value;

        public VdbObjectPropertyViewModel Parent
        {
            get { return _Parent; }
            private set
            {
                if (_Parent != value)
                {
                    _Parent = value;
                    NotifyPropertyChanged();
                    NotifyPropertyChanged(nameof(ParentPadding));
                    NotifyPropertyChanged(nameof(PaddingMargin));
                }
            }
        }
        private VdbObjectPropertyViewModel _Parent;

        public int ParentPadding
        {
            get { return (Parent == null) ? 0 : 1 + Parent.ParentPadding; }
        }

        public System.Windows.Thickness PaddingMargin
        {
            get
            {
                int parentPadding = Math.Max(0, ParentPadding - 1);
                System.Windows.Thickness margin = new System.Windows.Thickness(parentPadding * 15, 0, 0, 0);
                return margin;
            }
        }

        public ObservableCollection<VdbObjectPropertyViewModel> Children { get; set; }

        public bool IsExpanded
        {
            get { return _IsExpanded; }
            set
            {
                if (_IsExpanded != value)
                {
                    _IsExpanded = value;
                    NotifyPropertyChanged();
                    NotifyChildrenOfNodeExpansion(_IsExpanded, true);
                }
            }
        }
        private bool _IsExpanded;

        public void ExpandAll()
        {
            using (new hkResetOnDispose(this, nameof(m_isExpandingAll), true))
            {
                IsExpanded = true;
            }
        }

        private void NotifyChildrenOfNodeExpansion(bool parentExpanded, bool root)
        {
            if (!root || m_isExpandingAll)
            {
                NotifyPropertyChanged(nameof(IsExpanded));
                int thisIndexInView = m_rootFlattenedPropertyVms.IndexOf(this);

                if (!parentExpanded)
                {
                    if (thisIndexInView >= 0)
                    {
                        m_rootFlattenedPropertyVms.RemoveAt(thisIndexInView);
                    }
                }
                else if (thisIndexInView < 0 && Parent != null && Parent.IsExpanded)
                {
                    int parentIndex = m_rootFlattenedPropertyVms.IndexOf(Parent);
                    if (parentIndex >= 0)
                    {
                        m_rootFlattenedPropertyVms.Insert(parentIndex + 1, this);
                    }
                }
            }

            if (Children != null)
            {
                for (int i = Children.Count - 1; i >= 0; i--)
                {
                    VdbObjectPropertyViewModel node = Children[i];

                    if (m_isExpandingAll)
                    {
                        node.ExpandAll();
                    }
                    else
                    {
                        node.NotifyChildrenOfNodeExpansion(parentExpanded, false);
                    }
                }
            }
        }

        /// <summary>
        /// Set the name and propValue for this vm.
        /// Also specify the string formatting and if we want to build raw properties.
        /// </summary>
        public void Set(string name, object value, StringFormat format, bool displayAsRaw)
        {
            Name = name;
            m_propValue = value;
            SetChildValues(value, format, displayAsRaw);
            SetFormat(format);
        }

        /// <summary>
        /// Set the current string display format.
        /// </summary>
        public void SetFormat(StringFormat format)
        {
            Value = (m_propValue != null) ? m_propValue.ToString(format) : "[null]";
        }

        private void SetChildValues(object value, StringFormat format, bool displayAsRaw)
        {
            if (value is Havok.Vdb.Object)
            {
                SetChildValuesInternal(value as Havok.Vdb.Object, format, displayAsRaw);
            }
            else if (!(value is string) && (value is System.Collections.IEnumerable))
            {
                SetChildValuesInternal(value as System.Collections.IEnumerable, format, displayAsRaw);
            }
            else
            {
                SetChildValuesInternal(value, format, displayAsRaw);
            }
        }

        private void SetChildValuesInternal(Havok.Vdb.Object vdbObject, StringFormat format, bool displayAsRaw)
        {
            System.Diagnostics.Debug.Assert(vdbObject != null, "Not expecting null vdbObject");

            // Grab our effective properties
            IReadOnlyDictionary<string, Havok.Vdb.IReadOnlyProperty> newProperties = (displayAsRaw) ? vdbObject.RawProperties : vdbObject.Properties;

            // If we are switching types, the "expanded state" of our children is no longer meaningful/synced
            if (m_vdbObjectTypePtr != vdbObject.Type)
            {
                Children.Clear();
                m_vdbObjectTypePtr = vdbObject.Type;
            }

            // If we aren't the root and our number of children is changing, then we need to collapse
            // during this operation so that the correct number of rows exists in the m_rootFlattenedPropertyVms.
            bool newChildCount = (Children != m_rootFlattenedPropertyVms) && (Children.Count != newProperties.Count);
            using (new hkConditionalResetOnDispose(newChildCount, this, nameof(IsExpanded), false))
            {
                int childIdx = 0;
                foreach (var roProp in newProperties.Values)
                {
                    while (true)
                    {
                        // Expand as needed
                        if (childIdx >= Children.Count)
                        {
                            Children.Add(new VdbObjectPropertyViewModel(this, m_rootFlattenedPropertyVms));
                        }

                        // Skip children that aren't ours
                        if (Children[childIdx]._Parent == this)
                        {
                            break;
                        }

                        // Advance
                        childIdx++;
                    }

                    // Set our data and advance
                    Children[childIdx].Set(roProp.Name, roProp.Value, format, displayAsRaw);
                    childIdx++;
                }

                // Shrink as needed
                while (childIdx < Children.Count)
                {
                    Children.RemoveAt(Children.Count - 1);
                }
            }
        }

        private void SetChildValuesInternal(System.Collections.IEnumerable list, StringFormat format, bool displayAsRaw)
        {
            System.Diagnostics.Debug.Assert(list != null, "Not expecting null vdbObject");

            // If our number of children is changing, then we need to collapse during this operation so that
            // the correct number of rows exists in the m_rootFlattenedPropertyVms.
            bool newChildCount = (Children.Count != list.OfType<object>().Count());
            using (new hkConditionalResetOnDispose(newChildCount, this, nameof(IsExpanded), false))
            {
                int childIdx = 0;
                foreach (object item in list)
                {
                    // Expand as needed
                    if (childIdx >= Children.Count)
                    {
                        Children.Add(new VdbObjectPropertyViewModel(this, m_rootFlattenedPropertyVms));
                    }

                    // Set our data and advance
                    Children[childIdx].Set(childIdx.ToString(), item, format, displayAsRaw);
                    childIdx++;
                }

                // Shrink as needed
                while (childIdx < Children.Count)
                {
                    Children.RemoveAt(Children.Count - 1);
                }
            }
        }

        private void SetChildValuesInternal(object generic, StringFormat format, bool displayAsRaw)
        {
            Type valueType = (generic != null) ? generic.GetType() : null;
            if ((valueType != null) && !valueType.IsPrimitive && !valueType.Equals(typeof(string)))
            {
                // Grab our members (props and fields)
                List<MemberInfo> memberInfos = new List<MemberInfo>();
                try
                {
                    BindingFlags getFieldsAndPropertiesFlags = BindingFlags.Instance | BindingFlags.Public | BindingFlags.FlattenHierarchy;
                    memberInfos.AddRange(valueType.GetFields(getFieldsAndPropertiesFlags));
                    memberInfos.AddRange(valueType.GetProperties(getFieldsAndPropertiesFlags));
                }
                catch
                {
                    Children.Clear();
                    return;
                }

                // If our number of children is changing, then we need to collapse during this operation so that
                // the correct number of rows exists in the m_rootFlattenedPropertyVms.
                bool newChildCount = (Children.Count != memberInfos.Count());
                using (new hkConditionalResetOnDispose(newChildCount, this, nameof(IsExpanded), false))
                {
                    int childIdx = 0;
                    foreach (MemberInfo info in memberInfos)
                    {
                        try
                        {
                            // Expand as needed
                            if (childIdx >= Children.Count)
                            {
                                Children.Add(new VdbObjectPropertyViewModel(this, m_rootFlattenedPropertyVms));
                            }

                            // Set our data
                            if (info is PropertyInfo)
                            {
                                Children[childIdx].Set(info.Name, (info as PropertyInfo).GetValue(generic), format, displayAsRaw);
                            }
                            else if (info is FieldInfo)
                            {
                                Children[childIdx].Set(info.Name, (info as FieldInfo).GetValue(generic), format, displayAsRaw);
                            }
                            else
                            {
                                System.Diagnostics.Debug.Assert(false, "Unexpected MemberInfo type");
                                Children[childIdx].Set(info.Name, "[unknown]", format, displayAsRaw);
                            }
                        }
                        catch
                        {
                            // Set our data to unknown if there was an error
                            Children[childIdx].Set(info.Name, "[unknown]", format, displayAsRaw);
                        }

                        // Advance
                        childIdx++;
                    }

                    // Shrink as needed
                    while (childIdx < Children.Count)
                    {
                        Children.RemoveAt(Children.Count - 1);
                    }
                }
            }
            else
            {
                // We are a basic type, we have no children
                Children.Clear();
            }
        }

        // Tracks whether an Expand All operation is currently happening.
        private bool m_isExpandingAll;

        // The prop value from the IReadOnlyPropertySet.
        private object m_propValue;

        // If m_propValue is a Havok.Vdb.Object, this points to it's c++ type.
        private IntPtr m_vdbObjectTypePtr;

        // The flat list of properties we are expanding into.
        private ObservableCollection<VdbObjectPropertyViewModel> m_rootFlattenedPropertyVms;
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
