// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM		: WIN32 X64
// PRODUCT		: COMMON
// VISIBILITY	: CLIENT
//
// ------------------------------------------------------TKBMS v1.0

using System;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Input;

namespace HavokVisualDebugger
{
    
    public abstract class ViewModelNotifyPropertyChanged : INotifyPropertyChanged
    {
        public event PropertyChangedEventHandler PropertyChanged;
        protected virtual void NotifyPropertyChanged([System.Runtime.CompilerServices.CallerMemberName] String propertyName = "")
        {
            OnPropertyChanged(new PropertyChangedEventArgs(propertyName));
        }
        protected virtual void OnPropertyChanged(PropertyChangedEventArgs e)
        {
            PropertyChanged?.Invoke(this, e);
        }
    }

    
    public class VdbModelRelayCommand<T> : ICommand where T : class
    {
        private Action<T> m_execute = null;
        private Predicate<T> m_canExecute = null;

        public VdbModelRelayCommand(Action<T> execute)
            : this(execute, null)
        {
        }

        public VdbModelRelayCommand(Action<T> execute, Predicate<T> canExecute)
        {
            if (execute != null)
            {
                m_execute = execute;
                m_canExecute = canExecute;
            }
        }

        public bool CanExecute(object parameter)
        {
            return (m_canExecute == null) ? true : m_canExecute(parameter as T);
        }
        
        public event EventHandler CanExecuteChanged
        {
            add { CommandManager.RequerySuggested += value; }
            remove { CommandManager.RequerySuggested -= value; }
        }
        
        public void Execute(object parameter)
        {
            m_execute(parameter as T);
        }
    }

    public class SingleEventArg : EventArgs
    {
        private readonly object m_arg;

        public SingleEventArg(object arg)
        {
            m_arg = arg;
        }

        public object Arg
        {
            get { return m_arg;  }
        }
    }

    // From http://stackoverflow.com/questions/2984803/how-to-automatically-scroll-scrollviewer-only-if-the-user-did-not-change-scrol
    public static class ScrollViewerExtensions
    {
        public static readonly DependencyProperty AlwaysScrollToEndProperty = DependencyProperty.RegisterAttached("AlwaysScrollToEnd", typeof(bool),
            typeof(ScrollViewerExtensions), new PropertyMetadata(false, AlwaysScrollToEndChanged));
        private static bool _autoScroll;

        private static void AlwaysScrollToEndChanged(object sender, DependencyPropertyChangedEventArgs e)
        {
            ScrollViewer scroll = sender as ScrollViewer;
            if (scroll != null)
            {
                bool alwaysScrollToEnd = (e.NewValue != null) && (bool)e.NewValue;
                if (alwaysScrollToEnd)
                {
                    scroll.ScrollToEnd();
                    scroll.ScrollChanged += ScrollChanged;
                }
                else { scroll.ScrollChanged -= ScrollChanged; }
            }
            else { throw new InvalidOperationException("The attached AlwaysScrollToEnd property can only be applied to ScrollViewer instances."); }
        }

        public static bool GetAlwaysScrollToEnd(ScrollViewer scroll)
        {
            if (scroll == null) { throw new ArgumentNullException("scroll"); }
            return (bool)scroll.GetValue(AlwaysScrollToEndProperty);
        }

        public static void SetAlwaysScrollToEnd(ScrollViewer scroll, bool alwaysScrollToEnd)
        {
            if (scroll == null) { throw new ArgumentNullException("scroll"); }
            scroll.SetValue(AlwaysScrollToEndProperty, alwaysScrollToEnd);
        }

        private static void ScrollChanged(object sender, ScrollChangedEventArgs e)
        {
            ScrollViewer scroll = sender as ScrollViewer;
            if (scroll == null) { throw new InvalidOperationException("The attached AlwaysScrollToEnd property can only be applied to ScrollViewer instances."); }

            // User scroll event : set or unset autoscroll mode
            if (e.ExtentHeightChange == 0) { _autoScroll = scroll.VerticalOffset == scroll.ScrollableHeight; }

            // Content scroll event : autoscroll eventually
            if (_autoScroll && e.ExtentHeightChange != 0) { scroll.ScrollToVerticalOffset(scroll.ExtentHeight); }
        }
    }

    public static class FocusExtension
    {
        public static bool GetIsFocused(DependencyObject obj)
        {
            return (bool)obj.GetValue(IsFocusedProperty);
        }

        public static void SetIsFocused(DependencyObject obj, bool value)
        {
            obj.SetValue(IsFocusedProperty, value);
        }

        public static readonly DependencyProperty IsFocusedProperty =
            DependencyProperty.RegisterAttached("IsFocused", typeof(bool), typeof(FocusExtension), new UIPropertyMetadata(false, null, OnCoerceValue));
        
        private static object OnCoerceValue(DependencyObject d, object baseValue)
        {
            UIElement uiElement = (UIElement)d;
            if ((bool)baseValue && uiElement != null)
            {
                uiElement.Focus();
            }
            else if (uiElement != null && uiElement.IsFocused)
            {
                Keyboard.ClearFocus();
            }
            return ((bool)baseValue);
        }
    }

    public class ComboBoxAttachedProperties : DependencyObject
    {
        /// <summary>
        /// DeleteItemCommand
        /// </summary>
        public static readonly DependencyProperty DeleteItemCommandProperty =
           DependencyProperty.RegisterAttached("DeleteItemCommand", typeof(ICommand), typeof(ComboBoxAttachedProperties),
               new FrameworkPropertyMetadata(null, FrameworkPropertyMetadataOptions.BindsTwoWayByDefault | FrameworkPropertyMetadataOptions.Inherits));

        public static ICommand GetDeleteItemCommand(DependencyObject d)
        {
            return (ICommand)d.GetValue(DeleteItemCommandProperty);
        }

        public static void SetDeleteItemCommand(DependencyObject d, ICommand value)
        {
            d.SetValue(DeleteItemCommandProperty, value);
        }
    }

    public class LogViewerAttachedProperties : DependencyObject
    {
        /// <summary>
        /// IsFilterSelected
        /// </summary>
        public static readonly DependencyProperty IsFilterSelectedProperty =
            DependencyProperty.RegisterAttached("IsFilterSelected", typeof(bool), typeof(LogViewerAttachedProperties),
                new FrameworkPropertyMetadata(false, FrameworkPropertyMetadataOptions.BindsTwoWayByDefault | FrameworkPropertyMetadataOptions.Inherits));

        public static bool GetIsFilterSelected(DependencyObject d)
        {
            return (bool)d.GetValue(IsFilterSelectedProperty);
        }

        public static void SetIsFilterSelected(DependencyObject d, bool value)
        {
            d.SetValue(IsFilterSelectedProperty, value);
        }

        /// <summary>
        /// IsErrorSelected
        /// </summary>
        public static readonly DependencyProperty IsErrorSelectedProperty =
           DependencyProperty.RegisterAttached("IsErrorSelected", typeof(bool), typeof(LogViewerAttachedProperties),
               new FrameworkPropertyMetadata(false, FrameworkPropertyMetadataOptions.BindsTwoWayByDefault | FrameworkPropertyMetadataOptions.Inherits));

        public static bool GetIsErrorSelected(DependencyObject d)
        {
            return (bool)d.GetValue(IsErrorSelectedProperty);
        }

        public static void SetIsErrorSelected(DependencyObject d, bool value)
        {
            d.SetValue(IsErrorSelectedProperty, value);
        }

        /// <summary>
        /// IsWarningSelected
        /// </summary>
        public static readonly DependencyProperty IsWarningSelectedProperty =
           DependencyProperty.RegisterAttached("IsWarningSelected", typeof(bool), typeof(LogViewerAttachedProperties),
               new FrameworkPropertyMetadata(false, FrameworkPropertyMetadataOptions.BindsTwoWayByDefault | FrameworkPropertyMetadataOptions.Inherits));

        public static bool GetIsWarningSelected(DependencyObject d)
        {
            return (bool)d.GetValue(IsWarningSelectedProperty);
        }

        public static void SetIsWarningSelected(DependencyObject d, bool value)
        {
            d.SetValue(IsWarningSelectedProperty, value);
        }

        /// <summary>
        /// IsInfoSelected
        /// </summary>
        public static readonly DependencyProperty IsInfoSelectedProperty =
           DependencyProperty.RegisterAttached("IsInfoSelected", typeof(bool), typeof(LogViewerAttachedProperties),
               new FrameworkPropertyMetadata(false, FrameworkPropertyMetadataOptions.BindsTwoWayByDefault | FrameworkPropertyMetadataOptions.Inherits));

        public static bool GetIsInfoSelected(DependencyObject d)
        {
            return (bool)d.GetValue(IsInfoSelectedProperty);
        }

        public static void SetIsInfoSelected(DependencyObject d, bool value)
        {
            d.SetValue(IsInfoSelectedProperty, value);
        }

        /// <summary>
        /// IsLocalLog
        /// </summary>
        public static readonly DependencyProperty IsLocalLogProperty =
           DependencyProperty.RegisterAttached("IsLocalLog", typeof(bool), typeof(LogViewerAttachedProperties),
               new FrameworkPropertyMetadata(false, FrameworkPropertyMetadataOptions.BindsTwoWayByDefault | FrameworkPropertyMetadataOptions.Inherits));

        public static bool GetIsLocalLog(DependencyObject d)
        {
            return (bool)d.GetValue(IsLocalLogProperty);
        }

        public static void SetIsLocalLog(DependencyObject d, bool value)
        {
            d.SetValue(IsLocalLogProperty, value);
        }

        /// <summary>
        /// LogView
        /// </summary>
        public static readonly DependencyProperty LogViewProperty =
           DependencyProperty.RegisterAttached("LogView", typeof(ObservableCollection<VdbLogViewModel>), typeof(LogViewerAttachedProperties),
               new FrameworkPropertyMetadata(null, FrameworkPropertyMetadataOptions.BindsTwoWayByDefault | FrameworkPropertyMetadataOptions.Inherits));

        public static ObservableCollection<VdbLogViewModel> GetLogView(DependencyObject d)
        {
            return (ObservableCollection<VdbLogViewModel>)d.GetValue(LogViewProperty);
        }

        public static void SetLogView(DependencyObject d, ObservableCollection<VdbLogViewModel> value)
        {
            d.SetValue(LogViewProperty, value);
        }


        /// <summary>
        /// AppBadgeValue
        /// </summary>
        public static readonly DependencyProperty AppBadgeValueProperty =
           DependencyProperty.RegisterAttached("AppBadgeValue", typeof(string), typeof(LogViewerAttachedProperties),
               new FrameworkPropertyMetadata("", FrameworkPropertyMetadataOptions.BindsTwoWayByDefault | FrameworkPropertyMetadataOptions.Inherits));

        public static string GetAppBadgeValue(DependencyObject d)
        {
            return (string)d.GetValue(AppBadgeValueProperty);
        }

        public static void SetAppBadgeValue(DependencyObject d, string value)
        {
            d.SetValue(AppBadgeValueProperty, value);
        }

        /// <summary>
        /// ShowAppBadge
        /// </summary>
        public static readonly DependencyProperty ShowAppBadgeProperty =
           DependencyProperty.RegisterAttached("ShowAppBadge", typeof(bool), typeof(LogViewerAttachedProperties),
               new FrameworkPropertyMetadata(false, FrameworkPropertyMetadataOptions.BindsTwoWayByDefault | FrameworkPropertyMetadataOptions.Inherits));

        public static bool GetShowAppBadge(DependencyObject d)
        {
            return (bool)d.GetValue(ShowAppBadgeProperty);
        }

        public static void SetShowAppBadge(DependencyObject d, bool value)
        {
            d.SetValue(ShowAppBadgeProperty, value);
        }
    }

    public class BindingProxy : Freezable
    {
        protected override Freezable CreateInstanceCore()
        {
            return new BindingProxy();
        }

        public object Data
        {
            get { return (object)GetValue(DataProperty); }
            set { SetValue(DataProperty, value); }
        }

        // Using a DependencyProperty as the backing store for Data.
        // This enables animation, styling, binding, etc...
        public static readonly DependencyProperty DataProperty = DependencyProperty.Register("Data", typeof(object), typeof(BindingProxy), new UIPropertyMetadata(null));
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
