// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM		: WIN32 X64
// PRODUCT		: COMMON
// VISIBILITY	: CLIENT
//
// ------------------------------------------------------TKBMS v1.0

//#define ENABLE_DISPATCHER_DEBUG_CHECKS
//#define ENABLE_MAX_THROUGHPUT_TICKING

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.Threading;
using System.Windows;
using System.Windows.Threading;

namespace HavokVisualDebugger
{
    public class VdbModel
    {
        public class ViewerInformation
        {
            public List<string> ViewerNames { get; set; }
            public string ImagePath { get; set; }
            public string LongDescription { get; set; }
            public string ShortDescription { get; set; }

            public ViewerInformation(List<string> viewerNames, string imagePath, string longDescription, string shortDescription)
            {
                ViewerNames = viewerNames;
                ImagePath = imagePath;
                LongDescription = longDescription;
                ShortDescription = shortDescription;
            }

            public ViewerInformation()
            {
                ViewerNames = new List<string>();
                ImagePath = null;
                LongDescription = "";
                ShortDescription = "";
            }

            static public Dictionary<string, ViewerInformation> VdbProcessInformationDictionary = new Dictionary<string, ViewerInformation>();
        }


        
        
        
        
        
        

        
        
        

        internal class VdbClientLock : IDisposable
        {
            public VdbClientLock(Havok.Vdb.Client client)
            {
                m_client = client;
                Controls.VdbClientProgressModalWindow.WaitForAccess(client);
            }

            public void Dispose()
            {
                Monitor.Exit(m_client);
            }

            Havok.Vdb.Client m_client;
        }

        public delegate void VdbClientInteropErroredEventHandler(Exception e);
        public class VdbClientInterop
        {
            public event VdbClientInteropErroredEventHandler Errored = delegate { };
            public const DispatcherPriority ApplicationDispatchPriority = DispatcherPriority.Background;

            protected VdbClientInterop(VdbClientInterop parent, VdbClient clientInterop, Havok.Vdb.Client client, object obj)
            {
                m_parent = parent;
                m_clientInterop = clientInterop;
                m_client = client;
                m_object = obj;
            }

            public bool IsValid()
            {
                return (m_object != null);
            }

            public object Call(string methodName, params object[] parameters)
            {
                if (m_object == null) return null;

                try
                {
                    using (new VdbClientLock(m_client))
                    {
                        return m_object.GetType().GetMethod(methodName, new List<object>(parameters).ConvertAll<Type>(o => o.GetType()).ToArray()).Invoke(m_object, parameters);
                    }
                }
                catch (Exception e)
                {
                    FwdErrored(e);
                    return null;
                }
            }

            public T Call<T>(string methodName, params object[] parameters)
            {
                return (T)Call(methodName, parameters);
            }

            public void CallAsync(string methodName, params object[] parameters)
            {
                CallAsync(null, methodName, parameters);
            }

            public void CallAsync(Action<object> completed, string methodName, params object[] parameters)
            {
                System.Diagnostics.Debug.Assert(m_clientInterop != null, "Cannot call this method before processing has started on the client thread");
                if (m_clientInterop != null)
                {
                    m_clientInterop.InvokeAsync(() =>
                    {
                        object ret = Call(methodName, parameters);
                        if (completed != null)
                        {
                            Application.Current.Dispatcher.Invoke(completed, ret);
                        }
                    });
                }
                else
                {
                    Call(methodName, parameters);
                }
            }

            public T GetProperty<T>(string propName)
            {
                if (m_object == null)
                {
                    return default(T);
                }

                try
                {
                    using (new VdbClientLock(m_client))
                    {
                        return (T)m_object.GetType().GetProperty(propName).GetValue(m_object);
                    }
                }
                catch (Exception e)
                {
                    FwdErrored(e);
                    return default(T);
                }
            }

            private void _SetProperty(string propName, object val)
            {
                if (m_object == null) return;

                try
                {
                    using (new VdbClientLock(m_client))
                    {
                        m_object.GetType().GetProperty(propName).SetValue(m_object, val);
                    }
                }
                catch (Exception e)
                {
                    FwdErrored(e);
                }
            }

            public void SetPropertyAsync(string propName, object val)
            {
                SetPropertyAsync(null, propName, val);
            }

            public void SetPropertyAsync(Action completed, string propName, object val)
            {
                System.Diagnostics.Debug.Assert(m_clientInterop != null, "Cannot call this method before processing has started on the client thread");
                if (m_clientInterop != null)
                {
                    m_clientInterop.InvokeAsync(() =>
                    {
                        _SetProperty(propName, val);
                        if (completed != null)
                        {
                            Application.Current.Dispatcher.Invoke(completed);
                        }
                    });
                }
                else
                {
                    _SetProperty(propName, val);
                }
            }

            public Delegate ListenTo<HandlerType, P1>(string eventName, Action<P1> callback)
            {
                if (m_object == null) return null;

                try
                {
                    using (new VdbClientLock(m_client))
                    {
                        Delegate fwdToUiThread =
                            Delegate.CreateDelegate(typeof(HandlerType), new DelegateInterop<P1>(m_clientInterop, callback), "Call");

                        m_object.GetType().GetEvent(eventName).AddEventHandler(m_object, fwdToUiThread);

                        return fwdToUiThread;
                    }
                }
                catch (Exception e)
                {
                    FwdErrored(e);
                }

                return null;
            }

            public Delegate ListenTo<HandlerType, P1, P2>(string eventName, Action<P1, P2> callback)
            {
                if (m_object == null) return null;

                try
                {
                    using (new VdbClientLock(m_client))
                    {
                        Delegate fwdToUiThread =
                            Delegate.CreateDelegate(typeof(HandlerType), new DelegateInterop<P1, P2>(m_clientInterop, callback), "Call");

                        m_object.GetType().GetEvent(eventName).AddEventHandler(m_object, fwdToUiThread);

                        return fwdToUiThread;
                    }
                }
                catch (Exception e)
                {
                    FwdErrored(e);
                }

                return null;
            }

            public void UnListenTo(string eventName, Delegate fwdToUiThreadDelegate)
            {
                if (m_object == null) return;

                try
                {
                    using (new VdbClientLock(m_client))
                    {
                        m_object.GetType().GetEvent(eventName).RemoveEventHandler(m_object, fwdToUiThreadDelegate);
                    }
                }
                catch (Exception e)
                {
                    FwdErrored(e);
                }
            }

            public VdbClientInterop GetSubInterop(string objectPropertyOrMethodName, params object[] parameters)
            {
                if (m_object == null) return new VdbClientInterop();

                try
                {
                    // Note:
                    // We do not expect our sub-interops to change during client process ticks.
                    // This call must be lockless so that the UI can change properties on sub interops
                    // via SetPropertyAsync; allowing the progress bar to work for long property set operations.
                    // using (new VdbClientLock(m_client))
                    {
                        return GetCachedSubInterop(objectPropertyOrMethodName, parameters);
                    }
                }
                catch (Exception e)
                {
                    FwdErrored(e);
                    return new VdbClientInterop();
                }
            }

            public IDisposable AcquireLock()
            {
                return new VdbClientLock(m_client);
            }

            public bool HasLock()
            {
                return Monitor.IsEntered(m_client);
            }

            private class DelegateInterop<P1>
            {
                public DelegateInterop(VdbClient clientInterop, Action<P1> callback)
                {
                    m_clientInterop = clientInterop;
                    m_callback = callback;
                }

                public void Call(P1 param1)
                {
                    if (m_clientInterop != null)
                    {
                        m_clientInterop.QueueCallback(() =>
                        {
                            
                            // This is the lock-step approach, the UI will not get behind the sim *and* rendering won't hitch
                            Application.Current.Dispatcher.Invoke(() =>
                            {
                                m_callback(param1);
                            });
                        });
                    }
                    else
                    {
                        // This shouldn't happen anymore, but just in case dispatch async to UI thread so no deadlocks occur.
                        System.Diagnostics.Debug.Assert(false);
                        Dispatcher.CurrentDispatcher.InvokeAsync(() =>
                        {
                            // Note: repeated InvokeAsyncs are inherently dangerous as we can grow without bound
                            Application.Current.Dispatcher.Invoke(() =>
                            {
                                m_callback(param1);
                            });
                        });
                    }
                }

                private VdbClient m_clientInterop;
                private Action<P1> m_callback;
            }

            private class DelegateInterop<P1, P2>
            {
                public DelegateInterop(VdbClient clientInterop, Action<P1, P2> callback)
                {
                    m_clientInterop = clientInterop;
                    m_callback = callback;
                }

                public void Call(P1 param1, P2 param2)
                {
                    if (m_clientInterop != null)
                    {
                        m_clientInterop.QueueCallback(() =>
                        {
                            
                            // This is the lock-step approach, the UI will not get behind the sim *and* rendering won't hitch
                            Application.Current.Dispatcher.Invoke(() =>
                            {
                                m_callback(param1, param2);
                            });
                        });
                    }
                    else
                    {
                        // This shouldn't happen anymore, but just in case dispatch async to UI thread so no deadlocks occur.
                        System.Diagnostics.Debug.Assert(false);
                        Dispatcher.CurrentDispatcher.InvokeAsync(() =>
                        {
                            // Note: repeated InvokeAsyncs are inherently dangerous we can grow without bound
                            Application.Current.Dispatcher.Invoke(() =>
                            {
                                m_callback(param1, param2);
                            });
                        });
                    }
                }

                private VdbClient m_clientInterop;
                private Action<P1, P2> m_callback;
            }

            

            private VdbClientInterop GetCachedSubInterop(string objectPropertyOrMethodName, object[] parameters)
            {
                VdbClientInterop interop;
                // If it's a prop, we can cache it
                if ((parameters == null) || (parameters.Length == 0))
                {
                    if (!m_subInterops.TryGetValue(objectPropertyOrMethodName, out interop))
                    {
                        object subObject = m_object.GetType().GetProperty(objectPropertyOrMethodName).GetValue(m_object);
                        interop = new VdbClientInterop(
                            this,
                            m_clientInterop,
                            m_client,
                            subObject);

                        m_subInterops.Add(objectPropertyOrMethodName, interop);
                    }
                }
                // If it's based on parameters return a new one, we can't cache it (or if we did, we'd need to cache taking into account parameters; or cache by returned object instead)
                else
                {
                    object subObject = m_object.GetType().GetMethod(objectPropertyOrMethodName, new List<object>(parameters).ConvertAll<Type>(o => o.GetType()).ToArray()).Invoke(m_object, parameters);
                    interop = new VdbClientInterop(
                        this,
                        m_clientInterop,
                        m_client,
                        subObject);
                }
                return interop;
            }

            protected VdbClientInterop() { }
            protected void FwdErrored(Exception e) { if (m_parent != null) m_parent.FwdErrored(e); else Errored(e); }
            protected VdbClient m_clientInterop;
            protected Havok.Vdb.Client m_client;
            protected object m_object;
            VdbClientInterop m_parent;
            Dictionary<string, VdbClientInterop> m_subInterops = new Dictionary<string, VdbClientInterop>();
        }

        public class VdbClient : VdbClientInterop
        {
            internal event Action ClientTicked;

            internal VdbClient()
            {
                m_clientInterop = this;
                m_client = new Havok.Vdb.Client();
                _ServerDiscoveryPort = Havok.Vdb.Client.ADVERTISE_DEFAULT_SERVER_PORT;
                _IsServerDiscoveryEnabled = m_client.IsServerDiscoveryEnabled;
                m_object = m_client;
                DiscoveredServers = new ObservableCollection<Havok.Vdb.ServerInfo>();
            }
            internal Havok.Vdb.RenderSurface RenderSurface { get { return m_client.RenderSurface; } }

            #region Processing Enable/Disable/Suspension
            internal void StartProcessing(object nativeRenderSurface, bool enableMSAA)
            {
                // Initialize our render surface with our platform specific surface.
                // Widgets collection is populated after this call, so we notify of property change.
                m_client.RenderSurface.Initialize(nativeRenderSurface as System.Windows.Controls.Canvas, enableMSAA);

                // Start rendering.
                
                
                m_client.RenderSurface.StartRenderLoop();

                ManualResetEvent clientThreadInitialized = new ManualResetEvent(false);
                {
                    Thread clientThread = new Thread(new ThreadStart(() =>
                    {
                        // Thread initialization
                        {
                            m_client.InitBackgroundThread();
                            m_clientThreadDispatcher = Dispatcher.CurrentDispatcher;
#if ENABLE_DISPATCHER_DEBUG_CHECKS
                            m_clientThreadDispatcher.Hooks.DispatcherInactive += (s,e) => { m_pendingInvokes = 0; };
                            m_clientThreadDispatcher.Hooks.OperationAborted += (s, e) => { if (e.Operation.Status.HasFlag(DispatcherOperationStatus.Pending)) { m_pendingInvokes--; } m_invokeExecuting = false; };
                            m_clientThreadDispatcher.Hooks.OperationCompleted += (s, e) => { m_invokeExecuting = false; };
                            m_clientThreadDispatcher.Hooks.OperationPosted += (s, e) => { m_pendingInvokes++; };
                            m_clientThreadDispatcher.Hooks.OperationStarted += (s, e) => { m_pendingInvokes--; m_invokeExecuting = true; };
#endif

                            // Clear pending callbacks when the frame changes
                            m_client.PlaybackHandler.PlaybackCmdReceived +=
                                delegate (Havok.Vdb.PlaybackHandler sender, Havok.Vdb.PlaybackCmdReceivedArgs args)
                                {
                                    // While connected to a network, we wait until the end of a frame boundary before
                                    // processing callbacks on the UI thread to minimize lock contention.
                                    // We know the server will have a lot of work to do after a frame completes.
                                    // This is not true for files, in which all the data is present.
                                    if ((m_client.ConnectedSource == Havok.Vdb.ConnectedSource.Network) &&
                                        args.FlagWasSet(Havok.Vdb.PlaybackFlags.FrameEnded) )
                                    {
                                        InvokePendingCallbacksAsync(true);
                                    }
                                };

#if ENABLE_MAX_THROUGHPUT_TICKING
                            // Tick at regular intervals to achieve smooth playback.
                            m_clientTickTimer = new DispatcherTimer(/*DispatcherPriority.Send*/);
                            m_clientTickTimer.Tick += TickClient;
                            m_clientTickTimer.Interval = new TimeSpan(0); // As quick as possible, throttling happens in TickClient
                            m_clientTickTimer.Start();
                            m_clientTickStopwatch = new Stopwatch();
                            m_clientTickStopwatch.Start();
#else
                            // Tick at regular intervals to achieve smooth playback.
                            m_clientTickTimer = new DispatcherTimer();
                            m_clientTickTimer.Tick += TickClient;
                            m_clientTickTimer.Interval = new TimeSpan(10000); // Note: 1 tick == 100ns
                            m_clientTickTimer.Start();

                            // Auto reconnect-timer
                            m_autoReconnectTimer = new DispatcherTimer();
                            m_autoReconnectTimer.Tick += TickAutoReconnect;
                            m_autoReconnectTimer.Interval = new TimeSpan(1000000); // Note: 1 tick == 100ns
                            _ServerDiscoveryTimeoutMs = 3000; // 3 x Interval to help ensure timeout doesn't happen between ticks
                            m_autoReconnectTimer.Start();
#endif

                            // Hook up progress reporting
                            m_client.ProgressReporter.UpdateFrequency = .1f;
                            m_client.ProgressReporter.WorkEstimateUpdated += delegate (Havok.Vdb.ProgressReporter sender, Havok.Vdb.WorkEstimateUpdatedEventArgs args)
                            {
                                // If we are completing client work, flush our pending queue in a way that reports progress.
                                if (args.EstimatedTimeRemaining == 0)
                                {
                                    ProgressReportingPendingCallbackExecution pendingCallbackExec = new ProgressReportingPendingCallbackExecution(this);
                                    pendingCallbackExec.InvokeAsync();
                                }
                                // Else, just update our progress directly
                                else
                                {
                                    Controls.VdbClientProgressModalWindow.UpdateProgress(m_client, args);
                                }
                            };
                        }
                        clientThreadInitialized.Set();

                        // Enter dispatcher loop
                        try
                        {
                            Dispatcher.Run();
                        }
                        catch (Exception e)
                        {
                            System.Diagnostics.Debug.Assert(false, "Process thread prematurely aborted: " + e.Message);
                        }

                        // Tear-down the thread
                        m_client.QuitBackgroundThread();
                    }));
                    clientThread.Name = "VDB Client Process Thread";
                    clientThread.Start();
                }
                clientThreadInitialized.WaitOne();

                // When we get a modal dialog, always suspend processing.
                System.Windows.Interop.ComponentDispatcher.EnterThreadModal += delegate
                {
                    SuspendProcessing();
                };
                System.Windows.Interop.ComponentDispatcher.LeaveThreadModal += delegate
                {
                    ResumeProcessing();
                };
            }
            internal void StopProcessing()
            {
                try
                {
                    // Kill the render loop (do this first so that our dispatch loop doesn't do a bunch of unnecessary rendering!)
                    m_client.RenderSurface.StopRenderLoop();

                    // Note: The invoke queue may have methods which sync-invoke back on the UI thread, so we can't wait for shutdown
                    m_clientThreadDispatcher.BeginInvokeShutdown(DispatcherPriority.Normal);

                    // Clear the invoke queue, but some of that queue may rely on this thread's dispatcher, so we need to do dispatcher work while we wait
                    while (!m_clientThreadDispatcher.HasShutdownFinished)
                    {
                        DispatcherFrame frame = new DispatcherFrame();
                        Dispatcher.CurrentDispatcher.BeginInvoke(DispatcherPriority.ApplicationIdle, new DispatcherOperationCallback(delegate
                        {
                            frame.Continue = false;
                            return null;
                        }), null);
                        Dispatcher.PushFrame(frame);
                    }
                }
                // We are executing arbitrary code, likely in a shutdown context where things are going away.
                // Avoid exceptions from being problematic.
                // One example is that some UI might be dependent on the RenderSurface's render loop running, which
                // we intentionally stop first for speedier shutdown.
                catch { }
            }

            private void SuspendProcessing()
            {
                // Don't allow rendering the scene to potential cause lock contention with
                // process thread while we are suspended.
                if (m_suspendCount == 0)
                {
                    m_client.RenderSurface.StopRenderLoop();
                }
                m_suspendCount = hkMath.Clamp(m_suspendCount + 1, 0, Int32.MaxValue);
            }
            private void ResumeProcessing()
            {
                if (m_suspendCount == 1)
                {
                    m_client.RenderSurface.StartRenderLoop();
                }
                m_suspendCount = hkMath.Clamp(m_suspendCount - 1, 0, Int32.MaxValue);
            }
            private int m_suspendCount;
            #endregion

            #region Server Discovery
            internal Tuple<string, UInt16> AttemptAutoReconnect { get; set; }

            internal bool IsServerDiscoveryEnabled
            {
                get { return _IsServerDiscoveryEnabled; }
                set
                {
                    if (_IsServerDiscoveryEnabled != value)
                    {
                        _IsServerDiscoveryEnabled = value;
                        if (_IsServerDiscoveryEnabled)
                        {
                            CallAsync(nameof(Havok.Vdb.Client.EnableServerDiscovery), _ServerDiscoveryPort, _ServerDiscoveryTimeoutMs);
                        }
                        else
                        {
                            CallAsync(nameof(Havok.Vdb.Client.DisableServerDiscovery));
                        }
                        DiscoveredServers.Clear();
                    }
                }
            }
            private bool _IsServerDiscoveryEnabled;
            internal UInt16 ServerDiscoveryPort
            {
                get { return _ServerDiscoveryPort; }
                set
                {
                    if (_ServerDiscoveryPort != value)
                    {
                        _ServerDiscoveryPort = value;
                        if (_IsServerDiscoveryEnabled)
                        {
                            CallAsync(nameof(Havok.Vdb.Client.EnableServerDiscovery), _ServerDiscoveryPort, _ServerDiscoveryTimeoutMs);
                        }
                        DiscoveredServers.Clear();
                    }
                }
            }
            private UInt16 _ServerDiscoveryPort;
            // Note: currently this relates closely with auto reconnect tick interval...so we don't expose it
            private float _ServerDiscoveryTimeoutMs = 3000;

            internal ObservableCollection<Havok.Vdb.ServerInfo> DiscoveredServers { get; private set; }
            #endregion

            #region Deferred Callback Invoking
            internal void QueueCallback(Action callback)
            {
                lock (m_pendingCallbackQueue)
                {
                    m_pendingCallbackQueue.Add(callback);
                }
            }
            internal DispatcherOperation InvokeAsync(Action action)
            {
                return m_clientThreadDispatcher.InvokeAsync(action);
            }
            private void InvokePendingCallbacksAsync(bool waitToResumeProcessing)
            {
                lock (m_pendingCallbackQueue)
                {
                    // Queue ClientTicked at the end of all the other callbacks so the UI can update itself.
                    QueueCallback(() =>
                        {
                            Application.Current.Dispatcher.Invoke(() =>
                            {
                                ClientTicked?.Invoke();
                            });
                        });

                    if (waitToResumeProcessing)
                    {
                        Dispatcher.CurrentDispatcher.InvokeAsync(() =>
                        {
                            // Stop any process ticking so as not to overwhelm UI processing tasks
                            // Note: Stop() will abort the current dispatch task, which is why we queue this operation instead of calling it directly.
                            m_clientTickTimer.Stop();
                        });
                    }

                    // Push to the back of our queue all pending operations.
                    m_invokedCallbackQueue = m_pendingCallbackQueue.ConvertAll(c => Dispatcher.CurrentDispatcher.InvokeAsync(c));
                    m_pendingCallbackQueue.Clear();

                    if (waitToResumeProcessing)
                    {
                        // After those are done, start ticking again.
                        Dispatcher.CurrentDispatcher.InvokeAsync(
#if ENABLE_MAX_THROUGHPUT_TICKING
                        () =>
#else
                        async () =>
#endif
                        {
#if !ENABLE_MAX_THROUGHPUT_TICKING
                            // Do a yield before explicitly ticking client so we let queued events a chance to run.
                            // We do this to cover cases where there's not much data coming through and each TickClient
                            // keeps adding to the queue as it processes enough data to reach a step and thus invoke
                            // pending callbacks again!
                            // The queue grows and it never has a chance to "consider" other dispatch timers for
                            // enqueue.
                            await Dispatcher.Yield();

                            // Do an immediate tick since Start() will not tick again until interval has elapsed.
                            TickClient(null, null);
#endif
                            // Start ticking normally.
                            m_clientTickTimer.Start();
                        });
                    }
                }
            }
            private class ProgressReportingPendingCallbackExecution : Havok.Vdb.WorkEstimate
            {
                public ProgressReportingPendingCallbackExecution(VdbClient clientInterop)
                {
                    m_clientInterop = clientInterop;
                }

                public void InvokeAsync()
                {
                    lock (m_clientInterop.m_pendingCallbackQueue)
                    {
                        // Steal the pending queue, we will be processing from it directly
                        m_callbackQueue.AddRange(m_clientInterop.m_pendingCallbackQueue);
                        m_clientInterop.m_pendingCallbackQueue.Clear();
                        m_clientInterop.m_invokedCallbackQueue.Clear();

                        // Queue our processing loop
                        Dispatcher.CurrentDispatcher.InvokeAsync(() =>
                        {
                            // Updates until completion (this works because UpdateProgress will cause the modal dialog which
                            // participates in executing the queued callbacks).
                            DateTime lastUpdateTime = DateTime.MinValue;
                            float updateFreq = m_clientInterop.m_client.ProgressReporter.UpdateFrequency;
                            for (m_currentCallbackIdx = 0;
                                m_currentCallbackIdx < m_callbackQueue.Count;
                                m_currentCallbackIdx++)
                            {
                                m_callbackQueue[m_currentCallbackIdx]?.Invoke();
                                DateTime updateTime = DateTime.Now;
                                if ((updateTime - lastUpdateTime).TotalSeconds > updateFreq)
                                {
                                    Controls.VdbClientProgressModalWindow.UpdateProgress(m_clientInterop.m_client, this);
                                    lastUpdateTime = updateTime;
                                }
                            }

                            // Always notify completed, since we are called in lieu of normal work completed signal
                            Controls.VdbClientProgressModalWindow.UpdateProgress(m_clientInterop.m_client, this);
                        });
                    }
                }

                public UInt32 CompletedWorkItems { get { return (UInt32)(m_currentCallbackIdx); } }
                public Double EstimatedTimeRemaining { get { return (m_currentCallbackIdx < m_callbackQueue.Count) ? -1 : 0; } }
                public UInt32 EstimatedWorkItems { get { return (UInt32)(m_callbackQueue.Count); } }
                public String Message { get { return "Executing Process Callbacks"; } }
                public bool CanCancel { get { return false; } }
                public void Cancel() { }

                private VdbClient m_clientInterop;
                private List<Action> m_callbackQueue = new List<Action>();
                private int m_currentCallbackIdx;
            }
            private List<Action> m_pendingCallbackQueue = new List<Action>();
            private List<DispatcherOperation> m_invokedCallbackQueue = new List<DispatcherOperation>();
#if ENABLE_DISPATCHER_DEBUG_CHECKS
            private int m_pendingInvokes = 0;
            private bool m_invokeExecuting = false;
#endif
#endregion

#region Ticking
            private void TickAutoReconnect(object sender, EventArgs e)
            {
                if (m_suspendCount == 0)
                {
                    try
                    {
                        lock (m_client)
                        {
                            // We've timed-out, try to connect again...
                            if ((AttemptAutoReconnect != null) &&
                                ((m_client.ConnectedSource == Havok.Vdb.ConnectedSource.None) ||
                                ((m_client.ConnectedSource == Havok.Vdb.ConnectedSource.Network) &&
                                (m_client.ConnectedState != Havok.Vdb.ConnectedState.Connecting) &&
                                (m_client.ConnectedState != Havok.Vdb.ConnectedState.Connected))))
                            {
                                m_client.ConnectToMachine(AttemptAutoReconnect.Item1, AttemptAutoReconnect.Item2);
                            }

                            // Do polling (for server discovery)
                            m_client.Poll();

                            // Update our discovered servers list
                            // Note: We cannot actually update DiscoveredServers here because it's created during ctor from the UI thread
                            //       and throws an exception if modification occurs on a different thread.
                            // O(n^m) - but happens infrequently and over a smallish-set
                            List<Havok.Vdb.ServerInfo> clientDiscoveredServers = new List<Havok.Vdb.ServerInfo>(m_client.GetDiscoveredServers());
                            List<Havok.Vdb.ServerInfo> serversToAdd = new List<Havok.Vdb.ServerInfo>();
                            List<Havok.Vdb.ServerInfo> serversToRemove = new List<Havok.Vdb.ServerInfo>();
                            bool clearAll = false;
                            if (clientDiscoveredServers.Count > 0)
                            {
                                // Look for servers that are no longer around
                                foreach (Havok.Vdb.ServerInfo oldDiscoveredServer in DiscoveredServers)
                                {
                                    int foundIdx = clientDiscoveredServers.FindIndex(server => oldDiscoveredServer.ToString().Equals(server.ToString()));

                                    // We found the server, so it sticks around.  Remove it from our discoveredServers list.
                                    if (foundIdx != -1)
                                    {
                                        clientDiscoveredServers.RemoveAt(foundIdx);
                                        System.Diagnostics.Debug.Assert(
                                            clientDiscoveredServers.FindIndex(server => oldDiscoveredServer.ToString().Equals(server.ToString())) == -1,
                                            "clientDiscoveredServers contains duplicates");
                                    }
                                    // The server no longer exists, add it to our remove list.
                                    else
                                    {
                                        serversToRemove.Add(oldDiscoveredServer);
                                    }
                                }

                                // Any servers remaining in clientDiscoveredServers are new
                                clientDiscoveredServers.ForEach(
                                    server =>
                                    {
                                        serversToAdd.Add(server);
                                    });
                            }
                            else
                            {
                                clearAll = true;
                            }

                            // Debug check for no duplicates
                            System.Diagnostics.Debug.Assert(serversToAdd.TrueForAll(server => !DiscoveredServers.Contains(server)), "DiscoveredServers already contains server to add");
                            System.Diagnostics.Debug.Assert(serversToRemove.TrueForAll(server => DiscoveredServers.Contains(server)), "DiscoveredServers does not contain server to remove");

                            // Submit changes to the UI thread
                            Application.Current.Dispatcher.InvokeAsync(() =>
                            {
                                if (clearAll)
                                {
                                    DiscoveredServers.Clear();
                                }
                                else
                                {
                                    serversToAdd.ForEach(server => DiscoveredServers.Add(server));
                                    serversToRemove.ForEach(server => DiscoveredServers.Remove(server));
                                }
                            });
                        }
                    }
                    catch
                    {
                        /*Client.Disconnect();*/
                    }
                }
            }
#if ENABLE_MAX_THROUGHPUT_TICKING
            private void TickProcess(bool highFreq)
            {
                if (m_suspendCount == 0)
                {
                    try
                    {
                        lock (m_client)
                        {
                            // Process more from the client
                            float processSeconds =
                                (highFreq) ?
                                    HighFreqProcessTickFreqInSeconds :
                                    ( (m_client.ConnectedSource == Havok.Vdb.ConnectedSource.File) ?
                                    // If we are connected to a file, we want to consume as much as we can in our time interval
                                    ((float)LowFreqProcessTickFreqInSeconds) :
                                    // If we are connected to the network, a great deal of work happens between server frames, don't
                                    // wasted time processing when we know the next frame won't be for awhile.
                                    -1.0f );
                            m_client.Process(processSeconds);

                            // If our client won't advance in frames we need to process callbacks that will let the UI refresh
                            // (see m_client.PlaybackHandler.PlaybackCmdReceived += ... in initialization code)
                            if ((!highFreq) && (
                                // Not connected to anything, definitely won't get frame callbacks.
                                (m_client.ConnectedState != Havok.Vdb.ConnectedState.Connected) ||
                                // Connected to a file, we will get frame callbacks, but no need to wait since all data is available.
                                // We wait in the server case to reduce client locks.
                                (m_client.ConnectedSource == Havok.Vdb.ConnectedSource.File) ||
                                // Paused, won't be getting frame callbacks.
                                (m_client.PlaybackHandler.PlaybackState == Havok.Vdb.PlaybackState.Paused)))
                            {
                                InvokePendingCallbacksAsync(false);
                            }
                        }
                    }
                    catch (Exception ex)
                    {
                        Dispatcher.CurrentDispatcher.InvokeAsync(() =>
                        {
                        // Note: repeated InvokeAsyncs are inherently dangerous as we can grow without bound
                        Application.Current.Dispatcher.Invoke(() =>
                            {
                                FwdErrored(ex);
                            });
                        });
                    }
                }
            }
            private void TickClient(object sender, EventArgs e)
            {
                long ticks = m_clientTickStopwatch.ElapsedTicks;
                if (ticks > m_nextSmallProcessTick)
                {
                    TickProcess(true);
                    m_nextSmallProcessTick = (long)(ticks + Stopwatch.Frequency * HighFreqProcessTickFreqInSeconds);
                }
                if (ticks > m_nextFullProcessTick)
                {
                    TickProcess(false);
                    m_nextFullProcessTick = (long)(ticks + Stopwatch.Frequency * LowFreqProcessTickFreqInSeconds);
                }
                if (ticks > m_nextAutoReconnectTick)
                {
                    // Ensure a high enough frequency so we don't get erroneously dropped discovered entries
                    float autoReconnectTickFreqInSeconds = (_ServerDiscoveryTimeoutMs / 3);
                    TickAutoReconnect(null, null);
                    m_nextAutoReconnectTick = (long)(ticks + Stopwatch.Frequency * autoReconnectTickFreqInSeconds);
                }
            }
#else
            private void TickClient(object sender, EventArgs e)
            {
                if (m_suspendCount == 0)
                {
                    Havok.Vdb.StepState state = Havok.Vdb.StepState.NotStarted;
                    try
                    {
                        lock (m_client)
                        {
                            // Process more from the client
                            state = m_client.Process(
                                (m_client.ConnectedSource == Havok.Vdb.ConnectedSource.File) ?
                                // If we are connected to a file, we want to consume as much as we can in our time interval
                                ((float)m_clientTickTimer.Interval.TotalSeconds) :
                                // If we are connected to the network, a great deal of work happens between server frames, don't
                                // wasted time processing when we know the next frame won't be for awhile.
                                -1.0f);

                            // If our client won't advance in frames we need to process callbacks that will let the UI refresh
                            // (see m_client.PlaybackHandler.PlaybackCmdReceived += ... in initialization code)
                            if (// Not connected to anything, definitely won't get frame callbacks.
                                (m_client.ConnectedState != Havok.Vdb.ConnectedState.Connected) ||
                                // Connected to a file, we will get frame callbacks, but no need to wait since all data is available.
                                // We wait in the server case to reduce client locks.
                                (m_client.ConnectedSource == Havok.Vdb.ConnectedSource.File) ||
                                // Paused, won't be getting frame callbacks.
                                (m_client.PlaybackHandler.PlaybackState == Havok.Vdb.PlaybackState.Paused))
                            {
                                InvokePendingCallbacksAsync(false);
                            }
                        }
                    }
                    catch (Exception ex)
                    {
                        Dispatcher.CurrentDispatcher.InvokeAsync(() =>
                        {
                            // Note: repeated InvokeAsyncs are inherently dangerous as we can grow without bound
                            Application.Current.Dispatcher.Invoke(() =>
                            {
                                FwdErrored(ex);
                            });
                        });
                    }
                }
            }
#endif
            private Dispatcher m_clientThreadDispatcher;
            private DispatcherTimer m_clientTickTimer;
#if ENABLE_MAX_THROUGHPUT_TICKING
            private Stopwatch m_clientTickStopwatch;
            private long m_nextSmallProcessTick;
            private long m_nextFullProcessTick;
            private long m_nextAutoReconnectTick;
            private const float HighFreqProcessTickFreqInSeconds = 0.001f;
            private const float LowFreqProcessTickFreqInSeconds = 0.01f;
#else
            private DispatcherTimer m_autoReconnectTimer;
#endif
#endregion
        }
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
