// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM		: WIN32 X64
// PRODUCT		: COMMON
// VISIBILITY	: CLIENT
//
// ------------------------------------------------------TKBMS v1.0

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.ComponentModel;
using System.IO;
using System.Linq;
using System.Xml.Serialization;

namespace HavokVisualDebugger
{
    public enum VdbFilterLayerMode
    {
        Disabled = 0,
        Enabled,
        Isolated
    }

    public abstract class VdbFilterLayerViewModel : ViewModelNotifyPropertyChanged
    {
        public Guid LayerGuid { get; set; }

        #region VdbFilterLayerViewModel Abstract

        public abstract bool IsMutable { get; }
        public abstract int FilteredObjectCount { get; }
        public abstract bool IsAffectingSimulation { get; }

        #endregion

        public VdbFilterLayerViewModel()
        {
            LayerGuid = Guid.NewGuid();
            SetModeCommand = new ViewModelDelegateCommand(
                this,
                (o) => Mode = (VdbFilterLayerMode)o);
        }

        #region VdbFilterLayerViewModel View Commands and Properties

        public string Name
        {
            get { return _Name; }
            set
            {
                if (_Name != value)
                {
                    _Name = value;
                    NotifyPropertyChanged();
                }
            }
        }
        protected string _Name;

        public string Filter
        {
            get { return _Filter; }
            set
            {
                if (_Filter != value)
                {
                    string oldFilter = _Filter;
                    _Filter = value;

                    // Update Name if name hadn't been changed
                    if (GetNameFromFilter(oldFilter) == _Name)
                    {
                        Name = GetNameFromFilter(_Filter);
                    }

                    NotifyPropertyChanged();
                }
            }
        }
        protected string _Filter;

        [XmlIgnore]
        public bool IsSetFilterShown
        {
            get { return _IsSetFilterShown; }
            set
            {
                if (_IsSetFilterShown != value)
                {
                    _IsSetFilterShown = value;
                    NotifyPropertyChanged();
                }
            }
        }
        protected bool _IsSetFilterShown;

        public VdbFilterLayerMode Mode
        {
            get { return _Mode; }
            set
            {
                if (_Mode != value)
                {
                    _Mode = value;
                    NotifyPropertyChanged();
                }
            }
        }
        protected VdbFilterLayerMode _Mode = VdbFilterLayerMode.Enabled;

        [XmlIgnore]
        public bool IsSetModeShown
        {
            get { return _IsSetModeShown; }
            set
            {
                if (_IsSetModeShown != value)
                {
                    _IsSetModeShown = value;
                    NotifyPropertyChanged();
                }
            }
        }
        protected bool _IsSetModeShown;

        [XmlIgnore]
        public System.Windows.Input.ICommand SetModeCommand { get; private set; }

        #endregion

        protected static string GetNameFromFilter(string filter)
        {
            return '"' + filter + '"' + " " + HavokVisualDebugger.Properties.Resources.ObjectTree_FilterLayer;
        }
    }

    public class VdbMutableFilterLayerViewModel : VdbFilterLayerViewModel, IDisposable
    {
        #region VdbMutableFilterLayerViewModel Inherited

        public override bool IsMutable { get { return true; } }
        public override int FilteredObjectCount { get { return m_renderObjects.Count; } }
        public override bool IsAffectingSimulation
        {
            get
            {
                return
                    (_Mode == VdbFilterLayerMode.Isolated) ||
                    (!m_owningCollection.AreAnyLayersIsolated && (_Mode == VdbFilterLayerMode.Enabled));
            }
        }

        #endregion

        #region VdbMutableFilterLayerViewModel Events

        public event Action<object> Disposed;
        public event Action<VdbMutableFilterLayerViewModel, IEnumerable<Havok.Vdb.RenderObject>, IEnumerable<Havok.Vdb.RenderObject>> Refreshed;

        #endregion

        #region VdbMutableFilterLayerViewModel Initialization

        public VdbMutableFilterLayerViewModel()
        {
            DisposeCommand = new ViewModelDelegateCommand(
                this,
                (o) => { Dispose(); });
            RefreshCommand = new ViewModelDelegateCommand(
                this,
                Refresh,
                new List<string> {
                    nameof(IsDirty),
                },
                new List<Func<object, bool>> {
                delegate(object o) {
                    return IsDirty;
                }});
            MoveUpCommand = new ViewModelDelegateCommand(
                this,
                (o) => m_owningCollection.Move(LayerIndex, LayerIndex - 1),
                new List<string> {
                    nameof(LayerIndex),
                },
                new List<Func<object, bool>> {
                    delegate(object o) {
                        return LayerIndex > 0;
                }});
            MoveDownCommand = new ViewModelDelegateCommand(
                this,
                (o) => m_owningCollection.Move(LayerIndex, LayerIndex + 1),
                new List<string> {
                    nameof(LayerIndex),
                },
                new List<Func<object, bool>> {
                    delegate(object o) {
                        return LayerIndex < (m_owningCollection.Count - 1);
                }});
        }

        public VdbMutableFilterLayerViewModel(
            VdbViewModel vm,
            VdbMutableFilterLayerCollection owningCollection,
            VdbObjectMultiParentTree objectTree,
            Havok.Vdb.RenderSurface renderSurface,
            string filter,
            ReadOnlyCollection<VdbObjectViewModel> matchingTreeViewObjects) :
            this()
        {
            _IsDirty = (matchingTreeViewObjects == null);
            _Filter = filter;
            _Name = GetNameFromFilter(filter);
            PostDeserializeInit(vm, owningCollection, objectTree, renderSurface);
            CollectRenderObjects(matchingTreeViewObjects);
        }

        private void PostDeserializeInit(
            VdbViewModel vm,
            VdbMutableFilterLayerCollection owningCollection,
            VdbObjectMultiParentTree objectTree,
            Havok.Vdb.RenderSurface renderSurface)
        {
            m_vm = vm;
            m_owningCollection = owningCollection;
            m_owningCollection.CollectionChanged += OwningCollection_CollectionChanged;
            m_objectTree = objectTree;
            m_objectTree.TreeModified += ObjectTree_TreeModified;
            m_objectTree.TreeItemsUpdated += ObjectTree_TreeItemsUpdated;
            m_renderSurface = renderSurface;
            Refresh(null);
        }

        public void Dispose()
        {
            m_objectTree.TreeModified -= ObjectTree_TreeModified;
            m_objectTree.TreeItemsUpdated -= ObjectTree_TreeItemsUpdated;
            m_owningCollection.CollectionChanged -= OwningCollection_CollectionChanged;
            Disposed?.Invoke(this);
        }

        #endregion

        #region VdbMutableFilterLayerViewModel View Commands and Properties

        [XmlIgnore]
        public bool IsDirty
        {
            get { return _IsDirty; }
            protected set
            {
                if (_IsDirty != value)
                {
                    _IsDirty = value;
                    NotifyPropertyChanged();
                }
            }
        }
        protected bool _IsDirty;

        public bool IsVisible
        {
            get { return _IsVisible; }
            set
            {
                if (_IsVisible != value)
                {
                    _IsVisible = value;
                    NotifyPropertyChanged();
                }
            }
        }
        private bool _IsVisible = true;

        public System.Windows.Media.Color Color
        {
            get { return _Color; }
            set
            {
                if (_Color != value)
                {
                    _Color = value;
                    NotifyPropertyChanged();
                }
            }
        }
        private System.Windows.Media.Color _Color = System.Windows.Media.Colors.White;

        [XmlIgnore]
        public bool IsSetColorShown
        {
            get { return _IsSetColorShown; }
            set
            {
                if (_IsSetColorShown != value)
                {
                    _IsSetColorShown = value;
                    NotifyPropertyChanged();
                }
            }
        }
        private bool _IsSetColorShown;

        public bool IsColorSet
        {
            get { return _IsColorSet; }
            set
            {
                if (_IsColorSet != value)
                {
                    _IsColorSet = value;
                    NotifyPropertyChanged();
                }
            }
        }
        private bool _IsColorSet = false;

        public int LayerIndex
        {
            get { return m_owningCollection.IndexOf(this); }
            set { /*required so serialization will save this property*/ }
        }

        public IEnumerable<Havok.Vdb.RenderObject> RenderObjects { get { return m_renderObjects; } }

        [XmlIgnore]
        public System.Windows.Input.ICommand DisposeCommand { get; private set; }
        [XmlIgnore]
        public System.Windows.Input.ICommand RefreshCommand { get; private set; }
        [XmlIgnore]
        public System.Windows.Input.ICommand MoveUpCommand { get; private set; }
        [XmlIgnore]
        public System.Windows.Input.ICommand MoveDownCommand { get; private set; }

        #endregion

        #region VdbMutableFilterLayerViewModel Public Methods

        public static string Serialize(VdbMutableFilterLayerViewModel layer)
        {
            try
            {
                XmlSerializer xmlSerializer = new XmlSerializer(typeof(VdbMutableFilterLayerViewModel));
                using (StringWriter textWriter = new StringWriter())
                {
                    xmlSerializer.Serialize(textWriter, layer);
                    return textWriter.ToString();
                }
            }
            catch (Exception e)
            {
                
                System.Diagnostics.Debug.Assert(false, e.Message);
            }
            return "";
        }

        public static VdbMutableFilterLayerViewModel Deserialize(
            string stream,
            VdbViewModel vm,
            VdbMutableFilterLayerCollection owningCollection,
            VdbObjectMultiParentTree objectTree,
            Havok.Vdb.RenderSurface renderSurface)
        {
            try
            {
                XmlSerializer xmlSerializer = new XmlSerializer(typeof(VdbMutableFilterLayerViewModel));
                using (System.Xml.XmlReader reader = System.Xml.XmlReader.Create(new StringReader(stream)))
                {
                    VdbMutableFilterLayerViewModel deserialized = (VdbMutableFilterLayerViewModel)xmlSerializer.Deserialize(reader);
                    if (deserialized != null)
                    {
                        deserialized.PostDeserializeInit(vm, owningCollection, objectTree, renderSurface);
                        return deserialized;
                    }
                }
            }
            catch (Exception e)
            {
                
                System.Diagnostics.Debug.Assert(false, e.Message);
            }
            return null;
        }

        #endregion

        #region VdbMutableFilterLayerViewModel Private Methods and Properties

        
        private void ObjectTree_TreeModified(object sender, VdbObjectMultiParentTreeModifiedEvent e) { IsDirty = true; }
        private void ObjectTree_TreeItemsUpdated(object sender, VdbObjectMultiParentTreeItemsUpdatedEvent e) { IsDirty = true; }
        private void OwningCollection_CollectionChanged(object sender, NotifyCollectionChangedEventArgs e) { NotifyPropertyChanged(nameof(LayerIndex)); }
        protected override void OnPropertyChanged(PropertyChangedEventArgs e)
        {
            // Don't send property changes if we aren't initialized
            if (m_vm == null)
            {
                return;
            }

            switch (e.PropertyName)
            {
                case nameof(Filter):
                {
                    // If the filter changed, we are no longer up-to-date
                    IsDirty = true;

                    // Refresh the object set since a new filter has been created
                    if (RefreshCommand.CanExecute(null))
                    {
                        RefreshCommand.Execute(null);
                    }
                    break;
                }
                case nameof(Mode):
                {
                    // Changes to the mode can impact if other layers are affecting the simulation.
                    foreach (VdbMutableFilterLayerViewModel layer in m_owningCollection)
                    {
                        layer.NotifyPropertyChanged(nameof(IsAffectingSimulation));
                    }
                    break;
                }
            }
            base.OnPropertyChanged(e);
        }

        private void Refresh(object sender)
        {
            List<SearchFilter> searchFilters = new List<SearchFilter>();
            SearchFilter.GenerateSearchFilters(Filter, searchFilters);

            Dictionary<VdbObjectModel, VdbObjectViewModel> objectVms = new Dictionary<VdbObjectModel, VdbObjectViewModel>();
            List<VdbObjectViewModel> matchingTreeObjectVms;
            m_objectTree.Root.CreateTreeView(
                m_vm,
                searchFilters,
                objectVms,
                out matchingTreeObjectVms);

            IEnumerable<Havok.Vdb.RenderObject> oldRenderObjects = m_renderObjects;
            HashSet<UInt64> oldRenderObjectsSet = new HashSet<UInt64>(oldRenderObjects.Select(r => r.Id));
            CollectRenderObjects(matchingTreeObjectVms.AsReadOnly());

            IEnumerable<Havok.Vdb.RenderObject> newRenderObjects = m_renderObjects;
            HashSet<UInt64> newRenderObjectsSet = new HashSet<UInt64>(newRenderObjects.Select(r => r.Id));
            foreach (VdbObjectViewModel ovm in objectVms.Values)
            {
                ovm.Dispose();
            }

            Refreshed?.Invoke(
                this,
                // These are objects which were removed
                oldRenderObjects.Where(r => !newRenderObjectsSet.Contains(r.Id)),
                // These are objects which were added
                newRenderObjects.Where(r => !oldRenderObjectsSet.Contains(r.Id)));

            IsDirty = false;
        }
        private void CollectRenderObjects(ReadOnlyCollection<VdbObjectViewModel> treeViewObjects)
        {
            m_renderObjects = new List<Havok.Vdb.RenderObject>();

            if (treeViewObjects != null)
            {
                HashSet<UInt64> processedIds = new HashSet<UInt64>();
                foreach (VdbObjectViewModel treeViewObject in treeViewObjects)
                {
                    VdbObjectViewModel displayable = treeViewObject.FindContainingDisplayable();
                    if (displayable != null)
                    {
                        m_renderObjects.AddRange(displayable.GetRenderObjects(processedIds));
                    }
                }
            }

            NotifyPropertyChanged(nameof(RenderObjects));
            NotifyPropertyChanged(nameof(FilteredObjectCount));
        }

        private VdbViewModel m_vm;
        private VdbMutableFilterLayerCollection m_owningCollection;
        private VdbObjectMultiParentTree m_objectTree;
        private Havok.Vdb.RenderSurface m_renderSurface;
        private List<Havok.Vdb.RenderObject> m_renderObjects = new List<Havok.Vdb.RenderObject>();

        #endregion
    }

    public class VdbFixedFilterLayerViewModel : VdbFilterLayerViewModel
    {
        #region VdbFixedFilterLayerViewModel Inherited

        public override bool IsMutable { get { return false; } }
        public override int FilteredObjectCount { get { return _FilteredObjectCount; } }
        private int _FilteredObjectCount = 0;
        public override bool IsAffectingSimulation { get { return (_FilteredObjectCount > 0); } }

        #endregion

        public VdbFixedFilterLayerViewModel(VdbSelection selection)
        {
            _Name = Properties.Resources.ObjectTree_BaseLayerTitle;
            _Filter = Properties.Resources.ObjectTree_BaseLayerDescription;
            selection.SelectedObjects.CollectionChanged += HighlightedOrSelectedVdbObjects_CollectionChanged;
            selection.HighlightedObjects.CollectionChanged += HighlightedOrSelectedVdbObjects_CollectionChanged;
        }

        private void HighlightedOrSelectedVdbObjects_CollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            System.Diagnostics.Debug.Assert(e.Action != NotifyCollectionChangedAction.Reset, "Not expecting this event");
            _FilteredObjectCount += (e.NewItems != null) ? e.NewItems.Count : 0;
            _FilteredObjectCount -= (e.OldItems != null) ? e.OldItems.Count : 0;
            NotifyPropertyChanged(nameof(FilteredObjectCount));
            NotifyPropertyChanged(nameof(IsAffectingSimulation));
        }
    }

    public class VdbMutableFilterLayerCollection : ObservableCollection<VdbMutableFilterLayerViewModel>
    {
        #region VdbMutableFilterLayerCollection Initialization
        public VdbMutableFilterLayerCollection(
            VdbViewModel owner,
            Havok.Vdb.RenderSurface renderSurface,
            VdbFixedFilterLayerViewModel baseLayer,
            hkNonResettableObservableCollection<VdbProcessesViewModel> processesCollection)
        {
            m_owner = owner;
            m_renderSurface = renderSurface;
            m_baseLayer = baseLayer;
            CreateNewLayerCommand = new ViewModelDelegateCommand(
                this,
                (o) => CreateNewLayer(o as string));
            SetModeOnAllLayersCommand = new ViewModelDelegateCommand(
                this,
                (o) => SetModeOnAllLayers((VdbFilterLayerMode)o));
            SetVisibilityOnAllLayersCommand = new ViewModelDelegateCommand(
                this,
                (o) => SetVisibilityOnAllLayers(null));
            RefreshAllLayersCommand = new ViewModelDelegateCommand(
                this,
                (o) => RefreshAllLayers(true));
            ClearAllLayersCommand = new ViewModelDelegateCommand(
                this,
                (o) => Clear());
            m_owner.AutoRefreshTick += delegate
            {
                if (AutoRefreshEnabledLayers)
                {
                    RefreshAllLayers(false);
                }
            };

            m_baseLayer.PropertyChanged += VdbFilterLayerViewModel_PropertyChanged;
            processesCollection.CollectionChanged += ProcessesCollection_NotifyCollectionChanged;
            foreach (VdbProcessesViewModel processes in processesCollection)
            {
                OnProcessesAdded(processes);
            }

            owner.Client.GetSubInterop(nameof(Havok.Vdb.Client.PlaybackHandler)).
                ListenTo<Havok.Vdb.PlaybackCmdReceivedHandler, Havok.Vdb.PlaybackHandler, Havok.Vdb.PlaybackCmdReceivedArgs>(
                    nameof(Havok.Vdb.PlaybackHandler.PlaybackCmdReceived),
                    delegate (Havok.Vdb.PlaybackHandler sender, Havok.Vdb.PlaybackCmdReceivedArgs args)
                    {
                        if (args.FlagWasSet(Havok.Vdb.PlaybackFlags.FrameEnded))
                        {
                            // This seems valuable enough for the "initialization" case, and hidden enough
                            // by the turning on of viewers, that we should just do it regardless of
                            // AutoRefreshEnabledLayers...but revisit if it's too slow
                            if (// AutoRefreshEnabledLayers &&
                                m_viewerCreatedLastFrame)
                            {
                                // Because this happens so infrequently, and is part of a viewer enabled
                                // action, let's just refresh all layers.  This helps in the connection case
                                // to where someone may expect all the layers (enabled or not) to be *initially*
                                // correct after connection.
                                RefreshAllLayers();
                            }
                            m_viewerCreatedLastFrame = false;
                        }
                    });
        }
        #endregion

        #region VdbMutableFilterLayerCollection Factory Methods and Commands

        public VdbMutableFilterLayerViewModel CreateNewLayerFromStream(string stream)
        {
            VdbMutableFilterLayerViewModel layer = VdbMutableFilterLayerViewModel.Deserialize(
                stream,
                m_owner,
                this,
                m_owner.VdbObjectTree,
                m_renderSurface);

            if (layer != null)
            {
                // Goes first since we apply layers sequentially, last one in is the winner.
                Insert(0, layer);
            }
            return layer;
        }

        public VdbMutableFilterLayerViewModel CreateNewLayer(string searchFilter = null)
        {
            bool refreshResults = (searchFilter != null) && (searchFilter != m_owner.VdbObjectQuery);

            VdbMutableFilterLayerViewModel layer = new VdbMutableFilterLayerViewModel(
                m_owner,
                this,
                m_owner.VdbObjectTree,
                m_renderSurface,
                searchFilter ?? m_owner.VdbObjectQuery,
                refreshResults ? (new List<VdbObjectViewModel>()).AsReadOnly() : m_owner.VdbObjectQueryResults);

            // Goes first since we apply layers sequentially, last one in is the winner.
            Insert(0, layer);
            if (refreshResults)
            {
                layer.RefreshCommand.Execute(null);
            }

            return layer;
        }

        public System.Windows.Input.ICommand CreateNewLayerCommand { get; private set; }

        #endregion

        #region VdbMutableFilterLayerCollection Public Methods, Properties, and Commands

        public bool AreAnyLayersIsolated
        {
            get { return this.Any(x => (x.Mode == VdbFilterLayerMode.Isolated)); }
        }

        public bool AreAnyLayersVisible
        {
            get { return this.Any(x => x.IsVisible); }
        }

        public bool AreAllLayersVisible
        {
            get { return (Count > 0) && this.All(x => x.IsVisible); }
        }

        public bool AreAllLayersEnabled
        {
            get { return (Count > 0) && this.All(x => (x.Mode == VdbFilterLayerMode.Enabled)); }
        }

        public bool AreAllLayersDisabled
        {
            get { return (Count > 0) && this.All(x => (x.Mode == VdbFilterLayerMode.Disabled)); }
        }

        public bool AreAllLayersIsolated
        {
            get { return (Count > 0) && this.All(x => (x.Mode == VdbFilterLayerMode.Isolated)); }
        }

        public bool AutoRefreshEnabledLayers
        {
            get
            {
                return VdbSettings.GetValue<bool>(nameof(Properties.Settings.VdbUi_AutoRefreshLayersEnabled));
            }
            set
            {
                if (AutoRefreshEnabledLayers != value)
                {
                    OnPropertyChanged(new PropertyChangedEventArgs(nameof(AutoRefreshEnabledLayers)));
                    if (value)
                    {
                        RefreshAllLayers(false);
                    }

                    VdbSettings.SetValue(nameof(Properties.Settings.VdbUi_AutoRefreshLayersEnabled), value);
                }
            }
        }

        public void SetModeOnAllLayers(VdbFilterLayerMode mode)
        {
            using (new hkResetOnDispose(this, nameof(ModifyingAllLayers), true))
            {
                foreach (VdbMutableFilterLayerViewModel layer in this)
                {
                    layer.Mode = mode;
                }
            }
        }

        public void SetVisibilityOnAllLayers(bool? isVisible)
        {
            if (!isVisible.HasValue)
            {
                isVisible = !AreAllLayersVisible;
            }

            using (new hkResetOnDispose(this, nameof(ModifyingAllLayers), true))
            {
                foreach (VdbMutableFilterLayerViewModel layer in this)
                {
                    layer.IsVisible = isVisible.Value;
                }
            }
        }

        public void RefreshAllLayers(bool refreshDisabledLayers = true)
        {
            using (new hkResetOnDispose(this, nameof(ModifyingAllLayers), true))
            {
                foreach (VdbMutableFilterLayerViewModel layer in this)
                {
                    if (((layer.Mode != VdbFilterLayerMode.Disabled) || refreshDisabledLayers) &&
                        layer.RefreshCommand.CanExecute(this))
                    {
                        layer.RefreshCommand.Execute(this);
                    }
                }
            }
        }

        public System.Windows.Input.ICommand SetModeOnAllLayersCommand { get; private set; }
        public System.Windows.Input.ICommand SetVisibilityOnAllLayersCommand { get; private set; }
        public System.Windows.Input.ICommand RefreshAllLayersCommand { get; private set; }
        public System.Windows.Input.ICommand ClearAllLayersCommand { get; private set; }

        #endregion

        #region VdbMutableFilterLayerCollection Private Methods and Properties

        private void ProcessesCollection_NotifyCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
        {
            System.Diagnostics.Debug.Assert(e.Action != NotifyCollectionChangedAction.Reset, "Unexpected; use hkObservableCollection");

            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                case NotifyCollectionChangedAction.Remove:
                case NotifyCollectionChangedAction.Replace:
                case NotifyCollectionChangedAction.Move:
                {
                    if (e.Action != NotifyCollectionChangedAction.Add)
                    {
                        foreach (VdbProcessesViewModel processes in e.OldItems)
                        {
                            OnProcessesRemoved(processes);
                        }
                    }
                    if (e.Action != NotifyCollectionChangedAction.Remove)
                    {
                        foreach (VdbProcessesViewModel processes in e.NewItems)
                        {
                            OnProcessesAdded(processes);
                        }
                    }
                    break;
                }
            }
        }

        private void OnProcessesAdded(VdbProcessesViewModel processes)
        {
            processes.Processes.RegisterProcessCallback(OnProcessRegistered);
        }

        private void OnProcessesRemoved(VdbProcessesViewModel processes)
        {
            processes.Processes.UnRegisterProcessCallback(OnProcessRegistered);
        }

        private void OnProcessRegistered(VdbProcessViewModel process, VdbProcessCollection.ProcessEvent evt)
        {
            if (evt == VdbProcessCollection.ProcessEvent.FirstStep)
            {
                m_viewerCreatedLastFrame = true;
            }
        }

        private void VdbFilterLayerViewModel_PropertyChanged(object sender, PropertyChangedEventArgs e)
        {
            VdbFilterLayerViewModel layer = (sender as VdbFilterLayerViewModel);
            VdbMutableFilterLayerViewModel mutableLayer = (sender as VdbMutableFilterLayerViewModel);
            System.Diagnostics.Debug.Assert(layer.IsMutable == (mutableLayer != null), "Inconsistent IsMutable reporting");

            // Affect our render objects
            switch (e.PropertyName)
            {
                case nameof(VdbFilterLayerViewModel.FilteredObjectCount):
                case nameof(VdbMutableFilterLayerViewModel.RenderObjects):
                {
                    // For mutable layers, this is handled during "Refreshed" signal.
                    if (!layer.IsMutable)
                    {
                        UpdateRenderObjects(e.PropertyName);
                    }
                    break;
                }
                // Since LayerIndex is *derived* from the layers place in the collection, we don't need to do
                // anything, we've already updated render objects when we received the collection changed event.
                case nameof(VdbMutableFilterLayerViewModel.LayerIndex):
                // These don't affect render objects
                
                case nameof(VdbFilterLayerViewModel.Name):
                case nameof(VdbFilterLayerViewModel.Filter):
                case nameof(VdbFilterLayerViewModel.IsSetFilterShown):
                case nameof(VdbFilterLayerViewModel.IsSetModeShown):
                case nameof(VdbFilterLayerViewModel.IsAffectingSimulation):
                case nameof(VdbMutableFilterLayerViewModel.IsDirty):
                case nameof(VdbMutableFilterLayerViewModel.IsSetColorShown):
                {
                    // No-op
                    break;
                }
                // If these change, the render objects need to be refreshed
                case nameof(VdbFilterLayerViewModel.Mode):
                case nameof(VdbMutableFilterLayerViewModel.IsVisible):
                case nameof(VdbMutableFilterLayerViewModel.IsColorSet):
                case nameof(VdbMutableFilterLayerViewModel.Color):
                {
                    UpdateRenderObjects(e.PropertyName);
                    break;
                }
                default:
                {
                    System.Diagnostics.Debug.Assert(false, "New property");
                    break;
                }
            }

            // Do dependent notifies
            switch (e.PropertyName)
            {
                case nameof(VdbFilterLayerViewModel.Mode):
                {
                    NotifyLayerModeChangedProperties();
                    break;
                }
                case nameof(VdbMutableFilterLayerViewModel.IsVisible):
                {
                    NotifyLayerVisibilityChangedProperties();
                    break;
                }
            }

            // Update saved settings for mutable filter layers
            if ((mutableLayer != null) &&
                !hkReflect.IsXmlIgnored(typeof(VdbMutableFilterLayerViewModel), e.PropertyName))
            {
                StringCollection layers = VdbSettings.GetValue<StringCollection>(nameof(Properties.Settings.VdbLayers));
                int foundIdx = FindLayerInSettings(mutableLayer, layers);
                if (foundIdx >= 0)
                {
                    layers[foundIdx] = VdbMutableFilterLayerViewModel.Serialize(mutableLayer);
                }
                Properties.Settings.Default.Save();
            }
        }

        private void VdbMutableFilterLayerViewModel_Disposed(object sender)
        {
            VdbMutableFilterLayerViewModel layer = (VdbMutableFilterLayerViewModel)sender;
            Remove(layer);
        }

        protected override void ClearItems()
        {
            using (new hkResetOnDispose(this, nameof(ModifyingAllLayers), true))
            {
                foreach (VdbMutableFilterLayerViewModel layer in this)
                {
                    OnFilterLayerRemoved(layer);
                }
                base.ClearItems();
            }
        }

        protected override void OnCollectionChanged(NotifyCollectionChangedEventArgs e)
        {
            switch (e.Action)
            {
                case NotifyCollectionChangedAction.Add:
                case NotifyCollectionChangedAction.Remove:
                case NotifyCollectionChangedAction.Replace:
                case NotifyCollectionChangedAction.Move:
                {
                    if (e.Action != NotifyCollectionChangedAction.Add)
                    {
                        foreach (VdbMutableFilterLayerViewModel layer in e.OldItems)
                        {
                            OnFilterLayerRemoved(layer);
                        }
                    }
                    if (e.Action != NotifyCollectionChangedAction.Remove)
                    {
                        List<Havok.Vdb.RenderObject> modifiedItems = new List<Havok.Vdb.RenderObject>(e.NewItems.Count * 100);
                        foreach (VdbMutableFilterLayerViewModel layer in e.NewItems)
                        {
                            OnFilterLayerAdded(layer);
                            modifiedItems.AddRange(layer.RenderObjects);
                        }
                        SortRenderObjectsRegistry(modifiedItems);
                    }
                    UpdateRenderObjects();
                    break;
                }
                // Note NotifyCollectionChangedAction.Reset happens after clearing items and doesn't provide them here, so we catch that in ClearItems() override
            }
            base.OnCollectionChanged(e);
        }

        private void OnFilterLayerAdded(VdbMutableFilterLayerViewModel layer)
        {
            UpdateRenderObjectsRegistry(layer, new List<Havok.Vdb.RenderObject>(), layer.RenderObjects);
            layer.PropertyChanged += VdbFilterLayerViewModel_PropertyChanged;
            layer.Disposed += VdbMutableFilterLayerViewModel_Disposed;
            layer.Refreshed += VdbMutableFilterLayerViewModel_Refreshed;

            UpdateAppSettings();
            m_owner.IsFilterLayerPanelShown = true;
            NotifyLayerCollectionChangedProperties();
        }

        private void OnFilterLayerRemoved(VdbMutableFilterLayerViewModel layer)
        {
            layer.PropertyChanged -= VdbFilterLayerViewModel_PropertyChanged;
            layer.Disposed -= VdbMutableFilterLayerViewModel_Disposed;
            layer.Refreshed -= VdbMutableFilterLayerViewModel_Refreshed;

            UpdateRenderObjectsRegistry(layer, layer.RenderObjects, new List<Havok.Vdb.RenderObject>());
            UpdateAppSettings();
            NotifyLayerCollectionChangedProperties();
        }

        private void VdbMutableFilterLayerViewModel_Refreshed(
            VdbMutableFilterLayerViewModel layer,
            IEnumerable<Havok.Vdb.RenderObject> removedItems,
            IEnumerable<Havok.Vdb.RenderObject> addedItems)
        {
            UpdateRenderObjectsRegistry(layer, removedItems, addedItems);
            // We can't make any assumption about the layer's layerIndex (like we can with OnFilterLayerAdded).
            // So we must do a full resort operation here.
            SortRenderObjectsRegistry(addedItems);
            UpdateRenderObjects();
        }

        private void UpdateRenderObjectsRegistry(
            VdbMutableFilterLayerViewModel layer,
            IEnumerable<Havok.Vdb.RenderObject> removedItems,
            IEnumerable<Havok.Vdb.RenderObject> addedItems)
        {
#if DEBUG
            HashSet<UInt64> encounteredIds = new HashSet<UInt64>();
#endif
            foreach (Havok.Vdb.RenderObject addedItem in addedItems)
            {
                RenderObjectWrapper wrapper;
                if (!m_renderObjects.TryGetValue(addedItem.Id, out wrapper))
                {
                    wrapper = new RenderObjectWrapper(addedItem);
                    m_renderObjects.Add(addedItem.Id, wrapper);
                }
                if (!wrapper.m_referencingLayers.Contains(layer))
                {
                    // Note: these get sorted at various times by layer index in collection, but
                    // placing it first here ensures that in the added case, if we neglect to sort
                    // it will give expected behavior since we apply layers sequentially.
                    wrapper.m_referencingLayers.Insert(0, layer);
                }
                else
                {
                    // This is case is unexpected, but since the # of referencevms should be pretty low,
                    // we still check that m_referencingLayers doesn't contain the layer, even in release.
                    System.Diagnostics.Debug.Assert(false);
                }
#if DEBUG
                System.Diagnostics.Debug.Assert(encounteredIds.Add(addedItem.Id), "Duplicated Id encountered");
#endif
            }

            foreach (Havok.Vdb.RenderObject removedItem in removedItems)
            {
                RenderObjectWrapper wrapper;
                if (m_renderObjects.TryGetValue(removedItem.Id, out wrapper))
                {
                    bool removed = wrapper.m_referencingLayers.Remove(layer);
                    System.Diagnostics.Debug.Assert(removed, "Trying to remove referencing layer that was never added");
                    if (wrapper.m_referencingLayers.Count == 0)
                    {
                        wrapper.Dispose();
                        m_renderObjects.Remove(removedItem.Id);
                    }
                }
                else
                {
                    System.Diagnostics.Debug.Assert(false, "Trying to remove object that was never added");
                }
#if DEBUG
                System.Diagnostics.Debug.Assert(encounteredIds.Add(removedItem.Id), "Duplicated Id encountered");
#endif
            }
        }

        private void SortRenderObjectsRegistry(IEnumerable<Havok.Vdb.RenderObject> modifiedItems)
        {
            LayerComparer comparer = new LayerComparer(this);
            foreach (Havok.Vdb.RenderObject modifiedItem in modifiedItems)
            {
                RenderObjectWrapper wrapper;
                if (m_renderObjects.TryGetValue(modifiedItem.Id, out wrapper))
                {
                    wrapper.m_referencingLayers.Sort(comparer);
                }
                else
                {
                    System.Diagnostics.Debug.Assert(false, "Trying to sort layers for object that was never added");
                }
            }
        }

        private void UpdateRenderObjects(string optionalPropertyName = null)
        {
            // Early out for batch modifying
            if (ModifyingAllLayers)
            {
                return;
            }

            // Enable the global isolation mechanic if needed
            VdbFilterLayerMode enabledMode;
            if (AreAnyLayersIsolated || ((m_baseLayer.Mode == VdbFilterLayerMode.Isolated) && (m_baseLayer.IsAffectingSimulation)))
            {
                m_owner.Client.RenderSurface.RenderFlags |= Havok.Vdb.RenderFlags.HideAll;
                enabledMode = VdbFilterLayerMode.Isolated;
            }
            else
            {
                m_owner.Client.RenderSurface.RenderFlags &= ~Havok.Vdb.RenderFlags.HideAll;
                enabledMode = VdbFilterLayerMode.Enabled;
            }

            // Now affect all the render objects
            foreach (RenderObjectWrapper renderObjectWrapper in m_renderObjects.Values)
            {
                // Determine combined effect on render object.
                System.Windows.Media.Color? color = null;
                Havok.Vdb.RenderObjectVisibility? visibility = null;

                // Apply implicit/base layer affects
                // Note: If the object is highlighted or selected it's visible via the backend which we represent
                //       via the highlighted/selected base layer, but this ensures it's properly colored
                //       by the selection system.
                if (renderObjectWrapper.m_renderObject.IsHighlighted || renderObjectWrapper.m_renderObject.IsSelected)
                {
                    visibility = Havok.Vdb.RenderObjectVisibility.AlwaysVisible;
                }

                // Process mutable layers
                {
                    int disabledLayerCount = 0;
                    foreach (VdbMutableFilterLayerViewModel layer in renderObjectWrapper.m_referencingLayers)
                    {
                        if (layer.Mode == enabledMode)
                        {
                            if (layer.IsVisible)
                            {
                                if ((color == null) && (layer.IsColorSet)) color = layer.Color;
                                if (visibility == null) visibility = Havok.Vdb.RenderObjectVisibility.AlwaysVisible;
                            }
                        }
                        else
                        {
                            disabledLayerCount++;
                        }

                        // If we've set both, we can break.
                        if ((color != null) && (visibility != null))
                        {
                            break;
                        }
                    }

                    // If some enabled layer existed which set the object to *not* visible,
                    // and there was *no* enabled layer to set it *to* visible, we need the
                    // the object to be hidden.
                    if ((visibility == null) &&
                        (disabledLayerCount != renderObjectWrapper.m_referencingLayers.Count))
                    {
                        visibility = Havok.Vdb.RenderObjectVisibility.NeverVisible;
                    }
                }

                // Apply to render object.
                {
                    // Hide *first* in case it helps back end optimize.
                    if (visibility.HasValue && (visibility.Value == Havok.Vdb.RenderObjectVisibility.NeverVisible))
                    {
                        renderObjectWrapper.SetVisibility(Havok.Vdb.RenderObjectVisibility.NeverVisible);
                    }

                    // Set other properties in between.
                    {
                        if (color.HasValue)
                        {
                            renderObjectWrapper.SetColor(color.Value);
                        }
                        else
                        {
                            renderObjectWrapper.RevertColor();
                        }
                    }

                    // Show *last* in case it helps back end optimize (original value is *most likely* Hideable and therefore visible in most cases).
                    if (visibility.HasValue && (visibility.Value == Havok.Vdb.RenderObjectVisibility.AlwaysVisible))
                    {
                        renderObjectWrapper.SetVisibility(visibility.Value);
                    }

                    if (!visibility.HasValue)
                    {
                        renderObjectWrapper.RevertVisibility();
                    }
                }
            }
        }

        private void UpdateAppSettings()
        {
            // Early out for batch modifying
            if (ModifyingAllLayers)
            {
                return;
            }

            // Get layers from our settings
            StringCollection layers = VdbSettings.GetValue<StringCollection>(nameof(Properties.Settings.VdbLayers));

            // Update the saved layers list
            layers.Clear();
            foreach (VdbMutableFilterLayerViewModel layer in this)
            {
                layers.Add(VdbMutableFilterLayerViewModel.Serialize(layer));
            }

            // Save back to settings
            Properties.Settings.Default.Save();
        }

        private void NotifyLayerModeChangedProperties()
        {
            OnPropertyChanged(new PropertyChangedEventArgs(nameof(AreAllLayersDisabled)));
            OnPropertyChanged(new PropertyChangedEventArgs(nameof(AreAllLayersEnabled)));
            OnPropertyChanged(new PropertyChangedEventArgs(nameof(AreAllLayersIsolated)));
            OnPropertyChanged(new PropertyChangedEventArgs(nameof(AreAnyLayersIsolated)));
        }

        private void NotifyLayerVisibilityChangedProperties()
        {
            OnPropertyChanged(new PropertyChangedEventArgs(nameof(AreAllLayersVisible)));
            OnPropertyChanged(new PropertyChangedEventArgs(nameof(AreAnyLayersVisible)));
        }

        private void NotifyLayerCollectionChangedProperties()
        {
            NotifyLayerModeChangedProperties();
            NotifyLayerVisibilityChangedProperties();
        }

        private bool ModifyingAllLayers
        {
            get { return _ModifyingAllLayers; }
            set
            {
                if (_ModifyingAllLayers != value)
                {
                    _ModifyingAllLayers = value;
                    if (!_ModifyingAllLayers)
                    {
                        UpdateRenderObjects();
                        UpdateAppSettings();
                        // These notifications impact all properties affected by unknown layer changes
                        NotifyLayerCollectionChangedProperties();
                    }
                }
            }
        }
        private bool _ModifyingAllLayers;

        public static int FindLayerInSettings(VdbMutableFilterLayerViewModel layer, StringCollection layerSettings = null)
        {
            StringCollection layers = layerSettings ?? VdbSettings.GetValue<StringCollection>(nameof(Properties.Settings.VdbLayers));

            int foundIdx = -1;
            for (int i = 0; i < layers.Count; i++)
            {
                string layerXml = layers[i];
                if (layerXml.Contains(layer.LayerGuid.ToString()))
                {
                    foundIdx = i;
                    break;
                }
            }

            return foundIdx;
        }

        private class LayerComparer : Comparer<VdbMutableFilterLayerViewModel>
        {
            public LayerComparer(ObservableCollection<VdbMutableFilterLayerViewModel> collection)
            {
                m_collection = collection;
            }
            public override int Compare(VdbMutableFilterLayerViewModel layer1, VdbMutableFilterLayerViewModel layer2)
            {
                int idx1 = m_collection.IndexOf(layer1);
                int idx2 = m_collection.IndexOf(layer2);
                System.Diagnostics.Debug.Assert(idx1 != -1 && idx2 != -1);
                return idx1 - idx2;
            }
            private ObservableCollection<VdbMutableFilterLayerViewModel> m_collection;
        }

        // This wrapper holds layer references and also stores the value of render object properties
        // at the time they are set by this system so they can be properly reverted.
        private class RenderObjectWrapper : IDisposable
        {
            public RenderObjectWrapper(Havok.Vdb.RenderObject renderObject)
            {
                m_renderObject = renderObject;
            }

            public void SetVisibility(Havok.Vdb.RenderObjectVisibility visibility)
            {
                if (m_revertVisibility == null)
                {
                    m_revertVisibility = new hkRevertOnDispose(
                        m_renderObject,
                        nameof(Havok.Vdb.RenderObject.Visibility));
                }
                m_renderObject.Visibility = visibility;
            }

            public void RevertVisibility()
            {
                if (m_revertVisibility != null)
                {
                    m_revertVisibility.Dispose();
                    m_revertVisibility = null;
                }
            }

            public void SetColor(System.Windows.Media.Color color)
            {
                if (m_revertColor == null)
                {
                    m_revertColor = new hkRevertOnDispose(
                        m_renderObject,
                        nameof(Havok.Vdb.RenderObject.UserColor),
                        nameof(Havok.Vdb.RenderObject.EnableUserColor));
                }
                m_renderObject.UserColor = color;
                m_renderObject.EnableUserColor = true;
            }

            public void RevertColor()
            {
                if (m_revertColor != null)
                {
                    m_revertColor.Dispose();
                    m_revertColor = null;
                }
            }

            public void Dispose()
            {
                RevertVisibility();
                RevertColor();
            }

            public Havok.Vdb.RenderObject m_renderObject;
            public List<VdbMutableFilterLayerViewModel> m_referencingLayers = new List<VdbMutableFilterLayerViewModel>();
            private hkRevertOnDispose m_revertVisibility;
            private hkRevertOnDispose m_revertColor;
        }

        private VdbViewModel m_owner;
        private Havok.Vdb.RenderSurface m_renderSurface;
        private VdbFixedFilterLayerViewModel m_baseLayer;
        private Dictionary<UInt64, RenderObjectWrapper> m_renderObjects = new Dictionary<UInt64, RenderObjectWrapper>();
        private bool m_viewerCreatedLastFrame;

        #endregion
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
