// TKBMS v1.0 -----------------------------------------------------
//
// PLATFORM		: WIN32 X64
// PRODUCT		: COMMON
// VISIBILITY	: CLIENT
//
// ------------------------------------------------------TKBMS v1.0

using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Globalization;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;

namespace HavokVisualDebugger
{
    class BoolToVisibilityConverter : IValueConverter
    {
        public object Convert(object value, Type targetType, object parameter, CultureInfo cultureInfo)
        {
            if (!(value is bool))
            {
                if (parameter as string == "Hidden")
                {
                    return Visibility.Hidden;
                }
                return Visibility.Collapsed;
            }

            bool objValue = (bool)value;
            if (objValue)
            {
                return Visibility.Visible;
            }

            if (parameter as string == "Hidden")
            {
                return Visibility.Hidden;
            }
            return Visibility.Collapsed;
        }
        public object ConvertBack(object value, Type targetType, object parameter, CultureInfo cultureInfo)
        {
            try
            {
                if ((bool)value)
                {
                    return Visibility.Visible;
                }
            }
            catch { }
            return Visibility.Collapsed;
        }
    }

    class NegateBoolToVisibilityConverter : IValueConverter
    {
        public object Convert(object value, Type targetType, object parameter, CultureInfo cultureInfo)
        {
            if (!(value is bool))
            {
                return Visibility.Visible;
            }

            bool objValue = (bool)value;
            if (!objValue)
            {
                return Visibility.Visible;
            }

            if (parameter as string == "Hidden")
            {
                return Visibility.Hidden;
            }
            return Visibility.Collapsed;
        }
        public object ConvertBack(object value, Type targetType, object parameter, CultureInfo cultureInfo)
        {
            try
            {
                if ((bool)value)
                {
                    return Visibility.Collapsed;
                }
            }
            catch { }
            return Visibility.Visible;
        }
    }

    public class BooleanToVisibilityAllMultiConverter : IMultiValueConverter
    {
        public object Convert(object[] values, Type targetType, object parameter, CultureInfo culture)
        {
            return values.OfType<bool>().All(b => b) ? Visibility.Visible : Visibility.Collapsed;
        }

        public object[] ConvertBack(object value, Type[] targetTypes, object parameter, CultureInfo culture)
        {
            throw new NotImplementedException();
        }
    }

    public class BooleanToVisibilityAnyMultiConverter : IMultiValueConverter
    {
        public object Convert(object[] values, Type targetType, object parameter, CultureInfo culture)
        {
            return values.OfType<bool>().Any(b => b) ? Visibility.Visible : Visibility.Hidden;
        }

        public object[] ConvertBack(object value, Type[] targetTypes, object parameter, CultureInfo culture)
        {
            throw new NotImplementedException();
        }
    }

    public class NegateValueConverter : IValueConverter
    {
        public object Convert(object value, System.Type targetType, object parameter, System.Globalization.CultureInfo culture)
        {
            bool? binary = (value as bool?);
            if (binary.HasValue)
            {
                return !binary.Value;
            }
            else
            {
                double? d = (value as double?);
                if (d.HasValue)
                {
                    return -d.Value;
                }
                else
                {
                    Visibility? visibility = (value as Visibility?);
                    if (visibility.HasValue)
                    {
                        return ((visibility == Visibility.Collapsed) || (visibility == Visibility.Hidden)) ? Visibility.Visible : Visibility.Collapsed;
                    }
                }
            }
            System.Diagnostics.Debug.Assert(false, "Failed to negate");
            return value;
        }

        public object ConvertBack(object value, System.Type targetType, object parameter, System.Globalization.CultureInfo culture)
        {
            return Convert(value, targetType, parameter, culture);
        }
    }

    public class GridHasWidthConverter : IValueConverter
    {
        public object Convert(object value, System.Type targetType, object parameter, System.Globalization.CultureInfo culture)
        {
            Grid grid = value as Grid;
            if (grid != null)
            {
                return grid.Width > 0;
            }
            return false;
        }

        public object ConvertBack(object value, System.Type targetType, object parameter, System.Globalization.CultureInfo culture)
        {
            return false;
        }
    }

    public class GridLengthValueConverter : IValueConverter
    {
        GridLengthConverter m_converter = new GridLengthConverter();

        public object Convert(object value, Type targetType, object parameter, System.Globalization.CultureInfo culture)
        {
            return m_converter.ConvertFrom(value);
        }

        public object ConvertBack(object value, Type targetType, object parameter, System.Globalization.CultureInfo culture)
        {
            return m_converter.ConvertTo(value, targetType);
        }
    }

    public class ClampValueConverter : IValueConverter
    {
        public object Convert(object value, System.Type targetType, object parameter, System.Globalization.CultureInfo culture)
        {
            string parameterStr = (parameter as string);
            if (parameterStr != null)
            {
                string parameterStrNoWS = parameterStr.Trim();
                string[] tokens = parameterStrNoWS.Trim("[(])".ToArray()).Split(',');
                object min = value;
                object max = value;
                bool minExclusive = parameterStrNoWS.StartsWith("(");
                bool maxExclusive = parameterStrNoWS.EndsWith(")");
                if (tokens.Length == 2)
                {
                    switch (Type.GetTypeCode(targetType))
                    {
                        // Note TryParse will fail on null or whitespace
                        
                        case TypeCode.Byte:
                            {
                                Byte result;
                                Byte rmin = (Byte.TryParse(tokens[0], out result)) ? result : Byte.MinValue;
                                Byte rmax = (Byte.TryParse(tokens[1], out result)) ? result : Byte.MaxValue;
                                if (minExclusive) rmin += (Byte)1;
                                if (maxExclusive) rmax -= (Byte)1;
                                min = rmin;
                                max = rmax;
                            }
                            break;
                        case TypeCode.SByte:
                            {
                                SByte result;
                                SByte rmin = (SByte.TryParse(tokens[0], out result)) ? result : SByte.MinValue;
                                SByte rmax = (SByte.TryParse(tokens[1], out result)) ? result : SByte.MaxValue;
                                if (minExclusive) rmin += (SByte)1;
                                if (maxExclusive) rmax -= (SByte)1;
                                min = rmin;
                                max = rmax;
                            }
                            break;
                        case TypeCode.UInt16:
                            {
                                UInt16 result;
                                UInt16 rmin = (UInt16.TryParse(tokens[0], out result)) ? result : UInt16.MinValue;
                                UInt16 rmax = (UInt16.TryParse(tokens[1], out result)) ? result : UInt16.MaxValue;
                                if (minExclusive) rmin += (UInt16)1;
                                if (maxExclusive) rmax -= (UInt16)1;
                                min = rmin;
                                max = rmax;
                            }
                            break;
                        case TypeCode.UInt32:
                            {
                                UInt32 result;
                                UInt32 rmin = (UInt32.TryParse(tokens[0], out result)) ? result : UInt32.MinValue;
                                UInt32 rmax = (UInt32.TryParse(tokens[1], out result)) ? result : UInt32.MaxValue;
                                if (minExclusive) rmin += (UInt32)1;
                                if (maxExclusive) rmax -= (UInt32)1;
                                min = rmin;
                                max = rmax;
                            }
                            break;
                        case TypeCode.UInt64:
                            {
                                UInt64 result;
                                UInt64 rmin = (UInt64.TryParse(tokens[0], out result)) ? result : UInt64.MinValue;
                                UInt64 rmax = (UInt64.TryParse(tokens[1], out result)) ? result : UInt64.MaxValue;
                                if (minExclusive) rmin += (UInt64)1;
                                if (maxExclusive) rmax -= (UInt64)1;
                                min = rmin;
                                max = rmax;
                            }
                            break;
                        case TypeCode.Int16:
                            {
                                Int16 result;
                                Int16 rmin = (Int16.TryParse(tokens[0], out result)) ? result : Int16.MinValue;
                                Int16 rmax = (Int16.TryParse(tokens[1], out result)) ? result : Int16.MaxValue;
                                if (minExclusive) rmin += (Int16)1;
                                if (maxExclusive) rmax -= (Int16)1;
                                min = rmin;
                                max = rmax;
                            }
                            break;
                        case TypeCode.Int32:
                            {
                                Int32 result;
                                Int32 rmin = (Int32.TryParse(tokens[0], out result)) ? result : Int32.MinValue;
                                Int32 rmax = (Int32.TryParse(tokens[1], out result)) ? result : Int32.MaxValue;
                                if (minExclusive) rmin += (Int32)1;
                                if (maxExclusive) rmax -= (Int32)1;
                                min = rmin;
                                max = rmax;
                            }
                            break;
                        case TypeCode.Int64:
                            {
                                Int64 result;
                                Int64 rmin = (Int64.TryParse(tokens[0], out result)) ? result : Int64.MinValue;
                                Int64 rmax = (Int64.TryParse(tokens[1], out result)) ? result : Int64.MaxValue;
                                if (minExclusive) rmin += (Int64)1;
                                if (maxExclusive) rmax -= (Int64)1;
                                min = rmin;
                                max = rmax;
                            }
                            break;
                        case TypeCode.Decimal:
                            {
                                Decimal result;
                                Decimal rmin = (Decimal.TryParse(tokens[0], out result)) ? result : Decimal.MinValue;
                                Decimal rmax = (Decimal.TryParse(tokens[1], out result)) ? result : Decimal.MaxValue;
                                if (minExclusive) rmin += (Decimal)Double.Epsilon;
                                if (maxExclusive) rmax -= (Decimal)Double.Epsilon;
                                min = rmin;
                                max = rmax;
                            }
                            break;
                        case TypeCode.Double:
                            {
                                Double result;
                                Double rmin = (Double.TryParse(tokens[0], out result)) ? result : Double.MinValue;
                                Double rmax = (Double.TryParse(tokens[1], out result)) ? result : Double.MaxValue;
                                if (minExclusive) rmin += Double.Epsilon;
                                if (maxExclusive) rmax -= Double.Epsilon;
                                min = rmin;
                                max = rmax;
                            }
                            break;
                        case TypeCode.Single:
                            {
                                Single result;
                                Single rmin = (Single.TryParse(tokens[0], out result)) ? result : Single.MinValue;
                                Single rmax = (Single.TryParse(tokens[1], out result)) ? result : Single.MaxValue;
                                if (minExclusive) rmin += Single.Epsilon;
                                if (maxExclusive) rmax -= Single.Epsilon;
                                min = rmin;
                                max = rmax;
                            }
                            break;
                        default:
                            return value;
                    }
                    return hkMath.Clamp(value as IComparable, min as IComparable, max as IComparable);
                }
            }
            return value;
        }

        public object ConvertBack(object value, System.Type targetType, object parameter, System.Globalization.CultureInfo culture)
        {
            return value;
        }
    }

    public class AddValueConverter : IValueConverter
    {
        public object Convert(object value, System.Type targetType, object parameter, System.Globalization.CultureInfo culture)
        {
            double paramValue = System.Convert.ToDouble(parameter, culture);
            double val = (double)value;
            System.Diagnostics.Debug.Assert(!Double.IsNaN(val), "Binding to a NaN value.  If this is Width/Height, try ActualWidth/ActualHeight instead.");

            double finalValue = val + paramValue;
            if (finalValue < 0)
            {
                return Binding.DoNothing;
            }

            return finalValue;
        }

        public object ConvertBack(object value, System.Type targetType, object parameter, System.Globalization.CultureInfo culture)
        {
            double paramValue = System.Convert.ToDouble(parameter, culture);
            return (double)value - paramValue;
        }
    }

    public class AddValuesConverter : IMultiValueConverter
    {
        public object Convert(object[] values, Type targetType, object parameter, CultureInfo culture)
        {
            IEnumerable<double> doubleValues = values.Cast<double>();
            return doubleValues.Sum();
        }

        public object[] ConvertBack(object value, Type[] targetTypes, object parameter, CultureInfo culture)
        {
            throw new NotImplementedException();
        }
    }

    public class ScaleValueConverter : IValueConverter
    {
        public object Convert(object value, System.Type targetType, object parameter, System.Globalization.CultureInfo culture)
        {
            double paramValue = System.Convert.ToDouble(parameter, culture);
            return (double)value * paramValue;
        }

        public object ConvertBack(object value, System.Type targetType, object parameter, System.Globalization.CultureInfo culture)
        {
            double paramValue = System.Convert.ToDouble(parameter, culture);
            if (paramValue != 0)
            {
                return (double)value / paramValue;
            }
            return null;
        }
    }

    public class ColorToBrushConverter : IValueConverter
    {
        public object Convert(object value, Type targetType, object parameter, CultureInfo cultureInfo)
        {
            System.Windows.Media.Color? color = (value as System.Windows.Media.Color?);
            if (color.HasValue)
            {
                System.Windows.Media.SolidColorBrush brush = new System.Windows.Media.SolidColorBrush(color.Value);
                brush.Freeze();
                return brush;
            }
            else
            {
                return value;
            }
        }
        public object ConvertBack(object value, Type targetType, object parameter, CultureInfo cultureInfo)
        {
            System.Windows.Media.SolidColorBrush brush = (value as System.Windows.Media.SolidColorBrush);
            if (brush != null)
            {
                return brush.Color;
            }
            else
            {
                return value;
            }
        }
    }

    public class BrushToColorConverter : IValueConverter
    {
        public object Convert(object value, Type targetType, object parameter, CultureInfo cultureInfo)
        {
            System.Windows.Media.SolidColorBrush brush = (value as System.Windows.Media.SolidColorBrush);
            if (brush != null)
            {
                return brush.Color;
            }
            else
            {
                return value;
            }
        }
        public object ConvertBack(object value, Type targetType, object parameter, CultureInfo cultureInfo)
        {
            System.Windows.Media.Color? color = (value as System.Windows.Media.Color?);
            if (color.HasValue)
            {
                System.Windows.Media.SolidColorBrush brush = new System.Windows.Media.SolidColorBrush(color.Value);
                brush.Freeze();
                return brush;
            }
            else
            {
                return value;
            }
        }
    }

    public class IntToBrushConverter : IValueConverter
    {
        public object Convert(object value, Type targetType, object parameter, CultureInfo cultureInfo)
        {
            int? color = (value as int?);
            if (color.HasValue)
            {
                System.Windows.Media.SolidColorBrush brush = new System.Windows.Media.SolidColorBrush(hkColorUtils.ArgbToColor(color.Value));
                brush.Freeze();
                return brush;
            }
            else
            {
                return value;
            }
        }
        public object ConvertBack(object value, Type targetType, object parameter, CultureInfo cultureInfo)
        {
            System.Windows.Media.SolidColorBrush brush = (value as System.Windows.Media.SolidColorBrush);
            if (brush != null)
            {
                return hkColorUtils.ColorToArgb(brush.Color);
            }
            else
            {
                return value;
            }
        }
    }

    public class ColorToIntConverter : IValueConverter
    {
        public object Convert(object value, Type targetType, object parameter, CultureInfo cultureInfo)
        {
            System.Windows.Media.Color? color = (value as System.Windows.Media.Color?);
            if (color.HasValue)
            {
                return hkColorUtils.ColorToArgb(color.Value);
            }
            else
            {
                return value;
            }
        }
        public object ConvertBack(object value, Type targetType, object parameter, CultureInfo cultureInfo)
        {
            int? color = (value as int?);
            if (color.HasValue)
            {
                return hkColorUtils.ArgbToColor(color.Value);
            }
            else
            {
                return value;
            }
        }
    }

    public class SettingsMarginConverter : IValueConverter
    {
        public object Convert(object value, System.Type targetType, object parameter, System.Globalization.CultureInfo culture)
        {
            return new Thickness(System.Convert.ToDouble(value, culture)-2, 0, 0, 0);
        }

        public object ConvertBack(object value, System.Type targetType, object parameter, System.Globalization.CultureInfo culture)
        {
            return null;
        }
    }

    public class NullImageConverter : IValueConverter
    {
        public object Convert(object value, Type targetType, object parameter, CultureInfo culture)
        {
            string stringValue = value as string;
            if (String.IsNullOrEmpty(stringValue))
            {
                return DependencyProperty.UnsetValue;
            }

            return value;
        }

        public object ConvertBack(object value, Type targetType, object parameter, CultureInfo culture)
        {
            return Binding.DoNothing;
        }
    }

    public class IsNullConverter : IValueConverter
    {
        public object Convert(object value, Type targetType, object parameter, CultureInfo culture)
        {
            return (value == null);
        }

        public object ConvertBack(object value, Type targetType, object parameter, CultureInfo culture)
        {
            return Binding.DoNothing;
        }
    }

    public class VectorToStringConverter : IValueConverter
    {
        public object Convert(object value, System.Type targetType, object parameter, System.Globalization.CultureInfo culture)
        {
            Havok.Vdb.Vector vector = (Havok.Vdb.Vector)value;
            return "( " + vector.X + ", " + vector.Y + ", " + vector.Z + " )";
        }

        public object ConvertBack(object value, System.Type targetType, object parameter, System.Globalization.CultureInfo culture)
        {
            return null;
        }
    }

    public class MultiValueConverter : IMultiValueConverter
    {
        public object Convert(object[] values, System.Type targetType, object parameter, System.Globalization.CultureInfo culture)
        {
            return values.Clone();
        }

        public object[] ConvertBack(object value, System.Type[] targetTypes, object parameter, System.Globalization.CultureInfo culture)
        {
            return null;
        }
    }

    public class EnumToBooleanConverter : IValueConverter
    {
        public object Convert(object value, Type targetType, object parameter, CultureInfo culture)
        {
            string stringParam = parameter as string;
            if (stringParam == null || (Enum.IsDefined(value.GetType(), value) == false))
            {
                return DependencyProperty.UnsetValue;
            }

            object paramValueAsEnum = Enum.Parse(value.GetType(), stringParam);
            return paramValueAsEnum.Equals(value);
        }

        public object ConvertBack(object value, Type targetType, object parameter, CultureInfo culture)
        {
            string stringParam = parameter as string;

            if ((stringParam == null) || !(value is bool) || (bool)value == false)
            {
                return DependencyProperty.UnsetValue;
            }
            return Enum.Parse(targetType, stringParam);
        }
    }

    public class BindingPropertyPassthroughConverter : IValueConverter
    {
        public object Convert(object value, Type targetType, object parameter, CultureInfo culture)
        {
            string propertyPath = (parameter as string);
            if ((value != null) && !(string.IsNullOrWhiteSpace(propertyPath)))
            {
                string[] propertyPathTokens = propertyPath.Split('.');
                try
                {
                    object accessorObject = null;
                    System.Reflection.PropertyInfo accessorInfo = null;
                    {
                        object propertyObject = value;
                        int i = 0; while (true)
                        {
                            string propertyName = propertyPathTokens[i];
                            System.Reflection.PropertyInfo propertyInfo = propertyObject.GetType().GetProperty(propertyName);
                            if (i < propertyPathTokens.Length - 1)
                            {
                                propertyObject = propertyInfo.GetValue(propertyObject);
                                if (propertyObject == null)
                                {
                                    return value;
                                }
                            }
                            else
                            {
                                accessorObject = propertyObject;
                                accessorInfo = propertyInfo;
                                break;
                            }
                            i++;
                        }
                    }

                    if (accessorObject != null)
                    {
                        return Tuple.Create(accessorObject, accessorInfo);
                    }
                    else
                    {
                        System.Diagnostics.Debug.Assert(false, "Couldn't match property name \"" + parameter + "\" for object of type \"" + value.GetType() + "\"");
                    }
                }
                catch { }
            }
            return value;
        }

        public object ConvertBack(object value, Type targetType, object parameter, CultureInfo culture)
        {
            return value;
        }
    }

    public class StringFormatConverter : IMultiValueConverter
    {
        public object Convert(object[] values, System.Type targetType, object parameter, System.Globalization.CultureInfo culture)
        {
            string baseString = values[0]?.ToString();
            List<string> extraValues = new List<string>();
            for (int i = 1; i < values.Length; i++)
            {
                extraValues.Add(values[i]?.ToString());
            }

            return String.Format(baseString, extraValues.ToArray());
        }

        public object[] ConvertBack(object value, System.Type[] targetTypes, object parameter, System.Globalization.CultureInfo culture)
        {
            return null;
        }
    }

    public class StringConcatenateConverter : IMultiValueConverter
    {
        public object Convert(object[] values, System.Type targetType, object parameter, System.Globalization.CultureInfo culture)
        {
            try
            {
                if (parameter is string)
                {
                    return String.Format(parameter as string, values);
                }
                else
                {
                    StringBuilder finalString = new StringBuilder();
                    foreach (object value in values)
                    {
                        string valueAsString = value?.ToString();
                        finalString.Append(valueAsString);
                    }

                    return finalString;
                }
            }
            catch(Exception e)
            {
                System.Diagnostics.Debug.Assert(false, "Failed concatenating string in converter: " + e.Message);
            }

            return Binding.DoNothing;
        }

        public object[] ConvertBack(object value, System.Type[] targetTypes, object parameter, System.Globalization.CultureInfo culture)
        {
            return null;
        }
    }

    public class AccessListFromIndexConverter<T> : IValueConverter
    {
        public object Convert(object value, Type targetType, object parameter, CultureInfo culture)
        {
            List<T> collection = value as List<T>;
            if (collection == null || collection.Count == 0)
            {
                return Binding.DoNothing;
            }

            int index;
            if (Int32.TryParse(parameter as string, out index) && (index >= 0) && index < collection.Count)
            {
                return collection[index];
            }

            return Binding.DoNothing;
        }

        public object ConvertBack(object value, Type targetType, object parameter, CultureInfo culture)
        {
            return value;
        }
    }

    public class AccessFloatListFromIndexConverter : AccessListFromIndexConverter<float> { }
    public class AccessUIntListFromIndexConverter : AccessListFromIndexConverter<uint> { }

    public class AddLeftPaddingToStringConverter : IValueConverter
    {
        public object Convert(object value, Type targetType, object parameter, CultureInfo culture)
        {
            int numPadding = (int)value;
            return "".PadLeft(numPadding * 4);
        }

        public object ConvertBack(object value, Type targetType, object parameter, CultureInfo culture)
        {
            return value;
        }
    }

    public class ValueGroupConverter : List<IValueConverter>, IValueConverter
    {
        public object Convert(object value, Type targetType, object parameter, CultureInfo culture)
        {
            return this.Aggregate(value, (current, converter) => converter.Convert(current, targetType, parameter, culture));
        }

        public object ConvertBack(object value, Type targetType, object parameter, CultureInfo culture)
        {
            return value;
        }
    }
}

/*
 * Havok SDK - Product file, BUILD(#20180110)
 * 
 * Confidential Information of Microsoft Corporation.
 * Not for disclosure or distribution without Microsoft's prior written
 * consent.  This software contains code, techniques and know-how which
 * is confidential and proprietary to Microsoft.  Product and Trade Secret
 * source code contains trade secrets of Microsoft.  Havok Software (C)
 * Copyright 1999-2018 Microsoft Corporation.
 * All Rights Reserved. Use of this software is subject to the
 * terms of an end user license agreement.
 * 
 * The Havok Logo, and the Havok buzzsaw logo are trademarks of Microsoft.
 * Title, ownership rights, and intellectual property rights in the Havok
 * software remain in Microsoft and/or its suppliers.
 * 
 * Use of this software for evaluation purposes is subject to and
 * indicates acceptance of the End User licence Agreement for this
 * product. A copy of the license is included with this software and is
 * also available from Havok Support.
 * 
 */
